# -*- coding: utf-8 -*-
# =============================================================================
# Ural Shortened Url Testing Function
# =============================================================================
#
# Function testing whether the given url is probably a shortened url or not.
#
import re

from ural.hostname_trie_set import HostnameTrieSet
from ural.is_homepage import is_homepage
from ural.utils import safe_urlsplit

DOMAIN_STARTS_L_RE = re.compile(r"^/[0-9a-zA-Z]{3,}/?$")

# Some domains taken from: https://github.com/timleland/url-shorteners
SHORTENER_DOMAINS = [
    "0rz.tw",
    "1-url.net",
    "126.am",
    "1link.in",
    "1tk.us",
    "1un.fr",
    "1url.com",
    "1url.cz",
    "1wb2.net",
    "2.gp",
    "2.ht",
    "23o.net",
    "2ad.in",
    "2big.at",
    "2doc.net",
    "2fear.com",
    "2long.cc",
    "2no.co",
    "2pl.us",
    "2tu.us",
    "2ty.in",
    "2u.xf.cz",
    "2ya.com",
    "3.ly",
    "307.to",
    "3ra.be",
    "3x.si",
    "4i.ae",
    "4ms.me",
    "4sq.com",
    "4url.cc",
    "4view.me",
    "5em.cz",
    "5url.net",
    "5z8.info",
    "6fr.ru",
    "6g6.eu",
    "6url.com",
    "7.ly",
    "76.gd",
    "77.ai",
    "7fth.cc",
    "7li.in",
    "7vd.cn",
    "8u.cz",
    "944.la",
    "98.to",
    "9m.no",
    "9qr.de",
    "a.co",
    "a.gg",
    "a.msn.com",
    "a.nf",
    "a.sgdf.fr",
    "a0.fr",
    "a2a.me",
    "aa.cx",
    "abbr.sk",
    "abbrr.com",
    "abcn.ws",
    "abcurl.net",
    "abnb.me",
    "aca.re",
    "acmi.land",
    "acortar.link",
    "act.gp",
    "ad-med.cz",
    "ad.vu",
    "ad5.eu",
    "ad7.biz",
    "adb.ug",
    "adec.co",
    "adf.ly",
    "adfa.st",
    "adfly.fr",
    "adfoc.us",
    "adjix.com",
    "adli.pw",
    "admy.link",
    "adv.li",
    "afx.cc",
    "ajn.me",
    "aka.gr",
    "al.ly",
    "alil.in",
    "alturl.com",
    "amn.st",
    "amp.gs",
    "amz.run",
    "amzn.eu",
    "amzn.to",
    "any.gs",
    "apne.ws",
    "apple.co",
    "aqva.pl",
    "ar.gy",
    "arab.news",
    "arcg.is",
    "ares.tl",
    "arst.ch",
    "assnat.fr",
    "asso.in",
    "atu.ca",
    "au.ms",
    "ava.st",
    "ayt.fr",
    "azali.fr",
    "azc.cc",
    "b00.fr",
    "b23.ru",
    "b2l.me",
    "b54.in",
    "bacn.me",
    "baid.us",
    "barca.link",
    "bbc.in",
    "bc.vc",
    "bcool.bz",
    "bddy.me",
    "bee4.biz",
    "beth.games",
    "bigaston.link",
    "bim.im",
    "binged.it",
    "bit.do",
    "bit.ly",
    "bitly.com",
    "bitly.ws",
    "bitw.in",
    "bityl.co",
    "bizj.us",
    "bkite.com",
    "blap.net",
    "ble.pl",
    "blip.tv",
    "blizz.ly",
    "bloat.me",
    "bnc.lt",
    "bnpp.lk",
    "boi.re",
    "bote.me",
    "bougn.at",
    "boxd.it",
    "br4.in",
    "bravo.ly",
    "bre.is",
    "brk.to",
    "brzu.net",
    "bsa.ly",
    "budurl.com",
    "buff.ly",
    "buk.me",
    "bul.lu",
    "burnurl.com",
    "bxl.me",
    "bzh.me",
    "c-o.in",
    "c.bnf.fr",
    "c11.kr",
    "c3po.link",
    "cachor.ro",
    "canurl.com",
    "capfr.link",
    "captur.in",
    "catchylink.com",
    "cbs.so",
    "cbsn.ws",
    "cbug.cc",
    "cc.cc",
    "cciparisidf.fr",
    "ccj.im",
    "cf.ly",
    "cf2.me",
    "cf6.co",
    "chck.me",
    "chilp.it",
    "chl.li,",
    "chzb.gr",
    "cjb.net",
    "cl.lk",
    "cl.ly",
    "clck.ru",
    "cldp.ly",
    "cleanuri.com",
    "cli.gs",
    "cliccami.info",
    "clickmeter.com",
    "clickmetertracking.com",
    "clickthru.ca",
    "clikk.in",
    "clop.in",
    "cmpdnt.cc",
    "cn86.org",
    "cnhv.co",
    "cnn.it",
    "coinurl.com",
    "colibris.link",
    "conta.cc",
    "cort.as",
    "cot.ag",
    "couic.fr",
    "cr.tl",
    "crks.me",
    "crwd.fr",
    "cstu.io",
    "ctt.ec",
    "ctvr.us",
    "cudder.it",
    "cur.lv",
    "curl.im",
    "cut.ci",
    "cut.pe",
    "cut.sk",
    "cutt.eu",
    "cutt.ly",
    "cutt.us",
    "cutu.me",
    "cuturl.com",
    "cybr.fr",
    "cyonix.to",
    "d75.eu",
    "daa.pl",
    "dai.ly",
    "db.tt",
    "dbd.game",
    "dd.ma",
    "ddp.net",
    "decenturl.com",
    "dfl8.me",
    "dft.ba",
    "dgl.li",
    "di.sn",
    "digbig.com",
    "digg.com",
    "disq.us",
    "dlbs.fr",
    "dld.bz",
    "dlvr.it",
    "dncgastrnt.co",
    "do.my",
    "doiop.com",
    "dolp.cc",
    "dopen.us",
    "dopice.sk",
    "droid.ws",
    "dv.gd",
    "dwarfurl.com",
    "dy.fi",
    "dy.si",
    "dyo.gs",
    "e.vg",
    "e37.eu",
    "easyuri.com",
    "easyurl.com",
    "easyurl.net",
    "ebx.sh",
    "econ.st",
    "ecra.se",
    "ed.gr",
    "edcollector.fr",
    "eepurl.com",
    "efs.link",
    "eiie.io",
    "ely.re",
    "emojilink.me",
    "energy-citi.es",
    "eng.mg",
    "ept.ms",
    "equid.la",
    "erax.cz",
    "erw.cz",
    "es.pn",
    "eset.la",
    "esyurl.com",
    "etoro.tw",
    "europa.eu",
    "eweri.com",
    "ewerl.com",
    "ex9.co",
    "ezurl.cc",
    "f24.my",
    "fa.b",
    "fa.by",
    "fal.cn",
    "fav.me",
    "fb.me",
    "fb.watch",
    "fbshare.me",
    "fcld.ly",
    "fdip.fr",
    "feedproxy.google.com",
    "ff.im",
    "fff.re",
    "fff.to",
    "fff.wf",
    "fhurl.com",
    "filz.fr",
    "fire.to",
    "firsturl.de",
    "firsturl.net",
    "flic.kr",
    "flip.it",
    "flq.us",
    "fly2.ws",
    "fnk.es",
    "foe.hn",
    "folu.me",
    "fon.gs",
    "fr.rog.gg",
    "frama.link",
    "freak.to",
    "freze.it",
    "from.ubs",
    "fur.ly",
    "fuseurl.com",
    "fuzzy.to",
    "fw.to",
    "fwd4.me",
    "fwib.net",
    "fxn.ws",
    "g.co",
    "g.ro.lt",
    "g00.me",
    "g2g.to",
    "gd.lu",
    "geni.us",
    "gerd.fm",
    "gg.gg",
    "git.io",
    "gizmo.do",
    "gl.am",
    "glo.bo",
    "go.9nl.com",
    "go.asse.fr",
    "go.cegeka.com",
    "go.ign.com",
    "go.letelegramme.fr",
    "go.nasa.gov",
    "go.one.org",
    "go.shr.lc",
    "go.tass.ru",
    "go.usa.gov",
    "go2.me",
    "go2cut.com",
    "goha.bs",
    "goo.gl",
    "goo-gl.me",
    "goo.gle",
    "goo.lu",
    "good.ly",
    "goog.le",
    "goshrink.com",
    "got.cr",
    "gotrim.me",
    "gowat.ch",
    "grabify.link",
    "grem.io",
    "gri.ms",
    "guiama.is",
    "gurl.es",
    "hadej.co",
    "han.gl",
    "hbsd.shop",
    "hc.hockey",
    "hec.su",
    "hellotxt.com",
    "hex.io",
    "hide.my",
    "hiderefer.com",
    "hill.cm",
    "hjkl.fr",
    "hmm.ph",
    "hops.me",
    "hover.com",
    "href.in",
    "href.li",
    "hsblinks.com",
    "ht.ly",
    "htl.li",
    "httpslink.com",
    "htxt.it",
    "hubs.la",
    "hubs.li",
    "hubs.ly",
    "huff.to",
    "hugeurl.com",
    "hulu.com",
    "hurl.it",
    "hurl.me",
    "hurl.ws",
    "hyperurl.co",
    "i-2.co",
    "i99.cz",
    "iaddict.co",
    "ibit.ly",
    "icanhaz.com",
    "icit.fr",
    "ick.li",
    "icks.ro",
    "idek.net",
    "iea.li",
    "ift.tt",
    "ihe.art",
    "iiiii.in",
    "iky.fr",
    "ilix.in",
    "indeedhi.re",
    "info.ms",
    "inreply.to",
    "io.webhelp.com",
    "is.gd",
    "iscool.net",
    "isra.li",
    "isra.liiterasi.net",
    "isw.pub",
    "iterasi.net",
    "itm.im",
    "its.my",
    "ity.im",
    "ix.lt",
    "ix.sk",
    "j.gs",
    "j.mp",
    "jdem.cz",
    "jieb.be",
    "jijr.com",
    "jmp2.net",
    "jp22.net",
    "jpeg.ly",
    "jqw.de",
    "just.as",
    "juve.it",
    "kas.pr",
    "kask.us",
    "kbn.im",
    "kd2.org",
    "kfd.pl",
    "kissa.be",
    "kl.am",
    "klck.me",
    "kloo.me",
    "korta.nu",
    "kr3w.de",
    "krat.si",
    "kratsi.cz",
    "krod.cz",
    "krunchd.com",
    "kuc.cz",
    "kutt.it",
    "kxb.me",
    "l-fff.co",
    "l-k.be",
    "l.gg",
    "l.linklyhq.com",
    "l8r.it",
    "l9.fr",
    "l9k.net",
    "labur.eus",
    "lat.ms",
    "lc-s.co",
    "lc.cx",
    "lcut.in",
    "letop10.",
    "lety.io",
    "lew.ag",
    "lext.so",
    "libero.it",
    "lick.my",
    "lien.li",
    "lien.pl",
    "lifehac.kr",
    "liip.to",
    "liltext.com",
    "lin.cr",
    "lin.io",
    "link.emakinainflux.com",
    "link.infini.fr",
    "link.medium.com",
    "linkbee.com",
    "linkbun.ch",
    "linkn.co",
    "liurl.cn",
    "llu.ch",
    "ln-s.net",
    "ln-s.ru",
    "lnk.bvh.fyi",
    "lnk.co",
    "lnk.gd",
    "lnk.in",
    "lnk.ly",
    "lnk.ms",
    "lnk.sk",
    "lnkd.in",
    "lnked.in",
    "lnki.nl",
    "lnkiy.com,",
    "lnks.fr",
    "lnkurl.com",
    "lnky.fr",
    "lnp.sn",
    "loom.ly",
    "loopt.us",
    "lp25.fr",
    "lru.jp",
    "lsdl.es",
    "lt.tl",
    "lttr.ai",
    "lurl.no",
    "lvvk.com",
    "lynk.my",
    "m1p.fr",
    "m3mi.com",
    "macte.ch",
    "make.my",
    "mash.to",
    "mbe.io",
    "mcaf.ee",
    "mdl29.net",
    "mdpt.fr",
    "merky.de",
    "metamark.net",
    "mic.fr",
    "micro.mn",
    "migre.me",
    "minilien.com",
    "miniurl.be",
    "miniurl.com",
    "minu.me",
    "minurl.fr",
    "mke.me",
    "moby.to",
    "mon.actu.io",
    "monurl.ca",
    "moourl.com",
    "more.sh",
    "mrte.ch",
    "msft.social",
    "msi.gm",
    "mtr.cool",
    "mut.lu",
    "my.sociabble.com",
    "my.wolters-kluwer.fr",
    "myloc.me",
    "myurl.in",
    "n.pr",
    "n9.cl",
    "naver.me",
    "nbc.co",
    "nbcnews.to",
    "nblo.gs",
    "ne1.net",
    "net.ms",
    "net46.net",
    "nicou.ch",
    "nie.mn",
    "nig.gr",
    "ninten.do",
    "niny.io",
    "njx.me",
    "nn.nf",
    "non.li",
    "not.my",
    "notlong.com",
    "nov.io",
    "nq.st",
    "nsfw.in",
    "nutshellurl.com",
    "nvda.ws",
    "nvth.fr",
    "nxy.in",
    "nyti.ms",
    "o-x.fr",
    "oc1.us",
    "oe.cd",
    "okok.fr",
    "om.ly",
    "omf.gd",
    "omoikane.net",
    "on.cnn.com",
    "on.ft.com",
    "on.mktw.net",
    "on.msnbc.com",
    "on.unesco.org",
    "onforb.es",
    "oran.ge",
    "orz.se",
    "ou.af",
    "ou.gd",
    "oua.be",
    "ouo.io",
    "ouvaton.link",
    "ow.ly",
    "owl.li",
    "p.pw",
    "p3t.co",
    "p4k.in",
    "para.pt",
    "parky.tv",
    "past.is",
    "pd.am",
    "pdh.co",
    "ph.dog",
    "ph.ly",
    "pic.gd",
    "pich.in",
    "picta.fr",
    "pin.it",
    "pin.st",
    "ping.fm",
    "piurl.com",
    "pkmn.news",
    "play.st",
    "playst.cc",
    "pli.gs",
    "plots.fr",
    "pnt.me",
    "po.st",
    "politi.co",
    "poprl.com",
    "post.ly",
    "posted.at",
    "pp.gg",
    "ppfr.it",
    "ppst.me",
    "ppt.cc",
    "ppt.li",
    "prejit.cz",
    "prez.ly",
    "prn.to",
    "profile.to",
    "ptab.it",
    "ptiturl.com",
    "ptm.ro",
    "pub.vitrue.com",
    "pulse.ly",
    "pw2.ro",
    "pwne.ws",
    "py6.ru",
    "q.gs",
    "qbn.ru",
    "qdlink.co",
    "qicute.com",
    "qlnk.net",
    "qqc.co",
    "qr.net",
    "qrtag.fr",
    "qte.me",
    "qu.tc",
    "quip-art.com",
    "qxp.cz",
    "qxp.sk",
    "qy.fi",
    "r.im",
    "rb.gy",
    "rb6.co",
    "rb6.me",
    "rc.ca",
    "rcknr.io",
    "rdo.to",
    "rdz.me",
    "read.bi",
    "readthis.ca",
    "reallytinyurl.com",
    "rebrand.ly",
    "rebrandlydomain.com",
    "redir.ec",
    "redir.fr",
    "redirects.ca",
    "redirx.com",
    "redu.it",
    "ref.so",
    "reise.lc",
    "rel.ink",
    "relink.fr",
    "retwt.me",
    "reut.rs",
    "rfi.my",
    "rflsnk.link",
    "ri.ms",
    "rickroll.it",
    "riot.com",
    "riz.cz",
    "riz.gd",
    "rng.rs",
    "rod.gs",
    "roflc.at",
    "rsmonkey.com",
    "rt.nu",
    "rt.se",
    "rt.tc",
    "ru.ly",
    "rubyurl.com",
    "rurl.org",
    "rww.tw",
    "s-url.fr",
    "s.42l.fr",
    "s.bulk.ly",
    "s.click.aliexpress.com",
    "s.id",
    "s.swissin.fo",
    "s4c.in",
    "s7y.us",
    "safe.mn",
    "sagyap.tk",
    "sameurl.com",
    "sci3nc.es",
    "sco.lt",
    "scq.io",
    "sdu.sk",
    "sdut.us",
    "seeme.at",
    "segue.se",
    "sg.fr",
    "sh.st",
    "shar.as",
    "shar.es",
    "sharein.com",
    "sharetabs.com",
    "shdw.me",
    "shink.de",
    "shope.ee",
    "shorl.com",
    "short.cc",
    "short.ie",
    "short.nr",
    "short.pk",
    "short.to",
    "shorte.st",
    "shortlinks.co.uk",
    "shortna.me",
    "shorturl.at",
    "shorturl.com",
    "shoturl.us",
    "shout.to",
    "show.my",
    "shrinkee.com",
    "shrinkify.com",
    "shrinkr.com",
    "shrinkster.com",
    "shrinkurl.in",
    "shrt.fr",
    "shrt.in",
    "shrt.st",
    "shrtco.de",
    "shrten.com",
    "shrunkin.com",
    "shw.me",
    "shy.si",
    "sicax.net",
    "simurl.com",
    "sina.lt",
    "sk.gy",
    "sk.mu",
    "skr.sk",
    "skroc.pl",
    "slate.me",
    "smallr.com",
    "smarturl.it",
    "smll.co",
    "smsh.me",
    "smurl.name",
    "sn.cf",
    "sn.im",
    "sn.vc",
    "snipr.com",
    "snipurl.com",
    "snsw.us",
    "snurl.com",
    "so.arte",
    "soc.cr",
    "soc.fm",
    "social.ora.cl",
    "sohub.io",
    "soo.gd",
    "sp2.ro",
    "spedr.com",
    "spkl.io",
    "spn.sr",
    "spoti.fi",
    "spr.ly",
    "sptfy.com",
    "sq6.ru",
    "sqex.link",
    "sqex.to",
    "sqrl.it",
    "srnk.net",
    "srs.li",
    "ssl.gs",
    "starturl.com",
    "strms.net",
    "sturly.com",
    "su.pr",
    "sumof.us",
    "supr.cl",
    "sur.om.fr",
    "surl.co.uk",
    "surl.hu",
    "surl.me",
    "sux.cz",
    "svlint.org",
    "swll.to",
    "sy.pe",
    "t.cn",
    "t.co",
    "t.lh.com",
    "t.ly",
    "t.me",
    "t1p.de",
    "t2m.io",
    "ta.gd",
    "tabzi.com",
    "tau.pe",
    "tbd.ly",
    "tcrn.ch",
    "tdjt.cz",
    "tek.io",
    "tex.nz",
    "tgr.me",
    "tgr.ph",
    "thesa.us",
    "thinfi.com",
    "thls.co",
    "thrdl.es",
    "ti.me",
    "tidd.ly",
    "tighturl.com",
    "tin.li",
    "tini.cc",
    "tiniuri.com",
    "tiny.cc",
    "tiny.lt",
    "tiny.ly",
    "tiny.ms",
    "tiny.one",
    "tiny.pl",
    "tiny123.com",
    "tinyarro.ws",
    "tinyl.io",
    "tinylink.in",
    "tinytw.it",
    "tinyuri.ca",
    "tinyurl.com",
    "tinyurl.hu",
    "tinyvid.io",
    "tixsu.com",
    "tk.",
    "tl.gd",
    "tldr.sk",
    "tldrify.com",
    "tllg.net",
    "tmblr.co",
    "tmi.me",
    "tnij.org",
    "tnw.to",
    "tny.com",
    "tny.cz",
    "tny.im",
    "to.ly",
    "to8.cc",
    "togoto.us",
    "tohle.de",
    "totc.us",
    "toysr.us",
    "tpm.ly",
    "tpmr.com",
    "tr.im",
    "tr.my",
    "tr5.in",
    "tra.kz",
    "traceurl.com",
    "trbr.io",
    "trck.me",
    "trib.al",
    "trick.ly",
    "trkr.ws",
    "trunc.it",
    "tuppu.net",
    "turo.us",
    "tv5m.tv",
    "tweetburner.com",
    "twet.fr",
    "twhub.com",
    "twi.im",
    "twib.in",
    "twirl.at",
    "twit.ac",
    "twitclicks.com",
    "twitterpan.com",
    "twitterurl.net",
    "twitterurl.org",
    "twitthis.com",
    "twiturl.de",
    "twlr.me",
    "twurl.cc",
    "twurl.nl",
    "u.afp.com",
    "u.mavrev.com",
    "u.nu",
    "u.to",
    "u6e.de",
    "u76.org",
    "ub0.cc",
    "ubi.li",
    "ubxm.fr",
    "uby.es",
    "ucam.me",
    "ug.cz",
    "ulmt.in",
    "ulu.lu",
    "uni.cf",
    "unlc.us",
    "unwo.men",
    "updating.me",
    "upzat.com",
    "ur1.ca",
    "url.az",
    "url.co.uk",
    "url.ie",
    "url.inrae.fr",
    "url.kr",
    "url2.fr",
    "url360.me",
    "url4.eu",
    "url5.org",
    "urlao.com",
    "urlborg.com",
    "urlbrief.com",
    "urlcover.com",
    "urlcut.com",
    "urlenco.de",
    "urlhawk.com",
    "urli.nl",
    "urlin.it",
    "urlkiss.com",
    "urlkr.com",
    "urlot.com",
    "urlpire.com",
    "urlr.me",
    "urls.fr",
    "urls.im",
    "urlshorteningservicefortwitter.com",
    "urlx.ie",
    "urlx.org",
    "urlz.fr",
    "urlz.host",
    "urlzen.com",
    "urub.us",
    "usat.ly",
    "use.my",
    "uslsc.cr",
    "utfg.sk",
    "v.gd",
    "v.ht",
    "v5.gd",
    "vaaa.fr",
    "valv.im",
    "vaza.me",
    "vb.ly",
    "vbly.us",
    "vd55.com",
    "verd.in",
    "vgn.am",
    "vgn.me",
    "viid.me",
    "virl.com",
    "vl.am",
    "vm.lc",
    "vm.tiktok.com",
    "vov.li",
    "vsll.eu",
    "vt802.us",
    "vu.fr",
    "vur.me",
    "vv.vg",
    "w.wiki",
    "w1p.fr",
    "w3t.org",
    "w55.de",
    "wa.link",
    "wa.me",
    "waa.ai",
    "wapo.st",
    "wapurl.co.uk",
    "wb1.eu",
    "wclink.co",
    "we.tl",
    "web99.eu",
    "wed.li",
    "weverse.onelink.me",
    "wideo.fr",
    "wipi.es",
    "wn.nr",
    "wow.link",
    "wp.me",
    "wrfr.me",
    "wrld.bg",
    "wtc.la",
    "wu.cz",
    "ww7.fr",
    "www.sharee.me",
    "www.tmz.me",
    "wwy.me",
    "x.co",
    "x.nu",
    "x.se",
    "x.vu",
    "x10.mx",
    "x2c.eu",
    "x2c.eumx",
    "xaddr.com",
    "xav.cc",
    "xbx.lv",
    "xeeurl.com",
    "xfru.it",
    "xgd.in",
    "xhtxs.cn",
    "xib.me",
    "xl8.eu",
    "xoe.cz",
    "xq.ms",
    "xr.com",
    "xrb.link",
    "xrl.in",
    "xrl.us",
    "xt3.me",
    "xua.me",
    "xub.me",
    "xurl.es",
    "xurl.jp",
    "xurls.co",
    "xzb.cc",
    "y.ahoo.it",
    "y2u.be",
    "yagoa.fr",
    "yagoa.me",
    "yatuc.com",
    "yau.sh",
    "ye.pe",
    "yeca.eu",
    "yect.com",
    "yep.it",
    "yfrog.com",
    "yhoo.it",
    "yiyd.com",
    "yogh.me",
    "yon.ir",
    "youfap.me",
    "youtu.be",
    "ysear.ch",
    "yuarel.com",
    "yweb.com",
    "yyv.co",
    "z0p.de",
    "z9.fr",
    "zapit.nu",
    "zcu.io",
    "zeek.ir",
    "zi.ma",
    "zi.mu",
    "zi.pe",
    "zip.net",
    "zipmyurl.com",
    "zkr.cz",
    "zkrat.me",
    "zkrt.cz",
    "zoodl.com",
    "zpag.es",
    "zpr.io",
    "zsms.net",
    "zti.me",
    "zubb.it",
    "zud.me",
    "zurl.io",
    "zurl.ws",
    "zws.im",
    "zxq.net",
    "zyva.org",
    "zz.gd",
    "zzang.kr",
    "zzb.bz",
    "🏹.to",
    "😻🍕.ws",
    "🍊.ws",
    "›.ws",
    "✩.ws",
    "✿.ws",
    "❥.ws",
    "➔.ws",
    "➞.ws",
    "➡.ws",
    "➨.ws",
    "➯.ws",
    "➹.ws",
    "➽.ws",
]

# NOTE: we use a trie to perform efficient queries and so we don't
# need to test every domain/subdomain linearly
SHORTENER_DOMAINS_TRIE = HostnameTrieSet()

for domain in SHORTENER_DOMAINS:
    SHORTENER_DOMAINS_TRIE.add(domain)


def is_l_shortened_domain(url):
    parsed = safe_urlsplit(url)

    return parsed.hostname.startswith("l.") and re.search(
        DOMAIN_STARTS_L_RE, parsed.path
    )


def is_shortened_url(url):
    parsed = safe_urlsplit(url)

    # NOTE: shortener domain homepages are not shortened urls per se
    if is_homepage(parsed):
        return False

    # Shortener domains starting with 'l.'
    if is_l_shortened_domain(parsed):
        return True

    return SHORTENER_DOMAINS_TRIE.match(parsed)
