""" CLI for code pipeline """

import argparse
import boto3

from .watch import watch_pipeline
from .rollback import rollback_pipeline


def process_args():
    """Process CLI args."""
    parser = argparse.ArgumentParser(
        description="Manage AWS CodePipeline",
    )
    subparsers = parser.add_subparsers()

    watch = subparsers.add_parser("watch", help="watch pipeline activity")
    watch.set_defaults(func=watch_pipeline)
    watch.add_argument(
        "-p",
        "--pipeline",
        help="Name of the pipeline to watch",
        required=True,
    )
    watch.add_argument(
        "-e",
        "--execution-id",
        help="Id of the pipeline to watch",
    )
    watch.add_argument(
        "-f",
        "--follow-logs",
        help="Follow build logs",
        action="store_true",
    )

    rollback = subparsers.add_parser(
        "rollback", help="rollback pipeline stage to prior state"
    )
    rollback.add_argument(
        "-d",
        "--dryrun",
        help="Dryrun mode",
        action="store_true",
    )
    rollback.add_argument(
        "-p",
        "--pipeline",
        help="Name of the pipeline to rollback",
        required=True,
    )
    rollback.add_argument(
        "-s",
        "--stage",
        help="Name of the pipeline stage to rollback",
    )
    rollback.add_argument(
        "-c",
        "--commit",
        help="Commit to rollback to (default: last successful commit for stage)",
    )
    rollback.add_argument(
        "-n",
        "--no-wait",
        action="store_true",
        help="Don't wait for the rollback - run in background",
    )
    rollback.add_argument(
        "-f",
        "--follow-logs",
        help="Follow build logs",
        action="store_true",
    )
    rollback.set_defaults(func=rollback_pipeline)
    args = vars(parser.parse_args())
    if "func" in args:
        args["func"](session=boto3.session.Session(), **args)
    else:
        parser.print_help()


def main():
    """Main entrypoint."""
    process_args()


if __name__ == "__main__":
    main()
