# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/04_hirise.ipynb (unless otherwise specified).

__all__ = ['storage_root', 'baseurl', 'rdrindex', 'RGB_NOMAP', 'RGB_NOMAPCollection']

# Cell

import warnings

import rasterio
from dask import compute, delayed
from fastcore.utils import Path
from yarl import URL

import hvplot.xarray  # noqa
import xarray as xr
from .config import config
from .pds.apps import get_index
from .utils import url_retrieve
from pyrise import products as prod

warnings.filterwarnings("ignore", category=rasterio.errors.NotGeoreferencedWarning)

# Cell
storage_root = config.storage_root / "missions/mro/hirise"
baseurl = URL("https://hirise-pds.lpl.arizona.edu/download/PDS")
rdrindex = get_index("mro.hirise", "rdr")

# Cell
class RGB_NOMAP:
    def __init__(self, obsid):
        self.obsid = obsid
        if self.local_path.exists():
            self.read()  # this is fine, as it's using dask chunks, cheap op

    @property
    def product_id(self):
        return self.obsid + "_COLOR"

    @property
    def filename(self):
        return self.obsid + ".JP2"

    @property
    def pid(self):
        pid = prod.PRODUCT_ID(self.obsid)
        pid.kind = "RGB"
        return pid

    @property
    def meta(self):
        s = rdrindex.query("PRODUCT_ID == @self.product_id").squeeze()
        s.index = s.index.str.lower()
        return s

    # several things in the PDS path either have changed or I did it wrong in pyrise
    @property
    def nomap_jp2_path(self):
        p = Path("EXTRAS/") / self.pid.nomap_jp2_path.replace("/EXTRAS", "")
        return p

    @property
    def url(self):
        return baseurl / str(self.nomap_jp2_path)

    @property
    def local_path(self):
        full = self.nomap_jp2_path
        return storage_root / (f"EXTRAS/RDR/{full.parent.name}/{full.name}")

    def download(self, overwrite=False):
        self.local_path.parent.mkdir(parents=True, exist_ok=True)
        if self.local_path.exists() and not overwrite:
            print("File exists. Use `overwrite=True` to download fresh.")
            return
        url_retrieve(self.url, self.local_path)

    def read(self):
        self.da = xr.open_rasterio(self.local_path, chunks=(1, 2024, 2024))
        return self.da

    def plot_da(self, xslice=None, yslice=None):
        if xslice is not None or yslice is not None:
            data = self.da.isel(x=xslice, y=yslice)
        else:
            data = self.da

        return data.hvplot.image(
            x="x",
            y="y",
            rasterize=True,
            widget_location="top_left",
            cmap="gray",
            frame_height=800,
            frame_width=800,
            flip_yaxis=True,
        )

# Cell


class RGB_NOMAPCollection:
    """Class to deal with a set of RGB_NOMAP products."""

    def __init__(self, obsids):
        self.obsids = obsids

    def get_urls(self):
        """Get URLs for list of obsids.

        Returns
        -------
        List[yarl.URL]
            List of URL objects with the respective PDS URL for download.
        """
        urls = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            urls.append(rgb.url)
        self.urls = urls
        return urls

    @property
    def local_paths(self):
        paths = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            paths.append(rgb.local_path)
        return paths

    def download_collection(self):
        lazys = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            lazys.append(delayed(rgb.download)())
        print("Launching parallel download...")
        compute(*lazys)
        print("Done.")