# -*- coding: utf-8 -*-
# Copyright 2021 IRT Saint Exupéry, https://www.irt-saintexupery.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License version 3 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

# Contributors:
#    INITIAL AUTHORS - API and implementation and/or documentation
#        :author: Francois Gallard
#    OTHER AUTHORS   - MACROSCOPIC CHANGES
"""A formulation for uncoupled or weakly coupled problems."""
from __future__ import division, unicode_literals

from typing import List, Sequence, Tuple

from gemseo.algos.design_space import DesignSpace
from gemseo.core.chain import MDOChain
from gemseo.core.discipline import MDODiscipline
from gemseo.core.execution_sequence import ExecutionSequence, ExecutionSequenceFactory
from gemseo.core.formulation import MDOFormulation
from gemseo.utils.data_conversion import DataConversion


class DisciplinaryOpt(MDOFormulation):
    """The disciplinary optimization.

    This formulation draws the architecture of a mono-disciplinary optimization process
    from an ordered list of disciplines, an objective function and a design space.
    """

    def __init__(
        self,
        disciplines,  # type: Sequence[MDODiscipline]
        objective_name,  # type: str
        design_space,  # type: DesignSpace
        maximize_objective=False,  # type: bool
    ):  # type: (...) -> None
        self.chain = None
        if len(disciplines) > 1:
            self.chain = MDOChain(disciplines)
        super(DisciplinaryOpt, self).__init__(
            disciplines,
            objective_name,
            design_space,
            maximize_objective=maximize_objective,
        )
        self._filter_design_space()
        self._set_defaultinputs_from_ds()
        # Build the objective from its objective name
        self._build_objective_from_disc(objective_name)

    def get_expected_workflow(
        self,
    ):  # type: (...) -> List[ExecutionSequence,Tuple[ExecutionSequence]]
        if self.chain is None:
            return ExecutionSequenceFactory.serial(self.disciplines[0])
        return self.chain.get_expected_workflow()

    def get_expected_dataflow(
        self,
    ):  # type: (...) -> List[Tuple[MDODiscipline,MDODiscipline,List[str]]]
        if self.chain is None:
            return []
        return self.chain.get_expected_dataflow()

    def get_top_level_disc(self):  # type: (...) -> List[MDODiscipline]
        if self.chain is not None:
            return [self.chain]
        return self.disciplines

    def _filter_design_space(self):  # type: (...) -> None
        """Filter the design space to keep only available variables."""
        all_inpts = DataConversion.get_all_inputs(self.get_top_level_disc())
        kept = set(self.design_space.variables_names) & set(all_inpts)
        self.design_space.filter(kept)
