import React, { useEffect, useState } from 'react';
import { i18nStrings } from '../../../constants';
import * as Styles from '../../styles';
import { JobEnvironmentProps } from './jobEnvironment';
import { URLExt } from '@jupyterlab/coreutils';
import { ServerConnection } from '@jupyterlab/services';
import { DropdownItem } from '../../../components/selectinput';
import { SelectInputContainer } from '../SelectInputContainer';
import { getPreSelectedNotebookKernelSpec } from '../Studio/studioHelpers';

const tooltipsStrings = i18nStrings.ScheduleNoteBook.MainPanel.Tooltips;

export const JupyterLabJobEnvironment: React.FC<JobEnvironmentProps> = ({
  setFormState,
  formState,
  isDisabled,
  formErrors,
  setFormErrors,
  contentsManager,
  model,
}) => {
  const [imagesOptions, setImagesOptions] = useState<DropdownItem[]>([]);
  const [kernelOptions, setKernelOptions] = useState<DropdownItem[]>([]);

  const fetchImages = async () => {
    const settings = ServerConnection.makeSettings();
    const url = URLExt.join(settings.baseUrl, '/sagemaker_images');
    const response = await ServerConnection.makeRequest(url, {}, settings);

    if (response.status == 200) {
      const responseJson = await response.json();
      return responseJson.map((imageMetadata: any) => ({
        label: imageMetadata.image_display_name,
        value: imageMetadata.image_arn,
      }));
    }
    return [];
  };

  const fetchKernelNames = async () => {
    const settings = ServerConnection.makeSettings();
    const url = URLExt.join(settings.baseUrl, '/api/kernelspecs');
    const response = await ServerConnection.makeRequest(url, {}, settings);
    let defaultKernelName = null;
    const kernelNames: string[] = [];
    const kernelOptions: DropdownItem[] = [];

    if (response.status === 200) {
      const kernelSpecsResponse = await response.json();
      defaultKernelName = kernelSpecsResponse.default;
      if (kernelSpecsResponse.kernelspecs) {
        Object.values(kernelSpecsResponse.kernelspecs).forEach((kernelSpec: any) => {
          if (kernelSpec) {
            kernelNames.push(kernelSpec.name);
            let kernelDisplayName = kernelSpec.name;
            if (kernelSpec.spec) {
              kernelDisplayName = kernelSpec.spec.display_name;
            }
            kernelOptions.push({ label: kernelDisplayName, value: kernelSpec.name });
          }
        });
      }
    }

    return { defaultKernelName, kernelNames, kernelOptions };
  }

  useEffect(() => {
    Promise.all([
      getPreSelectedNotebookKernelSpec(model.inputFile, contentsManager),
      fetchImages(),
      fetchKernelNames()
    ]).then((values) => {
      const preSelectedNotebookKernelName = values[0];
      const imagesOptions = values[1];
      const kernelDetails = values[2];
      let defaultImageValue: string;
      let defaultKernelValue: string;

      if (imagesOptions && imagesOptions.length > 0) {
        setImagesOptions(imagesOptions);
      }

      if (model.runtimeEnvironmentParameters && model.runtimeEnvironmentParameters.sm_image) {
        defaultImageValue = model.runtimeEnvironmentParameters.sm_image as string;
      } else if (imagesOptions && imagesOptions.length > 0) {
        defaultImageValue = imagesOptions[0].value;
      }
      
      if (kernelDetails && kernelDetails.kernelOptions && kernelDetails.kernelOptions.length > 0) {
        setKernelOptions(kernelDetails.kernelOptions);
      }

      if (model.runtimeEnvironmentParameters && model.runtimeEnvironmentParameters.sm_kernel) {
        defaultKernelValue = model.runtimeEnvironmentParameters.sm_kernel as string;
      } else {
        if (kernelDetails.kernelNames.indexOf(preSelectedNotebookKernelName) >= 0) {
          defaultKernelValue = preSelectedNotebookKernelName;
        } else {
          defaultKernelValue = kernelDetails.defaultKernelName || ''; 
        }
      }

      setFormState(state => ({
        ...state,
        sm_image: defaultImageValue ?? '',
        sm_kernel: defaultKernelValue ?? '',
      }));
    })
    .catch((error) => console.error(error));
  }, []);

  const handleImageSelection = (
    item: DropdownItem | string,
  ) => {
    if (!item || typeof item === 'string') {
      return;
    }

    setFormState({
      ...formState,
      sm_image: item.value,
    });
  };

  const handleKernelSelection = (
    item: DropdownItem | string,
  ) => {
    if (!item || typeof item === 'string') {
      return;
    }

    setFormState({
      ...formState,
      sm_kernel: item.value,
    });
  };

  return (<div className={Styles.WidgetFieldsContainer}>
    <SelectInputContainer
      data-testid="sm_image_dropdown"
      options={imagesOptions}
      value={formState.sm_image}
      label={i18nStrings.ImageSelector.label}
      onChange={handleImageSelection}
      readOnly={isDisabled}
      toolTipText={tooltipsStrings.ImageTooltipText}
      required
    />

    <SelectInputContainer
      data-testid="sm_kernel_dropdown"
      options={kernelOptions}
      value={formState.sm_kernel}
      label={i18nStrings.KernelSelector.label}
      onChange={handleKernelSelection}
      readOnly={isDisabled}
      toolTipText={tooltipsStrings.KernelTooltipText}
      required
    />
  </div>)
}
