"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const constructs_1 = require("constructs");
const common_1 = require("./common");
/**
 * GitHub Actions runner provider using CodeBuild to execute the actions.
 *
 * Creates a project that gets started for each job.
 */
class CodeBuildRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.label = props.label || 'codebuild';
        this.vpc = props.vpc;
        this.securityGroup = props.securityGroup;
        const buildSpec = {
            version: '0.2',
            env: {
                variables: {
                    RUNNER_TOKEN: 'unspecified',
                    RUNNER_NAME: 'unspecified',
                    RUNNER_LABEL: 'unspecified',
                    OWNER: 'unspecified',
                    REPO: 'unspecified',
                    GITHUB_DOMAIN: 'github.com',
                },
            },
            phases: {
                install: {
                    commands: [
                        'sudo -Hu runner /home/runner/config.sh --unattended --url "https://${GITHUB_DOMAIN}/${OWNER}/${REPO}" --token "${RUNNER_TOKEN}" --ephemeral --work _work --labels "${RUNNER_LABEL}" --disableupdate --name "${RUNNER_NAME}"',
                    ],
                },
                build: {
                    commands: [
                        'sudo -Hu runner /home/runner/run.sh',
                    ],
                },
            },
        };
        this.project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            vpc: this.vpc,
            securityGroups: this.securityGroup ? [this.securityGroup] : undefined,
            subnetSelection: props.subnetSelection,
            timeout: props.timeout || aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.fromAsset(this, 'image', {
                    directory: path.join(__dirname, 'docker-images', 'codebuild'),
                    buildArgs: {
                        RUNNER_VERSION: props.runnerVersion ? props.runnerVersion.version : common_1.RunnerVersion.latest().version,
                    },
                }),
                computeType: props.computeType || aws_codebuild_1.ComputeType.SMALL,
            },
            logging: {
                cloudWatch: {
                    logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                        retention: props.logRetention || aws_logs_1.RetentionDays.ONE_MONTH,
                    }),
                },
            },
        });
        this.grantPrincipal = this.project.grantPrincipal;
    }
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CodeBuildStartBuild(this, 'Linux CodeBuild Runner', {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            project: this.project,
            environmentVariablesOverride: {
                RUNNER_TOKEN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerTokenPath,
                },
                RUNNER_NAME: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerNamePath,
                },
                RUNNER_LABEL: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: this.label,
                },
                GITHUB_DOMAIN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.githubDomainPath,
                },
                OWNER: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.ownerPath,
                },
                REPO: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.repoPath,
                },
            },
        });
    }
    get connections() {
        return this.project.connections;
    }
}
exports.CodeBuildRunner = CodeBuildRunner;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildRunner", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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