import * as asg from 'aws-cdk-lib/aws-autoscaling';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as sns from 'aws-cdk-lib/aws-sns';
import { Construct } from 'constructs';
import { DockerVolumes } from './gitlab-runner-interfaces';
/**
 * GitlabRunnerAutoscaling Props.
 */
export interface GitlabRunnerAutoscalingProps {
    /**
     * Gitlab Runner version
     * Please give me gitlab runner version.
     */
    readonly gitlabRunnerVersion: string;
    /**
     * Gitlab token.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN' });
     */
    readonly gitlabToken: string;
    /**
     * Image URL of Gitlab Runner.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', gitlabRunnerImage: 'gitlab/gitlab-runner:alpine' });
     *
     * @default public.ecr.aws/gitlab/gitlab-runner:latest
     *
     */
    readonly gitlabRunnerImage?: string;
    /**
     * Runner default EC2 instance type.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', instanceType: 't3.small' });
     *
     * @default - t3.micro
     *
     */
    readonly instanceType?: string;
    /**
     * VPC for the Gitlab Runner .
     *
     * @example
     * const newVpc = new Vpc(stack, 'NewVPC', {
     *   ipAddresses: IpAddresses.cidr('10.0.0.0/16'),
     *   maxAzs: 2,
     *   subnetConfiguration: [{
     *     cidrMask: 26,
     *     name: 'RunnerVPC',
     *     subnetType: SubnetType.PUBLIC,
     *   }],
     *   natGateways: 0,
     * });
     *
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', vpc: newVpc });
     *
     * @default - A new VPC will be created.
     *
     */
    readonly vpc?: ec2.IVpc;
    /**
     * IAM role for the Gitlab Runner Instance .
     *
     * @example
     * const role = new Role(stack, 'runner-role', {
     *   assumedBy: new ServicePrincipal('ec2.amazonaws.com'),
     *   description: 'For Gitlab Runner Test Role',
     *   roleName: 'Runner-Role',
     * });
     *
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', instanceRole: role });
     *
     * @default - new Role for Gitlab Runner Instance , attach AmazonSSMManagedInstanceCore Policy .
     *
     */
    readonly instanceRole?: iam.IRole;
    /**
     * Run worker nodes as EC2 Spot
     *
     * @default - false
     */
    readonly spotInstance?: boolean;
    /**
     * Minimum capacity limit for autoscaling group.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', minCapacity: 2 });
     *
     * @default - minCapacity: 1
     *
     */
    readonly minCapacity?: number;
    /**
     * Maximum capacity limit for autoscaling group.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', maxCapacity: 4 });
     *
     * @default - desiredCapacity
     *
     */
    readonly maxCapacity?: number;
    /**
     * Desired capacity limit for autoscaling group.
     *
     * @example
     * new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', desiredCapacity: 2 });
     *
     * @default - minCapacity, and leave unchanged during deployment
     *
     */
    readonly desiredCapacity?: number;
    /**
     * tags for the runner
     *
     * @default - ['runner', 'gitlab', 'awscdk']
     */
    readonly tags?: string[];
    /**
     * Gitlab Runner register url .
     *
     * @example
     * const runner = new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN',gitlabUrl: 'https://gitlab.com/'});
     *
     * @default - https://gitlab.com/ , The trailing slash is mandatory.
     *
     */
    readonly gitlabUrl?: string;
    /**
     * Gitlab Runner instance EBS size .
     *
     * @example
     * const runner = new GitlabRunnerAutoscaling(stack, 'runner', { gitlabToken: 'GITLAB_TOKEN', ebsSize: 100});
     *
     * @default - ebsSize=60
     *
     */
    readonly ebsSize?: number;
    /**
     * VPC subnet
     *
     * @example
     * const vpc = new Vpc(stack, 'nat', {
     * natGateways: 1,
     * maxAzs: 2,
     * });
     * const runner = new GitlabRunnerAutoscaling(stack, 'testing', {
     *   gitlabToken: 'GITLAB_TOKEN',
     *   instanceType: 't3.large',
     *   instanceRole: role,
     *   ebsSize: 100,
     *   vpc: vpc,
     *   vpcSubnet: {
     *     subnetType: SubnetType.PUBLIC,
     *   },
     * });
     *
     * @default - SubnetType.PRIVATE subnet
     */
    readonly vpcSubnet?: ec2.SubnetSelection;
    /**
     * add another Gitlab Container Runner Docker Volumes Path at job runner runtime.
     *
     * more detail see https://docs.gitlab.com/runner/configuration/advanced-configuration.html#the-runnersdocker-section
     *
     * @default - already mount "/var/run/docker.sock:/var/run/docker.sock"
     *
     * @example
     * dockerVolumes: [
     *   {
     *     hostPath: '/tmp/cache',
     *     containerPath: '/tmp/cache',
     *   },
     * ],
     */
    readonly dockerVolumes?: DockerVolumes[];
    /**
     * Parameters of put_metric_alarm function
     *
     * https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/cloudwatch.html#CloudWatch.Client.put_metric_alarm
     *
     * @default - [{
     *     AlarmName: 'GitlabRunnerDiskUsage',
     *     MetricName: 'disk_used_percent',
     * }]
     *
     */
    readonly alarms?: object[];
}
/**
 * GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 */
export declare class GitlabRunnerAutoscaling extends Construct {
    /**
     * The IAM role assumed by the Runner instance.
     */
    readonly instanceRole: iam.IRole;
    /**
     * This represents a Runner Auto Scaling Group
     */
    readonly autoscalingGroup: asg.AutoScalingGroup;
    /**
     * The EC2 runner's VPC.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The EC2 runner's default SecurityGroup.
     */
    readonly securityGroup: ec2.ISecurityGroup;
    /**
     * The SNS topic to suscribe alarms for EC2 runner's metrics.
     */
    readonly topicAlarm: sns.ITopic;
    constructor(scope: Construct, id: string, props: GitlabRunnerAutoscalingProps);
    private dockerVolumesList;
    /**
     * @param props
     * @returns Array.
     */
    createUserData(props: GitlabRunnerAutoscalingProps): string[];
}
