# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['inertia', 'inertia.tests', 'inertia.tests.testapp']

package_data = \
{'': ['*'], 'inertia': ['templates/*']}

install_requires = \
['django>=4.0,<5.0', 'requests>=2,<3']

setup_kwargs = {
    'name': 'inertia-django',
    'version': '0.5.1',
    'description': 'Django adapter for the InertiaJS framework',
    'long_description': '![image](https://user-images.githubusercontent.com/6599653/114456558-032e2200-9bab-11eb-88bc-a19897f417ba.png)\n\n\n# Inertia.js Django Adapter\n\n## Installation\n\n### Backend\n\nInstall the following python package via pip\n```bash\npip install inertia-django\n```\n\nAdd the Inertia app to your `INSTALLED_APPS` in `settings.py`\n```python\nINSTALLED_APPS = [\n  # django apps,\n  \'inertia\',\n  # your project\'s apps,\n]\n```\n\nAdd the Inertia middleware to your `MIDDLEWARE` in `settings.py`\n```python\nMIDDLEWARE = [\n  # django middleware,\n  \'inertia.middleware.InertiaMiddleware\',\n  # your project\'s middleware,\n]\n```\n\nFinally, create a layout which exposes `{% block inertia %}{% endblock %}` in the body and set the path to this layout as `INERTIA_LAYOUT` in your `settings.py` file.\n\nNow you\'re all set!\n\n### Frontend\n\nDjango specific frontend docs coming soon. For now, we recommend installing [django_vite](https://github.com/MrBin99/django-vite) \nand following the commits on the Django Vite [example repo](https://github.com/MrBin99/django-vite-example). Once Vite is setup with\nyour frontend of choice, just replace the contents of `entry.js` with [this file (example in react)](https://github.com/BrandonShar/inertia-rails-template/blob/main/app/frontend/entrypoints/application.jsx)\n\n\nYou can also check out the official Inertia docs at https://inertiajs.com/. \n\n### CSRF\n\nDjango\'s CSRF tokens are tightly coupled with rendering templates so Inertia Django automatically handles adding the CSRF cookie for you to each Inertia response. Because the default names Django users for the CSRF headers don\'t match Axios (the Javascript request library Inertia uses), we\'ll need to either modify Axios\'s defaults OR Django\'s settings. \n\n**You only need to choose one of the following options, just pick whichever makes the most sense to you!**\n\nIn your `entry.js` file\n```javascript\naxios.defaults.xsrfHeaderName = "X-CSRFToken"\naxios.defaults.xsrfCookieName = "csrftoken"\n```\nOR\n\nIn your Django `settings.py` file\n```python\nCSRF_HEADER_NAME = \'HTTP_X_XSRF_TOKEN\'\nCSRF_COOKIE_NAME = \'XSRF-TOKEN\'\n```\n\n## Usage\n\n### Responses\n\nRender Inertia responses is simple, you can either use the provided inertia render function or, for the most common use case, the inertia decorator. The render function accepts four arguments, the first is your request object. The second is the name of the component you want to render from within your pages directory (without extension). The third argument is a dict of `props` that should be provided to your components. The final argument is `template_data`, for any variables you want to provide to your template, but this is much less common.\n\n```python\nfrom inertia import render\nfrom .models import Event\n\ndef index(request):\n  return render(request, \'Event/Index\', props={\n    \'events\': Event.objects.all()\n  })\n```\n\nOr use the simpler decorator for the most common use cases\n\n```python\nfrom inertia import inertia\nfrom .models import Event\n\n@inertia(\'Event/Index\')\ndef index(request):\n  return {\n    \'events\': Event.objects.all(),\n  }\n```\n\n### Shared Data\n\nIf you have data that you want to be provided as a prop to every component (a common use-case is information about the authenticated user) you can use the `share` method. A common place to put this would be in some custom middleware.\n\n```python\nfrom inertia import share\nfrom django.conf import settings\nfrom .models import User\n\ndef inertia_share(get_response):\n  def middleware(request):\n    share(request, \n      app_name=settings.APP_NAME,\n      user_count=lambda: User.objects.count(), # evaluated lazily at render time\n      user=lambda: request.user, # evaluated lazily at render time\n    )\n\n    return get_response(request)\n  return middleware\n```\n\n### Lazy Props\nOn the front end, Inertia supports the concept of "partial reloads" where only the props requested\nare returned by the server. Sometimes, you may want to use this flow to avoid processing a particularly slow prop on the intial load. In this case, you can use `Lazy props`. Lazy props aren\'t evaluated unless they\'re specifically requested by name in a partial reload.\n\n```python\nfrom inertia import lazy, inertia\n\n@inertia(\'ExampleComponent\')\ndef example(request):\n  return {\n    \'name\': lambda: \'Brandon\', # this will be rendered on the first load as usual\n    \'data\': lazy(lambda: some_long_calculation()), # this will only be run when specifically requested by partial props and WILL NOT be included on the initial load\n  }\n```\n\n### Json Encoding\n\nInertia Django ships with a custom JsonEncoder at `inertia.utils.InertiaJsonEncoder` that extends Django\'s \n`DjangoJSONEncoder` with additional logic to handle encoding models and Querysets. If you have other json \nencoding logic you\'d prefer, you can set a new JsonEncoder via the settings.\n\n### SSR \n\n#### Backend\nEnable SSR via the `INERTIA_SSR_URL` and `INERTIA_SSR_ENABLED` settings\n\n#### Frontend\nComing Soon!\n\n## Settings\n\nInertia Django has a few different settings options that can be set from within your project\'s `settings.py` file. Some of them have defaults.\n\nThe default config is shown below\n\n```python\nINERTIA_VERSION = \'1.0\' # defaults to \'1.0\'\nINERTIA_LAYOUT = \'layout.html\' # required and has no default\nINERTIA_JSON_ENCODER = CustomJsonEncoder # defaults to inertia.utils.InertiaJsonEncoder\nINERTIA_SSR_URL = \'http://localhost:13714\' # defaults to http://localhost:13714\nINERTIA_SSR_ENABLED = False # defaults to False\n```\n\n## Testing\n\nInertia Django ships with a custom TestCase to give you some nice helper methods and assertions.\nTo use it, just make sure your TestCase inherits from `InertiaTestCase`. `InertiaTestCase` inherits from Django\'s `django.test.TestCase` so it includes transaction support and a client.\n\n```python\nfrom inertia.test import InertiaTestCase\n\nclass ExampleTestCase(InertiaTestCase):\n  def test_show_assertions(self):\n    self.client.get(\'/events/\')\n\n    # check the component\n    self.assertComponentUsed(\'Event/Index\')\n    \n    # access the component name\n    self.assertEqual(self.component(), \'Event/Index\')\n    \n    # props (including shared props)\n    self.assertHasExactProps({name: \'Brandon\', sport: \'hockey\'})\n    self.assertIncludesProps({sport: \'hockey\'})\n    \n    # access props\n    self.assertEquals(self.props()[\'name\'], \'Brandon\')\n    \n    # template data\n    self.assertHasExactTemplateData({name: \'Brian\', sport: \'basketball\'})\n    self.assertIncludesTemplateData({sport: \'basketball\'})\n    \n    # access template data \n    self.assertEquals(self.template_data()[\'name\'], \'Brian\')\n```\n\nThe inertia test helper also includes a special `inertia` client that pre-sets the inertia headers\nfor you to simulate an inertia response. You can access and use it just like the normal client with commands like `self.inertia.get(\'/events/\')`. When using the inertia client, inertia custom assertions **are not** enabled though, so only use it if you want to directly assert against the json response.\n\n## Thank you\n\nA huge thank you to the community members who have worked on InertiaJS for Django before us. Parts of this repo were particularly inspired by [Andres Vargas](https://github.com/zodman) and [Samuel Girardin](https://github.com/girardinsamuel). Additional thanks to Andres for the Pypi project.\n\n*Maintained and sponsored by the team at [bellaWatt](https://bellawatt.com/)*\n\n[![bellaWatt Logo](https://user-images.githubusercontent.com/6599653/114456832-5607d980-9bab-11eb-99c8-ab39867c384e.png)](https://bellawatt.com/)\n',
    'author': 'Brandon Shar',
    'author_email': 'brandon@bellawatt.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/inertiajs/inertia-django',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
