"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An EC2 Key Pair.
 *
 * @stability stable
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair.
     *
     * The private key will be stored in AWS Secrets Manager
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.privateKeyArn = '';
        /**
         * ARN of the public key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.publicKeyArn = '';
        /**
         * The public key.
         *
         * Only filled, when `exposePublicKey = true`
         *
         * @stability stable
         */
        this.publicKeyValue = '';
        /**
         * Name of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairName = '';
        /**
         * ID of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairID = '';
        /**
         * @stability stable
         */
        this.prefix = '';
        if (props.removeKeySecretsAfterDays &&
            (props.removeKeySecretsAfterDays < 0 ||
                (props.removeKeySecretsAfterDays > 0 &&
                    props.removeKeySecretsAfterDays < 7) ||
                props.removeKeySecretsAfterDays > 30)) {
            cdk.Annotations.of(this).addError(`Parameter removeKeySecretsAfterDays must be 0 or between 7 and 30. Got ${props.removeKeySecretsAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag(createdByTag, ID);
        const kmsPrivate = props.kmsPrivateKey || props.kms;
        const kmsPublic = props.kmsPublicKey || props.kms;
        const key = new cdk.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KmsPrivate: (kmsPrivate === null || kmsPrivate === void 0 ? void 0 : kmsPrivate.keyArn) || 'alias/aws/secretsmanager',
                KmsPublic: (kmsPublic === null || kmsPublic === void 0 ? void 0 : kmsPublic.keyArn) || 'alias/aws/secretsmanager',
                StorePublicKey: props.storePublicKey || false,
                ExposePublicKey: props.exposePublicKey || false,
                RemoveKeySecretsAfterDays: props.removeKeySecretsAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-ssh-key/',
                StackName: stack,
                Tags: cdk.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPrivateKey !== 'undefined') {
            props.kmsPrivateKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPrivateKey);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPublicKey !== 'undefined') {
            props.kmsPublicKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPublicKey);
            key.node.addDependency(this.lambda.role);
        }
        this.privateKeyArn = key.getAttString('PrivateKeyARN');
        this.publicKeyArn = key.getAttString('PublicKeyARN');
        this.publicKeyValue = key.getAttString('PublicKeyValue');
        this.keyPairName = key.getAttString('KeyPairName');
        this.keyPairID = key.getAttString('KeyPairID');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2() // generally allow to inspect key pairs
                    .allow()
                    .toDescribeKeyPairs(),
                new statement.Ec2() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateKeyPair()
                    .toCreateTags()
                    .onKeyPair('*')
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Ec2() // allow delete/update, only if createdByTag is set
                    .allow()
                    .toDeleteKeyPair()
                    .toCreateTags()
                    .toDeleteTags()
                    .onKeyPair('*')
                    .ifResourceTag(createdByTag, ID),
                new statement.Secretsmanager() // generally allow to list secrets. we need this to check if a secret exists before attempting to delete it
                    .allow()
                    .toListSecrets(),
                new statement.Secretsmanager() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateSecret()
                    .toTagResource()
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Secretsmanager() // allow delete/update, only if createdByTag is set
                    .allow()
                    .allMatchingActions('/^(Describe|Delete|Put|Update)/')
                    .toGetSecretValue()
                    .toGetResourcePolicy()
                    .toRestoreSecret()
                    .toListSecretVersionIds()
                    .toUntagResource()
                    .ifResourceTag(createdByTag, ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPrivateKey(grantee) {
        return this.grantRead(this.privateKeyArn, grantee);
    }
    /**
     * Grants read access to the public key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPublicKey(grantee) {
        return this.grantRead(this.publicKeyArn, grantee);
    }
    grantRead(arn, grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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