"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        this.adminRole = this.createAdminRole(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: this.adminRole,
            subnets: props.subnets,
            vpc: props.vpc,
            environment: props.environment,
        });
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.adminRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.adminRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    createAdminRole(props) {
        const stack = core_1.Stack.of(this);
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        const creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.list({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.string({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        // see https://github.com/aws/aws-cdk/issues/9027
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['ec2:DescribeVpcs'],
            resources: [stack.formatArn({
                    service: 'ec2',
                    resource: 'vpc',
                    resourceName: props.vpc.vpcId,
                })],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        return creationRole;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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