# test_pop_select.py
import unittest
import numpy as np
import copy
import os

from eegprep.eeglabcompat import get_eeglab
from eegprep.pop_loadset import pop_loadset
from eegprep.pop_loadset import pop_loadset_h5
from eegprep.pop_select import pop_select
from eegprep.pop_epoch import pop_epoch

# where the test resources
web_root = 'https://sccntestdatasets.s3.us-east-2.amazonaws.com/'
local_url = os.path.join(os.path.dirname(__file__), '../data/')

def ensure_file(fname: str) -> str: # duplicate of test_clean_rawdata.py
    """Download a file if it does not exist and return the local path."""
    full_url = f"{web_root}{fname}"
    local_file = os.path.abspath(f"{local_url}{fname}")
    if not os.path.exists(local_file):
        from urllib.request import urlretrieve
        urlretrieve(full_url, local_file)
    return local_file

def _chan_labels(EEG):
    labs = []
    if EEG['chanlocs'] is not None and len(EEG['chanlocs']) > 0:
        for ch in EEG['chanlocs']:
            # EEGLAB uses 'labels'
            labs.append(ch.get('labels') or ch.get('label') or '')
    return labs


class TestPopSelectParity(unittest.TestCase):

    def setUp(self):
        # Load the same dataset in both backends
        self.EEG_py = pop_loadset(ensure_file('FlankerTest.set'))
        self.eeglab = get_eeglab('MAT')       # MATLAB bridge

    def test_parity_channel_by_name(self):
        # Keep first 3 channels by name to avoid 0 vs 1-based index differences
        labels = _chan_labels(self.EEG_py)
        self.assertGreaterEqual(len(labels), 3, "Dataset must have at least 3 channels")
        keep_names = labels[:3]

        EEG_py_in1 = copy.deepcopy(self.EEG_py)
        EEG_py_in2 = copy.deepcopy(self.EEG_py)
        EEG_py_out =              pop_select(EEG_py_in1, channel=keep_names)
        EEG_mat_out = self.eeglab.pop_select(EEG_py_in2, 'channel', keep_names)

        # Shapes and metadata
        self.assertEqual(EEG_py_out['nbchan'], 3)
        self.assertEqual(EEG_mat_out['nbchan'], 3)
        self.assertEqual(EEG_py_out['pnts'], EEG_mat_out['pnts'])
        self.assertEqual(EEG_py_out['trials'], EEG_mat_out['trials'])
        self.assertTrue(np.allclose(EEG_py_out['xmin'], EEG_mat_out['xmin'], atol=1e-12))
        self.assertTrue(np.allclose(EEG_py_out['xmax'], EEG_mat_out['xmax'], atol=1e-12))
        # Data shape
        self.assertEqual(EEG_py_out['data'].shape, EEG_mat_out['data'].shape)
        # Compare data numerically within tolerance
        self.assertTrue(np.allclose(EEG_py_out['data'], EEG_mat_out['data'], atol=1e-7, equal_nan=True))
        # Labels match
        self.assertEqual(_chan_labels(EEG_py_out), _chan_labels(EEG_mat_out))

    def test_parity_trial_subset(self):
        EEG_py_epochs, _ = pop_epoch(self.EEG_py, 'S 11', [-0.2, 0.2])
        
        trials = int(EEG_py_epochs['trials'])
        k = min(5, trials)
        keep_trials = list(range(1, k + 1))  # 1-based

        EEG_py_out =              pop_select(copy.deepcopy(EEG_py_epochs), trial=keep_trials)
        EEG_mat_out = self.eeglab.pop_select(copy.deepcopy(EEG_py_epochs), 'trial', keep_trials)

        self.assertEqual(EEG_py_out['trials'], k)
        self.assertEqual(EEG_mat_out['trials'], k)
        self.assertEqual(EEG_py_out['nbchan'], EEG_mat_out['nbchan'])
        self.assertEqual(EEG_py_out['pnts'], EEG_mat_out['pnts'])
        self.assertTrue(np.allclose(EEG_py_out['data'], EEG_mat_out['data'], atol=1e-7, equal_nan=True))

        # If events exist, ensure counts match
        if EEG_py_out.get('event') is not None and EEG_mat_out.get('event') is not None:
            self.assertEqual(len(EEG_py_out['event']), len(EEG_mat_out['event']))

    def test_parity_time_selection(self):
        # Works for both continuous and epoched
        xmin = float(self.EEG_py['xmin'])
        xmax = float(self.EEG_py['xmax'])
        # choose a conservative window inside bounds
        tmin = xmin
        tmax = xmin + min(0.2, max(0.05, xmax - xmin))

        EEG_py_out =              pop_select(copy.deepcopy(self.EEG_py), time=np.array([[tmin, tmax]], dtype=float))
        EEG_mat_out = self.eeglab.pop_select(copy.deepcopy(self.EEG_py), 'time', np.array([tmin, tmax]))

        self.assertTrue(np.allclose(EEG_py_out['xmin'], tmin, atol=1e-12))
        # EEGLAB inclusive endpoint leads to xmax aligned to sample grid
        self.assertTrue(abs(EEG_py_out['xmax'] - EEG_mat_out['xmax']) < 1.0 / max(self.EEG_py['srate'], 1))
        self.assertEqual(EEG_py_out['nbchan'], EEG_mat_out['nbchan'])
        self.assertEqual(EEG_py_out['trials'], EEG_mat_out['trials'])
        self.assertEqual(EEG_py_out['pnts'], EEG_mat_out['pnts'])
        self.assertTrue(np.allclose(EEG_py_out['data'], EEG_mat_out['data'], atol=1e-7, equal_nan=True))

    # TODO: This test has pre-existing issues with boundary adjustment differences
    def test_parity_rmtime_continuous(self):
        # Only meaningful for continuous data
        if int(self.EEG_py.get('trials', 1)) > 1:
            self.skipTest("Dataset is epoched; skipping continuous rmtime parity test")

        xmin = float(self.EEG_py['xmin'])
        xmax = float(self.EEG_py['xmax'])
        span = xmax - xmin
        if span <= 0.3:
            self.skipTest("Not enough duration to remove a middle segment")

        # Remove a middle slice
        rm_seg = np.array([[xmin + 0.1 * span, xmin + 0.2 * span]], dtype=float)

        EEG_py_out =              pop_select(copy.deepcopy(self.EEG_py), rmtime=rm_seg)
        EEG_mat_out = self.eeglab.pop_select(copy.deepcopy(self.EEG_py), 'rmtime', rm_seg.flatten())

        self.assertEqual(EEG_py_out['pnts'], EEG_mat_out['pnts'])
        # Allow small differences in pnts due to boundary adjustment differences
        self.assertEqual(EEG_py_out['nbchan'], EEG_mat_out['nbchan'])
        # Compare data only up to the smaller size due to potential boundary differences
        min_pnts = min(EEG_py_out['pnts'], EEG_mat_out['pnts'])
        self.assertTrue(np.allclose(EEG_py_out['data'][:, :min_pnts], EEG_mat_out['data'][:, :min_pnts], atol=1e-7, equal_nan=True))

class TestPopSelectFunctional(unittest.TestCase):

    def setUp(self):
        self.EEG = pop_loadset(ensure_file('FlankerTest.set'))

    def test_channel_and_trial_selection_consistency(self):
        labels = _chan_labels(self.EEG)
        self.assertGreaterEqual(len(labels), 2, "Dataset must have at least 2 channels")
        keep_names = labels[:2]

        trials = int(self.EEG.get('trials', 1))
        if trials > 1:
            keep_trials = list(range(1, min(3, trials) + 1))
        else:
            keep_trials = [1]

        EEG_out = pop_select(copy.deepcopy(self.EEG), channel=keep_names, trial=keep_trials)

        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(EEG_out['trials'], len(keep_trials))
        self.assertEqual(EEG_out['data'].shape[0], 2)
        if EEG_out['trials'] > 1:
            self.assertEqual(EEG_out['data'].shape[2], len(keep_trials))

        # Events, if present, must have valid latencies
        if EEG_out.get('event'):
            total_pts = EEG_out['pnts'] * EEG_out['trials']
            for ev in EEG_out['event']:
                if 'latency' in ev:
                    self.assertTrue(1 <= float(ev['latency']) <= total_pts)

    def test_time_then_point_precedence(self):
        # If both point and time are provided, point should take precedence for computing time
        srate = float(self.EEG['srate'])
        xmin = float(self.EEG['xmin'])
        # choose a 10-sample window via points
        pt_start = 1
        pt_end = min(self.EEG['pnts'], 10)
        EEG_out = pop_select(copy.deepcopy(self.EEG), point=[pt_start, pt_end], time=[[xmin, xmin + 1000.0]])  # absurd time to force precedence

        # Expect exactly pt_end - pt_start + 1 samples in output
        expected_pnts = pt_end - pt_start + 1
        self.assertEqual(EEG_out['pnts'], expected_pnts)
        # xmin should match eeg_point2lat mapping of pt_start
        # xmin_out equals old xmin plus (pt_start-1)/srate
        self.assertAlmostEqual(EEG_out['xmin'], xmin + (pt_start - 1) / srate, places=7)

    def test_nochannel_records_removed(self):
        labels = _chan_labels(self.EEG)
        if len(labels) < 4:
            self.skipTest("Need at least 4 channels to test removal bookkeeping")

        drop = labels[2:4]
        EEG_out = pop_select(copy.deepcopy(self.EEG), nochannel=drop)

        self.assertEqual(EEG_out['nbchan'], len(labels) - 2)
        # removedchans bookkeeping
        chaninfo = EEG_out.get('chaninfo', {})
        removed = chaninfo.get('removedchans', [])
        self.assertGreaterEqual(len(removed), 2)

    def test_event_epoch_field_removal_on_single_trial(self):
        EEG = copy.deepcopy(self.EEG)
        if int(EEG.get('trials')) > 2:
            EEG_out = pop_select(EEG, trial=[1,2])

            self.assertTrue(EEG_out.get('epoch') == [])


class TestPopSelectEdgeCases(unittest.TestCase):
    """Test edge cases and error conditions in pop_select."""
    
    def setUp(self):
        """Create a minimal test EEG structure."""
        self.EEG = {
            'data': np.random.randn(4, 100, 3),  # 4 channels, 100 points, 3 trials
            'nbchan': 4,
            'pnts': 100,
            'trials': 3,
            'srate': 250.0,
            'xmin': -0.2,
            'xmax': 0.2,
            'chanlocs': [
                {'labels': 'Fz', 'X': 0, 'Y': 1, 'Z': 0},
                {'labels': 'Cz', 'X': 0, 'Y': 0, 'Z': 1},
                {'labels': 'Pz', 'X': 0, 'Y': -1, 'Z': 0},
                {'labels': 'Oz', 'X': 0, 'Y': -1, 'Z': -1}
            ],
            'event': [
                {'type': 'stimulus', 'latency': 50, 'epoch': 1},
                {'type': 'response', 'latency': 150, 'epoch': 2},
                {'type': 'stimulus', 'latency': 250, 'epoch': 3}
            ],
            'epoch': [{}, {}, {}],
            'icaact': None,
            'icawinv': None,
            'icaweights': None,
            'icasphere': None,
            'icachansind': None,
            'specdata': None,
            'specicaact': None,
            'reject': {},
            'stats': {},
            'dipfit': None,
            'roi': None,
            'chaninfo': {}
        }
    
    def test_missing_data_error(self):
        """Test error when EEG data is missing."""
        EEG = copy.deepcopy(self.EEG)
        EEG['data'] = None
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG)
        self.assertIn('EEG["data"] is required', str(cm.exception))
    
    def test_wrong_trial_range_error(self):
        """Test error for invalid trial ranges."""
        EEG = copy.deepcopy(self.EEG)
        
        # Trial index too low
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, trial=[0, 1, 2])
        self.assertIn('Wrong trial range', str(cm.exception))
        
        # Trial index too high
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, trial=[1, 2, 5])
        self.assertIn('Wrong trial range', str(cm.exception))
    
    def test_empty_dataset_error(self):
        """Test error when all trials are removed."""
        EEG = copy.deepcopy(self.EEG)
        EEG['filename'] = 'test.set'
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, notrial=[1, 2, 3])
        self.assertIn('dataset test.set is empty', str(cm.exception))
    
    def test_channel_name_and_type_conflict_error(self):
        """Test error when both channel names and types are specified."""
        EEG = copy.deepcopy(self.EEG)
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, channel=['Fz'], chantype=['EEG'])
        self.assertIn('Select channels by name OR by type, not both', str(cm.exception))
    
    def test_empty_channel_selection_error(self):
        """Test error when all channels are removed."""
        EEG = copy.deepcopy(self.EEG)
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, nochannel=['Fz', 'Cz', 'Pz', 'Oz'])
        self.assertIn('Empty channel selection', str(cm.exception))
    
    def test_invalid_time_range_columns_error(self):
        """Test error for time/point ranges with wrong number of columns."""
        EEG = copy.deepcopy(self.EEG)
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, time=[[0.1, 0.2, 0.3]])  # 3 columns instead of 2
        self.assertIn('Time/point range must contain exactly 2 columns', str(cm.exception))
    
    def test_notime_epoched_boundary_error(self):
        """Test error for notime ranges that don't touch epoch boundaries."""
        EEG = copy.deepcopy(self.EEG)
        
        # notime range that doesn't touch boundaries
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, notime=[[-0.1, 0.1]])  # middle of epoch
        self.assertIn('Wrong notime range for epoched data; must include an epoch boundary', str(cm.exception))
    
    def test_multiple_time_windows_epoched_error(self):
        """Test error for multiple time windows in epoched data."""
        EEG = copy.deepcopy(self.EEG)
        
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, time=[[-0.2, -0.1], [0.1, 0.2]])
        self.assertIn('Epoched data requires a single [tmin tmax] window', str(cm.exception))
    
    def test_invalid_time_window_points_error(self):
        """Test error for invalid time window mapping to points."""
        EEG = copy.deepcopy(self.EEG)
        
        # Time window that maps to invalid points
        with self.assertRaises(ValueError) as cm:
            pop_select(EEG, time=[[0.3, 0.1]])  # end before start
        self.assertIn('Invalid time window mapped to points', str(cm.exception))
    
    def test_channel_selection_by_indices(self):
        """Test channel selection by integer indices."""
        EEG = copy.deepcopy(self.EEG)
        
        # Select channels by 0-based indices
        EEG_out = pop_select(EEG, channel=[0, 2])  # Fz and Pz
        
        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(len(EEG_out['chanlocs']), 2)
        self.assertEqual(EEG_out['chanlocs'][0]['labels'], 'Fz')
        self.assertEqual(EEG_out['chanlocs'][1]['labels'], 'Pz')
        self.assertEqual(EEG_out['data'].shape[0], 2)
    
    def test_channel_selection_by_labels(self):
        """Test channel selection by string labels."""
        EEG = copy.deepcopy(self.EEG)
        
        EEG_out = pop_select(EEG, channel=['Cz', 'Oz'])
        
        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(EEG_out['chanlocs'][0]['labels'], 'Cz')
        self.assertEqual(EEG_out['chanlocs'][1]['labels'], 'Oz')
        self.assertEqual(EEG_out['data'].shape[0], 2)
    
    def test_mixed_channel_indices_labels(self):
        """Test mixed channel selection (indices and labels)."""
        EEG = copy.deepcopy(self.EEG)
        
        # This should work - mix of int index and string label
        EEG_out = pop_select(EEG, channel=[0, 'Oz'])
        
        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(EEG_out['data'].shape[0], 2)
    
    def test_negative_indices_error(self):
        """Test that negative channel indices raise appropriate errors."""
        EEG = copy.deepcopy(self.EEG)
        
        # This should fail during eeg_decodechan call
        with self.assertRaises(Exception):  # Could be ValueError or IndexError
            pop_select(EEG, channel=[-1])
    
    def test_float_indices_error(self):
        """Test that float channel indices raise appropriate errors."""
        EEG = copy.deepcopy(self.EEG)
        
        # Float indices should cause issues
        with self.assertRaises(Exception):
            pop_select(EEG, channel=[1.5])
    
    def test_out_of_range_channel_indices(self):
        """Test out-of-range channel indices."""
        EEG = copy.deepcopy(self.EEG)
        
        # Channel index beyond available channels
        with self.assertRaises(Exception):
            pop_select(EEG, channel=[10])  # Only 4 channels available (0-3)
    
    def test_time_range_selection(self):
        """Test time range selection functionality."""
        EEG = copy.deepcopy(self.EEG)
        
        # Select middle portion of epoch
        EEG_out = pop_select(EEG, time=[[-0.1, 0.1]])
        
        # Should have fewer points (unless the window covers the whole epoch)
        # The time window [-0.1, 0.1] might cover the whole epoch [-0.2, 0.2]
        # Let's use a smaller window
        EEG_out2 = pop_select(copy.deepcopy(self.EEG), time=[[-0.05, 0.05]])
        self.assertLessEqual(EEG_out2['pnts'], EEG['pnts'])
        self.assertAlmostEqual(EEG_out2['xmin'], -0.05, places=6)
        self.assertEqual(EEG_out['trials'], EEG['trials'])
        self.assertEqual(EEG_out['nbchan'], EEG['nbchan'])
    
    def test_trial_selection_with_events(self):
        """Test trial selection updates events correctly."""
        EEG = copy.deepcopy(self.EEG)
        
        # Select first two trials
        EEG_out = pop_select(EEG, trial=[1, 2])
        
        self.assertEqual(EEG_out['trials'], 2)
        # Should have 2 events (from epochs 1 and 2)
        self.assertEqual(len(EEG_out['event']), 2)
        # Check epoch numbers are updated
        self.assertEqual(EEG_out['event'][0]['epoch'], 1)
        self.assertEqual(EEG_out['event'][1]['epoch'], 2)
    
    def test_trial_reordering_preserves_order(self):
        """Test that trial selection with sorttrial=off preserves order."""
        EEG = copy.deepcopy(self.EEG)
        
        # Select trials in non-sorted order
        EEG_out = pop_select(EEG, trial=[3, 1], sorttrial='off')
        
        self.assertEqual(EEG_out['trials'], 2)
        # Events should reflect the new trial mapping
        # When trials [3, 1] are selected, they become new trials [1, 2]
        # So original epoch 3 becomes new epoch 1, and original epoch 1 becomes new epoch 2
        if len(EEG_out['event']) >= 2:
            # Find events by their original type to verify mapping
            event_types = [ev['type'] for ev in EEG_out['event']]
            # The event from original epoch 3 should now be in epoch 1
            # The event from original epoch 1 should now be in epoch 2
            self.assertEqual(len(EEG_out['event']), 2)
    
    def test_continuous_data_handling(self):
        """Test handling of continuous (non-epoched) data."""
        EEG = copy.deepcopy(self.EEG)
        # Convert to continuous
        EEG['data'] = EEG['data'][:, :, 0]  # Take first trial as continuous
        EEG['trials'] = 1
        EEG['event'] = [{'type': 'stimulus', 'latency': 50}]  # Remove epoch field
        EEG['epoch'] = []
        
        # Select subset of channels
        EEG_out = pop_select(EEG, channel=[0, 1])
        
        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(EEG_out['trials'], 1)
        self.assertEqual(EEG_out['data'].shape, (2, 100))
    
    def test_empty_selection_parameters(self):
        """Test that empty selection parameters work correctly."""
        EEG = copy.deepcopy(self.EEG)
        
        # Empty lists should be ignored - but trial=[] means no trials, causing empty dataset error
        # So test with None instead
        EEG_out = pop_select(EEG, channel=None, time=[])
        
        # Should return original data unchanged
        self.assertEqual(EEG_out['nbchan'], EEG['nbchan'])
        self.assertEqual(EEG_out['trials'], EEG['trials'])
        self.assertEqual(EEG_out['pnts'], EEG['pnts'])
        np.testing.assert_array_equal(EEG_out['data'], EEG['data'])
    
    def test_metadata_updates(self):
        """Test that metadata is properly updated after selection."""
        EEG = copy.deepcopy(self.EEG)
        
        EEG_out = pop_select(EEG, channel=[0, 2], trial=[1, 3])
        
        # Check all metadata is updated
        self.assertEqual(EEG_out['nbchan'], 2)
        self.assertEqual(EEG_out['trials'], 2)
        self.assertEqual(EEG_out['data'].shape, (2, 100, 2))
        self.assertEqual(len(EEG_out['chanlocs']), 2)
        self.assertEqual(len(EEG_out['epoch']), 2)
        
        # Check reject and stats are reset - both have specific fields
        self.assertIn('rejmanual', EEG_out['reject'])
        self.assertIn('jp', EEG_out['stats'])
    
    def test_ica_data_handling(self):
        """Test proper handling of ICA-related data structures."""
        EEG = copy.deepcopy(self.EEG)
        
        # Add some ICA data
        EEG['icaact'] = np.random.randn(2, 100, 3)  # 2 components
        EEG['icawinv'] = np.random.randn(4, 2)
        EEG['icaweights'] = np.random.randn(2, 4)
        EEG['icasphere'] = np.eye(4)
        EEG['icachansind'] = [0, 1, 2, 3]
        
        # Select subset of channels
        EEG_out = pop_select(EEG, channel=[0, 2])
        
        # ICA structures should be updated
        if EEG_out['icaact'] is not None:
            self.assertEqual(EEG_out['icaact'].shape[2], 3)  # Same trials
        
        # icachansind should be updated to new channel indices
        if EEG_out['icachansind'] is not None:
            self.assertEqual(len(EEG_out['icachansind']), 2)
    
    def test_dipfit_removal_warning(self):
        """Test that dipfit is removed when channels are removed."""
        EEG = copy.deepcopy(self.EEG)
        EEG['dipfit'] = [{'some': 'dipole_info'}]
        EEG['roi'] = [{'some': 'roi_info'}]
        
        # Remove one channel
        EEG_out = pop_select(EEG, channel=[0, 1, 2])  # Remove channel 3
        
        # dipfit and roi should be cleared
        self.assertEqual(EEG_out['dipfit'], [])
        self.assertEqual(EEG_out['roi'], [])
    
    def test_single_trial_epoch_field_removal(self):
        """Test that epoch fields are removed from events when only one trial remains."""
        EEG = copy.deepcopy(self.EEG)
        
        # Select only first trial
        EEG_out = pop_select(EEG, trial=[1])
        
        self.assertEqual(EEG_out['trials'], 1)
        # epoch field should be removed from events
        for event in EEG_out['event']:
            self.assertNotIn('epoch', event)
        # epoch list should be empty
        self.assertEqual(EEG_out['epoch'], [])
    
    def test_event_latency_bounds_checking(self):
        """Test that events with invalid latencies are removed."""
        EEG = copy.deepcopy(self.EEG)
        
        # Add an event with invalid latency
        EEG['event'].append({'type': 'invalid', 'latency': 1000, 'epoch': 1})  # Beyond data
        
        EEG_out = pop_select(EEG, trial=[1])
        
        # Invalid event should be removed
        latencies = [ev['latency'] for ev in EEG_out['event'] if 'latency' in ev]
        total_pts = EEG_out['pnts'] * EEG_out['trials']
        for lat in latencies:
            self.assertTrue(1 <= lat <= total_pts)
    
    def test_vector_format_deprecation_warning(self):
        """Test that vector format for point/time ranges shows deprecation warning."""
        EEG = copy.deepcopy(self.EEG)
        
        # This should trigger the deprecation warning for vector format
        # We can't easily test the print statement, but we can verify it works
        EEG_out = pop_select(EEG, point=[1, 5, 10, 20, 30])  # Vector with >2 elements
        
        # Should select from first to last point
        expected_pnts = 30 - 1 + 1  # From point 1 to 30
        self.assertEqual(EEG_out['pnts'], expected_pnts)


if __name__ == '__main__':
    unittest.main()