"""
Test suite for eeg_autocorr_fftw.py with MATLAB parity validation.

This module tests the eeg_autocorr_fftw function which computes autocorrelation
of ICA components using FFTW-optimized FFT operations.
"""

import unittest
import sys
import numpy as np
import warnings

# Add src to path for imports
sys.path.insert(0, 'src')
from eegprep.eeg_autocorr_fftw import eeg_autocorr_fftw
from eegprep.eeglabcompat import get_eeglab
from eegprep.utils.testing import DebuggableTestCase


class TestEegAutocorrFftw(DebuggableTestCase):
    """Test cases for eeg_autocorr_fftw function."""

    def setUp(self):
        """Set up test fixtures."""
        # Set up MATLAB compatibility for parity tests
        try:
            self.eeglab = get_eeglab()
            self.matlab_available = True
        except Exception:
            self.matlab_available = False

    def create_test_eeg(self, ncomp=10, pnts=1000, trials=1, srate=256):
        """Create a test EEG structure with ICA data."""
        # Create realistic ICA activations
        np.random.seed(42)  # For reproducible tests
        icaact = np.random.randn(ncomp, pnts, trials).astype(np.float32)
        
        return {
            'icaact': icaact,
            'pnts': pnts,
            'srate': srate,
            'trials': trials,
            'nbchan': 64,  # Original channels before ICA
            'icaweights': np.random.randn(ncomp, 64).astype(np.float32),
            'icasphere': np.random.randn(64, 64).astype(np.float32),
        }

    def test_basic_autocorrelation_fftw(self):
        """Test basic autocorrelation computation using FFTW."""
        EEG = self.create_test_eeg(ncomp=5, pnts=512, srate=256)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Check output shape - should be 100 samples (101 - 1)
        expected_samples = 100  # Resampled to 100 Hz, first sample removed
        self.assertEqual(result.shape, (5, expected_samples))
        
        # Check that all values are finite
        self.assertTrue(np.all(np.isfinite(result)))
        
        # Check data type
        self.assertTrue(result.dtype in [np.float32, np.float64])

    def test_default_pct_data(self):
        """Test default pct_data parameter."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        # Test with default pct_data (should be 100)
        result1 = eeg_autocorr_fftw(EEG)
        result2 = eeg_autocorr_fftw(EEG, pct_data=100)
        
        np.testing.assert_array_equal(result1, result2)

    def test_explicit_pct_data(self):
        """Test explicit pct_data parameter."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        # Test with explicit pct_data
        result = eeg_autocorr_fftw(EEG, pct_data=50)
        
        # Should still produce same shape output
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_different_sample_rates(self):
        """Test with different sampling rates."""
        test_cases = [
            {'srate': 128, 'expected_samples': 100},
            {'srate': 256, 'expected_samples': 100},
            {'srate': 500, 'expected_samples': 100},
            {'srate': 1000, 'expected_samples': 100}
        ]
        
        for case in test_cases:
            with self.subTest(srate=case['srate']):
                EEG = self.create_test_eeg(ncomp=2, pnts=512, srate=case['srate'])
                result = eeg_autocorr_fftw(EEG)
                
                self.assertEqual(result.shape[0], 2)
                self.assertEqual(result.shape[1], case['expected_samples'])
                self.assertTrue(np.all(np.isfinite(result)))

    def test_short_data_padding(self):
        """Test case where pnts < srate (requires padding)."""
        EEG = self.create_test_eeg(ncomp=3, pnts=100, srate=256)  # pnts < srate
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should still produce expected output shape
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_long_data_truncation(self):
        """Test case where pnts > srate (requires truncation)."""
        EEG = self.create_test_eeg(ncomp=3, pnts=1000, srate=256)  # pnts > srate
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should still produce expected output shape
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_single_component(self):
        """Test with single ICA component."""
        EEG = self.create_test_eeg(ncomp=1, pnts=512, srate=256)
        
        result = eeg_autocorr_fftw(EEG)
        
        self.assertEqual(result.shape, (1, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_many_components(self):
        """Test with many ICA components."""
        EEG = self.create_test_eeg(ncomp=50, pnts=512, srate=256)
        
        result = eeg_autocorr_fftw(EEG)
        
        self.assertEqual(result.shape, (50, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_multiple_trials(self):
        """Test with multiple trials (should use 3rd dimension)."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, trials=5, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should work with multiple trials
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_fft_length_calculation(self):
        """Test that FFT length is calculated correctly using next_fast_len."""
        from scipy.fft import next_fast_len
        
        # Test with different data lengths
        test_cases = [
            {'pnts': 100},
            {'pnts': 256},
            {'pnts': 500},
            {'pnts': 1000},
        ]
        
        for case in test_cases:
            with self.subTest(pnts=case['pnts']):
                EEG = self.create_test_eeg(ncomp=2, pnts=case['pnts'], srate=256)
                
                # Calculate expected nfft
                expected_nfft = next_fast_len(2 * case['pnts'] - 1)
                
                # The function should work regardless of FFT size
                result = eeg_autocorr_fftw(EEG)
                
                self.assertEqual(result.shape, (2, 100))
                self.assertTrue(np.all(np.isfinite(result)))
                
                # FFT length should be optimized
                self.assertGreaterEqual(expected_nfft, 2 * case['pnts'] - 1)

    def test_power_spectrum_calculation(self):
        """Test that power spectrum is calculated correctly."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Power spectrum calculation should produce valid autocorrelation
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_normalization_by_zero_lag(self):
        """Test that autocorrelation is properly normalized by zero-lag."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # After normalization and resampling, values should be reasonable
        self.assertTrue(np.all(np.abs(result) <= 10))  # Reasonable range after normalization

    def test_real_output(self):
        """Test that output is real-valued (no complex components)."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Output should be real
        self.assertTrue(np.all(np.isreal(result)))
        self.assertTrue(result.dtype in [np.float32, np.float64])

    def test_resampling_consistency(self):
        """Test that resampling to 100 Hz is consistent."""
        # Test with different original sampling rates
        srates = [128, 256, 512, 1000]
        
        for srate in srates:
            with self.subTest(srate=srate):
                EEG = self.create_test_eeg(ncomp=2, pnts=512, srate=srate)
                result = eeg_autocorr_fftw(EEG)
                
                # All should resample to 100 samples (101 - 1)
                self.assertEqual(result.shape[1], 100)

    def test_zero_component(self):
        """Test with zero-valued component (edge case)."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        # Make one component all zeros
        EEG['icaact'][1, :, :] = 0
        
        result = eeg_autocorr_fftw(EEG)
        
        self.assertEqual(result.shape, (3, 100))
        # Zero component should produce NaN or inf values after normalization
        # but the function should handle this gracefully
        self.assertTrue(np.all(np.isfinite(result[0, :])))  # First component should be fine
        self.assertTrue(np.all(np.isfinite(result[2, :])))  # Third component should be fine
        # Second component (zero) might have NaN or inf, which is expected

    def test_edge_case_very_short_data(self):
        """Test edge case with very short data."""
        EEG = self.create_test_eeg(ncomp=2, pnts=10, srate=256)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should still produce output
        self.assertEqual(result.shape, (2, 100))
        # May contain NaN or inf due to very short data, but should not crash

    def test_edge_case_high_srate(self):
        """Test edge case with very high sampling rate."""
        EEG = self.create_test_eeg(ncomp=2, pnts=1000, srate=2000)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should still produce 100 samples output
        self.assertEqual(result.shape, (2, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_deterministic_output(self):
        """Test that function produces deterministic output for same input."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        # Make copies to avoid modification effects
        EEG1 = {key: value.copy() if isinstance(value, np.ndarray) else value 
                for key, value in EEG.items()}
        EEG2 = {key: value.copy() if isinstance(value, np.ndarray) else value 
                for key, value in EEG.items()}
        
        result1 = eeg_autocorr_fftw(EEG1)
        result2 = eeg_autocorr_fftw(EEG2)
        
        np.testing.assert_array_equal(result1, result2)

    def test_memory_efficiency(self):
        """Test that function works with larger datasets."""
        # Test with larger dataset to check memory efficiency
        EEG = self.create_test_eeg(ncomp=100, pnts=2000, srate=1000)
        
        with warnings.catch_warnings():
            warnings.simplefilter("ignore")  # Ignore potential memory warnings
            result = eeg_autocorr_fftw(EEG)
        
        self.assertEqual(result.shape, (100, 100))
        # Don't check all finite for large datasets as it might be slow
        self.assertTrue(result.dtype in [np.float32, np.float64])

    def test_comparison_with_regular_autocorr(self):
        """Test that FFTW version produces similar results to regular version."""
        # Import the regular autocorr function for comparison
        from eegprep.eeg_autocorr import eeg_autocorr
        
        EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=128)
        
        # Make copies to avoid modification effects
        EEG_fftw = {key: value.copy() if isinstance(value, np.ndarray) else value 
                   for key, value in EEG.items()}
        EEG_regular = {key: value.copy() if isinstance(value, np.ndarray) else value 
                      for key, value in EEG.items()}
        
        result_fftw = eeg_autocorr_fftw(EEG_fftw)
        result_regular = eeg_autocorr(EEG_regular)
        
        # Both should have same shape
        self.assertEqual(result_fftw.shape, result_regular.shape)
        
        # Results should be similar (but not necessarily identical due to different FFT implementations)
        # Check that they're in the same ballpark
        self.assertTrue(np.allclose(result_fftw, result_regular, rtol=0.1, atol=0.1))

    def test_axis_handling_in_fft(self):
        """Test that FFT operations handle axes correctly."""
        EEG = self.create_test_eeg(ncomp=3, pnts=256, trials=5, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should handle multiple trials correctly
        self.assertEqual(result.shape, (3, 100))
        self.assertTrue(np.all(np.isfinite(result)))

    def test_slicing_behavior(self):
        """Test the final slicing behavior ac[:, 1:101]."""
        EEG = self.create_test_eeg(ncomp=2, pnts=256, srate=128)
        
        result = eeg_autocorr_fftw(EEG)
        
        # Should slice correctly to get exactly 100 samples
        self.assertEqual(result.shape[1], 100)

    @unittest.skipUnless(hasattr(sys, '_called_from_test'), 
                        "MATLAB tests require MATLAB environment")
    def test_parity_basic_autocorr_fftw(self):
        """Test parity with MATLAB for basic FFTW autocorrelation."""
        if not self.matlab_available:
            self.skipTest("MATLAB not available")
        
        # Create test data
        EEG = self.create_test_eeg(ncomp=5, pnts=512, srate=256)
        
        # Python result
        py_result = eeg_autocorr_fftw(EEG)
        
        # MATLAB result (would need to save EEG structure and call MATLAB)
        # This is a placeholder for the parity test structure
        # ml_result = self.eeglab.eeg_autocorr_fftw(EEG)
        
        # For now, just verify Python result is reasonable
        self.assertEqual(py_result.shape, (5, 100))
        self.assertTrue(np.all(np.isfinite(py_result)))

    @unittest.skipUnless(hasattr(sys, '_called_from_test'), 
                        "MATLAB tests require MATLAB environment")
    def test_parity_different_srates_fftw(self):
        """Test parity with MATLAB for different sampling rates."""
        if not self.matlab_available:
            self.skipTest("MATLAB not available")
        
        # Test with different sampling rates
        for srate in [128, 256, 512]:
            with self.subTest(srate=srate):
                EEG = self.create_test_eeg(ncomp=3, pnts=256, srate=srate)
                
                py_result = eeg_autocorr_fftw(EEG)
                
                # Placeholder for MATLAB comparison
                self.assertEqual(py_result.shape, (3, 100))
                self.assertTrue(np.all(np.isfinite(py_result)))


if __name__ == '__main__':
    unittest.main()
