"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const cdk_nag_1 = require("cdk-nag");
const sonar_code_scanner_1 = require("./code_scanner/sonar-code-scanner");
__exportStar(require("./code_scanner/sonar-code-scanner"), exports);
const DEFAULT_BRANCH_NAME = "mainline";
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
class PDKPipeline extends pipelines_1.CodePipeline {
    constructor(scope, id, props) {
        const codeRepository = new aws_codecommit_1.Repository(scope, "CodeRepository", {
            repositoryName: props.repositoryName,
        });
        codeRepository.applyRemovalPolicy(props.codeCommitRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN);
        const artifactBucket = new aws_s3_1.Bucket(scope, "ArtifactsBucket", {
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            serverAccessLogsPrefix: "access-logs",
        });
        const codePipeline = new aws_codepipeline_1.Pipeline(scope, "CodePipeline", {
            enableKeyRotation: true,
            restartExecutionOnUpdate: true,
            crossAccountKeys: props.crossAccountKeys,
            artifactBucket,
        });
        // ignore input and primaryOutputDirectory
        const { input, primaryOutputDirectory, commands, ...synthShellStepPartialProps } = props.synthShellStepPartialProps || {};
        const synthShellStep = new pipelines_1.ShellStep("Synth", {
            input: pipelines_1.CodePipelineSource.codeCommit(codeRepository, props.defaultBranchName || DEFAULT_BRANCH_NAME),
            installCommands: [
                "npm install -g aws-cdk",
                "yarn install --frozen-lockfile || npx projen && yarn install --frozen-lockfile",
            ],
            commands: commands && commands.length > 0
                ? commands
                : ["npx nx run-many --target=build --all"],
            primaryOutputDirectory: props.primarySynthDirectory,
            ...(synthShellStepPartialProps || {}),
        });
        synthShellStep.addOutputDirectory(".");
        const codePipelineProps = {
            codePipeline,
            ...props,
            crossAccountKeys: undefined,
            synth: synthShellStep,
        };
        super(scope, id, codePipelineProps);
        this.id = id;
        this.codeRepository = codeRepository;
        this.sonarCodeScannerConfig = props.sonarCodeScannerConfig
            ? {
                cdkOutDir: props.primarySynthDirectory,
                ...props.sonarCodeScannerConfig,
            }
            : undefined;
        new aws_cdk_lib_1.CfnOutput(scope, "CodeRepositoryGRCUrl", {
            exportName: "CodeRepositoryGRCUrl",
            value: this.codeRepository.repositoryCloneUrlGrc,
        });
    }
    /**
     * @inheritDoc
     */
    addStage(stage, options) {
        // Add any root Aspects to the stage level as currently this doesn't happen automatically
        aws_cdk_lib_1.Aspects.of(stage.node.root).all.forEach((aspect) => aws_cdk_lib_1.Aspects.of(stage).add(aspect));
        return super.addStage(stage, options);
    }
    buildPipeline() {
        super.buildPipeline();
        this.sonarCodeScannerConfig &&
            new sonar_code_scanner_1.SonarCodeScanner(this, "SonarCodeScanner", {
                artifactBucketArn: this.pipeline.artifactBucket.bucketArn,
                artifactBucketKeyArn: this.pipeline.artifactBucket.encryptionKey?.keyArn,
                synthBuildArn: this.synthProject.projectArn,
                ...this.sonarCodeScannerConfig,
            });
        this.suppressCDKViolations();
    }
    suppressCDKViolations() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Role/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                appliesTo: [
                    {
                        regex: "/^Action::s3:.*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                appliesTo: [
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                    },
                ],
            },
        ]);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Source/CodeCommit/CodePipelineActionRole/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                appliesTo: [
                    {
                        regex: "/^Action::s3:.*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                appliesTo: [
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                    },
                ],
            },
        ]);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Build/Synth/CdkBuildProject/Role/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                appliesTo: [
                    {
                        regex: "/^Action::s3:.*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                appliesTo: [
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<CodePipelineBuildSynthCdkBuildProject.*>:\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<CodePipelineBuildSynthCdkBuildProject.*>-\\*$/g`,
                    },
                ],
            },
        ]);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/UpdatePipeline/SelfMutation/Role/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<${this.id}UpdatePipelineSelfMutation.*>:\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<${this.id}UpdatePipelineSelfMutation.*>-\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to assume a role from within the current account in order to deploy.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:\\*:iam::${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:role/\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                appliesTo: [
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to list all buckets and stacks.",
                appliesTo: [
                    {
                        regex: "/^Resource::\\*$/g",
                    },
                ],
            },
        ]);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/UpdatePipeline/SelfMutation/Role/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                appliesTo: [
                    {
                        regex: "/^Action::s3:.*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:(<AWS::AccountId>|${stack.account}):log-group:/aws/codebuild/<${this.id}UpdatePipelineSelfMutation.*>:\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<${this.id}UpdatePipelineSelfMutation.*>-\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to assume a role from within the current account in order to deploy.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:\\*:iam::${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:role/\\*$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                appliesTo: [
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "CodePipeline requires access to list all buckets and stacks.",
                appliesTo: [
                    {
                        regex: "/^Resource::\\*$/g",
                    },
                ],
            },
        ]);
        pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/Assets/FileRole/DefaultPolicy/Resource`, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Asset role requires access to the Artifacts Bucket",
            },
        ]);
        cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
            {
                id: "AwsSolutions-CB4",
                reason: "Encryption of Codebuild is not required.",
            },
        ]);
    }
}
exports.PDKPipeline = PDKPipeline;
_a = JSII_RTTI_SYMBOL_1;
PDKPipeline[_a] = { fqn: "@aws-prototyping-sdk/pipeline.PDKPipeline", version: "0.11.9" };
//# sourceMappingURL=data:application/json;base64,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