import { containsGeneratedCodeOfAnalysis, containsMitosheetCallWithAnyAnalysisToReplay, containsMitosheetCallWithSpecificAnalysisToReplay, getArgsFromMitosheetCallCode, getCodeString, isMitosheetCallCode } from "../../utils/code";
export function getCellAtIndex(index) {
    var _a, _b;
    return (_b = (_a = window.Jupyter) === null || _a === void 0 ? void 0 : _a.notebook) === null || _b === void 0 ? void 0 : _b.get_cell(index);
}
export function getCellText(cell) {
    return (cell === null || cell === void 0 ? void 0 : cell.get_text()) || '';
}
/*
    Returns True if the passed cell is empty.
    Returns False if the passed cells is either not empty or undefined
*/
export function isEmptyCell(cell) {
    if (cell === undefined) {
        return false;
    }
    const currentCode = getCellText(cell);
    return currentCode.trim() === '';
}
/**
 * Returns the cell that has the mitosheet.sheet(analysis_to_replay={analysisName}) in it,
 * or undefined if no such cell exists
 */
export function getCellCallingMitoshetWithAnalysis(analysisName) {
    var _a, _b;
    const cells = (_b = (_a = window.Jupyter) === null || _a === void 0 ? void 0 : _a.notebook) === null || _b === void 0 ? void 0 : _b.get_cells();
    if (cells === undefined) {
        return undefined;
    }
    let cellIndex = 0;
    for (const cell of cells) {
        if (containsMitosheetCallWithSpecificAnalysisToReplay(getCellText(cell), analysisName)) {
            return [cell, cellIndex];
        }
        cellIndex++;
    }
    return undefined;
}
/**
 * A function that returns the [cell, index] pair of the mitosheet.sheet() call that contains
 * the analysis name.
 *
 * If no mitosheet.sheet() call contains this analysis name, then we assume it hasen't been
 * written yet, and take our best guess at which sheet this is.
 *
 * Returns undefined if it can find no good guess for a calling mitosheet cell.
 */
export function getMostLikelyMitosheetCallingCell(analysisName) {
    var _a, _b, _c, _d, _e, _f, _g, _h;
    // First, we check if this analysis name is in a mitosheet call, in which case things are easy
    if (analysisName) {
        const mitosheetCallCellAndIndex = getCellCallingMitoshetWithAnalysis(analysisName);
        if (mitosheetCallCellAndIndex !== undefined) {
            return mitosheetCallCellAndIndex;
        }
    }
    const cells = (_b = (_a = window.Jupyter) === null || _a === void 0 ? void 0 : _a.notebook) === null || _b === void 0 ? void 0 : _b.get_cells();
    if (cells == undefined) {
        return;
    }
    const activeCell = (_d = (_c = window.Jupyter) === null || _c === void 0 ? void 0 : _c.notebook) === null || _d === void 0 ? void 0 : _d.get_cell((_f = (_e = window.Jupyter) === null || _e === void 0 ? void 0 : _e.notebook) === null || _f === void 0 ? void 0 : _f.get_anchor_index());
    const activeCellIndex = ((_h = (_g = window.Jupyter) === null || _g === void 0 ? void 0 : _g.notebook) === null || _h === void 0 ? void 0 : _h.get_anchor_index()) || 0;
    const previousCell = getCellAtIndex(activeCellIndex - 1);
    // As the most common way for a user to run a cell for the first time is to run and advanced, this 
    // means that the active cell will most likely be one below the mitosheet.sheet() call we want to 
    // write to, so we check this first
    if (previousCell && isMitosheetCallCode(getCellText(previousCell)) && !containsMitosheetCallWithAnyAnalysisToReplay(getCellText(previousCell))) {
        return [previousCell, activeCellIndex - 1];
    }
    // The next case we check is if they did a run and not advance, which means that the currently
    // selected cell is the mitosheet.sheet call
    if (activeCell && isMitosheetCallCode(getCellText(activeCell)) && !containsMitosheetCallWithAnyAnalysisToReplay(getCellText(activeCell))) {
        return [activeCell, activeCellIndex];
    }
    // The last case is that the user did some sort of run all, in which case we cross our fingers
    // that there is only one cell that does not have a mitosheet call with an analysis_to_replay, 
    // and go looking for it
    let index = activeCellIndex;
    while (index >= 0) {
        const cell = getCellAtIndex(index);
        if (cell && isMitosheetCallCode(getCellText(cell)) && !containsMitosheetCallWithAnyAnalysisToReplay(getCellText(cell))) {
            return [cell, index];
        }
        index--;
    }
    return undefined;
}
export function writeToCell(cell, code) {
    if (cell == undefined) {
        return;
    }
    cell.set_text(code);
}
/**
 * Given a cell, will check if it has a mitosheet.sheet() call with the old
 * analysis to replay, and if so will replace it with the new analysis to
 * replay
 */
export function tryOverwriteAnalysisToReplayParameter(cell, oldAnalysisName, newAnalysisName) {
    if (isMitosheetCallCode(getCellText(cell)) && containsMitosheetCallWithSpecificAnalysisToReplay(getCellText(cell), oldAnalysisName)) {
        const currentCode = getCellText(cell);
        const newCode = currentCode.replace(`analysis_to_replay="${oldAnalysisName}")`, `analysis_to_replay="${newAnalysisName}")`);
        writeToCell(cell, newCode);
        return true;
    }
    return false;
}
/**
 * Given a cell, will check if it has a mitosheet.sheet() call with no
 * analysis_to_replay, and if so add the analysisName as a parameter to
 * this cell. It will return true in this case.
 *
 * Otherwise, if this is not a mitosheet.sheet() call, or if it already has
 * a analysis_to_replay parameter, this will return false.
 */
export function tryWriteAnalysisToReplayParameter(cell, analysisName) {
    if (isMitosheetCallCode(getCellText(cell)) && !containsMitosheetCallWithAnyAnalysisToReplay(getCellText(cell))) {
        const currentCode = getCellText(cell);
        // We know the mitosheet.sheet() call is the last thing in the cell, so we 
        // just replace the last closing paren
        const lastIndex = currentCode.lastIndexOf(')');
        let replacement = ``;
        if (currentCode.includes('sheet()')) {
            replacement = `analysis_to_replay="${analysisName}")`;
        }
        else {
            replacement = `, analysis_to_replay="${analysisName}")`;
        }
        const newCode = currentCode.substring(0, lastIndex) + replacement + currentCode.substring(lastIndex + 1);
        writeToCell(cell, newCode);
        return true;
    }
    return false;
}
export const notebookGetArgs = (analysisToReplayName) => {
    const cellAndIndex = getMostLikelyMitosheetCallingCell(analysisToReplayName);
    if (cellAndIndex) {
        const [cell,] = cellAndIndex;
        return getArgsFromMitosheetCallCode(getCellText(cell));
    }
    else {
        return [];
    }
};
export const notebookWriteAnalysisToReplayToMitosheetCall = (analysisName, mitoAPI) => {
    const cellAndIndex = getMostLikelyMitosheetCallingCell(analysisName);
    if (cellAndIndex) {
        const [cell,] = cellAndIndex;
        const written = tryWriteAnalysisToReplayParameter(cell, analysisName);
        if (written) {
            return;
        }
    }
    // Log if we are unable to write this param for any reason
    void mitoAPI.log('write_analysis_to_replay_to_mitosheet_call_failed');
};
export const notebookOverwriteAnalysisToReplayToMitosheetCall = (oldAnalysisName, newAnalysisName, mitoAPI) => {
    const mitosheetCallCellAndIndex = getCellCallingMitoshetWithAnalysis(oldAnalysisName);
    if (mitosheetCallCellAndIndex === undefined) {
        return;
    }
    const [mitosheetCallCell,] = mitosheetCallCellAndIndex;
    const overwritten = tryOverwriteAnalysisToReplayParameter(mitosheetCallCell, oldAnalysisName, newAnalysisName);
    if (!overwritten) {
        void mitoAPI.log('overwrite_analysis_to_replay_to_mitosheet_call_failed');
    }
};
export const notebookWriteGeneratedCodeToCell = (analysisName, codeLines, telemetryEnabled) => {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
    const code = getCodeString(analysisName, codeLines, telemetryEnabled);
    // Find the cell that made the mitosheet.sheet call, and if it does not exist, give
    // up immediately
    const mitosheetCallCellAndIndex = getCellCallingMitoshetWithAnalysis(analysisName);
    if (mitosheetCallCellAndIndex === undefined) {
        return;
    }
    const [, mitosheetCallIndex] = mitosheetCallCellAndIndex;
    const cells = (_b = (_a = window.Jupyter) === null || _a === void 0 ? void 0 : _a.notebook) === null || _b === void 0 ? void 0 : _b.get_cells();
    if (cells === undefined) {
        return;
    }
    const activeCellIndex = ((_d = (_c = window.Jupyter) === null || _c === void 0 ? void 0 : _c.notebook) === null || _d === void 0 ? void 0 : _d.get_anchor_index()) || 0;
    const codeCell = getCellAtIndex(mitosheetCallIndex + 1);
    if (isEmptyCell(codeCell) || containsGeneratedCodeOfAnalysis(getCellText(codeCell), analysisName)) {
        writeToCell(codeCell, code);
    }
    else {
        // If we cannot write to the cell below, we have to go back a new cell below, 
        // which can eb a bit of an involve process
        if (mitosheetCallIndex !== activeCellIndex) {
            // We have to move our selection back up to the cell that we 
            // make the mitosheet call to 
            if (mitosheetCallIndex < activeCellIndex) {
                for (let i = 0; i < (activeCellIndex - mitosheetCallIndex); i++) {
                    (_f = (_e = window.Jupyter) === null || _e === void 0 ? void 0 : _e.notebook) === null || _f === void 0 ? void 0 : _f.select_prev();
                }
            }
            else if (mitosheetCallIndex > activeCellIndex) {
                for (let i = 0; i < (mitosheetCallIndex - activeCellIndex); i++) {
                    (_h = (_g = window.Jupyter) === null || _g === void 0 ? void 0 : _g.notebook) === null || _h === void 0 ? void 0 : _h.select_next();
                }
            }
        }
        // And then write to this new cell below, which is not the active cell but we
        // should make it the actice cell
        (_k = (_j = window.Jupyter) === null || _j === void 0 ? void 0 : _j.notebook) === null || _k === void 0 ? void 0 : _k.insert_cell_below();
        (_m = (_l = window.Jupyter) === null || _l === void 0 ? void 0 : _l.notebook) === null || _m === void 0 ? void 0 : _m.select_next();
        const activeCell = (_p = (_o = window.Jupyter) === null || _o === void 0 ? void 0 : _o.notebook) === null || _p === void 0 ? void 0 : _p.get_cell((_r = (_q = window.Jupyter) === null || _q === void 0 ? void 0 : _q.notebook) === null || _r === void 0 ? void 0 : _r.get_anchor_index());
        writeToCell(activeCell, code);
    }
};
//# sourceMappingURL=pluginUtils.js.map