// Copyright (c) Mito
import React, { Fragment, useRef, useState } from 'react';
import MultiToggleBox from '../../../elements/MultiToggleBox';
import Select from '../../../elements/Select';
import Col from '../../../layout/Col';
import Row from '../../../layout/Row';
import { areFiltersEqual, getAllDoesNotContainsFilterValues, getExclusiveFilterData, getFilterDisabledMessage } from '../FilterAndSortTab/filter/utils';
import MultiToggleItem from '../../../elements/MultiToggleItem';
import DropdownItem from '../../../elements/DropdownItem';
import { useDebouncedEffect } from '../../../../hooks/useDebouncedEffect';
import { isFilterGroup } from '../FilterAndSortTab/filter/filterTypes';
import { formatCellData } from '../../../../utils/formatColumns';
import OpenFillNaN from '../../FillNa/OpenFillNaN';
export var UniqueValueSortType;
(function (UniqueValueSortType) {
    UniqueValueSortType["ASCENDING_ALPHABETICAL"] = "Ascending Value";
    UniqueValueSortType["DESCENDING_ALPHABETICAL"] = "Descending Value";
    UniqueValueSortType["ASCENDING_PERCENT_OCCURENCE"] = "Ascending Occurence";
    UniqueValueSortType["DESCENDING_PERCENT_OCCURENCE"] = "Descending Occurence";
})(UniqueValueSortType || (UniqueValueSortType = {}));
const sortUniqueValueCounts = (uniqueValueCounts, uniqueValueSortType) => {
    if (uniqueValueSortType === UniqueValueSortType.ASCENDING_ALPHABETICAL) {
        return uniqueValueCounts.sort(function (a, b) {
            if (a.value < b.value) {
                return -1;
            }
            if (a.value > b.value) {
                return 1;
            }
            return 0;
        });
    }
    else if (uniqueValueSortType === UniqueValueSortType.DESCENDING_ALPHABETICAL) {
        return uniqueValueCounts.sort(function (a, b) {
            if (a.value > b.value) {
                return -1;
            }
            if (a.value < b.value) {
                return 1;
            }
            return 0;
        });
    }
    else if (uniqueValueSortType === UniqueValueSortType.ASCENDING_PERCENT_OCCURENCE) {
        return uniqueValueCounts.sort(function (a, b) {
            return a.percentOccurence - b.percentOccurence;
        });
    }
    else {
        return uniqueValueCounts.sort(function (a, b) {
            return b.percentOccurence - a.percentOccurence;
        });
    }
};
export function ValuesTab(props) {
    const [loading, setLoading] = useState(true);
    const [isAllData, setIsAllData] = useState(false); // Start false, so we go get the data in the start
    const [uniqueValueCounts, setUniqueValueCounts] = useState([]);
    const [searchString, setSearchString] = useState('');
    const [sort, setSort] = useState(UniqueValueSortType.ASCENDING_ALPHABETICAL);
    /**
     * In the past, we used to send all the unique values to the front-end
     * at once, but for large data-sets this pretty much crashed the page.
     *
     * Now, we only send the first 1000 values (with that sort, and search),
     * and then let the front-end further process them. Note that this means
     * that we sort and filter and both the front-end and backend, to give
     * the user the most responsive possible experience.
     *
     * We reload data from the backend under the following conditions:
     * 1. We do not have all the data, and the search or sort changes
     * 2. We do have all the data, and the search is made more inclusive
     */
    const lastSearchTerm = useRef('so it rerenders the first time');
    const lastSort = useRef(sort);
    useDebouncedEffect(() => {
        if (!isAllData ||
            (isAllData &&
                ((!searchString.startsWith(lastSearchTerm.current)) || (searchString.length < lastSearchTerm.current.length)))) {
            void loadUniqueValueCounts();
        }
        lastSearchTerm.current = searchString;
        lastSort.current = sort;
    }, [searchString, sort], 500);
    async function loadUniqueValueCounts() {
        setLoading(true);
        const _uniqueValueObj = await props.mitoAPI.getUniqueValueCounts(props.selectedSheetIndex, props.columnID, searchString, sort);
        if (_uniqueValueObj !== undefined) {
            const _uniqueValueObjs = _uniqueValueObj.uniqueValueCounts;
            /*
                Add back all of the values that were filtered out of the column, so the user can toggle
                them back on. Note that this lets users toggle them back on even if they were removed in a previous step!
            */
            const allDoesNotContainsFilters = getAllDoesNotContainsFilterValues(props.filters, props.columnDtype);
            allDoesNotContainsFilters.forEach(key => {
                _uniqueValueObjs.push({
                    value: key,
                    percentOccurence: 0,
                    countOccurence: 0,
                    isNotFiltered: false,
                });
            });
            setUniqueValueCounts(_uniqueValueObjs);
            setIsAllData(_uniqueValueObj.isAllData);
        }
        else {
            setUniqueValueCounts([]);
        }
        setLoading(false);
    }
    /*
        Helper function for getting the index of the UniqueValueCount in UniqueValueCounts from the index
        of the UniqueValueCount in the searchedUniqueValueCounts. It exploits the invariant that
        these are _unique value_ counts.
    */
    const getUniqueValueCountIndexFromSortedIndex = (index) => {
        const value = sortedUniqueValueCounts[index].value;
        return uniqueValueCounts.findIndex(uniqueValueCount => {
            return uniqueValueCount.value === value;
        });
    };
    /*
        Toggles the exclusive filter for a specific value. An exclusive filter is the NOT_EXACTLY filter condition
        for strings, numbers, and dates. For booleans its either the IS_TRUE or IS_FALSE conditions.

        For a specific value in this column, the toggleExclusiveFilter determines if there is an exclusive filter that is excluding the value.
        If there is, it removes it. If there's is not, it applies it.
    */
    const toggleExclusiveFilters = (values) => {
        // Generate the filter
        props.setFilters((prevFilters) => {
            let newFilters = [...prevFilters];
            values.forEach(value => {
                const exclusiveFilter = getExclusiveFilterData(props.columnDtype, value);
                const originalFilterLength = newFilters.length;
                // Remove the filter if it exists
                newFilters = newFilters.filter(filter => {
                    return isFilterGroup(filter) || !areFiltersEqual(filter, exclusiveFilter);
                });
                // If the filter didn't exist, then add it. 
                if (newFilters.length === originalFilterLength) {
                    newFilters.push(exclusiveFilter);
                }
            });
            return newFilters;
        });
    };
    const sortedUniqueValueCounts = sortUniqueValueCounts(uniqueValueCounts, sort);
    const disabledMessage = getFilterDisabledMessage(props.columnDtype);
    return (React.createElement(Fragment, null,
        React.createElement(Row, { justify: 'space-between' },
            React.createElement(Col, { flex: '1', offsetRight: 1 },
                React.createElement("p", { className: 'text-header-2' }, "Unique Values")),
            React.createElement(Col, null,
                React.createElement(Select, { value: sort, onChange: (newSortType) => {
                        setSort(newSortType);
                    }, width: 'medium', dropdownWidth: 'medium' }, Object.values(UniqueValueSortType).map(sortType => {
                    return (React.createElement(DropdownItem, { key: sortType, title: sortType }));
                })))),
        React.createElement("div", { style: { height: 'calc(100% - 40px)' } },
            React.createElement(MultiToggleBox, { loading: loading, searchable: true, searchState: {
                    searchString: searchString,
                    setSearchString: setSearchString
                }, isSubset: !isAllData, message: disabledMessage, disabled: disabledMessage !== undefined }, sortedUniqueValueCounts.map((uniqueValueCount, index) => {
                const valueToDisplay = formatCellData(uniqueValueCount.value, props.columnDtype, props.columnFormatType);
                /**
                 * If this is an NaN value, we display additional text that allows the user to navigate
                 * to the fill NaN taskpane easily
                 */
                if (valueToDisplay === 'NaN') {
                    return (React.createElement(MultiToggleItem, { key: index, title: React.createElement("span", null,
                            valueToDisplay,
                            " ",
                            React.createElement(OpenFillNaN, { setUIState: props.setUIState, columnID: props.columnID })), rightText: uniqueValueCount.countOccurence + ' (' + uniqueValueCount.percentOccurence.toFixed(2).toString() + '%' + ')', toggled: uniqueValueCount.isNotFiltered, index: index, onToggle: () => {
                            // Manually change the toggle status so it updates instantaneously
                            const uniqueValueCountIndex = getUniqueValueCountIndexFromSortedIndex(index);
                            setUniqueValueCounts(oldUniqueValueCounts => {
                                const newUniqueValueCounts = oldUniqueValueCounts.slice();
                                newUniqueValueCounts[uniqueValueCountIndex].isNotFiltered = !uniqueValueCounts[uniqueValueCountIndex].isNotFiltered;
                                return newUniqueValueCounts;
                            });
                            toggleExclusiveFilters([uniqueValueCount.value]);
                        } }));
                }
                return ((React.createElement(MultiToggleItem, { key: index, title: valueToDisplay, rightText: uniqueValueCount.countOccurence + ' (' + uniqueValueCount.percentOccurence.toFixed(2).toString() + '%' + ')', toggled: uniqueValueCount.isNotFiltered, index: index, onToggle: () => {
                        // Manually change the toggle status so it updates instantaneously
                        const uniqueValueCountIndex = getUniqueValueCountIndexFromSortedIndex(index);
                        setUniqueValueCounts(oldUniqueValueCounts => {
                            const newUniqueValueCounts = oldUniqueValueCounts.slice();
                            newUniqueValueCounts[uniqueValueCountIndex].isNotFiltered = !uniqueValueCounts[uniqueValueCountIndex].isNotFiltered;
                            return newUniqueValueCounts;
                        });
                        toggleExclusiveFilters([uniqueValueCount.value]);
                    } })));
            })))));
}
//# sourceMappingURL=ValuesTab.js.map