import os
import shutil

from django.conf import settings
from django.test import TestCase
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.contrib.sites.models import Site
from django.utils.text import slugify

from .util import to_gsm7

from .auth import get_userprofile_model
from .settings import appkit_settings


SiteProfile = appkit_settings.SITEPROFILE_MODEL
User = get_user_model()
UserProfile = get_userprofile_model()

DEFAULT_USER_PASSWORD = 'asdf;#KLVCSUkal3dsfui#@FS@!'
DEFAULT_SITE_NAME = 'My App'
DEFAULT_SITE_DOMAIN = 'testserver' # IMPORTANT: this matches the hostname generated by the test runner


class AppkitTestCaseMixin(object):
    site = None
    site_profile = None
    superuser = None

    @classmethod
    def setUpTestData(cls):
        cls.superuser = User._default_manager.db_manager().create_superuser(**{
            User.USERNAME_FIELD: 'super-user',
            User.EMAIL_FIELD: 'superuser@appkit.com',
            'first_name': 'Super',
            'last_name': 'User',
            'password': DEFAULT_USER_PASSWORD,
        })

        cls.site = Site.objects.create(name=DEFAULT_SITE_NAME, domain=DEFAULT_SITE_DOMAIN)
        cls.site_profile = SiteProfile.objects.create(created_by=cls.superuser, site=cls.site)

        UserProfile.objects.create(
            created_by=cls.superuser,
            site=cls.site,
            user=cls.superuser,
            phone='+18005551234'
        )

    @classmethod
    def tearDownClass(cls):
        # Clear any test media that was generated
        test_media_paths = [
            os.path.join(settings.MEDIA_ROOT, '__sized__', 'test'),
            os.path.join(settings.MEDIA_ROOT, 'test'),
        ]
        for media_path in test_media_paths:
            if os.path.exists(media_path):
                shutil.rmtree(media_path)

        super().tearDownClass()


    def create_document(self, DocumentClass, **kwargs):
        defaults = {
            'created_by': self.superuser,
            'site': self.site
        }

        document_info = dict(defaults)
        document_info.update(kwargs)
        return DocumentClass.objects.create(**document_info)


    def create_site(self, name=DEFAULT_SITE_NAME, domain=DEFAULT_SITE_DOMAIN):
        site = Site.objects.create(name=name, domain=domain)
        self.create_document(SiteProfile, site=site)
        return site


    def create_user(self, first_name='TestUser', last_name='', email=None, phone=None,
                    is_active=True, is_staff=False, site=None, created_by=None):
        full_name = '{} {}'.format(first_name, last_name)
        username = slugify(full_name)

        user = User._default_manager.db_manager().create_user(**{
            User.USERNAME_FIELD: username,
            User.EMAIL_FIELD: email if email else f'{username}@noundb.com',
            'first_name': first_name,
            'last_name': last_name,
            'is_active': is_active,
            'is_staff': is_staff,
            'password': DEFAULT_USER_PASSWORD,
        })

        UserProfile.objects.create(
            created_by=created_by if created_by else self.superuser,
            site=site if site else self.site,
            user=user,
            phone=phone,
        )

        return user

    def create_tpb_users(self):
        TRAILER_PARK_BOYS = (
            {'first_name': 'Jim', 'last_name': 'Lahey', 'is_staff': True, 'phone': '+17825556969'},
            {'first_name': 'Randy', 'last_name': 'BoBandy', 'is_staff': False},
            {'first_name': 'Sam', 'last_name': 'Lasco', 'is_staff': False, 'phone': '+17825551234'},
            {'first_name': 'Ricky', 'is_staff': False},
            {'first_name': 'Julian', 'is_staff': False},
            {'first_name': 'Bubbles', 'is_staff': False},
        )

        tpb_users = {}
        for user_info in TRAILER_PARK_BOYS:
            user = self.create_user(**user_info)
            tpb_users[user.username] = user

        tpb_group = Group.objects.create(name='trailer-park-boys')
        tpb_group.user_set.add(*tpb_users.values())
        self.tpb_group = tpb_group

        return tpb_users


class GSMTestCase(TestCase):
    VALID_GSM_STRING = 'This message contains only GSM-7 characters'
    INVALID_GSM_STRING = "🥳 [It's] a $#!T 🥳 party 🥳"

    def test_gsm7_to_gsm7(self):
        self.assertEqual(self.VALID_GSM_STRING, to_gsm7(self.VALID_GSM_STRING))

    def test_emoji_to_gsm7(self):
        # Note that leading and trailing whitespace are valid and therefore _NOT_ removed
        self.assertEqual(to_gsm7(self.INVALID_GSM_STRING), " [It's] a $#!T  party ")

    def test_to_gsm7_max_length(self):
        gsm_msg = to_gsm7(to_gsm7(self.INVALID_GSM_STRING), max_length=18)
        self.assertEqual(gsm_msg, " [It's] a $#!T  pa")
