import os,json
from typing import List
from behave import given, when, then, step # pylint: disable=no-name-in-module (https://github.com/behave/behave/issues/641)
from behave_pandas import table_to_dataframe, dataframe_to_table
from pathlib import Path
from omnata_plugin_runtime.configuration import OutboundSyncConfigurationParameters,InboundSyncConfigurationParameters,StoredStreamConfiguration
from omnata_plugin_runtime.omnata_plugin import ApiLimits, OutboundSyncRequest,InboundSyncRequest
import pandas
import unittest
import datetime
from types import ModuleType
from snowflake.snowpark.functions import row_number, col, when_matched, when_not_matched, lit, current_timestamp, iff
import vcr
from pandas.testing import assert_frame_equal
from pydantic import parse_obj_as # pylint: disable=no-name-in-module
import logging
import sys
import importlib
case = unittest.TestCase()

@given('the following records')
def step_impl_records(context):
    column_names = []
    row_strings = []
    column_bindings = []
    print(f"context.table: {context.table}")
    print(f"context.table.headings: {context.table.headings}")
    context.source_records = table_to_dataframe(context.table)
    print(f"source dataframe: {context.source_records}")

@given('the following streams state')
def step_impl_streams_state(context):
    context.streams_state = {}
    for row in context.table:
        if 'Stream' not in row.headings:
            raise ValueError('Streams state must have a Stream column')
        if 'Value' not in row.headings:
            raise ValueError('Streams state must have a Value column')
        context.streams_state[row['Stream']] = json.loads(row['Value'])
    print(f"streams state: {context.streams_state}")


@when('we perform an outbound sync with configuration parameters')
def step_impl_outbound_sync(context):
    if context.plugin_class is None:
        raise ValueError('You must define which plugin class and module is to be used ("we use the x class from the y module")')
    strategy=None
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    api_limits = {}
    field_mappings = []
    connection_method = None

    for row in context.table:
        if row['Property']=='strategy':
            strategy = json.loads(row['Value'])
        elif row['Property']=='connection_method':
            connection_method = row['Value']
        elif row['Property']=='connection_parameters':
            connection_parameters = json.loads(row['Value'])
        elif row['Property']=='api_limits':
            api_limits = json.loads(row['Value'])
        elif row['Property']=='connection_secrets':
            connection_secrets = json.loads(row['Value'])
        elif row['Property']=='sync_parameters':
            sync_parameters = json.loads(row['Value'])
        elif row['Property']=='field_mappings':
            field_mappings = json.loads(row['Value'])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")
    
    parameters = OutboundSyncConfigurationParameters.parse_obj({
        "sync_strategy": strategy,
        "connection_method": connection_method,
        "connection_parameters": connection_parameters,
        "connection_secrets": connection_secrets,
        "sync_parameters": sync_parameters,
        "field_mappings": field_mappings
    })
    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    outbound_sync_request = OutboundSyncRequest(
                                            run_id=None,
                                            session=None,
                                            source_app_name=None,
                                            records_schema_name='',
                                            records_table_name='',
                                            results_schema_name='',
                                            results_table_name='',
                                            plugin_instance=context.plugin_instance,
                                            api_limits={},
                                            rate_limit_state={},
                                            # make it 5 mins since we should be using vcr
                                            run_deadline=datetime.datetime.now() + datetime.timedelta(minutes=5),
                                            development_mode=True,
                                            test_replay_mode=True)
    
    outbound_sync_request._prebaked_record_state = context.source_records
    context.error = None
    try:
        context.sync_direction = 'outbound'
        context.sync_outbound_result = context.plugin_instance.sync_outbound(parameters,outbound_sync_request)
        # when using a managed_outbound_processing decorator, the results aren't returned from sync_outbound
        if context.sync_outbound_result is None:
            context.sync_outbound_result = outbound_sync_request.get_queued_results()
    except Exception as e:
        context.error = e
        logging.exception(e)


@when('we perform an inbound sync with configuration parameters')
def step_impl_inbound_sync(context):
    if context.plugin_class is None:
        raise ValueError('You must define which plugin class and module is to be used ("we use the x class from the y module")')
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    api_limits = {}
    streams = []
    connection_method = None

    for row in context.table:
        if row['Property']=='connection_method':
            connection_method = row['Value']
        elif row['Property']=='connection_parameters':
            connection_parameters = json.loads(row['Value'])
        elif row['Property']=='api_limits':
            api_limits = json.loads(row['Value'])
        elif row['Property']=='connection_secrets':
            connection_secrets = json.loads(row['Value'])
        elif row['Property']=='sync_parameters':
            sync_parameters = json.loads(row['Value'])
        elif row['Property']=='streams':
            streams = json.loads(row['Value'])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")

    parameters = InboundSyncConfigurationParameters.parse_obj({
        "connection_method": connection_method,
        "connection_parameters": connection_parameters,
        "connection_secrets": connection_secrets,
        "sync_parameters": sync_parameters
    })
    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    streams_list = parse_obj_as(List[StoredStreamConfiguration],streams)
    for stream in streams_list:
        if stream.stream_name in context.streams_state:
            stream.latest_state = context.streams_state[stream.stream_name]
    inbound_sync_request = InboundSyncRequest(
                                            run_id=None,
                                            session=None,
                                            source_app_name=None,
                                            results_schema_name='',
                                            results_table_name='',
                                            plugin_instance=context.plugin_instance,
                                            api_limits={},
                                            rate_limit_state={},
                                            run_deadline=datetime.datetime.now() + datetime.timedelta(minutes=5),
                                            development_mode=True,
                                            streams=streams_list,
                                            test_replay_mode=True)
    
    #outbound_sync_request._prebaked_record_state = context.source_records
    context.error = None
    try:
        context.sync_direction = 'inbound'
        context.plugin_instance.sync_inbound(parameters,inbound_sync_request)
        if len(inbound_sync_request._apply_results)==0:
            raise ValueError('No results returned from sync_inbound')
        context.sync_inbound_result = inbound_sync_request._apply_results
    except Exception as e:
        context.error = e
        logging.exception(e)

def load_script_from_file(app_name,action_name):
    f = open(os.path.join('scripts',app_name,f"{action_name}.py"))
    script_contents=f.read()
    mod = ModuleType('whatever.py','')
    exec(script_contents, mod.__dict__)
    return mod      

@step('we use the {plugin_class} class from the {plugin_module} module')
def step_impl_use_plugin_class(context, plugin_class, plugin_module):
    # Assuming the module is defined in the parent of the tests directory
    sys.path.insert(0, os.path.abspath(os.path.join(context._root['config'].base_dir,'..')))
    module = importlib.import_module(plugin_module)
    context.plugin_class = getattr(module, plugin_class)


@step('we use the HTTP recordings from {filename}')
def step_impl_use_http_recordings(context, filename):
    if 'cassette' in context:
        context.cassette.__exit__()
    file_name = os.path.join(context._root['config'].base_dir,'vcr_cassettes',filename)
    print(f'using cassette at {file_name}')
    my_vcr = vcr.VCR(
        record_mode='none'
    )
    if 'VCRPY_LOG_LEVEL' in os.environ:
        logging.basicConfig(format='%(asctime)s %(levelname)-8s %(message)s',
                    level=logging.INFO,
                    datefmt='%Y-%m-%d %H:%M:%S')
        vcr_log = logging.getLogger("vcr")
        vcr_log.setLevel(logging._nameToLevel[os.environ['VCRPY_LOG_LEVEL']])
    my_vcr.filter_query_parameters
    context.cassette = my_vcr.use_cassette(file_name,filter_query_parameters=['q'])
    context.cassette.__enter__()

@step('the response will be')
def step_impl_response_outbound(context):
    column_names = []
    row_strings = []
    column_bindings = []
    expected_result = table_to_dataframe(context.table)
    expected_result = expected_result.sort_values(by=['IDENTIFIER']).sort_index(axis=1).reset_index(drop=True)
    context.sync_outbound_result = context.sync_outbound_result.sort_values(by=['IDENTIFIER']).sort_index(axis=1).reset_index(drop=True)
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    #expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    context.sync_outbound_result['RESULT'] = context.sync_outbound_result['RESULT'].apply(json.dumps)
    assert list(expected_result.columns)==list(context.sync_outbound_result.columns),f"Column headings didn't match. Expected: {expected_result.columns}, actual: {context.sync_outbound_result.columns}"
    pandas.set_option('display.max_columns', 10)
    pandas.set_option('display.width', 150)
    print(f"expected_result: {expected_result}")
    print(f"expected_result single: {expected_result['RESULT'][0]}")
    print(f"expected_result dtypes: {expected_result.dtypes}")
    print(f"apply_result: {context.sync_outbound_result}")
    print(f"apply_result single: {context.sync_outbound_result['RESULT'][0]}")
    print(f"apply_result dtypes: {context.sync_outbound_result.dtypes}")
    print(f"differences: {expected_result.compare(context.sync_outbound_result)}")
    case.assertCountEqual(expected_result.columns.to_list(),context.sync_outbound_result.columns.to_list())
    assert_frame_equal(expected_result,context.sync_outbound_result)

@step('the response for the {stream_name} stream will be')
def step_impl_response_inbound(context,stream_name):
    expected_result = table_to_dataframe(context.table)
    expected_result = expected_result.sort_values(by=['APP_IDENTIFIER']).sort_index(axis=1).reset_index(drop=True)
    if stream_name not in context.sync_inbound_result:
        print(context.sync_inbound_result)
        raise ValueError(f"Stream {stream_name} not found in sync_inbound_result. Keys: {list(context.sync_inbound_result.keys())}")
    context.sync_inbound_result = pandas.concat(context.sync_inbound_result[stream_name]).sort_values(by=['APP_IDENTIFIER']).sort_index(axis=1).reset_index(drop=True)
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    #expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    context.sync_inbound_result['RECORD_DATA'] = context.sync_inbound_result['RECORD_DATA'].apply(json.dumps)
    # exclude RETRIEVE_DATE column from both dataframes, naturally they will differ
    expected_result = expected_result.drop(columns=['RETRIEVE_DATE'])
    context.sync_inbound_result = context.sync_inbound_result.drop(columns=['RETRIEVE_DATE'])
    assert list(expected_result.columns)==list(context.sync_inbound_result.columns),f"Column headings didn't match. Expected: {expected_result.columns}, actual: {context.sync_outbound_result.columns}"
    pandas.set_option('display.max_columns', 10)
    pandas.set_option('display.width', 150)
    print(f"expected_result: {expected_result}")
    #print(f"expected_result single: {expected_result['RECORD_DATA'][0]}")
    print(f"expected_result dtypes: {expected_result.dtypes}")
    print(f"sync_inbound_result: {context.sync_inbound_result}")
    #print(f"sync_inbound_result single: {context.sync_inbound_result['RECORD_DATA'][0]}")
    print(f"sync_inbound_result dtypes: {context.sync_inbound_result.dtypes}")
    print(f"differences: {expected_result.compare(context.sync_inbound_result)}")
    case.assertCountEqual(expected_result.columns.to_list(),context.sync_inbound_result.columns.to_list())
    assert_frame_equal(expected_result,context.sync_inbound_result)

@then('no error will be raised')
def step_impl_no_error(context):
    assert context.error is None,f"Expected no error from action, instead got {context.error}"