"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const pinger_1 = require("./pinger/pinger");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        const secretsEncryptionKey = new kms.Key(this, 'SecretsKey');
        // just need one nat gateway to simplify the test
        this.vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_17,
            secretsEncryptionKey,
        });
        this.assertFargateProfile();
        this.assertCapacityX86();
        this.assertCapacityArm();
        this.assertBottlerocket();
        this.assertSpotCapacity();
        this.assertInferenceInstances();
        this.assertNodeGroupX86();
        this.assertNodeGroupArm();
        this.assertNodeGroupCustomAmi();
        this.assertSimpleManifest();
        this.assertSimpleHelmChart();
        this.assertCreateNamespace();
        this.assertServiceAccount();
        this.assertServiceLoadBalancerAddress();
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: this.cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: this.cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: this.cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: this.cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: this.cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
    }
    assertServiceAccount() {
        // add a service account connected to a IAM role
        this.cluster.addServiceAccount('MyServiceAccount');
    }
    assertCreateNamespace() {
        // deploy an nginx ingress in a namespace
        const nginxNamespace = this.cluster.addManifest('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = this.cluster.addChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
    }
    assertSimpleHelmChart() {
        // deploy the Kubernetes dashboard through a helm chart
        this.cluster.addChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
    }
    assertSimpleManifest() {
        // apply a kubernetes manifest
        this.cluster.addManifest('HelloApp', ...hello.resources);
    }
    assertNodeGroupX86() {
        // add a extra nodegroup
        this.cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertNodeGroupCustomAmi() {
        var _a, _b;
        // add a extra nodegroup
        const userData = ec2.UserData.forLinux();
        userData.addCommands('set -o xtrace', `/etc/eks/bootstrap.sh ${this.cluster.clusterName}`);
        const lt = new ec2.CfnLaunchTemplate(this, 'LaunchTemplate', {
            launchTemplateData: {
                imageId: new eks.EksOptimizedImage().getImage(this).imageId,
                instanceType: new ec2.InstanceType('t3.small').toString(),
                userData: core_1.Fn.base64(userData.render()),
            },
        });
        this.cluster.addNodegroup('extra-ng2', {
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: ((_a = this.cluster.defaultNodegroup) === null || _a === void 0 ? void 0 : _a.role) || ((_b = this.cluster.defaultCapacity) === null || _b === void 0 ? void 0 : _b.role),
            launchTemplate: {
                id: lt.ref,
                version: lt.attrDefaultVersionNumber,
            },
        });
    }
    assertNodeGroupArm() {
        // add a extra nodegroup
        this.cluster.addNodegroup('extra-ng-arm', {
            instanceType: new ec2.InstanceType('m6g.medium'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertInferenceInstances() {
        // inference instances
        this.cluster.addCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
    }
    assertSpotCapacity() {
        // spot instances (up to 10)
        this.cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
    }
    assertBottlerocket() {
        // add bottlerocket nodes
        this.cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
    }
    assertCapacityX86() {
        // add some x86_64 capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
    }
    assertCapacityArm() {
        // add some arm64 capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addCapacity('NodesArm', {
            instanceType: new ec2.InstanceType('m6g.medium'),
            minCapacity: 1,
        });
    }
    assertFargateProfile() {
        // fargate profile for resources in the "default" namespace
        this.cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
    }
    assertServiceLoadBalancerAddress() {
        const serviceName = 'webservice';
        const labels = { app: 'simple-web' };
        const containerPort = 80;
        const servicePort = 9000;
        const pingerSecurityGroup = new ec2.SecurityGroup(this, 'WebServiceSecurityGroup', {
            vpc: this.vpc,
        });
        pingerSecurityGroup.addIngressRule(pingerSecurityGroup, ec2.Port.tcp(servicePort), `allow http ${servicePort} access from myself`);
        this.cluster.addManifest('simple-web-pod', {
            kind: 'Pod',
            apiVersion: 'v1',
            metadata: { name: 'webpod', labels: labels },
            spec: {
                containers: [{
                        name: 'simplewebcontainer',
                        image: 'nginx',
                        ports: [{ containerPort: containerPort }],
                    }],
            },
        });
        this.cluster.addManifest('simple-web-service', {
            kind: 'Service',
            apiVersion: 'v1',
            metadata: {
                name: serviceName,
                annotations: {
                    // this is furtile soil for cdk8s-plus! :)
                    'service.beta.kubernetes.io/aws-load-balancer-internal': 'true',
                    'service.beta.kubernetes.io/aws-load-balancer-extra-security-groups': pingerSecurityGroup.securityGroupId,
                },
            },
            spec: {
                type: 'LoadBalancer',
                ports: [{ port: servicePort, targetPort: containerPort }],
                selector: labels,
            },
        });
        const loadBalancerAddress = this.cluster.getServiceLoadBalancerAddress(serviceName);
        // create a resource that hits the load balancer to make sure
        // everything is wired properly.
        const pinger = new pinger_1.Pinger(this, 'ServicePinger', {
            url: `http://${loadBalancerAddress}:${servicePort}`,
            securityGroup: pingerSecurityGroup,
            vpc: this.vpc,
        });
        // this should display a proper nginx response
        // <title>Welcome to nginx!</title>...
        new core_1.CfnOutput(this, 'Response', {
            value: pinger.response,
        });
    }
}
// this test uses both the bottlerocket image and the inf1 instance, which are only supported in these
// regions. see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-eks#bottlerocket
// and https://aws.amazon.com/about-aws/whats-new/2019/12/introducing-amazon-ec2-inf1-instances-high-performance-and-the-lowest-cost-machine-learning-inference-in-the-cloud/
const supportedRegions = [
    'us-east-1',
    'us-west-2',
];
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
const stack = new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
if (process.env.CDK_INTEG_ACCOUNT !== '12345678') {
    // only validate if we are about to actually deploy.
    // TODO: better way to determine this, right now the 'CDK_INTEG_ACCOUNT' seems like the only way.
    if (core_1.Token.isUnresolved(stack.region)) {
        throw new Error(`region (${stack.region}) cannot be a token and must be configured to one of: ${supportedRegions}`);
    }
    if (!supportedRegions.includes(stack.region)) {
        throw new Error(`region (${stack.region}) must be configured to one of: ${supportedRegions}`);
    }
}
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZWtzLWNsdXN0ZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5la3MtY2x1c3Rlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLG1DQUFtQztBQUNuQyx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBb0U7QUFDcEUsOEJBQThCO0FBQzlCLHFDQUFxQztBQUNyQyw0Q0FBeUM7QUFDekMsaUNBQW1DO0FBR25DLE1BQU0sZUFBZ0IsU0FBUSxnQkFBUztJQUtyQyxZQUFZLEtBQVUsRUFBRSxFQUFVO1FBQ2hDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsNEVBQTRFO1FBQzVFLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO1lBQ2xELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxvQkFBb0IsRUFBRTtTQUMxQyxDQUFDLENBQUM7UUFFSCxNQUFNLG9CQUFvQixHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFFN0QsaURBQWlEO1FBQ2pELElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRW5FLHVEQUF1RDtRQUN2RCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzlDLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLFdBQVc7WUFDWCxlQUFlLEVBQUUsQ0FBQztZQUNsQixPQUFPLEVBQUUsR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUs7WUFDcEMsb0JBQW9CO1NBQ3JCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBRTVCLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBRXpCLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBRXpCLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBRTFCLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBRTFCLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRWhDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBRTFCLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBRTFCLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRWhDLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBRTVCLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO1FBRTdCLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO1FBRTdCLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBRTVCLElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxDQUFDO1FBRXhDLElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBQ2hGLElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUN0RSxJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLGlDQUFpQyxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsK0JBQStCLEVBQUUsQ0FBQyxDQUFDO1FBQ2hILElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxzQkFBc0IsRUFBRSxDQUFDLENBQUM7UUFDOUYsSUFBSSxnQkFBUyxDQUFDLElBQUksRUFBRSwrQkFBK0IsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLDZCQUE2QixFQUFFLENBQUMsQ0FBQztRQUM1RyxJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVPLG9CQUFvQjtRQUMxQixnREFBZ0Q7UUFDaEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFTyxxQkFBcUI7UUFDM0IseUNBQXlDO1FBQ3pDLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLGlCQUFpQixFQUFFO1lBQ2pFLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLElBQUksRUFBRSxXQUFXO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUixJQUFJLEVBQUUsT0FBTzthQUNkO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsZUFBZSxFQUFFO1lBQzFELEtBQUssRUFBRSxlQUFlO1lBQ3RCLFVBQVUsRUFBRSwrQkFBK0I7WUFDM0MsU0FBUyxFQUFFLE9BQU87WUFDbEIsSUFBSSxFQUFFLElBQUk7WUFDVixlQUFlLEVBQUUsS0FBSztZQUN0QixPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsbURBQW1EO1FBQ25ELFlBQVksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBR2xELENBQUM7SUFDTyxxQkFBcUI7UUFDM0IsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRTtZQUNqQyxLQUFLLEVBQUUsc0JBQXNCO1lBQzdCLFVBQVUsRUFBRSx5Q0FBeUM7U0FDdEQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUNPLG9CQUFvQjtRQUMxQiw4QkFBOEI7UUFDOUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFDTyxrQkFBa0I7UUFDeEIsd0JBQXdCO1FBQ3hCLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRTtZQUNwQyxZQUFZLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQztZQUM5QyxPQUFPLEVBQUUsQ0FBQztZQUNWLHNFQUFzRTtZQUN0RSxRQUFRLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUztTQUN2RixDQUFDLENBQUM7SUFDTCxDQUFDO0lBQ08sd0JBQXdCOztRQUM5Qix3QkFBd0I7UUFDeEIsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUN6QyxRQUFRLENBQUMsV0FBVyxDQUNsQixlQUFlLEVBQ2YseUJBQXlCLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLENBQ3BELENBQUM7UUFDRixNQUFNLEVBQUUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDM0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLE9BQU8sRUFBRSxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPO2dCQUMzRCxZQUFZLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFFBQVEsRUFBRTtnQkFDekQsUUFBUSxFQUFFLFNBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO2FBQ3ZDO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO1lBQ3JDLE9BQU8sRUFBRSxDQUFDO1lBQ1Ysc0VBQXNFO1lBQ3RFLFFBQVEsRUFBRSxPQUFBLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLDBDQUFFLElBQUksWUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsMENBQUUsSUFBSSxDQUFBO1lBQ25GLGNBQWMsRUFBRTtnQkFDZCxFQUFFLEVBQUUsRUFBRSxDQUFDLEdBQUc7Z0JBQ1YsT0FBTyxFQUFFLEVBQUUsQ0FBQyx3QkFBd0I7YUFDckM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBQ08sa0JBQWtCO1FBQ3hCLHdCQUF3QjtRQUN4QixJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUU7WUFDeEMsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUM7WUFDaEQsT0FBTyxFQUFFLENBQUM7WUFDVixzRUFBc0U7WUFDdEUsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdkYsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUNPLHdCQUF3QjtRQUM5QixzQkFBc0I7UUFDdEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQUU7WUFDN0MsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUM7WUFDbEQsV0FBVyxFQUFFLENBQUM7U0FDZixDQUFDLENBQUM7SUFDTCxDQUFDO0lBQ08sa0JBQWtCO1FBQ3hCLDRCQUE0QjtRQUM1QixJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUU7WUFDL0IsU0FBUyxFQUFFLFFBQVE7WUFDbkIsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUM7WUFDOUMsV0FBVyxFQUFFLEVBQUU7WUFDZixnQkFBZ0IsRUFBRTtnQkFDaEIsZ0JBQWdCLEVBQUUsK0JBQStCO2dCQUNqRCxtQkFBbUIsRUFBRSxDQUFDO2FBQ3ZCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUNPLGtCQUFrQjtRQUN4Qix5QkFBeUI7UUFDekIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsbUJBQW1CLEVBQUU7WUFDNUMsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUM7WUFDOUMsV0FBVyxFQUFFLENBQUM7WUFDZCxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsWUFBWTtTQUNwRCxDQUFDLENBQUM7SUFFTCxDQUFDO0lBQ08saUJBQWlCO1FBQ3ZCLHNFQUFzRTtRQUN0RSwyRUFBMkU7UUFDM0UsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFO1lBQ2hDLFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDO1lBQy9DLFdBQVcsRUFBRSxDQUFDO1NBQ2YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQjtRQUN2QixxRUFBcUU7UUFDckUsMkVBQTJFO1FBQzNFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLFVBQVUsRUFBRTtZQUNuQyxZQUFZLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQztZQUNoRCxXQUFXLEVBQUUsQ0FBQztTQUNmLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxvQkFBb0I7UUFDMUIsMkRBQTJEO1FBQzNELElBQUksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFO1lBQ3hDLFNBQVMsRUFBRSxDQUFDLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxDQUFDO1NBQ3RDLENBQUMsQ0FBQztJQUVMLENBQUM7SUFFTyxnQ0FBZ0M7UUFFdEMsTUFBTSxXQUFXLEdBQUcsWUFBWSxDQUFDO1FBQ2pDLE1BQU0sTUFBTSxHQUFHLEVBQUUsR0FBRyxFQUFFLFlBQVksRUFBRSxDQUFDO1FBQ3JDLE1BQU0sYUFBYSxHQUFHLEVBQUUsQ0FBQztRQUN6QixNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUM7UUFFekIsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFO1lBQ2pGLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztTQUNkLENBQUMsQ0FBQztRQUVILG1CQUFtQixDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLFdBQVcscUJBQXFCLENBQUMsQ0FBQztRQUVuSSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRTtZQUN6QyxJQUFJLEVBQUUsS0FBSztZQUNYLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRTtZQUM1QyxJQUFJLEVBQUU7Z0JBQ0osVUFBVSxFQUFFLENBQUM7d0JBQ1gsSUFBSSxFQUFFLG9CQUFvQjt3QkFDMUIsS0FBSyxFQUFFLE9BQU87d0JBQ2QsS0FBSyxFQUFFLENBQUMsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLENBQUM7cUJBQzFDLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLG9CQUFvQixFQUFFO1lBQzdDLElBQUksRUFBRSxTQUFTO1lBQ2YsVUFBVSxFQUFFLElBQUk7WUFDaEIsUUFBUSxFQUFFO2dCQUNSLElBQUksRUFBRSxXQUFXO2dCQUNqQixXQUFXLEVBQUU7b0JBQ1gsMENBQTBDO29CQUMxQyx1REFBdUQsRUFBRSxNQUFNO29CQUMvRCxvRUFBb0UsRUFBRSxtQkFBbUIsQ0FBQyxlQUFlO2lCQUMxRzthQUNGO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLElBQUksRUFBRSxjQUFjO2dCQUNwQixLQUFLLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLGFBQWEsRUFBRSxDQUFDO2dCQUN6RCxRQUFRLEVBQUUsTUFBTTthQUNqQjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyw2QkFBNkIsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUVwRiw2REFBNkQ7UUFDN0QsZ0NBQWdDO1FBQ2hDLE1BQU0sTUFBTSxHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDL0MsR0FBRyxFQUFFLFVBQVUsbUJBQW1CLElBQUksV0FBVyxFQUFFO1lBQ25ELGFBQWEsRUFBRSxtQkFBbUI7WUFDbEMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsOENBQThDO1FBQzlDLHNDQUFzQztRQUN0QyxJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM5QixLQUFLLEVBQUUsTUFBTSxDQUFDLFFBQVE7U0FDdkIsQ0FBQyxDQUFDO0lBRUwsQ0FBQztDQUNGO0FBRUQsc0dBQXNHO0FBQ3RHLG1HQUFtRztBQUNuRyw2S0FBNks7QUFDN0ssTUFBTSxnQkFBZ0IsR0FBRztJQUN2QixXQUFXO0lBQ1gsV0FBVztDQUNaLENBQUM7QUFFRixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0FBRXRCLHNFQUFzRTtBQUN0RSxpREFBaUQ7QUFDakQsTUFBTSxLQUFLLEdBQUcsSUFBSSxlQUFlLENBQUMsR0FBRyxFQUFFLDBCQUEwQixDQUFDLENBQUM7QUFFbkUsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLGlCQUFpQixLQUFLLFVBQVUsRUFBRTtJQUVoRCxvREFBb0Q7SUFDcEQsaUdBQWlHO0lBRWpHLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssQ0FBQyxNQUFNLHlEQUF5RCxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7S0FDckg7SUFFRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsS0FBSyxDQUFDLE1BQU0sbUNBQW1DLGdCQUFnQixFQUFFLENBQUMsQ0FBQztLQUMvRjtDQUVGO0FBR0QsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLy8vICFjZGstaW50ZWcgcHJhZ21hOmlnbm9yZS1hc3NldHNcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCB7IEFwcCwgQ2ZuT3V0cHV0LCBEdXJhdGlvbiwgVG9rZW4sIEZuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBla3MgZnJvbSAnLi4vbGliJztcbmltcG9ydCAqIGFzIGhlbGxvIGZyb20gJy4vaGVsbG8tazhzJztcbmltcG9ydCB7IFBpbmdlciB9IGZyb20gJy4vcGluZ2VyL3Bpbmdlcic7XG5pbXBvcnQgeyBUZXN0U3RhY2sgfSBmcm9tICcuL3V0aWwnO1xuXG5cbmNsYXNzIEVrc0NsdXN0ZXJTdGFjayBleHRlbmRzIFRlc3RTdGFjayB7XG5cbiAgcHJpdmF0ZSBjbHVzdGVyOiBla3MuQ2x1c3RlcjtcbiAgcHJpdmF0ZSB2cGM6IGVjMi5JVnBjO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBBcHAsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgLy8gYWxsb3cgYWxsIGFjY291bnQgdXNlcnMgdG8gYXNzdW1lIHRoaXMgcm9sZSBpbiBvcmRlciB0byBhZG1pbiB0aGUgY2x1c3RlclxuICAgIGNvbnN0IG1hc3RlcnNSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdBZG1pblJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uQWNjb3VudFJvb3RQcmluY2lwYWwoKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHNlY3JldHNFbmNyeXB0aW9uS2V5ID0gbmV3IGttcy5LZXkodGhpcywgJ1NlY3JldHNLZXknKTtcblxuICAgIC8vIGp1c3QgbmVlZCBvbmUgbmF0IGdhdGV3YXkgdG8gc2ltcGxpZnkgdGhlIHRlc3RcbiAgICB0aGlzLnZwYyA9IG5ldyBlYzIuVnBjKHRoaXMsICdWcGMnLCB7IG1heEF6czogMywgbmF0R2F0ZXdheXM6IDEgfSk7XG5cbiAgICAvLyBjcmVhdGUgdGhlIGNsdXN0ZXIgd2l0aCBhIGRlZmF1bHQgbm9kZWdyb3VwIGNhcGFjaXR5XG4gICAgdGhpcy5jbHVzdGVyID0gbmV3IGVrcy5DbHVzdGVyKHRoaXMsICdDbHVzdGVyJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIG1hc3RlcnNSb2xlLFxuICAgICAgZGVmYXVsdENhcGFjaXR5OiAyLFxuICAgICAgdmVyc2lvbjogZWtzLkt1YmVybmV0ZXNWZXJzaW9uLlYxXzE3LFxuICAgICAgc2VjcmV0c0VuY3J5cHRpb25LZXksXG4gICAgfSk7XG5cbiAgICB0aGlzLmFzc2VydEZhcmdhdGVQcm9maWxlKCk7XG5cbiAgICB0aGlzLmFzc2VydENhcGFjaXR5WDg2KCk7XG5cbiAgICB0aGlzLmFzc2VydENhcGFjaXR5QXJtKCk7XG5cbiAgICB0aGlzLmFzc2VydEJvdHRsZXJvY2tldCgpO1xuXG4gICAgdGhpcy5hc3NlcnRTcG90Q2FwYWNpdHkoKTtcblxuICAgIHRoaXMuYXNzZXJ0SW5mZXJlbmNlSW5zdGFuY2VzKCk7XG5cbiAgICB0aGlzLmFzc2VydE5vZGVHcm91cFg4NigpO1xuXG4gICAgdGhpcy5hc3NlcnROb2RlR3JvdXBBcm0oKTtcblxuICAgIHRoaXMuYXNzZXJ0Tm9kZUdyb3VwQ3VzdG9tQW1pKCk7XG5cbiAgICB0aGlzLmFzc2VydFNpbXBsZU1hbmlmZXN0KCk7XG5cbiAgICB0aGlzLmFzc2VydFNpbXBsZUhlbG1DaGFydCgpO1xuXG4gICAgdGhpcy5hc3NlcnRDcmVhdGVOYW1lc3BhY2UoKTtcblxuICAgIHRoaXMuYXNzZXJ0U2VydmljZUFjY291bnQoKTtcblxuICAgIHRoaXMuYXNzZXJ0U2VydmljZUxvYWRCYWxhbmNlckFkZHJlc3MoKTtcblxuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ0NsdXN0ZXJFbmRwb2ludCcsIHsgdmFsdWU6IHRoaXMuY2x1c3Rlci5jbHVzdGVyRW5kcG9pbnQgfSk7XG4gICAgbmV3IENmbk91dHB1dCh0aGlzLCAnQ2x1c3RlckFybicsIHsgdmFsdWU6IHRoaXMuY2x1c3Rlci5jbHVzdGVyQXJuIH0pO1xuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ0NsdXN0ZXJDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGEnLCB7IHZhbHVlOiB0aGlzLmNsdXN0ZXIuY2x1c3RlckNlcnRpZmljYXRlQXV0aG9yaXR5RGF0YSB9KTtcbiAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsICdDbHVzdGVyU2VjdXJpdHlHcm91cElkJywgeyB2YWx1ZTogdGhpcy5jbHVzdGVyLmNsdXN0ZXJTZWN1cml0eUdyb3VwSWQgfSk7XG4gICAgbmV3IENmbk91dHB1dCh0aGlzLCAnQ2x1c3RlckVuY3J5cHRpb25Db25maWdLZXlBcm4nLCB7IHZhbHVlOiB0aGlzLmNsdXN0ZXIuY2x1c3RlckVuY3J5cHRpb25Db25maWdLZXlBcm4gfSk7XG4gICAgbmV3IENmbk91dHB1dCh0aGlzLCAnQ2x1c3Rlck5hbWUnLCB7IHZhbHVlOiB0aGlzLmNsdXN0ZXIuY2x1c3Rlck5hbWUgfSk7XG4gIH1cblxuICBwcml2YXRlIGFzc2VydFNlcnZpY2VBY2NvdW50KCkge1xuICAgIC8vIGFkZCBhIHNlcnZpY2UgYWNjb3VudCBjb25uZWN0ZWQgdG8gYSBJQU0gcm9sZVxuICAgIHRoaXMuY2x1c3Rlci5hZGRTZXJ2aWNlQWNjb3VudCgnTXlTZXJ2aWNlQWNjb3VudCcpO1xuICB9XG5cbiAgcHJpdmF0ZSBhc3NlcnRDcmVhdGVOYW1lc3BhY2UoKSB7XG4gICAgLy8gZGVwbG95IGFuIG5naW54IGluZ3Jlc3MgaW4gYSBuYW1lc3BhY2VcbiAgICBjb25zdCBuZ2lueE5hbWVzcGFjZSA9IHRoaXMuY2x1c3Rlci5hZGRNYW5pZmVzdCgnbmdpbngtbmFtZXNwYWNlJywge1xuICAgICAgYXBpVmVyc2lvbjogJ3YxJyxcbiAgICAgIGtpbmQ6ICdOYW1lc3BhY2UnLFxuICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAgbmFtZTogJ25naW54JyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBuZ2lueEluZ3Jlc3MgPSB0aGlzLmNsdXN0ZXIuYWRkQ2hhcnQoJ25naW54LWluZ3Jlc3MnLCB7XG4gICAgICBjaGFydDogJ25naW54LWluZ3Jlc3MnLFxuICAgICAgcmVwb3NpdG9yeTogJ2h0dHBzOi8vaGVsbS5uZ2lueC5jb20vc3RhYmxlJyxcbiAgICAgIG5hbWVzcGFjZTogJ25naW54JyxcbiAgICAgIHdhaXQ6IHRydWUsXG4gICAgICBjcmVhdGVOYW1lc3BhY2U6IGZhbHNlLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgfSk7XG5cbiAgICAvLyBtYWtlIHN1cmUgbmFtZXNwYWNlIGlzIGRlcGxveWVkIGJlZm9yZSB0aGUgY2hhcnRcbiAgICBuZ2lueEluZ3Jlc3Mubm9kZS5hZGREZXBlbmRlbmN5KG5naW54TmFtZXNwYWNlKTtcblxuXG4gIH1cbiAgcHJpdmF0ZSBhc3NlcnRTaW1wbGVIZWxtQ2hhcnQoKSB7XG4gICAgLy8gZGVwbG95IHRoZSBLdWJlcm5ldGVzIGRhc2hib2FyZCB0aHJvdWdoIGEgaGVsbSBjaGFydFxuICAgIHRoaXMuY2x1c3Rlci5hZGRDaGFydCgnZGFzaGJvYXJkJywge1xuICAgICAgY2hhcnQ6ICdrdWJlcm5ldGVzLWRhc2hib2FyZCcsXG4gICAgICByZXBvc2l0b3J5OiAnaHR0cHM6Ly9rdWJlcm5ldGVzLmdpdGh1Yi5pby9kYXNoYm9hcmQvJyxcbiAgICB9KTtcbiAgfVxuICBwcml2YXRlIGFzc2VydFNpbXBsZU1hbmlmZXN0KCkge1xuICAgIC8vIGFwcGx5IGEga3ViZXJuZXRlcyBtYW5pZmVzdFxuICAgIHRoaXMuY2x1c3Rlci5hZGRNYW5pZmVzdCgnSGVsbG9BcHAnLCAuLi5oZWxsby5yZXNvdXJjZXMpO1xuICB9XG4gIHByaXZhdGUgYXNzZXJ0Tm9kZUdyb3VwWDg2KCkge1xuICAgIC8vIGFkZCBhIGV4dHJhIG5vZGVncm91cFxuICAgIHRoaXMuY2x1c3Rlci5hZGROb2RlZ3JvdXAoJ2V4dHJhLW5nJywge1xuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1pblNpemU6IDEsXG4gICAgICAvLyByZXVzaW5nIHRoZSBkZWZhdWx0IGNhcGFjaXR5IG5vZGVncm91cCBpbnN0YW5jZSByb2xlIHdoZW4gYXZhaWxhYmxlXG4gICAgICBub2RlUm9sZTogdGhpcy5jbHVzdGVyLmRlZmF1bHRDYXBhY2l0eSA/IHRoaXMuY2x1c3Rlci5kZWZhdWx0Q2FwYWNpdHkucm9sZSA6IHVuZGVmaW5lZCxcbiAgICB9KTtcbiAgfVxuICBwcml2YXRlIGFzc2VydE5vZGVHcm91cEN1c3RvbUFtaSgpIHtcbiAgICAvLyBhZGQgYSBleHRyYSBub2RlZ3JvdXBcbiAgICBjb25zdCB1c2VyRGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgIHVzZXJEYXRhLmFkZENvbW1hbmRzKFxuICAgICAgJ3NldCAtbyB4dHJhY2UnLFxuICAgICAgYC9ldGMvZWtzL2Jvb3RzdHJhcC5zaCAke3RoaXMuY2x1c3Rlci5jbHVzdGVyTmFtZX1gLFxuICAgICk7XG4gICAgY29uc3QgbHQgPSBuZXcgZWMyLkNmbkxhdW5jaFRlbXBsYXRlKHRoaXMsICdMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgIGxhdW5jaFRlbXBsYXRlRGF0YToge1xuICAgICAgICBpbWFnZUlkOiBuZXcgZWtzLkVrc09wdGltaXplZEltYWdlKCkuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZCxcbiAgICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgndDMuc21hbGwnKS50b1N0cmluZygpLFxuICAgICAgICB1c2VyRGF0YTogRm4uYmFzZTY0KHVzZXJEYXRhLnJlbmRlcigpKSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy5jbHVzdGVyLmFkZE5vZGVncm91cCgnZXh0cmEtbmcyJywge1xuICAgICAgbWluU2l6ZTogMSxcbiAgICAgIC8vIHJldXNpbmcgdGhlIGRlZmF1bHQgY2FwYWNpdHkgbm9kZWdyb3VwIGluc3RhbmNlIHJvbGUgd2hlbiBhdmFpbGFibGVcbiAgICAgIG5vZGVSb2xlOiB0aGlzLmNsdXN0ZXIuZGVmYXVsdE5vZGVncm91cD8ucm9sZSB8fCB0aGlzLmNsdXN0ZXIuZGVmYXVsdENhcGFjaXR5Py5yb2xlLFxuICAgICAgbGF1bmNoVGVtcGxhdGU6IHtcbiAgICAgICAgaWQ6IGx0LnJlZixcbiAgICAgICAgdmVyc2lvbjogbHQuYXR0ckRlZmF1bHRWZXJzaW9uTnVtYmVyLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuICBwcml2YXRlIGFzc2VydE5vZGVHcm91cEFybSgpIHtcbiAgICAvLyBhZGQgYSBleHRyYSBub2RlZ3JvdXBcbiAgICB0aGlzLmNsdXN0ZXIuYWRkTm9kZWdyb3VwKCdleHRyYS1uZy1hcm0nLCB7XG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBlYzIuSW5zdGFuY2VUeXBlKCdtNmcubWVkaXVtJyksXG4gICAgICBtaW5TaXplOiAxLFxuICAgICAgLy8gcmV1c2luZyB0aGUgZGVmYXVsdCBjYXBhY2l0eSBub2RlZ3JvdXAgaW5zdGFuY2Ugcm9sZSB3aGVuIGF2YWlsYWJsZVxuICAgICAgbm9kZVJvbGU6IHRoaXMuY2x1c3Rlci5kZWZhdWx0Q2FwYWNpdHkgPyB0aGlzLmNsdXN0ZXIuZGVmYXVsdENhcGFjaXR5LnJvbGUgOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH1cbiAgcHJpdmF0ZSBhc3NlcnRJbmZlcmVuY2VJbnN0YW5jZXMoKSB7XG4gICAgLy8gaW5mZXJlbmNlIGluc3RhbmNlc1xuICAgIHRoaXMuY2x1c3Rlci5hZGRDYXBhY2l0eSgnSW5mZXJlbmNlSW5zdGFuY2VzJywge1xuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgnaW5mMS4yeGxhcmdlJyksXG4gICAgICBtaW5DYXBhY2l0eTogMSxcbiAgICB9KTtcbiAgfVxuICBwcml2YXRlIGFzc2VydFNwb3RDYXBhY2l0eSgpIHtcbiAgICAvLyBzcG90IGluc3RhbmNlcyAodXAgdG8gMTApXG4gICAgdGhpcy5jbHVzdGVyLmFkZENhcGFjaXR5KCdzcG90Jywge1xuICAgICAgc3BvdFByaWNlOiAnMC4xMDk0JyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IGVjMi5JbnN0YW5jZVR5cGUoJ3QzLmxhcmdlJyksXG4gICAgICBtYXhDYXBhY2l0eTogMTAsXG4gICAgICBib290c3RyYXBPcHRpb25zOiB7XG4gICAgICAgIGt1YmVsZXRFeHRyYUFyZ3M6ICctLW5vZGUtbGFiZWxzIGZvbz1iYXIsZ29vPWZhcicsXG4gICAgICAgIGF3c0FwaVJldHJ5QXR0ZW1wdHM6IDUsXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG4gIHByaXZhdGUgYXNzZXJ0Qm90dGxlcm9ja2V0KCkge1xuICAgIC8vIGFkZCBib3R0bGVyb2NrZXQgbm9kZXNcbiAgICB0aGlzLmNsdXN0ZXIuYWRkQ2FwYWNpdHkoJ0JvdHRsZXJvY2tldE5vZGVzJywge1xuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1pbkNhcGFjaXR5OiAyLFxuICAgICAgbWFjaGluZUltYWdlVHlwZTogZWtzLk1hY2hpbmVJbWFnZVR5cGUuQk9UVExFUk9DS0VULFxuICAgIH0pO1xuXG4gIH1cbiAgcHJpdmF0ZSBhc3NlcnRDYXBhY2l0eVg4NigpIHtcbiAgICAvLyBhZGQgc29tZSB4ODZfNjQgY2FwYWNpdHkgdG8gdGhlIGNsdXN0ZXIuIFRoZSBJQU0gaW5zdGFuY2Ugcm9sZSB3aWxsXG4gICAgLy8gYXV0b21hdGljYWxseSBiZSBtYXBwZWQgdmlhIGF3cy1hdXRoIHRvIGFsbG93IG5vZGVzIHRvIGpvaW4gdGhlIGNsdXN0ZXIuXG4gICAgdGhpcy5jbHVzdGVyLmFkZENhcGFjaXR5KCdOb2RlcycsIHtcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IGVjMi5JbnN0YW5jZVR5cGUoJ3QyLm1lZGl1bScpLFxuICAgICAgbWluQ2FwYWNpdHk6IDMsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGFzc2VydENhcGFjaXR5QXJtKCkge1xuICAgIC8vIGFkZCBzb21lIGFybTY0IGNhcGFjaXR5IHRvIHRoZSBjbHVzdGVyLiBUaGUgSUFNIGluc3RhbmNlIHJvbGUgd2lsbFxuICAgIC8vIGF1dG9tYXRpY2FsbHkgYmUgbWFwcGVkIHZpYSBhd3MtYXV0aCB0byBhbGxvdyBub2RlcyB0byBqb2luIHRoZSBjbHVzdGVyLlxuICAgIHRoaXMuY2x1c3Rlci5hZGRDYXBhY2l0eSgnTm9kZXNBcm0nLCB7XG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBlYzIuSW5zdGFuY2VUeXBlKCdtNmcubWVkaXVtJyksXG4gICAgICBtaW5DYXBhY2l0eTogMSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYXNzZXJ0RmFyZ2F0ZVByb2ZpbGUoKSB7XG4gICAgLy8gZmFyZ2F0ZSBwcm9maWxlIGZvciByZXNvdXJjZXMgaW4gdGhlIFwiZGVmYXVsdFwiIG5hbWVzcGFjZVxuICAgIHRoaXMuY2x1c3Rlci5hZGRGYXJnYXRlUHJvZmlsZSgnZGVmYXVsdCcsIHtcbiAgICAgIHNlbGVjdG9yczogW3sgbmFtZXNwYWNlOiAnZGVmYXVsdCcgfV0sXG4gICAgfSk7XG5cbiAgfVxuXG4gIHByaXZhdGUgYXNzZXJ0U2VydmljZUxvYWRCYWxhbmNlckFkZHJlc3MoKSB7XG5cbiAgICBjb25zdCBzZXJ2aWNlTmFtZSA9ICd3ZWJzZXJ2aWNlJztcbiAgICBjb25zdCBsYWJlbHMgPSB7IGFwcDogJ3NpbXBsZS13ZWInIH07XG4gICAgY29uc3QgY29udGFpbmVyUG9ydCA9IDgwO1xuICAgIGNvbnN0IHNlcnZpY2VQb3J0ID0gOTAwMDtcblxuICAgIGNvbnN0IHBpbmdlclNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1dlYlNlcnZpY2VTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICB9KTtcblxuICAgIHBpbmdlclNlY3VyaXR5R3JvdXAuYWRkSW5ncmVzc1J1bGUocGluZ2VyU2VjdXJpdHlHcm91cCwgZWMyLlBvcnQudGNwKHNlcnZpY2VQb3J0KSwgYGFsbG93IGh0dHAgJHtzZXJ2aWNlUG9ydH0gYWNjZXNzIGZyb20gbXlzZWxmYCk7XG5cbiAgICB0aGlzLmNsdXN0ZXIuYWRkTWFuaWZlc3QoJ3NpbXBsZS13ZWItcG9kJywge1xuICAgICAga2luZDogJ1BvZCcsXG4gICAgICBhcGlWZXJzaW9uOiAndjEnLFxuICAgICAgbWV0YWRhdGE6IHsgbmFtZTogJ3dlYnBvZCcsIGxhYmVsczogbGFiZWxzIH0sXG4gICAgICBzcGVjOiB7XG4gICAgICAgIGNvbnRhaW5lcnM6IFt7XG4gICAgICAgICAgbmFtZTogJ3NpbXBsZXdlYmNvbnRhaW5lcicsXG4gICAgICAgICAgaW1hZ2U6ICduZ2lueCcsXG4gICAgICAgICAgcG9ydHM6IFt7IGNvbnRhaW5lclBvcnQ6IGNvbnRhaW5lclBvcnQgfV0sXG4gICAgICAgIH1dLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuY2x1c3Rlci5hZGRNYW5pZmVzdCgnc2ltcGxlLXdlYi1zZXJ2aWNlJywge1xuICAgICAga2luZDogJ1NlcnZpY2UnLFxuICAgICAgYXBpVmVyc2lvbjogJ3YxJyxcbiAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgIG5hbWU6IHNlcnZpY2VOYW1lLFxuICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgIC8vIHRoaXMgaXMgZnVydGlsZSBzb2lsIGZvciBjZGs4cy1wbHVzISA6KVxuICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1pbnRlcm5hbCc6ICd0cnVlJyxcbiAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItZXh0cmEtc2VjdXJpdHktZ3JvdXBzJzogcGluZ2VyU2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgc3BlYzoge1xuICAgICAgICB0eXBlOiAnTG9hZEJhbGFuY2VyJyxcbiAgICAgICAgcG9ydHM6IFt7IHBvcnQ6IHNlcnZpY2VQb3J0LCB0YXJnZXRQb3J0OiBjb250YWluZXJQb3J0IH1dLFxuICAgICAgICBzZWxlY3RvcjogbGFiZWxzLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGxvYWRCYWxhbmNlckFkZHJlc3MgPSB0aGlzLmNsdXN0ZXIuZ2V0U2VydmljZUxvYWRCYWxhbmNlckFkZHJlc3Moc2VydmljZU5hbWUpO1xuXG4gICAgLy8gY3JlYXRlIGEgcmVzb3VyY2UgdGhhdCBoaXRzIHRoZSBsb2FkIGJhbGFuY2VyIHRvIG1ha2Ugc3VyZVxuICAgIC8vIGV2ZXJ5dGhpbmcgaXMgd2lyZWQgcHJvcGVybHkuXG4gICAgY29uc3QgcGluZ2VyID0gbmV3IFBpbmdlcih0aGlzLCAnU2VydmljZVBpbmdlcicsIHtcbiAgICAgIHVybDogYGh0dHA6Ly8ke2xvYWRCYWxhbmNlckFkZHJlc3N9OiR7c2VydmljZVBvcnR9YCxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IHBpbmdlclNlY3VyaXR5R3JvdXAsXG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgIH0pO1xuXG4gICAgLy8gdGhpcyBzaG91bGQgZGlzcGxheSBhIHByb3BlciBuZ2lueCByZXNwb25zZVxuICAgIC8vIDx0aXRsZT5XZWxjb21lIHRvIG5naW54ITwvdGl0bGU+Li4uXG4gICAgbmV3IENmbk91dHB1dCh0aGlzLCAnUmVzcG9uc2UnLCB7XG4gICAgICB2YWx1ZTogcGluZ2VyLnJlc3BvbnNlLFxuICAgIH0pO1xuXG4gIH1cbn1cblxuLy8gdGhpcyB0ZXN0IHVzZXMgYm90aCB0aGUgYm90dGxlcm9ja2V0IGltYWdlIGFuZCB0aGUgaW5mMSBpbnN0YW5jZSwgd2hpY2ggYXJlIG9ubHkgc3VwcG9ydGVkIGluIHRoZXNlXG4vLyByZWdpb25zLiBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL3RyZWUvbWFzdGVyL3BhY2thZ2VzLyU0MGF3cy1jZGsvYXdzLWVrcyNib3R0bGVyb2NrZXRcbi8vIGFuZCBodHRwczovL2F3cy5hbWF6b24uY29tL2Fib3V0LWF3cy93aGF0cy1uZXcvMjAxOS8xMi9pbnRyb2R1Y2luZy1hbWF6b24tZWMyLWluZjEtaW5zdGFuY2VzLWhpZ2gtcGVyZm9ybWFuY2UtYW5kLXRoZS1sb3dlc3QtY29zdC1tYWNoaW5lLWxlYXJuaW5nLWluZmVyZW5jZS1pbi10aGUtY2xvdWQvXG5jb25zdCBzdXBwb3J0ZWRSZWdpb25zID0gW1xuICAndXMtZWFzdC0xJyxcbiAgJ3VzLXdlc3QtMicsXG5dO1xuXG5jb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbi8vIHNpbmNlIHRoZSBFS1Mgb3B0aW1pemVkIEFNSSBpcyBoYXJkLWNvZGVkIGhlcmUgYmFzZWQgb24gdGhlIHJlZ2lvbixcbi8vIHdlIG5lZWQgdG8gYWN0dWFsbHkgcGFzcyBpbiBhIHNwZWNpZmljIHJlZ2lvbi5cbmNvbnN0IHN0YWNrID0gbmV3IEVrc0NsdXN0ZXJTdGFjayhhcHAsICdhd3MtY2RrLWVrcy1jbHVzdGVyLXRlc3QnKTtcblxuaWYgKHByb2Nlc3MuZW52LkNES19JTlRFR19BQ0NPVU5UICE9PSAnMTIzNDU2NzgnKSB7XG5cbiAgLy8gb25seSB2YWxpZGF0ZSBpZiB3ZSBhcmUgYWJvdXQgdG8gYWN0dWFsbHkgZGVwbG95LlxuICAvLyBUT0RPOiBiZXR0ZXIgd2F5IHRvIGRldGVybWluZSB0aGlzLCByaWdodCBub3cgdGhlICdDREtfSU5URUdfQUNDT1VOVCcgc2VlbXMgbGlrZSB0aGUgb25seSB3YXkuXG5cbiAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChzdGFjay5yZWdpb24pKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGByZWdpb24gKCR7c3RhY2sucmVnaW9ufSkgY2Fubm90IGJlIGEgdG9rZW4gYW5kIG11c3QgYmUgY29uZmlndXJlZCB0byBvbmUgb2Y6ICR7c3VwcG9ydGVkUmVnaW9uc31gKTtcbiAgfVxuXG4gIGlmICghc3VwcG9ydGVkUmVnaW9ucy5pbmNsdWRlcyhzdGFjay5yZWdpb24pKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGByZWdpb24gKCR7c3RhY2sucmVnaW9ufSkgbXVzdCBiZSBjb25maWd1cmVkIHRvIG9uZSBvZjogJHtzdXBwb3J0ZWRSZWdpb25zfWApO1xuICB9XG5cbn1cblxuXG5hcHAuc3ludGgoKTtcbiJdfQ==