"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
const instance_types_1 = require("./instance-types");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2 (x86-64)
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    /**
     * Amazon Linux 2 (ARM-64)
     */
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: (_e = props.amiType) !== null && _e !== void 0 ? _e : (props.instanceType ? getAmiTypeForInstanceType(props.instanceType).toString() :
                undefined),
            diskSize: props.diskSize,
            forceUpdateEnabled: (_f = props.forceUpdate) !== null && _f !== void 0 ? _f : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        if (props.launchTemplate) {
            if (props.diskSize) {
                // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
                // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
                throw new Error('diskSize must be specified within the launch template');
            }
            if (props.instanceType) {
                // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
                // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
                throw new Error('Instance types must be specified within the launch template');
            }
            // TODO: update this when the L1 resource spec is updated.
            resource.addPropertyOverride('LaunchTemplate', {
                Id: props.launchTemplate.id,
                Version: props.launchTemplate.version,
            });
        }
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
function getAmiTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? NodegroupAmiType.AL2_ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_ARM_64 :
            instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_X86_64_GPU :
                instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodegroupAmiType.AL2_X86_64_GPU :
                    NodegroupAmiType.AL2_X86_64;
}
//# sourceMappingURL=data:application/json;base64,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