"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const core_1 = require("@aws-cdk/core");
const k8s_manifest_1 = require("./k8s-manifest");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_manifest_1.KubernetesManifest(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.assertSameStack(role);
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.assertSameStack(user);
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    assertSameStack(construct) {
        const thisStack = core_1.Stack.of(this);
        if (core_1.Stack.of(construct) !== thisStack) {
            // aws-auth is always part of the cluster stack, and since resources commonly take
            // a dependency on the cluster, allowing those resources to be in a different stack,
            // will create a circular dependency. granted, it won't always be the case,
            // but we opted for the more causious and restrictive approach for now.
            throw new Error(`${construct.node.uniqueId} should be defined in the scope of the ${thisStack.stackName} stack to prevent circular dependencies`);
        }
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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