"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An EC2 Key Pair.
 *
 * @stability stable
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair.
     *
     * The private key will be stored in AWS Secrets Manager
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.privateKeyArn = '';
        /**
         * ARN of the public key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.publicKeyArn = '';
        /**
         * The public key.
         *
         * Only filled, when `exposePublicKey = true`
         *
         * @stability stable
         */
        this.publicKeyValue = '';
        /**
         * Name of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairName = '';
        /**
         * ID of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairID = '';
        /**
         * @stability stable
         */
        this.prefix = '';
        if (props.removeKeySecretsAfterDays &&
            (props.removeKeySecretsAfterDays < 0 ||
                (props.removeKeySecretsAfterDays > 0 &&
                    props.removeKeySecretsAfterDays < 7) ||
                props.removeKeySecretsAfterDays > 30)) {
            cdk.Annotations.of(this).addError(`Parameter removeKeySecretsAfterDays must be 0 or between 7 and 30. Got ${props.removeKeySecretsAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag(createdByTag, ID);
        const kmsPrivate = props.kmsPrivateKey || props.kms;
        const kmsPublic = props.kmsPublicKey || props.kms;
        const key = new cdk.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KmsPrivate: (kmsPrivate === null || kmsPrivate === void 0 ? void 0 : kmsPrivate.keyArn) || 'alias/aws/secretsmanager',
                KmsPublic: (kmsPublic === null || kmsPublic === void 0 ? void 0 : kmsPublic.keyArn) || 'alias/aws/secretsmanager',
                StorePublicKey: props.storePublicKey || false,
                ExposePublicKey: props.exposePublicKey || false,
                RemoveKeySecretsAfterDays: props.removeKeySecretsAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-ssh-key/',
                StackName: stack,
                Tags: cdk.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPrivateKey !== 'undefined') {
            props.kmsPrivateKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPrivateKey);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPublicKey !== 'undefined') {
            props.kmsPublicKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPublicKey);
            key.node.addDependency(this.lambda.role);
        }
        this.privateKeyArn = key.getAttString('PrivateKeyARN');
        this.publicKeyArn = key.getAttString('PublicKeyARN');
        this.publicKeyValue = key.getAttString('PublicKeyValue');
        this.keyPairName = key.getAttString('KeyPairName');
        this.keyPairID = key.getAttString('KeyPairID');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2() // generally allow to inspect key pairs
                    .allow()
                    .toDescribeKeyPairs(),
                new statement.Ec2() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateKeyPair()
                    .toCreateTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Ec2() // allow delete/update, only if createdByTag is set
                    .allow()
                    .toDeleteKeyPair()
                    .toCreateTags()
                    .toDeleteTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifResourceTag(createdByTag, ID),
                new statement.Secretsmanager() // generally allow to list secrets. we need this to check if a secret exists before attempting to delete it
                    .allow()
                    .toListSecrets(),
                new statement.Secretsmanager() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateSecret()
                    .toTagResource()
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Secretsmanager() // allow delete/update, only if createdByTag is set
                    .allow()
                    .allMatchingActions('/^(Describe|Delete|Put|Update)/')
                    .toGetSecretValue()
                    .toGetResourcePolicy()
                    .toRestoreSecret()
                    .toListSecretVersionIds()
                    .toUntagResource()
                    .ifResourceTag(createdByTag, ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPrivateKey(grantee) {
        return this.grantRead(this.privateKeyArn, grantee);
    }
    /**
     * Grants read access to the public key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPublicKey(grantee) {
        return this.grantRead(this.publicKeyArn, grantee);
    }
    grantRead(arn, grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
_a = JSII_RTTI_SYMBOL_1;
KeyPair[_a] = { fqn: "cdk-ec2-key-pair.KeyPair", version: "2.2.0" };
//# sourceMappingURL=data:application/json;base64,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