# -*- coding: utf-8 -*-
"""
Messages.

https://gist.github.com/facelessuser/5749982

Licensed under MIT
Copyright (c) 2013 - 2015 Isaac Muse <isaacmuse@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
documentation files (the "Software"), to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions
of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.
"""
from __future__ import unicode_literals
import wx
import re
import sys
from collections import namedtuple
from wx.lib.embeddedimage import PyEmbeddedImage

# Styles
INFO = 0
PROMPT = 1
WARN = 2
ERROR = 3

# Icons
DEFAULT_ICON_SIZE = 64
DEFAULT_TEXT_MIN_SIZE = 250
DEFAULT_TEXT_MAX_SIZE = 500

HAS_CAPTION = re.compile(r"([^\r\n]+)(?:\r?\n){2,}(.*)", re.DOTALL | re.UNICODE)


# Icons by Isaac Muse
info_icon = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAAIAAAACACAYAAADDPmHLAAAABmJLR0QAAAAAAAD5Q7t/AAAA"
    "CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH3wcaBCIe+Iy8LgAAAB1pVFh0Q29tbWVudAAA"
    "AAAAQ3JlYXRlZCB3aXRoIEdJTVBkLmUHAAAgAElEQVR42u19e7hlV1Hnr2rtfc65r779SKc7"
    "jyYvICYhARSU8FBE/FBAzfhCUSQqCYg6n6Az43ygMYKMiiLOjAwSEqM4QngI+JgPIwYhwsdD"
    "SCCQhCTk3enupN99X+fsvVbNH+tVa5/T6dvpTkhj73w3fc85+5x79q5aVb/6Va0q4Phx/Dh+"
    "HD+OH8eP48fx4/hx/Dh+HD+OH/+BDvpWv8Dp115jABgZLXLvuRfbfRdz83Dnz1/t2O64vXY7"
    "7oDsvs+5vdvs8COXu+MKcAwcM795XUXrTjEL/+3sYXxu8DN/+nSaXvtUMD8Ro+XTUQ82oR3N"
    "wNQzMNUUQAyxI7TNEsQtox48CGcfBORO2Par7U0f+1zzhQ/ujsrRfP4DlX3g5vZbRSnoWF/d"
    "fMq5ZuEN548AYOqSq18IkYsg7kLY5hyApmAqgA3AlcBZCyIiCECUr18EACT/CoDIwLaAayHt"
    "CNSb3grIDSDz8fa269/XfOrKHeE79Jbe8bLRcQV4jI7+RZdx/cwfrxbecP6ofu7FW6qnfP+v"
    "olm5SNrRk6jqg6raBlkyBSnHSyWSCZctB3ksXhlEIASBOAHgAKqkGUFcO6R6cB2A9yy/++ff"
    "e6wqAx1Dqz3e3KmpS//q9zFc+hmqqg2o+gKIEIjToiYCoSNsIhCRX+LUuey07P15EIF46U84"
    "JSiGcyIkFiKVGy2Duf60ML99+cpf/ODs73yhbm78h2MCOzzuFWD69f/YX3rbS4Yzl/7VT7p2"
    "9EaIPZ970w4AEYH8yiYlPwIxTbhIGhPy5MfZIkhUAMn/CggiLitCfF5gRaxxzQhcD65cuuKV"
    "rwUwmn7tNWbpHS+zxxXgMI/Z37upt/CG80czr/6rN8ho+Q1UT02ByBLBeIFTR9hh1RMDIl4p"
    "SCnHhJVfyl3yk/GxBBUgAsRlqBCfl6wEAhddhojAudGioXr6uvaeGy4ZXfv2Ox+vikCP1xU/"
    "/Yvv/m0RdxnX00wkklc7w8tWCT4KOkiUmeHXKhXnATKuEBMEL4V3EL/io2twUlgHcf41cd7a"
    "O3HaWlgZLRsRfLa9/6sXTf3M2x/cdzHL4+l+8+MJ3AEA7dv+Q9M//+dLXA8ur/rTxAwwG2JT"
    "wRgDZgKzST/GMIwxICYYZlRVBWaGSecQmDn9S8xgCs8RgY3xP5Qtin6PMf6z/OcxuDL+J7yH"
    "2YCJ/eP4d4n934IY7g3Adf+Z9annbR9df/VHAfDMb143OG4BOuZ+5W9eX/e+6yc/T1XvXDZG"
    "AI/qiDmv8gjmgtln4rSi08qO5+nnDvJ47FDWQIILKCyDiFr56vf4mpNsG0QgzkHS6xYAWruy"
    "VMFUb1y68pLfezy4Bfpmr/rhRy53M6/+yzehGb6R+9OWQIY4CtuvUv+rN/FMDIqPuy4AXdeg"
    "BJ+jO+0Ryt9J4cAs3mT2QQQEoSbBB0zgFGaI7iArRgSNAhexRDO6f3Tvl8/rfe+rVyKP8R9K"
    "AWZ+87rB4u+/wM2+6qo7UNVbOPp2Zu/ng5ApCD256fhaB+R5C0FJmGS48PWkpOvdueYFyqCg"
    "wAHh/XF1AxRXs8IDWVHSf04SfvDg0ME5pTDOWWlWDMj894UrX/X7//EU4OJ3/jDV9UfJ9Cwz"
    "Gwp+Gcgm3rBJqz8BumDi43NppWtFIB380fgKHzMBB+eEBK7QEL/Qs+mPz3sL4JL5d0r4SRGC"
    "ErloRaJS2PbLC+/+xad9M4ikb4oCzP7Cu/6Cqt7FZCphZvJC47Ti/aqPiL807RyVggAwgwuu"
    "j9NVxWhhoo2njt/X2GCMD1DIvyCDXDb5UroK52xSkKgIMWQUCMRaeC7JReVo3Whom91bT62+"
    "7XsWFn//BSvfclFARPmzl1z1VR7MXGyqGsYY8mibwYYz8o6IOypDeM1UlUf7lT/fBCXx0UDl"
    "zwtonJiAgPqJPZhMiF9FEfF8j+ZJvd8EhWSQMeH95H9iJBLfG7+78dbLpM/037EK50arlSIG"
    "46MYJq647lW9Dac+JPfd+Jz5q131LWUBpi5+Z7189WuaNZdcuRv1YA0TGU5onpLAGJxNe3AJ"
    "eeWT8uWUrEVWZQZBQMTZ/JMEHRcFBsOK13zAxGBASupXkUNO08TRBcQIQDqgUKLZjy5B1MqP"
    "kYK3Hs5aETsi5+wvL1z1mnd8SyhA8Gt27lVXLHI9MMaYCsG0TxQ84FeLEjxFEMjKrEfQCPLs"
    "nyJ5dFTQjQgmJ38wyfmHOEC7AUFBHOqIwNn0XucUSZSUwZ/rrIPLQNArgAoxbdsCzsGuLP7x"
    "wl/9ym8c0woQVr5bc+lVQzK1MJuKDSffTkwwxCG8yzjAWwVKq5mNCu20QgRfD5H0Pi94CfDP"
    "f8448IvJIkpCpkK8mgWMvzuAcoiXw8CgCOE8F/w9ikggg8AYBdgoeCeeRnZeIfzHtXCtg7TL"
    "/2v/Vb/0n49JBehfdFk1/MjlmL/0L5bJVKAgfE5gL7JlMYFjQASvCAHoRbMPeAUB51UfkX9a"
    "9WAl7CzgxCGkcBAPY/6jUpQ8P2JoJ1nw0PE+onCRLYXOKDoHJzlq8C7Am35vBbKSuKBMXgks"
    "ZLj0tv1/+cu/fqwpAAGQta+++gBMNWDmio0JKzKY/egGgmCZxv1+IoI6gvfsIFJoSCrOTyRS"
    "8hTjFoBUilgrgTb5OldQhH0RCqB8Pq5kABCrGcCIC7IFEIUHIiZwgSxKCgGBtS3EtnBt8yv0"
    "vEv+fN/F3B4zFmD+Ve++nwdTmwhUmeDT06pXguYgzITckRm/SPuSAn1Z6CiSPd00sH8PH5wK"
    "1kmg7u8y5gy8UAiAooABwEYrEGlfbQEUwEvYQFxBHScrIM6HjFEZBLDO+s9oVmBXFl9oT7ng"
    "00c7RKRHSfif4MHUdxOIo/B9WJeTMXq1s1HIndkLz/gYn0i5g2ABMjOoXIFWBuZs8qnjEnQk"
    "0M0ITsAAifGL1qCbCwjK4oKwU7yvcgGIFHBUEhuoYedS9OCcg1NKEK2Ec85HB+2QsOe+TcMT"
    "z9t7NMmioxpvzl/tKrr+iv/KvcHz4wqkGFvDC5oUkKNE5GQWkJkUoPOmP1kJnQourEHX/FMh"
    "fK8snAGgShzpfwtiN/FDAnAsD0MoCAGIc5rYPyYPAEkxf+BgAQAGQ9gXk4hhjx/YwJAXMjOD"
    "hABnIcQQ498XPpBc1bey9tR7l97xssHj0gJMv/aantlzz5n1mhNuoboPCilUSqlVLjEAM2I4"
    "mNg/Zi+7AOhSREClr6cAHqOQSVmEnBfQdBeNs30KCFKKBjqGoAMSXagEKvIAqlBEXGT+clJo"
    "UhSQeACn8wT+s616zYlnDa0TiLNw4lppR7fsederLnhcuoD1l1w5pMGgYjbMZDwzRgcXPquE"
    "TokNctIn5QMm4QHyZGYmgGJsiIPwAV0cMJ4SKBVBUrGHjgiK1LAg+fqEBQJIdBNAn7OdzKBT"
    "GcPwXASForgC6xzEtXAOcM3KG9szLnzr0cgiHjUFWPfqv/g41/3nM5MhYpjKgCjQtZzBHxEB"
    "gXbV0QAor/qkLKrsS8f0hQUAISYEWIE+bQMiOBwLAQ9SHSRF8idmAF0RFYiuE1S+vljxoiyC"
    "RAvhVChYnu9sfM3CqXOSElgLEQsZLoG2fW3d0mnPPXCk9QRHrADTr//Hfr1v6/Pquv5n6vVL"
    "jhy5CiclfBQO4A4jGNF/YfJpEgfAaaWzonW1dQiOeQz8ZSZxQjlYcglI73OK1gUEsTQgcwCS"
    "zHgGgGUW0AVzHhUkAT1RINAGXqCrBNYmKxIVwLXOuna4Y/cVl5zyuLAAGy69ouH+rGEmopDk"
    "SGVRYfUjIfuc5dPCJyX46AZ0KMiR49eZvpQa1hYhU78HxQRl5UfmB8gLePNAcO6JU1gajnDH"
    "Q0t4sO2DpYUFK4IoKIPCAFGYhSUIws8s33geIK30jhKIRNPvIwSxzoeGwVrYZuWy0YYn/uGR"
    "hIZHpADzV7vKfObqq0x/8HJmMsSmFD4pf8+Uij2i+U/4IJh4wzlyiGCPFTdAxQ+SJaCOv5/E"
    "EXRj/0wZI4G6Hz9nDq959viiWlgZ4XXXfBG3HqjBdb9wE5no0YxgiP2lLA5xhel3KQzMjGB+"
    "LNYmnOCsUyGh89bBOchwATvf/UsGgHvMFaB/0WV8whfe3R+99PIl6s34QsmYAmUzQfg+harB"
    "IMXXKMT8UWlUdRAU6idSChLDR8ohX2IKNQWMUgkIMvE2XHXRFpw038/VRh0+gIjw5r/9HK69"
    "D+CpGVCoG+zWA2pTr8vBXIrrs1I4KVe+zwfYbDmUhXDWeorY2gASW7i2bW3b/rN9zi+85JFW"
    "Gx+RBdjw6is/zfXgu0xlDJGBqQwMcRI2aAL1G0NDJUyT2MCoOCHDR1RU65Iy7VEZuGP2tUUo"
    "dgnpWsGO+X/7D2zGGRum0TOHvh0/9kcfxb650yCml9C+9u35d5esQMoGapegQKG48TSxczYl"
    "iCJL6IJVsG0bQsQGGC5j5b6bT27OfPaDjwQQmkcY85vBiaed2Z8/8a1U1ZxXPifGD6xWN08W"
    "Pgfhx3OYKSWA9GMO72HFKcRii2jq9WuFcrDeSxC/k3+9MozzTzD4qaduhOFDC7+1DudsnsM/"
    "3rQN1WCmsz8BIBWFEhShpckpHZ10cIqnmTI3kR4Xya2Q9hYHCAFkLE2v/d5977r4zx+ziqCl"
    "d7zM9k9+8ntR92ysnkl18nHl8iTuH4XwU20+QQlGcQIdIBmVKyobEcEYpRTRqoz9jrB/QCkV"
    "+dv7yqefAOtWZz0rw3jaE0/FaP9DIHEwxqAynJTT6AhHha5Jmf0NUIqvy9/ifofOPYvAGeUi"
    "osizGDJVxc+Y/YFfe2LohfDoUsH9iy7jmWbf2or5mYjmPpjdyORFypfD/two3PSvEn7GCjo0"
    "zPUCKW9Q/C3yHkLlD5CKQ7xwdWgIVtvFUBaPnLNpJsf0q101i7tAEDATxHnF8ABPfF4jmHPD"
    "8Glg9lXIzoW/bhhwDobZ+3MmsAvkUUqQCcT57+iYwCDAMZzYcG8B57xyu2pgB6dfcNXOd7zs"
    "ux91CzC46DKYLRe8B71pm7j96MeJFOtHiccvwF50FR3h69XNandO/Owo/LjaUsLIhHMo1Amq"
    "kNIoy2CYCzdSEER0eFBoOBym/AIVSk2hTpE6Fk5bg/FFwUUInEvhY2Ks3LWUQ+loVYlguFl+"
    "HoB+rL181BRg38Xs2A5fzEzGX5TJ5jvczDHUTyjoXiIosEcTUsRa2P65ijtmM5reIHjtVkzM"
    "PCaFC+ebWHTKqIwHrAsro1Djd+gjArV6zYbAdKJwUVpRdRbUu6qg4OSjpNJNRrcYXZ/aE6HS"
    "4FoRTFQOE1zwYNZuuORdV9RPe6l51EDgt73pugHOecFlVd1/HrMhrny1bKziJbUHr1j5zIUJ"
    "N0lpMlBjfUFQK4MprepS+OHHxBsTsoVQq4hLV6LdTgRSJ9QNTt8wvToQ6Bze/dFP4PbFHvrz"
    "G0PUokKp4Ia4Y1nybuZ8bkw0ZmJSVFimoGJiMMPuJvV7LEohEcAJkR0+dc/bfuS3HzULcOtv"
    "vWCFm6VLiCGkTK/W1mQRgGzWlBAiR6CZwK5p1+bRKAAZV1KB/DG+mZM6UYTh7AJMcAfeClR4"
    "1+e2oza8igjAojYGV33yFgzWnwxTVen7GVNGKlHxOVgI7XqI8wrngvdQi0WFvByBs3pPVKrk"
    "XjxYJK56WP+qd758+rXX9I66Aky/9hoz87yfO72q6k0Ew7GOD5QFHWmXlLJl7dOQwGJp3lC4"
    "AL16uasoHPFGtAqULY16j1Go2zCVFLNyI8SMwfwJ+PVrvlhU94z5/FGDyhg8+zVvwQlnnY/+"
    "7HxgPCn76aRknK+VGYaRvktaFKwtH+XoKLnL+B6jLBh38IHCCenaKluT/NrhFIysWgH4lHPN"
    "9Lnf81uo+85HehH5m+yfYoiSfDjya9AFoSjAUF4h+kZSx3yXmICUYI0CRzrkyspkwjlUWgXD"
    "oKrG9mojfurP/gV3bts18do/+aVb8J2//MdYf+6FmNq4BVVdBz/PEyIaKhR1kkvisfCPi91Q"
    "yeKxCoVJr3qVHk+WE2Amw7Z95qPGBG6+5P88xIO5DVXdy/7fVAU5E3fqMOX6/giQSnJIl4l1"
    "MoMdRKxXOqlkECdOP59bMoEo6gIPtm1cRIB2hOX9uzDcdgfO3zwDIy32HFjCjffuwtT6zZjb"
    "fDrqufVgY1SmL9K6OfvnuvRvJ9OnGT+nN4q4zB7GeoCyLsAmKjk+b52D2BbWWv/ZtoVthxiN"
    "mp8/6ZQt773p8pcMjwoP0L/oMp77tz+brphPgMkJl7RVG4qRU/4qmveE9pVFiICtwAqFRpc5"
    "ASIdTehCUC6SQZOSRDnj2+kSolWD+5hZvxlTc+txbzuCsyNIH9iyqQJVfZCpAx6TtF8g/TCn"
    "58gBLpR8MflNo8yBAwhxPxPBwZeJOQmPScDw/xITWAJ/ERSJieCEQbD5vjHBOg6nCZwjELit"
    "CD930+UvufqoEUGnPvGc/v6T3vqrErj+tLkjugBWHD3rlcsJ+eZYuHyuYMyIipWdzWCH++9U"
    "BXP4wG4BCaibCh7fMZRyAgFxmWoKzg1yjwBdNi6+8DsxsZjww6FymAiOBewYDkEZEATJDHbO"
    "KwGTJ3TYp4qJCAyBi5RyuEcubYpjkFjfBsf5f524fJ5hg5XRc44qBvjGH/3UMtvRDxHBgrSw"
    "fcEFqxtNIJDoAk50zDgyFlAAseAQukQKl74vol9juPCnBRYI3EGtsIPp8AvRDdUc0XaM09kj"
    "e8MFmaRB5RjOUIpZoHtGAnD6fSkaiHgoLghiL+Rg2RINDMUHaCAbyuc41UMIVVXVm33uy09d"
    "DSm0aiq4ssOnE2ZigrZIeGjOvzT/lCp4MtqFCm86uYHkIjARLMW0cRF5kM4SegUzhuCEYNoV"
    "PH1zH2evr3HafI1NczWqYAAMEx7YP8LOpRZf3b6CT9+3BMysB4nzRjY2heC4ESQXEbloOwzB"
    "2Uz5ejo3rviYvmE49ubZm35JldCOghuAC+7B5vunXAZRaHxl22BJgsIguBLyboec9e6vGmD6"
    "2777Fetmen/ydWDliBQgtHGZ4aoeMEk264kEySa2qOMr2KuOJdBVQazP6eIAdDiFnNFDx1XE"
    "nIFrVvCDp8/ghWfOYH5qfQjvcimYdgBb1g1w6lrg6afM4pXPIGzft4w/vO5u7KvXgau+5/rF"
    "J93SLq/Angn5HcHMlDcKE4Jvp8D/I/D/nvGxTN4lkPXcfsQEiIrBEFiFXSQtJGslm/uAY4gA"
    "w4xWRHEDDAvXkrTft/3pl/4B8AtHZgGe8X0vNV8/8ZSXiObPQ4497XlnLbC8+1qvbiQTp5QA"
    "ivBQFkRz5zHJNKYUnX0Bs9zg0m+fx1M2bSgEHlf7QUMg9dLGuT7e9p/Owd/dcC/+fquA6oFf"
    "ebr+PwK/UCPqoiKA4BhgF5pJ+hJ/MAlcLFwBYEl8ZOIcSLzAnDh/X10AgM4nj8AEcsHXk7dq"
    "RN7FJowlLj2OJXNsmA3oKfsuZnfELmDbAw/UlbjnU9VzxMzJhxP7P4xMAOX6O7VXL2ir3tSh"
    "lYA1mCv2DKpzY2m/VoogvcoA564jvO5ZJ6ENad3V0LoTefGg5D/89CdgcenruO6hEarp+WCK"
    "Q2+hAPYce2UISM6jfhCE/B4DJxHg+b4lKeMXIgJfzeTP88CPfCYwRASGCa2LC41Dn2soV2dD"
    "2CtgQxBLECawI1gh4nZp01EBgXf+jx9ZAtx5BBECq80Y0dcr86M4e53QoA5TmPZqFCCQOmEh"
    "JmAB5CQKyLNsIPTJF8JUfPS2Ofz0c87GaPsdINeEayupbRMjGVbEjHJ1RUSTCjvUdRauTop9"
    "DLl8Le92Klyr+nuxWDaCwyAf4t4Aa57x0k2HAoKrigJY7Bl5R5aifCPDpws5Q548fXElPFJF"
    "H6lyJiznbvhXsGOqLsBQ6QZ8PuDoN99srMXFF56O5X0h95+ycCjwChNyPmJCOXvkMVhFL0xl"
    "NZCmikvl4VTKroVOqtI5369ckhT3XvSf8sLnnrV5XX00FGBTUi1IWZhJUF+wjLu5s3Vbr2SN"
    "9knt3ipvQI4YoOljKgmkQ+XzNau22qM2Bt/71LPQLi8A4lTpORc5/FTRFGltLr+/V1BtIUS1"
    "t5HOtnd979ijf90XObW/hZJDd69k/IAKxlQXDGc3H7EFIANUzHmfrd6VU4I5GkuBcrxo0nVz"
    "ZRNHw1r7MVZbAOokUzCuDJOyd/q4Z9tD2LX3wMMmfSZe/N77Qa4FB8umFlqRxTQTFBmYYK2I"
    "QKK4E20tRPMCkje8htxxlwAjIOCwgEeg2+KJhciZ/bOeKY8YBIYQ0MSa+mRmRBKq1SYq/x6V"
    "Q0JQpBRFRwCq0QNzx6RFhi9Qqcki6NAQ2QppwVfG4Mav34W/+8xN+Nzt27H25NNRT83A1D0s"
    "7tqBP3nl83HyhrkE+h7uWFMJyLUgDDwIDLWEQnmDqCf/QpmWpdS6TkL8L+L3CbugwBLDv/Bv"
    "ak0RQZyTVPJuE1HJiAyE5wkoWCZPG7uweFxcnJ7A2rh16ozRI48CRHjt8372LJiq2FwZW66w"
    "pl11FWyxOROdDRt5v16s0IvnJ0ZQcfipQCRtG+tQxR3z/xd//0m8/9++hvVbnoT1J5+Ns5//"
    "LJj+AKaq/XvOOg9vvu5u/PnLzl+VBThl/Sy2i0tuKNb3ifj6PV+Z4ZVdImcPX6QhgWpmAly0"
    "VHqRhM+10SUGJaFgacSFRRQiA0ggfQQJO8SEk2cQggVB2J42Wt54qFDwkGFg78QnrIlSIYlV"
    "LJQ2RnAssgxERnQPRm3ILHw762bOShE0PkguAN5cJguQ/0biGgAMLeFtH/kM/vWmu7Hu5NNw"
    "9vf8EHoz86jqHtgYoIOgTzj9HCyNWkzV5pD4YXllBXDOWz+EzSBBCJGVQxROEAin7JM/yVlv"
    "HcR6YYaSz3QPo/WMNyXTQZIigLL0KBeKR2xlkRdPSm8YPmQvgYdVgHPPO4+3bdu2Nu+IVP11"
    "JHFeIfSLIYhk/6XBS7QCImUU0KmPj5ag2wCCNeWr+HcBcMcBRoNN2PLtWzCYmgNXdagM6uKN"
    "/LkPLTbYstYcMh++d3HFN3QM8b1EIWslCPG4z4EQYkMREiR/H5+LSab0eviOEku/VKkYB6IJ"
    "arNqum5yKlzsWEQPEom4msbEnriHYQEsmem4Ev3ydGVCgnNal9KOXGXyI3EREWy0DIKipDyG"
    "Wpk11KBHUil0ihqgIgRjMDV/Qtn3P4RnfhV1kyfASXO9VRVD7Nq/jJM4p71TRk8kCdtn7wKr"
    "HzJzCSgG4icJlAlkFZAMzSlyTwTfRtaXnAd0zwA5ylYjgXEJ1K/NFsQ5UKApiag+lAIcEgVJ"
    "O0rJThIXgJcEX5Y7ZhbzuUiHLV0EW8azXQKEAqwdKzErVkyJwn32jlWCKSuICbG6UcknJqBX"
    "8arKwW1vOtX/FUCUslvyq1LGYnXW4XEUW8ROMX+hrjFx/zpiAlT4F87pEF7cbZYFDoWizh2q"
    "6OdhFaC/skfsrvuW4VwWZNxXp1uzBUWgYtdtnN+jCQ0aU5K8Hz/6wfFQKymOooGZ9DlIVqGI"
    "y8l/tkn4AWA4zLjlQwreOYfd+w6gPzPnFaD4XP13lfBBnfqHbM67xI3a0Viye50djKUSKFdM"
    "OTz09zDsUJbQrk5ExA5HR6QAN3zjAbEPfmMhNj6i6L8UBMkt2xX3D0WBqhg2milOSEVUdU8Z"
    "Q5M6jwtrkVeMjhqyUkh+nJQpCl9gCNgys7o9ALfftw2D6bmcx0BX6XI8z2EhUHF9UKBWcyEl"
    "E5pcbACNfoW7VOiS42yNi5R15VxanjCYOKAdLR+RAoDIDXfcvYts671fqGPLSZ6y7/LBXQHG"
    "zmW1YphQ6P04iFQ3X/QmTFE/2jWUShEfm3CjT1tjDrkfkIhww233Ys3atd69JNYOyUKxxiiF"
    "2dcJrJwQK4ZWRLfazRNIRvZQljGxiJ3hlzkryvmzYn9C8MrD+f9DKsDwI5fL8N6blhA2HyQz"
    "JMG1hAJIUqCk7NfbTb3qhFDGEih8oKY0JX3BbPKDn0xZRirdRBQOxJv98BmsmLUz1lY4lPtn"
    "ZtzxwE5UgxlFPI0rGanOo9p3J2tU5E5KZY/9DrMgpVOzIFkZRDuE7gSETvdy8krgRku7zvil"
    "d/aOhAoWACztCiS1wYiAsFthm+Mc6qxm6nZkgXIlxSrKilOQQZRNnKZ+Oa4gyoqSHgfQyAVo"
    "9AqxabZeVQRw/94h6t4gYIuu4OPnUfp7UOVtGtqhI9BYN1H4eJrQv7C7bRwoOANW54lzvhbR"
    "2diZnJ2z29qlfXQkCgAA1sLs9/3JrC9aSEqgO2l1VjJSZ8WxeL8o1JROrybJ5jRanez7KQEe"
    "fa8iNolALydUJD8PwMChapcPuSHUOYcDC0uY23Ai6l4v/T3Wrk/NE+D8RrW/XxIWmMh5FNem"
    "FlaMsDp1FoUSsGp5E9rKJkURrwQsLbml/ffzcFGOVAEcbLtTclfDsEXNqXCwW25FnXJsvQIk"
    "u4q0WqlEt/E87jRdoK4flk5TBh0NKBAm2RWcMkOrAoC33n0/pmbW5GJSSAE4S4OMzl5AlOBX"
    "nxcF33k+p9qQ6886uKqMApQyiA/RY2MpcgLYFsNtt98+Ndxtj0gB5p70DNs2zT2wbTAz3hKQ"
    "6rNftGLXAKfI1OQ4uOQQSCkGJatChVfpgCUtANFgqgSFSfgRsQM4c75aFQC88fZ7MDe/NuxI"
    "RunOREpwq0I1KEVJBBdQ0OLaepbzzaVY4eVrGG93HxlZXacmDiIWZEc4cOO1t9764f/dHJEC"
    "kKlds3fHrQCM/wNWDUVyWUpOCliSd746BW6U5kvJaCf/X8TCkunR4jlk/kAXlozFzBK4gGyO"
    "T5kzqwSAu9CfnklKRMrdlJFKLMWTh/3cEuBRcS2E8W5mSdEC9CqUQZB2H8V7IVH4zoFExFo7"
    "ivWrR6QAlRsOl+/5yo1kR6BzS7AAABgxSURBVJDQvjxmm+B0IiNcoJQTNcay9WmwM8YFW4RH"
    "k4VerqTynMItJOtQ+u6TQyewQx07FhpUvf54yEna9ejvrECoDk2BsW7kmhEt3KIiy0QUqI5a"
    "phpWFu4Dwfw7C/Lt5Jxt2/sALBxxTeDu2760vHjbZ++W4RLg2tDY2IZQUO2PKiKBLNDCl6GL"
    "/OQQAciE9m7SYb+QWbACKCXyJ7slbpZWBQB37tmH2bUnoK57Pv+ggalmQ/VlA2PWKoE3Kv/N"
    "1y7jIXPkWRJxRgogdgdgW+TpJNH8O0AsDRf33dI/+ezmiC0AAAtg2IC2Oeec2Na3NHfOWwRx"
    "StidxgbQN0gK01aaRSlID0jpU6F9Zy59UYqg/WpJmWqy5eQZszoAeNf9GMzOwRhT4BAdstAE"
    "pS7HFEphogtLIB1lnhAmFtrVAYZR2P4OBcHb1jeXdBawDS9vve1z1CwfuQUA4Oq1m5ZGC3u/"
    "ArHBDViIbcNFuSQ0UaPToyanVuoHSUoRMoAZQ7udC6eOBSgUo4uyi56//r2nrRIA3nr3VszN"
    "r0vp7azAWkknWSBMVBIgdyGnDheCQp8nkGcd91k+9KBcrA0uoAVsC4yWcOCGj31l5aF7DxwN"
    "BQCJO7B011eup7ZhBAWIaDMOS8rdSjtgMH9GcRGTOnbm+bzjFzvuMqQY3ToWOqkbTGHS58lz"
    "q2MAb71vB3pTM6rWTjpuDWPuDYWLUt9ZW4KDCJMmWYKD9S0SAKqXcBQ+bANxFta2rmnlAQAN"
    "gOHRUAAZ7Xto74GvfPwWaYcQa0WsjwS8O3DBF6EYsHSwMSylPHOL1RLpdoQqUliTMV87pgRa"
    "OfK0r5PWrA4A7lqyMHUvVOy4cZyjW/MWFkrhHg2GMc6F6PeOK/IkRZByZA1UfwLb+IHW1gHi"
    "aHhgz+fN7LplAO1RsQAAFgE0w5XhLeKsiG285gUMIKHR8dgFiBqrJnJQU1as6O7FF9u0pdPH"
    "X8aQsVaqdGOdQztaOWQJeSwbn123wTeCcKVVc/H7iFrdajyc5OGBiREtLIaUijlR4GO/inKx"
    "8b2BlRUBbGgc0bYQ14LtiPZ89kN/b4fLewJ+OyoKsGKm1ywubv/GP4lrvRuwje9d6wJBJA6I"
    "vkjcmBJ0V2iKIuBK7Q99cZPXVEqQLUKOgQV6hNu4ksWau1mzulLw5ZUV1L2p1IwpNn5wExRR"
    "UuTh1PfE2ArtvjYpL1II3LnOglAnhzkCIuJHytkWEoQv1oodjRZGu7ftRbOy+1ARwOEoQGuX"
    "9u/cc/37rkezAm8FWrhgCSR2uIZ4UxSLUSYMZkwERtdLaGTcGdJQzPYrFo7kqZ1StmWXTsfu"
    "mla/eyiXfAsc1KzgNC08JFuDEqyvHaS12V2oczNRM8HCQbeal3JknXSvW1RXCoeIxfwCbMNs"
    "oVaW9u/+V56aawHsW821rlYBBMBuALKytPgl1zbi/2j+8avfjZk4SctHxluy6osHOqNY8jSO"
    "OICpuGnIw5nGzXDozYMssN4qrpSZMTUYoB2NYEVgdffv0Pk7g1ngwlMq/O73nogzqv0YLe0r"
    "ZgdAKYoUlk8pkjs4TtJYyrkc4+dZAi7de9c2gG1hmiXe8f/+7D1uZXEfgOXVCPZwegXv5d5g"
    "tO+r//p/py/8kW8X2wbU2YCdgbMt2LCvbjXOu6nQMXNsDKMTwKjQULKp1qXXIrnkOtLNwsoj"
    "aC8btm/HF/xGjdjSRbDYrL6T+neevh53Ly1ianaNLyklrwBOgI09i+edPodzN8/CicBah8W9"
    "u9AOp8CDOX/9HauRTbsCclI2l0kDpzuurFgcDoBtvfCbBi4IX5yDaxs3au1WEWkB2bkaAHi4"
    "CrDiRiu7D3z9c3dv/I4X7atMPSemYccGYgyEKzjrW5s468ChFUecj1eYtmBiKc7jC2hbHANG"
    "SZacb4xEyo36avRsaY3KjEJgEBoykW+2LAxYETSH0RD6FS96Fv79jgdw575lkKkw06tw+voB"
    "ztk0CyJKXAIToZU00i0JzqrQL1sBJdDoFsJicBNAZGHZIvMaQ+82+H7bePPfjkAQ3nf3Le/l"
    "wQzcyuLO1fj/w1UAC2AHD2ZO3f/AXdesO+O8S13bgNjAVRUoWAJYC7DxN58sYEy6EApDF6Xw"
    "eZEmomQdKOy+0W4AahOFqNG+3tIg1cW5sMMmWpe4UWORD2/e4tPP3IxnBCDY7SXc7T8g40FL"
    "ahun+Y0ClwRB6yghjZBLUYdLH+hxjQ0TQwL+aj0Gc86KHS03ez77t58DsHe1/v9wMEC8zp1u"
    "ZfHAzn99zz9JsywRDNpmBGkbuGCWIkUcY/DY666L+qUzUydN6Y7Bo+tM2tLIX93wDPxQDGVO"
    "CB6+i8e2PYurbg2v9w3qziYHvzlxVFz4UaNfoOYHZ0AnWbkmAN44PhZxjpC1Ae3bcK9HcOFf"
    "cY7279z+QR7MVAAeXA0B9EgUIPIBO7k/XS08tO39zlnynEDwRbaBa/3vrrXhQm1aBam/HnKo"
    "6IFz58aIK4FeMYevO43TFQmRNJxJOtO5RPDJ2x887Nbwq14dYwOkOteFcoaQLrB10hk66fLE"
    "cefaUInVwtoWrh0Fv++JOHFOZLgoD157xYfcyuISgO2rNf+PRAEsgAfccGll+z+96/0yXBJn"
    "rTjbwLUNbFAC/8Varwyhm2USlkL62W9mISXU7CaMZ0/6oWbrBIQeX3cKtaeumsEc37E0wIGl"
    "4WFtDz/kDXEOdVUVtEdhup2KaqSMdkSHgQosOlXdI6HOz7YB8TcNXDsKPEwL5xrat+OBa3gw"
    "UwPYczjm/5EoAADsArCfB7P1vm33XCnOUlz10o7grFcEF0kKZ9PKTlpfdNuUlE+Qbvye2qqW"
    "N82l4czoKEJutSpBwaLw4+j2N//9l31nLfvIB24659L7/+2LN+HDn/4KhGv/t9Tqdh0XJ3p+"
    "oOTRcogDpiIbGTOtIdHjAudim2EQfrS4rchwuXnwX676gFtZHALYGnIAqz4eydAoB6CSdrRh"
    "8a4vf2Ptuc9+MVdVTf7wWmVU6/O4H5A6nb30lK+4j2/S1rFuQeWkUqkJ9XllVVLYf0+Eemoa"
    "H/rUjThj/QAb184lQa6my4gNe/pFBO+/9nq8/u1/jZv3AutOPwf1zBoQs5oFLIXiJ0UIeCQr"
    "SmYUrbPBaviowjUtnGth2xFs8Pnxd2kaiGtp5203vG24a+sOaUd7AXz9sVAABJCxmXuD6WZh"
    "/xfnTznrB2EqVfWaR8X61i6m7OcXz+Nyfg8X/0q5sxflriB0hy8UmUBSlbIx8ox5V8Zgdh7/"
    "ftcuXPvpf8eUEZy0YR5V9fAB0YO79uDaz3wJf3rNx/DXn7oF9y5X2PTkp2J+8xNQT8154UtJ"
    "SzvRvlxZL5Qj4hAfQzynL86PhmstnG3QjoZwTVCCZgjXNmjbxjVLCw9svfbKK6QdWQDfALDj"
    "cAX5SBERAfi28ENP+NHf+K2p+XUXVHWPTdVD3R+A6z6q3gBVrw9T91DVPVRVlTqMG2PygAnj"
    "LULVHRlTjH6LbeQ7ncO6DSVji1XOuxP09PhYzOnJkxEW9+/BcHEBe7ffi42zPfQZ2LhuDrZt"
    "ce+O3TiwMsKy1OjPrMFgdg5Ta9aDBzMwpoZQDBORTL5zCoMUbkzQdiaIRpdlYw8j28Jahza4"
    "UDsaYtSMYJsh7HAF7WgIGxVhuIjb3v9HP+3aUQNxBwB8ZjUlYEdLAQBgFsCFAOYA4OxX/M4H"
    "TW8ArmoydQ91rw/TG6Cq+0kJTFX7XvuxzbzJLeZT733KMwOoMzjS6M6iQNmnF7GbiN6y1R0m"
    "iXJ7FlHKWXg/61ecs63fzw8/A0GMAZGJZgtxqHSOOEpAWkwHDW3gbXjOppbxalR8mAZqrYVt"
    "G9i2hW0aNM0K2qaBHS57wY+GaNsRbNti3/b7PrjjE+/5gBsuNQBuDT+H3S7NHIECNAFEbuTB"
    "bK9p2y/Pb97yfYhdQqL/JpSdu4v5OYSx5hc03mJGq2qsLFZFxEUNXdn9U2UEgRJMhogBoecP"
    "2IBMBap6MD1vvajqgUwN5ir1YdPsnQRsEBtDOidj3EZh9q1L38eGdK5Yl8bERiW0wde37Qh2"
    "tAIXMYBt0Lat2JXlPfd85E/eIrZxAPYD+Orh+v6joQAICYcTpB31hzvu2jl9+tOeYPqDU72s"
    "qfTXRKolSmdlhqZLqcSrW0/I1Km8VecKJhZilvn0csRrUbKo6y517K7+mOtGLeHz8uDnblTj"
    "IxdbTArP56YoxbrM7lkL1wZSrRmhaYawoyGkHcE2TeBWWkE7otve+3sXs6kg4tqw8nc+UgEe"
    "qQLYkHQ4EYDZd+tnP7P+Kc/9QRDXaZGmuno1Lyh2vmcBkZq8NWGOn954kQsuu7tl8nY0kckl"
    "WmU9ioxZA1Fo/KAhagzfVGiZ/oWUz8nk1LQLs4NhHZxY2NbCujaAvga2GaFpvd93jff5MeaX"
    "dkTbb/7CW4Z7djzoRssWwENBAew3SwEAYAnADIB5NpXZ+eVP/N3GpzznZWAjpPY/leGa3h8o"
    "KtzrWAnkZkoYU4Jyp9B4NRGKhExyB05KJQApBlG7CZfj+hi6Baq5oJ0jDewmYIGClHJpRDyc"
    "hXU2oXzXeoKnbYZog6+3zRC28Xy/DYTawq4HP7Xjk3/zYWmHDr4N/E0ADhyJ8I6GArhAEW8U"
    "cT0ezNYre3d+av7UJ704N/HLGzREJtTCxyaJcSULpWwYaRBAesVPMPnUMf/SKbyIdqKzx1EU"
    "qBNV9KFDuZyP0EoQgR5yaIfYmdSfm0BfmPztXAuxLvl6Z1vYpg3CH/pahMaHfdI2sLaFta00"
    "y0sP3PXht/2Ouud3ALjvkQC/o60AADAKZuhEaUfSLh9Ydr2Z+2fXbbxQig5gTrn3ONYdRbuT"
    "ScSOKJ+d2yNI+L+UoK9bYSsyIbuO0p8rv59BG1INoJOyuEM0ANTDoop/XUroxBHwNvp664LZ"
    "b2CbBrZZQdsE0BfMfoxGbNs6Nxqu3P6+t1yq7vdOAF97pMDv0VCAmCgaAFgrbSNLW2+/rzrh"
    "NDu1Zt353fIm6lTSUjTNnPsb5MaESGnkDN4ordI0UZMkuYzsJTo8uyJqSqFrdq6jHFCVSTqC"
    "iCVjMTcRPsNF0kc86rcSUb4NtK4vorHN0CP9UTD7yuf7Qg8vfDRDd+vfvPnlbCoW77+WAHwl"
    "oH88nhQghiTrAEwDggN33nhz/+QnzfZn1jw5bQwNVR0kKGrniHxIpBtRpQENJJ1+AuE1pgIL"
    "aCWJFb0pBFR1iKWQtdnvAMNgBZwbdw9OpZ4TsSPeIrhQNu8Cn29jzV7r0Xw7akJcvxKEP0Ib"
    "Vr0v9mhg29aJbenmv37TT3JvYFw7cgFw33y4Gb/HSgEQvuACgBMA9ABg/+1fvKF/8pNmBzNr"
    "nlxsklDbvhHCJaLJ5eIo9lZ0iigm7sdwY5XBRc1hR5i50lsS8HPqXKfPi+bfuoLMiVlKG4Se"
    "ZvuFqt3E5Y/86o8r3wUr4Jx3B861cK11Yhvc/J43/QT3pio3WrFhgd0dfL89WgI72gqAgE6H"
    "QQmqqAT1xtPc1Ozc+RLaN6bq1jSMx5Ul304tVVLl5U558oLgQTHLRZxAlRyqDSilwJNggxkv"
    "c/u+zAwFfZt5e3ECZ2PEkBm9IMSO8EcB4I0CpRtp3ZA9DcUerm2ds21z83ve9HKu++yaFRu+"
    "7vbg90dHU1iPhgIgWAEBsD7+jQN33XSLTK/dOrt+44UAi0gonwwbHSjuLQw1dj5ks2V5dOiB"
    "R4KinDoVgBUlWcEP6w0iROMl493Us04xO83vQxWduGzynYNrfdrW2jYIvw3Cb2BHI+/rmxHc"
    "aIQ2Cr5N5E4o8XIQZ6Vthvtu+es3v5Krmp1P8sSK7K8E/49jQQEkFCZwwAQMCFYeuu+Bpd0P"
    "XT9/6hNfLCA/NSmZ5Li5wgYBWmUFwo1OK9iVpebF6gec2LL4UlkB6aRn0UnL+nNc9vHOKWUI"
    "qzxRuCFX7yxsKNnyK3oE14QV3zaKx1fpXBvIn7SxRrB0YO8tt1/zB68L1UHRIO0HcMPRAn1H"
    "Mxm0mqMOGcMzo7LxYLbnVhbac1/x21eZuj/LTOSnjxkwG1TG+J6/Ve2zhnVdPGZT+ySSMSCu"
    "/LAJNv45ipPN8kDJFEwYVhdMYx26odq7J4MTgWQs1kAM62yo/FH+PhZuBKHamMtvRinJk86x"
    "GSA62wpsSzu33v3erR//yw927t9+ADeGIhw5FhVgshKYip1t3Rk/+rr/smZ+7bPAtbBhIiJw"
    "GEZtqspnCeseOKaPqwqGK5Ax6TkyJqSUTapBMCYMUmKkRtEA0jwDnUfopI1UaVeuW4yFm5HJ"
    "c6FU29oQ3sWSLWdhG18OZ2Mo52z6PVoLNURapB3ZOz/5odctb/vGjgD2HjPhP1YKEJXgSQDO"
    "Cr+nY/3Tvu+8LU973uXCRtgYTtPHQ6qYTQUTBMymApsKxAam8gpBbGDYpNH1Jgy38BO0TOgw"
    "zqm8iHmMURoLI5zahOmts3IFYUu2i1RuLN+KGMAzdx4DxPOUogQlEjhLCwsLX73jA2+9jHtT"
    "JnD78ZvsDT5/96Mp/MdSARAigjMAPBlAPxUlDmZrt7LQPPEn/stlc3NzF4Ar8dNkTBx74hWC"
    "OZh6E4pJKpDxr1EcZc9VqB0IY+tNmAcQ/iVSbiBuMNDNFjXYTHV6mcmTiANCxbOkGv0o6Da5"
    "gyR8p4ggDyhF2qFs/ernfnfXl669aQJ2eigI/8CjLfzHWgFiOuBkAOeGgpLiWPOkZ56+5Vkv"
    "urzqDWaZWIiJYsEIMWeFIA6zATi7gPA6wu9+UkgcPc/ZKiD2I4/JiW7y0KmeO3r1B4ZPxfmx"
    "2tk661+LVdCSc/zqdyFpae/uPZ+666P/80+j4ncyq1sD0bP0WAnksVaA+DfXAzgPwAb9Hbg3"
    "MG60Yk9+4c+99MRTz7hYTI+M7wNLFCZ7k8mTuZl9lY6J/j9aCT+yOwg+DLdQdWFE42nnBPhC"
    "jVfaiOlylOI3ZEbGzwvW2tg6T1mKEMp6qOCEYWlhcfG229//1t+iui8dXx9zKd8IP6PHWhj4"
    "JinBFICzAWxBZ4taXB1PePGlP73+xJN+DKZHTBAKhX+sik5JTS6JwyWT1QCBTFYE6G6cY7cg"
    "Jn4pN7YI5FQq4AzcROT5RfUziKFc3JDinJNKLC0sLt95zyff/wfL2+/aCWJSW4Kjvh0AcEsg"
    "euw3QxDfzKMCcEoAiHPd78P96coNl9pTv/+VP7z+pCf8aF335iyRGGJQaKYfC0ySefevdYYp"
    "5nIjPkj5txSlBrpaN3fm8k/ZMmcQWEEvfIg4gXEj2ruwdMN91171tuG+XUue1GncBNp8K4Db"
    "Hyt//3hUgPgd5oISnNyNErxFmKndymKz7pwLz9x4wXf/3JrZ6fOFaz+wzcf7FNur5v0FrAYp"
    "8JjJH+sWJZ1gsNPpRIpsYd7mLSLiRGBcS6PRaN/e3buuvfcf3/k+kCFmImdbN0HXDgTBP3A0"
    "UrrHugJoa7A5hIqBPewoglpJp7zgZ1+w5qQtL5oZDJ4INnDEAhAMMxXNJdMQK9XE8mGuWnTB"
    "QCCZC3wQBA4IGdtg2NgDBxYPfH7bJ9//vuWdW3fzYKZyK4sH25s/BHA/gDsVXY7jClB+nwGA"
    "UwGcHiKFid/RZ8mWWwDY+B0vOn/tmec/f2p2zXkDdhup6sFRBYFzJExkQtqYmIriQDXGpqsZ"
    "IoEwDkklhhC5Fk3TjFaa9q4Dux76/P0fu+IfRKTl/nQF27gJZj4eDfyu3TtDbG8fTzf88XgQ"
    "fJ3hlqAMM3iYfYxc9w1MTW5loQWA+Sc/8wnzZz3tgun1J55d9/pbelW13kg7U9X9UH3MHYus"
    "aMGQgGpErHOydzgcbl9eXLh76aH7v7b9+g98Mfhu8GC2RrNiJ5j4ruB3hjTuzm+2uT+WFEB/"
    "v2kAJwVFWINVNLXgqmbfokTQCbmoml3Xn1q3aa6a3zhrZtZOEzG70fLQLuxZGu19cHHpwXv2"
    "dk0zT63poR1a1zaug+IPhidXwoq/L6z49vF8g4+Fg+ALTDYEoHhCcBWHvbuZTcXltAWF8gBf"
    "ew85XN8swazvD+HctuDj7bFwY4+1wwSXcAKATQDmH6kyHOEhyBVQO8OK3xuIHMExtLKO1YOC"
    "MgwArA3s4toAHGvk6W5H63BhRS+HMG43fEOGA8ea0L9VFGDStVRBIWYCtzAbMMQgKEWFPFbo"
    "YKs6mvM2hG3LYZUvBmEvB4G7Y1Xo36oKcLBrM+Gn6vwcTAGaoABNUAKL8dbgx4/jx/Hj+HH8"
    "OH4c68f/ByzousNOGCXfAAAAAElFTkSuQmCC")

# ----------------------------------------------------------------------
warn_icon = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAAIAAAACACAYAAADDPmHLAAAABmJLR0QAAAAAAAD5Q7t/AAAA"
    "CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH3wcaBCQNKmhadgAAAB1pVFh0Q29tbWVudAAA"
    "AAAAQ3JlYXRlZCB3aXRoIEdJTVBkLmUHAAAgAElEQVR42u19edRlV1Xnb+9z3/vGGjMSEkjI"
    "gIQOQq9EBSTakZYG0bZViNKC9FI03WovAXF1N2iM01IjAXR1GwUiQitzd4OggjQqDi24MExJ"
    "ICETSWWs1PhN79579u4/zrTPfa9SX1KVobDuWq++N9V77949/fZv77MPcPw4fhw/jh/Hj+PH"
    "8eP4cfw4fhw/jh/Hj39GB329n+B7r150ANzqmvIrXzr2fPr+7sHeL3du45tu9aOv3iq44y6V"
    "u+8Tf8VVEzmuAMfA8cn3LzVPPJXcU5+3MknPveWX5p+1fSt9IzPOWZ/gzPk5nNK2WBqNsNQ4"
    "LBCBvaDtWqyJYn1+Dvd5j/sUuKXv8aU//6v+0x/4SLcnKcf7P9I119/o+68XpaBj3brPP4/d"
    "BZestADwjjcvPF8V3yuCZ3c9nkbAQtMAjoGmgXqBJ4BUCUTl/DX8o+FTwx8iuN4DfQ+0nWJx"
    "gXap4lrH+MTffKZ/z9vf3d0bf8P40svW2uMK8Cgdl79mjn/gxaPmgktW2le+dHTGv764+emN"
    "Cb63bfXcuTnCqCGvQYZMRPn8iABVyvfToTr7sSatCH9VgoIIAU3bKXqvk/kxfRKEd/2HV6+/"
    "+1hVBjqGrD1d3IV3/vbCr6+t4983jk6YG0NVoUTEQLBsIkBRC5sIINCUwJPQa0UJDsE+nx4k"
    "xRCBKtSrollfF4xG/HfM+uYffe36B/7xT5dHH/lEd0xgh8e9Anz0nYtz3/WKtck737L00raT"
    "N3iPCxYXWMJvDzZOTOaECDSUMFEl9GLh04+tgqhopSDhNYVoVI5aKbwXdW0rmJ/jt//Iz6z9"
    "JwDte69edJdetuaPK8BDPL74yeXxBZestO98y9Lr1yf6+oU5WiCCB8gRU7boJOxk9cxRUMkb"
    "ROWYZfnh9GshZ+ErspCJABEApNXzyRsAgMQXVKGAyuqauMUF+uS1X+pf9ea3tbc8XhWBHq8W"
    "/7bfWvwFEb18cYFZlRQUrJ2Jgp0zVa49KYUqwMxQVRARCFSJOimMVuIvF0IHMUGhENESGpLV"
    "SxE8ESA+PZaicFC/vqEO0H/40lf6733zFQv38en79fF0vfnxBO4A4J776bt/7zcX1+bn+IrF"
    "xYZADOeYGsdwzoGY4Vy5MTu4xoGIwOTQuAZMDMcOjhlEDGYGcfjLTPk+McPFGzFnhcn/hxhM"
    "Ds45OHZg837XcP4tXP114Pj/FeTm5xlzY77o6eeN7nnH+9oPAeBPvn9p/rgHGLj71/zixuil"
    "3zP+zLih813DwVjBYFfcfYrlyYqZ2IQAC/rq50qsf/DTtWFAo4uvQ4IWDwCtvEF6bMNDCQsK"
    "LwoQ+rVV3zQN3vCq16396uMhLNBjbfVXXDWRP3zz0i9PWrxhcZE9gRxFKM8MEHEWXggBHBTB"
    "oPzw2oMoAtNsnz+8b+KAGnIgufuMBTJuiEIPiWIWvqh5XQpoVFWICABF2+HOz1/XPv0nXj7e"
    "SDzGPysF+OT7l+YvecmqXPOm5a+OGpzBxAAI7CjE+ShUAoEd55/LhKwA1htkJYjC5AH4y7Ff"
    "0//XnCpiKu7XCH/oEbzUOCDxBcULaCX8mDZGJYneQcRvbKhzDv/1x3525df/2SnA1b+x9D2j"
    "EX1o3JBnZkdMMe4WYTnnqtSOcjoXQR/MYzJu3qR9IXQMLHzgACrJU/1YVAf6oSY9VJMiSvYO"
    "mh9rfr+oWuEXpRBB7/XzP/ralWc+FkTSY6IAv3/l8h+MR/TKpiFlZqJo+RmEcQRiM1y7s96A"
    "AVDBsVykDo6Yweb5syDALCZwmCWoJQLiH7FpoHHxAOC9GJwQFKEwiwrvk1cIHkFU+8mG+F33"
    "dKd/27OblUtesrrxdZcFJJR/zZuWv7S0wK8cjRwcO2JmsOOMnAMydxHgBZDHEWE3Ce07B3YM"
    "IgdmgmNG41xB48QIHiWEAiIK73M885b+H5F5nP4fERyHv5SyBZOJMMXf70om4eJvCR7Nwbkm"
    "fj7F103GEM61GY+5Of0J4/s/d50+V+7c1nxdeYCrf31hdNl/We/e/sate+bnsZWIXRIuEeKF"
    "iiEANPV8AHJsyJ8a+AGIBBBFgSeDLcRQ8iIFE0zTwNb310avFTmkKjl85PcFbjhYvRRQqNk7"
    "SA4DyfLD/YIXvIi2nZIX+cnLfm7lf3xdKECMa/6tV25ZnV9g59g1Ad3TTMEnN28Fn1w7c5Ec"
    "59hv2L8hIDRUL4ge1OVPPReFmcOA1MxfiuHpsTe0sYYXgsAlCT9lCJKVQ0QqBVAo+t5DBFhd"
    "82/8qdev/OwxrQDR8uWaN22djEakjrnhKFDKrtmBjKCT9ZfH4X76scmdFxwQrZopfo6x8kjq"
    "DDn/RBtTjO8JSOrAHWjSBCAAOop/YWsFtSKIShC6SQNjnDeCBkT8lBcQCf+x9wIRwXqrv/Mf"
    "f+7Afz4mFeDy18w1V1w1wR+8adt60xCco6bExSC4lPoRhzgb7nO8b90+IjtXrH6aFCLDGRQB"
    "Z9dfWIFDuv/yHq1TwaQMJr/PSpEs3ZA+lfARuQObAmqx/KAwRRGCEih6L/BesLauV/3kfzvw"
    "2mNNAQiAvuPN2w82DeaZuUluPStABGoB7Rdh27gfPP604JMHqNNDmy3QTPYvewA29QFDBmnF"
    "+delYSt4mHw/M4GGABKJygDLB2CKH1CDCURKWEjP971H7xVdLz/1qh+i3+PT9/fHjAd42xu3"
    "3bmwwKcQqHHOFauP8d7GfxjBJ4AWsoASz1PIoMoDGMEPysAUkfuhqOCa9q3v17HeFoG0UojA"
    "E/ha6Jb1s6RQxAZDRUheQEShEIgvocCLh3jFRqtYXfPPf8bT/N8d7RTxEVGAt/3Wtr9cWOCL"
    "iYiL8GMxJqVTMOjfcXbLHAmeAPgG4SBKiTMmIGBGVsBsGz8odwPlZhGd/jszJBiXP1UNNO6+"
    "gDuT74sWIsi4fcTqokoCidHyIRAvWQlsSPC96KRTuuMunPL08yb7jiZZdFTzTblzW/PWP6Kf"
    "m5/nb08Wn/Le4Oq5gD8qQs6uP8f9xPtTpIc5CpaTredaf90TYBUiUb5RsVwNDq3204z0L3LG"
    "yH0fob4Lyp6AQpkYAEhBSsFyI20srGCKYUIUDAa5IFRmhUDhAAgcRAUsDHIE7wFmzR6HAKgD"
    "zUH86afq1y69bG3+cekB3nv14vj2Xe4pJ+4Y3TA3R3AuECLZ+hM5k0maFPsN55/CQfISOSRw"
    "5f6ZyYA5gwnIFH9Qx/5D9QGm5hG1xPAhPIKkgs8QCKqtBUgGhDnFy1af7ku2cEAhvny2iA/v"
    "8eGzvFeoePjgDfq20xt+7LV7n/G4DAFvf9POyfyYGueYXWTG2AC+aeEXcseGg0zzguDYVbl9"
    "BfooFIfUKoE5K9sQYrHBFOzXabdvX5YqAzCl4UAWlCwgZwYp7ksBfUn4XnKWMMwACjcg+bXE"
    "FYh49L0CKtiYyBuefWF/5dGoIh41BfiDN+/4xNyYv52IHTPBNS5asovu3lChXFx9qukzW/df"
    "nq9uzIUQqmI/cn/AbPRfnH1l+YP7VcFHgzBDq2ns/EGd3+e6gIn1GdnrQAky+pcY/6WqHKZ4"
    "rwp4kQgYayXw3sN7xdqG4robace3Xrh28Ej7CY5YAT76zsW5XfeOnjcaj/5ibhT59ia5fAbn"
    "po6UBdCUkNlxVe+vu3NiB0+K6RE/pGyPiKt8P91Ca1gN/no/wnkXvhPt5ODsfjDj/sfjLfj7"
    "j/8gTthBMSZnqqfiAIq7L2Vk6wUkU7+GGk5WDQMCRYqCGCXwUrxEUgAR8ZNO7n3Va/c88XHh"
    "Ad76Wyd0y4vsiJnYBbfNrgg/CDUVZjgjfSt8G/dLb0BJBVNaaCnhnBVQjQnI0MSZCCLGpG3w"
    "rG/7GFQPbzREDr/7xgvxTc9axNwow7HS8ROVIZeA1fL/w2YQiSVgzKwDWEu3SpBcvwy8QHq8"
    "0frLz3ly+5tHkhoeURYgd25r3vE+d838PBNoWvgBpQfh58pefq6wgRkIkqtp4sjuJQsmtrm/"
    "8QSoOQHMIIcUBRgSuU2d39paC+hyziCsmycoVF1G7KFbKAeJFJPAqtELMTQ1jEbrJ1IAAgZD"
    "WEDKAIeMwMfPJGJQfI1YwBo+h4jgvbvikpes/spjUg6+/DVzfMY3L49E3cuJyDlnQB4K6GOX"
    "TiKRQJyzghwmYtk0Cb+UgqPncMk7lP+XMUVs1iyl2lBiTf+/Yh71oRFpJ+wYhRQWqWxbGj6r"
    "+4bUYiZzfkkRE7uZzomr96bfl9JlGn5mvH6O69L1wgL1b7tq50fkzm30qCvAFVdN5Jdes/GJ"
    "hXl4MgieyUW3bCzf8P5FcCbW0zBdZDiX6vlFkKEfkCN3wKEvgB2Imyh0Z97vipJF5VpeemgO"
    "bzSKPQZNErbLtyxESsrgokCTQpJRCpjuZFdCXa6Ccqxj0JRCpWvlwofEVDmk2Uzc+H70wlf/"
    "Ik6Nq6AfnRDw3qsX3WeudWeSa55T+vKDdecfbt1+/OvMCRU8YCxlVu7/oDeeCgFkV30mYDmj"
    "22uzEMnFtJQcpRQgVxFFJLjxVCFUzstNmQlKKf5zCBUSrpVQhJKiYBc6TS2rqF7AzPBewjVS"
    "gY81E08Cxw69Ao4FCwvqn3bu6MOXXrb/okfNA1x62Zo/75y5d49H8OyCtTq2rgvR1dcuLlX7"
    "rPApWbRNASuXyhkr2NARLJ6y1RfLJ9P7nxpMi8fYFLYRwWSyEVxtwiqgnM6m7wodPU32Zuxc"
    "lbImJU3vJ3ttyBlMUwwkdRjZkEKmhO6Yc/U0GB055ubCn3nV8jkPxws0Dyf2719Z2s7cXMRV"
    "zZ1yHm4rfTAnOPOiUN0Mmk86dwXXdYMcT6msDuKqEQQx1pbUkGMdn4jhva8qjoc6VAVz47jQ"
    "JLaVKySabrRwpXi/iRmAxFVJcYUQu1QLzoAPzGAN1UEWByEBkQBIq5kCKAzORAEBiKInQPAA"
    "odGVwSSAY8zPiX/G0+avufSy3Rc/4h7g8tfM4xnnu3ctzsPXsc4VTt8IP4MhC5AySCvCrwGV"
    "qR3EHsAs/Az2qLCMcVVQBpPZwzhjrU1A4tJt6jxVtQBPdvH7XInjOQuxVswZzwQrr3+vjfUp"
    "EyKujcGGTbYewmCKzLDG8AQitz7h5wGYS72Xj5gC8On7ZTLhF1Fs5XamGbLE4oJaKbtgMpZX"
    "x3BLEdcXs7hD4qZysUycL3JoLDWYgt0AZZcwkYicwx3ra/tLdhGzjcrNWyBYKTbl8EAzFDQ1"
    "g9owSYOMADmFpkEdxIRJ5/KaCceE5SX2v/9bJ7z1xc8fuUcsBHzyg98wf8eu3a9n5pDZTtG5"
    "RbhM1t1TruSB6q6gUtuPlmEtwVhAfi9Q0cQYlIGBspoohwbOHQMQ2YDD/GEgoWJtbT9Gjcsx"
    "l3L9X8DsTENIvFGgChmcSSFmhkhy6QqRBEfDZ4S6P4NIoMTF1XMAjsQAB8YgcAUigIZrKV7C"
    "70qeyoO9b15+0YseeMUj5gEu+f4vb6yt86tApFy5fjKazyX2o3ZrqeXbdvJYl0+DOJ9cqi0D"
    "V99n8n+b/k19LpXUrZvs3ST+VzSNq0IWqlBjXb6L58oVyAVqAFhZ+KA7ynqCwhtE7wk2NDjn"
    "olbhOgjMROMx4+rf2Pmy9169OD7qHuC9Vy+6j/5fOmM0ak5xFNIiZyyOShE+x2eyJ2Pjn7Hs"
    "zA1wXQAqF+4Q6V9uAuVc3eeBR7DtYcPC0aYwwKBjWalw/8xp2XhqCuVIFktcMq7BktUFkKcK"
    "JYn1EYneIFi8IjJ90RMgegLHoSZATGBhCANkPUtcBq8SfmfD6luMfubSy/b88VFXgPPPY7ey"
    "tvDz43H41Ungzlir41pIMB7AAjaYzKEmhIYECA08ApU6ANmVRJw7hmuhmw6h3ETSbEr49993"
    "a16Oni50IBMk15DZhTYvZjUlYwaxBvRPDiQB+SOGAomZAEEA4Yz6hQUkEdkzw6uUHgmN6D9S"
    "yoiTSiwHAxGQY9d7vegRCQEXXLLStp37HgKoiStigEFvXqaCKTdqFIVwJtUjwxwOcYQVPlfY"
    "IIQCNlSxy4g8u2IyOXkGXHFtgRLWV27FjFWBU4dzDUYjl7OHACypfIcJayCuVjJlkDhg8xL6"
    "z38Hz+ewhoKniru3YJpKFTTjneCDFhcIV//mia/86B9fMHfUFODy18zxiSeeuMzcnEjJ76MA"
    "u8o9m3hV3DvH3zngAUDTwC/9TYLGMHayEYIpGHHkBajw8FVNIQ16cKPNYQCbjRjKukrtqnpD"
    "es6VUAYDbDHIclLIA1WYKV9Pgx+yAaQQBzKkGqpMjIl6oHnFd73si5OjFgKe9tTT5658w4Gf"
    "dk4DCHKcNTSv4hks7LRZgdXwnA3A5PWYtvbpdJFqgGT7BWLXBg1Cw6FwwGaOlYO7MRqNDd0c"
    "ev/isCBT9bNZQPibugmVjNunVMVLz5RYzxzCQQgDlHEA1IBdldwS7zUoj4dmPJBb7Flc2+G5"
    "RzUE/OBP3LzedvzdAHmqlmRFNmwgFOsdbIhIz9nuoGTZwIBCNWNc0g1U8wSWUMpov6okOsMN"
    "MEAN+m7fYUOAqqDv1jE3NxqwluUzk5caspwYhLNSGOOK30guvUqBUWojpeeBwTCt8TAl9Hyt"
    "EdPJwDk2rhm/7N8tn74ZUmjTGGDSNs9iCplparokg/iHJE8GeMZNp7QG+eS5IkNgM4bs9sqF"
    "DtVAV9PLljRiquMzDV21g8pkk+6/VCIzazf47JIhuEppYcOYqf4Rl3O3GRASqze4fik7Svcr"
    "Y9Pi+UK6WZpj5ucJF3/L4suf/OQzx0esAFGLtowanldiSt03zihBcrFVH59hslC5cypCN2Bv"
    "6O5tOMhWkmO7M1bHhVxCKRWTXWJe9QpsjgXwvqv6Ciprp8ILzLLs6jGShyvt7zWojbFfaQpY"
    "w8T61BZP9vV8LUzdgxmi6HtP3/HK77+nPWIM8OIXXuieeNpXvospzsmJQ5vUxH/L/cMKPBEb"
    "4MIXgE2Rh/IJMc0GhanINFSKaqUQU3Vh6hvyRQ2CkMMWhp1rcPDAvXjCjkBs5QbSdEfU9gzn"
    "2B7ucyz8KEACRUj5Ejup4sFMIUvMAgsxXD0M2NW8NsLHQhZReB989LyeINa44qkyMRO5f8Gn"
    "75cj9gB33X33SHzz7c2IhI1r4uyGiqbaVuwEwoLbGmQMzDVXYJs/B/WD7DbT+waPwdY6p2+5"
    "FBznC/lu96ZawpwzKaTh+EM4GGIRC15j0wpoNrPJRvFhibHSQgeTNmMAnDPhlcgpu5iGc5Ms"
    "rU34lKMCAv/tK25ZE8XTVUnZ2TU1ZHrtUBE+lQAH6/mKv+KaPzBuL6VDoGEJmPNK4fDYmTDD"
    "04CtoqFdpoQ3dWGs0E3cLqmuMwDOUrdU5/3E4RoNZhdYkqsMsabBWgaakWEZsJ09J6rKKkA0"
    "P2a8+Du3nnI4ILipiOiFz8qLqmxqNwCAYRgzzagODhTCeo3c4l3yZRBN1f9zcyi7QRhwRjiW"
    "fSRD0DR1A8omjr177gzdQFSDP8BVeAWYhQFo0CdBlZeCwTyFSq+9QQF+JYQRarodsM9xjnSh"
    "Sgs8/3lz37pj59mjo6EAp6Rrqlp4fqQJXMMfpubE7ImiLobAFj3AgwtbnoPBEBWNbIglEE0x"
    "iISajgY59GHvh8MTJA2hcbYIVM8uZBMObJ9i+V6L9st5qjEKNR3MFv0nwJe6icl0NieBJzkU"
    "74AKkDcN0DTuGaeeNOEjJYII6ho7sNGsyy1CrqzdFF6iCySjLEO+PsRkyp0us3oGZmYFoEpA"
    "pRuIqhSLzO92vNkJ7laACfLFdX+x9EscAGGZMOIz0IzWAk2/VTULU8KQgrj0tCxw0bgglYjz"
    "zOFQhKI4u6AoQdXqTtboIvYSeFU85aJnzenD9gAxfjTIy6aCm1FTc0+DF9NKWqj5gcrTimJc"
    "fdXKxTWFnFI3zXz3gDihOn1EwgTZGl3mExDvY9Yo+UMc7WQldOKCAbiKrbQuuwKoFSnE5vsR"
    "2T3TKYSaO7E4CmkVM08vm7dL4dNcpMJWlgkpMTSfdNZpu9qHrQAK8A9/3/azm8bahflyy+jZ"
    "njy7SnewYKOq1JF18zD3CxDKZEtqC0dB/bU3sbz7MB8vF0plc+sp+3419/9n1s7wDKjSVjKe"
    "jo2rL5eZMMjtqbCoxXLtORUOYahsMAO00mwjBZvFXgwBYX1dTzpcKnhYDPCkJ423FuY0mH9C"
    "pKlmrori6nJbt4s0uc3PLVquw4ElTfJ7NDVepotqLkZBPFO8gbV4UM3mqa5sSgFcnF9YGjJK"
    "PMcgS7G1DZAVGpcYzly4/aE3Su7chM1qCwzbx0CzFr9a4VOxXqL5I0oDn/608/nkE3h7vWNG"
    "UIK41C0CFcr6qllo1lWRmdNPxhNQjXDzGRmrBVXEUSE9TLpYKYbFDJzBX7JgtzkqEK4ZgeDi"
    "7w0hRQffg6HCmjJvepz+PzRXo9K2VJU3oLxYtVbuMt6u8CsEg6tsH4S5jgqiZkSLh2O9+PCW"
    "4BeThiYSjAxzV3r1KE/mssuxYYtGVQyurUEt6FJEl0bFOlAXnWCszSpBsdKSg9fxe3Pl4K1L"
    "yM2ciFStLe/WVctSh9ABIK4zIZ6RAhePVvUAZMYUObvJYdXA8NwVFIlJ0bBGMWYFoyNWgLZL"
    "ww4B0XKSaYBJmbJFg2JKEVwZ4jisHwwIkOwKUbeYzfIaA1CW4nNx0UlIrgA5YiwvbQ4ENqMm"
    "cA5ToK9WAo2CUgwBqVFOY0Cqw/p/aVjNOb+N58mDVthrOPyK80j8XKUVgujhee8HVYC9B+b0"
    "jrv8uiiyILNAbcqhaQ4f5UGNqnGaZ97CpYSPOjxEnKDlcyswSFy5SpjiSlGG6A20jtMaLRfR"
    "fefnNpsG5v9jvE76bmPRlfAxYDMxxDo1yB1OPcNgOHbCUIrCuuZrSWSWo1NZeRymjeik1faI"
    "FOCuXdfqzbf5lfTBRUghN7Wl04r7j6g91QHSDy7egs0sPi4ehHhQCYtMGLFFP+Y9RgmUMxKv"
    "Sq6ZauWsBIfrBww/MaaUOu15krJphQE4n3v+3Si8RHkNAy+RvCQZmEAx1Bp2z/rP5CnVeA0F"
    "VCJngFBsajusH5ECECC33TF5oO8JopSHGYBhTsjy+yYUKGbSvsNaQLGsEv2GILK6+EpZKGpu"
    "dYjggadI6RLHm06NhbUKMNlYx3g8CnWDGEJUpz1ODgM6zErq36FaL1xNFpuyp0oxlKaQvQqm"
    "vIAtAaeeC43/P0wrUTDrxuG6X/gwS8D1izdM1soPpjgerUzZDONQDNrHwNpLoCql2fg+iyHK"
    "ONeCETL5ZMKC5pjKGRETeEpJirCL60+CkU0tDzPhYgr5D4RreQwtLKJaTFD9vzIM0+KmpPh2"
    "sHXO99XYV8YKmq9THnCtaYErsLYuD1x95VnjIwGBCoA3WoWq5nF5BayU/XiS+9Ic85ABTKYP"
    "zfAMtfHPXEgd5L/2Z2YAlVMdNlbBOc1UGnoBKh4AfEjrLyRQl6eRl5g/4/Ni5pFDgVEWi+fr"
    "XYkHQE8HRbOBzy7GRGbJilGMtKI5jqn3cSwtVNl7uXv/gf7IsgAA3rE/IAp4QcwIqAYeZuuU"
    "hGarDGEq560JC4MMa2pZi8uDAY02nsJ4C63AYVIQGADoNnXK+/bek8vBGUNkHGHy+0yFh8+0"
    "YFkrLGDCl0HzWnL2AuaGWZPSgHIvYDpMG0Mm5zQaifeKXpgOHJA7V9dZj1QBpO+xW7xKSgfT"
    "zlhZyAOaquq+tfN5882EADPSvYDFWhFSsUNRu1LVQ+ED476V4+dFb6GMbnL/g4ZGYoZrGvMd"
    "XHiJGWkajPJXSjnFiZT4X54ejrAz00krjxEVXwfKEN8vQjkkqxD6HrjplslNe/Yt+COqBl74"
    "zC2+6/vbez96ssRxaF4BJ4m9qufyWi1mM6pV42WrwE8e5VYKHuXi5KpT2elLaTgJ0uz8VStC"
    "RuaWOdPUx9ccEhyXLV+cCS1irD7tN1aKX1ngRLmiVzGBYWV/Seni4CgMHH6aN0DV5xgMhVRd"
    "NLHehGWJDK0XRdsSPv7XB7/88b/+cndECjBqSO69v/vyiTsX/pX4EAa8F2AUtC1t4qECqEs7"
    "eSALT4wipC1crMaXOX5UuAPjGqHDKd/lApAthZJVGBMiUkxWjkrHuO/Oj+HW21exMfGoonNE"
    "Um17AGc+cRxXAQ8yHi3hCUZIdVgauDwUd18AcDq/okzZtWMQSnSADOJcYgv8knEGDoDUS9/F"
    "7hU9IgWYdM3kC9etf+7cs7ZhPIpDEJXDLFsG0CSLjhqpBBevi6rmyd7JBWja048SyVFfoOQZ"
    "MmlklMRu+lBhCMOwqU5T09kTKEGg2Lq4D+ee2aPvZSo4hfRphKWFxQHSLoqlhqHTwf6DKZ9P"
    "HAgMFhpiIiIyFlyKa4hpXNm0OiwDl4wT6vAhSgUACkG8SN/5OwActvJ1WAX4p8/vWQdw2wu/"
    "QzG/kFxMWPVaZwXI0zmHwKYYsAE0YmYqGUEWgWKKYJre6cNUFU1mgYwzOBI2pqOGGOM5wnhu"
    "VAiUfEnjKl6VohAqFfaoCZja24Uejnr4sGVPi/vP86ayEhQ0X/CEaNmtVIGqByB0aqU9itKo"
    "WcRhkqD9K5Mbnnr2XPeVmydHDAI9gAlRd7f0Ir1XeJ++SDPoKK3TNMR+lUJUFlG5TwwuLOWa"
    "g01/KmsbXOAqOzHWkkGlAXXBO7qYGTSBRk5ZgiV41H73MJVDjdK1pLtDw6jPu5yAVoOqzbkO"
    "vIZWOb5mLkaiZ+i9wvdBJl0PvvGr659e36DDeoBNZQGnnDRa23eg/UKI/2Ej5DC5Omhptp9q"
    "gHJJ8eJW6jO3aEvCEYNs1bj3nHmgXGDVWReLjDMvNQYLtiTHVIKqy8WitXWPrg+W7k1dAiYG"
    "Z2XQmn9X2Dh9CCWB3XzSnFulDeU7K/Q/U0GQp5RLlIlGJeg9sDYB/vyTB7/wtV0bB484BATB"
    "0sEvXLf2N6ectOUFYx++MC5MDm8AABYKSURBVPACCucVcCH2MJeTGhSsIJGvK5O5aQqHa/R5"
    "M3Z6zXjLYsx0RXImpjZJoOq7kgsVibGXz8TitmejGS1hMeGdjf144I6/wLalu6qwpdaLDfYQ"
    "BKatulIGnY3CqufU7FhGA6WfKY84aCZ6YVWg64Jceu9FpbsHQAdgcjQ8gN7/QLvvE586eMOk"
    "VXiv6r3Ce0HvS1oYGxHNRkoztHZwCcrcfVRIVwfWZHfxUIOSYb63dvkJzpn9f4xg5rb/ALae"
    "+Hy4ZrH6ReO5rXjSuT+AfvRvsG//Wpzbr1NUqwyssrxWcnix1jzYg6By6wMiTWXw2YP/V+1b"
    "HIdNd33wyuLDnIi9+yaf2bHdrQPoj4YCAMAqgG6yMbnBe9WuS/EmxiOvlfvPmppiVt5W7dBK"
    "Yffjmb6wqMATjOJYodvPqf9Gi1FFs/SdaEbbp9qq7OOdJ56D3j0Xe/au5Q0j0+CnHOMrBdWc"
    "Q1iae+p3Sf2bhtfiUJ6iXAfN4dXHz0gy6HtBL4q2Y/rgR/b+yfq63xvx21FRgI2tW9zqzbet"
    "fqz3yl6Aziu8JC8QPYGUnNS6q+HJJAuxf9PrYsauW3xuBS063Nyh7MyFQVEkeQLxirYdY2H5"
    "7MMuDSNiPOXci/GVmx7ApPXpl5lEsTCiqrWywmwpo3nruBnSVcw0iLTPYL1BlV2jEbepFQ0x"
    "vxf0vSYQqG3rV+6+t923McEebGIUymYVoD9w0O9+z//Z+zcbG4DvVfte0bVhT5uwr43dGRuF"
    "qLCIF3bX7drXKzBA0FKEPqVB5eKLUZRywetdOlQUSsB46Rs3tVcAAHjf45TTvw0HD65DvDdC"
    "tptE1tKRKYVQE5q02pco60vynNBqy7opjBHRfvKs3qeYHwC57xW9eN2zd+2vtixzD2D/Zs5z"
    "swqgQNCo1bWNf+o60T59ebx5Hy0TWscuLS1lOlB1u0dvuQhaI+e0I4fNn43A7dbteY/fbIGS"
    "PQRUMZp/Ih7KyOil5VMDFjD7/Zod46fCTVbFoRew4WqoSBhmA3XWUPYRQt5Ozse8v+/Dresk"
    "oP81x//9mnvftbom+wGsH00FAIB983Pc/tXf7/+jXoj7nHP66IKCNxCfCAnkXTKmBa+Vi1cd"
    "ACEzhLHCAhX4qS9w9grZ5UrZswcCqI+5/2YPgmvGaNvO/AYpXgb2u9ScVzonmYFNUCurYgpf"
    "2OtQK0K4CH2M+V0X3H/Xh51Eul7E9+0dqtqLYPdmAOBDVYCNjYns+fRnD962sdHv73uRrld0"
    "vaDzSRul2vxoyA3ULJixnFTFGlwYyduwm21Y7d68Uzt1yiF26BT0XrC+emfetWMzTm/P7l2x"
    "n1Hq3522iotkh8B8twKqvgZuGOw7bL1AZhsHrxkv57NBIXMwvRd0Mfa3nUKV+IYb9797aZEB"
    "YPdm4v9DVQAP4N6lRXa33n7gvd4rd50ETewEXfQAPtaoJboqa+HWf9qYZ7di1UFstduz1ei+"
    "RufFEwgyJteyRx8IOLDnC2De3GhE5xp87p/+DFuWxzk5t8Aub/Nm28tsyoHh3sI1MITqIbKX"
    "ghbSFrLpHL1X+F6y5edsTETXN3z7v/5076dX12TfZuP/Q1UABbB7dU0Ovuv9uz+2vqHqfQCD"
    "befRpVjUi6GIY++A2J2zB/hApy27aL/k3LdsyDiFFku8t5s1ig0jIQQsL+zGysF74f2Dt4R1"
    "XYtbb/kSdmzdwPLyXNj0IW3zmm4zfoOYm0/7BFYey5x3vgYyCAXFc2rMrpLwewmf2/WKtg1K"
    "0HYC8Ur33HPgA0uL3AC4bzME0MNRgMQH7F5c5Obuu1fe571QiEUhI+hiGOg6ge8lu68qrpsd"
    "szK5Y8mN7PprVy9qdtiyF9wQSmlzxrznnv0uDWHgy5+/KubP3SHbwUQEf/LBy3HO2SdhbkRG"
    "UHWMT0RRVrqK79DBPsOoNohM3il9dsoi0r7DacfxgK2A3gN959F28frGsCtedXVN9a3/874P"
    "rq7JGoB7Nuv+8RBRUcaqXacnf/YLKzc855t2vnTUEIiIwvx6VJscpdk9OlgDiDgqHWbZN2Ys"
    "niyvo6yDh1knX3fPmXUHNd9uqeflxTFuvOFPMem3YucJZ0yd4P/72w/jzz78K7joXz4Zp56y"
    "DOfilG7YrEKKkNNrUqg8sYAubyhYBK0WoFZeLO0u6vPegRIZ167v0XWCSesxaX3wAJ1H23v6"
    "2h3733vjLes3dZ3eB+AWbHYmPh7enkEPADiwvMSn3H7H/rd/w7k7fqzvBR1LmF1PAiZvNokM"
    "s3Ot1jO5UO9O2IAUNBi/HjZdBoTMUOTsdgXCFDdnRkXUpO3ZyjjGtHAjdPU4Rzj15K3Yt+cj"
    "+PiH/ghrk21YWj4Nd991M/ruPpx68lZc/JyzsH37wrTwrdBQ9gJMISH/ze+TqTAnU0oQw6IU"
    "/iTVWRLX33vJwm9bMamf6vq69te8+773xxPcFWsAeKQ8QCpZN22nJ3z+utWbn/NN21/UNDyi"
    "cKQpVWafmzTKBTNnBpXBk2UBZb0UDDMXk2Kq4kZT1bO6WljcsWPC4kKDE3Ys4oSdjC2LKzjt"
    "lBHOOnMHTjt1KxYXR7GvQQyfkKxba0AYvYLdELIioWaBv+H/j2HFJ8v3Qbn6PoSttvNoW4+u"
    "i/cnPoBvr3TtF3Zftevuyb1tp/sAfOXRUABEkHHq/BwvHjjYffbss7a9sHFFUGGYVBkA4cwU"
    "zTLcMC5mMBx8mULC1fKnqc7iqY3fyXZnhdfVNmnaHrzCyLFjjBrCeMyYm3MYj11coKnFZWey"
    "RmqBWQuHBZw2Q5HM4A3dfq0wkTlFSaV77+FjvJ90PdpW0LYekzY813W9rKx2d739j3a9te3U"
    "A7gZwL0PVZAPVwF6AOPe68677mkPXPiNi08dz7mTAxqo59pwGGOdV+hqbHpMHTpcrYfj6Zhu"
    "asB13B80jWK4BI1MSKFpQspSkYOqlCJZuaSO+5K6ihrPMBS8UYqpcBBmBE55CS0bSfs+vLfv"
    "PXyvaHtB23ZoW8EkKkAXlWJtHXTV7978EyIqGlq/rgfQPloKAAAbAE4BMPf3/7j/Uxd/y4mX"
    "hv0C0pZ9XO/8FV/gvDKW6nlDXLda10MPpluopxZLDIU/1TdQF+kLQhdby46WHgVfgbeyKTSq"
    "eoNM1x5MfC+C9jm3r3cNL2mzSEjtcsyf9Oh6j41W0LUSUz8P7wV337P/A1/88up10fpvAXD3"
    "wxHikShAF9PIk5aXeNz33efPOH3bd6SZtTCNGRnRsxkhyfXgqOIFSq/fNNKHGb0yXIswVApU"
    "eXdqNCw0i85A5YZ+zs+b98CkaYPYD8jA6nXgARTiCzXuvc9NHUn4vQ/pc9d6tL1H2/XY2PBo"
    "J9H6e0Hf97q+4fe+6erbf63rVAAcAPClhxr7j4YCIBYcTmw7nbv1a5Pdzzx/8UnzC+50xFAw"
    "3ME7uH3No+QyAASFyVvVAok6pusg5cOMBSlVrqrTSmDp1nplkmEec+utlJ081YQBteFgkM6Z"
    "LeODcviKlNJqJ/GgEF5Cc433gj6BvE4xaTtMJuF+13l0raDvRdsW9KtX3fjKpmGIaA/gy5H6"
    "xWOhAD7igZMBuH/47P6//9Zv3vlCZozKomYtCyi1nuGrcZuZamDksFHDZgC2c9h2/xJMD72p"
    "sNGwAxOmYjekZbXi33MB1+Tp4Tkp//eQ6Z/WWGDg8oMShDzf9x59vHVdsPQ2Cr/E/ZAKtq3S"
    "P157z6/de//kvvUN8QDujwrgHysFAIA1AEsAtjUNu7/8290ffu43n3Spc9B6iyazBApFIXTg"
    "xtOqmNLbr2aNwHChpcUAOuUBasp5UF0r7aOmY0myMth0z5JAIoZ2HijKFC4wHiKUdEMa6fuQ"
    "8nkfY34fYnxA+30Q/sQHy+8Ufe8hIrjv/pVP/fEH7/3fk1YlYrAvAjh4JMI7GgogkSI+SUTH"
    "y0s82v3AxqfOPXvbi7IPyJIoWYA1TLsJRel717L6B7YZdEYz6YN0ztoeAqTuZCkbNYvGVs+q"
    "oGSQgszK+dWkhSUrEEmKIaY24WPMD+/pRSCxm6rrfOzo8Zi0PSZtVIA2Ub5B+L33urba3XXV"
    "7976i+aafxXAHQ+F9n2kFAAx/fAATm471YMr/frSktx50gnLz4axYEkuXdKwzXoeUBqdmlcN"
    "wawEoaof2CwR06GHn6EQarqONGtlFdcNR5Bz/dzPNkD50AwAcQjXHyxeq/tePHyfSJ5k4R4b"
    "kwD4kvWH4loigrxMJrLxa2+66cfN9d4N4LqHC/weCQVIhaJ5ANu7TvWmm9fuePLpjd+xfeEC"
    "29iYeu1VUpWVahce15nZBZBUgrwBiWpChV1Pp1NhoVKSKn/HVFoHA+YqWtpmECa3D5Sw5rqA"
    "GNcfBO+D20/lch9SuUDr+mL1Jub3ERD23stkA/IrV335ZU3DLCHmrAH4QkT/eDwpQEpJdgBY"
    "VAU+96WD15979tzy1qW58+zkzLTerWqPAgWLoSF9O71SNgmnDJSgQV8txRheUsChwGvWbvo5"
    "WEFDZgBCMVXAFN+D5/BeAsBTw+r1Ec0HFg/txGOj82gnfQR+fSyna7R+L32v9MtvvP6l83Ps"
    "2k4kAu7rH2rF79FSgMQQrgA4EcAYAD77uQPXnnvO3PLW5fnzVKdTNE2ri0SrFTbZYgdIftgr"
    "j0qwgy4k49jr/gPUwhwINrtySj0NUrqAUornpQJ3qRfBex9LxLGS14WWuTYJP1bzJl0fhN8J"
    "Jm0fGjx6H8u/odvql6+8/iULC9xsBMQvAG6Lsd8fLYEdbQVIDOEkKkGTlODJTxzJli0LF4St"
    "Zyh3tx6qTTyD8sEaA9t4WRaIlObQsBXLsA+xbjezraVZoIYtrEkhyby+xGYQsWmflDY4r6l8"
    "K7FzJwo/grog8Aj4UqxvfQaDXhS9F+k76X75yutfNjdm3phI6ku/J8b99mgK65FQAEQvoAB2"
    "pu/44vUHb9i+TXeddOLys8OY4bA2PLc7K8Wum1AKRSyJxl7M0iELLUupbM9dXlI8aNA0tCwN"
    "GzuGpdoK1JWcXUWmu37ENoSkBo0Yv/sE8pLwg4tP1t5GoUdyJxaABF5UJxv9/l954w0/Mhox"
    "t10W/p4Y99eOtqAeKQXQ2JfGEROwKnDHro277t+99jfnPGXbi4jCSkA1Aw40LTOLSqBmDk5J"
    "vVC6agaNlAm0+0zPDix+puCnW8hTB5Ht9hEfYrzPrr/U6gPA8zF+J4FLiOsZ7Hl0bZ9Lu6GJ"
    "NihMOp99+9du+I3fvunVMYwlF3YAwLVHC/RNF8sf2WME4BsAPCUp2/ISj1dWpf+F151/zdzY"
    "LRMxhQ2aKG7U1MA1jJFjNI2L+/cyRiNG4xxGo7D1i2sYjQt7ATkicMN57Frah9COlKkmk8BW"
    "EWEmjmiFK0S0whASsYCPHsHHnD54gFK/77P7D9Su9z4UcfrUMxlAYgCIon0Huu1ru9/9h+/Z"
    "9YHB9TsA4HOxCUePRQWYqQRNw9z3Iq/+ibNet33H1m8ZNVBmJmJC48KY9qYJtflx49A0DBf/"
    "Ns7BMYXnnDOKEzepdGYvvbiRdVWPyN2QQ8Cpg4eal7iFMBAUIDRrSmzVDkKNq3JLGOjD4yBs"
    "n5UheQuJS7tERNtO/Qf/5JZX33zb+r0R7D1qwn+0FCApwbkAzo738/EdF+98+vOefcYVbqTq"
    "iJmZwqz+OK+/cUH4jsP9IHjKCuGY4DgoAjGhacI+BewYjhE3lbTTy3iYMQ6TCBNqQs4q0Lwq"
    "R7Tk896LKehYDOCzQpT3pLKvQETVe9DK6sqXrvydr16+MM9uvQhfAeyLMX/PIyn8R1MBUv/h"
    "WQDOA5C3Nl9e4tHKqnSv+6lzLt+ydcszmjjU0xFHaw7WzRwFz9EbuPB6Ch/MhIZd3DGL4rSv"
    "0HvguHQg5bYzmjWDwCzH0rKmMMR/rdx9WgeZEL+P5VzJmECztUsMGxFr6GSi+unP7vqlj//l"
    "A1+cgZ3uj8I/+EgL/9FWAERQeBqA8wEsD1+86Flbz3zB88+4Yn7cLDOxEhPZzZTZUe46djHm"
    "O6ZwP+5S6jh0JjsOrWfp/wFlNCvnjXgwIBWi4FPzR5rC4TWvC7DNG3183nsP9WF5dlaW4X0R"
    "7T3R3r37PvXbv3/rW5LiDyqruyLRs/ZoCeTRVoD0nTsBPB3ACfY3zM+x25iIf8Wlp734rDNP"
    "fuW4USJyGnZ7oSzktEmiYwYx4OJUz6AAadNl5J3E0ibMZTajndpdk0dl1bFt7oyNG0WYRsg+"
    "Losv1q5xAUckjNQr0+rqyo1X/s5NPz83RzqI9amWcnO8tY+2MPAYKcECgKcCOAOD9vRkHT/+"
    "I0/6oSecuvP7xw0IxMrRdNNuW8xlD0POW7MGS8+bU6ct5RimvZCmLkCihiiRToK86CR3/aQV"
    "P6apU+J6/TSrtzwv6n1D6+srt7zvQ7f/xq23r+9mAokOuU4cBHBDJHr8YyGIx/JoADwxAsQt"
    "w9+zuMjN2pr0P/KDp3/Pk87Y+X3j0WgLsde4kxeRaTNjNimgaTqphjATDr1zqNp5xXVnb/EQ"
    "YRl8bguDVotaQ/KuaDtHa6v7rr3mj++46oE9k7XRiLnrRGbQ5rsA3PRoxfvHowKk37AlKsFp"
    "wywBAJYWebS6Jt2zL9rxlIufc9IrFhe3XjBygUaKAqZha7ndf6dkAVQ3Gh2yr0DNKHxM08PV"
    "DAJVhaDvHbVdu/+B3fs+fvU7vvYexyBipr6fEnyy+psA3HU0SrrHugJYb3BqTBV3YMa6RWtJ"
    "P/wDT7zkjDO2vmB+bukc1wDMoqQAO0d2h7KQCdLMgZWHJjFLPYJQ+IDIFWisD1DXO3g/OXjw"
    "wMHPvO9Dd79n193re5YWuVldk0OtzZ8AuBOhi3flsbL6x6sCpN8zD+B0AGfGTGHmb1xY4GZ9"
    "PVzoF1xy0gUXnL/927duWXi6yPxJ4zlCwwJRFQ4LF9PwatLBl+khLoaGo1QShakXQtd1bd9v"
    "3Hr//Qc/89Z33fkRVe0XF7npOsgMN5+ODmHV7i0xt/ePpwv+eDwIoc/wjKgMS3iQlcxzc+xG"
    "DWhlNSjERc/c9qRnPmPbM04+cfGpc3OjM5pmvLP3bmlu3ABQ2K0DdaAIvk+zezqvKvsmk8k9"
    "Kyvrt91519p17//wPZ+NsRvLSzzamMDPcPFDwe+OZdzdj7W7P5YUwP6+RQBPiIqwFZuZcD5i"
    "blxoNB+kXLRjezN3ykkLW046oVnevs0tMhOvb8hk7z6/dt/udvX2O9b2DV3z1mUeTzr4rhUZ"
    "oPhDxZCNaPF3RIvvH88X+Fg4CKHB5IQIFE+MoeKhzjdA03DhgcwOrRKXAfS9qOpDjs0a3fqB"
    "mM7dHWO8PxYu7LF2uBgSTkRYmrbt4SrDER6K0gG1O1r8vkjkKI4hyzpWD4rKMA9ge2QXt0fg"
    "OEK9LfnROCRa9HpM4/YA2BvvH1NC/3pRgFnn0kSFWIrcwnLEEPNRKdJeMfwgVp3ceR/TtvVo"
    "5atR2OtR4HKsCv3rVQEOdW4u3prB7VAK0EUF6KISeAynwB8/jh/Hj+PH8eP4cewf/x/ZI2wF"
    "+F5o0AAAAABJRU5ErkJggg==")

# ----------------------------------------------------------------------
error_icon = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAAIAAAACACAYAAADDPmHLAAAABmJLR0QAAAAAAAD5Q7t/AAAA"
    "CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH3wcaBCI5XYYJRQAAAB1pVFh0Q29tbWVudAAA"
    "AAAAQ3JlYXRlZCB3aXRoIEdJTVBkLmUHAAAgAElEQVR42u19e7hdVXXvb8659j577/N+5kEC"
    "gcjDhIC2WlRARelDEJtWr/2uvddr7Xe9V9CipiLvAJFHH/pVtGgfii1VQeRN8VPbWhCw2quI"
    "EEKMQoBASDhJTnIe+7XmnPePNec6Y4091zk7OUlI6Fnft7/93nutNX7jN35jzLHmBOa3+W1+"
    "m9/mt/ltfpvf5rf5bX6b3+a3/0KbeKUf4GdKJQVAVQG5ulDQJ46PN2f6/OPd3fIZYwrPGoMX"
    "jTEvWatvaDTMPAAOg+3GcjkakVKdPTlZ969d1NHx2m4hTpbAq+rWLusQYkED6IySWxmANECj"
    "CUxZoNoBbNfAdgBPxcDjD8bxj74bxzs9OL7TbEa/MiZ+pYBCHO7evVxKtXpqqgEA15RKZ1pg"
    "tQHeGFv7agGUlRCQABRgLaABCCuEP3ABADZ5nNxZ699QGkBsLZoAykI8b4FHJPAvP9H65tub"
    "zW1uH4prarXGPAAO0nZusSh/K4qi1VNTjdVRtPRNUfTROrC6ae2xRSEQCaGdQaUQQtADte4p"
    "PWib89w6MLh7a5OXDICoaQy0EPUi8G8CuOmSWu0bhysYxGHk7f7klq8tla6rWfuHSsrBYuKz"
    "Vggh4TxbAAAztn/NBg7ashOSMb7/jGMG/54BLKzVFojqxkAJ8ZAE/uqyWu1bt1Qqhfvj+LDQ"
    "Doc8AL5YLnd8uFqtX1upvLdpzKXa2lVlpQwAAefiUpDDECL7nIChxcg5z4nVMwBJgeHAwUCh"
    "jbWqoTU6lPryxdXquQAanymV1JpaTc8DYC+3OyuV4uqpqca1lcolDWMuKUpZFoCGEEr6GM6N"
    "7V63AIRnA/d+yPMhRIuRBTG2N7j/Tc4CmdemgWFhrakZozqE+LcNcfy/b2o2nzpUgSAOVY+/"
    "qly+3BiztqyUtEJY4bxdCJHepu0oUoNbxwje4IKEAhsARMjwHhSpYcn+Gf+euzcOIP51Y0yG"
    "FWrGKFj7H7/UevWFpdL2E8fH7TwAcsTdDY2GWVcuv8dY+48lpcpCCGcz5+nemAQA3PCC032I"
    "/il4cgwPZnThFCCoxxMGoIzg3/NgMNbqhjEqsvaeS2u11TeWy8U/qlZr8wAgdP9n9XrhHYXC"
    "jyMpV0RSJoaXMqV44QBAvVoyIIALwFnAgBwxKIiBMyHB2oznWyoQrc2wg/Ws4F53IIqrWkcK"
    "uPTyavXqQyEsiEPB66+pVNY1rL20olQa41NjS5kaO71nsV/MAgQaEmxA+AmmEWxOnA+xADU2"
    "FYlUFxjyO8YYwFo0rd3yZLO58r2FQs3XMV6OLXoZq3alP6pWzdWdnb9UwNKKUhBe4FEjE2ML"
    "Iv64yGvRAQ441tosQNxnrbUZEFlK/9ZCWJtJGz0jRAA00wHCWkji8dT7hbsZANYYCClhARSM"
    "WXRcobD7MSEuAnDdfzkGWFsuv6sg5V2REFpJqYQQqdFSb6fPiYFBhKAg6h+OMUReZjBL6hcq"
    "CmW8nXg1F4jWmCxrGJMY3X+eMIFnAeMeG2sfvXRy8jUvRyHpZQHAlZXKjQUpPxAJYaVMCnZU"
    "6PnHIWpXDhSeGXyIACv+cK/PGJd6Pjc6SfVa0j+aHVCDs1CRZgLe6PR98poxxgMkrhujtzWb"
    "S16v1MTBFIgHDQA+3l/d2fl4UcqVcAJPBG40DFBjpsygVMa7W4xNKoI8XZxJ+BkKFm806tXZ"
    "8nBqbJFjbM4Wztsz+kG717UxummMqhpz5v+IovtPHB+PXzEAWNvRUbiyXm9+urt7ZwfQI6RU"
    "GYVPwEAFoCR0nj4OZABwAz6ZMBGi/bz0r6UAaINlYg+C1PspOzhvBon3lr1vCBAoQDxwtDG2"
    "aYwwxpx3xdTUDa8IALi4pq/q6prskFIppSLpvTZgeE/zGcMzLeANLshjauyWQhGh/WCcD7zG"
    "q4A8IwDLCAzxdqP1tNezcJEa37FHhjmsRaw1DICq1p/59OTknx7WAHCeb67u7q5HQlglZaSk"
    "zMR5KWWyEzQkcMO7+0x4cJskLJAq/IAO4FrAhwnrjAyXGSBHBGoPBl4lpOGBxX/rKJ96OgLG"
    "B9UEAGJjYIxBw5jPXzEx8SeHJQDOLRajGxoNXN3TUy0khkuMTwytiEf7x1LKaYM7cKRsQY3O"
    "xKHM0QEIgGKmQhBN+eg4QSgjMHwcgIDBUsM6MHDPzzyeDgOAtYgTXYCatZ9dNz6+5nADgABg"
    "r+3tHVdASUkZSef50gEgjfXOsJ4JpFLZPD9g+FDuLwJlYq74+XhAWtghhg+mh6FUMDQQRIHh"
    "6d8/JoamrOCfp2Vjwgyx1tAJGD7ybuBvDoQwPGAMsK6nZ0tJygVSiEgmRZ7U671HCxICPOVn"
    "Yr/bwUwdIJAhcFag4SKvFDxTRbBlQIhV/SgzkHp/pgqYoX7/O1oHQ4LxAtEb34UC7T5fNwY1"
    "Y848No4f2t8p4gEBwFU9Pd8vS/lmKYT0xpc0xlOjCwHl0rpMKsiKOqlWoOkgNTp73RLxxxtE"
    "bOA+bxAIjBksS+MMrQXk5Pu0MASSDlrm+TRUGBI6YmNswxixTYgFx9RqY/uzWLRfS8GPd3dH"
    "twlxQVmpt1JjKuLtIEJOEiZIQRGo+NGKoAiFgNAoIS0d81AQ6hbio4GuVBwcF7AW1olGxWr+"
    "gjyX1NulTJ77MOdiPYSANSYBuLXQ/r9JCVkBogjoBdY+u6ZWKx2SDPCZUqn4YhQd01cobCg6"
    "qpdKpUpfUpFHxR73eJYithie1QpaGMEbmxrcs0egPmDJ2AAYEFqygkD+bwLFHloFRCDue4r3"
    "JWHtPB1AKgI1ySKMCwXGmLhp7YZLx8ZOOiRDwNX9/fWiEFEkpRTOwFTwtRjfez0tBnlG8HGc"
    "pIC8WpjWAHjJmAGH1whaVH9W8WWUvyCGoqGBaoKQAMwVfKRARNPE9HPGZI3vQoJ2gtAag4bW"
    "l54Ux3+xP0YR9xsAru3r+5eClG+VbmBHOe8XxOg0JMCp+9TwrCIYLBHzwSEfJki9oB3Kn23j"
    "HUCC3Pt4TgtAhoEBOUofzPst+4wmItAEQBA7JqhZi18J0X/y5OT4XPsJ5gyAL5bLHS8VCqcX"
    "C4XvFZxxvYdz8UdpnxpfUq/336OU71PBUCZAjC5YKZgb3wbEIGYYAcwUfJgI9FQPNtJHjQ0m"
    "5rjnG0b9tExsmPEpCxhjdMOYbZft2nXEIcEA6wYGmmUplXIje4rF/hALZCqBzMMVqRlQTUAf"
    "cw3QQvFMEGKGcYC8TIB3+PD7lgEfamRG67RnkL+fagEOggAb6Gk9gIbWa5fU638+l9RwTgB4"
    "vLs7uiuKvtKh1PuklEoy43tjKV73p17OHsvZaJ8Chho9MHQs2XNqcK8NeLGHgsEGDB5MAUOV"
    "v9CNeD6IgW3gXlsLBPSAGzn0VUKs3bFDkXrWwQPAucWivGt4uOPD9fpUmVC/IvTPjU/DAjVw"
    "S/xnYUAwEZnJ7wkztKSCoTRwlr5AhFq/A4bmFcC0JYyzgH/OxgUsyfk1HRQiIKHMkAkNSW0A"
    "2pg41vp7q+P47H3tNp4TA3x6YOChkpSnKKWUFAKRUhk1z8MArQlwRqBDvhlDU0AxHcD7BsA+"
    "k0f/1PszHp8T/1vq/9zrWQponPcaWgTKaQbhdG8DIKCG92LQGoOmtahbi1/W64tPbjS274sg"
    "3KdC0GdKJfV4FC2TUfSmTLqWF/PzjB9IEwVjgUzsD91C4jAgCi0Dg8yrCYS6glxRxgKQNOcn"
    "Ys8D2H/GCpHcGzNdBPLFIWMAKWFcsckYk/5u2kvowSol4HoJhWMYpRRiAMoYdAD6mCi6e82e"
    "Pa8/qAxw1eDgjytK/VpEY39A/XPa58JQ5OkB+pyWhXl3EKkTZLqFSVeQZUDYm6YQPvKHABPQ"
    "OoEmI302NABEQwErCWeYwCn+TFGICEKtNWIXHurWYtPk5LEna/303rKA2JfYP9nZ2Xdssbij"
    "7BS7j/uRG8lTxOORAwbJB4SYQAxphAz1K5V07jIRSFvJMUNLGBeGedSfSQN5lw+t9QsB02ym"
    "INOkxs8FH9UHNNaHKN8wfaDJ69p9PzZG6zh++OIdO968t/aUew2Ajg4cp9RNJUC3pHDMyLnG"
    "JyN/odclEZTClZQzgJAS5eOPT+rm/juegejgE2El6fUJ1RSB6qTwoczvo/8dX9rm/+G+X1q+"
    "HNpXN8l+8mNNy99kX0Lng1ZSWxyCOZgQQtWUOh1Ax7nFojygADhxfNzEQpwlPPWTHQVP85gG"
    "kIzGgwftX6cn0J185U788vPPxynXX4/CmWeiZkxK/ZIYLKNDnOEUNai/+c/7m/+cez+tOfjf"
    "9GB0z4WUOOkv/xKn3nADGscdh4ZvcPGfoyB2n09HP/Oyo4AWakmBCdtKIdAphP704ODfvSWK"
    "1N7Yc68+/E/HH196A7C2EEWnR1IKb/j0hNGDCRSDQp4eYoIM0v1n3W8c89GP4sh3vhMAcNSp"
    "p2LLCy+gsXkzCs6YIWBxUIlZys10GDrzPu1UckY58brr0L9iBYSUOP7ss/HYww9D7N4NRUYT"
    "Q9c0InBvA6/zRhVLO5x8OEmei4a1J5+3e/flBwwAt+/YEZ/a1fWNolIV5QCgiCE9Gj06ZR7N"
    "8zSRGTpjSPL+0R/5SGp8vy077TRs2boV9aefRoGwhKLGnkFjZMrKrMOI7kdqGPL9Fddcg74V"
    "KyDU9Gl89TvficcffhhibAyKdzHlGF/kPKfG9gbndQw73dkkIAROLxQ2nQNs/G4c6/0KgM+U"
    "Sqrc0XHUkeXyJQUhRMQoNZjehcYEyGsqkCaGvFhKiaOJ5/Nt2amnYsvWrWhu3owiAYEIxPuZ"
    "vJ8WnGyo/Yx4/gmf/jT6Xv3qjPFDIIhI32OwS3kGJqAdySlIiPAls5VACAFjrdZCHHXR7t1f"
    "2u8aYLmU6jfK5cuKgGkp1ZLevpY4znJ6Kg65GEq/Q0SWEALLPvxhHHn22TPu35v/9E8RveUt"
    "qBqTpH0cfFQQuueK3CiLIbQ/5HiOX7cOvSecEDS+337vi19E49hjUSfUL1kI4uExLyvK9Eky"
    "HUUBK6VUWsrXHxARuHpqqtFU6l0AhPf8zGVcPv0KjNkL3hQSOiEEEPQ9KwQGTzkFutmcdR9P"
    "X7MG6vTTMaV1mh2k9QkaUtxrIKqfij5FFT5T/sddeeWsxvfbgjPOwHiz6Y0zPXTNRzPZcYNd"
    "JwHePRXQJdKFm5IQuGJw8ANfXrmyY78B4NxiUQ4NDXXJKBpSUop0p3xsZEKLozNV/wF2oKIq"
    "L/Z///3vx8ToaFsgOO0Tn4A67TRU4zj9T5BMoCXDYN4OKQGXEUiWNr5q7Vr0nHBC9nrEnO37"
    "t9yCB/78z1EpFFpS5LyCl6DGZyKZez6tjdDQKYWIRRS9/4/Xr6/vNwAcv2RJxxrgoyVrM/kp"
    "HbIFV9KhcfuAB9CcmhvGpzm9pRIe+OAHMbljR1sgOPXjH8f48uWoeTFHKDzzmP2npIzBqP+Y"
    "yy5Dz/HHt2X8f7/lFvz4r/4KK/r70V0sZo6HZz8tI6HUOVh6K3jZ3FdCCWiEECoGTt2vIeD8"
    "p56qxlKeI4TQoXIs32HODhm6Cog/0NDBWEC4CuNAuYwf/PEfY3LnTpg2QPA7112HPUuXToMg"
    "oAN4MSaUekJKLLv4YnQfd1x7xv/mN/Gf11+PlQMD6C2VMllS6D94OOCFHwTqAKFzi+mmGaGU"
    "Kr6zq2tJO0WhtjVAQ6nXws3FR6/RSxs7nCdz+hdZdGZ1AsvxkZMBCCkRSYnBSgUPfehDmNix"
    "oy0Q/OZ112H3EUegRkcIWXdSyBj0vWUXXti28e//5jfx0+uvx4kDA+gvlRI28SxCgc4YLnRO"
    "M42xtDeSO5tjZUGuhShKiddXKv/zqKOOKs4ZAA5F3ZFSJSmloJdyCdaBK3KoH6GUi1W7eBUx"
    "IwrdbyvHBD8891xMtMkEZ157LcYWLULN5dA81WxJE8nzpRdcgK5jj23P+Lfeip984QtYOTiI"
    "Puf5Gc8NHFeq7tk5AvP8dLjcfb4lhNIQnHw2jq19+9kvvtiYMwB+53WvU1cNDZ0dufxTOZTZ"
    "0MUeBOVgaU4LQMhjsDJxhi3895yxCkphoFLBj847r20QvP2aazC2aBHq/rejKLn5cnAUZcYK"
    "hFJY8slP7pXxH7nhBqwaHkZfuZzUIbhizwq1aUOy10Ilc0tifWg43LKhbymllFF04onj42bO"
    "AHjhhRcKVqm3FoQwGdpnwq/lun3WmydoezeP8zlpIb16iL5ekBIDnZ34z49+tG1N8Larr8bO"
    "BQtQY+EHJNeXUQQZRTjiE59A56te1ZbxH/jWt/Dol76Ek0ZG0M+M3yKGA6VtzOIQguT5ocKV"
    "5YBJ9lnUpVywX0Tg/928ecoKsdIIYTPGZYbOG8RouZgjUFJFYJAIpK6AQAwtSImBri78v499"
    "rG0QnLFuHXaOjKBKS70sM1h4/vntG/+22/Do3/wNTlywAL2lUkLRPN+nRiYhkw4SITAVXvCc"
    "snSQN8SSjEwUhcAZPT0LZhOCbYlALcTRbg8ytfL0RoZrLRUvDAR54BAsc2jx/JwY6pngp2vW"
    "YHLXLph49otn33rlldg5PJzRBB4MC//kT9C5fHlbxv/Bbbfhsb/9W6waGUkFX4ugnKHo03I9"
    "Q0gTkfmQ+LnKa4/34UQJgTd1dJw2tHx5Yc4AsEIsEG7GzpadoKNcgRk9M/V0dmIQMmygKpj7"
    "XIgEBJUKHlmzJmGCNkDwliuuwI7hYdQdCISUGPnIR1A55pj2jH/77Xjs7/8eqxYswEClMj0m"
    "EkolSXbDmc9SEc1a2iTXQ3RMIccOdCBJJeA+abBenzMDCKtUJHwFEGy+vlBOSrt1XZ9cqH8P"
    "rM8/iGwysBMUTU4YDnZ14dELLmgbBG9euxYvDQ3BSonh885r2/gP3n47Hv/yl3HywoWJ8Wma"
    "x0QwPUe0u8mywk/LeSSjj/53Le98CoGCAMpaqwEcs6qjw+4zAFz8iFKl6Q+AtF3RHQPtv/Mn"
    "w79P6ZAWNGg8pgfpxwKo1/DQQU5e5EDw2MUXtx0O3nz55ej80IdQOfro9ox/xx1Yf+ONeM2i"
    "ReivVFrSOJBmDeFKyrR9HSy3zxwPdQbuaMhOmEUZRHCGJZ3PUsrhkS1bGvsMAAvIc3p7l0es"
    "kYGmJi3TudEd4dO3BsID2Fg7pznJOmtA6/usxcqDYP0ll2CqTRAMtSn4HrzjDjzx1a/iZGL8"
    "YHgKeWTgvVCdJHP8vA7AQJMpCvlzl1o1+XzV2uHZUsFZj/yIjo4eyxsoPQhogwLbOUmonw8W"
    "ibw4lneyAicgT0f47OCJyy9vCwSyjVG9h+68Exv+8R9x8uLF6O/snFbuIU3Dq5yBgR4RagTh"
    "gzz0ghY2yUVLgysTi6RzaNa5BGYEwMoVK+Qg0MfbiC29FonFdEtrAuzgwA8q0CXDh5URiHUz"
    "gQC+WNTVhQ1XXtk2E+Qa/667sOGmm/CaJUsyxg8J05ZQ57IikDweITHnvms5GCg7UvCQ//BO"
    "mSkuOYQUhKjM1vk9+2CBtRX6wzbQBRxSpnyswAa8OVT+9CeLH6SYoVrIiyxUGG5Yt26fQfDQ"
    "XXfhyX/6p8T4XvCFhFtoUCswMsrrHS21fSr+aDynfYCMNSTRYnCOSbRBYc4AaNBLoymyffsS"
    "7c2naKSKlMb8QJzMm/yJM4DlAGLVvBRA7j5SCkPd3fjFNddgYnQ0ncCxne3he+7Bxm98A69d"
    "sgQDnZ1JJy8f0OEA5MfG2CAY/4EWwwo2KQZC9RLKyoH00bG0mRMAxjs67Datq5lr50Jz7jFF"
    "yq+7z+gBttOWhQ0uhkJagGuJllBAwBAphQW9vdi1fn1b8d5vT37/+zhh4UL0MdoXLNzw1I+L"
    "OMszIw7+QJ0fvGeQv0amwLfMLnTtoqa1jTkB4NlHHrHPaj2Rzn8fSj1Cna6kKghOXfzKnbxU"
    "kaaRVO0GPIwa3zIRJqRE5ayzcNTb375X9P/Bz34WzUWL0AgILQiRsAwDH9h+c4q3obBHbtyJ"
    "MpNfUM3l/suSDCDtICa/1bS2OicACMA8X6/vMITu06nTODWxoeGZDpSmLd5YlglEsLCBvHEI"
    "7lHe+50hOs86C/2/8Rv7JADf8qlPYXtPD6b8XADesIGRT17upTrGhrw7cLl6ZlCNAo4DnwOC"
    "Gd9PUCmBGvJnwZsdADc0GnZjvT5lABj2Bz4keGbILPMS+Fc+gkUXcmwpG9NQQk66JScz1Fxh"
    "iQdaKdH1jnfss/HTYtEFF2C0pyfTbSwoCBj4LKtuZo6ZgZ2vdWBD9E9/N3tC03ZxS0IBFYNV"
    "Y3asm6UpZDYRaAHIRvKDltcCqODjOkEwD7ZonXpdcIFHZ+1gZWXaiJqhPkat/nM9bRrfWtse"
    "CHp7UxCAgKBFB/gwxM5TSx3FC8BQGZ05hmXFNEu0VeY4kmsD/PtSW7t1XOu5ZQEAtNR6j0Wy"
    "8rKhgoOIkrxLnWwg7w9N2dpyeVROpmBYdtEiHqVE9154vhACup0BpAsuwI7eXkz5aV3zNI3z"
    "zDT/Z6N54ANB1OOp11PdwLxbTKv8TNrn/1cnjC3Gtd5STZbemxMATGztqDHGZC6l9vQTEoKB"
    "MnCLBwTUsWWDHxmBxGM/o2DPBN1nnYWBNo3/H9/+Nj7/h3+YXHbdLgh6elCji0nyfWXsJQJG"
    "RmD0zgZWOm2p+rFLxuh7xjmHB0QM4JlGY9N4paLnBIBVXV26GcfPGBfvjWcCsuMpjeZd6EgP"
    "2n3eBmb0nun6OeS8R09+71lnYbBN4//wvvuw8Y47cNqrX42Hr7iibRC89VOfwmh3dxIOAmxl"
    "AyNzYBkEje+Yab3DnPMnWofrpxnBCfUmgAfHx5/86saNzTkBIBLCjDabTwJQhtAOML1WHp9x"
    "i6LUMpRzowcXecwRQ1QPcGrtO/tsDJ5ySnvG//a38Yu77sKvHX00hnt6cNSCBfjhVVe1DYIz"
    "LrwQO7wm8OGPXf8XYj1ucEvOj53Bw0PORIU4nZXcJM5ljTENJNd+zi0ExIVCfWOt9rOGExh0"
    "QsR0nd4Z1tjJozQRiH2WeXXLej05cwHvq/H7u7showiFKMKRIyP44bp1bYPgbQ4EU1q3CFyE"
    "hCnbfzq9bei4Q6KannfuIOkSdUJAG2MacfwcgIk59wQ+tnNn9WeTk5trbiZrg2T5FEMGhejO"
    "8lU6giKPfM/OkH600CCvegHoO+ccDL3hDe2Vd++7D7+46y689qijUuN7MBWiCEuHh/HwXoDg"
    "7RddhJ19faiRNYJm2l8RurQ7APSW0AC0Cl5y/loYABATzeaGozs6mnNmABfy64jjrcZao8kf"
    "pZMeOhYQAYO1HChL9YIxPeDloQkUBs45B8P7anxa21cKloDgob0IB2defHESDtxiTy2Cj3v4"
    "bBbJEc58/mLqSN4ptY//Qsinq9Uf1YWYOwMAMEOFwtSeRuPnmhg9ZuLD56AZZLI4ldNv2IJw"
    "cIQzMBkAg+96F4bf+Ma2jb/xzjvx2qOOQl9XF6QzOIRIu3b8LZISS4eH8eCVV7YNgt+85BLs"
    "7O1Fza8IgvCqpLngyHGcUP0EDAiGhAXt7FKzFg+Mj//8hVptfH8AAEaI8Senpn4QCyG1+6N0"
    "SjOXFtrANOv8YGZ8zuNb3gkEMPy7v4uRvTD+k3feideEjM+KOB6EBSmxdGgID6xd2zYIfuuy"
    "y7Crvz8FAWbx9oxDMEObWUIjDb/eBrF7rI0xptl8AUATQH1/AMDubDTGHh4f39AAEFtrPd1o"
    "sgPc003OCQhNvWoYA7SwBzlpw6tXY+RNb2pvPP+++7DhttvwmiOPTNq4SBnW8hjKXoukxJFD"
    "Q7j/8svbBsFvX345xgYGUCOzhIaEccs+BOYpzj13bOEqYy1iwsrGWrG7Xv9xr1JVh4u5MwCA"
    "SQDNWq22wbh2UxoOjJ/cOAe5PATY/LpsGCzu+yPvfjcWnNrelc8P/fM/Y8OttyaeX6lAuJk5"
    "W+bqddOuWjLrp//PSEocOTiI+y+9tH0QrF2LsYEB1MkcggikbEGmzTt3AYY01HncMnPGWjSl"
    "FN8ZG7unqvUup9/2CwBqXUpNPjc19R2d1JgR+z/1IYHcm1mWXp3tYEPr9i14z3uw8LTT2jb+"
    "E7feipOPPBJ95fL0bF10NQ9jMjN5tgDDj6gJgSUDA/j+RRe1DYJ3XHEFdg8MpHUCmyPmbA4r"
    "zKSbDKZnI/U2iD0jW2ubcTyxvdEYawA729CbbQMgntB69N5du35Qd2EgdjugQ9OjA9OzZOcZ"
    "l9GZCRy8f77w3e/GotNPb8/4996L9bfcgpOWLEFfuewnT2o1sL9pnewrBQPZf1gLJQSWDg7i"
    "Xy+8sH0QXHUVJkZGUNO6ZV0Bamg+BS0PSVQz0NRZk0zMh2NtjB2rVv+9U8oYwO52zle7ALBw"
    "iJqqVn+qjbGUBfwO2JByDxg8xAgI0KOfcrVv1aq2evoeuvdePH7zzTh56VL0VyrTncvcsBwE"
    "7tayjAtlAgBL+vvxL5/8ZAKCNq5FHHjd6zA2NZXNDAi7zSQWLVub0CC7EIWP/bG1aDob1ISQ"
    "N23fflPVmN0AqvsTAAAw1iFl40e7d38tllJ6IehZwAMhXRMH0/PltlAZX4gBSJdOyQDBvf/w"
    "pz6F8Vmu+Hnwnnvw869/HauOOAK9pVKShYS83u+jMclqXEwH+M8hAAIF4Ij+fnxvzZpknv8Z"
    "QHD/7bfjoS99CeVicbqYQ9cSCGkD5jAZZiBZl7EWTTdRdNPn/saYuNl8zlobG2C0HQG4twCo"
    "1Y3Z+bOJic31ZnO3dkWh2M9aTYSIIcurmQC10dw1c++RziqLXcUi/uPCC5P5AAIgePDuu/Hz"
    "r30NJy1ejL5yeTpH9iDixqWxmc74TcQgDxN2emwEi/v78Z2PfSyXCe6/4w785CtfwcoFC9BZ"
    "KKTGD1I9W3AylCVRdvWer0nsbybFOLlpz55vVJKxktF24v/eAkAD2FaRUj07Pn5LbIyMHQKb"
    "LBWhFNeSajFK5uqfx2JrbahwgeEAAA9ZSURBVDJRVLmMH150ESZYi/eD99yDR2+6CasWL0Zv"
    "uTw93z+ZVj2P8q3W6Q10hm7HCKk2AKZBYy0iAIv7+/Ht88+HjuMMCLzxT1y4ED0dHYkGYWsJ"
    "2rwUkTGfD0lwbGq86HPGT8WfMbamdeO7u3b9aMqYsXbj/94CwAIYnTJm/M7R0e/UAZuKQb9T"
    "ZApzzVbS4IUiILzqNthyq+AguPhiTOzaBas1Hrz7bjzyD/+AExcvTmmfr9sHt/Ci1XrasO6x"
    "IWHAUEC452DLvdAp3CMAR/T24l8//nHoeh3WGNx/xx346Y03YuWCBeju6EjASJeTD2ghmsqB"
    "ebolawobGm6tRVPrxPmSwpPYNj7+rYqUEYDt7RSA/LZXcwW7uNJblnJwSRSZ3mLxxNBiUHRF"
    "MIRmvmITQWR63Wdo/pRCoBhFePp738NL1SoevflmrFi0CH2VSnrxiQiNNXiWCa0Q6m8k7tN4"
    "DRo+6AKRxkBYi1KhgPX33IOXJifx869/HSeMjKRgtDwEBVYTBVk6zpB9MI65NLnFcYymtWho"
    "jdgYNBImsFVj8JXnnruyaW0NwJMApg4UACySNHDk8cnJDb8+MPBeN2eQoEaXbCJEG7ioQbCF"
    "IELr/YQmSfAgqD/zDBZ0d6O7oyMz+wgfO7e0acIt/RICAA9PadwmCzpYeu++L6xFRxSh/vTT"
    "GO7sRFehkBV9lAHImkIhQFhufLKCWCOOE6NrnXi/MWhqjVhrsWXPnls2V6ubmtZuB/AU9mIV"
    "sb0FAAA0AIxUpOwZUWp3f6n065lZQNlcQXw+3KSnQ+bOetHSGQQEJ6QoKIViFGUmpODtaaGh"
    "VT6EisAysNTI1OiUCeDDBpJZvCMhUPB9i0xs8tSTah3v+V4vpWHKfU+7NC82BnWtEwA4IMTG"
    "2Kox8Vefe+6qprUxgE0uXceBBIABEDWtHdwwOfmrX+/rO0tJWUhnEHEGlmSaGMnny8+ZNoYb"
    "PHTNQWixKMoQNmdEja/9R1f2TEHg1/4hzy0DAw8HmTIyo3G+dhD3dAosOOMDSLIpr/Kdpzem"
    "PT4BQcIM4rHR0c9uq9e3Na0dA7DRDQIdUADAiYyFHVJWxpvNnyzr6XmHIj3sEtkJjySfH4iD"
    "gU0KlZn8gPXOz9Q6lts7QHsQeS2esUCLDvCfJd6MGUrJ/Pt03UCe8tLv+FXDY0//Ls57wUc9"
    "39G/mWw0Xvjm1q1/10yGZ34FYNveGnJfARADKGprB7Y3GntWlsvHd0TRCGTi63RpV7rKt5Ay"
    "7SYOXhHLGiRnvPwsAIJg4wnp0mlZCIoZjgIk49nMWDOOIXDaJ8WkvBVF0/UA3fdjt2JY01rU"
    "4xgNJ/iacZwYP1k1VPz9U0/9H2Otca1fT7jwfFAAACSXHS0A0PHTPXseeP3Q0B84zxbp9KVs"
    "KTcBJNOsM9FGQZMJFbwNii/9PpvxqRAkWUCm6pZj1JAYDIGAFpAMAw8HR0udg8R5tiLotNrX"
    "GnVP/R4cxmDbnj3f2jg5ud55/1MAtu6LEecCgKarIwxXpCw2G41Hj+jtfbv0M2+RZkbMtEoH"
    "snMO8YxgpmVV7AzsIPioGk0BcwwcDAFctHEj87GFPBAAmZVFjV8Emq0J7OspTaf26576/Xta"
    "25rWu77y7LPXxIn37wHw+N7G/v0BALgBh6GmtR1b6vXRE8rlI0tKLUnnFAx4sGCXNoNNjpgR"
    "iPR6AwaG3HbqnKFXsIUgQ69nilauCAMWDmYEQQ61G75WIFsbUDPjN1ysr5Nij8sEbNNa8YVN"
    "mz4QJf0Nscv7R/fVgHMFgHZ6YASA+tnu3Q+/bnDwHVKIgnAgaEnlvJFZmOALNoExyIwXjFDx"
    "R1bYAqs6ZhiBGIangiG1HtIGhtQJaGjILBFP6wB0IMqxgKf0pi/uOMFXd5Tv83+nCcQjL754"
    "zWi9vr1mjAbwkgOAfrkAAFd16gTQG0mpfjg6evfrhob+QALe1fOv8iF97XTu4XSRJJa/i5wr"
    "jwSv8AWMH+pB8IWcTO5PDc8Mm6aNOR4PanxfTvZ5PjW8tdBxnMb82Ct7J/h8vh87Bohdyfql"
    "iYkH7t627Y5GQv01AI8BGJ+L8fYHAIxrGRs21hYrUhZ2VKsPHNPXd5Ykk0pw6qaJm2RXEftL"
    "n/M8PsMSbOQw1G2TGXAKKH5eChas5JsRcmQZ17xQYIh20KSqlxZ3iNeneX4cp4Zv+Lzfiz6t"
    "7WSz+cKXN2++gpzzXwJ4rt1RvwMJAF8d1ABGmtbaqTiulrXeMtTV9ca83NwysQh2kaSdYeXv"
    "TAqYM7LGM4A8BuAjj2Dr94KNBGYGmYhesAwYhoQYTSjfeEonQo/m+A0SFuIELKZuTO2vN236"
    "EDmsUQDr91X4HQgA+MbREoC+prX26Wr1uUVRpPsqlVWYXt2ytT6PdM27aeHHGEKQ+W/4enot"
    "OoNer8g6aHn+n8kIZijqhB4bkh0YVgegK4JTBkiLO6SyV4/jjNfH7jNNb3xrzec3bnxfJKU0"
    "yU5PAfi5U/84lADgU5J+ABULYMPExBPLisWurlLpuNB1fi0Nk+RCS+vDALkXjLZbrrsjxjfU"
    "08l7NqcYExJ9oYpfUAew3gFNlL2v6mkf0wnle8N7FkjH+Y1BHMcmBsT1Gza8t0NK1UwQF7uC"
    "z4tzpf4DAQBfIZwAMASgCACP7dnzyLJisaubgID3Ahrq3WwQJ9QlS43tDS1CNM/H31n8FyTm"
    "h+r/KXME2MGwXkK/gqf2+b2v6sVxWsNvktG8hvN+MrCTUr/W2sQAPvfEE/+tJGVUTxS/AbDZ"
    "xX69vwy2vwHgK4R1B4LIg2BxFJluFw4En9OGMIHhtOsnQKAij9b2HRhMgE0E607KVAU5pRNw"
    "8AqeIUUhQw3uc3nyOM3ptUbTxfsmHcFzKR4d2fO0bxK2ME1jmp/bsOF9RSmlM751Xr9+X8q9"
    "BxsAcCxgAQz4/3hyfHxDtzHPD3V3v9FaawUgvCcb1h3jn/uLTbw+MORxSznXhwpuRH6VTl4d"
    "nwzIGCLo8gZ9NBnoSdu1vHhzho7J4E3DMUGd0H7aTOs7qZLWrt2ff/LJ/1WQUjanjb/Txf2p"
    "/W2oAwUA6/rSpNME0gLYWqu98NLk5A+O7us7C9YaCQhDPZ83iZJJjwyh5JZeQxrjWS8dVfcI"
    "GZ42awRqAFTJUwZIr8idVus+dqfDt2mMd/k9HdbVXugRgO6ZmtrwxU2bPu5A7zG7B8Aj+0v0"
    "tYyW48BuBQAnADjGg60iZXHKmPj8FSu+UpSySyqVLEPvlm/xt4JbJq6gFJSb8TNSCn7VcqVU"
    "ul6hJMvBWYe6tD2NFpoCYwm00MTLxZYykFf8BGC0R7AFCD7VI2HAF37SAaCENawWQjw7OvqN"
    "255//lvs/O0B8DMAO/aX6DvYAAiCIJJSxsaYDy5b9sn+3t43KMAqKYVwa90opVJAFNxqnxnj"
    "u4UklQOHIitxeVD4jlc+sihI8amlI5c9R6B13JK4r51RtW/OJEO52lG/8R7vP0/qAlZr2zRG"
    "37d588efqVa3uXh/0Ix/IEMATw93uce9AJSnt0fGxh4Wjcb6xT09ZwhjrBXJnGaU5vkFqOlE"
    "CP49MpyqjYFx0755T81MZkFoPZ6OuxmP9AZK/8vH9ThOvNulb02axvlxev8eLeU6RkhTwyTc"
    "2FgIMTE1tf6vN248b1LrKWJ8C2AMwKNo8/q+Q50B/BYBOBrAcQDSpc0rUhamjGl+aPnytT09"
    "PSdJa60UQtCeQuW8W/lQ4VhBkdAh/eukGVXxVbrJIhd+bqNMLYFQPi0W0eZMS1vfyRi+B5Uh"
    "hk5DBBWWWtumtfYnzz9/1Q927nwsoJ1ecoJv/EAb/2ADwLPyYgArAHTxN0/q6Vl2xtKlV3ZE"
    "UReESMNC2nFMjKzI0umKrVOcLg/r+gwU6zCSbJApQ1cBgUizAs9EhhR6YgIMTT7nGUcDsFrb"
    "WEoxNjb2wI1PP/05D3w2svq8K/RMHSyDHGwA+P8cALASwCDdhw4pVd0Y/fuLFr1z2cjIByJA"
    "QErL284pICgYBHss2RqEGSHIys30Kt1MNuLbubgAJECgpV/62HX5WiOlmJqc/MWXNm26rCiE"
    "ZbHej6X8yt0aB9sYeJlAUAZwPIClvmDEw8L7li797wsHBt4dJRa1yrmwYNQuyQrlaQZABKBk"
    "I4hChA+bp46GFpG87iCA4I0fNgsQq5US1cnJp+595pk/e65aHRWAsK1TBIwD2OAKPfrlMMTL"
    "uUUAjgBwLIBuvj9lKaOqMfF7lix515KBgd8vRFG3MMYKpSABQfsNudEziymTjiGZZ3yaErI6"
    "gx8WBgNFSy3BWmuNQayUmNi9+5Fbn3vus7vq9alICOnat3jZ/HkkvfwHJd4figDw+9DtQLDY"
    "pY0IMcKv9fcfc8rw8Psr3d2rosQrrWMDIfk6BchZtm6GA+dzGrR0EZOQQD5jrTHQSolmvb57"
    "dPfu737t2WdvloCQUorYGBP4m3Fn+Bf2x5Du4Q4AygYLASz31cOWDxBP+r3Fi9+2uLf3t0ud"
    "na+KkgEdCwBKSpHXPBKawj0QB7IDVrRJ1E3DapMLOESsFOJ6fXx8YuLH/7x1680vVqs7K1JG"
    "U8bkXZtfB7AFSRfvxMvl9YcqAPz+lAAsAbDMZQrBfSxJGdXciX7z0NCqE/r63tpdLq9EqTRc"
    "FAIyicVGCCHEdIUwM3c6zwJYR5H1JWIhBIwQQkuJuNFoNOv1p0fHx39885Yt91pr47KUUdNa"
    "E6B5vzWRXLX7lMvt9aF0wg/FTSDpM1zqwNCJGS5lL0qpIkB4zzupp+fIlX19Jw2Vy8cXi8Wl"
    "hUJhQEdRZyGKIIyZ8aC1Kx3bZlMbrcca9fqLE7Xa5q2Tk+vv27btJy52oyJloQHoAMVzw4+6"
    "YdzRl5vuDycA0P2rAFjkgNDDM4ZgLBFCRq6qyFIu0RtFHUPlcvdAFHX1KFWRQsia1vXdxkzt"
    "qNcnn69Wxzg1d0lZbAC6mayZMBttWyRD4tuR9OztRJvTtcwDYOb9LLq6wWIkvQYl7N0EF/Dj"
    "ED7OS6IKfXla22QUYC9/1k/ctcelc1tdjNeHw4k93DblQsIQkkvTevcVDHPc/MIcE47et7sa"
    "fuNQEHevZADQfVfO+H2uutjnhGPBAWJ/Hp9fLKXq0ridSAa5xg83o79SABA6lsgBotPVFrqc"
    "hig5UETuc3IGr/Z0Hru0req8fNIZu+oMbg5Xo79SAZB3bMrdInbLA0DTAaDpQEBnaZvf5rf5"
    "bX6b3+a3V8r2/wFtjVplE/yybgAAAABJRU5ErkJggg==")

# ----------------------------------------------------------------------
prompt_icon = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAAIAAAACACAYAAADDPmHLAAAABmJLR0QAAAAAAAD5Q7t/AAAA"
    "CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH3wcaBCMgIPaQxAAAAB1pVFh0Q29tbWVudAAA"
    "AAAAQ3JlYXRlZCB3aXRoIEdJTVBkLmUHAAAgAElEQVR42u19eZSdV3Hnr+p+r/v1ota+WFi2"
    "vGCDFWsgwXGMMXFYwoQsw5lMzAwJxDlAlskykAyZkwOM4yFkGWYIJJNAYmxIOAlgIHOSwCQQ"
    "YwiOAZsDNl4xxqssW5JlSd3q7b3vu7fmj7vV/d6T1NpsydF3Tp/ufkv3+27VrfrVr+pWAaeu"
    "U9ep69R16jp1nbpOXaeuU9ep69R16vpXdNGz/QbHL/mEAWCkmeORs66009dzfbDXL7/Csd1/"
    "f8fNfhcyv825xSds7+6r3SkFOAmuictvrGjsOWb2H87vxce6L3z/C6mz4t+A+FzYhc0w3fWw"
    "/QlwZwJcjQHEENuHq+chbgGmuwtidwHyIKS5q3niH2+pH/vUnqgc9bZPVnbmnubZohR0su9u"
    "nrrAzH7uwj4AjH3/R14ByGsg7hK4+vkAjYErgAxAlUCsBYiIxN86hfsXAQDJPwIAGUgDSANx"
    "fZAZ3w7IbSBzQ/PkTR+vH7p2Z/gMI/NffW3/lAI8Tdfolqu4c/p/qGY/d2G/s/nKTdWGV/4q"
    "7OJrxPWfSzwKMh0bpMhElO+PCARJP6dLZPjvIkkvRCCAEwAOoEpsHyJNj7h7IwgfXbj15z52"
    "sioDnUS7PS7u2NjFf/n7aOZ/mrhaDTMqgAiBmCgKVwk73SLB64MMuW1pLYlARIrHRSmIVwwn"
    "ImIhUjm7AObOzQJ+38LX3/ipyVd8vVM/8ZmTAjuc8AowftlnR+dv+tHexA/85RWu6b8DsBdy"
    "Z9xBQEQgv7P1Ro+Cht785S4HDRV62yKIk5b4BSIEEYdgFtTjsOKscbYPrrrXzt/ys/8ZQH/8"
    "kk+Y+a++1p5SgMO8Jl9158js5y7sT1z8l28Xu/B2qsbGALJEMBR3uRZ2NPHEQNAObwlICXFQ"
    "MbLPb1sMbwEkPiZOgYTCPQRlCc9DRATONXOGzPiNzZ7b3ty//30PnqiKQCfqjh+/6EP/XcRd"
    "xZ1xJoh4yEYgYhChFHyyAn4nMzNEpGUNSCmGVohBwYtoSCB+xxeuAeq7f85/B5x1+e+ArNgF"
    "I4KvNfvues3YC9+3a/p6lhNpvflEAncAQL0dPz7+oj+b50736mpknJgAZkNsKhhjwExgY8Ds"
    "vwwzDBsQEYxhVFUFZvavNQZMBGb272MGUf7Zf49/KytSeg+R//vGhP/N6vVGvZ/Td1MZsGH/"
    "vyCGTRdcjV7UWbFlR//hj/wtAJ64/MbuKQvQMveLt/96Z2TTFbdSNXIBsxEA5IXCapdrMIcg"
    "UEq+P4M9tB4b/vvgla1BBoHKMjhlAVrWIH5lbOAthwjC+yxAaGw9XwHVO+a//uZ3nwhugZ7p"
    "Xd+7+2o38QN/8S643ju4GrdEZLzpBkB+F2bhxV2cQ/ikHG1FoOz9Dy34YbogyedL4Sac8vs5"
    "WnAKG0R3IE6CogTQKALnwvtd/7H+nm9tGTnnFxYjj/GvSgEmLr+xO/ell7nJi6/7LrizKQmW"
    "GMTsPXoQLDPnD1tggFLwZdjPha8npiRYAiBEIJGhgWCJAyRbhBAhiLPJCmRcKIUliBZCosDh"
    "4ERCZCFwzlmxiwZkfmv2ljf9/r8+Bbjogz9B3Plb4hHLhg0RgZW5BwBjTDTqXsBMycTHxxCt"
    "BZQiKLLnwLufBsQ+1BA4V6iIaADoMlfgvK1P0YFzwQVEF+FcthZaKZxAXPOt2Vve+IJngkh6"
    "RhRg8vv//MPEI1eSqYTZE3Ye3WdzzvExLgXJbBSZx2B1B8RcWIAMC2iQ8TsYE9iKErKv12SQ"
    "SzZAWwAAcNFCSFaE9NqAB0QQFEEg4hpne7ae3X56tfYHZ+e+9LLFZ50CRH8/efF1d3E1uoU5"
    "CCmY56QAQchEyDs+WQAv4AIX0CAfkIVKhWyzWxgO/AAd7oXnBEX8X7iIZPJL0BeVJoeQgHNh"
    "50NKd+EcnDg4a6242th6/ys6577ln6ev5+ZZowBj3/fBzsI3frGeuuTaPeDuFDMZpmzCfchE"
    "CdVHwbOyAKx9flvoCRtIEHJUEAmRrmSFUcoxzPvnnS7D8UDY1fp5H/9LthaiQWH+3SuIqJ0f"
    "Hgtg0TkrYvvknP3l2Vt/8U+fFQoQ/JpddvE1c9zpGsOmArxghwseYDKF4OPOJdZmXYNBLkie"
    "QTpYW4I2IyhDlkKTPZRCQKh4wMtXRQRi0xPOKZIoKkN4rROXFMgrQlCGEFFY1wDiYOu5/z37"
    "9V/5rye1AoSd76Yuua5HpiNMpmIOKD+QLsZwCO+yG4DCAN5CqBiflUKEqAEiCiDmhA9Bv1cD"
    "P0JMCUche+PQigVEUb1wLVAYnkepCM45RRdLFi6yoAGB1VYguAb/XsC5xiuHXfjjmVt+6ddO"
    "SgUY3XJV1bv7aix/8YcXiCtQED6b4PeJc0xPBCLv970ieKDHJprqIEhiFQGgiBgIXCoKSREZ"
    "ZKVQiSGRIXiAFJGjw0KlDAUNjLyLVWSghe3Rfn7OiUtRgSssgEsRglcCC2nm3ztz6y//xsmm"
    "AARAVlz6kf3gqsvEFbM36xx2eTT7XmhReOHxJEgeKnhKvAAVkUMRBiZPMWgB2vkBvaPTY0Vd"
    "gAzE/SLDmEBXJIa0BQBKUy9w6WcXBA/JQFGcwNoGIg2crX+Fznjznx0PYHjcLMDyF3/oMTZj"
    "64moMsYghnpZ8OE7MUCK1lWMn070kOLqI1NYCF6TPUkRStRfKIPIYEow/Swt4yDK3EvhKqzT"
    "VgAp1temPQHBqAiuJI0iFnAxKgh/01rr/65dhK3nXmEntt58rENEOk7C/yKbsZcSEUfh+4QJ"
    "e1NtSpQfhZ3jdx/y+c1vWlEAFDNYuoKULWBOOziSRNklUCsTSAOIPpNAQ8Bga9dHZYmmPNPB"
    "KiwUbfahqOESEDqXlUAkuwQfHfQIi9vW90a37DuWZFF1TAV/hato2zW/yVX3ck3upMxZEHjc"
    "ncQMZqRQMIPBIOxACUeBs6KIC+sQlYc1K5iTR4NKQS39p4LSTfwQJ+4PiRYQCn9GVCoYICG4"
    "hAlieMcQ8grBjiFMwfRzUAIDQ17IyQJaCzEcrERyU+QwamXs9Efnb35t94S0AOOXfGLE9B45"
    "uzO+5l4yoyDyKdS0++Nuj0phGDEcHOD8Q8o2J4Co5e9Z7ewSE5SKoG+TBsBe/JmIBiKAYSDR"
    "m+bBeoBECiUKODKE7oA+P4K+GE5GDGCL3e9ZQ2v9d+dcI7Z/796vvGnrCekCVl16bY+qbsVk"
    "mI1Jod2BhB9dQQkKOdG8RIBJ1G8J+IijuAMBlGjgWA+YXyeqfIuJD5EQFuUNYqjoBsignP5F"
    "CvHaANC5dlJIkhINI4QgAmddAoUaH1jrPCB0gKsX39Esv+Q9xyKLeMwUYOWlH76Bq9HLmcgQ"
    "+cIIIobh6M812WPAhhIIJBUJ5FwADwidmVo8f2YNc32AoGMaQCVxejVgOuNl0gje7Drn/TcR"
    "q909SAa1q4Ikx4Qq4ycJwftEkLYI/vGoAKXws7/3SmDhlIJkJbA+j9DMg/bfvXJ+4iX7j7ae"
    "4KgVYPyyz452mu2XdarOP5EZ9f7eGDCVVTgR2JHCAYVlUGEgDxM86RRxBo2sFMAw4bzTK1z7"
    "9gsO+HmnZ3u4/9FZPLJjEd+8bxpfuWMaziwH4IIr0ArQqhAWgUty1zl/b8ZjxXDKAsbYXoE9"
    "nSBKIFC5BSdtJbCJKrbWQpyFE2dd09u55+Y3P+eEsACrX3JNzZ1Jw0REoUQrof4Q+sV6PjYB"
    "AAZrkIFhFnwEexxYQlIgEYRBDBB+ZyKcvUHw4XdeeIAajyCgkJzpVN4dLPYafOamx/E3X9qJ"
    "J2YmUDFQWymhgLiSIErKoJF+UBTX3tmSAKI7AAMYha+VQJwLmMDBWQeBg21sshbWLl7V75z7"
    "P48mNDwqBVh+havM4x+5zlTd1zGTITYt4ccQL5j3GAZy9OGBGUy0cIwOcqSQ2MA2Bkjp38wJ"
    "GCZsXudw3Tu+57DuQ0TQWK8Q+2Z6+LO/eRCfvw2oyKFvvYKJEnSbEygBYas8rIj3NfPnBeqs"
    "KCXIRSMiNrmMGEK6nDn0jzWz2P0vv2RCGvOIriMuCh3dchVP3rypY8i93pdxmSEsnxJ+MPmp"
    "IJO88Dk8roXPwRqwCYJPBZyZT4iKY4wq8Aw5hMPeBUTJGkxNjuBtb3gevvD+5+Pi82sYwzAk"
    "MMQw6n+nn9VniXkObf2iC/PfEQpOKVDimRr3f8ek+46MqF+j+HNwreF1ZMaaVS/+0GeWX+Ho"
    "GbEAqy+99mbudC82xqTdb+INGZNTvcV3oxYlCDHG+cbkBE6LF8hsIIr6AFZkkGHgzLUOf/5b"
    "Fxy1W4tl5Q9tn8bP/fY3MLZsA6yYVslXyfyVft6VqeGU9o0uIbwumHbtAiT6fs0SWoELVsHa"
    "JlQS1YBdwOLeezbWy16860gA4RFZgPFLPmGWbfmNcypDL+ZQxBk12ws/Z/EOJnxmL3xSWu2F"
    "7/+eMXlXc/qZU9m1ieEgq+f42FS6R7Jo0/pl+NI1L8O6sV0g6fnQNFkAlJEL53WIr6EEahlk"
    "GEwmAV0mgqk4fe60hokaD38f4X8FrGOMySypGbOd5c//uyONBo5otea/+lo7uvK8j8GMWG/+"
    "Qp08KeEP5f5RCD+7hJD35xIYxoUh7TLCz8b41xj1ukweHburCq7hL959Oc5evQfkeslKGTao"
    "lPsxwWzHswVJMU1+DfRrKaesM2PKRZ4kAWcqNxGF9SYiUxl+0eQFbzk39EI4vgowuuUqXrX1"
    "Lasq5ouIAqZXWguCEiApoVPxnTU2YBUachZyuauC4DnsPrWI6f3qnMDxuD7wzsuxcdkuGFOH"
    "XIYXcsVZIZNyBlCaLBerWkeOysvlmjAUc5rfm5/P6XMPr/yBFJiu7a7eet3T4gK6W66CWbH1"
    "o6jGreb2jd/GQ4TPitoNpptL4UMLvgX44okf7WaS2Y0ALC0mY2Tk4LfUWM+qHen14d95JaZ3"
    "3geiBmS4KGKNbqtQ8rbix82iLQRzIejIlmZQTK0N0q6ngGFZuAzAaDxhdVxB4PrLPiBmZLk3"
    "Q6YKWh/y/cGM+ZsIoM7EkM6knRGVAkNTxK1CUX8iNHP+iQpu1Qj6SBuL+3dgYfpxiDivJHYG"
    "413GWacvx/PPXoVLv28zvvd7zkxx+eHghrqx2LFrP974e/ehM7YK+hSRqDAPMUOoyR1VH2Cd"
    "zY+pLGAK+SJATBnBzApGgBiZQVtbONfYuun9da/7wjfO3nBRfVwU4Hk/dWN3985tb+9U/Hbm"
    "iriqQAjn5ow+d2dUmFPG+950m1LAPMgKJlNJ5JG+Mu+6hpAUYCMKh0LhFyaVjgXFIFiI7WFx"
    "YRa9+T24/AWTeNubLkV3tHPYruPX3v05PDyzCQ4jZVGIyyeFCmQPlQ9QeX9RqeJhyqDRfxJ+"
    "+Lu2seE5C58nmqOdN/0SHVcLsO6yD+yoRibXMo8wGwNjqiL21QctM+rPaNgkEBOZwcz6EUih"
    "Zm0aPbaIz8UcAKvCkIHj4iqNHENHH9qFzJ6tQdLH/P49uGxLg//25kuXvAbWOuzcPYMr33UX"
    "xibXA8SZwInhYQzfHMpMoHMq0ycHVgK101OtQKCQ/c4PdLGnhuGaBs710OvVP70oKz+11JqB"
    "Jdu+8Us+YSbOfcPmynTWExkmE2r3oj+L7BwBZS1AFBaS/0v+zFAh/EgMFcKP71eKYUx0I8NP"
    "/JoAynw4lj+DP7nrv6qRLnhkGSZWbcJtj63H637zn5asAMYwNq5fgUk8AYRqYA5Yxfv9HO0Y"
    "g1Z4R0W6u8AE5Ncpvi4h/sL3K1dpNNnEIKpsx8hbDqdgZMkKwFMXmPHTfvCd4FGn6/NiCTeH"
    "0KSIiSOqpTItnONbFLGzjqFL4EMDwFADpsQGxoUy+e/5KMWEcFGBNSZUxqBTdWBGpsBTz8Mv"
    "vuvmw7ICr7xkI+AWW8IpPxcN5UMO9ji30D6n+D+DSxTFMAlwExlGc9FxiQJmP3dh31D/J0BE"
    "zJVGoEogavdDnfwJC46ioIMGcgb6O7ESIkGhZaidHiwBws9GCzhaAKM4CaVgXP5OposZbMJX"
    "v7ULdeOW5Dy/b8tp6C/OgFkGsphZ+Fzu9IN+VxGOOtqWs6fI0Q/H8vlU8OhxkhnHmks/eOWF"
    "P/XZ0WOmAKNbruI1a9ZMVsxrOJDtmq1KNK3iBCJAi2Yd6gZYnQYqzR+V4V4gQTIwRBI8K2Uq"
    "2TgUWcgsZGqFmOWXMYTR0TFc/8W9KS9wKPD0/HM3wDWLWUGDa8oHXrKVSwdg0Nr5A68r10hv"
    "gOLUdEyFt3AXMTeVwRvu/OSP9o5ZTeDpm58/OrP6Pb8qxsCQTnRA+SkMmGZSVoKLQg8E5Wgx"
    "Zq3d4LFfK+4nGvr/hlUNDZSJDzkxVJwMEsF0vWZpppMZyybHIbYHEgGbUEvA8OlgJjgnYMNw"
    "1oGNPz7GhuAsg9nBWeTfjW8vQ0xgJ3AEkCD8TnBE4XcGOevb4FgbilpcUhpHbCD9S4+pC3jg"
    "s/9xgdH/cSJY6MUPFTjMqmY/VOZpQVPh61QYhxL4HIgtTF/aDyZTToqaLTODppVBLHAA68yk"
    "foyx2GuG1wgOSxq5uuUKUbozZQUTzjHxM+eNkPEQ5x4JhOKUdFEtxXkjgOMpaQKRUGWqkclz"
    "X3f6UkihJWOACr0XEoWyjHZWLnH2VC4Eqbx+2t1I31MOQOUENJbQZjAujmFlUVAmikgJU2OI"
    "DMyy0E36ClGF5uuXcDnn0DRNiAoG6dqkpAMsH4qIiFXlc1Ik5T7zWobNFtdXgguQkDiFarBh"
    "uhg/7aWv33zmmSNHrQBBi5ax6XSZhLIfMoWwkXa9KtpUNz3oIjQJlBWKdTIlKVEEgQh1hlxG"
    "BxEYJgygk0TKYrAOp5CzeQi+mHzWrTu6tGr5vdNzqEZGUxq85O21e/NCh7aCAwkyjamgSC4k"
    "F5cjoLwJ49lKhMf8FnUNSfPyHZNX9o8aA7zo0h8z9618zo9KOqEb0by04tSs3cnHa8oXKFPG"
    "rCt8lfIAgzHyUCQ95CstXAaluWJIFYS2S8ZNruiZGq2LdPABLYAIHnr0SVTVSMp+OgLYt4fz"
    "ZWfhdxdxigOshCPs5BIwjD48lqgzA+IoldGJs2AmOKf5llha5xQbGteZ2TB9z/T17I5aAZ54"
    "/PFOxe5y4hFHFE7yKRCYNBJlyJJBmKgj4FC5gVZMS2Wcn92BVixtGVBEFm1gONg4KtYTAjRA"
    "gFIQG+OHLxqDdeLD1kPUC9x6+4MYn1gOYypPiMF3NBTnzxJHoTEjfReQL1iOYA/Z8jkbFAn+"
    "/zcSNpphOGuVi/BAkJgAK2Amr3TO/x1riVjm1x8TEPjgZ/7dPOC2EEQoFXdCaaEkwSfAAs0J"
    "6BM8efdl4BiEyFRagPgaKk0pVAhoGAXpVPb/U2wjZ/ZQF3LE9xkmjHYIqydqvOx7Jw8pfF+B"
    "zLjhKw9gZHQqH1zh7MOLVDikaF1XArp4bK1MciUKm1uKXfRIiucccpgZ+2my6WJq84+tPxQQ"
    "5KVpiT0rdW7TlC9aNf9BIeIO1xxBoRDQuACppRurBE+ih9VCIWYFlRswih9gTUapQpOKoV4T"
    "XBEBhgidALTOWF3j6itXLRkAPr7jKTRmHTrd8QLhD/Q5Qs5h5LpFqBY4KAivor4i9j7Q692K"
    "OEhbxBjqMgFsMLrxFS85Z9PKzlHzAAy7Pm792IZFH7/KFqGMu7l1aFMXRyYr0YoOktlHGTGg"
    "zZqpv594gAFsALA0QAzVlMlnIoyNAOdv6uCHXtjFxjUrlk6fMuOPrrsBK9dsRtUZ9cdMRLzf"
    "Zs8HOAgonu1zHgd4SyDJ8olzfg0lriuFM4de+ELR94d7E5ciACIBCYPI+r+l7tn/2wqG3dae"
    "WfP/jlYByDAq1sewU8w5qPVEGZEiuQLVviWZumwSjQJw3NLuTHmqqACDVGosEWcdRjHjvb+w"
    "Ygk7euntextrsWv3NO56lHDaWeth2IQUNGBJEHigdMjEg2XvKDkAO4EDEyABGIL9cxJchRe6"
    "U+SWzzQSaUWJP+fNV4JXsYCcPbrmIjliFxD8RxWPWOW6CVHxsijfLgU/wCyDitKiceOH1jRo"
    "DpNC8+fEJmIojx7fYzib+egalkLqMC89K14Zg9e/5Vqs2XAeOqPjAwRVWcam2ttF4apjb9l1"
    "osBRYQun1DcAVQ+hraooC6jyArkDy9rtC2f1jxwDiPCK837mHFBVoN8YsnCB+PWxbv1atHY/"
    "CkvCbWCofJyOfXOJVZle1T6QCIUvPhzBLvV6w3/5UzznnIsxuXKDr4UgZbnSmoQOZWF7FiC3"
    "8OF5U1ELG7Cq/WP1vnK9ghqIgBOQjADbQezC2kOFgocEgSNTZ0whR1FKeJRq5yGSTF10DxFJ"
    "py4/rI95I4dk1PLnbbBTWABpEUsoSrPjghjOyrRjTw05ygbtjfU5/9f/2p+iHjkHy9ecA1ON"
    "KGCLFjYpAScKQEjpLEMOU5HMed7JpctMq6+sRduClTgHIObuofHdQa4Ltmxh7q5bobojBYEj"
    "nUbyR7NjO5YMdJKiMCXgk15PbeCIViUPBo6CDU0CkVpwysARlE0wH0WVcPTDu5+awQ//9P/C"
    "yKqtWHfGVnRGx0NBClqYRbmlJDwBa4UHgIE1CL9IuS6qcUrh+zkCysRtqLVB/DtCxNX4oaq+"
    "DgkCrTPjFKgS/2lccdY/HfRMZo/LODbWBZC0yJn2bpB8tCvsdlLPFfnwwuQr06jDqSD8uUUL"
    "OUyhW+vQ6VSYn+/hDz/0j7jzYcG5W/8txqc2gEwnF5OC4Bie0GEGhzo+0p/BeYYwKqdn80Km"
    "LySCxHoQCcMg6z8vM0FsdqlkI0gM3Q+SC2FYa31Wk0LtIzkfWYA6OERT5EMqgLg+/N8RELmk"
    "fUiEpwzQq22U6t/LA6FitHRlFKGaOwzEvGiZf+3vtSLkXbTQlyUJXbehueGmO/HZG+/G7vlJ"
    "rFhzJs48fyOqkfFUAeycBKQuYAlUL5wXNGXihwMd7AXuuca4hhLcmXWSrICIS4pinbpXNf1E"
    "xA3Q1EzeHuS19xsuNDU8cgswKnvFzm9bwNjmEK9mgeoW7FkhWPXkEwAmZKtiM0ceCiYzUyBh"
    "RgAG/GoZIQRBY9AKlBhBfOLooKVdFrff9SD+/oY78d1tM6jGNmB0bBmWrbkYZ44th6lG/WzJ"
    "WOgZi0qd/3+OKez8vCtJvFCciPLxSrlVW5qcURWQH4zjw8cBK5pda0T/3iCHtafQrSR3GxFx"
    "vf5RKcBt9zwuo/MPzGLN5lxRG7VR999Fi/tXFGcSMnNSHFbNl1h1/eQBYOj/nz8apUgnfTqG"
    "tPBLd2EY2LWnD7e5e8CIwIngvgd3YQabcdaFp8NUoyBTpRZzgtwKLhrT+NkcBCwIyZ4Yq/vn"
    "bBC+f53f+blVcfThKJQ1ggC/ZoEEsuFF0uqVDFeE3nEjerwVRhxKf+HoMACR6808/BRJA5Jc"
    "vmyioAUFv482Ci1O9GIwF0CZBczvQNnjh7QFaAHHuIhoATHSDOLBj4sREawQxiZXYmxiuReC"
    "UO7tIwIh3/zFBQXwvtwnf1xccK0Y7IdRRJ4nmfBk5SQMq5CBPEHsUqTdQ7S8IeUTaIIA9qC6"
    "rkWXm/oU8CIOMRThoFFA7+6rpbfnzvloYgrtSu1TnNq1grJfr15oFImk3MxJipvXiDaSQAVH"
    "EITOJAXpUihJChk9y3ewQKAyBjt2TcOYSuURwvsp7+JETXMWdAKoierOhoJJI/fSXSU6WrtE"
    "aKuaCbeSqCLVT1E1Pk8tbKA6mzm4Zv6ps17xwZGj4QEEAItd9NPwVK+cQmtjT53k05Hz8FHD"
    "B3oxSrnDKTZ8Lv1msbAF1xAjBSkJFfU7iWDfrD0kITQ60gkFHeI5BMoNpFlVLyXBslKygtCh"
    "pCxlr8JSoDp5U2RW9YMorYKefTGoGKFnEYXhVL6bCTtnn2gWp+los4HWipnxTZBsyE4p4AEU"
    "Y1bzTpaWMMv6wAM3bFf8dsEpQNGf5YJQoqpFCSCTRIe6Ht+5N2cVw99mFsWqiXItkhQ20+IJ"
    "USi/LAnrUHsNQMX6JUsY11a5DG0NBjadiOJjKL1fxIKlIdefeYxlTo5WARxss1vSoJvYFdsp"
    "RWifM6OC/i1ELVKi4YG8AhV/t/zKPj8Jpr24cceS/5wzc4fur0xsUFVGmXoJuzsTOazIG1b3"
    "W+CfgtnL91Oug7Q6mQ+6SKCcW6zJoGht27iBkBtLeSVq0Nt3//1jtMcelQIs2/Ai29j6EUgT"
    "wgtvCbIQVVgYbzoJUFOYuk+/4hAiK5LCQeUHI7chg7teU6gZTGWcQqEcC3RoHV/s2UBCZSuS"
    "hN8OxVphWAFOteIiP5ZdGFQkJQMbpIysWtyKcqvFu4Lp94vlAgC0IOlj/6Of//a3/+X/1EfF"
    "BBJ3XD2389tjk2t+COIA+AOJsUMmwIpskkKnKDJTiiaOz6eUchs4xhMxAhDLQH1e3j0yQBAB"
    "paAEhKdmO3j7B76NmaceCyeGSxLKGIPRqTNRjYwWFijemQuIPbF3mtghragHB5tx95JCcBHR"
    "Z/Od8VRStNaGiD875La1cecj9B4kEWmcrT0Rc/Ao4JAKUHGvt7D7jtuXr3suREYgEnLZweSQ"
    "Mu1+TV34vyFLpc/eK1ZwUMNl6C5IjwkBLCpUVAAiJY0yR54BKGPlqtOwYuW6Iuxk0oDTgEyV"
    "mD5BzOvrahzJoW/c5eG1GQ9EQUuR80BrCjkVINclQQsCELbR0QbqPamEK9c6GdfYftaC4OCc"
    "c7ZptgGYPeqi0D2PfXMBwMNy3o8AVTc0NrYwoj+MusF0Jl+UVWj5QUm2/SA8RR77Uu78ciFR"
    "oG7tc0VRxRWIqlwrGNO2EcBSLgyJsEpElDWRxN3nqEZKcigIOgVFaIE3F62SFOuWlUCZ9Kjm"
    "qT2tU68NSuIAzxLp6aQufLfUW5y+d3TF+XVv331HDQItgF5t6QnnnBNpAAmDDJyFwClhUwEE"
    "MeDflMUYuuuR4tm464pcg7tGUOkAABazSURBVJS7rQxJB6OTvMgxvMvpZSZBxc6HftSK3RMA"
    "lSI60eCMBu5Shbvt2BxShG+FiogMDRMVACruFQGAQ08tFwdxjZeHswBqXtjznVtIFo7eAgBw"
    "nfH18/3FfXeMdCdfGf+J2AZUdVKP3BSLpjFt3hXkJso81B1R8mMOBHPA3Z4AkERAiRbwVJFE"
    "MkYZY5yx1mLLmRU2rBqBiOCp6T7uuH8GT84vh3U2TxeRcocnP91SxMJsoy3EQYAnQx7Twi8V"
    "ZNgWGnzeu2EP+jwOaABpgGYe+x/5xzsA7D8WCgCC2z//5B03LVu+/lVwI5lscBbCxoOhQviD"
    "xoWgn8+CQWuRUiHlsImeBQJWY12EslIIKbJFAFfjTa9egclxU7CCE+MVNm+cQL9u8L6/ugfL"
    "V20CqJMZtvh5lA/3BZsyIOjBXd1ShqEVKVIIWgasghzYMyJPHInCB2rfL8g1rm5kB4AawCFP"
    "CC/FBUh//sl9+x+94V6xPYhY8T1pLLw7cMEXoTVPZ/gYllL5c0/dAYwwZAFzF2+FuPWQR8VU"
    "ijg01uKtV6zGWJeLqmRSLWWNYfzmlVuxY/t9sE0feh5AtmqlOY9+uvyfLdyjXiNDuBB9f3oN"
    "BpRKyntrj6/3fYlqb/q9a6De/N5bzejKBQDNsVAAAJgDUPf6vXvFWcn/0Ka25wI3ZCeUY9XK"
    "qYuDqi0y3JQWLdvRHtwsg68NP//kZYc+5WOY0TQWb3rNedg/sycXV7QFlVq+tj+fHipZpmuL"
    "z1UI8CACH7rlW4yry6xsxGMuTBhj6dPe73767229sDfgt2OiAItmZGpu7qkHPifSMMRCXJ2s"
    "gDiXPxTy+LRIJA7ejKiQxrWAjlNjV4dYg9Z79Hg26GkfdhFnb+wu6ZRPVRk857RVqGcegm36"
    "oY+gmvYh8S5KAWqLlC0Ecl7eSUb9h0y5oFiDYkPojSE2t5uTxgvexS8rtu7P9mef2Ae3uGcJ"
    "/3jJCtDY/szuvd/5+E2wi94N+Hl2HgeIzQ2UQwGls26IXwyEBWSIISgXwcmQ3d22iChHtmtF"
    "uGDzyJLP+MfCz3M3TaC/OJvbvYVd7ySHbVnhoJQgfjZXWoYC3UiaQqZvRk8h0/+jjTFElFsJ"
    "btcD8iB8sbC2kfm5PV/ikWUNgOml3PdSFUAA7AEgiwtz33S2lqx1DcQ2wSQ5pfFDbqwtEGkN"
    "YHJSLluawAGlOHlX5claUkzsBAkmumpqzBKvs89YjdnZ2dCXPxR1xK7e6YMM8d/K/Gsh5/Ex"
    "mi8pFelApl8UeeSLU52aQxhdr19/Z2vANTAyzztv/5OPuv7cNICFY6kAALCPq25/etuX/oqk"
    "YbFB65o6DDMKcagEd4A8AHHYTN5iRDtKEy56KEPLj+pmjIL2wIawcM5hbqHB4TYOf+iRnSBt"
    "Yl2cChIHP6Kc84NyyGR6rGU12oC2UAgRZfE0tkG5OeJ62CZwMHXoDViHvoO16/ftNhFpALd7"
    "KQDwcBVg0TWLe/Y/fsvDTb047VzjxNbBDdRBE5ti+FEiLYYAuWLHRqCI1k6J0zSlLWwpxrDq"
    "Hv2AwFrg3of2H1b3z8oY3HHvozAjYz5oDcLXwx2LgdASmMP0msyMWmUpNI7wMCFHE3q6WNvn"
    "i16zYF1jzC82+v46WOA+SISnn7z3Y9yZAIDdS/H/h6sAFsBO7kyYmace+oQ4y177ajgbNdGG"
    "kNDlGBVtQaMYtFDeONSolfauao1lVe93akaPE8/h9zGFx3fNLqkxtLUWe/fNYPdcF1VnPLdz"
    "DZ0+i90PPSdYcoSD0kMkEz8wTjZ3E21biGLcDCTgKKSxsyI2dAcN6x2iMeesWLvQ33v/39zi"
    "6rl9S/X/h6sAAmC3q+f2777no58TtyDirIhtYG0f0Rp4zYyuIO4UpwTmCt/f7rGrAZYrmjBr"
    "RUAa/hSVyemJXWHixvv+6tswhpMrOlBJuDEGv/7bH8HqdWeCq5E8zcNpKxQmfTilcE59FjUJ"
    "TFwgydQ8wLaLiz/rQVMldopr5uCi3090bw1n+3DhuzhHM3t3fIo7ExWAXUshgI5EASIfsJur"
    "8Wp2zxPXO2cpmf+m9koQtDP2uY2tVLXpFBUqZmugFkbckKFLqqduMYzRKcVxhTJMrToN73j/"
    "zeHwhCsUwTd58m3W3vwbf4ypdc/HxIoNADg0Zs5j4sSVViZaHRcFrj+TlOFpMvatAZHxXrIS"
    "QM0WzhPHXQTYroFtGjjXz9FX00DEidg52XXHNZ929dw8gB1LNf9IedvDswIQV6+b3fmNe1ed"
    "+eIriDsgPy8utX7PTSCpdaQZ6uSP6hmAYQdAcl8h6Lo5ynX17Y5/RSGFeLa+6kzib2+8D/un"
    "d2Ll8i6mJscBAE/tmcanP/sV/N6f/APWnL4Vq9adBdMZ9cx+GvGmB0OrbKHLJj/hgPCYp0Qy"
    "+E1Hz10WtFORRbkp4nzAMFzSWm89bYMmWFhre7BND872YW0N2/Rpevejn1jY+537xdW7ADyI"
    "w5gidiTDo58CMMOdyfXTTz5y7coNz3uTaxo4rsGW4RoDG3sICYPF981JppwcjHAI1/JYV4m5"
    "eGVSwX7x/eDlbHb9aRuXEkyJqNEndsJhFKEOlq/ciDu3zeFr99yPxflvoKnnwaaDsfFlOHfr"
    "y9DtTsJUnRTLpwnfKIWevgdM4IrXlFNCk6uAQGzGDlE5nHYLoaV8tEwpzxJwlAuhnm16cHXf"
    "g8CmhrhGpFlodt113SeD0LeHHACOlwWI1F4lrr967slvPbBi04tfzabqkL/CrmXVIJpyZ7Eh"
    "HTygLQN06xl9dKx9kFTV4aFdblXaq9Rnx4yiO74ME8tWY9mKDVi2YgPGJlej0+mCuEpCzaPf"
    "EaKAHKL552MLeDUFFO2BkAeeHZgtiWTXEsxJGiLhbGgP79G+tX3Yph8sgP9ZbA2RhnZvu+29"
    "vf3bd4rr7wNw39OhAAggYwNX3fF6YeYby9ec8yPgKlT9Qk36CDOE2CShJpMemhpCdQ/R7eRS"
    "eRYrQdIQU1/k6+NrqaielVRfwKEaKTep1gLP4FIKF1CYbWUJMhhUplzxA360WwCSrjT7flxc"
    "VpI8EaQJE0Ob4AJqNHUPrunD2j5c04NrajS2dnVv9vHtt197jbi+BfAAgJ2HK8gjVYAGwIi4"
    "ZlV/9vGZ8fUvOt90RtZFK8AoO4NHYcRDkcV5ONUVm4vav9wRE5wFS7oCpygUUUkYiZU5og5P"
    "ZFCmq6oHUklx1wef7QpGMoeaJSEENRlUzwcO7iCYeuuGcBp6irjzJJQNkZSzfdRNH67xgrdN"
    "RP41YOfpga+89xfEky6zAO4B0H+6FAAAFgGsBzA6/dhXvrzmzJe+1g+Q8OiMobqFRjcg5eSr"
    "DORiLyBV2t0CdOADCR3FmUW0S7YS4VJmWGNIGTN80fxH7t8pqloTPlKAv/BaV4azTpv/8JyN"
    "o2SisNU0cR/ixRExIZJq/M63toarF0OI3YdtfIQ1veeJT83tuvPusPsfBPDEkQjxaBSgDntz"
    "LXcmR+q6+dby1ZteHqd8664iFHraEut2MsgNDqCqt3XPHMrdRNpVwQMl4qAD1gnEg60JbUfg"
    "KFkJCvpZJ4KcQu5OlIUofb9H/zKIAxQ41NGBbXLmNI2Isd7s26aGdX00TR+2WYSro/mv0dhG"
    "bL2w95Fb/vB3xdUOwAyAuw7X9x8LBUBIOKwR1x/tTT+0e3ztC84wne7p3r23BzqF8+2hqDFW"
    "/kQFEUFRLo52vwHk4s32QRSiYZU4wxJOOU4nOUAmNwpWctWRa/v4gnyS8vnEDiJM/pYiKeTS"
    "rg/KE4prnLMJ8Into657sLYHaXy455oazjUC16fv/PO7r2RTQcQ1AL4dqF88EwpgAx5YB8BM"
    "b//aV1ad8ZIfAXGH1AkmHbfnVnIUTtuYfMKGsqVI5d7U6n2jhU9QpWXq5GwLFA6k1Fs+XQtW"
    "s5GFj3fZakRBO9tWAD0trPxdh4EukTwW1lpYG0geH9ejrvsh5OslxO/BYZ92PPT13+3N7dzl"
    "mgUL4MmgAPaZUgAAmAcwAWA5m8rsfuiLf7d286WvBRmh2M8ErdM8UM2iotlWuECcpF5/onvh"
    "FEpQnhSSXMg3UI4mqmyryK4JfG9fcUPCQJfBXgCEgpwckjgNrACHWfgJEKbfXZoYBmfTyLdI"
    "nzvrfb43+z3Y4P/F1mFYtMPs9K4v77z7r/+v2J4LGOzOpRR+Hm8FcIEiXiviRrgz2Vmc3f3l"
    "5eue++p8nFkS7yxFuZTy+6kW3gvfcyySGk+mZs+6mHRA4IM7f6DwAqTSyRHYUem/VVJKZyOL"
    "7J/CDZEX8KEd1Ii37AbyCLgmAD4LW/udbZvGC18pgLN9SOOFb20jdW/+8Ye++t7fVmv+XQDb"
    "Dof2PV4KgBB+WADrxPWl6e9fcDzx2OTU2ksSZAuNCygVBvsmCihmDshAJ5BUEEytIvEABkWG"
    "CV8f2lBVO0XuvnQFqjIhA7yCAJKB0M8p7r7NBsbdnodH+nl/kd61toF1tQd8zSKapp93fwr3"
    "GljXOGd7i/f/8+/+vLrR3QDuPlLgdzwUICaKugBWiKtlfs/926rJM+3YspUXlrVc+fgUVMMH"
    "CSAvH9TJpd7eLUD1R6AUeye8AMmuIMleBgpSipSrIn8KXn6AHCpJnggMfViXw0M/5bMcCmnj"
    "oEhrIWHQo20ip99H0/TQRJ9vo/AjA9g42J779hd/53VsKhZvuuYB3BHQP04kBYghyUoA44Bg"
    "/87b7xld8dzJ0bGp80hUE7zQDSuZ51hz7xz06aGiPp/1Bg/PMRV8gI4kfNZNh4DtegQMULSD"
    "wBDZl7fAYRJ89PfWqaRRnOhpIdYlYsc2Hs03TQ1b+xDPC7+PJoR5MbNqbeNEGrrnC++6gquu"
    "cbbvAuC+53Azfk+XAkSGcBbAGsAP1Z154hu3ja547mR3fOo8tMqdcrMJFxosyYEOEPgyAhpS"
    "RNGuNCeoquRoVXQHLRmC9NESrK4/yCnhIk/gXArpnEog+bGuziuE9ZSuuCZz+WH3e5/fh7Pe"
    "CniePwJC68TVuOcL7/oprsYq1yzasMEeDr7fHiuBHWsFiAxhLyhBFZWgM3mmGxtfdqGQattW"
    "HDBtl4lHNyHqNEx4m26XFv6MFCXVVFYZxVo7UuBPl2shm/HEBaiwL+X/ldBL5J9n/SZCx/nw"
    "TmwDW3s6N+32EOPbJqZ0Q5hnLZxtnHNNfc8X3vU6NqPs7GIsq9oR/H7/WArreCgAghUQAKvi"
    "/9i/6857pVqxfXJq7SUgFoFQMs3BJYg65uxlaQcqcQUu6ISkGjuJilCUjZfJl0gsxARNFrK0"
    "BjujjNlFo/9chOIkh3XOumDim7CDGy/8JmTvrAd4zgbhN4HSjTveNWFUvJWm7k3fe+Pv/Cyb"
    "Djvbt6oi+47g/3EyKICEujQOmIABweLMtsfnZ568afm6c18tCP1MU11gtABB+LGeUCQde3aS"
    "j30VpebJNcT8vFUnkaRVtiUFO5cKSmOJl4uj26HKuyQJ3IkNShRy9U3c7b5kKxRp+B0fBG2b"
    "Pmzd874/pXO9kjhpINbfz/zsvnvv//IfvDWUhEWtnwFw27ECfe2LcHyvDoDnATg7Kht3Jkdc"
    "Pdtc8PL/fp2pRieZiZgNiA049uphA+YOjKlgqk6Y+O1/Z9NJY+rJVKFps/HDGdO4+DxONrOG"
    "rc4kxWnc2PKOWqGiK2r0neiYPiiaNv9WWQDXwDUhl29t8O82uYf83kYgDe3e+fDHtt/+F59q"
    "rd8MgNtDEY6cjAowXAlMxc427qxL3vq2qeUrfgDUESYm31K+AhsOvfg5tGWv/Oh3rmC48spS"
    "VTBsQCYqjEnTTE0cboVcmwDkKSbDblxaOYRY5Jlq9mI9oo2WyMK6EN6JN/8S/X6q32u88F1W"
    "jIgTQk2AiO3bB7/16bcu7H1gZwB7T5vwny4FiErwXADnhJ/TtWrzy7dsOu+yq4WNMBv2u9ek"
    "2kI2lZ8AziYoh1cA/1gFMsYrQlAAY6pUlQQ2adhk3P68hFt2yvXkcE8VpVqVwIkVwEH4cfdH"
    "hfB0rwtWI77eCcTS7OzsXd/9l/dcxdWYCdx+1MJ9wefvOZ7CfzoVINYfngXgPABptDl3Jjuu"
    "nq3Pfcnbrlq2bNlWUCUAiI1RE8BNGE3rFcGwCYoQXAH71xpThZSzCSPjTDD/JlkD6MRTq0eB"
    "Po6lD6qk6l8bwZ8NWCHW60VlaEIUYFWWL+CXRAWLiO3J9gdu+R9PPfj5O4cYoieD8Pcfb+E/"
    "3QoQ0wEbAVwAYLL95NTGizZvuuBVV1dVd5KZxXsFNUw5jKgbphRxmCWS9TAJD5BqBQ9oizCM"
    "RXYJ7effnUoCZb/vD2UGijd06MgUsGv/LISG9u3b++WHvvZH74+K38qsbg9Ez/zTJZCnWwHi"
    "/1wFYAuA1fozcNU1rlm0G7e+4cfWnXbWlcIjZHyzPiJwGKacBy6xMWGWoAk7PygEcRq7muf1"
    "qj670BNFW4APyN03QkVvPgE0KFjfSdxmfiBxBH7wg2ucMFuanZ37zv03veedZEal5etjLuWB"
    "8NV/uoWBZ0gJxgCcD2ATWuXpcXec8aKf/0+rVp/2k+ARYoKEybVqiDKrGYac5/Mw+wJQ1nWJ"
    "ud6sIBwVjyyx2FDtfkAXcCKd+HHp8Erk/l1R8uWck4oszc4uPPjIHdf/wcK+h3aHXuLt7NV+"
    "APcGosc+E4J4Jq8KwHMCQFzW/jxcjVeumW9Of8HP/sSqtWf8+87IyDIrJH4IRBhkyXpsqq5I"
    "ViPWYpMIOfAksSIgbJ0AgmYKxQbmsTyNBAmdE53AoE/7ZuZv23bbde/tzT0170md2g2hzbcD"
    "uP/p8vcnogLEz7AsKMHGdpTgLcJEx9Vz9cpNl5y99uyXvmFqYvxCoY5vH+mFTcVJIsQ5eyh8"
    "PpV1ZgfMO4gCghEn5hq/gUOq4kRg0FC/15/eN/3U5x/9+gc/DjLETORs44bo2v4g+MePRUr3"
    "ZFcAbQ02hFAxsIctRVA76Tlbf+ZlU6s3vWpivHsuyMCJb+9tmEk3i0wDI2I2UrcPltYqiK4v"
    "idkCKs4xAiIhFUwGNXp9u3///v23PnH39R9fmNm+hzsTlasP2KG6B+Ax+Cre2Wdq15+oChA/"
    "TxfA6QA2h0hh6Gf0WbKFBgDWnvOqC1dsvPDysfGpLd2OW0s8AocKIs4RRYUQX6larHn7KJEG"
    "hSLJFfjG00RoUNd1f7HXPLR/+slbH/vmNZ8RkYar8QpSuyFmPl41/KndB0Nsb0+kBT8RL4Kv"
    "M9wUlGECBznJzNWoAXXI1bMNACzfeNEZy097wdbxqXXnd0ZGN41U1SrDzURVjariNG2RlSKE"
    "HETdiHVO9vX6vR0L87MPz+977O4d93zyG8F3gzuTHbhFO8TEtwW/O6Rxdz/T5v5kUgD9+cYB"
    "nBYUYQpLONDKpsOgyre0LkMuqrorR8eWrV9Wja2dNKMrxomYXbPQs7298/25XXPz+x7Z1zbN"
    "PDI1AtezrqldC8UfCE8uhh2/Lez45kRe4JPhIvgCk9UBKK4JruJw+xuATcVD568GGtDZ5kCt"
    "PQ8ldBv4+x3wp3RmTyRTf7IrgL5McAlr4I+mLT9SZTjKS5AroHaHHb8vEDmCk2hnnawXBWXo"
    "AlgR2MUVATh2kCdZHKvLhR29EMK4PQD2hp9PKqE/WxRg2L1UQSEmArcwGTBENyhFFV7HB9nV"
    "0Zw3IWxbCLt8Lgh7IQjcnaxCf7YqwIHuzYSvqvV1IAWogwLUQQlU8+BT16nr1HXqOnWdup41"
    "1/8H8uQNMsd64coAAAAASUVORK5CYII=")


class MessageIcon(namedtuple('MessageIcon', ['bitmap', 'width', 'height'])):
    """Message icon named tuple."""


class Messages(wx.Dialog):
    """Simple message dialog."""

    def __init__(self, parent, text, title=wx.EmptyString, style=INFO, bitmap=None, yes="Okay", no="Cancel"):
        """Init Messages object."""

        m = HAS_CAPTION.match(text)
        if m is not None:
            caption = m.group(1)
            msg = m.group(2)
        else:
            caption = None
            msg = text

        self.answer = wx.ID_CANCEL if style == PROMPT else wx.ID_OK

        wx.Dialog.__init__(
            self, parent, id=wx.ID_ANY, title=title, pos=wx.DefaultPosition,
            size=wx.DefaultSize, style=wx.DEFAULT_DIALOG_STYLE
        )

        self.SetSizeHints(wx.DefaultSize, wx.DefaultSize)
        b_dialog_sizer = wx.BoxSizer(wx.VERTICAL)

        self.m_message_panel = wx.Panel(self, wx.ID_ANY, wx.DefaultPosition, wx.DefaultSize, wx.TAB_TRAVERSAL)
        fg_panel_sizer = wx.FlexGridSizer(2, 1, 0, 0)
        fg_panel_sizer.AddGrowableCol(0)
        fg_panel_sizer.AddGrowableRow(0)
        fg_panel_sizer.SetFlexibleDirection(wx.BOTH)
        fg_panel_sizer.SetNonFlexibleGrowMode(wx.FLEX_GROWMODE_SPECIFIED)

        self._setup_message(fg_panel_sizer, msg, caption, style, bitmap)

        fg_button_row_sizer = wx.FlexGridSizer(0, 3, 0, 0)
        fg_button_row_sizer.AddGrowableCol(0)
        fg_button_row_sizer.AddGrowableCol(2)
        fg_button_row_sizer.SetFlexibleDirection(wx.BOTH)
        fg_button_row_sizer.SetNonFlexibleGrowMode(wx.FLEX_GROWMODE_SPECIFIED)
        fg_button_row_sizer.Add((0, 0), 1, wx.EXPAND, 5)

        self._setup_buttons(fg_button_row_sizer, style, yes, no)

        fg_button_row_sizer.Add((0, 0), 1, wx.EXPAND, 5)
        fg_panel_sizer.Add(fg_button_row_sizer, 1, wx.EXPAND, 5)

        self.m_message_panel.SetSizer(fg_panel_sizer)
        self.m_message_panel.Layout()
        fg_panel_sizer.Fit(self.m_message_panel)
        b_dialog_sizer.Add(self.m_message_panel, 1, wx.EXPAND | wx.ALL, 5)

        self.SetSizer(b_dialog_sizer)
        self.Layout()
        b_dialog_sizer.Fit(self)

        self.Centre(wx.BOTH)
        self.Bind(wx.EVT_CHAR_HOOK, self.on_char_hook)
        self.m_accept_button.SetFocus()

    def _setup_buttons(self, fg_button_row_sizer, style, yes, no):
        """Setup the appropriate buttons for the dialog."""

        fg_button_sizer = wx.FlexGridSizer(0, 2, 0, 0) if style == PROMPT else wx.FlexGridSizer(0, 1, 0, 0)
        fg_button_sizer.SetFlexibleDirection(wx.BOTH)
        fg_button_sizer.SetNonFlexibleGrowMode(wx.FLEX_GROWMODE_SPECIFIED)

        if style == PROMPT:
            self.m_cancel_button = wx.Button(self.m_message_panel, wx.ID_ANY, no, wx.DefaultPosition, wx.DefaultSize, 0)
            fg_button_sizer.Add(self.m_cancel_button, 0, wx.ALL, 5)
            self.m_cancel_button.Bind(wx.EVT_BUTTON, self.on_cancel)

        self.m_accept_button = wx.Button(self.m_message_panel, wx.ID_ANY, yes, wx.DefaultPosition, wx.DefaultSize, 0)
        fg_button_sizer.Add(self.m_accept_button, 0, wx.ALL, 5)
        self.m_accept_button.Bind(wx.EVT_BUTTON, self.on_accept)

        fg_button_row_sizer.Add(fg_button_sizer, 1, wx.EXPAND, 5)

    def _setup_message(self, fg_panel_sizer, msg, caption, style, bitmap):
        """Setup the message."""

        bm = self._get_bitmap(style, bitmap)

        if bm is not None:
            fg_message_sizer = wx.FlexGridSizer(1, 2, 0, 0)
            fg_message_sizer.AddGrowableCol(1)
        else:
            fg_message_sizer = wx.FlexGridSizer(1, 1, 0, 0)
            fg_message_sizer.AddGrowableCol(0)
        fg_message_sizer.AddGrowableRow(0)
        fg_message_sizer.SetFlexibleDirection(wx.BOTH)
        fg_message_sizer.SetNonFlexibleGrowMode(wx.FLEX_GROWMODE_SPECIFIED)

        if bm is not None:
            self.m_bitmap = wx.StaticBitmap(
                self.m_message_panel, wx.ID_ANY, bm.bitmap, wx.DefaultPosition, wx.Size(bm.width, bm.height), 0
            )
            fg_message_sizer.Add(self.m_bitmap, 0, wx.ALL, 5)

        self.m_message_text = wx.StaticText(self.m_message_panel, wx.ID_ANY, msg, wx.DefaultPosition, wx.DefaultSize, 0)
        sz = self.m_message_text.GetSize()
        if sz[0] < DEFAULT_TEXT_MIN_SIZE:
            self.m_message_text.SetMinSize(wx.Size(DEFAULT_TEXT_MIN_SIZE, -1))
        elif sz[0] > DEFAULT_TEXT_MAX_SIZE:
            self.m_message_text.Wrap(DEFAULT_TEXT_MAX_SIZE)
        else:
            self.m_message_text.Wrap(-1)

        if caption is not None:
            fg_text_sizer = wx.FlexGridSizer(2, 1, 0, 0)
            fg_text_sizer.AddGrowableCol(0)
            fg_text_sizer.AddGrowableRow(1)
            fg_text_sizer.SetFlexibleDirection(wx.BOTH)
            fg_text_sizer.SetNonFlexibleGrowMode(wx.FLEX_GROWMODE_SPECIFIED)
            self.m_caption_text = wx.StaticText(
                self.m_message_panel, wx.ID_ANY, caption, wx.DefaultPosition, wx.DefaultSize, 0
            )
            font = self.m_caption_text.GetFont()
            font.SetWeight(wx.FONTWEIGHT_BOLD)
            self.m_caption_text.SetFont(font)

            caption_sz = self.m_caption_text.GetSize()
            if caption_sz[0] < DEFAULT_TEXT_MIN_SIZE:
                self.m_caption_text.SetMinSize(wx.Size(DEFAULT_TEXT_MIN_SIZE, -1))
            elif caption_sz[0] > DEFAULT_TEXT_MAX_SIZE:
                self.m_caption_text.Wrap(DEFAULT_TEXT_MAX_SIZE)
            else:
                self.m_caption_text.Wrap(-1)

            fg_text_sizer.Add(self.m_caption_text, 1, wx.ALL | wx.EXPAND, 5)
            fg_text_sizer.Add(self.m_message_text, 1, wx.ALL | wx.EXPAND, 5)
            fg_message_sizer.Add(fg_text_sizer, 1, wx.EXPAND, 5)
        else:
            fg_message_sizer.Add(self.m_message_text, 1, wx.ALL | wx.EXPAND, 5)

        fg_panel_sizer.Add(fg_message_sizer, 1, wx.EXPAND, 5)

    def _get_bitmap(self, style, bitmap):
        """Get the appropriate icon."""

        icon = None

        if bitmap is not None:
            icon = bitmap
        else:
            if style == ERROR:
                icn = error_icon
            elif style == WARN:
                icn = warn_icon
            elif style == PROMPT:
                icn = prompt_icon
            elif style == INFO:
                icn = info_icon
            if icn is not None:
                # We use a large icon size but half it so it will
                # work on retina.
                if sys.platform == "darwin":
                    bm = icn.GetBitmap()
                    bm.SetHeight(DEFAULT_ICON_SIZE)
                    bm.SetWidth(DEFAULT_ICON_SIZE)
                    icon = MessageIcon(bm, DEFAULT_ICON_SIZE, DEFAULT_ICON_SIZE)
                else:
                    scaled = icn.GetImage().Rescale(DEFAULT_ICON_SIZE, DEFAULT_ICON_SIZE)
                    icon = MessageIcon(
                        scaled.ConvertToBitmap(),
                        DEFAULT_ICON_SIZE,
                        DEFAULT_ICON_SIZE
                    )

        return icon

    def __del__(self):
        """Delete."""

    def on_char_hook(self, event):
        """Cancel on escape."""

        if event.GetKeyCode() == wx.WXK_ESCAPE:
            self.answer = wx.ID_CANCEL
            self.EndModal(self.answer)

    def on_cancel(self, event):
        """Set cancel status."""

        self.answer = wx.ID_CANCEL
        self.EndModal(self.answer)

    def on_accept(self, event):
        """Set OK status."""

        self.answer = wx.ID_OK
        self.EndModal(self.answer)


def filepickermsg(msg, default_path="", wildcard="*", save=False):
    """File picker."""

    select = None
    style = wx.FD_OPEN | wx.FD_FILE_MUST_EXIST if not save else wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT
    dlg = wx.FileDialog(
        None, msg,
        default_path, wildcard=wildcard,
        style=style
    )
    if dlg.ShowModal() == wx.ID_OK:
        select = dlg.GetPath()
    dlg.Destroy()
    return select


def dirpickermsg(msg, default_path=""):
    """Directory picker."""

    select = None
    style = wx.DD_DEFAULT_STYLE | wx.DD_DIR_MUST_EXIST
    dlg = wx.DirDialog(
        None, msg,
        default_path,
        style=style
    )
    if dlg.ShowModal() == wx.ID_OK:
        select = dlg.GetPath()
    dlg.Destroy()
    return select


def promptmsg(question, caption='PROMPT', bitmap=None, yes="Okay", no="Cancel"):
    """Prompt with "yes" "no" type object."""

    dlg = Messages(None, question, caption, style=PROMPT, yes=yes, no=no, bitmap=bitmap)
    result = dlg.ShowModal() == wx.ID_OK
    dlg.Destroy()
    return result


def infomsg(msg, title="INFO", bitmap=None):
    """Info message."""

    dlg = Messages(None, msg, title, style=INFO, bitmap=bitmap)
    dlg.ShowModal()
    dlg.Destroy()


def errormsg(msg, title="ERROR", bitmap=None):
    """Error message."""

    dlg = Messages(None, msg, title, style=ERROR, bitmap=bitmap)
    dlg.ShowModal()
    dlg.Destroy()


def warnmsg(msg, title="WARNING", bitmap=None):
    """Warning message."""

    dlg = Messages(None, msg, title, style=WARN, bitmap=bitmap)
    dlg.ShowModal()
    dlg.Destroy()
