# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pantam', 'pantam.scripts', 'pantam.scripts.utils', 'pantam.services']

package_data = \
{'': ['*']}

install_requires = \
['PyInquirer>=1.0.3,<2.0.0',
 'colored>=1.4.2,<2.0.0',
 'shellingham>=1.3.2,<2.0.0',
 'starlette>=0.13.6,<0.14.0',
 'typer>=0.3.1,<0.4.0',
 'uvicorn>=0.11.6,<0.12.0']

entry_points = \
{'console_scripts': ['pantam = pantam.scripts.cli:run']}

setup_kwargs = {
    'name': 'pantam',
    'version': '0.0.1a1',
    'description': 'The microframework for microservices',
    'long_description': '# pantam\n\n<img src="https://raw.githubusercontent.com/flmnt/pantam/master/logo/logo.png" alt="Peter the Pantam Python" width="300">\n\nPantam is an extensible, ultra lightweight, Python framework for creating RESTful microservices.\n\nFeatures include:\n\n- built on top of [Starlette](https://www.starlette.io/)\n- simple REST routing\n- [segregated business logic](https://en.wikipedia.org/wiki/Separation_of_concerns)\n- [convention over configuration](https://en.wikipedia.org/wiki/Convention_over_configuration) paradigm\n- expressive logging (decent error reporting!)\n- live reloading in developer mode\n\nNeed to build a Javascript microservice? [Check out Bantam](https://github.com/flmnt/bantam)\n\n## Getting started\n\nOur goal with Pantam is reduce the work bootstrapping microservices.\n\nWith Pantam you can create a basic REST API in 5 minutes or less.\n\n### Examples\n\n- [Domain segregated example](https://github.com/flmnt/pantam/tree/master/example/domain)\n- [Event based example](https://github.com/flmnt/pantam/tree/master/example/event)\n\n### Installation\n\nWe recommend using [Poetry](https://python-poetry.org/) for dependency and environment management (this documentation will use Poetry throughout) but if you prefer `pip` or `conda` then crack on!\n\nIf you haven\'t done so already, setup your python environment and Poetry project:\n\n```\n% pyenv install 3.8.2\n% pyenv local 3.8.2\n% poetry init\n```\n\n_NB: if you specify pantam as a package during the setup process, run `poetry install` and skip the next step._\n\nNow install the Pantam package:\n\n```\n% poetry add pantam\n```\n\nOnce you have installed Pantam you can initialise your app.\n\n```\n% poetry shell\n% pantam init\n```\n\nFollow the CLI instructions and then start building your microservice!\n\n### Setup\n\nPantam expects the following folder structure:\n\n```\n| index.py       // can have any name, it\'s where you run your app\n| actions        // where your domain logic sits\n|  |  index.py   // primary logic lives here (might be all you need)\n|  |  other.py   // add as many other "domains" as you like (optional)\n```\n\nIn the root level `index.py` file add the following to run Pantam:\n\n```\nfrom pantam import Pantam\n\npantam = Pantam()\n\napp = pantam.build()\n```\n\nIn the `actions` folder create the following files.\n\n`actions/index.py`\n\n```\nclass Index:\n  def fetch_all(self):\n    pass\n\n  def fetch_single(self, uid):\n    pass\n\n  def create(self, data):\n    pass\n\n  def update(self, uid, data):\n    pass\n\n  def delete(self, uid):\n    pass\n```\n\n`actions/other.py`\n\n```\nclass Other:\n  def fetch_single(self, uid):\n    pass\n\n  def create(self, data):\n    pass\n```\n\nThe setup above will make the following routes available:\n\n```\nGET      /            // Index.fetch_all()\nGET      /:id         // Index.fetch_single()\nPOST     /            // Index.create()\nPATCH    /:id         // Index.update()\nDELETE   /:id         // Index.delete()\n\nGET      /other/:id   // Other.fetch_single()\nPOST     /other       // Other.create()\n```\n\nAnd that\'s you ready to go!\n\n### Development\n\nStart the development server with:\n\n```\n% pantam serve --dev\n```\n\nYour application will be served on http://localhost:5000\n\nIn development mode, when you make changes to files the application will update itself.\n\n### Production\n\nTo serve your microservice in production use:\n\n```\n% pantam serve\n```\n\nYour application is served at http://your-host:5000\n\nYou can change the port number via the configuration options.\n\n## .pantamrc.json\n\nAfter running `pantam init` you will have a `.pantamrc.json` file in your directory with some CLI config options like this:\n\n```\n{\n  "actions_folder": "actions"\n  "entrypoint": "example.py"\n  "dev_port": 5000\n  "port": 5000\n}\n```\n\nThe `.pantamrc.json` file provides configuration options for the CLI. You only need to change it if you want to serve the application on different ports, change your main file (entrypoint) or rename your actions folder.\n\n## Add New Routes\n\nTo add a new action (resource) you can either create a new file in the actions folder or use the CLI to make the file for you:\n\n```\n% pantam action index.py\n```\n\nYou can add the standard methods (`fetch_all`, `fetch_single`, `create`, `update`, `delete`) to an action class which will automatically create the standard routes.\n\nIf you\'d like to create custom methods for your action class you can create custom getters like this:\n\n```\n// GET -> /custom-method/\ndef get_custom_method(self):\n  # your code here\n```\n\nAnd custom setters like this:\n\n```\n// POST -> /custom-method/\ndef set_custom_method(self, data):\n  print(data)\n  # your code here\n```\n\nPantam will ignore methods that are not "standard" methods or do not start with `get` or `set`. However if you want to _ensure_ that your method will be ignored you can prefix the method with a double underscore, like this:\n\n```\ndef __my_hidden_method(self):\n  // do something secret\n```\n\n## Creating Responses\n\nTo create a response, make use of the [Starlette response API](https://www.starlette.io/responses/), you can import all responses from starlette or import common responses from Pantam directly, including: `JSONResponse`, `HTMLResponse`, `PlainTextResponse`, `FileResponse`, `RedirectResponse`.\n\nHere are a few examples:\n\n```\nfrom pantam import PlainTextResponse\n\nclass YourClass:\n  def fetch_all():\n    return PlainTextResponse("This is fetch all!")\n```\n\n```\nfrom pantam import JSONResponse\n\nclass YourClass:\n  def fetch_all():\n    return JSONResponse([{ "content": "example" }])\n```\n\nChanging status code is also simple.\n\n```\ndef fetch_all(self):\n  return PlainTextResponse("This is fetch all!", status_code=404)\n```\n\nAdjusting headers can also be achieved.\n\n```\ndef fetch_all(self):\n  headers = {\n    "Cache-Control": "no-cache"\n  }\n  return PlainTextResponse("This is fetch all!", headers=headers)\n```\n\n## Debugging\n\nIf you\'re struggling to debug and issue and unsure what routes Pantam has created for you, set the `debug` option to True.\n\n```\nfrom pantam import Pantam\n\npantam = Pantam(debug=True)\n\napp = pantam.build()\n```\n\nAlso check trailing slashes in your urls, these are important.\n\nIn the example below the url `test/1` and `test/custom-method` both trigger `fetch_single()` but the url `test/custom-method/` (with the trailing slash) triggers `get_custom_method()`.\n\n```\n// actions/test.py\n\n// GET -> test/custom-method\n// GET -> test/:id\nfetch_single()\n\n// GET -> test/custom-method/\nget_custom_method()\n```\n\n## Contribution\n\nWe welcome feedback, suggestions and contributions.\n\nIf you have an idea you want to discuss please [open an issue](https://github.com/flmnt/pantam/issues/new).\n\n## Licenses\n\nFree for personal and commerical use under the [MIT License](https://github.com/flmnt/pantam/blob/master/LICENSE.md)\n',
    'author': 'Matt Davies',
    'author_email': 'matt@filament.so',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
