# -*- coding: utf-8 -*-
# mypy: disable-error-code=assignment
# pylint: disable=line-too-long,consider-using-f-string
"""Application Layer Protocol Numbers
========================================

.. module:: pcapkit.const.reg.apptype

This module contains the constant enumeration for **Application Layer Protocol Numbers**,
which is automatically generated from :class:`pcapkit.vendor.reg.apptype.AppType`.

"""
from collections import defaultdict
from typing import TYPE_CHECKING

from aenum import IntFlag, StrEnum, auto, extend_enum

from pcapkit.utilities.compat import show_flag_values

__all__ = ['AppType']

if TYPE_CHECKING:
    from typing import Any, DefaultDict, Type


class TransportProtocol(IntFlag):
    """Transport layer protocol."""

    undefined = 0

    #: Transmission Control Protocol.
    tcp = auto()
    #: User Datagram Protocol.
    udp = auto()
    #: Stream Control Transmission Protocol.
    sctp = auto()
    #: Datagram Congestion Control Protocol.
    dccp = auto()

    @staticmethod
    def get(key: 'int | str') -> 'TransportProtocol':
        """Backport support for original codes.

        Args:
            key: Key to get enum item.

        :meta private:
        """
        if isinstance(key, int):
            return TransportProtocol(key)
        if key.lower() in TransportProtocol.__members__:
            return TransportProtocol[key.lower()]  # type: ignore[misc]
        max_val = max(TransportProtocol.__members__.values())
        return extend_enum(TransportProtocol, key.lower(), max_val * 2)


class AppType(StrEnum):
    """[AppType] Application Layer Protocol Numbers"""

    if TYPE_CHECKING:
        #: Service name.
        svc: 'str'
        #: Port number.
        port: 'int'
        #: Transport protocol.
        proto: 'TransportProtocol'

    #: Mapping of members based on transport protocol.
    __members_proto__: 'DefaultDict[TransportProtocol, dict[int, AppType]]' = defaultdict(dict)

    def __new__(cls, value: 'int', name: 'str' = '<null>',
                proto: 'TransportProtocol' = TransportProtocol.undefined) -> 'Type[AppType]':
        temp = '%s [%d - %s]' % (name, value, proto.name)

        obj = str.__new__(cls, temp)
        obj._value_ = temp

        obj.svc = name
        obj.port = value
        obj.proto = proto

        for namespace in show_flag_values(proto):
            cls.__members_proto__[TransportProtocol(namespace)][value] = obj
        if proto is TransportProtocol.undefined:
            cls.__members_proto__[proto][value] = obj

        return obj

    def __repr__(self) -> 'str':
        return "<%s.%s: %d [%s]>" % (self.__class__.__name__, self.svc, self.port, self.proto.name)

    def __str__(self) -> 'str':
        return '%s [%d - %s]' % (self.svc, self.port, self.proto.name)

    def __int__(self) -> 'int':
        return self.port

    def __lt__(self, other: 'AppType') -> 'bool':
        return self.port < other

    def __gt__(self, other: 'AppType') -> 'bool':
        return self.port > other

    def __le__(self, other: 'AppType') -> 'bool':
        return self.port <= other

    def __ge__(self, other: 'AppType') -> 'bool':
        return self.port >= other

    def __eq__(self, other: 'Any') -> 'bool':
        return self.port == other

    def __ne__(self, other: 'Any') -> 'bool':
        return self.port != other

    def __hash__(self) -> 'int':
        return hash(self.port)

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_0: 'AppType' = 0, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCP Port Service Multiplexer
    #: - [UDP] TCP Port Service Multiplexer
    tcpmux: 'AppType' = 1, 'tcpmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Management Utility
    #: - [UDP] Management Utility
    compressnet_2: 'AppType' = 2, 'compressnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compression Process
    #: - [UDP] Compression Process
    compressnet_3: 'AppType' = 3, 'compressnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_4: 'AppType' = 4, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Job Entry
    #: - [UDP] Remote Job Entry
    rje: 'AppType' = 5, 'rje', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_6: 'AppType' = 6, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Echo
    #: - [UDP] Echo
    echo: 'AppType' = 7, 'echo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_8: 'AppType' = 8, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Discard
    #: - [UDP] Discard
    #: - [SCTP] Discard [:rfc:`9260`]
    #: - [DCCP] Discard [:rfc:`4340`]
    discard: 'AppType' = 9, 'discard', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp') | TransportProtocol.get('dccp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_10: 'AppType' = 10, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active Users
    #: - [UDP] Active Users
    systat: 'AppType' = 11, 'systat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_12: 'AppType' = 12, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Daytime [:rfc:`867`]
    #: - [UDP] Daytime [:rfc:`867`]
    daytime: 'AppType' = 13, 'daytime', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_14: 'AppType' = 14, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Unassigned [was netstat]
    unassigned_was_netstat: 'AppType' = 15, 'unassigned_was_netstat', TransportProtocol.get('tcp')

    #: [UDP] Unassigned
    unassigned_15: 'AppType' = 15, 'unassigned', TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_16: 'AppType' = 16, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quote of the Day
    #: - [UDP] Quote of the Day
    qotd: 'AppType' = 17, 'qotd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Character Generator
    #: - [UDP] Character Generator
    chargen: 'AppType' = 19, 'chargen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] File Transfer [Default Data]
    #: - [UDP] File Transfer [Default Data]
    #: - [SCTP] FTP [:rfc:`9260`]
    ftp_data: 'AppType' = 20, 'ftp-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] File Transfer Protocol [Control] [:rfc:`959`]
    #: - [UDP] File Transfer Protocol [Control] [:rfc:`959`]
    #: - [SCTP] FTP [:rfc:`9260`]
    ftp: 'AppType' = 21, 'ftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Telnet [:rfc:`854`]
    #: - [UDP] Telnet [:rfc:`854`]
    telnet: 'AppType' = 23, 'telnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private mail system
    #: - [UDP] any private mail system
    any_private_mail_system: 'AppType' = 24, 'any_private_mail_system', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Mail Transfer [:rfc:`5321`]
    #: - [UDP] Simple Mail Transfer [:rfc:`5321`]
    smtp: 'AppType' = 25, 'smtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_26: 'AppType' = 26, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSW User System FE
    #: - [UDP] NSW User System FE
    nsw_fe: 'AppType' = 27, 'nsw-fe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_28: 'AppType' = 28, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSG ICP
    #: - [UDP] MSG ICP
    msg_icp: 'AppType' = 29, 'msg-icp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_30: 'AppType' = 30, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSG Authentication
    #: - [UDP] MSG Authentication
    msg_auth: 'AppType' = 31, 'msg-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_32: 'AppType' = 32, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Display Support Protocol
    #: - [UDP] Display Support Protocol
    dsp: 'AppType' = 33, 'dsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_34: 'AppType' = 34, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private printer server
    #: - [UDP] any private printer server
    any_private_printer_server: 'AppType' = 35, 'any_private_printer_server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_36: 'AppType' = 36, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Time
    #: - [UDP] Time
    time: 'AppType' = 37, 'time', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Resource Location Protocol
    #: - [UDP] Resource Location Protocol
    rlp: 'AppType' = 39, 'rlp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_40: 'AppType' = 40, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Graphics
    #: - [UDP] Graphics
    graphics: 'AppType' = 41, 'graphics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Host Name Server
    #: - [UDP] Host Name Server
    name: 'AppType' = 42, 'name', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Host Name Server
    #: - [UDP] Host Name Server
    nameserver: 'AppType' = 42, 'nameserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Who Is
    #: - [UDP] Who Is
    nicname: 'AppType' = 43, 'nicname', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPM FLAGS Protocol
    #: - [UDP] MPM FLAGS Protocol
    mpm_flags: 'AppType' = 44, 'mpm-flags', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Processing Module [recv]
    #: - [UDP] Message Processing Module [recv]
    mpm: 'AppType' = 45, 'mpm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPM [default send]
    #: - [UDP] MPM [default send]
    mpm_snd: 'AppType' = 46, 'mpm-snd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_47: 'AppType' = 47, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Digital Audit Daemon
    #: - [UDP] Digital Audit Daemon
    auditd: 'AppType' = 48, 'auditd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Login Host Protocol (TACACS)
    #: - [UDP] Login Host Protocol (TACACS)
    tacacs: 'AppType' = 49, 'tacacs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Mail Checking Protocol
    #: - [UDP] Remote Mail Checking Protocol
    re_mail_ck: 'AppType' = 50, 're-mail-ck', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Reserved
    reserved_51: 'AppType' = 51, 'reserved', TransportProtocol.get('undefined')

    #: - [TCP] XNS Time Protocol
    #: - [UDP] XNS Time Protocol
    xns_time: 'AppType' = 52, 'xns-time', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Domain Name Server
    #: - [UDP] Domain Name Server
    domain: 'AppType' = 53, 'domain', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XNS Clearinghouse
    #: - [UDP] XNS Clearinghouse
    xns_ch: 'AppType' = 54, 'xns-ch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISI Graphics Language
    #: - [UDP] ISI Graphics Language
    isi_gl: 'AppType' = 55, 'isi-gl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XNS Authentication
    #: - [UDP] XNS Authentication
    xns_auth: 'AppType' = 56, 'xns-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private terminal access
    #: - [UDP] any private terminal access
    any_private_terminal_access: 'AppType' = 57, 'any_private_terminal_access', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XNS Mail
    #: - [UDP] XNS Mail
    xns_mail: 'AppType' = 58, 'xns-mail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private file service
    #: - [UDP] any private file service
    any_private_file_service: 'AppType' = 59, 'any_private_file_service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_60: 'AppType' = 60, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_61: 'AppType' = 61, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACA Services
    #: - [UDP] ACA Services
    acas: 'AppType' = 62, 'acas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] whois++ IANA assigned this well-formed service name as a replacement
    #:   for "whois++".
    #: - [UDP] whois++ IANA assigned this well-formed service name as a replacement
    #:   for "whois++".
    whoispp: 'AppType' = 63, 'whoispp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] whois++
    #: - [UDP] whois++
    whois: 'AppType' = 63, 'whois++', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Communications Integrator (CI)
    #: - [UDP] Communications Integrator (CI)
    covia: 'AppType' = 64, 'covia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TACACS-Database Service
    #: - [UDP] TACACS-Database Service
    tacacs_ds: 'AppType' = 65, 'tacacs-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bootstrap Protocol Server [:rfc:`951`]
    #: - [UDP] Bootstrap Protocol Server
    bootps: 'AppType' = 67, 'bootps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bootstrap Protocol Client
    #: - [UDP] Bootstrap Protocol Client
    bootpc: 'AppType' = 68, 'bootpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trivial File Transfer
    #: - [UDP] Trivial File Transfer
    tftp: 'AppType' = 69, 'tftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gopher
    #: - [UDP] Gopher
    gopher: 'AppType' = 70, 'gopher', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Job Service
    #: - [UDP] Remote Job Service
    netrjs_1: 'AppType' = 71, 'netrjs-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Job Service
    #: - [UDP] Remote Job Service
    netrjs_2: 'AppType' = 72, 'netrjs-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Job Service
    #: - [UDP] Remote Job Service
    netrjs_3: 'AppType' = 73, 'netrjs-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Job Service
    #: - [UDP] Remote Job Service
    netrjs_4: 'AppType' = 74, 'netrjs-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private dial out service
    #: - [UDP] any private dial out service
    any_private_dial_out_service: 'AppType' = 75, 'any_private_dial_out_service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Distributed External Object Store
    #: - [UDP] Distributed External Object Store
    deos: 'AppType' = 76, 'deos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private RJE service
    #: - [UDP] any private RJE service
    any_private_rje_service: 'AppType' = 77, 'any_private_rje_service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vettcp
    #: - [UDP] vettcp
    vettcp: 'AppType' = 78, 'vettcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Finger
    #: - [UDP] Finger
    finger: 'AppType' = 79, 'finger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] World Wide Web HTTP [:rfc:`9110`]
    #: - [UDP] World Wide Web HTTP [:rfc:`9110`]
    #: - [SCTP] HTTP [:rfc:`9260`]
    http: 'AppType' = 80, 'http', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] World Wide Web HTTP [:rfc:`9110`]
    #: - [UDP] World Wide Web HTTP [:rfc:`9110`]
    www: 'AppType' = 80, 'www', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] World Wide Web HTTP
    #: - [UDP] World Wide Web HTTP
    www_http: 'AppType' = 80, 'www-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_81: 'AppType' = 81, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] XFER Utility
    #: - [UDP] XFER Utility
    xfer: 'AppType' = 82, 'xfer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Common Trace Facility
    #: - [UDP] Common Trace Facility
    ctf: 'AppType' = 84, 'ctf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIT ML Device
    #: - [UDP] MIT ML Device
    mit_ml_dev_83: 'AppType' = 83, 'mit-ml-dev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIT ML Device
    #: - [UDP] MIT ML Device
    mit_ml_dev_85: 'AppType' = 85, 'mit-ml-dev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Micro Focus Cobol
    #: - [UDP] Micro Focus Cobol
    mfcobol: 'AppType' = 86, 'mfcobol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] any private terminal link
    #: - [UDP] any private terminal link
    any_private_terminal_link: 'AppType' = 87, 'any_private_terminal_link', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kerberos [:rfc:`4120`]
    #: - [UDP] Kerberos [:rfc:`4120`]
    kerberos: 'AppType' = 88, 'kerberos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SU/MIT Telnet Gateway
    #: - [UDP] SU/MIT Telnet Gateway
    su_mit_tg: 'AppType' = 89, 'su-mit-tg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNSIX Securit Attribute Token Map
    #: - [UDP] DNSIX Securit Attribute Token Map
    dnsix: 'AppType' = 90, 'dnsix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIT Dover Spooler
    #: - [UDP] MIT Dover Spooler
    mit_dov: 'AppType' = 91, 'mit-dov', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Device Control Protocol
    #: - [UDP] Device Control Protocol
    dcp: 'AppType' = 93, 'dcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tivoli Object Dispatcher
    #: - [UDP] Tivoli Object Dispatcher
    objcall: 'AppType' = 94, 'objcall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SUPDUP
    #: - [UDP] SUPDUP
    supdup: 'AppType' = 95, 'supdup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIXIE Protocol Specification
    #: - [UDP] DIXIE Protocol Specification
    dixie: 'AppType' = 96, 'dixie', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Swift Remote Virtural File Protocol
    #: - [UDP] Swift Remote Virtural File Protocol
    swift_rvf: 'AppType' = 97, 'swift-rvf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TAC News
    #: - [UDP] TAC News
    tacnews: 'AppType' = 98, 'tacnews', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Metagram Relay
    #: - [UDP] Metagram Relay
    metagram: 'AppType' = 99, 'metagram', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_100: 'AppType' = 100, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] NIC Host Name Server
    #: - [UDP] NIC Host Name Server
    hostname: 'AppType' = 101, 'hostname', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISO-TSAP Class 0
    #: - [UDP] ISO-TSAP Class 0
    iso_tsap: 'AppType' = 102, 'iso-tsap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Genesis Point-to-Point Trans Net
    #: - [UDP] Genesis Point-to-Point Trans Net
    gppitnp: 'AppType' = 103, 'gppitnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACR-NEMA Digital Imag. & Comm. 300
    #: - [UDP] ACR-NEMA Digital Imag. & Comm. 300
    acr_nema: 'AppType' = 104, 'acr-nema', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCSO name server protocol
    #: - [UDP] CCSO name server protocol
    cso: 'AppType' = 105, 'cso', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mailbox Name Nameserver
    #: - [UDP] Mailbox Name Nameserver
    csnet_ns: 'AppType' = 105, 'csnet-ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3COM-TSMUX
    #: - [UDP] 3COM-TSMUX
    AppType_3com_tsmux: 'AppType' = 106, '3com-tsmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Telnet Service
    #: - [UDP] Remote Telnet Service
    rtelnet: 'AppType' = 107, 'rtelnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNA Gateway Access Server
    #: - [UDP] SNA Gateway Access Server
    snagas: 'AppType' = 108, 'snagas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Post Office Protocol - Version 2
    #: - [UDP] Post Office Protocol - Version 2
    pop2: 'AppType' = 109, 'pop2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Post Office Protocol - Version 3
    #: - [UDP] Post Office Protocol - Version 3
    pop3: 'AppType' = 110, 'pop3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SUN Remote Procedure Call
    #: - [UDP] SUN Remote Procedure Call
    sunrpc: 'AppType' = 111, 'sunrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] McIDAS Data Transmission Protocol
    #: - [UDP] McIDAS Data Transmission Protocol
    mcidas: 'AppType' = 112, 'mcidas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    ident: 'AppType' = 113, 'ident', TransportProtocol.get('tcp')

    #: - [TCP] Authentication Service
    #: - [UDP] Authentication Service
    auth: 'AppType' = 113, 'auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] unassigned
    unassigned_114: 'AppType' = 114, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Simple File Transfer Protocol
    #: - [UDP] Simple File Transfer Protocol
    sftp: 'AppType' = 115, 'sftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSA REX Notify
    #: - [UDP] ANSA REX Notify
    ansanotify: 'AppType' = 116, 'ansanotify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UUCP Path Service
    #: - [UDP] UUCP Path Service
    uucp_path: 'AppType' = 117, 'uucp-path', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SQL Services
    #: - [UDP] SQL Services
    sqlserv: 'AppType' = 118, 'sqlserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network News Transfer Protocol [:rfc:`3977`]
    #: - [UDP] Network News Transfer Protocol [:rfc:`3977`]
    nntp: 'AppType' = 119, 'nntp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CFDPTKT
    #: - [UDP] CFDPTKT
    cfdptkt: 'AppType' = 120, 'cfdptkt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Encore Expedited Remote Pro.Call
    #: - [UDP] Encore Expedited Remote Pro.Call
    erpc: 'AppType' = 121, 'erpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMAKYNET
    #: - [UDP] SMAKYNET
    smakynet: 'AppType' = 122, 'smakynet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Time Protocol [:rfc:`5905`]
    #: - [UDP] Network Time Protocol [:rfc:`5905`]
    ntp: 'AppType' = 123, 'ntp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSA REX Trader
    #: - [UDP] ANSA REX Trader
    ansatrader: 'AppType' = 124, 'ansatrader', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Locus PC-Interface Net Map Ser
    #: - [UDP] Locus PC-Interface Net Map Ser
    locus_map: 'AppType' = 125, 'locus-map', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NXEdit
    #: - [UDP] NXEdit
    nxedit: 'AppType' = 126, 'nxedit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Locus PC-Interface Conn Server
    #: - [UDP] Locus PC-Interface Conn Server
    locus_con: 'AppType' = 127, 'locus-con', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GSS X License Verification
    #: - [UDP] GSS X License Verification
    gss_xlicen: 'AppType' = 128, 'gss-xlicen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Password Generator Protocol
    #: - [UDP] Password Generator Protocol
    pwdgen: 'AppType' = 129, 'pwdgen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco FNATIVE
    #: - [UDP] cisco FNATIVE
    cisco_fna: 'AppType' = 130, 'cisco-fna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco TNATIVE
    #: - [UDP] cisco TNATIVE
    cisco_tna: 'AppType' = 131, 'cisco-tna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco SYSMAINT
    #: - [UDP] cisco SYSMAINT
    cisco_sys: 'AppType' = 132, 'cisco-sys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Statistics Service
    #: - [UDP] Statistics Service
    statsrv: 'AppType' = 133, 'statsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INGRES-NET Service
    #: - [UDP] INGRES-NET Service
    ingres_net: 'AppType' = 134, 'ingres-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DCE endpoint resolution
    #: - [UDP] DCE endpoint resolution
    epmap: 'AppType' = 135, 'epmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PROFILE Naming System
    #: - [UDP] PROFILE Naming System
    profile: 'AppType' = 136, 'profile', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETBIOS Name Service
    #: - [UDP] NETBIOS Name Service
    netbios_ns: 'AppType' = 137, 'netbios-ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETBIOS Datagram Service
    #: - [UDP] NETBIOS Datagram Service
    netbios_dgm: 'AppType' = 138, 'netbios-dgm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETBIOS Session Service
    #: - [UDP] NETBIOS Session Service
    netbios_ssn: 'AppType' = 139, 'netbios-ssn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMFIS Data Service
    #: - [UDP] EMFIS Data Service
    emfis_data: 'AppType' = 140, 'emfis-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMFIS Control Service
    #: - [UDP] EMFIS Control Service
    emfis_cntl: 'AppType' = 141, 'emfis-cntl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Britton-Lee IDM
    #: - [UDP] Britton-Lee IDM
    bl_idm: 'AppType' = 142, 'bl-idm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Internet Message Access Protocol [:rfc:`3501`][:rfc:`9051`]
    imap: 'AppType' = 143, 'imap', TransportProtocol.get('tcp')

    #: [UDP] Reserved [:rfc:`9051`]
    reserved_143: 'AppType' = 143, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] UAAC Protocol
    #: - [UDP] UAAC Protocol
    uaac: 'AppType' = 145, 'uaac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISO-IP0
    #: - [UDP] ISO-IP0
    iso_tp0: 'AppType' = 146, 'iso-tp0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISO-IP
    #: - [UDP] ISO-IP
    iso_ip: 'AppType' = 147, 'iso-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jargon
    #: - [UDP] Jargon
    jargon: 'AppType' = 148, 'jargon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AED 512 Emulation Service
    #: - [UDP] AED 512 Emulation Service
    aed_512: 'AppType' = 149, 'aed-512', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle SQL*NET IANA assigned this well-formed service name as a
    #:   replacement for "sql*net".
    #: - [TCP] Oracle SQL*NET
    #: - [UDP] Oracle SQL*NET IANA assigned this well-formed service name as a
    #:   replacement for "sql*net".
    #: - [UDP] Oracle SQL*NET
    sql_net_66: 'AppType' = 66, 'sql-net', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] SQL-NET
    #: - [UDP] SQL-NET
    sql_net_150: 'AppType' = 150, 'sql-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HEMS
    #: - [UDP] HEMS
    hems: 'AppType' = 151, 'hems', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Background File Transfer Program
    #: - [UDP] Background File Transfer Program
    bftp: 'AppType' = 152, 'bftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SGMP
    #: - [UDP] SGMP
    sgmp: 'AppType' = 153, 'sgmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETSC
    #: - [UDP] NETSC
    netsc_prod: 'AppType' = 154, 'netsc-prod', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETSC
    #: - [UDP] NETSC
    netsc_dev: 'AppType' = 155, 'netsc-dev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SQL Service
    #: - [UDP] SQL Service
    sqlsrv: 'AppType' = 156, 'sqlsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KNET/VM Command/Message Protocol
    #: - [UDP] KNET/VM Command/Message Protocol
    knet_cmp: 'AppType' = 157, 'knet-cmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCMail Server
    #: - [UDP] PCMail Server
    pcmail_srv: 'AppType' = 158, 'pcmail-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSS-Routing
    #: - [UDP] NSS-Routing
    nss_routing: 'AppType' = 159, 'nss-routing', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SGMP-TRAPS
    #: - [UDP] SGMP-TRAPS
    sgmp_traps: 'AppType' = 160, 'sgmp-traps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNMP
    #: - [UDP] SNMP
    snmp: 'AppType' = 161, 'snmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNMPTRAP
    #: - [UDP] SNMPTRAP
    snmptrap: 'AppType' = 162, 'snmptrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CMIP/TCP Manager
    #: - [UDP] CMIP/TCP Manager
    cmip_man: 'AppType' = 163, 'cmip-man', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CMIP/TCP Agent
    #: - [UDP] CMIP/TCP Agent
    cmip_agent: 'AppType' = 164, 'cmip-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xerox
    #: - [UDP] Xerox
    xns_courier: 'AppType' = 165, 'xns-courier', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sirius Systems
    #: - [UDP] Sirius Systems
    s_net: 'AppType' = 166, 's-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NAMP
    #: - [UDP] NAMP
    namp: 'AppType' = 167, 'namp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSVD
    #: - [UDP] RSVD
    rsvd: 'AppType' = 168, 'rsvd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SEND
    #: - [UDP] SEND
    send: 'AppType' = 169, 'send', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network PostScript
    #: - [UDP] Network PostScript
    print_srv: 'AppType' = 170, 'print-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Innovations Multiplex
    #: - [UDP] Network Innovations Multiplex
    multiplex: 'AppType' = 171, 'multiplex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Innovations CL/1 IANA assigned this well-formed service name
    #:   as a replacement for "cl/1".
    #: - [TCP] Network Innovations CL/1
    #: - [UDP] Network Innovations CL/1 IANA assigned this well-formed service name
    #:   as a replacement for "cl/1".
    #: - [UDP] Network Innovations CL/1
    cl_1: 'AppType' = 172, 'cl-1', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Xyplex
    #: - [UDP] Xyplex
    xyplex_mux: 'AppType' = 173, 'xyplex-mux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MAILQ
    #: - [UDP] MAILQ
    mailq: 'AppType' = 174, 'mailq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VMNET
    #: - [UDP] VMNET
    vmnet: 'AppType' = 175, 'vmnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GENRAD-MUX
    #: - [UDP] GENRAD-MUX
    genrad_mux: 'AppType' = 176, 'genrad-mux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] X Display Manager Control Protocol
    #: - [UDP] X Display Manager Control Protocol
    xdmcp: 'AppType' = 177, 'xdmcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NextStep Window Server
    #: - [UDP] NextStep Window Server
    nextstep: 'AppType' = 178, 'nextstep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Border Gateway Protocol
    #: - [UDP] Border Gateway Protocol
    #: - [SCTP] BGP [:rfc:`9260`]
    bgp: 'AppType' = 179, 'bgp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Intergraph
    #: - [UDP] Intergraph
    ris: 'AppType' = 180, 'ris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify
    #: - [UDP] Unify
    unify: 'AppType' = 181, 'unify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unisys Audit SITP
    #: - [UDP] Unisys Audit SITP
    audit: 'AppType' = 182, 'audit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OCBinder
    #: - [UDP] OCBinder
    ocbinder: 'AppType' = 183, 'ocbinder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OCServer
    #: - [UDP] OCServer
    ocserver: 'AppType' = 184, 'ocserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote-KIS
    #: - [UDP] Remote-KIS
    remote_kis: 'AppType' = 185, 'remote-kis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KIS Protocol
    #: - [UDP] KIS Protocol
    kis: 'AppType' = 186, 'kis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Application Communication Interface
    #: - [UDP] Application Communication Interface
    aci: 'AppType' = 187, 'aci', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Plus Five's MUMPS
    #: - [UDP] Plus Five's MUMPS
    mumps: 'AppType' = 188, 'mumps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Queued File Transport
    #: - [UDP] Queued File Transport
    qft: 'AppType' = 189, 'qft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gateway Access Control Protocol
    #: - [UDP] Gateway Access Control Protocol
    gacp: 'AppType' = 190, 'gacp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Directory Service
    #: - [UDP] Prospero Directory Service
    prospero: 'AppType' = 191, 'prospero', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OSU Network Monitoring System
    #: - [UDP] OSU Network Monitoring System
    osu_nms: 'AppType' = 192, 'osu-nms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spider Remote Monitoring Protocol
    #: - [UDP] Spider Remote Monitoring Protocol
    srmp: 'AppType' = 193, 'srmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Internet Relay Chat Protocol
    #: - [UDP] Internet Relay Chat Protocol
    irc: 'AppType' = 194, 'irc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNSIX Network Level Module Audit
    #: - [UDP] DNSIX Network Level Module Audit
    dn6_nlm_aud: 'AppType' = 195, 'dn6-nlm-aud', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNSIX Session Mgt Module Audit Redir
    #: - [UDP] DNSIX Session Mgt Module Audit Redir
    dn6_smm_red: 'AppType' = 196, 'dn6-smm-red', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Directory Location Service Monitor
    #: - [UDP] Directory Location Service Monitor
    dls_mon: 'AppType' = 198, 'dls-mon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMUX
    #: - [UDP] SMUX
    smux: 'AppType' = 199, 'smux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM System Resource Controller
    #: - [UDP] IBM System Resource Controller
    src: 'AppType' = 200, 'src', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Routing Maintenance
    #: - [UDP] AppleTalk Routing Maintenance
    at_rtmp: 'AppType' = 201, 'at-rtmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Name Binding
    #: - [UDP] AppleTalk Name Binding
    at_nbp: 'AppType' = 202, 'at-nbp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Unused
    #: - [UDP] AppleTalk Unused
    at_3: 'AppType' = 203, 'at-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Echo
    #: - [UDP] AppleTalk Echo
    at_echo: 'AppType' = 204, 'at-echo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Unused
    #: - [UDP] AppleTalk Unused
    at_5: 'AppType' = 205, 'at-5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Zone Information
    #: - [UDP] AppleTalk Zone Information
    at_zis: 'AppType' = 206, 'at-zis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Unused
    #: - [UDP] AppleTalk Unused
    at_7: 'AppType' = 207, 'at-7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleTalk Unused
    #: - [UDP] AppleTalk Unused
    at_8: 'AppType' = 208, 'at-8', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] The Quick Mail Transfer Protocol
    #: - [UDP] The Quick Mail Transfer Protocol
    qmtp: 'AppType' = 209, 'qmtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSI Z39.50 IANA assigned this well-formed service name as a
    #:   replacement for "z39.50".
    #: - [TCP] ANSI Z39.50
    #: - [UDP] ANSI Z39.50 IANA assigned this well-formed service name as a
    #:   replacement for "z39.50".
    #: - [UDP] ANSI Z39.50
    z39_50: 'AppType' = 210, 'z39-50', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Texas Instruments 914C/G Terminal IANA assigned this well-formed
    #:   service name as a replacement for "914c/g".
    #: - [TCP] Texas Instruments 914C/G Terminal
    #: - [UDP] Texas Instruments 914C/G Terminal IANA assigned this well-formed
    #:   service name as a replacement for "914c/g".
    #: - [UDP] Texas Instruments 914C/G Terminal
    AppType_914c_g: 'AppType' = 211, '914c-g', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ATEXSSTR
    #: - [UDP] ATEXSSTR
    anet: 'AppType' = 212, 'anet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPX
    #: - [UDP] IPX
    ipx: 'AppType' = 213, 'ipx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VM PWSCS
    #: - [UDP] VM PWSCS
    vmpwscs: 'AppType' = 214, 'vmpwscs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Insignia Solutions
    #: - [UDP] Insignia Solutions
    softpc: 'AppType' = 215, 'softpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Computer Associates Int'l License Server
    #: - [UDP] Computer Associates Int'l License Server
    cailic: 'AppType' = 216, 'cailic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dBASE Unix
    #: - [UDP] dBASE Unix
    dbase: 'AppType' = 217, 'dbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netix Message Posting Protocol
    #: - [UDP] Netix Message Posting Protocol
    mpp: 'AppType' = 218, 'mpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unisys ARPs
    #: - [UDP] Unisys ARPs
    uarps: 'AppType' = 219, 'uarps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interactive Mail Access Protocol v3
    #: - [UDP] Interactive Mail Access Protocol v3
    imap3: 'AppType' = 220, 'imap3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Berkeley rlogind with SPX auth
    #: - [UDP] Berkeley rlogind with SPX auth
    fln_spx: 'AppType' = 221, 'fln-spx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Berkeley rshd with SPX auth
    #: - [UDP] Berkeley rshd with SPX auth
    rsh_spx: 'AppType' = 222, 'rsh-spx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Certificate Distribution Center
    #: - [UDP] Certificate Distribution Center
    cdc: 'AppType' = 223, 'cdc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] masqdialer
    #: - [UDP] masqdialer
    masqdialer: 'AppType' = 224, 'masqdialer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct
    #: - [UDP] Direct
    direct: 'AppType' = 242, 'direct', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Survey Measurement
    #: - [UDP] Survey Measurement
    sur_meas: 'AppType' = 243, 'sur-meas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] inbusiness
    #: - [UDP] inbusiness
    inbusiness: 'AppType' = 244, 'inbusiness', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LINK
    #: - [UDP] LINK
    link: 'AppType' = 245, 'link', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Display Systems Protocol
    #: - [UDP] Display Systems Protocol
    dsp3270: 'AppType' = 246, 'dsp3270', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SUBNTBCST_TFTP IANA assigned this well-formed service name as a
    #:   replacement for "subntbcst_tftp".
    #: - [TCP] SUBNTBCST_TFTP
    #: - [UDP] SUBNTBCST_TFTP IANA assigned this well-formed service name as a
    #:   replacement for "subntbcst_tftp".
    #: - [UDP] SUBNTBCST_TFTP
    subntbcst_tftp: 'AppType' = 247, 'subntbcst-tftp', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] bhfhs
    #: - [UDP] bhfhs
    bhfhs: 'AppType' = 248, 'bhfhs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Route Access Protocol
    #: - [UDP] Route Access Protocol
    rap_38: 'AppType' = 38, 'rap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RAP
    #: - [UDP] RAP
    rap_256: 'AppType' = 256, 'rap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Electronic Transaction
    #: - [UDP] Secure Electronic Transaction
    set: 'AppType' = 257, 'set', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_258: 'AppType' = 258, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Efficient Short Remote Operations
    #: - [UDP] Efficient Short Remote Operations
    esro_gen: 'AppType' = 259, 'esro-gen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Openport
    #: - [UDP] Openport
    openport: 'AppType' = 260, 'openport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IIOP Name Service over TLS/SSL
    #: - [UDP] IIOP Name Service over TLS/SSL
    nsiiops: 'AppType' = 261, 'nsiiops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Arcisdms
    #: - [UDP] Arcisdms
    arcisdms: 'AppType' = 262, 'arcisdms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HDAP
    #: - [UDP] HDAP
    hdap: 'AppType' = 263, 'hdap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BGMP
    #: - [UDP] BGMP
    bgmp: 'AppType' = 264, 'bgmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] X-Bone CTL
    #: - [UDP] X-Bone CTL
    x_bone_ctl: 'AppType' = 265, 'x-bone-ctl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCSI on ST
    #: - [UDP] SCSI on ST
    sst: 'AppType' = 266, 'sst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tobit David Service Layer
    #: - [UDP] Tobit David Service Layer
    td_service: 'AppType' = 267, 'td-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tobit David Replica
    #: - [UDP] Tobit David Replica
    td_replica: 'AppType' = 268, 'td-replica', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MANET Protocols [:rfc:`5498`]
    #: - [UDP] MANET Protocols [:rfc:`5498`]
    manet: 'AppType' = 269, 'manet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_270: 'AppType' = 270, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Q-mode encapsulation for GIST messages [:rfc:`5971`]
    gist: 'AppType' = 270, 'gist', TransportProtocol.get('udp')

    #: [TCP] IETF Network Endpoint Assessment (NEA) Posture Transport Protocol over
    #: TLS (PT-TLS) [:rfc:`6876`]
    pt_tls: 'AppType' = 271, 'pt-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_271: 'AppType' = 271, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] http-mgmt
    #: - [UDP] http-mgmt
    http_mgmt: 'AppType' = 280, 'http-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Personal Link
    #: - [UDP] Personal Link
    personal_link: 'AppType' = 281, 'personal-link', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cable Port A/X
    #: - [UDP] Cable Port A/X
    cableport_ax: 'AppType' = 282, 'cableport-ax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rescap
    #: - [UDP] rescap
    rescap: 'AppType' = 283, 'rescap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] corerjd
    #: - [UDP] corerjd
    corerjd: 'AppType' = 284, 'corerjd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_285: 'AppType' = 285, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] K-BLOCK
    #: - [UDP] K-BLOCK
    k_block: 'AppType' = 287, 'k-block', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novastor Backup
    #: - [UDP] Novastor Backup
    novastorbakcup: 'AppType' = 308, 'novastorbakcup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EntrustTime
    #: - [UDP] EntrustTime
    entrusttime: 'AppType' = 309, 'entrusttime', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bhmds
    #: - [UDP] bhmds
    bhmds: 'AppType' = 310, 'bhmds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppleShare IP WebAdmin
    #: - [UDP] AppleShare IP WebAdmin
    asip_webadmin: 'AppType' = 311, 'asip-webadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSLMP
    #: - [UDP] VSLMP
    vslmp: 'AppType' = 312, 'vslmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Magenta Logic
    #: - [UDP] Magenta Logic
    magenta_logic: 'AppType' = 313, 'magenta-logic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Opalis Robot
    #: - [UDP] Opalis Robot
    opalis_robot: 'AppType' = 314, 'opalis-robot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DPSI
    #: - [UDP] DPSI
    dpsi: 'AppType' = 315, 'dpsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] decAuth
    #: - [UDP] decAuth
    decauth: 'AppType' = 316, 'decauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zannet
    #: - [UDP] Zannet
    zannet: 'AppType' = 317, 'zannet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PKIX TimeStamp
    #: - [UDP] PKIX TimeStamp
    pkix_timestamp: 'AppType' = 318, 'pkix-timestamp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PTP Event
    #: - [UDP] PTP Event
    ptp_event: 'AppType' = 319, 'ptp-event', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PTP General
    #: - [UDP] PTP General
    ptp_general: 'AppType' = 320, 'ptp-general', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTSPS
    #: - [UDP] RTSPS
    rtsps: 'AppType' = 322, 'rtsps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Resource PKI to Router Protocol [:rfc:`6810`]
    rpki_rtr: 'AppType' = 323, 'rpki-rtr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_323: 'AppType' = 323, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Resource PKI to Router Protocol over TLS [:rfc:`6810`]
    rpki_rtr_tls: 'AppType' = 324, 'rpki-rtr-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_324: 'AppType' = 324, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Texar Security Port
    #: - [UDP] Texar Security Port
    texar: 'AppType' = 333, 'texar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Data Access Protocol
    #: - [UDP] Prospero Data Access Protocol
    pdap: 'AppType' = 344, 'pdap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Perf Analysis Workbench
    #: - [UDP] Perf Analysis Workbench
    pawserv: 'AppType' = 345, 'pawserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zebra server
    #: - [UDP] Zebra server
    zserv: 'AppType' = 346, 'zserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fatmen Server
    #: - [UDP] Fatmen Server
    fatserv: 'AppType' = 347, 'fatserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cabletron Management Protocol
    #: - [UDP] Cabletron Management Protocol
    csi_sgwp: 'AppType' = 348, 'csi-sgwp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MATIP Type A [:rfc:`2351`]
    #: - [UDP] MATIP Type A [:rfc:`2351`]
    matip_type_a: 'AppType' = 350, 'matip-type-a', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MATIP Type B [:rfc:`2351`]
    #: - [UDP] MATIP Type B [:rfc:`2351`]
    matip_type_b: 'AppType' = 351, 'matip-type-b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bhoetty
    #: - [UDP] bhoetty
    bhoetty: 'AppType' = 351, 'bhoetty', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTAG
    #: - [UDP] DTAG
    dtag_ste_sb: 'AppType' = 352, 'dtag-ste-sb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bhoedap4
    #: - [UDP] bhoedap4
    bhoedap4: 'AppType' = 352, 'bhoedap4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDSAUTH
    #: - [UDP] NDSAUTH
    ndsauth: 'AppType' = 353, 'ndsauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bh611
    #: - [UDP] bh611
    bh611: 'AppType' = 354, 'bh611', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DATEX-ASN
    #: - [UDP] DATEX-ASN
    datex_asn: 'AppType' = 355, 'datex-asn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cloanto Net 1
    #: - [UDP] Cloanto Net 1
    cloanto_net_1: 'AppType' = 356, 'cloanto-net-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bhevent
    #: - [UDP] bhevent
    bhevent: 'AppType' = 357, 'bhevent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Shrinkwrap
    #: - [UDP] Shrinkwrap
    shrinkwrap: 'AppType' = 358, 'shrinkwrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_359: 'AppType' = 359, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] scoi2odialog
    #: - [UDP] scoi2odialog
    scoi2odialog: 'AppType' = 360, 'scoi2odialog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Semantix
    #: - [UDP] Semantix
    semantix: 'AppType' = 361, 'semantix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SRS Send
    #: - [UDP] SRS Send
    srssend: 'AppType' = 362, 'srssend', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSVP Tunnel IANA assigned this well-formed service name as a
    #:   replacement for "rsvp_tunnel".
    #: - [TCP] RSVP Tunnel
    #: - [UDP] RSVP Tunnel IANA assigned this well-formed service name as a
    #:   replacement for "rsvp_tunnel".
    #: - [UDP] RSVP Tunnel
    rsvp_tunnel: 'AppType' = 363, 'rsvp-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Aurora CMGR
    #: - [UDP] Aurora CMGR
    aurora_cmgr: 'AppType' = 364, 'aurora-cmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTK
    #: - [UDP] DTK
    dtk: 'AppType' = 365, 'dtk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ODMR
    #: - [UDP] ODMR
    odmr: 'AppType' = 366, 'odmr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MortgageWare
    #: - [UDP] MortgageWare
    mortgageware: 'AppType' = 367, 'mortgageware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QbikGDP
    #: - [UDP] QbikGDP
    qbikgdp: 'AppType' = 368, 'qbikgdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rpc2portmap
    #: - [UDP] rpc2portmap
    rpc2portmap: 'AppType' = 369, 'rpc2portmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] codaauth2
    #: - [UDP] codaauth2
    codaauth2: 'AppType' = 370, 'codaauth2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Clearcase
    #: - [UDP] Clearcase
    clearcase: 'AppType' = 371, 'clearcase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ListProcessor
    #: - [UDP] ListProcessor
    ulistproc: 'AppType' = 372, 'ulistproc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Legent Corporation
    #: - [UDP] Legent Corporation
    legent_1: 'AppType' = 373, 'legent-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Legent Corporation
    #: - [UDP] Legent Corporation
    legent_2: 'AppType' = 374, 'legent-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hassle
    #: - [UDP] Hassle
    hassle: 'AppType' = 375, 'hassle', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Amiga Envoy Network Inquiry Protocol
    #: - [UDP] Amiga Envoy Network Inquiry Protocol
    nip: 'AppType' = 376, 'nip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NEC Corporation
    #: - [UDP] NEC Corporation
    tnetos: 'AppType' = 377, 'tnetos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NEC Corporation
    #: - [UDP] NEC Corporation
    dsetos: 'AppType' = 378, 'dsetos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TIA/EIA/IS-99 modem client
    #: - [UDP] TIA/EIA/IS-99 modem client
    is99c: 'AppType' = 379, 'is99c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TIA/EIA/IS-99 modem server
    #: - [UDP] TIA/EIA/IS-99 modem server
    is99s: 'AppType' = 380, 'is99s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp performance data collector
    #: - [UDP] hp performance data collector
    hp_collector: 'AppType' = 381, 'hp-collector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp performance data managed node
    #: - [UDP] hp performance data managed node
    hp_managed_node: 'AppType' = 382, 'hp-managed-node', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp performance data alarm manager
    #: - [UDP] hp performance data alarm manager
    hp_alarm_mgr: 'AppType' = 383, 'hp-alarm-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A Remote Network Server System
    #: - [UDP] A Remote Network Server System
    arns: 'AppType' = 384, 'arns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Application
    #: - [UDP] IBM Application
    ibm_app: 'AppType' = 385, 'ibm-app', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASA Message Router Object Def.
    #: - [UDP] ASA Message Router Object Def.
    asa: 'AppType' = 386, 'asa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Appletalk Update-Based Routing Pro.
    #: - [UDP] Appletalk Update-Based Routing Pro.
    aurp: 'AppType' = 387, 'aurp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unidata LDM
    #: - [UDP] Unidata LDM
    unidata_ldm: 'AppType' = 388, 'unidata-ldm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lightweight Directory Access Protocol
    #: - [UDP] Lightweight Directory Access Protocol
    ldap: 'AppType' = 389, 'ldap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UIS
    #: - [UDP] UIS
    uis: 'AppType' = 390, 'uis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SynOptics SNMP Relay Port
    #: - [UDP] SynOptics SNMP Relay Port
    synotics_relay: 'AppType' = 391, 'synotics-relay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SynOptics Port Broker Port
    #: - [UDP] SynOptics Port Broker Port
    synotics_broker: 'AppType' = 392, 'synotics-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Meta5
    #: - [UDP] Meta5
    meta5: 'AppType' = 393, 'meta5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMBL Nucleic Data Transfer
    #: - [UDP] EMBL Nucleic Data Transfer
    embl_ndt: 'AppType' = 394, 'embl-ndt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetScout Control Protocol
    #: - [UDP] NetScout Control Protocol
    netcp: 'AppType' = 395, 'netcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novell Netware over IP
    #: - [UDP] Novell Netware over IP
    netware_ip: 'AppType' = 396, 'netware-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multi Protocol Trans. Net.
    #: - [UDP] Multi Protocol Trans. Net.
    mptn: 'AppType' = 397, 'mptn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kryptolan
    #: - [UDP] Kryptolan
    kryptolan: 'AppType' = 398, 'kryptolan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISO Transport Class 2 Non-Control over TCP
    #: - [UDP] ISO Transport Class 2 Non-Control over UDP
    iso_tsap_c2: 'AppType' = 399, 'iso-tsap-c2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle Secure Backup
    #: - [UDP] Oracle Secure Backup
    osb_sd: 'AppType' = 400, 'osb-sd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Uninterruptible Power Supply
    #: - [UDP] Uninterruptible Power Supply
    ups: 'AppType' = 401, 'ups', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Genie Protocol
    #: - [UDP] Genie Protocol
    genie: 'AppType' = 402, 'genie', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] decap
    #: - [UDP] decap
    decap: 'AppType' = 403, 'decap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nced
    #: - [UDP] nced
    nced: 'AppType' = 404, 'nced', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncld
    #: - [UDP] ncld
    ncld: 'AppType' = 405, 'ncld', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interactive Mail Support Protocol
    #: - [UDP] Interactive Mail Support Protocol
    imsp: 'AppType' = 406, 'imsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Timbuktu
    #: - [UDP] Timbuktu
    timbuktu: 'AppType' = 407, 'timbuktu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Resource Manager Sys. Man.
    #: - [UDP] Prospero Resource Manager Sys. Man.
    prm_sm: 'AppType' = 408, 'prm-sm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Resource Manager Node Man.
    #: - [UDP] Prospero Resource Manager Node Man.
    prm_nm: 'AppType' = 409, 'prm-nm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DECLadebug Remote Debug Protocol
    #: - [UDP] DECLadebug Remote Debug Protocol
    decladebug: 'AppType' = 410, 'decladebug', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote MT Protocol
    #: - [UDP] Remote MT Protocol
    rmt: 'AppType' = 411, 'rmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trap Convention Port
    #: - [UDP] Trap Convention Port
    synoptics_trap: 'AppType' = 412, 'synoptics-trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Storage Management Services Protocol
    #: - [UDP] Storage Management Services Protocol
    smsp: 'AppType' = 413, 'smsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InfoSeek
    #: - [UDP] InfoSeek
    infoseek: 'AppType' = 414, 'infoseek', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BNet
    #: - [UDP] BNet
    bnet: 'AppType' = 415, 'bnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Silverplatter
    #: - [UDP] Silverplatter
    silverplatter: 'AppType' = 416, 'silverplatter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Onmux
    #: - [UDP] Onmux
    onmux: 'AppType' = 417, 'onmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hyper-G
    #: - [UDP] Hyper-G
    hyper_g: 'AppType' = 418, 'hyper-g', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ariel 1
    #: - [UDP] Ariel 1
    ariel1: 'AppType' = 419, 'ariel1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMPTE
    #: - [UDP] SMPTE
    smpte: 'AppType' = 420, 'smpte', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ariel 2
    #: - [UDP] Ariel 2
    ariel2: 'AppType' = 421, 'ariel2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ariel 3
    #: - [UDP] Ariel 3
    ariel3: 'AppType' = 422, 'ariel3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Operations Planning and Control Start
    #: - [UDP] IBM Operations Planning and Control Start
    opc_job_start: 'AppType' = 423, 'opc-job-start', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Operations Planning and Control Track
    #: - [UDP] IBM Operations Planning and Control Track
    opc_job_track: 'AppType' = 424, 'opc-job-track', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICAD
    #: - [UDP] ICAD
    icad_el: 'AppType' = 425, 'icad-el', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] smartsdp
    #: - [UDP] smartsdp
    smartsdp: 'AppType' = 426, 'smartsdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Server Location
    #: - [UDP] Server Location
    svrloc: 'AppType' = 427, 'svrloc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OCS_CMU IANA assigned this well-formed service name as a replacement
    #:   for "ocs_cmu".
    #: - [TCP] OCS_CMU
    #: - [UDP] OCS_CMU IANA assigned this well-formed service name as a replacement
    #:   for "ocs_cmu".
    #: - [UDP] OCS_CMU
    ocs_cmu: 'AppType' = 428, 'ocs-cmu', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] OCS_AMU IANA assigned this well-formed service name as a replacement
    #:   for "ocs_amu".
    #: - [TCP] OCS_AMU
    #: - [UDP] OCS_AMU IANA assigned this well-formed service name as a replacement
    #:   for "ocs_amu".
    #: - [UDP] OCS_AMU
    ocs_amu: 'AppType' = 429, 'ocs-amu', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] UTMPSD
    #: - [UDP] UTMPSD
    utmpsd: 'AppType' = 430, 'utmpsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UTMPCD
    #: - [UDP] UTMPCD
    utmpcd: 'AppType' = 431, 'utmpcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IASD
    #: - [UDP] IASD
    iasd: 'AppType' = 432, 'iasd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NNTP for transit servers (NNSP) [:rfc:`3977`]
    #: - [UDP] NNTP for transit servers (NNSP) [:rfc:`3977`]
    nnsp: 'AppType' = 433, 'nnsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MobileIP-Agent
    #: - [UDP] MobileIP-Agent
    mobileip_agent: 'AppType' = 434, 'mobileip-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MobilIP-MN
    #: - [UDP] MobilIP-MN
    mobilip_mn: 'AppType' = 435, 'mobilip-mn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNA-CML
    #: - [UDP] DNA-CML
    dna_cml: 'AppType' = 436, 'dna-cml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] comscm
    #: - [UDP] comscm
    comscm: 'AppType' = 437, 'comscm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dsfgw
    #: - [UDP] dsfgw
    dsfgw: 'AppType' = 438, 'dsfgw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dasp
    #: - [UDP] dasp
    dasp: 'AppType' = 439, 'dasp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sgcp
    #: - [UDP] sgcp
    sgcp: 'AppType' = 440, 'sgcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] decvms-sysmgt
    #: - [UDP] decvms-sysmgt
    decvms_sysmgt: 'AppType' = 441, 'decvms-sysmgt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cvc_hostd IANA assigned this well-formed service name as a replacement
    #:   for "cvc_hostd".
    #: - [TCP] cvc_hostd
    #: - [UDP] cvc_hostd IANA assigned this well-formed service name as a replacement
    #:   for "cvc_hostd".
    #: - [UDP] cvc_hostd
    cvc_hostd: 'AppType' = 442, 'cvc-hostd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Network Paging Protocol [:rfc:`1568`]
    #: - [UDP] Simple Network Paging Protocol [:rfc:`1568`]
    snpp: 'AppType' = 444, 'snpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft-DS
    #: - [UDP] Microsoft-DS
    microsoft_ds: 'AppType' = 445, 'microsoft-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DDM-Remote Relational Database Access
    #: - [UDP] DDM-Remote Relational Database Access
    ddm_rdb: 'AppType' = 446, 'ddm-rdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DDM-Distributed File Management
    #: - [UDP] DDM-Distributed File Management
    ddm_dfm: 'AppType' = 447, 'ddm-dfm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DDM-Remote DB Access Using Secure Sockets
    #: - [UDP] DDM-Remote DB Access Using Secure Sockets
    ddm_ssl: 'AppType' = 448, 'ddm-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AS Server Mapper
    #: - [UDP] AS Server Mapper
    as_servermap: 'AppType' = 449, 'as-servermap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Computer Supported Telecomunication Applications
    #: - [UDP] Computer Supported Telecomunication Applications
    tserver: 'AppType' = 450, 'tserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cray Network Semaphore server
    #: - [UDP] Cray Network Semaphore server
    sfs_smp_net: 'AppType' = 451, 'sfs-smp-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cray SFS config server
    #: - [UDP] Cray SFS config server
    sfs_config: 'AppType' = 452, 'sfs-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] macon-tcp
    macon_tcp: 'AppType' = 456, 'macon-tcp', TransportProtocol.get('tcp')

    #: [UDP] macon-udp
    macon_udp: 'AppType' = 456, 'macon-udp', TransportProtocol.get('udp')

    #: - [TCP] scohelp
    #: - [UDP] scohelp
    scohelp: 'AppType' = 457, 'scohelp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] apple quick time
    #: - [UDP] apple quick time
    appleqtc: 'AppType' = 458, 'appleqtc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ampr-rcmd
    #: - [UDP] ampr-rcmd
    ampr_rcmd: 'AppType' = 459, 'ampr-rcmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] skronk
    #: - [UDP] skronk
    skronk: 'AppType' = 460, 'skronk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataRampSrv
    #: - [UDP] DataRampSrv
    datasurfsrv: 'AppType' = 461, 'datasurfsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataRampSrvSec
    #: - [UDP] DataRampSrvSec
    datasurfsrvsec: 'AppType' = 462, 'datasurfsrvsec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] alpes
    #: - [UDP] alpes
    alpes: 'AppType' = 463, 'alpes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] kpasswd
    #: - [UDP] kpasswd
    kpasswd: 'AppType' = 464, 'kpasswd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] URL Rendezvous Directory for SSM
    urd: 'AppType' = 465, 'urd', TransportProtocol.get('tcp')

    #: [TCP] Message Submission over TLS protocol [:rfc:`8314`]
    submissions: 'AppType' = 465, 'submissions', TransportProtocol.get('tcp')

    #: [UDP] IGMP over UDP for SSM
    igmpv3lite: 'AppType' = 465, 'igmpv3lite', TransportProtocol.get('udp')

    #: - [TCP] digital-vrc
    #: - [UDP] digital-vrc
    digital_vrc: 'AppType' = 466, 'digital-vrc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mylex-mapd
    #: - [UDP] mylex-mapd
    mylex_mapd: 'AppType' = 467, 'mylex-mapd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proturis
    #: - [UDP] proturis
    photuris: 'AppType' = 468, 'photuris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Radio Control Protocol
    #: - [UDP] Radio Control Protocol
    rcp: 'AppType' = 469, 'rcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] scx-proxy
    #: - [UDP] scx-proxy
    scx_proxy: 'AppType' = 470, 'scx-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mondex
    #: - [UDP] Mondex
    mondex: 'AppType' = 471, 'mondex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ljk-login
    #: - [UDP] ljk-login
    ljk_login: 'AppType' = 472, 'ljk-login', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hybrid-pop
    #: - [UDP] hybrid-pop
    hybrid_pop: 'AppType' = 473, 'hybrid-pop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] tn-tl-w1
    tn_tl_w1: 'AppType' = 474, 'tn-tl-w1', TransportProtocol.get('tcp')

    #: [UDP] tn-tl-w2
    tn_tl_w2: 'AppType' = 474, 'tn-tl-w2', TransportProtocol.get('udp')

    #: - [TCP] tcpnethaspsrv
    #: - [UDP] tcpnethaspsrv
    tcpnethaspsrv: 'AppType' = 475, 'tcpnethaspsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tn-tl-fd1
    #: - [UDP] tn-tl-fd1
    tn_tl_fd1: 'AppType' = 476, 'tn-tl-fd1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ss7ns
    #: - [UDP] ss7ns
    ss7ns: 'AppType' = 477, 'ss7ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] spsc
    #: - [UDP] spsc
    spsc: 'AppType' = 478, 'spsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iafserver
    #: - [UDP] iafserver
    iafserver: 'AppType' = 479, 'iafserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iafdbase
    #: - [UDP] iafdbase
    iafdbase: 'AppType' = 480, 'iafdbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ph service
    #: - [UDP] Ph service
    ph: 'AppType' = 481, 'ph', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bgs-nsi
    #: - [UDP] bgs-nsi
    bgs_nsi: 'AppType' = 482, 'bgs-nsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ulpnet
    #: - [UDP] ulpnet
    ulpnet: 'AppType' = 483, 'ulpnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Integra Software Management Environment
    #: - [UDP] Integra Software Management Environment
    integra_sme: 'AppType' = 484, 'integra-sme', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Air Soft Power Burst
    #: - [UDP] Air Soft Power Burst
    powerburst: 'AppType' = 485, 'powerburst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] avian
    #: - [UDP] avian
    avian: 'AppType' = 486, 'avian', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] saft Simple Asynchronous File Transfer
    #: - [UDP] saft Simple Asynchronous File Transfer
    saft: 'AppType' = 487, 'saft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gss-http
    #: - [UDP] gss-http
    gss_http: 'AppType' = 488, 'gss-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nest-protocol
    #: - [UDP] nest-protocol
    nest_protocol: 'AppType' = 489, 'nest-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] micom-pfs
    #: - [UDP] micom-pfs
    micom_pfs: 'AppType' = 490, 'micom-pfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] go-login
    #: - [UDP] go-login
    go_login: 'AppType' = 491, 'go-login', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Transport Independent Convergence for FNA
    #: - [UDP] Transport Independent Convergence for FNA
    ticf_1: 'AppType' = 492, 'ticf-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Transport Independent Convergence for FNA
    #: - [UDP] Transport Independent Convergence for FNA
    ticf_2: 'AppType' = 493, 'ticf-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] POV-Ray
    #: - [UDP] POV-Ray
    pov_ray: 'AppType' = 494, 'pov-ray', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] intecourier
    #: - [UDP] intecourier
    intecourier: 'AppType' = 495, 'intecourier', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PIM-RP-DISC
    #: - [UDP] PIM-RP-DISC
    pim_rp_disc: 'AppType' = 496, 'pim-rp-disc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] siam
    #: - [UDP] siam
    siam: 'AppType' = 498, 'siam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISO ILL Protocol
    #: - [UDP] ISO ILL Protocol
    iso_ill: 'AppType' = 499, 'iso-ill', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isakmp
    #: - [UDP] isakmp
    isakmp: 'AppType' = 500, 'isakmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STMF
    #: - [UDP] STMF
    stmf: 'AppType' = 501, 'stmf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Modbus Application Protocol
    #: - [UDP] Modbus Application Protocol
    mbap: 'AppType' = 502, 'mbap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intrinsa
    #: - [UDP] Intrinsa
    intrinsa: 'AppType' = 503, 'intrinsa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] citadel
    #: - [UDP] citadel
    citadel: 'AppType' = 504, 'citadel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mailbox-lm
    #: - [UDP] mailbox-lm
    mailbox_lm: 'AppType' = 505, 'mailbox-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ohimsrv
    #: - [UDP] ohimsrv
    ohimsrv: 'AppType' = 506, 'ohimsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] crs
    #: - [UDP] crs
    crs: 'AppType' = 507, 'crs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xvttp
    #: - [UDP] xvttp
    xvttp: 'AppType' = 508, 'xvttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] snare
    #: - [UDP] snare
    snare: 'AppType' = 509, 'snare', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FirstClass Protocol
    #: - [UDP] FirstClass Protocol
    fcp: 'AppType' = 510, 'fcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PassGo
    #: - [UDP] PassGo
    passgo: 'AppType' = 511, 'passgo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] remote process execution; authentication performed using passwords and
    #: UNIX login names
    exec: 'AppType' = 512, 'exec', TransportProtocol.get('tcp')

    #: [UDP]
    comsat: 'AppType' = 512, 'comsat', TransportProtocol.get('udp')

    #: [UDP] used by mail system to notify users of new mail received; currently
    #: receives messages only from processes on the same machine
    biff: 'AppType' = 512, 'biff', TransportProtocol.get('udp')

    #: [TCP] remote login a la telnet; automatic authentication performed based on
    #: priviledged port numbers and distributed data bases which identify
    #: "authentication domains"
    login: 'AppType' = 513, 'login', TransportProtocol.get('tcp')

    #: [UDP] maintains data bases showing who's logged in to machines on a local
    #: net and the load average of the machine
    who: 'AppType' = 513, 'who', TransportProtocol.get('udp')

    #: [TCP] cmd like exec, but automatic authentication is performed as for login
    #: server
    shell: 'AppType' = 514, 'shell', TransportProtocol.get('tcp')

    #: [UDP] [:rfc:`5426`]
    syslog: 'AppType' = 514, 'syslog', TransportProtocol.get('udp')

    #: - [TCP] spooler
    #: - [UDP] spooler
    printer: 'AppType' = 515, 'printer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] videotex
    #: - [UDP] videotex
    videotex: 'AppType' = 516, 'videotex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] like tenex link, but across machine - unfortunately, doesn't use link
    #:   protocol (this is actually just a rendezvous port from which a tcp
    #:   connection is established)
    #: - [UDP] like tenex link, but across machine - unfortunately, doesn't use link
    #:   protocol (this is actually just a rendezvous port from which a tcp
    #:   connection is established)
    talk: 'AppType' = 517, 'talk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    ntalk: 'AppType' = 518, 'ntalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] unixtime
    #: - [UDP] unixtime
    utime: 'AppType' = 519, 'utime', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] extended file name server
    efs: 'AppType' = 520, 'efs', TransportProtocol.get('tcp')

    #: [UDP] local routing process (on site); uses variant of Xerox NS routing
    #: information protocol - RIP
    router: 'AppType' = 520, 'router', TransportProtocol.get('udp')

    #: - [TCP] ripng
    #: - [UDP] ripng
    ripng: 'AppType' = 521, 'ripng', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ULP
    #: - [UDP] ULP
    ulp: 'AppType' = 522, 'ulp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM-DB2
    #: - [UDP] IBM-DB2
    ibm_db2: 'AppType' = 523, 'ibm-db2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NCP
    #: - [UDP] NCP
    ncp: 'AppType' = 524, 'ncp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] timeserver
    #: - [UDP] timeserver
    timed: 'AppType' = 525, 'timed', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] newdate
    #: - [UDP] newdate
    tempo: 'AppType' = 526, 'tempo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Stock IXChange
    #: - [UDP] Stock IXChange
    stx: 'AppType' = 527, 'stx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Customer IXChange
    #: - [UDP] Customer IXChange
    custix: 'AppType' = 528, 'custix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRC-SERV
    #: - [UDP] IRC-SERV
    irc_serv: 'AppType' = 529, 'irc-serv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rpc
    #: - [UDP] rpc
    courier: 'AppType' = 530, 'courier', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] chat
    #: - [UDP] chat
    conference: 'AppType' = 531, 'conference', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] readnews
    #: - [UDP] readnews
    netnews: 'AppType' = 532, 'netnews', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] for emergency broadcasts
    #: - [UDP] for emergency broadcasts
    netwall: 'AppType' = 533, 'netwall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] windream Admin
    #: - [UDP] windream Admin
    windream: 'AppType' = 534, 'windream', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iiop
    #: - [UDP] iiop
    iiop: 'AppType' = 535, 'iiop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] opalis-rdv
    #: - [UDP] opalis-rdv
    opalis_rdv: 'AppType' = 536, 'opalis-rdv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gdomap
    #: - [UDP] gdomap
    gdomap: 'AppType' = 538, 'gdomap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apertus Technologies Load Determination
    #: - [UDP] Apertus Technologies Load Determination
    apertus_ldp: 'AppType' = 539, 'apertus-ldp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] uucpd
    #: - [UDP] uucpd
    uucp: 'AppType' = 540, 'uucp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] uucp-rlogin
    #: - [UDP] uucp-rlogin
    uucp_rlogin: 'AppType' = 541, 'uucp-rlogin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] commerce
    #: - [UDP] commerce
    commerce: 'AppType' = 542, 'commerce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    klogin: 'AppType' = 543, 'klogin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] krcmd
    #: - [UDP] krcmd
    kshell: 'AppType' = 544, 'kshell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appleqtcsrvr
    #: - [UDP] appleqtcsrvr
    appleqtcsrvr: 'AppType' = 545, 'appleqtcsrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DHCPv6 Client
    #: - [UDP] DHCPv6 Client [:rfc:`8415`]
    dhcpv6_client: 'AppType' = 546, 'dhcpv6-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DHCPv6 Server [:rfc:`5460`]
    #: - [UDP] DHCPv6 Server [:rfc:`8415`]
    dhcpv6_server: 'AppType' = 547, 'dhcpv6-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AFP over TCP
    #: - [UDP] AFP over TCP
    afpovertcp: 'AppType' = 548, 'afpovertcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDFP
    #: - [UDP] IDFP
    idfp: 'AppType' = 549, 'idfp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] new-who [n/a]
    #: - [UDP] new-who [n/a]
    new_rwho: 'AppType' = 550, 'new-rwho', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cybercash [:rfc:`1898`]
    #: - [UDP] cybercash [:rfc:`1898`]
    cybercash: 'AppType' = 551, 'cybercash', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DeviceShare
    #: - [UDP] DeviceShare
    devshr_nts: 'AppType' = 552, 'devshr-nts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pirp
    #: - [UDP] pirp
    pirp: 'AppType' = 553, 'pirp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real Time Streaming Protocol (RTSP)
    #: - [UDP] Real Time Streaming Protocol (RTSP)
    rtsp: 'AppType' = 554, 'rtsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    dsf: 'AppType' = 555, 'dsf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rfs server
    #: - [UDP] rfs server
    remotefs: 'AppType' = 556, 'remotefs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] openvms-sysipc
    #: - [UDP] openvms-sysipc
    openvms_sysipc: 'AppType' = 557, 'openvms-sysipc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDNSKMP
    #: - [UDP] SDNSKMP
    sdnskmp: 'AppType' = 558, 'sdnskmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TEEDTAP
    #: - [UDP] TEEDTAP
    teedtap: 'AppType' = 559, 'teedtap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rmonitord
    #: - [UDP] rmonitord
    rmonitor: 'AppType' = 560, 'rmonitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    monitor: 'AppType' = 561, 'monitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] chcmd
    #: - [UDP] chcmd
    chshell: 'AppType' = 562, 'chshell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nntp protocol over TLS/SSL (was snntp) [:rfc:`4642`]
    #: - [UDP] nntp protocol over TLS/SSL (was snntp) [:rfc:`4642`]
    nntps: 'AppType' = 563, 'nntps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] plan 9 file service
    #: - [UDP] plan 9 file service
    AppType_9pfs: 'AppType' = 564, '9pfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] whoami
    #: - [UDP] whoami
    whoami: 'AppType' = 565, 'whoami', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] streettalk
    #: - [UDP] streettalk
    streettalk: 'AppType' = 566, 'streettalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] banyan-rpc
    #: - [UDP] banyan-rpc
    banyan_rpc: 'AppType' = 567, 'banyan-rpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] microsoft shuttle
    #: - [UDP] microsoft shuttle
    ms_shuttle: 'AppType' = 568, 'ms-shuttle', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] microsoft rome
    #: - [UDP] microsoft rome
    ms_rome: 'AppType' = 569, 'ms-rome', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] demon
    #: - [UDP] demon
    meter_570: 'AppType' = 570, 'meter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] udemon
    #: - [UDP] udemon
    meter_571: 'AppType' = 571, 'meter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sonar
    #: - [UDP] sonar
    sonar: 'AppType' = 572, 'sonar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] banyan-vip
    #: - [UDP] banyan-vip
    banyan_vip: 'AppType' = 573, 'banyan-vip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FTP Software Agent System
    #: - [UDP] FTP Software Agent System
    ftp_agent: 'AppType' = 574, 'ftp-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VEMMI
    #: - [UDP] VEMMI
    vemmi: 'AppType' = 575, 'vemmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ipcd
    #: - [UDP] ipcd
    ipcd: 'AppType' = 576, 'ipcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vnas
    #: - [UDP] vnas
    vnas: 'AppType' = 577, 'vnas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ipdd
    #: - [UDP] ipdd
    ipdd: 'AppType' = 578, 'ipdd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] decbsrv
    #: - [UDP] decbsrv
    decbsrv: 'AppType' = 579, 'decbsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNTP HEARTBEAT
    #: - [UDP] SNTP HEARTBEAT
    sntp_heartbeat: 'AppType' = 580, 'sntp-heartbeat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bundle Discovery Protocol
    #: - [UDP] Bundle Discovery Protocol
    bdp: 'AppType' = 581, 'bdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCC Security
    #: - [UDP] SCC Security
    scc_security: 'AppType' = 582, 'scc-security', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Philips Video-Conferencing
    #: - [UDP] Philips Video-Conferencing
    philips_vc: 'AppType' = 583, 'philips-vc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Key Server
    #: - [UDP] Key Server
    keyserver: 'AppType' = 584, 'keyserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Password Change
    #: - [UDP] Password Change
    password_chg: 'AppType' = 586, 'password-chg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Submission [:rfc:`6409`]
    #: - [UDP] Message Submission [:rfc:`6409`]
    submission: 'AppType' = 587, 'submission', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAL
    #: - [UDP] CAL
    cal: 'AppType' = 588, 'cal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EyeLink
    #: - [UDP] EyeLink
    eyelink: 'AppType' = 589, 'eyelink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNS CML
    #: - [UDP] TNS CML
    tns_cml: 'AppType' = 590, 'tns-cml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eudora Set
    #: - [UDP] Eudora Set
    eudora_set: 'AppType' = 592, 'eudora-set', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTTP RPC Ep Map
    #: - [UDP] HTTP RPC Ep Map
    http_rpc_epmap: 'AppType' = 593, 'http-rpc-epmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TPIP
    #: - [UDP] TPIP
    tpip: 'AppType' = 594, 'tpip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAB Protocol
    #: - [UDP] CAB Protocol
    cab_protocol: 'AppType' = 595, 'cab-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMSD
    #: - [UDP] SMSD
    smsd: 'AppType' = 596, 'smsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PTC Name Service
    #: - [UDP] PTC Name Service
    ptcnameservice: 'AppType' = 597, 'ptcnameservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO Web Server Manager 3
    #: - [UDP] SCO Web Server Manager 3
    sco_websrvrmg3: 'AppType' = 598, 'sco-websrvrmg3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aeolon Core Protocol
    #: - [UDP] Aeolon Core Protocol
    acp: 'AppType' = 599, 'acp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun IPC server
    #: - [UDP] Sun IPC server
    ipcserver: 'AppType' = 600, 'ipcserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reliable Syslog Service [:rfc:`3195`]
    #: - [UDP] Reliable Syslog Service [:rfc:`3195`]
    syslog_conn: 'AppType' = 601, 'syslog-conn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XML-RPC over BEEP [:rfc:`3529`]
    #: - [UDP] XML-RPC over BEEP [:rfc:`3529`]
    xmlrpc_beep: 'AppType' = 602, 'xmlrpc-beep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDXP [:rfc:`4767`]
    #: - [UDP] IDXP [:rfc:`4767`]
    idxp: 'AppType' = 603, 'idxp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TUNNEL [:rfc:`3620`]
    #: - [UDP] TUNNEL [:rfc:`3620`]
    tunnel: 'AppType' = 604, 'tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SOAP over BEEP [:rfc:`4227`]
    #: - [UDP] SOAP over BEEP [:rfc:`4227`]
    soap_beep: 'AppType' = 605, 'soap-beep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cray Unified Resource Manager
    #: - [UDP] Cray Unified Resource Manager
    urm: 'AppType' = 606, 'urm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nqs
    #: - [UDP] nqs
    nqs: 'AppType' = 607, 'nqs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sender-Initiated/Unsolicited File Transfer
    #: - [UDP] Sender-Initiated/Unsolicited File Transfer
    sift_uft: 'AppType' = 608, 'sift-uft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] npmp-trap
    #: - [UDP] npmp-trap
    npmp_trap: 'AppType' = 609, 'npmp-trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] npmp-local
    #: - [UDP] npmp-local
    npmp_local: 'AppType' = 610, 'npmp-local', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] npmp-gui
    #: - [UDP] npmp-gui
    npmp_gui: 'AppType' = 611, 'npmp-gui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HMMP Indication
    #: - [UDP] HMMP Indication
    hmmp_ind: 'AppType' = 612, 'hmmp-ind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HMMP Operation
    #: - [UDP] HMMP Operation
    hmmp_op: 'AppType' = 613, 'hmmp-op', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSLshell
    #: - [UDP] SSLshell
    sshell: 'AppType' = 614, 'sshell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Internet Configuration Manager
    #: - [UDP] Internet Configuration Manager
    sco_inetmgr: 'AppType' = 615, 'sco-inetmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO System Administration Server
    #: - [UDP] SCO System Administration Server
    sco_sysmgr: 'AppType' = 616, 'sco-sysmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO Desktop Administration Server
    #: - [UDP] SCO Desktop Administration Server
    sco_dtmgr: 'AppType' = 617, 'sco-dtmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DEI-ICDA
    #: - [UDP] DEI-ICDA
    dei_icda: 'AppType' = 618, 'dei-icda', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq EVM
    #: - [UDP] Compaq EVM
    compaq_evm: 'AppType' = 619, 'compaq-evm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO WebServer Manager
    #: - [UDP] SCO WebServer Manager
    sco_websrvrmgr: 'AppType' = 620, 'sco-websrvrmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESCP
    #: - [UDP] ESCP
    escp_ip: 'AppType' = 621, 'escp-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Collaborator
    #: - [UDP] Collaborator
    collaborator: 'AppType' = 622, 'collaborator', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DMTF out-of-band web services management protocol
    oob_ws_http: 'AppType' = 623, 'oob-ws-http', TransportProtocol.get('tcp')

    #: [UDP] ASF Remote Management and Control Protocol
    asf_rmcp: 'AppType' = 623, 'asf-rmcp', TransportProtocol.get('udp')

    #: - [TCP] Crypto Admin
    #: - [UDP] Crypto Admin
    cryptoadmin: 'AppType' = 624, 'cryptoadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DEC DLM IANA assigned this well-formed service name as a replacement
    #:   for "dec_dlm".
    #: - [TCP] DEC DLM
    #: - [UDP] DEC DLM IANA assigned this well-formed service name as a replacement
    #:   for "dec_dlm".
    #: - [UDP] DEC DLM
    dec_dlm: 'AppType' = 625, 'dec-dlm', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ASIA
    #: - [UDP] ASIA
    asia: 'AppType' = 626, 'asia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PassGo Tivoli
    #: - [UDP] PassGo Tivoli
    passgo_tivoli: 'AppType' = 627, 'passgo-tivoli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QMQP
    #: - [UDP] QMQP
    qmqp: 'AppType' = 628, 'qmqp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com AMP3
    #: - [UDP] 3Com AMP3
    AppType_3com_amp3: 'AppType' = 629, '3com-amp3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPP (Internet Printing Protocol) [:rfc:`8011`]
    #: - [UDP] IPP (Internet Printing Protocol) [:rfc:`8011`]
    ipp: 'AppType' = 631, 'ipp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Internet Printing Protocol over HTTPS [:rfc:`8011`]
    ipps: 'AppType' = 631, 'ipps', TransportProtocol.get('tcp')

    #: - [TCP] bmpp
    #: - [UDP] bmpp
    bmpp: 'AppType' = 632, 'bmpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Status update (Sterling Software)
    #: - [UDP] Service Status update (Sterling Software)
    servstat: 'AppType' = 633, 'servstat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ginad
    #: - [UDP] ginad
    ginad: 'AppType' = 634, 'ginad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RLZ DBase
    #: - [UDP] RLZ DBase
    rlzdbase: 'AppType' = 635, 'rlzdbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ldap protocol over TLS/SSL (was sldap)
    #: - [UDP] ldap protocol over TLS/SSL (was sldap)
    ldaps: 'AppType' = 636, 'ldaps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] lanserver
    #: - [UDP] lanserver
    lanserver: 'AppType' = 637, 'lanserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mcns-sec
    #: - [UDP] mcns-sec
    mcns_sec: 'AppType' = 638, 'mcns-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSDP
    #: - [UDP] MSDP
    msdp: 'AppType' = 639, 'msdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] entrust-sps
    #: - [UDP] entrust-sps
    entrust_sps: 'AppType' = 640, 'entrust-sps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] repcmd
    #: - [UDP] repcmd
    repcmd: 'AppType' = 641, 'repcmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESRO-EMSDP V1.3
    #: - [UDP] ESRO-EMSDP V1.3
    esro_emsdp: 'AppType' = 642, 'esro-emsdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SANity
    #: - [UDP] SANity
    sanity: 'AppType' = 643, 'sanity', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dwr
    #: - [UDP] dwr
    dwr: 'AppType' = 644, 'dwr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PSSC
    #: - [UDP] PSSC
    pssc: 'AppType' = 645, 'pssc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LDP
    #: - [UDP] LDP
    ldp: 'AppType' = 646, 'ldp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DHCP Failover [:rfc:`8156`]
    #: - [UDP] DHCP Failover
    dhcp_failover: 'AppType' = 647, 'dhcp-failover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Registry Registrar Protocol (RRP)
    #: - [UDP] Registry Registrar Protocol (RRP)
    rrp: 'AppType' = 648, 'rrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadview-3d - streaming 3d models over the internet
    #: - [UDP] Cadview-3d - streaming 3d models over the internet
    cadview_3d: 'AppType' = 649, 'cadview-3d', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OBEX
    #: - [UDP] OBEX
    obex: 'AppType' = 650, 'obex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IEEE MMS
    #: - [UDP] IEEE MMS
    ieee_mms: 'AppType' = 651, 'ieee-mms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HELLO_PORT
    #: - [UDP] HELLO_PORT
    hello_port: 'AppType' = 652, 'hello-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RepCmd
    #: - [UDP] RepCmd
    repscmd: 'AppType' = 653, 'repscmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AODV
    #: - [UDP] AODV
    aodv: 'AppType' = 654, 'aodv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TINC
    #: - [UDP] TINC
    tinc: 'AppType' = 655, 'tinc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPMP
    #: - [UDP] SPMP
    spmp: 'AppType' = 656, 'spmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMC
    #: - [UDP] RMC
    rmc: 'AppType' = 657, 'rmc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TenFold
    #: - [UDP] TenFold
    tenfold: 'AppType' = 658, 'tenfold', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MacOS Server Admin
    #: - [UDP] MacOS Server Admin
    mac_srvr_admin: 'AppType' = 660, 'mac-srvr-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HAP
    #: - [UDP] HAP
    hap: 'AppType' = 661, 'hap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PFTP
    #: - [UDP] PFTP
    pftp: 'AppType' = 662, 'pftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PureNoise
    #: - [UDP] PureNoise
    purenoise: 'AppType' = 663, 'purenoise', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DMTF out-of-band secure web services management protocol
    oob_ws_https: 'AppType' = 664, 'oob-ws-https', TransportProtocol.get('tcp')

    #: [UDP] ASF Secure Remote Management and Control Protocol
    asf_secure_rmcp: 'AppType' = 664, 'asf-secure-rmcp', TransportProtocol.get('udp')

    #: - [TCP] Sun DR
    #: - [UDP] Sun DR
    sun_dr: 'AppType' = 665, 'sun-dr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    mdqs: 'AppType' = 666, 'mdqs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] doom Id Software
    #: - [UDP] doom Id Software
    doom: 'AppType' = 666, 'doom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] campaign contribution disclosures - SDR Technologies
    #: - [UDP] campaign contribution disclosures - SDR Technologies
    disclose: 'AppType' = 667, 'disclose', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MeComm
    #: - [UDP] MeComm
    mecomm: 'AppType' = 668, 'mecomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MeRegister
    #: - [UDP] MeRegister
    meregister: 'AppType' = 669, 'meregister', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VACDSM-SWS
    #: - [UDP] VACDSM-SWS
    vacdsm_sws: 'AppType' = 670, 'vacdsm-sws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VACDSM-APP
    #: - [UDP] VACDSM-APP
    vacdsm_app: 'AppType' = 671, 'vacdsm-app', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VPPS-QUA
    #: - [UDP] VPPS-QUA
    vpps_qua: 'AppType' = 672, 'vpps-qua', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CIMPLEX
    #: - [UDP] CIMPLEX
    cimplex: 'AppType' = 673, 'cimplex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACAP
    #: - [UDP] ACAP
    acap: 'AppType' = 674, 'acap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DCTP
    #: - [UDP] DCTP
    dctp: 'AppType' = 675, 'dctp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VPPS Via
    #: - [UDP] VPPS Via
    vpps_via: 'AppType' = 676, 'vpps-via', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Presence Protocol
    #: - [UDP] Virtual Presence Protocol
    vpp: 'AppType' = 677, 'vpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GNU Generation Foundation NCP
    #: - [UDP] GNU Generation Foundation NCP
    ggf_ncp: 'AppType' = 678, 'ggf-ncp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MRM
    #: - [UDP] MRM
    mrm: 'AppType' = 679, 'mrm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] entrust-aaas
    #: - [UDP] entrust-aaas
    entrust_aaas: 'AppType' = 680, 'entrust-aaas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] entrust-aams
    #: - [UDP] entrust-aams
    entrust_aams: 'AppType' = 681, 'entrust-aams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XFR
    #: - [UDP] XFR
    xfr: 'AppType' = 682, 'xfr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CORBA IIOP
    #: - [UDP] CORBA IIOP
    corba_iiop: 'AppType' = 683, 'corba-iiop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CORBA IIOP SSL
    #: - [UDP] CORBA IIOP SSL
    corba_iiop_ssl: 'AppType' = 684, 'corba-iiop-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MDC Port Mapper
    #: - [UDP] MDC Port Mapper
    mdc_portmapper: 'AppType' = 685, 'mdc-portmapper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hardware Control Protocol Wismar
    #: - [UDP] Hardware Control Protocol Wismar
    hcp_wismar: 'AppType' = 686, 'hcp-wismar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] asipregistry
    #: - [UDP] asipregistry
    asipregistry: 'AppType' = 687, 'asipregistry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ApplianceWare managment protocol
    #: - [UDP] ApplianceWare managment protocol
    realm_rusd: 'AppType' = 688, 'realm-rusd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMAP
    #: - [UDP] NMAP
    nmap: 'AppType' = 689, 'nmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Velneo Application Transfer Protocol
    #: - [UDP] Velneo Application Transfer Protocol
    vatp: 'AppType' = 690, 'vatp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Exchange Routing
    #: - [UDP] MS Exchange Routing
    msexch_routing: 'AppType' = 691, 'msexch-routing', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hyperwave-ISP
    #: - [UDP] Hyperwave-ISP
    hyperwave_isp: 'AppType' = 692, 'hyperwave-isp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] almanid Connection Endpoint
    #: - [UDP] almanid Connection Endpoint
    connendp: 'AppType' = 693, 'connendp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ha-cluster
    #: - [UDP] ha-cluster
    ha_cluster: 'AppType' = 694, 'ha-cluster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IEEE-MMS-SSL
    #: - [UDP] IEEE-MMS-SSL
    ieee_mms_ssl: 'AppType' = 695, 'ieee-mms-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RUSHD
    #: - [UDP] RUSHD
    rushd: 'AppType' = 696, 'rushd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UUIDGEN
    #: - [UDP] UUIDGEN
    uuidgen: 'AppType' = 697, 'uuidgen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OLSR
    #: - [UDP] OLSR
    olsr: 'AppType' = 698, 'olsr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Access Network
    #: - [UDP] Access Network
    accessnetwork: 'AppType' = 699, 'accessnetwork', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Link Management Protocol (LMP) [:rfc:`4204`]
    #: - [UDP] Link Management Protocol (LMP) [:rfc:`4204`]
    lmp: 'AppType' = 701, 'lmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRIS over BEEP [:rfc:`3983`]
    #: - [UDP] IRIS over BEEP [:rfc:`3983`]
    iris_beep: 'AppType' = 702, 'iris-beep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_703: 'AppType' = 703, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] errlog copy/server daemon
    #: - [UDP] errlog copy/server daemon
    elcsd: 'AppType' = 704, 'elcsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AgentX
    #: - [UDP] AgentX
    agentx: 'AppType' = 705, 'agentx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SILC
    #: - [UDP] SILC
    silc: 'AppType' = 706, 'silc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Borland DSJ
    #: - [UDP] Borland DSJ
    borland_dsj: 'AppType' = 707, 'borland-dsj', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_708: 'AppType' = 708, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Entrust Key Management Service Handler
    #: - [UDP] Entrust Key Management Service Handler
    entrust_kmsh: 'AppType' = 709, 'entrust-kmsh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Entrust Administration Service Handler
    #: - [UDP] Entrust Administration Service Handler
    entrust_ash: 'AppType' = 710, 'entrust-ash', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco TDP
    #: - [UDP] Cisco TDP
    cisco_tdp: 'AppType' = 711, 'cisco-tdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TBRPF [:rfc:`3684`]
    #: - [UDP] TBRPF [:rfc:`3684`]
    tbrpf: 'AppType' = 712, 'tbrpf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRIS over XPC
    #: - [UDP] IRIS over XPC
    iris_xpc: 'AppType' = 713, 'iris-xpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRIS over XPCS [:rfc:`4992`]
    #: - [UDP] IRIS over XPCS [:rfc:`4992`]
    iris_xpcs: 'AppType' = 714, 'iris-xpcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRIS-LWZ [:rfc:`4993`]
    #: - [UDP] IRIS-LWZ [:rfc:`4993`]
    iris_lwz: 'AppType' = 715, 'iris-lwz', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] PANA Messages [:rfc:`5191`]
    pana: 'AppType' = 716, 'pana', TransportProtocol.get('udp')

    #: - [TCP] IBM NetView DM/6000 Server/Client
    #: - [UDP] IBM NetView DM/6000 Server/Client
    netviewdm1: 'AppType' = 729, 'netviewdm1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM NetView DM/6000 send/tcp
    #: - [UDP] IBM NetView DM/6000 send/tcp
    netviewdm2: 'AppType' = 730, 'netviewdm2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM NetView DM/6000 receive/tcp
    #: - [UDP] IBM NetView DM/6000 receive/tcp
    netviewdm3: 'AppType' = 731, 'netviewdm3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netGW
    #: - [UDP] netGW
    netgw: 'AppType' = 741, 'netgw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network based Rev. Cont. Sys.
    #: - [UDP] Network based Rev. Cont. Sys.
    netrcs: 'AppType' = 742, 'netrcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_743: 'AppType' = 743, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Flexible License Manager
    #: - [UDP] Flexible License Manager
    flexlm: 'AppType' = 744, 'flexlm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Device Control
    #: - [UDP] Fujitsu Device Control
    fujitsu_dev: 'AppType' = 747, 'fujitsu-dev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Russell Info Sci Calendar Manager
    #: - [UDP] Russell Info Sci Calendar Manager
    ris_cm: 'AppType' = 748, 'ris-cm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] kerberos administration
    #: - [UDP] kerberos administration
    kerberos_adm: 'AppType' = 749, 'kerberos-adm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    rfile: 'AppType' = 750, 'rfile', TransportProtocol.get('tcp')

    #: [UDP]
    loadav: 'AppType' = 750, 'loadav', TransportProtocol.get('udp')

    #: [UDP] kerberos version iv
    kerberos_iv: 'AppType' = 750, 'kerberos-iv', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    pump: 'AppType' = 751, 'pump', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    qrh: 'AppType' = 752, 'qrh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    rrh: 'AppType' = 753, 'rrh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] send
    #: - [UDP] send
    tell: 'AppType' = 754, 'tell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    nlogin: 'AppType' = 758, 'nlogin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    con: 'AppType' = 759, 'con', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    ns: 'AppType' = 760, 'ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    rxe: 'AppType' = 761, 'rxe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    quotad: 'AppType' = 762, 'quotad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    cycleserv: 'AppType' = 763, 'cycleserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    omserv: 'AppType' = 764, 'omserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    webster: 'AppType' = 765, 'webster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_766: 'AppType' = 766, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] phone
    #: - [UDP] phone
    phonebook: 'AppType' = 767, 'phonebook', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_768: 'AppType' = 768, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP]
    #: - [UDP]
    vid: 'AppType' = 769, 'vid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    cadlock: 'AppType' = 770, 'cadlock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    rtip: 'AppType' = 771, 'rtip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    cycleserv2: 'AppType' = 772, 'cycleserv2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    submit: 'AppType' = 773, 'submit', TransportProtocol.get('tcp')

    #: [UDP]
    notify: 'AppType' = 773, 'notify', TransportProtocol.get('udp')

    #: [TCP]
    rpasswd: 'AppType' = 774, 'rpasswd', TransportProtocol.get('tcp')

    #: - [UDP] IANA assigned this well-formed service name as a replacement for
    #:   "acmaint_dbd".
    #: - [UDP]
    acmaint_dbd: 'AppType' = 774, 'acmaint-dbd', TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [TCP]
    entomb: 'AppType' = 775, 'entomb', TransportProtocol.get('tcp')

    #: - [UDP] IANA assigned this well-formed service name as a replacement for
    #:   "acmaint_transd".
    #: - [UDP]
    acmaint_transd: 'AppType' = 775, 'acmaint-transd', TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    wpages: 'AppType' = 776, 'wpages', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multiling HTTP
    #: - [UDP] Multiling HTTP
    multiling_http: 'AppType' = 777, 'multiling-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    wpgs: 'AppType' = 780, 'wpgs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_786: 'AppType' = 786, 'unassigned', TransportProtocol.get('undefined')

    #: [N/A] Unassigned
    unassigned_787: 'AppType' = 787, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] IANA assigned this well-formed service name as a replacement for
    #:   "mdbs_daemon".
    #: - [TCP]
    #: - [UDP] IANA assigned this well-formed service name as a replacement for
    #:   "mdbs_daemon".
    #: - [UDP]
    mdbs_daemon: 'AppType' = 800, 'mdbs-daemon', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    device: 'AppType' = 801, 'device', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Modbus Application Protocol Secure
    #: - [UDP] Modbus Application Protocol Secure
    mbap_s: 'AppType' = 802, 'mbap-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FCP
    #: - [UDP] FCP Datagram
    fcp_udp: 'AppType' = 810, 'fcp-udp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] itm-mcell-s
    #: - [UDP] itm-mcell-s
    itm_mcell_s: 'AppType' = 828, 'itm-mcell-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PKIX-3 CA/RA
    #: - [UDP] PKIX-3 CA/RA
    pkix_3_ca_ra: 'AppType' = 829, 'pkix-3-ca-ra', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETCONF over SSH [:rfc:`6242`]
    #: - [UDP] NETCONF over SSH [:rfc:`6242`]
    netconf_ssh: 'AppType' = 830, 'netconf-ssh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETCONF over BEEP [:rfc:`4744`]
    #: - [UDP] NETCONF over BEEP [:rfc:`4744`]
    netconf_beep: 'AppType' = 831, 'netconf-beep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETCONF for SOAP over HTTPS [:rfc:`4743`]
    #: - [UDP] NETCONF for SOAP over HTTPS [:rfc:`4743`]
    netconfsoaphttp: 'AppType' = 832, 'netconfsoaphttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETCONF for SOAP over BEEP [:rfc:`4743`]
    #: - [UDP] NETCONF for SOAP over BEEP [:rfc:`4743`]
    netconfsoapbeep: 'AppType' = 833, 'netconfsoapbeep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dhcp-failover 2
    #: - [UDP] dhcp-failover 2
    dhcp_failover2: 'AppType' = 847, 'dhcp-failover2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GDOI [:rfc:`3547`]
    #: - [UDP] GDOI [:rfc:`3547`]
    gdoi: 'AppType' = 848, 'gdoi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNS query-response protocol run over TLS [:rfc:`7858`]
    #: - [UDP] DNS query-response protocol run over DTLS or QUIC
    #:   [:rfc:`7858`][:rfc:`8094`][:rfc:`9250`]
    domain_s: 'AppType' = 853, 'domain-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic Link Exchange Protocol (DLEP) [:rfc:`8175`]
    #: - [UDP] Dynamic Link Exchange Protocol (DLEP) [:rfc:`8175`]
    dlep: 'AppType' = 854, 'dlep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iSCSI [:rfc:`7143`]
    #: - [UDP] iSCSI [:rfc:`7143`]
    iscsi: 'AppType' = 860, 'iscsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] OWAMP-Control [:rfc:`4656`]
    owamp_control: 'AppType' = 861, 'owamp-control', TransportProtocol.get('tcp')

    #: [UDP] OWAMP-Test [:rfc:`8545`]
    owamp_test: 'AppType' = 861, 'owamp-test', TransportProtocol.get('udp')

    #: [TCP] TWAMP-Control [:rfc:`5357`]
    twamp_control: 'AppType' = 862, 'twamp-control', TransportProtocol.get('tcp')

    #: [UDP] TWAMP-Test Receiver Port [:rfc:`8545`]
    twamp_test: 'AppType' = 862, 'twamp-test', TransportProtocol.get('udp')

    #: - [TCP] rsync
    #: - [UDP] rsync
    rsync: 'AppType' = 873, 'rsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICL coNETion locate server
    #: - [UDP] ICL coNETion locate server
    iclcnet_locate: 'AppType' = 886, 'iclcnet-locate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICL coNETion server info IANA assigned this well-formed service name
    #:   as a replacement for "iclcnet_svinfo".
    #: - [TCP] ICL coNETion server info
    #: - [UDP] ICL coNETion server info IANA assigned this well-formed service name
    #:   as a replacement for "iclcnet_svinfo".
    #: - [UDP] ICL coNETion server info
    iclcnet_svinfo: 'AppType' = 887, 'iclcnet-svinfo', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] AccessBuilder
    #: - [UDP] AccessBuilder
    accessbuilder: 'AppType' = 888, 'accessbuilder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CD Database Protocol
    cddbp: 'AppType' = 888, 'cddbp', TransportProtocol.get('tcp')

    #: - [TCP] OMG Initial Refs
    #: - [UDP] OMG Initial Refs
    omginitialrefs: 'AppType' = 900, 'omginitialrefs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMPNAMERES
    #: - [UDP] SMPNAMERES
    smpnameres: 'AppType' = 901, 'smpnameres', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] self documenting Telnet Door
    #: - [UDP] self documenting Door: send 0x00 for info
    ideafarm_door: 'AppType' = 902, 'ideafarm-door', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] self documenting Telnet Panic Door
    #: - [UDP] self documenting Panic Door: send 0x00 for info
    ideafarm_panic: 'AppType' = 903, 'ideafarm-panic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kerberized Internet Negotiation of Keys (KINK) [:rfc:`4430`]
    #: - [UDP] Kerberized Internet Negotiation of Keys (KINK) [:rfc:`4430`]
    kink: 'AppType' = 910, 'kink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xact-backup
    #: - [UDP] xact-backup
    xact_backup: 'AppType' = 911, 'xact-backup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APEX relay-relay service
    #: - [UDP] APEX relay-relay service
    apex_mesh: 'AppType' = 912, 'apex-mesh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APEX endpoint-relay service [:rfc:`3340`]
    #: - [UDP] APEX endpoint-relay service [:rfc:`3340`]
    apex_edge: 'AppType' = 913, 'apex-edge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_914: 'AppType' = 914, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Routing in Fat Trees Link Information Element [RFC-ietf-rift-rift-24]
    rift_lies: 'AppType' = 914, 'rift-lies', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_915: 'AppType' = 915, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Routing in Fat Trees Topology Information Element [RFC-ietf-rift-
    #: rift-24]
    rift_ties: 'AppType' = 915, 'rift-ties', TransportProtocol.get('udp')

    #: [TCP] BIND9 remote name daemon controller
    rndc: 'AppType' = 953, 'rndc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_953: 'AppType' = 953, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ftp protocol, data, over TLS/SSL
    #: - [UDP] ftp protocol, data, over TLS/SSL
    ftps_data: 'AppType' = 989, 'ftps-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ftp protocol, control, over TLS/SSL
    #: - [UDP] ftp protocol, control, over TLS/SSL
    ftps: 'AppType' = 990, 'ftps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netnews Administration System [:rfc:`4707`]
    #: - [UDP] Netnews Administration System [:rfc:`4707`]
    nas: 'AppType' = 991, 'nas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] telnet protocol over TLS/SSL
    #: - [UDP] telnet protocol over TLS/SSL
    telnets: 'AppType' = 992, 'telnets', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] IMAP over TLS protocol [:rfc:`3501`][:rfc:`8314`][:rfc:`9051`]
    imaps: 'AppType' = 993, 'imaps', TransportProtocol.get('tcp')

    #: [UDP] Reserved [:rfc:`9051`]
    reserved_993: 'AppType' = 993, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_994: 'AppType' = 994, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] POP3 over TLS protocol [:rfc:`8314`]
    #: - [UDP] pop3 protocol over TLS/SSL (was spop3)
    pop3s: 'AppType' = 995, 'pop3s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vsinet
    #: - [UDP] vsinet
    vsinet: 'AppType' = 996, 'vsinet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    maitrd: 'AppType' = 997, 'maitrd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    busboy: 'AppType' = 998, 'busboy', TransportProtocol.get('tcp')

    #: [UDP]
    puparp: 'AppType' = 998, 'puparp', TransportProtocol.get('udp')

    #: [TCP]
    garcon: 'AppType' = 999, 'garcon', TransportProtocol.get('tcp')

    #: [UDP] Applix ac
    applix: 'AppType' = 999, 'applix', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    puprouter: 'AppType' = 999, 'puprouter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    cadlock2: 'AppType' = 1000, 'cadlock2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] HTTP Web Push [:rfc:`8030`]
    webpush: 'AppType' = 1001, 'webpush', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_1001: 'AppType' = 1001, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Possibly used by Sun Solaris????
    possibly_used_by_sun_solaris: 'AppType' = 1008, 'possibly_used_by_sun_solaris', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_1009: 'AppType' = 1009, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] surf
    #: - [UDP] surf
    surf: 'AppType' = 1010, 'surf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RFC3692-style Experiment 1 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [UDP] RFC3692-style Experiment 1 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [SCTP] RFC3692-style Experiment 1 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [DCCP] RFC3692-style Experiment 1 [1][:rfc:`4727`][:rfc:`6335`]
    exp1: 'AppType' = 1021, 'exp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp') | TransportProtocol.get('dccp')

    #: - [TCP] RFC3692-style Experiment 2 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [UDP] RFC3692-style Experiment 2 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [SCTP] RFC3692-style Experiment 2 [1][:rfc:`4727`][:rfc:`6335`]
    #: - [DCCP] RFC3692-style Experiment 2 [1][:rfc:`4727`][:rfc:`6335`]
    exp2: 'AppType' = 1022, 'exp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp') | TransportProtocol.get('dccp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_1023: 'AppType' = 1023, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_1024: 'AppType' = 1024, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] network blackjack
    #: - [UDP] network blackjack
    blackjack: 'AppType' = 1025, 'blackjack', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Calendar Access Protocol
    #: - [UDP] Calendar Access Protocol
    cap: 'AppType' = 1026, 'cap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] IPv6 Behind NAT44 CPEs [:rfc:`6751`]
    AppType_6a44: 'AppType' = 1027, '6a44', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_1027: 'AppType' = 1027, 'reserved', TransportProtocol.get('tcp')

    #: [N/A] Deprecated
    deprecated: 'AppType' = 1028, 'deprecated', TransportProtocol.get('undefined')

    #: - [TCP] Solid Mux Server
    #: - [UDP] Solid Mux Server
    solid_mux: 'AppType' = 1029, 'solid-mux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Reserved
    reserved_1030: 'AppType' = 1030, 'reserved', TransportProtocol.get('undefined')

    #: [N/A] Reserved
    reserved_1031: 'AppType' = 1031, 'reserved', TransportProtocol.get('undefined')

    #: [N/A] Reserved
    reserved_1032: 'AppType' = 1032, 'reserved', TransportProtocol.get('undefined')

    #: - [TCP] local netinfo port
    #: - [UDP] local netinfo port
    netinfo_local: 'AppType' = 1033, 'netinfo-local', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ActiveSync Notifications
    #: - [UDP] ActiveSync Notifications
    activesync: 'AppType' = 1034, 'activesync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MX-XR RPC
    #: - [UDP] MX-XR RPC
    mxxrlogin: 'AppType' = 1035, 'mxxrlogin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nebula Secure Segment Transfer Protocol
    #: - [UDP] Nebula Secure Segment Transfer Protocol
    nsstp: 'AppType' = 1036, 'nsstp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMS
    #: - [UDP] AMS
    ams: 'AppType' = 1037, 'ams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Tracking Query Protocol [:rfc:`3887`]
    #: - [UDP] Message Tracking Query Protocol [:rfc:`3887`]
    mtqp: 'AppType' = 1038, 'mtqp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Streamlined Blackhole
    #: - [UDP] Streamlined Blackhole
    sbl: 'AppType' = 1039, 'sbl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netarx Netcare
    #: - [UDP] Netarx Netcare
    netarx: 'AppType' = 1040, 'netarx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AK2 Product
    #: - [UDP] AK2 Product
    danf_ak2: 'AppType' = 1041, 'danf-ak2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Subnet Roaming
    #: - [UDP] Subnet Roaming
    afrog: 'AppType' = 1042, 'afrog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BOINC Client Control
    #: - [UDP] BOINC Client Control
    boinc_client: 'AppType' = 1043, 'boinc-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dev Consortium Utility
    #: - [UDP] Dev Consortium Utility
    dcutility: 'AppType' = 1044, 'dcutility', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fingerprint Image Transfer Protocol
    #: - [UDP] Fingerprint Image Transfer Protocol
    fpitp: 'AppType' = 1045, 'fpitp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WebFilter Remote Monitor
    #: - [UDP] WebFilter Remote Monitor
    wfremotertm: 'AppType' = 1046, 'wfremotertm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun's NEO Object Request Broker
    #: - [UDP] Sun's NEO Object Request Broker
    neod1: 'AppType' = 1047, 'neod1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun's NEO Object Request Broker
    #: - [UDP] Sun's NEO Object Request Broker
    neod2: 'AppType' = 1048, 'neod2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tobit David Postman VPMN
    #: - [UDP] Tobit David Postman VPMN
    td_postman: 'AppType' = 1049, 'td-postman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CORBA Management Agent
    #: - [UDP] CORBA Management Agent
    cma: 'AppType' = 1050, 'cma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optima VNET
    #: - [UDP] Optima VNET
    optima_vnet: 'AppType' = 1051, 'optima-vnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic DNS Tools
    #: - [UDP] Dynamic DNS Tools
    ddt: 'AppType' = 1052, 'ddt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Assistant (RA)
    #: - [UDP] Remote Assistant (RA)
    remote_as: 'AppType' = 1053, 'remote-as', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BRVREAD
    #: - [UDP] BRVREAD
    brvread: 'AppType' = 1054, 'brvread', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSYS - License Manager
    #: - [UDP] ANSYS - License Manager
    ansyslmd: 'AppType' = 1055, 'ansyslmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VFO
    #: - [UDP] VFO
    vfo: 'AppType' = 1056, 'vfo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STARTRON
    #: - [UDP] STARTRON
    startron: 'AppType' = 1057, 'startron', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nim
    #: - [UDP] nim
    nim: 'AppType' = 1058, 'nim', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nimreg
    #: - [UDP] nimreg
    nimreg: 'AppType' = 1059, 'nimreg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] POLESTAR
    #: - [UDP] POLESTAR
    polestar: 'AppType' = 1060, 'polestar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KIOSK
    #: - [UDP] KIOSK
    kiosk: 'AppType' = 1061, 'kiosk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Veracity
    #: - [UDP] Veracity
    veracity: 'AppType' = 1062, 'veracity', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KyoceraNetDev
    #: - [UDP] KyoceraNetDev
    kyoceranetdev: 'AppType' = 1063, 'kyoceranetdev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JSTEL
    #: - [UDP] JSTEL
    jstel: 'AppType' = 1064, 'jstel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SYSCOMLAN
    #: - [UDP] SYSCOMLAN
    syscomlan: 'AppType' = 1065, 'syscomlan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FPO-FNS
    #: - [UDP] FPO-FNS
    fpo_fns: 'AppType' = 1066, 'fpo-fns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Installation Bootstrap Proto. Serv. IANA assigned this well-formed
    #:   service name as a replacement for "instl_boots".
    #: - [TCP] Installation Bootstrap Proto. Serv.
    #: - [UDP] Installation Bootstrap Proto. Serv. IANA assigned this well-formed
    #:   service name as a replacement for "instl_boots".
    #: - [UDP] Installation Bootstrap Proto. Serv.
    instl_boots: 'AppType' = 1067, 'instl-boots', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Installation Bootstrap Proto. Cli. IANA assigned this well-formed
    #:   service name as a replacement for "instl_bootc".
    #: - [TCP] Installation Bootstrap Proto. Cli.
    #: - [UDP] Installation Bootstrap Proto. Cli. IANA assigned this well-formed
    #:   service name as a replacement for "instl_bootc".
    #: - [UDP] Installation Bootstrap Proto. Cli.
    instl_bootc: 'AppType' = 1068, 'instl-bootc', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] COGNEX-INSIGHT
    #: - [UDP] COGNEX-INSIGHT
    cognex_insight: 'AppType' = 1069, 'cognex-insight', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GMRUpdateSERV
    #: - [UDP] GMRUpdateSERV
    gmrupdateserv: 'AppType' = 1070, 'gmrupdateserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BSQUARE-VOIP
    #: - [UDP] BSQUARE-VOIP
    bsquare_voip: 'AppType' = 1071, 'bsquare-voip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CARDAX
    #: - [UDP] CARDAX
    cardax: 'AppType' = 1072, 'cardax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bridge Control
    #: - [UDP] Bridge Control
    bridgecontrol: 'AppType' = 1073, 'bridgecontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Warmspot Management Protocol
    #: - [UDP] Warmspot Management Protocol
    warmspotmgmt: 'AppType' = 1074, 'warmspotmgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RDRMSHC
    #: - [UDP] RDRMSHC
    rdrmshc: 'AppType' = 1075, 'rdrmshc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DAB STI-C
    #: - [UDP] DAB STI-C
    dab_sti_c: 'AppType' = 1076, 'dab-sti-c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IMGames
    #: - [UDP] IMGames
    imgames: 'AppType' = 1077, 'imgames', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avocent Proxy Protocol
    #: - [UDP] Avocent Proxy Protocol
    avocent_proxy: 'AppType' = 1078, 'avocent-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASPROVATalk
    #: - [UDP] ASPROVATalk
    asprovatalk: 'AppType' = 1079, 'asprovatalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Socks
    #: - [UDP] Socks
    socks: 'AppType' = 1080, 'socks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PVUNIWIEN
    #: - [UDP] PVUNIWIEN
    pvuniwien: 'AppType' = 1081, 'pvuniwien', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMT-ESD-PROT
    #: - [UDP] AMT-ESD-PROT
    amt_esd_prot: 'AppType' = 1082, 'amt-esd-prot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anasoft License Manager
    #: - [UDP] Anasoft License Manager
    ansoft_lm_1: 'AppType' = 1083, 'ansoft-lm-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anasoft License Manager
    #: - [UDP] Anasoft License Manager
    ansoft_lm_2: 'AppType' = 1084, 'ansoft-lm-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Web Objects
    #: - [UDP] Web Objects
    webobjects: 'AppType' = 1085, 'webobjects', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPL Scrambler Logging
    #: - [UDP] CPL Scrambler Logging
    cplscrambler_lg: 'AppType' = 1086, 'cplscrambler-lg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPL Scrambler Internal
    #: - [UDP] CPL Scrambler Internal
    cplscrambler_in: 'AppType' = 1087, 'cplscrambler-in', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPL Scrambler Alarm Log
    #: - [UDP] CPL Scrambler Alarm Log
    cplscrambler_al: 'AppType' = 1088, 'cplscrambler-al', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FF Annunciation
    #: - [UDP] FF Annunciation
    ff_annunc: 'AppType' = 1089, 'ff-annunc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FF Fieldbus Message Specification
    #: - [UDP] FF Fieldbus Message Specification
    ff_fms: 'AppType' = 1090, 'ff-fms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FF System Management
    #: - [UDP] FF System Management
    ff_sm: 'AppType' = 1091, 'ff-sm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Open Business Reporting Protocol
    #: - [UDP] Open Business Reporting Protocol
    obrpd: 'AppType' = 1092, 'obrpd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PROOFD
    #: - [UDP] PROOFD
    proofd: 'AppType' = 1093, 'proofd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ROOTD
    #: - [UDP] ROOTD
    rootd: 'AppType' = 1094, 'rootd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NICELink
    #: - [UDP] NICELink
    nicelink: 'AppType' = 1095, 'nicelink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Common Name Resolution Protocol
    #: - [UDP] Common Name Resolution Protocol
    cnrprotocol: 'AppType' = 1096, 'cnrprotocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Cluster Manager
    #: - [UDP] Sun Cluster Manager
    sunclustermgr: 'AppType' = 1097, 'sunclustermgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMI Activation
    #: - [UDP] RMI Activation
    rmiactivation: 'AppType' = 1098, 'rmiactivation', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMI Registry
    #: - [UDP] RMI Registry
    rmiregistry: 'AppType' = 1099, 'rmiregistry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCTP
    #: - [UDP] MCTP
    mctp: 'AppType' = 1100, 'mctp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PT2-DISCOVER
    #: - [UDP] PT2-DISCOVER
    pt2_discover: 'AppType' = 1101, 'pt2-discover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ADOBE SERVER 1
    #: - [UDP] ADOBE SERVER 1
    adobeserver_1: 'AppType' = 1102, 'adobeserver-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ADOBE SERVER 2
    #: - [UDP] ADOBE SERVER 2
    adobeserver_2: 'AppType' = 1103, 'adobeserver-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XRL
    #: - [UDP] XRL
    xrl: 'AppType' = 1104, 'xrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FTRANHC
    #: - [UDP] FTRANHC
    ftranhc: 'AppType' = 1105, 'ftranhc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISOIPSIGPORT-1
    #: - [UDP] ISOIPSIGPORT-1
    isoipsigport_1: 'AppType' = 1106, 'isoipsigport-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISOIPSIGPORT-2
    #: - [UDP] ISOIPSIGPORT-2
    isoipsigport_2: 'AppType' = 1107, 'isoipsigport-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ratio-adp
    #: - [UDP] ratio-adp
    ratio_adp: 'AppType' = 1108, 'ratio-adp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Reserved - IANA
    reserved_iana: 'AppType' = 1109, 'reserved_iana', TransportProtocol.get('undefined')

    #: [TCP] Start web admin server
    webadmstart: 'AppType' = 1110, 'webadmstart', TransportProtocol.get('tcp')

    #: [UDP] Client status info
    nfsd_keepalive: 'AppType' = 1110, 'nfsd-keepalive', TransportProtocol.get('udp')

    #: - [TCP] LM Social Server
    #: - [UDP] LM Social Server
    lmsocialserver: 'AppType' = 1111, 'lmsocialserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intelligent Communication Protocol
    #: - [UDP] Intelligent Communication Protocol
    icp: 'AppType' = 1112, 'icp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Licklider Transmission Protocol [:rfc:`5326`]
    #: - [UDP] Licklider Transmission Protocol [:rfc:`5326`][:rfc:`7122`]
    #: - [DCCP] Licklider Transmission Protocol [:rfc:`7122`]
    ltp_deepspace: 'AppType' = 1113, 'ltp-deepspace', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('dccp')

    #: - [TCP] Mini SQL
    #: - [UDP] Mini SQL
    mini_sql: 'AppType' = 1114, 'mini-sql', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDUS Transfer
    #: - [UDP] ARDUS Transfer
    ardus_trns: 'AppType' = 1115, 'ardus-trns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDUS Control
    #: - [UDP] ARDUS Control
    ardus_cntl: 'AppType' = 1116, 'ardus-cntl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDUS Multicast Transfer
    #: - [UDP] ARDUS Multicast Transfer
    ardus_mtrns: 'AppType' = 1117, 'ardus-mtrns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SACRED [:rfc:`3767`]
    #: - [UDP] SACRED [:rfc:`3767`]
    sacred: 'AppType' = 1118, 'sacred', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Battle.net Chat/Game Protocol
    #: - [UDP] Battle.net Chat/Game Protocol
    bnetgame: 'AppType' = 1119, 'bnetgame', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Battle.net File Transfer Protocol
    #: - [UDP] Battle.net File Transfer Protocol
    bnetfile: 'AppType' = 1120, 'bnetfile', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Datalode RMPP
    #: - [UDP] Datalode RMPP
    rmpp: 'AppType' = 1121, 'rmpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] availant-mgr
    #: - [UDP] availant-mgr
    availant_mgr: 'AppType' = 1122, 'availant-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Murray
    #: - [UDP] Murray
    murray: 'AppType' = 1123, 'murray', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP VMM Control
    #: - [UDP] HP VMM Control
    hpvmmcontrol: 'AppType' = 1124, 'hpvmmcontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP VMM Agent
    #: - [UDP] HP VMM Agent
    hpvmmagent: 'AppType' = 1125, 'hpvmmagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP VMM Agent
    #: - [UDP] HP VMM Agent
    hpvmmdata: 'AppType' = 1126, 'hpvmmdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KWDB Remote Communication
    #: - [UDP] KWDB Remote Communication
    kwdb_commn: 'AppType' = 1127, 'kwdb-commn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAPHostControl over SOAP/HTTP
    #: - [UDP] SAPHostControl over SOAP/HTTP
    saphostctrl: 'AppType' = 1128, 'saphostctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAPHostControl over SOAP/HTTPS
    #: - [UDP] SAPHostControl over SOAP/HTTPS
    saphostctrls: 'AppType' = 1129, 'saphostctrls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAC App Service Protocol
    #: - [UDP] CAC App Service Protocol
    casp: 'AppType' = 1130, 'casp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAC App Service Protocol Encripted
    #: - [UDP] CAC App Service Protocol Encripted
    caspssl: 'AppType' = 1131, 'caspssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KVM-via-IP Management Service
    #: - [UDP] KVM-via-IP Management Service
    kvm_via_ip: 'AppType' = 1132, 'kvm-via-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Flow Network
    #: - [UDP] Data Flow Network
    dfn: 'AppType' = 1133, 'dfn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MicroAPL APLX
    #: - [UDP] MicroAPL APLX
    aplx: 'AppType' = 1134, 'aplx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OmniVision Communication Service
    #: - [UDP] OmniVision Communication Service
    omnivision: 'AppType' = 1135, 'omnivision', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HHB Gateway Control
    #: - [UDP] HHB Gateway Control
    hhb_gateway: 'AppType' = 1136, 'hhb-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIM Workgroup Service
    #: - [UDP] TRIM Workgroup Service
    trim: 'AppType' = 1137, 'trim', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] encrypted admin requests IANA assigned this well-formed service name
    #:   as a replacement for "encrypted_admin".
    #: - [TCP] encrypted admin requests
    #: - [UDP] encrypted admin requests IANA assigned this well-formed service name
    #:   as a replacement for "encrypted_admin".
    #: - [UDP] encrypted admin requests
    encrypted_admin: 'AppType' = 1138, 'encrypted-admin', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Virtual Manager
    #: - [UDP] Enterprise Virtual Manager
    evm: 'AppType' = 1139, 'evm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AutoNOC Network Operations Protocol
    #: - [UDP] AutoNOC Network Operations Protocol
    autonoc: 'AppType' = 1140, 'autonoc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] User Message Service
    #: - [UDP] User Message Service
    mxomss: 'AppType' = 1141, 'mxomss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] User Discovery Service
    #: - [UDP] User Discovery Service
    edtools: 'AppType' = 1142, 'edtools', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Infomatryx Exchange
    #: - [UDP] Infomatryx Exchange
    imyx: 'AppType' = 1143, 'imyx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fusion Script
    #: - [UDP] Fusion Script
    fuscript: 'AppType' = 1144, 'fuscript', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] X9 iCue Show Control
    #: - [UDP] X9 iCue Show Control
    x9_icue: 'AppType' = 1145, 'x9-icue', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] audit transfer
    #: - [UDP] audit transfer
    audit_transfer: 'AppType' = 1146, 'audit-transfer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAPIoverLAN
    #: - [UDP] CAPIoverLAN
    capioverlan: 'AppType' = 1147, 'capioverlan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Elfiq Replication Service
    #: - [UDP] Elfiq Replication Service
    elfiq_repl: 'AppType' = 1148, 'elfiq-repl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BlueView Sonar Service
    #: - [UDP] BlueView Sonar Service
    bvtsonar: 'AppType' = 1149, 'bvtsonar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blaze File Server
    #: - [UDP] Blaze File Server
    blaze: 'AppType' = 1150, 'blaze', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unizensus Login Server
    #: - [UDP] Unizensus Login Server
    unizensus: 'AppType' = 1151, 'unizensus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Winpopup LAN Messenger
    #: - [UDP] Winpopup LAN Messenger
    winpoplanmess: 'AppType' = 1152, 'winpoplanmess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSI C12.22 Port [:rfc:`6142`]
    #: - [UDP] ANSI C12.22 Port [:rfc:`6142`]
    c1222_acse: 'AppType' = 1153, 'c1222-acse', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Community Service
    #: - [UDP] Community Service
    resacommunity: 'AppType' = 1154, 'resacommunity', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network File Access
    #: - [UDP] Network File Access
    nfa: 'AppType' = 1155, 'nfa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iasControl OMS
    #: - [UDP] iasControl OMS
    iascontrol_oms: 'AppType' = 1156, 'iascontrol-oms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle iASControl
    #: - [UDP] Oracle iASControl
    iascontrol: 'AppType' = 1157, 'iascontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dbControl OMS
    #: - [UDP] dbControl OMS
    dbcontrol_oms: 'AppType' = 1158, 'dbcontrol-oms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle OMS
    #: - [UDP] Oracle OMS
    oracle_oms: 'AppType' = 1159, 'oracle-oms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DB Lite Mult-User Server
    #: - [UDP] DB Lite Mult-User Server
    olsv: 'AppType' = 1160, 'olsv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Health Polling
    #: - [UDP] Health Polling
    health_polling: 'AppType' = 1161, 'health-polling', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Health Trap
    #: - [UDP] Health Trap
    health_trap: 'AppType' = 1162, 'health-trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SmartDialer Data Protocol
    #: - [UDP] SmartDialer Data Protocol
    sddp: 'AppType' = 1163, 'sddp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSM Proxy Service
    #: - [UDP] QSM Proxy Service
    qsm_proxy: 'AppType' = 1164, 'qsm-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSM GUI Service
    #: - [UDP] QSM GUI Service
    qsm_gui: 'AppType' = 1165, 'qsm-gui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSM RemoteExec
    #: - [UDP] QSM RemoteExec
    qsm_remote: 'AppType' = 1166, 'qsm-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco IP SLAs Control Protocol
    #: - [UDP] Cisco IP SLAs Control Protocol
    #: - [SCTP] Cisco IP SLAs Control Protocol
    cisco_ipsla: 'AppType' = 1167, 'cisco-ipsla', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] VChat Conference Service
    #: - [UDP] VChat Conference Service
    vchat: 'AppType' = 1168, 'vchat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIPWIRE
    #: - [UDP] TRIPWIRE
    tripwire: 'AppType' = 1169, 'tripwire', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AT+C License Manager
    #: - [UDP] AT+C License Manager
    atc_lm: 'AppType' = 1170, 'atc-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AT+C FmiApplicationServer
    #: - [UDP] AT+C FmiApplicationServer
    atc_appserver: 'AppType' = 1171, 'atc-appserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNA Protocol
    #: - [UDP] DNA Protocol
    dnap: 'AppType' = 1172, 'dnap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D-Cinema Request-Response
    #: - [UDP] D-Cinema Request-Response
    d_cinema_rrp: 'AppType' = 1173, 'd-cinema-rrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FlashNet Remote Admin
    #: - [UDP] FlashNet Remote Admin
    fnet_remote_ui: 'AppType' = 1174, 'fnet-remote-ui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dossier Server
    #: - [UDP] Dossier Server
    dossier: 'AppType' = 1175, 'dossier', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Indigo Home Server
    #: - [UDP] Indigo Home Server
    indigo_server: 'AppType' = 1176, 'indigo-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DKMessenger Protocol
    #: - [UDP] DKMessenger Protocol
    dkmessenger: 'AppType' = 1177, 'dkmessenger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SGI Storage Manager
    #: - [UDP] SGI Storage Manager
    sgi_storman: 'AppType' = 1178, 'sgi-storman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Backup To Neighbor
    #: - [UDP] Backup To Neighbor
    b2n: 'AppType' = 1179, 'b2n', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Millicent Client Proxy
    #: - [UDP] Millicent Client Proxy
    mc_client: 'AppType' = 1180, 'mc-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com Net Management
    #: - [UDP] 3Com Net Management
    AppType_3comnetman: 'AppType' = 1181, '3comnetman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] AcceleNet Control
    accelenet: 'AppType' = 1182, 'accelenet', TransportProtocol.get('tcp')

    #: [UDP] AcceleNet Data
    accelenet_data: 'AppType' = 1182, 'accelenet-data', TransportProtocol.get('udp')

    #: - [TCP] LL Surfup HTTP
    #: - [UDP] LL Surfup HTTP
    llsurfup_http: 'AppType' = 1183, 'llsurfup-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LL Surfup HTTPS
    #: - [UDP] LL Surfup HTTPS
    llsurfup_https: 'AppType' = 1184, 'llsurfup-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Catchpole port
    #: - [UDP] Catchpole port
    catchpole: 'AppType' = 1185, 'catchpole', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MySQL Cluster Manager
    #: - [UDP] MySQL Cluster Manager
    mysql_cluster: 'AppType' = 1186, 'mysql-cluster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alias Service
    #: - [UDP] Alias Service
    alias: 'AppType' = 1187, 'alias', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Web Admin
    #: - [UDP] HP Web Admin
    hp_webadmin: 'AppType' = 1188, 'hp-webadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unet Connection
    #: - [UDP] Unet Connection
    unet: 'AppType' = 1189, 'unet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CommLinx GPS / AVL System
    #: - [UDP] CommLinx GPS / AVL System
    commlinx_avl: 'AppType' = 1190, 'commlinx-avl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] General Parallel File System
    #: - [UDP] General Parallel File System
    gpfs: 'AppType' = 1191, 'gpfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] caids sensors channel
    #: - [UDP] caids sensors channel
    caids_sensor: 'AppType' = 1192, 'caids-sensor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Five Across Server
    #: - [UDP] Five Across Server
    fiveacross: 'AppType' = 1193, 'fiveacross', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenVPN
    #: - [UDP] OpenVPN
    openvpn: 'AppType' = 1194, 'openvpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSF-1 clustering
    #: - [UDP] RSF-1 clustering
    rsf_1: 'AppType' = 1195, 'rsf-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Magic
    #: - [UDP] Network Magic
    netmagic: 'AppType' = 1196, 'netmagic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Carrius Remote Access
    #: - [UDP] Carrius Remote Access
    carrius_rshell: 'AppType' = 1197, 'carrius-rshell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cajo reference discovery
    #: - [UDP] cajo reference discovery
    cajo_discovery: 'AppType' = 1198, 'cajo-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DMIDI
    #: - [UDP] DMIDI
    dmidi: 'AppType' = 1199, 'dmidi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCOL
    #: - [UDP] SCOL
    scol: 'AppType' = 1200, 'scol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nucleus Sand Database Server
    #: - [UDP] Nucleus Sand Database Server
    nucleus_sand: 'AppType' = 1201, 'nucleus-sand', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] caiccipc
    #: - [UDP] caiccipc
    caiccipc: 'AppType' = 1202, 'caiccipc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] License Validation
    #: - [UDP] License Validation
    ssslic_mgr: 'AppType' = 1203, 'ssslic-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Log Request Listener
    #: - [UDP] Log Request Listener
    ssslog_mgr: 'AppType' = 1204, 'ssslog-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Accord-MGC
    #: - [UDP] Accord-MGC
    accord_mgc: 'AppType' = 1205, 'accord-mgc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anthony Data
    #: - [UDP] Anthony Data
    anthony_data: 'AppType' = 1206, 'anthony-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaSage
    #: - [UDP] MetaSage
    metasage: 'AppType' = 1207, 'metasage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SEAGULL AIS
    #: - [UDP] SEAGULL AIS
    seagull_ais: 'AppType' = 1208, 'seagull-ais', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPCD3
    #: - [UDP] IPCD3
    ipcd3: 'AppType' = 1209, 'ipcd3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EOSS
    #: - [UDP] EOSS
    eoss: 'AppType' = 1210, 'eoss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Groove DPP
    #: - [UDP] Groove DPP
    groove_dpp: 'AppType' = 1211, 'groove-dpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] lupa
    #: - [UDP] lupa
    lupa: 'AppType' = 1212, 'lupa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Medtronic/Physio-Control LIFENET
    #: - [UDP] Medtronic/Physio-Control LIFENET
    mpc_lifenet: 'AppType' = 1213, 'mpc-lifenet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KAZAA
    #: - [UDP] KAZAA
    kazaa: 'AppType' = 1214, 'kazaa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] scanSTAT 1.0
    #: - [UDP] scanSTAT 1.0
    scanstat_1: 'AppType' = 1215, 'scanstat-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ETEBAC 5
    #: - [UDP] ETEBAC 5
    etebac5: 'AppType' = 1216, 'etebac5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPSS NonDCE Gateway
    #: - [UDP] HPSS NonDCE Gateway
    hpss_ndapi: 'AppType' = 1217, 'hpss-ndapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AeroFlight-ADs
    #: - [UDP] AeroFlight-ADs
    aeroflight_ads: 'AppType' = 1218, 'aeroflight-ads', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AeroFlight-Ret
    #: - [UDP] AeroFlight-Ret
    aeroflight_ret: 'AppType' = 1219, 'aeroflight-ret', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QT SERVER ADMIN
    #: - [UDP] QT SERVER ADMIN
    qt_serveradmin: 'AppType' = 1220, 'qt-serveradmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SweetWARE Apps
    #: - [UDP] SweetWARE Apps
    sweetware_apps: 'AppType' = 1221, 'sweetware-apps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNI R&D network
    #: - [UDP] SNI R&D network
    nerv: 'AppType' = 1222, 'nerv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TrulyGlobal Protocol
    #: - [UDP] TrulyGlobal Protocol
    tgp: 'AppType' = 1223, 'tgp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VPNz
    #: - [UDP] VPNz
    vpnz: 'AppType' = 1224, 'vpnz', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SLINKYSEARCH
    #: - [UDP] SLINKYSEARCH
    slinkysearch: 'AppType' = 1225, 'slinkysearch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STGXFWS
    #: - [UDP] STGXFWS
    stgxfws: 'AppType' = 1226, 'stgxfws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNS2Go
    #: - [UDP] DNS2Go
    dns2go: 'AppType' = 1227, 'dns2go', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FLORENCE
    #: - [UDP] FLORENCE
    florence: 'AppType' = 1228, 'florence', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ZENworks Tiered Electronic Distribution
    #: - [UDP] ZENworks Tiered Electronic Distribution
    zented: 'AppType' = 1229, 'zented', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Periscope
    #: - [UDP] Periscope
    periscope: 'AppType' = 1230, 'periscope', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] menandmice-lpm
    #: - [UDP] menandmice-lpm
    menandmice_lpm: 'AppType' = 1231, 'menandmice-lpm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote systems monitoring
    #: - [UDP] Remote systems monitoring
    first_defense: 'AppType' = 1232, 'first-defense', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Universal App Server
    #: - [UDP] Universal App Server
    univ_appserver: 'AppType' = 1233, 'univ-appserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Infoseek Search Agent
    #: - [UDP] Infoseek Search Agent
    search_agent: 'AppType' = 1234, 'search-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mosaicsyssvc1
    #: - [UDP] mosaicsyssvc1
    mosaicsyssvc1: 'AppType' = 1235, 'mosaicsyssvc1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bvcontrol
    #: - [UDP] bvcontrol
    bvcontrol: 'AppType' = 1236, 'bvcontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tsdos390
    #: - [UDP] tsdos390
    tsdos390: 'AppType' = 1237, 'tsdos390', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hacl-qs
    #: - [UDP] hacl-qs
    hacl_qs: 'AppType' = 1238, 'hacl-qs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMSD
    #: - [UDP] NMSD
    nmsd: 'AppType' = 1239, 'nmsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instantia
    #: - [UDP] Instantia
    instantia: 'AppType' = 1240, 'instantia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nessus
    #: - [UDP] nessus
    nessus: 'AppType' = 1241, 'nessus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMAS over IP
    #: - [UDP] NMAS over IP
    nmasoverip: 'AppType' = 1242, 'nmasoverip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SerialGateway
    #: - [UDP] SerialGateway
    serialgateway: 'AppType' = 1243, 'serialgateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isbconference1
    #: - [UDP] isbconference1
    isbconference1: 'AppType' = 1244, 'isbconference1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isbconference2
    #: - [UDP] isbconference2
    isbconference2: 'AppType' = 1245, 'isbconference2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] payrouter
    #: - [UDP] payrouter
    payrouter: 'AppType' = 1246, 'payrouter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VisionPyramid
    #: - [UDP] VisionPyramid
    visionpyramid: 'AppType' = 1247, 'visionpyramid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hermes
    #: - [UDP] hermes
    hermes: 'AppType' = 1248, 'hermes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mesa Vista Co
    #: - [UDP] Mesa Vista Co
    mesavistaco: 'AppType' = 1249, 'mesavistaco', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swldy-sias
    #: - [UDP] swldy-sias
    swldy_sias: 'AppType' = 1250, 'swldy-sias', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] servergraph
    #: - [UDP] servergraph
    servergraph: 'AppType' = 1251, 'servergraph', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bspne-pcc
    #: - [UDP] bspne-pcc
    bspne_pcc: 'AppType' = 1252, 'bspne-pcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] q55-pcc
    #: - [UDP] q55-pcc
    q55_pcc: 'AppType' = 1253, 'q55-pcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] de-noc
    #: - [UDP] de-noc
    de_noc: 'AppType' = 1254, 'de-noc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] de-cache-query
    #: - [UDP] de-cache-query
    de_cache_query: 'AppType' = 1255, 'de-cache-query', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] de-server
    #: - [UDP] de-server
    de_server: 'AppType' = 1256, 'de-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Shockwave 2
    #: - [UDP] Shockwave 2
    shockwave2: 'AppType' = 1257, 'shockwave2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Open Network Library
    #: - [UDP] Open Network Library
    opennl: 'AppType' = 1258, 'opennl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Open Network Library Voice
    #: - [UDP] Open Network Library Voice
    opennl_voice: 'AppType' = 1259, 'opennl-voice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ibm-ssd
    #: - [UDP] ibm-ssd
    ibm_ssd: 'AppType' = 1260, 'ibm-ssd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpshrsv
    #: - [UDP] mpshrsv
    mpshrsv: 'AppType' = 1261, 'mpshrsv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QNTS-ORB
    #: - [UDP] QNTS-ORB
    qnts_orb: 'AppType' = 1262, 'qnts-orb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dka
    #: - [UDP] dka
    dka: 'AppType' = 1263, 'dka', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PRAT
    #: - [UDP] PRAT
    prat: 'AppType' = 1264, 'prat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DSSIAPI
    #: - [UDP] DSSIAPI
    dssiapi: 'AppType' = 1265, 'dssiapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DELLPWRAPPKS
    #: - [UDP] DELLPWRAPPKS
    dellpwrappks: 'AppType' = 1266, 'dellpwrappks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eTrust Policy Compliance
    #: - [UDP] eTrust Policy Compliance
    epc: 'AppType' = 1267, 'epc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PROPEL-MSGSYS
    #: - [UDP] PROPEL-MSGSYS
    propel_msgsys: 'AppType' = 1268, 'propel-msgsys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WATiLaPP
    #: - [UDP] WATiLaPP
    watilapp: 'AppType' = 1269, 'watilapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Operations Manager
    #: - [UDP] Microsoft Operations Manager
    opsmgr: 'AppType' = 1270, 'opsmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eXcW
    #: - [UDP] eXcW
    excw: 'AppType' = 1271, 'excw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSPMLockMgr
    #: - [UDP] CSPMLockMgr
    cspmlockmgr: 'AppType' = 1272, 'cspmlockmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMC-Gateway
    #: - [UDP] EMC-Gateway
    emc_gateway: 'AppType' = 1273, 'emc-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] t1distproc
    #: - [UDP] t1distproc
    t1distproc: 'AppType' = 1274, 't1distproc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ivcollector
    #: - [UDP] ivcollector
    ivcollector: 'AppType' = 1275, 'ivcollector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_1276: 'AppType' = 1276, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mqs
    #: - [UDP] mqs
    miva_mqs: 'AppType' = 1277, 'miva-mqs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dell Web Admin 1
    #: - [UDP] Dell Web Admin 1
    dellwebadmin_1: 'AppType' = 1278, 'dellwebadmin-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dell Web Admin 2
    #: - [UDP] Dell Web Admin 2
    dellwebadmin_2: 'AppType' = 1279, 'dellwebadmin-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pictrography
    #: - [UDP] Pictrography
    pictrography: 'AppType' = 1280, 'pictrography', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] healthd
    #: - [UDP] healthd
    healthd: 'AppType' = 1281, 'healthd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Emperion
    #: - [UDP] Emperion
    emperion: 'AppType' = 1282, 'emperion', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Product Information
    #: - [UDP] Product Information
    productinfo: 'AppType' = 1283, 'productinfo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IEE-QFX
    #: - [UDP] IEE-QFX
    iee_qfx: 'AppType' = 1284, 'iee-qfx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] neoiface
    #: - [UDP] neoiface
    neoiface: 'AppType' = 1285, 'neoiface', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netuitive
    #: - [UDP] netuitive
    netuitive: 'AppType' = 1286, 'netuitive', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RouteMatch Com
    #: - [UDP] RouteMatch Com
    routematch: 'AppType' = 1287, 'routematch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NavBuddy
    #: - [UDP] NavBuddy
    navbuddy: 'AppType' = 1288, 'navbuddy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JWalkServer
    #: - [UDP] JWalkServer
    jwalkserver: 'AppType' = 1289, 'jwalkserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WinJaServer
    #: - [UDP] WinJaServer
    winjaserver: 'AppType' = 1290, 'winjaserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SEAGULLLMS
    #: - [UDP] SEAGULLLMS
    seagulllms: 'AppType' = 1291, 'seagulllms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dsdn
    #: - [UDP] dsdn
    dsdn: 'AppType' = 1292, 'dsdn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PKT-KRB-IPSec
    #: - [UDP] PKT-KRB-IPSec
    pkt_krb_ipsec: 'AppType' = 1293, 'pkt-krb-ipsec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CMMdriver
    #: - [UDP] CMMdriver
    cmmdriver: 'AppType' = 1294, 'cmmdriver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] End-by-Hop Transmission Protocol
    #: - [UDP] End-by-Hop Transmission Protocol
    ehtp: 'AppType' = 1295, 'ehtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dproxy
    #: - [UDP] dproxy
    dproxy: 'AppType' = 1296, 'dproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sdproxy
    #: - [UDP] sdproxy
    sdproxy: 'AppType' = 1297, 'sdproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] lpcp
    #: - [UDP] lpcp
    lpcp: 'AppType' = 1298, 'lpcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp-sci
    #: - [UDP] hp-sci
    hp_sci: 'AppType' = 1299, 'hp-sci', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.323 Secure Call Control Signalling
    #: - [UDP] H.323 Secure Call Control Signalling
    h323hostcallsc: 'AppType' = 1300, 'h323hostcallsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_1301: 'AppType' = 1301, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_1302: 'AppType' = 1302, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sftsrv
    #: - [UDP] sftsrv
    sftsrv: 'AppType' = 1303, 'sftsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Boomerang
    #: - [UDP] Boomerang
    boomerang: 'AppType' = 1304, 'boomerang', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pe-mike
    #: - [UDP] pe-mike
    pe_mike: 'AppType' = 1305, 'pe-mike', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RE-Conn-Proto
    #: - [UDP] RE-Conn-Proto
    re_conn_proto: 'AppType' = 1306, 're-conn-proto', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pacmand
    #: - [UDP] Pacmand
    pacmand: 'AppType' = 1307, 'pacmand', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optical Domain Service Interconnect (ODSI)
    #: - [UDP] Optical Domain Service Interconnect (ODSI)
    odsi: 'AppType' = 1308, 'odsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JTAG server
    #: - [UDP] JTAG server
    jtag_server: 'AppType' = 1309, 'jtag-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Husky
    #: - [UDP] Husky
    husky: 'AppType' = 1310, 'husky', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RxMon
    #: - [UDP] RxMon
    rxmon: 'AppType' = 1311, 'rxmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STI Envision
    #: - [UDP] STI Envision
    sti_envision: 'AppType' = 1312, 'sti-envision', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC_PATROLDB IANA assigned this well-formed service name as a
    #:   replacement for "bmc_patroldb".
    #: - [TCP] BMC_PATROLDB
    #: - [UDP] BMC_PATROLDB IANA assigned this well-formed service name as a
    #:   replacement for "bmc_patroldb".
    #: - [UDP] BMC_PATROLDB
    bmc_patroldb: 'AppType' = 1313, 'bmc-patroldb', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Photoscript Distributed Printing System
    #: - [UDP] Photoscript Distributed Printing System
    pdps: 'AppType' = 1314, 'pdps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] E.L.S., Event Listener Service
    #: - [UDP] E.L.S., Event Listener Service
    els: 'AppType' = 1315, 'els', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Exbit-ESCP
    #: - [UDP] Exbit-ESCP
    exbit_escp: 'AppType' = 1316, 'exbit-escp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vrts-ipcserver
    #: - [UDP] vrts-ipcserver
    vrts_ipcserver: 'AppType' = 1317, 'vrts-ipcserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] krb5gatekeeper
    #: - [UDP] krb5gatekeeper
    krb5gatekeeper: 'AppType' = 1318, 'krb5gatekeeper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMX-ICSP
    #: - [UDP] AMX-ICSP
    amx_icsp: 'AppType' = 1319, 'amx-icsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMX-AXBNET
    #: - [UDP] AMX-AXBNET
    amx_axbnet: 'AppType' = 1320, 'amx-axbnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PIP
    #: - [UDP] PIP
    pip_321: 'AppType' = 321, 'pip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PIP
    #: - [UDP] PIP
    pip_1321: 'AppType' = 1321, 'pip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novation
    #: - [UDP] Novation
    novation: 'AppType' = 1322, 'novation', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] brcd
    #: - [UDP] brcd
    brcd: 'AppType' = 1323, 'brcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] delta-mcp
    #: - [UDP] delta-mcp
    delta_mcp: 'AppType' = 1324, 'delta-mcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DX-Instrument
    #: - [UDP] DX-Instrument
    dx_instrument: 'AppType' = 1325, 'dx-instrument', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WIMSIC
    #: - [UDP] WIMSIC
    wimsic: 'AppType' = 1326, 'wimsic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ultrex
    #: - [UDP] Ultrex
    ultrex: 'AppType' = 1327, 'ultrex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EWALL
    #: - [UDP] EWALL
    ewall: 'AppType' = 1328, 'ewall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netdb-export
    #: - [UDP] netdb-export
    netdb_export: 'AppType' = 1329, 'netdb-export', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] StreetPerfect
    #: - [UDP] StreetPerfect
    streetperfect: 'AppType' = 1330, 'streetperfect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] intersan
    #: - [UDP] intersan
    intersan: 'AppType' = 1331, 'intersan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCIA RXP-B
    #: - [UDP] PCIA RXP-B
    pcia_rxp_b: 'AppType' = 1332, 'pcia-rxp-b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Password Policy
    #: - [UDP] Password Policy
    passwrd_policy: 'AppType' = 1333, 'passwrd-policy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] writesrv
    #: - [UDP] writesrv
    writesrv: 'AppType' = 1334, 'writesrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Digital Notary Protocol
    #: - [UDP] Digital Notary Protocol
    digital_notary: 'AppType' = 1335, 'digital-notary', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instant Service Chat
    #: - [UDP] Instant Service Chat
    ischat: 'AppType' = 1336, 'ischat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] menandmice DNS
    #: - [UDP] menandmice DNS
    menandmice_dns: 'AppType' = 1337, 'menandmice-dns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WMC-log-svr
    #: - [UDP] WMC-log-svr
    wmc_log_svc: 'AppType' = 1338, 'wmc-log-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] kjtsiteserver
    #: - [UDP] kjtsiteserver
    kjtsiteserver: 'AppType' = 1339, 'kjtsiteserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NAAP
    #: - [UDP] NAAP
    naap: 'AppType' = 1340, 'naap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QuBES
    #: - [UDP] QuBES
    qubes: 'AppType' = 1341, 'qubes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESBroker
    #: - [UDP] ESBroker
    esbroker: 'AppType' = 1342, 'esbroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] re101
    #: - [UDP] re101
    re101: 'AppType' = 1343, 're101', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICAP
    #: - [UDP] ICAP
    icap: 'AppType' = 1344, 'icap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VPJP
    #: - [UDP] VPJP
    vpjp: 'AppType' = 1345, 'vpjp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alta Analytics License Manager
    #: - [UDP] Alta Analytics License Manager
    alta_ana_lm: 'AppType' = 1346, 'alta-ana-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] multi media conferencing
    #: - [UDP] multi media conferencing
    bbn_mmc: 'AppType' = 1347, 'bbn-mmc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] multi media conferencing
    #: - [UDP] multi media conferencing
    bbn_mmx: 'AppType' = 1348, 'bbn-mmx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Registration Network Protocol
    #: - [UDP] Registration Network Protocol
    sbook: 'AppType' = 1349, 'sbook', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Registration Network Protocol
    #: - [UDP] Registration Network Protocol
    editbench: 'AppType' = 1350, 'editbench', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Digital Tool Works (MIT)
    #: - [UDP] Digital Tool Works (MIT)
    equationbuilder: 'AppType' = 1351, 'equationbuilder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lotus Note
    #: - [UDP] Lotus Note
    lotusnote: 'AppType' = 1352, 'lotusnote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Relief Consulting
    #: - [UDP] Relief Consulting
    relief: 'AppType' = 1353, 'relief', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Five Across XSIP Network
    #: - [UDP] Five Across XSIP Network
    xsip_network: 'AppType' = 1354, 'xsip-network', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intuitive Edge
    #: - [UDP] Intuitive Edge
    intuitive_edge: 'AppType' = 1355, 'intuitive-edge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CuillaMartin Company
    #: - [UDP] CuillaMartin Company
    cuillamartin: 'AppType' = 1356, 'cuillamartin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Electronic PegBoard
    #: - [UDP] Electronic PegBoard
    pegboard: 'AppType' = 1357, 'pegboard', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CONNLCLI
    #: - [UDP] CONNLCLI
    connlcli: 'AppType' = 1358, 'connlcli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FTSRV
    #: - [UDP] FTSRV
    ftsrv: 'AppType' = 1359, 'ftsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIMER
    #: - [UDP] MIMER
    mimer: 'AppType' = 1360, 'mimer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LinX
    #: - [UDP] LinX
    linx: 'AppType' = 1361, 'linx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TimeFlies
    #: - [UDP] TimeFlies
    timeflies: 'AppType' = 1362, 'timeflies', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network DataMover Requester
    #: - [UDP] Network DataMover Requester
    ndm_requester: 'AppType' = 1363, 'ndm-requester', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network DataMover Server
    #: - [UDP] Network DataMover Server
    ndm_server: 'AppType' = 1364, 'ndm-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Software Associates
    #: - [UDP] Network Software Associates
    adapt_sna: 'AppType' = 1365, 'adapt-sna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novell NetWare Comm Service Platform
    #: - [UDP] Novell NetWare Comm Service Platform
    netware_csp: 'AppType' = 1366, 'netware-csp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DCS
    #: - [UDP] DCS
    dcs: 'AppType' = 1367, 'dcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ScreenCast
    #: - [UDP] ScreenCast
    screencast: 'AppType' = 1368, 'screencast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GlobalView to Unix Shell
    #: - [UDP] GlobalView to Unix Shell
    gv_us: 'AppType' = 1369, 'gv-us', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unix Shell to GlobalView
    #: - [UDP] Unix Shell to GlobalView
    us_gv: 'AppType' = 1370, 'us-gv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Config Protocol
    #: - [UDP] Fujitsu Config Protocol
    fc_cli: 'AppType' = 1371, 'fc-cli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Config Protocol
    #: - [UDP] Fujitsu Config Protocol
    fc_ser: 'AppType' = 1372, 'fc-ser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Chromagrafx
    #: - [UDP] Chromagrafx
    chromagrafx: 'AppType' = 1373, 'chromagrafx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPI Software Systems
    #: - [UDP] EPI Software Systems
    molly: 'AppType' = 1374, 'molly', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bytex
    #: - [UDP] Bytex
    bytex: 'AppType' = 1375, 'bytex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Person to Person Software
    #: - [UDP] IBM Person to Person Software
    ibm_pps: 'AppType' = 1376, 'ibm-pps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cichlid License Manager
    #: - [UDP] Cichlid License Manager
    cichlid: 'AppType' = 1377, 'cichlid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Elan License Manager
    #: - [UDP] Elan License Manager
    elan: 'AppType' = 1378, 'elan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Integrity Solutions
    #: - [UDP] Integrity Solutions
    dbreporter: 'AppType' = 1379, 'dbreporter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Telesis Network License Manager
    #: - [UDP] Telesis Network License Manager
    telesis_licman: 'AppType' = 1380, 'telesis-licman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple Network License Manager
    #: - [UDP] Apple Network License Manager
    apple_licman: 'AppType' = 1381, 'apple-licman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GW Hannaway Network License Manager
    #: - [UDP] GW Hannaway Network License Manager
    gwha: 'AppType' = 1383, 'gwha', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Objective Solutions License Manager
    #: - [UDP] Objective Solutions License Manager
    os_licman: 'AppType' = 1384, 'os-licman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Atex Publishing License Manager IANA assigned this well-formed service
    #:   name as a replacement for "atex_elmd".
    #: - [TCP] Atex Publishing License Manager
    #: - [UDP] Atex Publishing License Manager IANA assigned this well-formed service
    #:   name as a replacement for "atex_elmd".
    #: - [UDP] Atex Publishing License Manager
    atex_elmd: 'AppType' = 1385, 'atex-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] CheckSum License Manager
    #: - [UDP] CheckSum License Manager
    checksum: 'AppType' = 1386, 'checksum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Computer Aided Design Software Inc LM
    #: - [UDP] Computer Aided Design Software Inc LM
    cadsi_lm: 'AppType' = 1387, 'cadsi-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Objective Solutions DataBase Cache
    #: - [UDP] Objective Solutions DataBase Cache
    objective_dbc: 'AppType' = 1388, 'objective-dbc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Document Manager
    #: - [UDP] Document Manager
    iclpv_dm: 'AppType' = 1389, 'iclpv-dm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Storage Controller
    #: - [UDP] Storage Controller
    iclpv_sc: 'AppType' = 1390, 'iclpv-sc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Storage Access Server
    #: - [UDP] Storage Access Server
    iclpv_sas: 'AppType' = 1391, 'iclpv-sas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Print Manager
    #: - [UDP] Print Manager
    iclpv_pm: 'AppType' = 1392, 'iclpv-pm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Log Server
    #: - [UDP] Network Log Server
    iclpv_nls: 'AppType' = 1393, 'iclpv-nls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Log Client
    #: - [UDP] Network Log Client
    iclpv_nlc: 'AppType' = 1394, 'iclpv-nlc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PC Workstation Manager software
    #: - [UDP] PC Workstation Manager software
    iclpv_wsm: 'AppType' = 1395, 'iclpv-wsm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DVL Active Mail
    #: - [UDP] DVL Active Mail
    dvl_activemail: 'AppType' = 1396, 'dvl-activemail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Audio Active Mail
    #: - [UDP] Audio Active Mail
    audio_activmail: 'AppType' = 1397, 'audio-activmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Video Active Mail
    #: - [UDP] Video Active Mail
    video_activmail: 'AppType' = 1398, 'video-activmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadkey License Manager
    #: - [UDP] Cadkey License Manager
    cadkey_licman: 'AppType' = 1399, 'cadkey-licman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadkey Tablet Daemon
    #: - [UDP] Cadkey Tablet Daemon
    cadkey_tablet: 'AppType' = 1400, 'cadkey-tablet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Goldleaf License Manager
    #: - [UDP] Goldleaf License Manager
    goldleaf_licman: 'AppType' = 1401, 'goldleaf-licman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Resource Manager
    #: - [UDP] Prospero Resource Manager
    prm_sm_np: 'AppType' = 1402, 'prm-sm-np', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Resource Manager
    #: - [UDP] Prospero Resource Manager
    prm_nm_np: 'AppType' = 1403, 'prm-nm-np', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Infinite Graphics License Manager
    #: - [UDP] Infinite Graphics License Manager
    igi_lm: 'AppType' = 1404, 'igi-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Remote Execution Starter
    #: - [UDP] IBM Remote Execution Starter
    ibm_res: 'AppType' = 1405, 'ibm-res', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetLabs License Manager
    #: - [UDP] NetLabs License Manager
    netlabs_lm: 'AppType' = 1406, 'netlabs-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] TIBET Data Server
    tibet_server: 'AppType' = 1407, 'tibet-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_1407: 'AppType' = 1407, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Sophia License Manager
    #: - [UDP] Sophia License Manager
    sophia_lm: 'AppType' = 1408, 'sophia-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Here License Manager
    #: - [UDP] Here License Manager
    here_lm: 'AppType' = 1409, 'here-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HiQ License Manager
    #: - [UDP] HiQ License Manager
    hiq: 'AppType' = 1410, 'hiq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AudioFile
    #: - [UDP] AudioFile
    af: 'AppType' = 1411, 'af', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InnoSys
    #: - [UDP] InnoSys
    innosys: 'AppType' = 1412, 'innosys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Innosys-ACL
    #: - [UDP] Innosys-ACL
    innosys_acl: 'AppType' = 1413, 'innosys-acl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM MQSeries
    #: - [UDP] IBM MQSeries
    ibm_mqseries: 'AppType' = 1414, 'ibm-mqseries', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DBStar
    #: - [UDP] DBStar
    dbstar: 'AppType' = 1415, 'dbstar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novell LU6.2 IANA assigned this well-formed service name as a
    #:   replacement for "novell-lu6.2".
    #: - [TCP] Novell LU6.2
    #: - [UDP] Novell LU6.2 IANA assigned this well-formed service name as a
    #:   replacement for "novell-lu6.2".
    #: - [UDP] Novell LU6.2
    novell_lu6_2: 'AppType' = 1416, 'novell-lu6-2', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Timbuktu Service 1 Port
    #: - [UDP] Timbuktu Service 1 Port
    timbuktu_srv1: 'AppType' = 1417, 'timbuktu-srv1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Timbuktu Service 2 Port
    #: - [UDP] Timbuktu Service 2 Port
    timbuktu_srv2: 'AppType' = 1418, 'timbuktu-srv2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Timbuktu Service 3 Port
    #: - [UDP] Timbuktu Service 3 Port
    timbuktu_srv3: 'AppType' = 1419, 'timbuktu-srv3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Timbuktu Service 4 Port
    #: - [UDP] Timbuktu Service 4 Port
    timbuktu_srv4: 'AppType' = 1420, 'timbuktu-srv4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gandalf License Manager
    #: - [UDP] Gandalf License Manager
    gandalf_lm: 'AppType' = 1421, 'gandalf-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Autodesk License Manager
    #: - [UDP] Autodesk License Manager
    autodesk_lm: 'AppType' = 1422, 'autodesk-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Essbase Arbor Software
    #: - [UDP] Essbase Arbor Software
    essbase: 'AppType' = 1423, 'essbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hybrid Encryption Protocol
    #: - [UDP] Hybrid Encryption Protocol
    hybrid: 'AppType' = 1424, 'hybrid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zion Software License Manager
    #: - [UDP] Zion Software License Manager
    zion_lm: 'AppType' = 1425, 'zion-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Satellite-data Acquisition System 1
    #: - [UDP] Satellite-data Acquisition System 1
    sais: 'AppType' = 1426, 'sais', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mloadd monitoring tool
    #: - [UDP] mloadd monitoring tool
    mloadd: 'AppType' = 1427, 'mloadd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Informatik License Manager
    #: - [UDP] Informatik License Manager
    informatik_lm: 'AppType' = 1428, 'informatik-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hypercom NMS
    #: - [UDP] Hypercom NMS
    nms: 'AppType' = 1429, 'nms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hypercom TPDU
    #: - [UDP] Hypercom TPDU
    tpdu: 'AppType' = 1430, 'tpdu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reverse Gossip Transport
    #: - [UDP] Reverse Gossip Transport
    rgtp: 'AppType' = 1431, 'rgtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blueberry Software License Manager
    #: - [UDP] Blueberry Software License Manager
    blueberry_lm: 'AppType' = 1432, 'blueberry-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft-SQL-Server
    #: - [UDP] Microsoft-SQL-Server
    ms_sql_s: 'AppType' = 1433, 'ms-sql-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft-SQL-Monitor
    #: - [UDP] Microsoft-SQL-Monitor
    ms_sql_m: 'AppType' = 1434, 'ms-sql-m', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM CICS
    #: - [UDP] IBM CICS
    ibm_cics: 'AppType' = 1435, 'ibm-cics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Satellite-data Acquisition System 2
    #: - [UDP] Satellite-data Acquisition System 2
    saism: 'AppType' = 1436, 'saism', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tabula
    #: - [UDP] Tabula
    tabula: 'AppType' = 1437, 'tabula', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eicon Security Agent/Server
    #: - [UDP] Eicon Security Agent/Server
    eicon_server: 'AppType' = 1438, 'eicon-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eicon X25/SNA Gateway
    #: - [UDP] Eicon X25/SNA Gateway
    eicon_x25: 'AppType' = 1439, 'eicon-x25', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eicon Service Location Protocol
    #: - [UDP] Eicon Service Location Protocol
    eicon_slp: 'AppType' = 1440, 'eicon-slp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadis License Management
    #: - [UDP] Cadis License Management
    cadis_1: 'AppType' = 1441, 'cadis-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadis License Management
    #: - [UDP] Cadis License Management
    cadis_2: 'AppType' = 1442, 'cadis-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Integrated Engineering Software
    #: - [UDP] Integrated Engineering Software
    ies_lm: 'AppType' = 1443, 'ies-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Marcam License Management
    #: - [UDP] Marcam License Management
    marcam_lm: 'AppType' = 1444, 'marcam-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proxima License Manager
    #: - [UDP] Proxima License Manager
    proxima_lm: 'AppType' = 1445, 'proxima-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optical Research Associates License Manager
    #: - [UDP] Optical Research Associates License Manager
    ora_lm: 'AppType' = 1446, 'ora-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Applied Parallel Research LM
    #: - [UDP] Applied Parallel Research LM
    apri_lm: 'AppType' = 1447, 'apri-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenConnect License Manager
    #: - [UDP] OpenConnect License Manager
    oc_lm: 'AppType' = 1448, 'oc-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PEport
    #: - [UDP] PEport
    peport: 'AppType' = 1449, 'peport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tandem Distributed Workbench Facility
    #: - [UDP] Tandem Distributed Workbench Facility
    dwf: 'AppType' = 1450, 'dwf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Information Management
    #: - [UDP] IBM Information Management
    infoman: 'AppType' = 1451, 'infoman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GTE Government Systems License Man
    #: - [UDP] GTE Government Systems License Man
    gtegsc_lm: 'AppType' = 1452, 'gtegsc-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Genie License Manager
    #: - [UDP] Genie License Manager
    genie_lm: 'AppType' = 1453, 'genie-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] interHDL License Manager IANA assigned this well-formed service name
    #:   as a replacement for "interhdl_elmd".
    #: - [TCP] interHDL License Manager
    #: - [UDP] interHDL License Manager IANA assigned this well-formed service name
    #:   as a replacement for "interhdl_elmd".
    #: - [UDP] interHDL License Manager
    interhdl_elmd: 'AppType' = 1454, 'interhdl-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ESL License Manager
    #: - [UDP] ESL License Manager
    esl_lm: 'AppType' = 1455, 'esl-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DCA
    #: - [UDP] DCA
    dca: 'AppType' = 1456, 'dca', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Valisys License Manager
    #: - [UDP] Valisys License Manager
    valisys_lm: 'AppType' = 1457, 'valisys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nichols Research Corp.
    #: - [UDP] Nichols Research Corp.
    nrcabq_lm: 'AppType' = 1458, 'nrcabq-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proshare Notebook Application
    #: - [UDP] Proshare Notebook Application
    proshare1: 'AppType' = 1459, 'proshare1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proshare Notebook Application
    #: - [UDP] Proshare Notebook Application
    proshare2: 'AppType' = 1460, 'proshare2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Wireless LAN IANA assigned this well-formed service name as a
    #:   replacement for "ibm_wrless_lan".
    #: - [TCP] IBM Wireless LAN
    #: - [UDP] IBM Wireless LAN IANA assigned this well-formed service name as a
    #:   replacement for "ibm_wrless_lan".
    #: - [UDP] IBM Wireless LAN
    ibm_wrless_lan: 'AppType' = 1461, 'ibm-wrless-lan', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] World License Manager
    #: - [UDP] World License Manager
    world_lm: 'AppType' = 1462, 'world-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nucleus
    #: - [UDP] Nucleus
    nucleus: 'AppType' = 1463, 'nucleus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSL License Manager IANA assigned this well-formed service name as a
    #:   replacement for "msl_lmd".
    #: - [TCP] MSL License Manager
    #: - [UDP] MSL License Manager IANA assigned this well-formed service name as a
    #:   replacement for "msl_lmd".
    #: - [UDP] MSL License Manager
    msl_lmd: 'AppType' = 1464, 'msl-lmd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Pipes Platform
    #: - [UDP] Pipes Platform
    pipes: 'AppType' = 1465, 'pipes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ocean Software License Manager
    #: - [UDP] Ocean Software License Manager
    oceansoft_lm: 'AppType' = 1466, 'oceansoft-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active Analysis Limited License Manager
    #: - [UDP] Active Analysis Limited License Manager
    aal_lm: 'AppType' = 1469, 'aal-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Universal Analytics
    #: - [UDP] Universal Analytics
    uaiact: 'AppType' = 1470, 'uaiact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSDMBASE
    #: - [UDP] CSDMBASE
    csdmbase_1467: 'AppType' = 1467, 'csdmbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] csdmbase
    #: - [UDP] csdmbase
    csdmbase_1471: 'AppType' = 1471, 'csdmbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSDM
    #: - [UDP] CSDM
    csdm_1468: 'AppType' = 1468, 'csdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] csdm
    #: - [UDP] csdm
    csdm_1472: 'AppType' = 1472, 'csdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenMath
    #: - [UDP] OpenMath
    openmath: 'AppType' = 1473, 'openmath', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Telefinder
    #: - [UDP] Telefinder
    telefinder: 'AppType' = 1474, 'telefinder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Taligent License Manager
    #: - [UDP] Taligent License Manager
    taligent_lm: 'AppType' = 1475, 'taligent-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] clvm-cfg
    #: - [UDP] clvm-cfg
    clvm_cfg: 'AppType' = 1476, 'clvm-cfg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ms-sna-server
    #: - [UDP] ms-sna-server
    ms_sna_server: 'AppType' = 1477, 'ms-sna-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ms-sna-base
    #: - [UDP] ms-sna-base
    ms_sna_base: 'AppType' = 1478, 'ms-sna-base', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dberegister
    #: - [UDP] dberegister
    dberegister: 'AppType' = 1479, 'dberegister', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PacerForum
    #: - [UDP] PacerForum
    pacerforum: 'AppType' = 1480, 'pacerforum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AIRS
    #: - [UDP] AIRS
    airs: 'AppType' = 1481, 'airs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Miteksys License Manager
    #: - [UDP] Miteksys License Manager
    miteksys_lm: 'AppType' = 1482, 'miteksys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AFS License Manager
    #: - [UDP] AFS License Manager
    afs: 'AppType' = 1483, 'afs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Confluent License Manager
    #: - [UDP] Confluent License Manager
    confluent: 'AppType' = 1484, 'confluent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LANSource
    #: - [UDP] LANSource
    lansource: 'AppType' = 1485, 'lansource', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nms_topo_serv IANA assigned this well-formed service name as a
    #:   replacement for "nms_topo_serv".
    #: - [TCP] nms_topo_serv
    #: - [UDP] nms_topo_serv IANA assigned this well-formed service name as a
    #:   replacement for "nms_topo_serv".
    #: - [UDP] nms_topo_serv
    nms_topo_serv: 'AppType' = 1486, 'nms-topo-serv', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] LocalInfoSrvr
    #: - [UDP] LocalInfoSrvr
    localinfosrvr: 'AppType' = 1487, 'localinfosrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DocStor
    #: - [UDP] DocStor
    docstor: 'AppType' = 1488, 'docstor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dmdocbroker
    #: - [UDP] dmdocbroker
    dmdocbroker: 'AppType' = 1489, 'dmdocbroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] insitu-conf
    #: - [UDP] insitu-conf
    insitu_conf: 'AppType' = 1490, 'insitu-conf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_1491: 'AppType' = 1491, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] stone-design-1
    #: - [UDP] stone-design-1
    stone_design_1: 'AppType' = 1492, 'stone-design-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netmap_lm IANA assigned this well-formed service name as a replacement
    #:   for "netmap_lm".
    #: - [TCP] netmap_lm
    #: - [UDP] netmap_lm IANA assigned this well-formed service name as a replacement
    #:   for "netmap_lm".
    #: - [UDP] netmap_lm
    netmap_lm: 'AppType' = 1493, 'netmap-lm', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ica
    #: - [UDP] ica
    ica: 'AppType' = 1494, 'ica', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cvc
    #: - [UDP] cvc
    cvc: 'AppType' = 1495, 'cvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] liberty-lm
    #: - [UDP] liberty-lm
    liberty_lm: 'AppType' = 1496, 'liberty-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rfx-lm
    #: - [UDP] rfx-lm
    rfx_lm: 'AppType' = 1497, 'rfx-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sybase SQL Any
    #: - [UDP] Sybase SQL Any
    sybase_sqlany: 'AppType' = 1498, 'sybase-sqlany', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Federico Heinz Consultora
    #: - [UDP] Federico Heinz Consultora
    fhc: 'AppType' = 1499, 'fhc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VLSI License Manager
    #: - [UDP] VLSI License Manager
    vlsi_lm: 'AppType' = 1500, 'vlsi-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Satellite-data Acquisition System 3
    #: - [UDP] Satellite-data Acquisition System 3
    saiscm: 'AppType' = 1501, 'saiscm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Shiva
    #: - [UDP] Shiva
    shivadiscovery: 'AppType' = 1502, 'shivadiscovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Databeam
    #: - [UDP] Databeam
    imtc_mcs: 'AppType' = 1503, 'imtc-mcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EVB Software Engineering License Manager
    #: - [UDP] EVB Software Engineering License Manager
    evb_elm: 'AppType' = 1504, 'evb-elm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Funk Software, Inc.
    #: - [UDP] Funk Software, Inc.
    funkproxy: 'AppType' = 1505, 'funkproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Universal Time daemon (utcd)
    #: - [UDP] Universal Time daemon (utcd)
    utcd: 'AppType' = 1506, 'utcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] symplex
    #: - [UDP] symplex
    symplex: 'AppType' = 1507, 'symplex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] diagmond
    #: - [UDP] diagmond
    diagmond: 'AppType' = 1508, 'diagmond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Robcad, Ltd. License Manager
    #: - [UDP] Robcad, Ltd. License Manager
    robcad_lm: 'AppType' = 1509, 'robcad-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Midland Valley Exploration Ltd. Lic. Man.
    #: - [UDP] Midland Valley Exploration Ltd. Lic. Man.
    mvx_lm: 'AppType' = 1510, 'mvx-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3l-l1
    #: - [UDP] 3l-l1
    AppType_3l_l1: 'AppType' = 1511, '3l-l1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft's Windows Internet Name Service
    #: - [UDP] Microsoft's Windows Internet Name Service
    wins: 'AppType' = 1512, 'wins', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Systems Business of America, Inc
    #: - [UDP] Fujitsu Systems Business of America, Inc
    fujitsu_dtc: 'AppType' = 1513, 'fujitsu-dtc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Systems Business of America, Inc
    #: - [UDP] Fujitsu Systems Business of America, Inc
    fujitsu_dtcns: 'AppType' = 1514, 'fujitsu-dtcns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ifor-protocol
    #: - [UDP] ifor-protocol
    ifor_protocol: 'AppType' = 1515, 'ifor-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Places Audio data
    #: - [UDP] Virtual Places Audio data
    vpad: 'AppType' = 1516, 'vpad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Places Audio control
    #: - [UDP] Virtual Places Audio control
    vpac: 'AppType' = 1517, 'vpac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Places Video data
    #: - [UDP] Virtual Places Video data
    vpvd: 'AppType' = 1518, 'vpvd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Places Video control
    #: - [UDP] Virtual Places Video control
    vpvc: 'AppType' = 1519, 'vpvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] atm zip office
    #: - [UDP] atm zip office
    atm_zip_office: 'AppType' = 1520, 'atm-zip-office', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nCube License Manager
    #: - [UDP] nCube License Manager
    ncube_lm: 'AppType' = 1521, 'ncube-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cichild
    #: - [UDP] cichild
    cichild_lm: 'AppType' = 1523, 'cichild-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ingres
    #: - [UDP] ingres
    ingreslock: 'AppType' = 1524, 'ingreslock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] oracle
    #: - [UDP] oracle
    orasrv: 'AppType' = 1525, 'orasrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Directory Service non-priv
    #: - [UDP] Prospero Directory Service non-priv
    prospero_np: 'AppType' = 1525, 'prospero-np', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prospero Data Access Prot non-priv
    #: - [UDP] Prospero Data Access Prot non-priv
    pdap_np: 'AppType' = 1526, 'pdap-np', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] oracle
    #: - [UDP] oracle
    tlisrv: 'AppType' = 1527, 'tlisrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Not Only a Routeing Protocol
    #: - [UDP] Not Only a Routeing Protocol
    #: - [SCTP] Not Only a Routeing Protocol
    norp: 'AppType' = 1528, 'norp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] oracle
    #: - [UDP] oracle
    coauthor: 'AppType' = 1529, 'coauthor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rap-service
    #: - [UDP] rap-service
    rap_service: 'AppType' = 1530, 'rap-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rap-listen
    #: - [UDP] rap-listen
    rap_listen: 'AppType' = 1531, 'rap-listen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] miroconnect
    #: - [UDP] miroconnect
    miroconnect: 'AppType' = 1532, 'miroconnect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Places Software
    #: - [UDP] Virtual Places Software
    virtual_places: 'AppType' = 1533, 'virtual-places', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] micromuse-lm
    #: - [UDP] micromuse-lm
    micromuse_lm: 'AppType' = 1534, 'micromuse-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ampr-info
    #: - [UDP] ampr-info
    ampr_info: 'AppType' = 1535, 'ampr-info', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ampr-inter
    #: - [UDP] ampr-inter
    ampr_inter: 'AppType' = 1536, 'ampr-inter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isi-lm
    #: - [UDP] isi-lm
    sdsc_lm: 'AppType' = 1537, 'sdsc-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3ds-lm
    #: - [UDP] 3ds-lm
    AppType_3ds_lm: 'AppType' = 1538, '3ds-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intellistor License Manager
    #: - [UDP] Intellistor License Manager
    intellistor_lm: 'AppType' = 1539, 'intellistor-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rds
    #: - [UDP] rds
    rds: 'AppType' = 1540, 'rds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rds2
    #: - [UDP] rds2
    rds2: 'AppType' = 1541, 'rds2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gridgen-elmd
    #: - [UDP] gridgen-elmd
    gridgen_elmd: 'AppType' = 1542, 'gridgen-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simba-cs
    #: - [UDP] simba-cs
    simba_cs: 'AppType' = 1543, 'simba-cs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aspeclmd
    #: - [UDP] aspeclmd
    aspeclmd: 'AppType' = 1544, 'aspeclmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vistium-share
    #: - [UDP] vistium-share
    vistium_share: 'AppType' = 1545, 'vistium-share', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] abbaccuray
    #: - [UDP] abbaccuray
    abbaccuray: 'AppType' = 1546, 'abbaccuray', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] laplink
    #: - [UDP] laplink
    laplink: 'AppType' = 1547, 'laplink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Axon License Manager
    #: - [UDP] Axon License Manager
    axon_lm: 'AppType' = 1548, 'axon-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Shiva Hose
    shivahose: 'AppType' = 1549, 'shivahose', TransportProtocol.get('tcp')

    #: [UDP] Shiva Sound
    shivasound: 'AppType' = 1549, 'shivasound', TransportProtocol.get('udp')

    #: - [TCP] Image Storage license manager 3M Company
    #: - [UDP] Image Storage license manager 3M Company
    AppType_3m_image_lm: 'AppType' = 1550, '3m-image-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HECMTL-DB
    #: - [UDP] HECMTL-DB
    hecmtl_db: 'AppType' = 1551, 'hecmtl-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pciarray
    #: - [UDP] pciarray
    pciarray: 'AppType' = 1552, 'pciarray', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sna-cs
    #: - [UDP] sna-cs
    sna_cs: 'AppType' = 1553, 'sna-cs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CACI Products Company License Manager
    #: - [UDP] CACI Products Company License Manager
    caci_lm: 'AppType' = 1554, 'caci-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] livelan
    #: - [UDP] livelan
    livelan: 'AppType' = 1555, 'livelan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS Private Branch Exchange IANA assigned this well-formed service
    #:   name as a replacement for "veritas_pbx".
    #: - [TCP] VERITAS Private Branch Exchange
    #: - [UDP] VERITAS Private Branch Exchange IANA assigned this well-formed service
    #:   name as a replacement for "veritas_pbx".
    #: - [UDP] VERITAS Private Branch Exchange
    veritas_pbx: 'AppType' = 1556, 'veritas-pbx', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ArborText License Manager
    #: - [UDP] ArborText License Manager
    arbortext_lm: 'AppType' = 1557, 'arbortext-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xingmpeg
    #: - [UDP] xingmpeg
    xingmpeg: 'AppType' = 1558, 'xingmpeg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] web2host
    #: - [UDP] web2host
    web2host: 'AppType' = 1559, 'web2host', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASCI-RemoteSHADOW
    #: - [UDP] ASCI-RemoteSHADOW
    asci_val: 'AppType' = 1560, 'asci-val', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] facilityview
    #: - [UDP] facilityview
    facilityview: 'AppType' = 1561, 'facilityview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pconnectmgr
    #: - [UDP] pconnectmgr
    pconnectmgr: 'AppType' = 1562, 'pconnectmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadabra License Manager
    #: - [UDP] Cadabra License Manager
    cadabra_lm: 'AppType' = 1563, 'cadabra-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pay-Per-View
    #: - [UDP] Pay-Per-View
    pay_per_view: 'AppType' = 1564, 'pay-per-view', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WinDD
    #: - [UDP] WinDD
    winddlb: 'AppType' = 1565, 'winddlb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CORELVIDEO
    #: - [UDP] CORELVIDEO
    corelvideo: 'AppType' = 1566, 'corelvideo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jlicelmd
    #: - [UDP] jlicelmd
    jlicelmd: 'AppType' = 1567, 'jlicelmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tsspmap
    #: - [UDP] tsspmap
    tsspmap: 'AppType' = 1568, 'tsspmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ets
    #: - [UDP] ets
    ets: 'AppType' = 1569, 'ets', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] orbixd
    #: - [UDP] orbixd
    orbixd: 'AppType' = 1570, 'orbixd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle Remote Data Base
    #: - [UDP] Oracle Remote Data Base
    rdb_dbs_disp: 'AppType' = 1571, 'rdb-dbs-disp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Chipcom License Manager
    #: - [UDP] Chipcom License Manager
    chip_lm: 'AppType' = 1572, 'chip-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] itscomm-ns
    #: - [UDP] itscomm-ns
    itscomm_ns: 'AppType' = 1573, 'itscomm-ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mvel-lm
    #: - [UDP] mvel-lm
    mvel_lm: 'AppType' = 1574, 'mvel-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] oraclenames
    #: - [UDP] oraclenames
    oraclenames: 'AppType' = 1575, 'oraclenames', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Moldflow License Manager
    #: - [UDP] Moldflow License Manager
    moldflow_lm: 'AppType' = 1576, 'moldflow-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hypercube-lm
    #: - [UDP] hypercube-lm
    hypercube_lm: 'AppType' = 1577, 'hypercube-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jacobus License Manager
    #: - [UDP] Jacobus License Manager
    jacobus_lm: 'AppType' = 1578, 'jacobus-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ioc-sea-lm
    #: - [UDP] ioc-sea-lm
    ioc_sea_lm: 'AppType' = 1579, 'ioc-sea-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] tn-tl-r1
    tn_tl_r1: 'AppType' = 1580, 'tn-tl-r1', TransportProtocol.get('tcp')

    #: [UDP] tn-tl-r2
    tn_tl_r2: 'AppType' = 1580, 'tn-tl-r2', TransportProtocol.get('udp')

    #: - [TCP] MIL-2045-47001
    #: - [UDP] MIL-2045-47001
    mil_2045_47001: 'AppType' = 1581, 'mil-2045-47001', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSIMS
    #: - [UDP] MSIMS
    msims: 'AppType' = 1582, 'msims', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simbaexpress
    #: - [UDP] simbaexpress
    simbaexpress: 'AppType' = 1583, 'simbaexpress', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tn-tl-fd2
    #: - [UDP] tn-tl-fd2
    tn_tl_fd2: 'AppType' = 1584, 'tn-tl-fd2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] intv
    #: - [UDP] intv
    intv: 'AppType' = 1585, 'intv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ibm-abtact
    #: - [UDP] ibm-abtact
    ibm_abtact: 'AppType' = 1586, 'ibm-abtact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pra_elmd IANA assigned this well-formed service name as a replacement
    #:   for "pra_elmd".
    #: - [TCP] pra_elmd
    #: - [UDP] pra_elmd IANA assigned this well-formed service name as a replacement
    #:   for "pra_elmd".
    #: - [UDP] pra_elmd
    pra_elmd: 'AppType' = 1587, 'pra-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] triquest-lm
    #: - [UDP] triquest-lm
    triquest_lm: 'AppType' = 1588, 'triquest-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VQP
    #: - [UDP] VQP
    vqp: 'AppType' = 1589, 'vqp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gemini-lm
    #: - [UDP] gemini-lm
    gemini_lm: 'AppType' = 1590, 'gemini-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncpm-pm
    #: - [UDP] ncpm-pm
    ncpm_pm: 'AppType' = 1591, 'ncpm-pm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] commonspace
    #: - [UDP] commonspace
    commonspace: 'AppType' = 1592, 'commonspace', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mainsoft-lm
    #: - [UDP] mainsoft-lm
    mainsoft_lm: 'AppType' = 1593, 'mainsoft-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sixtrak
    #: - [UDP] sixtrak
    sixtrak: 'AppType' = 1594, 'sixtrak', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] radio
    #: - [UDP] radio
    radio: 'AppType' = 1595, 'radio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] radio-sm
    radio_sm: 'AppType' = 1596, 'radio-sm', TransportProtocol.get('tcp')

    #: [UDP] radio-bc
    radio_bc: 'AppType' = 1596, 'radio-bc', TransportProtocol.get('udp')

    #: - [TCP] orbplus-iiop
    #: - [UDP] orbplus-iiop
    orbplus_iiop: 'AppType' = 1597, 'orbplus-iiop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] picknfs
    #: - [UDP] picknfs
    picknfs: 'AppType' = 1598, 'picknfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simbaservices
    #: - [UDP] simbaservices
    simbaservices: 'AppType' = 1599, 'simbaservices', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] issd
    #: - [UDP] issd
    issd: 'AppType' = 1600, 'issd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aas
    #: - [UDP] aas
    aas: 'AppType' = 1601, 'aas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] inspect
    #: - [UDP] inspect
    inspect: 'AppType' = 1602, 'inspect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pickodbc
    #: - [UDP] pickodbc
    picodbc: 'AppType' = 1603, 'picodbc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] icabrowser
    #: - [UDP] icabrowser
    icabrowser: 'AppType' = 1604, 'icabrowser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Salutation Manager (Salutation Protocol)
    #: - [UDP] Salutation Manager (Salutation Protocol)
    slp: 'AppType' = 1605, 'slp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Salutation Manager (SLM-API)
    #: - [UDP] Salutation Manager (SLM-API)
    slm_api: 'AppType' = 1606, 'slm-api', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] stt
    #: - [UDP] stt
    stt: 'AppType' = 1607, 'stt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Smart Corp. License Manager
    #: - [UDP] Smart Corp. License Manager
    smart_lm: 'AppType' = 1608, 'smart-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isysg-lm
    #: - [UDP] isysg-lm
    isysg_lm: 'AppType' = 1609, 'isysg-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] taurus-wh
    #: - [UDP] taurus-wh
    taurus_wh: 'AppType' = 1610, 'taurus-wh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Inter Library Loan
    #: - [UDP] Inter Library Loan
    ill: 'AppType' = 1611, 'ill', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetBill Transaction Server
    #: - [UDP] NetBill Transaction Server
    netbill_trans: 'AppType' = 1612, 'netbill-trans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetBill Key Repository
    #: - [UDP] NetBill Key Repository
    netbill_keyrep: 'AppType' = 1613, 'netbill-keyrep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetBill Credential Server
    #: - [UDP] NetBill Credential Server
    netbill_cred: 'AppType' = 1614, 'netbill-cred', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetBill Authorization Server
    #: - [UDP] NetBill Authorization Server
    netbill_auth: 'AppType' = 1615, 'netbill-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetBill Product Server
    #: - [UDP] NetBill Product Server
    netbill_prod: 'AppType' = 1616, 'netbill-prod', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nimrod Inter-Agent Communication
    #: - [UDP] Nimrod Inter-Agent Communication
    nimrod_agent: 'AppType' = 1617, 'nimrod-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] skytelnet
    #: - [UDP] skytelnet
    skytelnet: 'AppType' = 1618, 'skytelnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xs-openstorage
    #: - [UDP] xs-openstorage
    xs_openstorage: 'AppType' = 1619, 'xs-openstorage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] faxportwinport
    #: - [UDP] faxportwinport
    faxportwinport: 'AppType' = 1620, 'faxportwinport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] softdataphone
    #: - [UDP] softdataphone
    softdataphone: 'AppType' = 1621, 'softdataphone', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ontime
    #: - [UDP] ontime
    ontime: 'AppType' = 1622, 'ontime', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jaleosnd
    #: - [UDP] jaleosnd
    jaleosnd: 'AppType' = 1623, 'jaleosnd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] udp-sr-port
    #: - [UDP] udp-sr-port
    udp_sr_port: 'AppType' = 1624, 'udp-sr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] svs-omagent
    #: - [UDP] svs-omagent
    svs_omagent: 'AppType' = 1625, 'svs-omagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Shockwave
    #: - [UDP] Shockwave
    shockwave: 'AppType' = 1626, 'shockwave', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] T.128 Gateway
    #: - [UDP] T.128 Gateway
    t128_gateway: 'AppType' = 1627, 't128-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LonTalk normal
    #: - [UDP] LonTalk normal
    lontalk_norm: 'AppType' = 1628, 'lontalk-norm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LonTalk urgent
    #: - [UDP] LonTalk urgent
    lontalk_urgnt: 'AppType' = 1629, 'lontalk-urgnt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle Net8 Cman
    #: - [UDP] Oracle Net8 Cman
    oraclenet8cman: 'AppType' = 1630, 'oraclenet8cman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Visit view
    #: - [UDP] Visit view
    visitview: 'AppType' = 1631, 'visitview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PAMMRATC
    #: - [UDP] PAMMRATC
    pammratc: 'AppType' = 1632, 'pammratc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PAMMRPC
    #: - [UDP] PAMMRPC
    pammrpc: 'AppType' = 1633, 'pammrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Log On America Probe
    #: - [UDP] Log On America Probe
    loaprobe: 'AppType' = 1634, 'loaprobe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDB Server 1
    #: - [UDP] EDB Server 1
    edb_server1: 'AppType' = 1635, 'edb-server1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISP shared public data control
    #: - [UDP] ISP shared public data control
    isdc: 'AppType' = 1636, 'isdc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISP shared local data control
    #: - [UDP] ISP shared local data control
    islc: 'AppType' = 1637, 'islc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISP shared management control
    #: - [UDP] ISP shared management control
    ismc: 'AppType' = 1638, 'ismc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cert-initiator
    #: - [UDP] cert-initiator
    cert_initiator: 'AppType' = 1639, 'cert-initiator', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cert-responder
    #: - [UDP] cert-responder
    cert_responder: 'AppType' = 1640, 'cert-responder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InVision
    #: - [UDP] InVision
    invision: 'AppType' = 1641, 'invision', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isis-am
    #: - [UDP] isis-am
    isis_am: 'AppType' = 1642, 'isis-am', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isis-ambc
    #: - [UDP] isis-ambc
    isis_ambc: 'AppType' = 1643, 'isis-ambc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Satellite-data Acquisition System 4
    #: - [UDP] Satellite-data Acquisition System 4
    saiseh: 'AppType' = 1644, 'saiseh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SightLine
    #: - [UDP] SightLine
    sightline: 'AppType' = 1645, 'sightline', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sa-msg-port
    #: - [UDP] sa-msg-port
    sa_msg_port: 'AppType' = 1646, 'sa-msg-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rsap
    #: - [UDP] rsap
    rsap: 'AppType' = 1647, 'rsap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] concurrent-lm
    #: - [UDP] concurrent-lm
    concurrent_lm: 'AppType' = 1648, 'concurrent-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] kermit
    #: - [UDP] kermit
    kermit: 'AppType' = 1649, 'kermit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nkdn
    #: - [UDP] nkd
    nkd: 'AppType' = 1650, 'nkd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] shiva_confsrvr IANA assigned this well-formed service name as a
    #:   replacement for "shiva_confsrvr".
    #: - [TCP] shiva_confsrvr
    #: - [UDP] shiva_confsrvr IANA assigned this well-formed service name as a
    #:   replacement for "shiva_confsrvr".
    #: - [UDP] shiva_confsrvr
    shiva_confsrvr: 'AppType' = 1651, 'shiva-confsrvr', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] xnmp
    #: - [UDP] xnmp
    xnmp: 'AppType' = 1652, 'xnmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] alphatech-lm
    #: - [UDP] alphatech-lm
    alphatech_lm: 'AppType' = 1653, 'alphatech-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] stargatealerts
    #: - [UDP] stargatealerts
    stargatealerts: 'AppType' = 1654, 'stargatealerts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dec-mbadmin
    #: - [UDP] dec-mbadmin
    dec_mbadmin: 'AppType' = 1655, 'dec-mbadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dec-mbadmin-h
    #: - [UDP] dec-mbadmin-h
    dec_mbadmin_h: 'AppType' = 1656, 'dec-mbadmin-h', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fujitsu-mmpdc
    #: - [UDP] fujitsu-mmpdc
    fujitsu_mmpdc: 'AppType' = 1657, 'fujitsu-mmpdc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sixnetudr
    #: - [UDP] sixnetudr
    sixnetudr: 'AppType' = 1658, 'sixnetudr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Silicon Grail License Manager
    #: - [UDP] Silicon Grail License Manager
    sg_lm: 'AppType' = 1659, 'sg-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] skip-mc-gikreq
    #: - [UDP] skip-mc-gikreq
    skip_mc_gikreq: 'AppType' = 1660, 'skip-mc-gikreq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-1
    #: - [UDP] netview-aix-1
    netview_aix_1: 'AppType' = 1661, 'netview-aix-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-2
    #: - [UDP] netview-aix-2
    netview_aix_2: 'AppType' = 1662, 'netview-aix-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-3
    #: - [UDP] netview-aix-3
    netview_aix_3: 'AppType' = 1663, 'netview-aix-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-4
    #: - [UDP] netview-aix-4
    netview_aix_4: 'AppType' = 1664, 'netview-aix-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-5
    #: - [UDP] netview-aix-5
    netview_aix_5: 'AppType' = 1665, 'netview-aix-5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-6
    #: - [UDP] netview-aix-6
    netview_aix_6: 'AppType' = 1666, 'netview-aix-6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-7
    #: - [UDP] netview-aix-7
    netview_aix_7: 'AppType' = 1667, 'netview-aix-7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-8
    #: - [UDP] netview-aix-8
    netview_aix_8: 'AppType' = 1668, 'netview-aix-8', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-9
    #: - [UDP] netview-aix-9
    netview_aix_9: 'AppType' = 1669, 'netview-aix-9', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-10
    #: - [UDP] netview-aix-10
    netview_aix_10: 'AppType' = 1670, 'netview-aix-10', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-11
    #: - [UDP] netview-aix-11
    netview_aix_11: 'AppType' = 1671, 'netview-aix-11', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netview-aix-12
    #: - [UDP] netview-aix-12
    netview_aix_12: 'AppType' = 1672, 'netview-aix-12', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intel Proshare Multicast
    #: - [UDP] Intel Proshare Multicast
    proshare_mc_1: 'AppType' = 1673, 'proshare-mc-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intel Proshare Multicast
    #: - [UDP] Intel Proshare Multicast
    proshare_mc_2: 'AppType' = 1674, 'proshare-mc-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pacific Data Products
    #: - [UDP] Pacific Data Products
    pdp: 'AppType' = 1675, 'pdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] netcomm1
    netcomm1: 'AppType' = 1676, 'netcomm1', TransportProtocol.get('tcp')

    #: [UDP] netcomm2
    netcomm2: 'AppType' = 1676, 'netcomm2', TransportProtocol.get('udp')

    #: - [TCP] groupwise
    #: - [UDP] groupwise
    groupwise: 'AppType' = 1677, 'groupwise', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] prolink
    #: - [UDP] prolink
    prolink: 'AppType' = 1678, 'prolink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] darcorp-lm
    #: - [UDP] darcorp-lm
    darcorp_lm: 'AppType' = 1679, 'darcorp-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] microcom-sbp
    #: - [UDP] microcom-sbp
    microcom_sbp: 'AppType' = 1680, 'microcom-sbp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sd-elmd
    #: - [UDP] sd-elmd
    sd_elmd: 'AppType' = 1681, 'sd-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] lanyon-lantern
    #: - [UDP] lanyon-lantern
    lanyon_lantern: 'AppType' = 1682, 'lanyon-lantern', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncpm-hip
    #: - [UDP] ncpm-hip
    ncpm_hip: 'AppType' = 1683, 'ncpm-hip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SnareSecure
    #: - [UDP] SnareSecure
    snaresecure: 'AppType' = 1684, 'snaresecure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] n2nremote
    #: - [UDP] n2nremote
    n2nremote: 'AppType' = 1685, 'n2nremote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cvmon
    #: - [UDP] cvmon
    cvmon: 'AppType' = 1686, 'cvmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nsjtp-ctrl
    #: - [UDP] nsjtp-ctrl
    nsjtp_ctrl: 'AppType' = 1687, 'nsjtp-ctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nsjtp-data
    #: - [UDP] nsjtp-data
    nsjtp_data: 'AppType' = 1688, 'nsjtp-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] firefox
    #: - [UDP] firefox
    firefox: 'AppType' = 1689, 'firefox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ng-umds
    #: - [UDP] ng-umds
    ng_umds: 'AppType' = 1690, 'ng-umds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] empire-empuma
    #: - [UDP] empire-empuma
    empire_empuma: 'AppType' = 1691, 'empire-empuma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sstsys-lm
    #: - [UDP] sstsys-lm
    sstsys_lm: 'AppType' = 1692, 'sstsys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rrirtr
    #: - [UDP] rrirtr
    rrirtr: 'AppType' = 1693, 'rrirtr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rrimwm
    #: - [UDP] rrimwm
    rrimwm: 'AppType' = 1694, 'rrimwm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rrilwm
    #: - [UDP] rrilwm
    rrilwm: 'AppType' = 1695, 'rrilwm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rrifmm
    #: - [UDP] rrifmm
    rrifmm: 'AppType' = 1696, 'rrifmm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rrisat
    #: - [UDP] rrisat
    rrisat: 'AppType' = 1697, 'rrisat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSVP-ENCAPSULATION-1
    #: - [UDP] RSVP-ENCAPSULATION-1
    rsvp_encap_1: 'AppType' = 1698, 'rsvp-encap-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSVP-ENCAPSULATION-2
    #: - [UDP] RSVP-ENCAPSULATION-2
    rsvp_encap_2: 'AppType' = 1699, 'rsvp-encap-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mps-raft
    #: - [UDP] mps-raft
    mps_raft: 'AppType' = 1700, 'mps-raft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] l2f
    #: - [UDP] l2f
    l2f: 'AppType' = 1701, 'l2f', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] l2tp
    #: - [UDP] l2tp
    l2tp: 'AppType' = 1701, 'l2tp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] deskshare
    #: - [UDP] deskshare
    deskshare: 'AppType' = 1702, 'deskshare', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hb-engine
    #: - [UDP] hb-engine
    hb_engine: 'AppType' = 1703, 'hb-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bcs-broker
    #: - [UDP] bcs-broker
    bcs_broker: 'AppType' = 1704, 'bcs-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] slingshot
    #: - [UDP] slingshot
    slingshot: 'AppType' = 1705, 'slingshot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jetform
    #: - [UDP] jetform
    jetform: 'AppType' = 1706, 'jetform', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vdmplay
    #: - [UDP] vdmplay
    vdmplay: 'AppType' = 1707, 'vdmplay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gat-lmd
    #: - [UDP] gat-lmd
    gat_lmd: 'AppType' = 1708, 'gat-lmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] centra
    #: - [UDP] centra
    centra: 'AppType' = 1709, 'centra', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] impera
    #: - [UDP] impera
    impera: 'AppType' = 1710, 'impera', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pptconference
    #: - [UDP] pptconference
    pptconference: 'AppType' = 1711, 'pptconference', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] resource monitoring service
    #: - [UDP] resource monitoring service
    registrar: 'AppType' = 1712, 'registrar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConferenceTalk
    #: - [UDP] ConferenceTalk
    conferencetalk: 'AppType' = 1713, 'conferencetalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sesi-lm
    #: - [UDP] sesi-lm
    sesi_lm: 'AppType' = 1714, 'sesi-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] houdini-lm
    #: - [UDP] houdini-lm
    houdini_lm: 'AppType' = 1715, 'houdini-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xmsg
    #: - [UDP] xmsg
    xmsg: 'AppType' = 1716, 'xmsg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fj-hdnet
    #: - [UDP] fj-hdnet
    fj_hdnet: 'AppType' = 1717, 'fj-hdnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.323 Multicast Gatekeeper Discover
    #: - [UDP] H.323 Multicast Gatekeeper Discover
    h323gatedisc: 'AppType' = 1718, 'h323gatedisc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.323 Unicast Gatekeeper Signaling
    #: - [UDP] H.323 Unicast Gatekeeper Signaling
    h323gatestat: 'AppType' = 1719, 'h323gatestat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.323 Call Control Signalling
    #: - [UDP] H.323 Call Control Signalling
    #: - [SCTP] H.323 Call Control
    h323hostcall: 'AppType' = 1720, 'h323hostcall', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] caicci
    #: - [UDP] caicci
    caicci: 'AppType' = 1721, 'caicci', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HKS License Manager
    #: - [UDP] HKS License Manager
    hks_lm: 'AppType' = 1722, 'hks-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pptp [:rfc:`2637`]
    #: - [UDP] pptp [:rfc:`2637`]
    pptp: 'AppType' = 1723, 'pptp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] csbphonemaster
    #: - [UDP] csbphonemaster
    csbphonemaster: 'AppType' = 1724, 'csbphonemaster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iden-ralp
    #: - [UDP] iden-ralp
    iden_ralp: 'AppType' = 1725, 'iden-ralp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBERIAGAMES
    #: - [UDP] IBERIAGAMES
    iberiagames: 'AppType' = 1726, 'iberiagames', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] winddx
    #: - [UDP] winddx
    winddx: 'AppType' = 1727, 'winddx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TELINDUS
    #: - [UDP] TELINDUS
    telindus: 'AppType' = 1728, 'telindus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CityNL License Management
    #: - [UDP] CityNL License Management
    citynl: 'AppType' = 1729, 'citynl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] roketz
    #: - [UDP] roketz
    roketz: 'AppType' = 1730, 'roketz', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSICCP
    #: - [UDP] MSICCP
    msiccp: 'AppType' = 1731, 'msiccp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proxim
    #: - [UDP] proxim
    proxim: 'AppType' = 1732, 'proxim', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SIMS - SIIPAT Protocol for Alarm Transmission
    #: - [UDP] SIMS - SIIPAT Protocol for Alarm Transmission
    siipat: 'AppType' = 1733, 'siipat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Camber Corporation License Management
    #: - [UDP] Camber Corporation License Management
    cambertx_lm: 'AppType' = 1734, 'cambertx-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PrivateChat
    #: - [UDP] PrivateChat
    privatechat: 'AppType' = 1735, 'privatechat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] street-stream
    #: - [UDP] street-stream
    street_stream: 'AppType' = 1736, 'street-stream', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ultimad
    #: - [UDP] ultimad
    ultimad: 'AppType' = 1737, 'ultimad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GameGen1
    #: - [UDP] GameGen1
    gamegen1: 'AppType' = 1738, 'gamegen1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] webaccess
    #: - [UDP] webaccess
    webaccess: 'AppType' = 1739, 'webaccess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] encore
    #: - [UDP] encore
    encore: 'AppType' = 1740, 'encore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco-net-mgmt
    #: - [UDP] cisco-net-mgmt
    cisco_net_mgmt: 'AppType' = 1741, 'cisco-net-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com-nsd
    #: - [UDP] 3Com-nsd
    AppType_3com_nsd: 'AppType' = 1742, '3com-nsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cinema Graphics License Manager
    #: - [UDP] Cinema Graphics License Manager
    cinegrfx_lm: 'AppType' = 1743, 'cinegrfx-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncpm-ft
    #: - [UDP] ncpm-ft
    ncpm_ft: 'AppType' = 1744, 'ncpm-ft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] remote-winsock
    #: - [UDP] remote-winsock
    remote_winsock: 'AppType' = 1745, 'remote-winsock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ftrapid-1
    #: - [UDP] ftrapid-1
    ftrapid_1: 'AppType' = 1746, 'ftrapid-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ftrapid-2
    #: - [UDP] ftrapid-2
    ftrapid_2: 'AppType' = 1747, 'ftrapid-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] oracle-em1
    #: - [UDP] oracle-em1
    oracle_em1: 'AppType' = 1748, 'oracle-em1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aspen-services
    #: - [UDP] aspen-services
    aspen_services: 'AppType' = 1749, 'aspen-services', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Socket Library's PortMaster
    #: - [UDP] Simple Socket Library's PortMaster
    sslp: 'AppType' = 1750, 'sslp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SwiftNet
    #: - [UDP] SwiftNet
    swiftnet: 'AppType' = 1751, 'swiftnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Leap of Faith Research License Manager
    #: - [UDP] Leap of Faith Research License Manager
    lofr_lm: 'AppType' = 1752, 'lofr-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Predatar Comms Service
    predatar_comms: 'AppType' = 1753, 'predatar-comms', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_1753: 'AppType' = 1753, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] oracle-em2
    #: - [UDP] oracle-em2
    oracle_em2: 'AppType' = 1754, 'oracle-em2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ms-streaming
    #: - [UDP] ms-streaming
    ms_streaming: 'AppType' = 1755, 'ms-streaming', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] capfast-lmd
    #: - [UDP] capfast-lmd
    capfast_lmd: 'AppType' = 1756, 'capfast-lmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cnhrp
    #: - [UDP] cnhrp
    cnhrp: 'AppType' = 1757, 'cnhrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tftp-mcast
    #: - [UDP] tftp-mcast
    tftp_mcast: 'AppType' = 1758, 'tftp-mcast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPSS License Manager
    #: - [UDP] SPSS License Manager
    spss_lm: 'AppType' = 1759, 'spss-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] www-ldap-gw
    #: - [UDP] www-ldap-gw
    www_ldap_gw: 'AppType' = 1760, 'www-ldap-gw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-0
    #: - [UDP] cft-0
    cft_0: 'AppType' = 1761, 'cft-0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-1
    #: - [UDP] cft-1
    cft_1: 'AppType' = 1762, 'cft-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-2
    #: - [UDP] cft-2
    cft_2: 'AppType' = 1763, 'cft-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-3
    #: - [UDP] cft-3
    cft_3: 'AppType' = 1764, 'cft-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-4
    #: - [UDP] cft-4
    cft_4: 'AppType' = 1765, 'cft-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-5
    #: - [UDP] cft-5
    cft_5: 'AppType' = 1766, 'cft-5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-6
    #: - [UDP] cft-6
    cft_6: 'AppType' = 1767, 'cft-6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cft-7
    #: - [UDP] cft-7
    cft_7: 'AppType' = 1768, 'cft-7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bmc-net-adm
    #: - [UDP] bmc-net-adm
    bmc_net_adm: 'AppType' = 1769, 'bmc-net-adm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bmc-net-svc
    #: - [UDP] bmc-net-svc
    bmc_net_svc: 'AppType' = 1770, 'bmc-net-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vaultbase
    #: - [UDP] vaultbase
    vaultbase: 'AppType' = 1771, 'vaultbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EssWeb Gateway
    #: - [UDP] EssWeb Gateway
    essweb_gw: 'AppType' = 1772, 'essweb-gw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KMSControl
    #: - [UDP] KMSControl
    kmscontrol: 'AppType' = 1773, 'kmscontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] global-dtserv
    #: - [UDP] global-dtserv
    global_dtserv: 'AppType' = 1774, 'global-dtserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] data interchange between visual processing containers
    vdab: 'AppType' = 1775, 'vdab', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_1775: 'AppType' = 1775, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Federal Emergency Management Information System
    #: - [UDP] Federal Emergency Management Information System
    femis: 'AppType' = 1776, 'femis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] powerguardian
    #: - [UDP] powerguardian
    powerguardian: 'AppType' = 1777, 'powerguardian', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] prodigy-internet
    #: - [UDP] prodigy-internet
    prodigy_intrnet: 'AppType' = 1778, 'prodigy-intrnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pharmasoft
    #: - [UDP] pharmasoft
    pharmasoft: 'AppType' = 1779, 'pharmasoft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dpkeyserv
    #: - [UDP] dpkeyserv
    dpkeyserv: 'AppType' = 1780, 'dpkeyserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] answersoft-lm
    #: - [UDP] answersoft-lm
    answersoft_lm: 'AppType' = 1781, 'answersoft-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp-hcip
    #: - [UDP] hp-hcip
    hp_hcip: 'AppType' = 1782, 'hp-hcip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Decomissioned Port 04/14/00, ms
    decomissioned_port_04_14_00_ms: 'AppType' = 1783, 'decomissioned_port_04_14_00_ms', TransportProtocol.get('undefined')

    #: - [TCP] Finle License Manager
    #: - [UDP] Finle License Manager
    finle_lm: 'AppType' = 1784, 'finle-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wind River Systems License Manager
    #: - [UDP] Wind River Systems License Manager
    windlm: 'AppType' = 1785, 'windlm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] funk-logger
    #: - [UDP] funk-logger
    funk_logger: 'AppType' = 1786, 'funk-logger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] funk-license
    #: - [UDP] funk-license
    funk_license: 'AppType' = 1787, 'funk-license', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] psmond
    #: - [UDP] psmond
    psmond: 'AppType' = 1788, 'psmond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hello
    #: - [UDP] hello
    hello: 'AppType' = 1789, 'hello', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Networked Media Streaming Protocol
    #: - [UDP] Networked Media Streaming Protocol
    nmsp_537: 'AppType' = 537, 'nmsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Narrative Media Streaming Protocol
    #: - [UDP] Narrative Media Streaming Protocol
    nmsp_1790: 'AppType' = 1790, 'nmsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EA1
    #: - [UDP] EA1
    ea1: 'AppType' = 1791, 'ea1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ibm-dt-2
    #: - [UDP] ibm-dt-2
    ibm_dt_2: 'AppType' = 1792, 'ibm-dt-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rsc-robot
    #: - [UDP] rsc-robot
    rsc_robot: 'AppType' = 1793, 'rsc-robot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cera-bcm
    #: - [UDP] cera-bcm
    cera_bcm: 'AppType' = 1794, 'cera-bcm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dpi-proxy
    #: - [UDP] dpi-proxy
    dpi_proxy: 'AppType' = 1795, 'dpi-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vocaltec Server Administration
    #: - [UDP] Vocaltec Server Administration
    vocaltec_admin: 'AppType' = 1796, 'vocaltec-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Universal Management Architecture
    #: - [UDP] Universal Management Architecture
    uma_144: 'AppType' = 144, 'uma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UMA
    #: - [UDP] UMA
    uma_1797: 'AppType' = 1797, 'uma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Event Transfer Protocol
    #: - [UDP] Event Transfer Protocol
    etp: 'AppType' = 1798, 'etp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETRISK
    #: - [UDP] NETRISK
    netrisk: 'AppType' = 1799, 'netrisk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSYS-License manager
    #: - [UDP] ANSYS-License manager
    ansys_lm: 'AppType' = 1800, 'ansys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Message Que
    #: - [UDP] Microsoft Message Que
    msmq: 'AppType' = 1801, 'msmq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConComp1
    #: - [UDP] ConComp1
    concomp1: 'AppType' = 1802, 'concomp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP-HCIP-GWY
    #: - [UDP] HP-HCIP-GWY
    hp_hcip_gwy: 'AppType' = 1803, 'hp-hcip-gwy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ENL
    #: - [UDP] ENL
    enl: 'AppType' = 1804, 'enl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ENL-Name
    #: - [UDP] ENL-Name
    enl_name: 'AppType' = 1805, 'enl-name', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Musiconline
    #: - [UDP] Musiconline
    musiconline: 'AppType' = 1806, 'musiconline', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Hot Standby Protocol
    #: - [UDP] Fujitsu Hot Standby Protocol
    fhsp: 'AppType' = 1807, 'fhsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle-VP2
    #: - [UDP] Oracle-VP2
    oracle_vp2: 'AppType' = 1808, 'oracle-vp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle-VP1
    #: - [UDP] Oracle-VP1
    oracle_vp1: 'AppType' = 1809, 'oracle-vp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jerand License Manager
    #: - [UDP] Jerand License Manager
    jerand_lm: 'AppType' = 1810, 'jerand-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scientia-SDB
    #: - [UDP] Scientia-SDB
    scientia_sdb: 'AppType' = 1811, 'scientia-sdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADIUS [:rfc:`2865`]
    #: - [UDP] RADIUS [:rfc:`2865`]
    radius: 'AppType' = 1812, 'radius', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADIUS Accounting [:rfc:`2866`]
    #: - [UDP] RADIUS Accounting [:rfc:`2866`]
    radius_acct: 'AppType' = 1813, 'radius-acct', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TDP Suite
    #: - [UDP] TDP Suite
    tdp_suite: 'AppType' = 1814, 'tdp-suite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Manufacturing messaging protocol for factory transmission
    #: - [UDP] Manufacturing messaging protocol for factory transmission
    mmpft: 'AppType' = 1815, 'mmpft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HARP
    #: - [UDP] HARP
    harp: 'AppType' = 1816, 'harp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RKB-OSCS
    #: - [UDP] RKB-OSCS
    rkb_oscs: 'AppType' = 1817, 'rkb-oscs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enhanced Trivial File Transfer Protocol
    #: - [UDP] Enhanced Trivial File Transfer Protocol
    etftp: 'AppType' = 1818, 'etftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Plato License Manager
    #: - [UDP] Plato License Manager
    plato_lm: 'AppType' = 1819, 'plato-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mcagent
    #: - [UDP] mcagent
    mcagent: 'AppType' = 1820, 'mcagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] donnyworld
    #: - [UDP] donnyworld
    donnyworld: 'AppType' = 1821, 'donnyworld', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] es-elmd
    #: - [UDP] es-elmd
    es_elmd: 'AppType' = 1822, 'es-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unisys Natural Language License Manager
    #: - [UDP] Unisys Natural Language License Manager
    unisys_lm: 'AppType' = 1823, 'unisys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] metrics-pas
    #: - [UDP] metrics-pas
    metrics_pas: 'AppType' = 1824, 'metrics-pas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirecPC Video
    #: - [UDP] DirecPC Video
    direcpc_video: 'AppType' = 1825, 'direcpc-video', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDT
    #: - [UDP] ARDT
    ardt: 'AppType' = 1826, 'ardt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASI
    #: - [UDP] ASI
    asi: 'AppType' = 1827, 'asi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] itm-mcell-u
    #: - [UDP] itm-mcell-u
    itm_mcell_u: 'AppType' = 1828, 'itm-mcell-u', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optika eMedia
    #: - [UDP] Optika eMedia
    optika_emedia: 'AppType' = 1829, 'optika-emedia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle Net8 CMan Admin
    #: - [UDP] Oracle Net8 CMan Admin
    net8_cman: 'AppType' = 1830, 'net8-cman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Myrtle
    #: - [UDP] Myrtle
    myrtle: 'AppType' = 1831, 'myrtle', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ThoughtTreasure
    #: - [UDP] ThoughtTreasure
    tht_treasure: 'AppType' = 1832, 'tht-treasure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] udpradio
    #: - [UDP] udpradio
    udpradio: 'AppType' = 1833, 'udpradio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDUS Unicast
    #: - [UDP] ARDUS Unicast
    ardusuni: 'AppType' = 1834, 'ardusuni', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARDUS Multicast
    #: - [UDP] ARDUS Multicast
    ardusmul: 'AppType' = 1835, 'ardusmul', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ste-smsc
    #: - [UDP] ste-smsc
    ste_smsc: 'AppType' = 1836, 'ste-smsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] csoft1
    #: - [UDP] csoft1
    csoft1: 'AppType' = 1837, 'csoft1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TALNET
    #: - [UDP] TALNET
    talnet: 'AppType' = 1838, 'talnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netopia-vo1
    #: - [UDP] netopia-vo1
    netopia_vo1: 'AppType' = 1839, 'netopia-vo1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netopia-vo2
    #: - [UDP] netopia-vo2
    netopia_vo2: 'AppType' = 1840, 'netopia-vo2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netopia-vo3
    #: - [UDP] netopia-vo3
    netopia_vo3: 'AppType' = 1841, 'netopia-vo3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netopia-vo4
    #: - [UDP] netopia-vo4
    netopia_vo4: 'AppType' = 1842, 'netopia-vo4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netopia-vo5
    #: - [UDP] netopia-vo5
    netopia_vo5: 'AppType' = 1843, 'netopia-vo5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirecPC-DLL
    #: - [UDP] DirecPC-DLL
    direcpc_dll: 'AppType' = 1844, 'direcpc-dll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] altalink
    #: - [UDP] altalink
    altalink: 'AppType' = 1845, 'altalink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tunstall PNC
    #: - [UDP] Tunstall PNC
    tunstall_pnc: 'AppType' = 1846, 'tunstall-pnc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SLP Notification [:rfc:`3082`]
    #: - [UDP] SLP Notification [:rfc:`3082`]
    slp_notify: 'AppType' = 1847, 'slp-notify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fjdocdist
    #: - [UDP] fjdocdist
    fjdocdist: 'AppType' = 1848, 'fjdocdist', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ALPHA-SMS
    #: - [UDP] ALPHA-SMS
    alpha_sms: 'AppType' = 1849, 'alpha-sms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GSI
    #: - [UDP] GSI
    gsi: 'AppType' = 1850, 'gsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ctcd
    #: - [UDP] ctcd
    ctcd: 'AppType' = 1851, 'ctcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Time
    #: - [UDP] Virtual Time
    virtual_time: 'AppType' = 1852, 'virtual-time', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VIDS-AVTP
    #: - [UDP] VIDS-AVTP
    vids_avtp: 'AppType' = 1853, 'vids-avtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Buddy Draw
    #: - [UDP] Buddy Draw
    buddy_draw: 'AppType' = 1854, 'buddy-draw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fiorano RtrSvc
    #: - [UDP] Fiorano RtrSvc
    fiorano_rtrsvc: 'AppType' = 1855, 'fiorano-rtrsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fiorano MsgSvc
    #: - [UDP] Fiorano MsgSvc
    fiorano_msgsvc: 'AppType' = 1856, 'fiorano-msgsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataCaptor
    #: - [UDP] DataCaptor
    datacaptor: 'AppType' = 1857, 'datacaptor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PrivateArk
    #: - [UDP] PrivateArk
    privateark: 'AppType' = 1858, 'privateark', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gamma Fetcher Server
    #: - [UDP] Gamma Fetcher Server
    gammafetchsvr: 'AppType' = 1859, 'gammafetchsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SunSCALAR Services
    #: - [UDP] SunSCALAR Services
    sunscalar_svc: 'AppType' = 1860, 'sunscalar-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LeCroy VICP
    #: - [UDP] LeCroy VICP
    lecroy_vicp: 'AppType' = 1861, 'lecroy-vicp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MySQL Cluster Manager Agent
    #: - [UDP] MySQL Cluster Manager Agent
    mysql_cm_agent: 'AppType' = 1862, 'mysql-cm-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSNP
    #: - [UDP] MSNP
    msnp: 'AppType' = 1863, 'msnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Paradym 31 Port
    #: - [UDP] Paradym 31 Port
    paradym_31port: 'AppType' = 1864, 'paradym-31port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ENTP
    #: - [UDP] ENTP
    entp: 'AppType' = 1865, 'entp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swrmi
    #: - [UDP] swrmi
    swrmi: 'AppType' = 1866, 'swrmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UDRIVE
    #: - [UDP] UDRIVE
    udrive: 'AppType' = 1867, 'udrive', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VizibleBrowser
    #: - [UDP] VizibleBrowser
    viziblebrowser: 'AppType' = 1868, 'viziblebrowser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TransAct
    #: - [UDP] TransAct
    transact: 'AppType' = 1869, 'transact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SunSCALAR DNS Service
    #: - [UDP] SunSCALAR DNS Service
    sunscalar_dns: 'AppType' = 1870, 'sunscalar-dns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cano Central 0
    #: - [UDP] Cano Central 0
    canocentral0: 'AppType' = 1871, 'canocentral0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cano Central 1
    #: - [UDP] Cano Central 1
    canocentral1: 'AppType' = 1872, 'canocentral1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fjmpjps
    #: - [UDP] Fjmpjps
    fjmpjps: 'AppType' = 1873, 'fjmpjps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fjswapsnp
    #: - [UDP] Fjswapsnp
    fjswapsnp: 'AppType' = 1874, 'fjswapsnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] westell stats
    #: - [UDP] westell stats
    westell_stats: 'AppType' = 1875, 'westell-stats', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ewcappsrv
    #: - [UDP] ewcappsrv
    ewcappsrv: 'AppType' = 1876, 'ewcappsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp-webqosdb
    #: - [UDP] hp-webqosdb
    hp_webqosdb: 'AppType' = 1877, 'hp-webqosdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] drmsmc
    #: - [UDP] drmsmc
    drmsmc: 'AppType' = 1878, 'drmsmc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NettGain NMS
    #: - [UDP] NettGain NMS
    nettgain_nms: 'AppType' = 1879, 'nettgain-nms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gilat VSAT Control
    #: - [UDP] Gilat VSAT Control
    vsat_control: 'AppType' = 1880, 'vsat-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM WebSphere MQ Everyplace
    #: - [UDP] IBM WebSphere MQ Everyplace
    ibm_mqseries2: 'AppType' = 1881, 'ibm-mqseries2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA eTrust Common Services
    #: - [UDP] CA eTrust Common Services
    ecsqdmn: 'AppType' = 1882, 'ecsqdmn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Internet Distance Map Svc
    #: - [UDP] Internet Distance Map Svc
    idmaps: 'AppType' = 1884, 'idmaps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Veritas Trap Server
    #: - [UDP] Veritas Trap Server
    vrtstrapserver: 'AppType' = 1885, 'vrtstrapserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Leonardo over IP
    #: - [UDP] Leonardo over IP
    leoip: 'AppType' = 1886, 'leoip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileX Listening Port
    #: - [UDP] FileX Listening Port
    filex_lport: 'AppType' = 1887, 'filex-lport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NC Config Port
    #: - [UDP] NC Config Port
    ncconfig: 'AppType' = 1888, 'ncconfig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify Web Adapter Service
    #: - [UDP] Unify Web Adapter Service
    unify_adapter: 'AppType' = 1889, 'unify-adapter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] wilkenListener
    #: - [UDP] wilkenListener
    wilkenlistener: 'AppType' = 1890, 'wilkenlistener', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ChildKey Notification
    #: - [UDP] ChildKey Notification
    childkey_notif: 'AppType' = 1891, 'childkey-notif', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ChildKey Control
    #: - [UDP] ChildKey Control
    childkey_ctrl: 'AppType' = 1892, 'childkey-ctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ELAD Protocol
    #: - [UDP] ELAD Protocol
    elad: 'AppType' = 1893, 'elad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] O2Server Port
    #: - [UDP] O2Server Port
    o2server_port: 'AppType' = 1894, 'o2server-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] unassigned
    #: - [UDP] unassigned
    unassigned_1895: 'AppType' = 1895, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] b-novative license server
    #: - [UDP] b-novative license server
    b_novative_ls: 'AppType' = 1896, 'b-novative-ls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaAgent
    #: - [UDP] MetaAgent
    metaagent: 'AppType' = 1897, 'metaagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cymtec secure management
    #: - [UDP] Cymtec secure management
    cymtec_port: 'AppType' = 1898, 'cymtec-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MC2Studios
    #: - [UDP] MC2Studios
    mc2studios: 'AppType' = 1899, 'mc2studios', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSDP
    #: - [UDP] SSDP
    ssdp: 'AppType' = 1900, 'ssdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu ICL Terminal Emulator Program A
    #: - [UDP] Fujitsu ICL Terminal Emulator Program A
    fjicl_tep_a: 'AppType' = 1901, 'fjicl-tep-a', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu ICL Terminal Emulator Program B
    #: - [UDP] Fujitsu ICL Terminal Emulator Program B
    fjicl_tep_b: 'AppType' = 1902, 'fjicl-tep-b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Local Link Name Resolution
    #: - [UDP] Local Link Name Resolution
    linkname: 'AppType' = 1903, 'linkname', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu ICL Terminal Emulator Program C
    #: - [UDP] Fujitsu ICL Terminal Emulator Program C
    fjicl_tep_c: 'AppType' = 1904, 'fjicl-tep-c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure UP.Link Gateway Protocol
    #: - [UDP] Secure UP.Link Gateway Protocol
    sugp: 'AppType' = 1905, 'sugp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TPortMapperReq
    #: - [UDP] TPortMapperReq
    tpmd: 'AppType' = 1906, 'tpmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IntraSTAR
    #: - [UDP] IntraSTAR
    intrastar: 'AppType' = 1907, 'intrastar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dawn
    #: - [UDP] Dawn
    dawn: 'AppType' = 1908, 'dawn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Global World Link
    #: - [UDP] Global World Link
    global_wlink: 'AppType' = 1909, 'global-wlink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UltraBac Software communications port
    #: - [UDP] UltraBac Software communications port
    ultrabac: 'AppType' = 1910, 'ultrabac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Starlight Networks Multimedia Transport Protocol
    #: - [UDP] Starlight Networks Multimedia Transport Protocol
    mtp: 'AppType' = 1911, 'mtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rhp-iibp
    #: - [UDP] rhp-iibp
    rhp_iibp: 'AppType' = 1912, 'rhp-iibp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] armadp
    #: - [UDP] armadp
    armadp: 'AppType' = 1913, 'armadp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Elm-Momentum
    #: - [UDP] Elm-Momentum
    elm_momentum: 'AppType' = 1914, 'elm-momentum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FACELINK
    #: - [UDP] FACELINK
    facelink: 'AppType' = 1915, 'facelink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Persoft Persona
    #: - [UDP] Persoft Persona
    persona: 'AppType' = 1916, 'persona', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nOAgent
    #: - [UDP] nOAgent
    noagent: 'AppType' = 1917, 'noagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Tivole Directory Service - NDS
    #: - [UDP] IBM Tivole Directory Service - NDS
    can_nds: 'AppType' = 1918, 'can-nds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Tivoli Directory Service - DCH
    #: - [UDP] IBM Tivoli Directory Service - DCH
    can_dch: 'AppType' = 1919, 'can-dch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Tivoli Directory Service - FERRET
    #: - [UDP] IBM Tivoli Directory Service - FERRET
    can_ferret: 'AppType' = 1920, 'can-ferret', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NoAdmin
    #: - [UDP] NoAdmin
    noadmin: 'AppType' = 1921, 'noadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tapestry
    #: - [UDP] Tapestry
    tapestry: 'AppType' = 1922, 'tapestry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPICE
    #: - [UDP] SPICE
    spice: 'AppType' = 1923, 'spice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XIIP
    #: - [UDP] XIIP
    xiip: 'AppType' = 1924, 'xiip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Surrogate Discovery Port
    #: - [UDP] Surrogate Discovery Port
    discovery_port: 'AppType' = 1925, 'discovery-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Evolution Game Server
    #: - [UDP] Evolution Game Server
    egs: 'AppType' = 1926, 'egs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Videte CIPC Port
    #: - [UDP] Videte CIPC Port
    videte_cipc: 'AppType' = 1927, 'videte-cipc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Expnd Maui Srvr Dscovr
    #: - [UDP] Expnd Maui Srvr Dscovr
    emsd_port: 'AppType' = 1928, 'emsd-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bandwiz System - Server
    #: - [UDP] Bandwiz System - Server
    bandwiz_system: 'AppType' = 1929, 'bandwiz-system', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Drive AppServer
    #: - [UDP] Drive AppServer
    driveappserver: 'AppType' = 1930, 'driveappserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMD SCHED
    #: - [UDP] AMD SCHED
    amdsched: 'AppType' = 1931, 'amdsched', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CTT Broker
    #: - [UDP] CTT Broker
    ctt_broker: 'AppType' = 1932, 'ctt-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM LM MT Agent
    #: - [UDP] IBM LM MT Agent
    xmapi: 'AppType' = 1933, 'xmapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM LM Appl Agent
    #: - [UDP] IBM LM Appl Agent
    xaapi: 'AppType' = 1934, 'xaapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Macromedia Flash Communications Server MX
    #: - [UDP] Macromedia Flash Communications server MX
    macromedia_fcs: 'AppType' = 1935, 'macromedia-fcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JetCmeServer Server Port
    #: - [UDP] JetCmeServer Server Port
    jetcmeserver: 'AppType' = 1936, 'jetcmeserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JetVWay Server Port
    #: - [UDP] JetVWay Server Port
    jwserver: 'AppType' = 1937, 'jwserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JetVWay Client Port
    #: - [UDP] JetVWay Client Port
    jwclient: 'AppType' = 1938, 'jwclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JetVision Server Port
    #: - [UDP] JetVision Server Port
    jvserver: 'AppType' = 1939, 'jvserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JetVision Client Port
    #: - [UDP] JetVision Client Port
    jvclient: 'AppType' = 1940, 'jvclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIC-Aida
    #: - [UDP] DIC-Aida
    dic_aida: 'AppType' = 1941, 'dic-aida', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real Enterprise Service
    #: - [UDP] Real Enterprise Service
    res: 'AppType' = 1942, 'res', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Beeyond Media
    #: - [UDP] Beeyond Media
    beeyond_media: 'AppType' = 1943, 'beeyond-media', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] close-combat
    #: - [UDP] close-combat
    close_combat: 'AppType' = 1944, 'close-combat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dialogic-elmd
    #: - [UDP] dialogic-elmd
    dialogic_elmd: 'AppType' = 1945, 'dialogic-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tekpls
    #: - [UDP] tekpls
    tekpls: 'AppType' = 1946, 'tekpls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SentinelSRM
    #: - [UDP] SentinelSRM
    sentinelsrm: 'AppType' = 1947, 'sentinelsrm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eye2eye
    #: - [UDP] eye2eye
    eye2eye: 'AppType' = 1948, 'eye2eye', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISMA Easdaq Live
    #: - [UDP] ISMA Easdaq Live
    ismaeasdaqlive: 'AppType' = 1949, 'ismaeasdaqlive', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISMA Easdaq Test
    #: - [UDP] ISMA Easdaq Test
    ismaeasdaqtest: 'AppType' = 1950, 'ismaeasdaqtest', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bcs-lmserver
    #: - [UDP] bcs-lmserver
    bcs_lmserver: 'AppType' = 1951, 'bcs-lmserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpnjsc
    #: - [UDP] mpnjsc
    mpnjsc: 'AppType' = 1952, 'mpnjsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rapid Base
    #: - [UDP] Rapid Base
    rapidbase: 'AppType' = 1953, 'rapidbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ABR-API (diskbridge)
    #: - [UDP] ABR-API (diskbridge)
    abr_api: 'AppType' = 1954, 'abr-api', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ABR-Secure Data (diskbridge)
    #: - [UDP] ABR-Secure Data (diskbridge)
    abr_secure: 'AppType' = 1955, 'abr-secure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vertel VMF DS
    #: - [UDP] Vertel VMF DS
    vrtl_vmf_ds: 'AppType' = 1956, 'vrtl-vmf-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] unix-status
    #: - [UDP] unix-status
    unix_status: 'AppType' = 1957, 'unix-status', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA Administration Daemon
    #: - [UDP] CA Administration Daemon
    dxadmind: 'AppType' = 1958, 'dxadmind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SIMP Channel
    #: - [UDP] SIMP Channel
    simp_all: 'AppType' = 1959, 'simp-all', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Merit DAC NASmanager
    #: - [UDP] Merit DAC NASmanager
    nasmanager: 'AppType' = 1960, 'nasmanager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BTS APPSERVER
    #: - [UDP] BTS APPSERVER
    bts_appserver: 'AppType' = 1961, 'bts-appserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BIAP-MP
    #: - [UDP] BIAP-MP
    biap_mp: 'AppType' = 1962, 'biap-mp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WebMachine
    #: - [UDP] WebMachine
    webmachine: 'AppType' = 1963, 'webmachine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SOLID E ENGINE
    #: - [UDP] SOLID E ENGINE
    solid_e_engine: 'AppType' = 1964, 'solid-e-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tivoli NPM
    #: - [UDP] Tivoli NPM
    tivoli_npm: 'AppType' = 1965, 'tivoli-npm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Slush
    #: - [UDP] Slush
    slush: 'AppType' = 1966, 'slush', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Quote
    #: - [UDP] SNS Quote
    sns_quote: 'AppType' = 1967, 'sns-quote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LIPSinc
    #: - [UDP] LIPSinc
    lipsinc: 'AppType' = 1968, 'lipsinc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LIPSinc 1
    #: - [UDP] LIPSinc 1
    lipsinc1: 'AppType' = 1969, 'lipsinc1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetOp Remote Control
    #: - [UDP] NetOp Remote Control
    netop_rc: 'AppType' = 1970, 'netop-rc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetOp School
    #: - [UDP] NetOp School
    netop_school: 'AppType' = 1971, 'netop-school', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cache
    #: - [UDP] Cache
    intersys_cache: 'AppType' = 1972, 'intersys-cache', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Link Switching Remote Access Protocol
    #: - [UDP] Data Link Switching Remote Access Protocol
    dlsrap: 'AppType' = 1973, 'dlsrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DRP
    #: - [UDP] DRP
    drp: 'AppType' = 1974, 'drp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCO Flash Agent
    #: - [UDP] TCO Flash Agent
    tcoflashagent: 'AppType' = 1975, 'tcoflashagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCO Reg Agent
    #: - [UDP] TCO Reg Agent
    tcoregagent: 'AppType' = 1976, 'tcoregagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCO Address Book
    #: - [UDP] TCO Address Book
    tcoaddressbook: 'AppType' = 1977, 'tcoaddressbook', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniSQL
    #: - [UDP] UniSQL
    unisql: 'AppType' = 1978, 'unisql', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniSQL Java
    #: - [UDP] UniSQL Java
    unisql_java: 'AppType' = 1979, 'unisql-java', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PearlDoc XACT
    #: - [UDP] PearlDoc XACT
    pearldoc_xact: 'AppType' = 1980, 'pearldoc-xact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] p2pQ
    #: - [UDP] p2pQ
    p2pq: 'AppType' = 1981, 'p2pq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Evidentiary Timestamp
    #: - [UDP] Evidentiary Timestamp
    estamp: 'AppType' = 1982, 'estamp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Loophole Test Protocol
    #: - [UDP] Loophole Test Protocol
    lhtp: 'AppType' = 1983, 'lhtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BB
    #: - [UDP] BB
    bb: 'AppType' = 1984, 'bb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hot Standby Router Protocol [:rfc:`2281`]
    #: - [UDP] Hot Standby Router Protocol [:rfc:`2281`]
    hsrp: 'AppType' = 1985, 'hsrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco license management
    #: - [UDP] cisco license management
    licensedaemon: 'AppType' = 1986, 'licensedaemon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco RSRB Priority 1 port
    #: - [UDP] cisco RSRB Priority 1 port
    tr_rsrb_p1: 'AppType' = 1987, 'tr-rsrb-p1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco RSRB Priority 2 port
    #: - [UDP] cisco RSRB Priority 2 port
    tr_rsrb_p2: 'AppType' = 1988, 'tr-rsrb-p2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco RSRB Priority 3 port
    #: - [UDP] cisco RSRB Priority 3 port
    tr_rsrb_p3: 'AppType' = 1989, 'tr-rsrb-p3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MHSnet system
    #: - [UDP] MHSnet system
    mshnet: 'AppType' = 1989, 'mshnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco STUN Priority 1 port
    #: - [UDP] cisco STUN Priority 1 port
    stun_p1: 'AppType' = 1990, 'stun-p1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco STUN Priority 2 port
    #: - [UDP] cisco STUN Priority 2 port
    stun_p2: 'AppType' = 1991, 'stun-p2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco STUN Priority 3 port
    #: - [UDP] cisco STUN Priority 3 port
    stun_p3: 'AppType' = 1992, 'stun-p3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPsendmsg
    #: - [UDP] IPsendmsg
    ipsendmsg: 'AppType' = 1992, 'ipsendmsg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco SNMP TCP port
    #: - [UDP] cisco SNMP TCP port
    snmp_tcp_port: 'AppType' = 1993, 'snmp-tcp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco serial tunnel port
    #: - [UDP] cisco serial tunnel port
    stun_port: 'AppType' = 1994, 'stun-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco perf port
    #: - [UDP] cisco perf port
    perf_port: 'AppType' = 1995, 'perf-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco Remote SRB port
    #: - [UDP] cisco Remote SRB port
    tr_rsrb_port: 'AppType' = 1996, 'tr-rsrb-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco Gateway Discovery Protocol
    #: - [UDP] cisco Gateway Discovery Protocol
    gdp_port: 'AppType' = 1997, 'gdp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco X.25 service (XOT)
    #: - [UDP] cisco X.25 service (XOT)
    x25_svc_port: 'AppType' = 1998, 'x25-svc-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cisco identification port
    #: - [UDP] cisco identification port
    tcp_id_port: 'AppType' = 1999, 'tcp-id-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco SCCP
    #: - [UDP] Cisco SCCp
    cisco_sccp: 'AppType' = 2000, 'cisco-sccp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    dc: 'AppType' = 2001, 'dc', TransportProtocol.get('tcp')

    #: [UDP] curry
    wizard: 'AppType' = 2001, 'wizard', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    globe: 'AppType' = 2002, 'globe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Brutus Server
    #: - [UDP] Brutus Server
    brutus: 'AppType' = 2003, 'brutus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    mailbox: 'AppType' = 2004, 'mailbox', TransportProtocol.get('tcp')

    #: [UDP] CCWS mm conf
    emce: 'AppType' = 2004, 'emce', TransportProtocol.get('udp')

    #: [TCP]
    berknet: 'AppType' = 2005, 'berknet', TransportProtocol.get('tcp')

    #: [UDP]
    oracle: 'AppType' = 2005, 'oracle', TransportProtocol.get('udp')

    #: [TCP]
    invokator: 'AppType' = 2006, 'invokator', TransportProtocol.get('tcp')

    #: [TCP]
    dectalk: 'AppType' = 2007, 'dectalk', TransportProtocol.get('tcp')

    #: [TCP]
    conf: 'AppType' = 2008, 'conf', TransportProtocol.get('tcp')

    #: [TCP]
    news: 'AppType' = 2009, 'news', TransportProtocol.get('tcp')

    #: [TCP]
    search: 'AppType' = 2010, 'search', TransportProtocol.get('tcp')

    #: - [UDP] IANA assigned this well-formed service name as a replacement for
    #:   "pipe_server".
    #: - [UDP]
    pipe_server: 'AppType' = 2010, 'pipe-server', TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [TCP] raid
    raid_cc: 'AppType' = 2011, 'raid-cc', TransportProtocol.get('tcp')

    #: [UDP]
    servserv: 'AppType' = 2011, 'servserv', TransportProtocol.get('udp')

    #: [TCP]
    ttyinfo: 'AppType' = 2012, 'ttyinfo', TransportProtocol.get('tcp')

    #: [UDP]
    raid_ac: 'AppType' = 2012, 'raid-ac', TransportProtocol.get('udp')

    #: [UDP]
    raid_am_2007: 'AppType' = 2007, 'raid-am', TransportProtocol.get('udp')

    #: [TCP]
    raid_am_2013: 'AppType' = 2013, 'raid-am', TransportProtocol.get('tcp')

    #: [UDP] raid
    raid_cd_2006: 'AppType' = 2006, 'raid-cd', TransportProtocol.get('udp')

    #: [UDP]
    raid_cd_2013: 'AppType' = 2013, 'raid-cd', TransportProtocol.get('udp')

    #: [TCP]
    troff: 'AppType' = 2014, 'troff', TransportProtocol.get('tcp')

    #: [UDP]
    raid_sf: 'AppType' = 2014, 'raid-sf', TransportProtocol.get('udp')

    #: [TCP]
    cypress: 'AppType' = 2015, 'cypress', TransportProtocol.get('tcp')

    #: [UDP]
    raid_cs: 'AppType' = 2015, 'raid-cs', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    bootserver: 'AppType' = 2016, 'bootserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    cypress_stat: 'AppType' = 2017, 'cypress-stat', TransportProtocol.get('tcp')

    #: [UDP]
    bootclient: 'AppType' = 2017, 'bootclient', TransportProtocol.get('udp')

    #: [UDP]
    terminaldb_2008: 'AppType' = 2008, 'terminaldb', TransportProtocol.get('udp')

    #: [TCP]
    terminaldb_2018: 'AppType' = 2018, 'terminaldb', TransportProtocol.get('tcp')

    #: [UDP]
    rellpack: 'AppType' = 2018, 'rellpack', TransportProtocol.get('udp')

    #: [UDP]
    whosockami_2009: 'AppType' = 2009, 'whosockami', TransportProtocol.get('udp')

    #: [TCP]
    whosockami_2019: 'AppType' = 2019, 'whosockami', TransportProtocol.get('tcp')

    #: [UDP]
    about: 'AppType' = 2019, 'about', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    xinupageserver: 'AppType' = 2020, 'xinupageserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    servexec: 'AppType' = 2021, 'servexec', TransportProtocol.get('tcp')

    #: [UDP]
    xinuexpansion1: 'AppType' = 2021, 'xinuexpansion1', TransportProtocol.get('udp')

    #: [TCP]
    down: 'AppType' = 2022, 'down', TransportProtocol.get('tcp')

    #: [UDP]
    xinuexpansion2: 'AppType' = 2022, 'xinuexpansion2', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    xinuexpansion3: 'AppType' = 2023, 'xinuexpansion3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    xinuexpansion4: 'AppType' = 2024, 'xinuexpansion4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP]
    ellpack: 'AppType' = 2025, 'ellpack', TransportProtocol.get('tcp')

    #: [UDP]
    xribs: 'AppType' = 2025, 'xribs', TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    scrabble: 'AppType' = 2026, 'scrabble', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    shadowserver: 'AppType' = 2027, 'shadowserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    submitserver: 'AppType' = 2028, 'submitserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hot Standby Router Protocol IPv6
    #: - [UDP] Hot Standby Router Protocol IPv6
    hsrpv6: 'AppType' = 2029, 'hsrpv6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    device2: 'AppType' = 2030, 'device2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mobrien-chat
    #: - [UDP] mobrien-chat
    mobrien_chat: 'AppType' = 2031, 'mobrien-chat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    blackboard: 'AppType' = 2032, 'blackboard', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    glogger: 'AppType' = 2033, 'glogger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    scoremgr: 'AppType' = 2034, 'scoremgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    imsldoc: 'AppType' = 2035, 'imsldoc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ethernet WS DP network
    #: - [UDP] Ethernet WS DP network
    e_dpnet: 'AppType' = 2036, 'e-dpnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APplus Application Server
    #: - [UDP] APplus Application Server
    applus: 'AppType' = 2037, 'applus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    objectmanager: 'AppType' = 2038, 'objectmanager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prizma Monitoring Service
    #: - [UDP] Prizma Monitoring Service
    prizma: 'AppType' = 2039, 'prizma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    lam: 'AppType' = 2040, 'lam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    interbase: 'AppType' = 2041, 'interbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isis
    #: - [UDP] isis
    isis: 'AppType' = 2042, 'isis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isis-bcast
    #: - [UDP] isis-bcast
    isis_bcast: 'AppType' = 2043, 'isis-bcast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    rimsl: 'AppType' = 2044, 'rimsl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    cdfunc: 'AppType' = 2045, 'cdfunc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    sdfunc: 'AppType' = 2046, 'sdfunc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Directory Location Service
    #: - [UDP] Directory Location Service
    dls_197: 'AppType' = 197, 'dls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    dls_2047: 'AppType' = 2047, 'dls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    dls_monitor: 'AppType' = 2048, 'dls-monitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    shilp: 'AppType' = 2049, 'shilp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network File System - Sun Microsystems
    #: - [UDP] Network File System - Sun Microsystems
    #: - [SCTP] Network File System [:rfc:`5665`]
    nfs: 'AppType' = 2049, 'nfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Avaya EMB Config Port
    #: - [UDP] Avaya EMB Config Port
    av_emb_config: 'AppType' = 2050, 'av-emb-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPNSDP
    #: - [UDP] EPNSDP
    epnsdp: 'AppType' = 2051, 'epnsdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] clearVisn Services Port
    #: - [UDP] clearVisn Services Port
    clearvisn: 'AppType' = 2052, 'clearvisn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lot105 DSuper Updates
    #: - [UDP] Lot105 DSuper Updates
    lot105_ds_upd: 'AppType' = 2053, 'lot105-ds-upd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Weblogin Port
    #: - [UDP] Weblogin Port
    weblogin: 'AppType' = 2054, 'weblogin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Iliad-Odyssey Protocol
    #: - [UDP] Iliad-Odyssey Protocol
    iop: 'AppType' = 2055, 'iop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OmniSky Port
    #: - [UDP] OmniSky Port
    omnisky: 'AppType' = 2056, 'omnisky', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rich Content Protocol
    #: - [UDP] Rich Content Protocol
    rich_cp: 'AppType' = 2057, 'rich-cp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NewWaveSearchables RMI
    #: - [UDP] NewWaveSearchables RMI
    newwavesearch: 'AppType' = 2058, 'newwavesearch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC Messaging Service
    #: - [UDP] BMC Messaging Service
    bmc_messaging: 'AppType' = 2059, 'bmc-messaging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Telenium Daemon IF
    #: - [UDP] Telenium Daemon IF
    teleniumdaemon: 'AppType' = 2060, 'teleniumdaemon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetMount
    #: - [UDP] NetMount
    netmount: 'AppType' = 2061, 'netmount', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICG SWP Port
    #: - [UDP] ICG SWP Port
    icg_swp: 'AppType' = 2062, 'icg-swp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICG Bridge Port
    #: - [UDP] ICG Bridge Port
    icg_bridge: 'AppType' = 2063, 'icg-bridge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICG IP Relay Port
    #: - [UDP] ICG IP Relay Port
    icg_iprelay: 'AppType' = 2064, 'icg-iprelay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Link Switch Read Port Number
    #: - [UDP] Data Link Switch Read Port Number
    dlsrpn: 'AppType' = 2065, 'dlsrpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AVM USB Remote Architecture
    #: - [UDP] AVM USB Remote Architecture
    aura: 'AppType' = 2066, 'aura', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Link Switch Write Port Number
    #: - [UDP] Data Link Switch Write Port Number
    dlswpn: 'AppType' = 2067, 'dlswpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avocent AuthSrv Protocol
    #: - [UDP] Avocent AuthSrv Protocol
    avauthsrvprtcl: 'AppType' = 2068, 'avauthsrvprtcl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTTP Event Port
    #: - [UDP] HTTP Event Port
    event_port: 'AppType' = 2069, 'event-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AH and ESP Encapsulated in UDP packet
    #: - [UDP] AH and ESP Encapsulated in UDP packet
    ah_esp_encap: 'AppType' = 2070, 'ah-esp-encap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Axon Control Protocol
    #: - [UDP] Axon Control Protocol
    acp_port: 'AppType' = 2071, 'acp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GlobeCast mSync
    #: - [UDP] GlobeCast mSync
    msync: 'AppType' = 2072, 'msync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataReel Database Socket
    #: - [UDP] DataReel Database Socket
    gxs_data_port: 'AppType' = 2073, 'gxs-data-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vertel VMF SA
    #: - [UDP] Vertel VMF SA
    vrtl_vmf_sa: 'AppType' = 2074, 'vrtl-vmf-sa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Newlix ServerWare Engine
    #: - [UDP] Newlix ServerWare Engine
    newlixengine: 'AppType' = 2075, 'newlixengine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Newlix JSPConfig
    #: - [UDP] Newlix JSPConfig
    newlixconfig: 'AppType' = 2076, 'newlixconfig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Old Tivoli Storage Manager
    #: - [UDP] Old Tivoli Storage Manager
    tsrmagt: 'AppType' = 2077, 'tsrmagt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Total Productivity Center Server
    #: - [UDP] IBM Total Productivity Center Server
    tpcsrvr: 'AppType' = 2078, 'tpcsrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDWARE Router Port
    #: - [UDP] IDWARE Router Port
    idware_router: 'AppType' = 2079, 'idware-router', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Autodesk NLM (FLEXlm)
    #: - [UDP] Autodesk NLM (FLEXlm)
    autodesk_nlm: 'AppType' = 2080, 'autodesk-nlm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KME PRINTER TRAP PORT
    #: - [UDP] KME PRINTER TRAP PORT
    kme_trap_port: 'AppType' = 2081, 'kme-trap-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Infowave Mobility Server
    #: - [UDP] Infowave Mobility Server
    infowave: 'AppType' = 2082, 'infowave', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Radius Service [:rfc:`6614`]
    #: - [UDP] Secure Radius Service [:rfc:`7360`]
    radsec: 'AppType' = 2083, 'radsec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SunCluster Geographic
    #: - [UDP] SunCluster Geographic
    sunclustergeo: 'AppType' = 2084, 'sunclustergeo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ADA Control
    #: - [UDP] ADA Control
    ada_cip: 'AppType' = 2085, 'ada-cip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GNUnet
    #: - [UDP] GNUnet
    gnunet: 'AppType' = 2086, 'gnunet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ELI - Event Logging Integration
    #: - [UDP] ELI - Event Logging Integration
    eli: 'AppType' = 2087, 'eli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IP Busy Lamp Field
    #: - [UDP] IP Busy Lamp Field
    ip_blf: 'AppType' = 2088, 'ip-blf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Security Encapsulation Protocol - SEP
    #: - [UDP] Security Encapsulation Protocol - SEP
    sep: 'AppType' = 2089, 'sep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Load Report Protocol
    #: - [UDP] Load Report Protocol
    lrp: 'AppType' = 2090, 'lrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PRP
    #: - [UDP] PRP
    prp: 'AppType' = 2091, 'prp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Descent 3
    #: - [UDP] Descent 3
    descent3: 'AppType' = 2092, 'descent3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NBX CC
    #: - [UDP] NBX CC
    nbx_cc: 'AppType' = 2093, 'nbx-cc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NBX AU
    #: - [UDP] NBX AU
    nbx_au: 'AppType' = 2094, 'nbx-au', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NBX SER
    #: - [UDP] NBX SER
    nbx_ser: 'AppType' = 2095, 'nbx-ser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NBX DIR
    #: - [UDP] NBX DIR
    nbx_dir: 'AppType' = 2096, 'nbx-dir', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jet Form Preview
    #: - [UDP] Jet Form Preview
    jetformpreview: 'AppType' = 2097, 'jetformpreview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dialog Port
    #: - [UDP] Dialog Port
    dialog_port: 'AppType' = 2098, 'dialog-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.225.0 Annex G Signalling
    #: - [UDP] H.225.0 Annex G Signalling
    h2250_annex_g: 'AppType' = 2099, 'h2250-annex-g', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Amiga Network Filesystem
    #: - [UDP] Amiga Network Filesystem
    amiganetfs: 'AppType' = 2100, 'amiganetfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rtcm-sc104
    #: - [UDP] rtcm-sc104
    rtcm_sc104: 'AppType' = 2101, 'rtcm-sc104', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zephyr server
    #: - [UDP] Zephyr server
    zephyr_srv: 'AppType' = 2102, 'zephyr-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zephyr serv-hm connection
    #: - [UDP] Zephyr serv-hm connection
    zephyr_clt: 'AppType' = 2103, 'zephyr-clt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zephyr hostmanager
    #: - [UDP] Zephyr hostmanager
    zephyr_hm: 'AppType' = 2104, 'zephyr-hm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MiniPay
    #: - [UDP] MiniPay
    minipay: 'AppType' = 2105, 'minipay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MZAP
    #: - [UDP] MZAP
    mzap: 'AppType' = 2106, 'mzap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BinTec Admin
    #: - [UDP] BinTec Admin
    bintec_admin: 'AppType' = 2107, 'bintec-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Comcam
    #: - [UDP] Comcam
    comcam: 'AppType' = 2108, 'comcam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ergolight
    #: - [UDP] Ergolight
    ergolight: 'AppType' = 2109, 'ergolight', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UMSP
    #: - [UDP] UMSP
    umsp: 'AppType' = 2110, 'umsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OPNET Dynamic Sampling Agent Transaction Protocol
    #: - [UDP] OPNET Dynamic Sampling Agent Transaction Protocol
    dsatp: 'AppType' = 2111, 'dsatp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Idonix MetaNet
    #: - [UDP] Idonix MetaNet
    idonix_metanet: 'AppType' = 2112, 'idonix-metanet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HSL StoRM
    #: - [UDP] HSL StoRM
    hsl_storm: 'AppType' = 2113, 'hsl-storm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Classical Music Meta-Data Access and Enhancement
    #: - [UDP] Classical Music Meta-Data Access and Enhancement
    ariascribe: 'AppType' = 2114, 'ariascribe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Key Distribution Manager
    #: - [UDP] Key Distribution Manager
    kdm: 'AppType' = 2115, 'kdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCOWCMR
    #: - [UDP] CCOWCMR
    ccowcmr: 'AppType' = 2116, 'ccowcmr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MENTACLIENT
    #: - [UDP] MENTACLIENT
    mentaclient: 'AppType' = 2117, 'mentaclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MENTASERVER
    #: - [UDP] MENTASERVER
    mentaserver: 'AppType' = 2118, 'mentaserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GSIGATEKEEPER
    #: - [UDP] GSIGATEKEEPER
    gsigatekeeper: 'AppType' = 2119, 'gsigatekeeper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quick Eagle Networks CP
    #: - [UDP] Quick Eagle Networks CP
    qencp: 'AppType' = 2120, 'qencp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCIENTIA-SSDB
    #: - [UDP] SCIENTIA-SSDB
    scientia_ssdb: 'AppType' = 2121, 'scientia-ssdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CauPC Remote Control
    #: - [UDP] CauPC Remote Control
    caupc_remote: 'AppType' = 2122, 'caupc-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GTP-Control Plane (3GPP)
    #: - [UDP] GTP-Control Plane (3GPP)
    gtp_control: 'AppType' = 2123, 'gtp-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ELATELINK
    #: - [UDP] ELATELINK
    elatelink: 'AppType' = 2124, 'elatelink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LOCKSTEP
    #: - [UDP] LOCKSTEP
    lockstep: 'AppType' = 2125, 'lockstep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PktCable-COPS
    #: - [UDP] PktCable-COPS
    pktcable_cops: 'AppType' = 2126, 'pktcable-cops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INDEX-PC-WB
    #: - [UDP] INDEX-PC-WB
    index_pc_wb: 'AppType' = 2127, 'index-pc-wb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Steward Control
    #: - [UDP] Net Steward Control
    net_steward: 'AppType' = 2128, 'net-steward', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cs-live.com
    #: - [UDP] cs-live.com
    cs_live: 'AppType' = 2129, 'cs-live', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XDS
    #: - [UDP] XDS
    xds: 'AppType' = 2130, 'xds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avantageb2b
    #: - [UDP] Avantageb2b
    avantageb2b: 'AppType' = 2131, 'avantageb2b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SoleraTec End Point Map
    #: - [UDP] SoleraTec End Point Map
    solera_epmap: 'AppType' = 2132, 'solera-epmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ZYMED-ZPP
    #: - [UDP] ZYMED-ZPP
    zymed_zpp: 'AppType' = 2133, 'zymed-zpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AVENUE
    #: - [UDP] AVENUE
    avenue: 'AppType' = 2134, 'avenue', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Grid Resource Information Server
    #: - [UDP] Grid Resource Information Server
    gris: 'AppType' = 2135, 'gris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APPWORXSRV
    #: - [UDP] APPWORXSRV
    appworxsrv: 'AppType' = 2136, 'appworxsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CONNECT
    #: - [UDP] CONNECT
    connect: 'AppType' = 2137, 'connect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UNBIND-CLUSTER
    #: - [UDP] UNBIND-CLUSTER
    unbind_cluster: 'AppType' = 2138, 'unbind-cluster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-AUTH
    #: - [UDP] IAS-AUTH
    ias_auth: 'AppType' = 2139, 'ias-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-REG
    #: - [UDP] IAS-REG
    ias_reg: 'AppType' = 2140, 'ias-reg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-ADMIND
    #: - [UDP] IAS-ADMIND
    ias_admind: 'AppType' = 2141, 'ias-admind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TDM OVER IP [:rfc:`5087`]
    #: - [UDP] TDM OVER IP [:rfc:`5087`]
    tdmoip: 'AppType' = 2142, 'tdmoip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Live Vault Job Control
    #: - [UDP] Live Vault Job Control
    lv_jc: 'AppType' = 2143, 'lv-jc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Live Vault Fast Object Transfer
    #: - [UDP] Live Vault Fast Object Transfer
    lv_ffx: 'AppType' = 2144, 'lv-ffx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Live Vault Remote Diagnostic Console Support
    #: - [UDP] Live Vault Remote Diagnostic Console Support
    lv_pici: 'AppType' = 2145, 'lv-pici', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Live Vault Admin Event Notification
    #: - [UDP] Live Vault Admin Event Notification
    lv_not: 'AppType' = 2146, 'lv-not', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Live Vault Authentication
    #: - [UDP] Live Vault Authentication
    lv_auth: 'AppType' = 2147, 'lv-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS UNIVERSAL COMMUNICATION LAYER
    #: - [UDP] VERITAS UNIVERSAL COMMUNICATION LAYER
    veritas_ucl: 'AppType' = 2148, 'veritas-ucl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACPTSYS
    #: - [UDP] ACPTSYS
    acptsys: 'AppType' = 2149, 'acptsys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DYNAMIC3D
    #: - [UDP] DYNAMIC3D
    dynamic3d: 'AppType' = 2150, 'dynamic3d', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DOCENT
    #: - [UDP] DOCENT
    docent: 'AppType' = 2151, 'docent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GTP-User Plane (3GPP)
    #: - [UDP] GTP-User Plane (3GPP)
    gtp_user: 'AppType' = 2152, 'gtp-user', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Control Protocol
    #: - [UDP] Control Protocol
    ctlptc: 'AppType' = 2153, 'ctlptc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Standard Protocol
    #: - [UDP] Standard Protocol
    stdptc: 'AppType' = 2154, 'stdptc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bridge Protocol
    #: - [UDP] Bridge Protocol
    brdptc: 'AppType' = 2155, 'brdptc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talari Reliable Protocol
    #: - [UDP] Talari Reliable Protocol
    trp: 'AppType' = 2156, 'trp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xerox Network Document Scan Protocol
    #: - [UDP] Xerox Network Document Scan Protocol
    xnds: 'AppType' = 2157, 'xnds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TouchNetPlus Service
    #: - [UDP] TouchNetPlus Service
    touchnetplus: 'AppType' = 2158, 'touchnetplus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GDB Remote Debug Port
    #: - [UDP] GDB Remote Debug Port
    gdbremote: 'AppType' = 2159, 'gdbremote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 2160
    #: - [UDP] APC 2160
    apc_2160: 'AppType' = 2160, 'apc-2160', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 2161
    #: - [UDP] APC 2161
    apc_2161: 'AppType' = 2161, 'apc-2161', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Navisphere
    #: - [UDP] Navisphere
    navisphere: 'AppType' = 2162, 'navisphere', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Navisphere Secure
    #: - [UDP] Navisphere Secure
    navisphere_sec: 'AppType' = 2163, 'navisphere-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic DNS Version 3
    #: - [UDP] Dynamic DNS Version 3
    ddns_v3: 'AppType' = 2164, 'ddns-v3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] X-Bone API
    #: - [UDP] X-Bone API
    x_bone_api: 'AppType' = 2165, 'x-bone-api', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iwserver
    #: - [UDP] iwserver
    iwserver: 'AppType' = 2166, 'iwserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Raw Async Serial Link
    #: - [UDP] Raw Async Serial Link
    raw_serial: 'AppType' = 2167, 'raw-serial', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] easy-soft Multiplexer
    #: - [UDP] easy-soft Multiplexer
    easy_soft_mux: 'AppType' = 2168, 'easy-soft-mux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Backbone for Academic Information Notification (BRAIN)
    #: - [UDP] Backbone for Academic Information Notification (BRAIN)
    brain: 'AppType' = 2169, 'brain', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EyeTV Server Port
    #: - [UDP] EyeTV Server Port
    eyetv: 'AppType' = 2170, 'eyetv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Firewall Storage
    #: - [UDP] MS Firewall Storage
    msfw_storage: 'AppType' = 2171, 'msfw-storage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Firewall SecureStorage
    #: - [UDP] MS Firewall SecureStorage
    msfw_s_storage: 'AppType' = 2172, 'msfw-s-storage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Firewall Replication
    #: - [UDP] MS Firewall Replication
    msfw_replica: 'AppType' = 2173, 'msfw-replica', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Firewall Intra Array
    #: - [UDP] MS Firewall Intra Array
    msfw_array: 'AppType' = 2174, 'msfw-array', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Desktop AirSync Protocol
    #: - [UDP] Microsoft Desktop AirSync Protocol
    airsync: 'AppType' = 2175, 'airsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft ActiveSync Remote API
    #: - [UDP] Microsoft ActiveSync Remote API
    rapi: 'AppType' = 2176, 'rapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qWAVE Bandwidth Estimate
    #: - [UDP] qWAVE Bandwidth Estimate
    qwave: 'AppType' = 2177, 'qwave', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Peer Services for BITS
    #: - [UDP] Peer Services for BITS
    bitspeer: 'AppType' = 2178, 'bitspeer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft RDP for virtual machines
    #: - [UDP] Microsoft RDP for virtual machines
    vmrdp: 'AppType' = 2179, 'vmrdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Millicent Vendor Gateway Server
    #: - [UDP] Millicent Vendor Gateway Server
    mc_gt_srv: 'AppType' = 2180, 'mc-gt-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eforward
    #: - [UDP] eforward
    eforward: 'AppType' = 2181, 'eforward', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CGN status
    #: - [UDP] CGN status
    cgn_stat: 'AppType' = 2182, 'cgn-stat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Code Green configuration
    #: - [UDP] Code Green configuration
    cgn_config: 'AppType' = 2183, 'cgn-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OnBase Distributed Disk Services
    #: - [UDP] OnBase Distributed Disk Services
    onbase_dds: 'AppType' = 2185, 'onbase-dds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Guy-Tek Automated Update Applications
    #: - [UDP] Guy-Tek Automated Update Applications
    gtaua: 'AppType' = 2186, 'gtaua', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Sepehr System Management Control
    ssmc: 'AppType' = 2187, 'ssmc', TransportProtocol.get('tcp')

    #: [UDP] Sepehr System Management Data
    ssmd: 'AppType' = 2187, 'ssmd', TransportProtocol.get('udp')

    #: [TCP] Radware Resource Pool Manager
    radware_rpm: 'AppType' = 2188, 'radware-rpm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2188: 'AppType' = 2188, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Secure Radware Resource Pool Manager
    radware_rpm_s: 'AppType' = 2189, 'radware-rpm-s', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2189: 'AppType' = 2189, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] TiVoConnect Beacon
    #: - [UDP] TiVoConnect Beacon
    tivoconnect: 'AppType' = 2190, 'tivoconnect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TvBus Messaging
    #: - [UDP] TvBus Messaging
    tvbus: 'AppType' = 2191, 'tvbus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASDIS software management
    #: - [UDP] ASDIS software management
    asdis: 'AppType' = 2192, 'asdis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dr.Web Enterprise Management Service
    #: - [UDP] Dr.Web Enterprise Management Service
    drwcs: 'AppType' = 2193, 'drwcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MNP data exchange
    #: - [UDP] MNP data exchange
    mnp_exchange: 'AppType' = 2197, 'mnp-exchange', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OneHome Remote Access
    #: - [UDP] OneHome Remote Access
    onehome_remote: 'AppType' = 2198, 'onehome-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OneHome Service Port
    #: - [UDP] OneHome Service Port
    onehome_help: 'AppType' = 2199, 'onehome-help', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_2200: 'AppType' = 2200, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Advanced Training System Program
    #: - [UDP] Advanced Training System Program
    ats: 'AppType' = 2201, 'ats', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Int. Multimedia Teleconferencing Cosortium
    #: - [UDP] Int. Multimedia Teleconferencing Cosortium
    imtc_map: 'AppType' = 2202, 'imtc-map', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] b2 Runtime Protocol
    #: - [UDP] b2 Runtime Protocol
    b2_runtime: 'AppType' = 2203, 'b2-runtime', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] b2 License Server
    #: - [UDP] b2 License Server
    b2_license: 'AppType' = 2204, 'b2-license', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Java Presentation Server
    #: - [UDP] Java Presentation Server
    jps: 'AppType' = 2205, 'jps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP OpenCall bus
    #: - [UDP] HP OpenCall bus
    hpocbus: 'AppType' = 2206, 'hpocbus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Status and Services
    #: - [UDP] HP Status and Services
    hpssd: 'AppType' = 2207, 'hpssd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP I/O Backend
    #: - [UDP] HP I/O Backend
    hpiod: 'AppType' = 2208, 'hpiod', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP RIM for Files Portal Service
    #: - [UDP] HP RIM for Files Portal Service
    rimf_ps: 'AppType' = 2209, 'rimf-ps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NOAAPORT Broadcast Network
    #: - [UDP] NOAAPORT Broadcast Network
    noaaport: 'AppType' = 2210, 'noaaport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMWIN
    #: - [UDP] EMWIN
    emwin: 'AppType' = 2211, 'emwin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LeeCO POS Server Service
    #: - [UDP] LeeCO POS Server Service
    leecoposserver: 'AppType' = 2212, 'leecoposserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kali
    #: - [UDP] Kali
    kali: 'AppType' = 2213, 'kali', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RDQ Protocol Interface
    #: - [UDP] RDQ Protocol Interface
    rpi: 'AppType' = 2214, 'rpi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPCore.co.za GPRS
    #: - [UDP] IPCore.co.za GPRS
    ipcore: 'AppType' = 2215, 'ipcore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VTU data service
    #: - [UDP] VTU data service
    vtu_comms: 'AppType' = 2216, 'vtu-comms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GoToDevice Device Management
    #: - [UDP] GoToDevice Device Management
    gotodevice: 'AppType' = 2217, 'gotodevice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bounzza IRC Proxy
    #: - [UDP] Bounzza IRC Proxy
    bounzza: 'AppType' = 2218, 'bounzza', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetIQ NCAP Protocol
    #: - [UDP] NetIQ NCAP Protocol
    netiq_ncap: 'AppType' = 2219, 'netiq-ncap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetIQ End2End
    #: - [UDP] NetIQ End2End
    netiq: 'AppType' = 2220, 'netiq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EtherNet/IP over TLS
    #: - [UDP] EtherNet/IP over DTLS
    ethernet_ip_s: 'AppType' = 2221, 'ethernet-ip-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EtherNet/IP I/O IANA assigned this well-formed service name as a
    #:   replacement for "EtherNet/IP-1".
    #: - [TCP] EtherNet/IP I/O
    #: - [UDP] EtherNet/IP I/O IANA assigned this well-formed service name as a
    #:   replacement for "EtherNet/IP-1".
    #: - [UDP] EtherNet/IP I/O
    ethernet_ip_1: 'AppType' = 2222, 'ethernet-ip-1', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Rockwell CSP2
    #: - [UDP] Rockwell CSP2
    rockwell_csp2: 'AppType' = 2223, 'rockwell-csp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Easy Flexible Internet/Multiplayer Games
    #: - [UDP] Easy Flexible Internet/Multiplayer Games
    efi_mg: 'AppType' = 2224, 'efi-mg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Resource Connection Initiation Protocol
    #: - [SCTP] Resource Connection Initiation Protocol
    rcip_itu: 'AppType' = 2225, 'rcip-itu', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_2225: 'AppType' = 2225, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Digital Instinct DRM
    #: - [UDP] Digital Instinct DRM
    di_drm: 'AppType' = 2226, 'di-drm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DI Messaging Service
    #: - [UDP] DI Messaging Service
    di_msg: 'AppType' = 2227, 'di-msg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eHome Message Server
    #: - [UDP] eHome Message Server
    ehome_ms: 'AppType' = 2228, 'ehome-ms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataLens Service
    #: - [UDP] DataLens Service
    datalens: 'AppType' = 2229, 'datalens', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaSoft Job Queue Administration Service
    #: - [UDP] MetaSoft Job Queue Administration Service
    queueadm: 'AppType' = 2230, 'queueadm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WiMAX ASN Control Plane Protocol
    #: - [UDP] WiMAX ASN Control Plane Protocol
    wimaxasncp: 'AppType' = 2231, 'wimaxasncp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IVS Video default
    #: - [UDP] IVS Video default
    ivs_video: 'AppType' = 2232, 'ivs-video', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INFOCRYPT
    #: - [UDP] INFOCRYPT
    infocrypt: 'AppType' = 2233, 'infocrypt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirectPlay
    #: - [UDP] DirectPlay
    directplay: 'AppType' = 2234, 'directplay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sercomm-WLink
    #: - [UDP] Sercomm-WLink
    sercomm_wlink: 'AppType' = 2235, 'sercomm-wlink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nani
    #: - [UDP] Nani
    nani: 'AppType' = 2236, 'nani', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optech Port1 License Manager
    #: - [UDP] Optech Port1 License Manager
    optech_port1_lm: 'AppType' = 2237, 'optech-port1-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AVIVA SNA SERVER
    #: - [UDP] AVIVA SNA SERVER
    aviva_sna: 'AppType' = 2238, 'aviva-sna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Image Query
    #: - [UDP] Image Query
    imagequery: 'AppType' = 2239, 'imagequery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RECIPe
    #: - [UDP] RECIPe
    recipe: 'AppType' = 2240, 'recipe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IVS Daemon
    #: - [UDP] IVS Daemon
    ivsd: 'AppType' = 2241, 'ivsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Folio Remote Server
    #: - [UDP] Folio Remote Server
    foliocorp: 'AppType' = 2242, 'foliocorp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Magicom Protocol
    #: - [UDP] Magicom Protocol
    magicom: 'AppType' = 2243, 'magicom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMS Server
    #: - [UDP] NMS Server
    nmsserver: 'AppType' = 2244, 'nmsserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HaO
    #: - [UDP] HaO
    hao: 'AppType' = 2245, 'hao', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PacketCable MTA Addr Map
    #: - [UDP] PacketCable MTA Addr Map
    pc_mta_addrmap: 'AppType' = 2246, 'pc-mta-addrmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Antidote Deployment Manager Service
    #: - [UDP] Antidote Deployment Manager Service
    antidotemgrsvr: 'AppType' = 2247, 'antidotemgrsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] User Management Service
    #: - [UDP] User Management Service
    ums: 'AppType' = 2248, 'ums', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RISO File Manager Protocol
    #: - [UDP] RISO File Manager Protocol
    rfmp: 'AppType' = 2249, 'rfmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] remote-collab
    #: - [UDP] remote-collab
    remote_collab: 'AppType' = 2250, 'remote-collab', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Distributed Framework Port
    #: - [UDP] Distributed Framework Port
    dif_port: 'AppType' = 2251, 'dif-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NJENET using SSL
    #: - [UDP] NJENET using SSL
    njenet_ssl: 'AppType' = 2252, 'njenet-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTV Channel Request
    #: - [UDP] DTV Channel Request
    dtv_chan_req: 'AppType' = 2253, 'dtv-chan-req', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Seismic P.O.C. Port
    #: - [UDP] Seismic P.O.C. Port
    seispoc: 'AppType' = 2254, 'seispoc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VRTP - ViRtue Transfer Protocol
    #: - [UDP] VRTP - ViRtue Transfer Protocol
    vrtp: 'AppType' = 2255, 'vrtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCC MFP
    #: - [UDP] PCC MFP
    pcc_mfp: 'AppType' = 2256, 'pcc-mfp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simple text/file transfer
    #: - [UDP] simple text/file transfer
    simple_tx_rx: 'AppType' = 2257, 'simple-tx-rx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rotorcraft Communications Test System
    #: - [UDP] Rotorcraft Communications Test System
    rcts: 'AppType' = 2258, 'rcts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BIF identifiers resolution service
    #: - [UDP] BIF identifiers resolution service
    bid_serv: 'AppType' = 2259, 'bid-serv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 2260
    #: - [UDP] APC 2260
    apc_2260: 'AppType' = 2260, 'apc-2260', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CoMotion Master Server
    #: - [UDP] CoMotion Master Server
    comotionmaster: 'AppType' = 2261, 'comotionmaster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CoMotion Backup Server
    #: - [UDP] CoMotion Backup Server
    comotionback: 'AppType' = 2262, 'comotionback', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ECweb Configuration Service
    #: - [UDP] ECweb Configuration Service
    ecwcfg: 'AppType' = 2263, 'ecwcfg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Audio Precision Apx500 API Port 1
    #: - [UDP] Audio Precision Apx500 API Port 1
    apx500api_1: 'AppType' = 2264, 'apx500api-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Audio Precision Apx500 API Port 2
    #: - [UDP] Audio Precision Apx500 API Port 2
    apx500api_2: 'AppType' = 2265, 'apx500api-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] M-Files Server
    #: - [UDP] M-files Server
    mfserver: 'AppType' = 2266, 'mfserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OntoBroker
    #: - [UDP] OntoBroker
    ontobroker: 'AppType' = 2267, 'ontobroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMT [:rfc:`7450`]
    #: - [UDP] AMT [:rfc:`7450`]
    amt: 'AppType' = 2268, 'amt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIKEY
    #: - [UDP] MIKEY
    mikey: 'AppType' = 2269, 'mikey', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] starSchool
    #: - [UDP] starSchool
    starschool: 'AppType' = 2270, 'starschool', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Meeting Maker Scheduling
    #: - [UDP] Secure Meeting Maker Scheduling
    mmcals: 'AppType' = 2271, 'mmcals', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Meeting Maker Scheduling
    #: - [UDP] Meeting Maker Scheduling
    mmcal: 'AppType' = 2272, 'mmcal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MySQL Instance Manager
    #: - [UDP] MySQL Instance Manager
    mysql_im: 'AppType' = 2273, 'mysql-im', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCTTunneller
    #: - [UDP] PCTTunneller
    pcttunnell: 'AppType' = 2274, 'pcttunnell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iBridge Conferencing
    #: - [UDP] iBridge Conferencing
    ibridge_data: 'AppType' = 2275, 'ibridge-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iBridge Management
    #: - [UDP] iBridge Management
    ibridge_mgmt: 'AppType' = 2276, 'ibridge-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bt device control proxy
    #: - [UDP] Bt device control proxy
    bluectrlproxy: 'AppType' = 2277, 'bluectrlproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Stacked Sequences Database
    #: - [UDP] Simple Stacked Sequences Database
    s3db: 'AppType' = 2278, 's3db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xmquery
    #: - [UDP] xmquery
    xmquery: 'AppType' = 2279, 'xmquery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVPOLLER
    #: - [UDP] LNVPOLLER
    lnvpoller: 'AppType' = 2280, 'lnvpoller', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVCONSOLE
    #: - [UDP] LNVCONSOLE
    lnvconsole: 'AppType' = 2281, 'lnvconsole', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVALARM
    #: - [UDP] LNVALARM
    lnvalarm: 'AppType' = 2282, 'lnvalarm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVSTATUS
    #: - [UDP] LNVSTATUS
    lnvstatus: 'AppType' = 2283, 'lnvstatus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVMAPS
    #: - [UDP] LNVMAPS
    lnvmaps: 'AppType' = 2284, 'lnvmaps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LNVMAILMON
    #: - [UDP] LNVMAILMON
    lnvmailmon: 'AppType' = 2285, 'lnvmailmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NAS-Metering
    #: - [UDP] NAS-Metering
    nas_metering: 'AppType' = 2286, 'nas-metering', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNA
    #: - [UDP] DNA
    dna: 'AppType' = 2287, 'dna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETML
    #: - [UDP] NETML
    netml: 'AppType' = 2288, 'netml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lookup dict server
    #: - [UDP] Lookup dict server
    dict_lookup: 'AppType' = 2289, 'dict-lookup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sonus Logging Services
    #: - [UDP] Sonus Logging Services
    sonus_logging: 'AppType' = 2290, 'sonus-logging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPSON Advanced Printer Share Protocol
    #: - [UDP] EPSON Advanced Printer Share Protocol
    eapsp: 'AppType' = 2291, 'eapsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sonus Element Management Services
    #: - [UDP] Sonus Element Management Services
    mib_streaming: 'AppType' = 2292, 'mib-streaming', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Platform Debug Manager
    #: - [UDP] Network Platform Debug Manager
    npdbgmngr: 'AppType' = 2293, 'npdbgmngr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Konshus License Manager (FLEX)
    #: - [UDP] Konshus License Manager (FLEX)
    konshus_lm: 'AppType' = 2294, 'konshus-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Advant License Manager
    #: - [UDP] Advant License Manager
    advant_lm: 'AppType' = 2295, 'advant-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Theta License Manager (Rainbow)
    #: - [UDP] Theta License Manager (Rainbow)
    theta_lm: 'AppType' = 2296, 'theta-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2K DataMover 1
    #: - [UDP] D2K DataMover 1
    d2k_datamover1: 'AppType' = 2297, 'd2k-datamover1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2K DataMover 2
    #: - [UDP] D2K DataMover 2
    d2k_datamover2: 'AppType' = 2298, 'd2k-datamover2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PC Telecommute
    #: - [UDP] PC Telecommute
    pc_telecommute: 'AppType' = 2299, 'pc-telecommute', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CVMMON
    #: - [UDP] CVMMON
    cvmmon: 'AppType' = 2300, 'cvmmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq HTTP
    #: - [UDP] Compaq HTTP
    cpq_wbem: 'AppType' = 2301, 'cpq-wbem', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bindery Support
    #: - [UDP] Bindery Support
    binderysupport: 'AppType' = 2302, 'binderysupport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proxy Gateway
    #: - [UDP] Proxy Gateway
    proxy_gateway: 'AppType' = 2303, 'proxy-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Attachmate UTS
    #: - [UDP] Attachmate UTS
    attachmate_uts: 'AppType' = 2304, 'attachmate-uts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MT ScaleServer
    #: - [UDP] MT ScaleServer
    mt_scaleserver: 'AppType' = 2305, 'mt-scaleserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TAPPI BoxNet
    #: - [UDP] TAPPI BoxNet
    tappi_boxnet: 'AppType' = 2306, 'tappi-boxnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pehelp
    #: - [UDP] pehelp
    pehelp: 'AppType' = 2307, 'pehelp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sdhelp
    #: - [UDP] sdhelp
    sdhelp: 'AppType' = 2308, 'sdhelp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SD Server
    #: - [UDP] SD Server
    sdserver: 'AppType' = 2309, 'sdserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SD Client
    #: - [UDP] SD Client
    sdclient: 'AppType' = 2310, 'sdclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Service
    #: - [UDP] Message Service
    messageservice: 'AppType' = 2311, 'messageservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WANScaler Communication Service
    #: - [UDP] WANScaler Communication Service
    wanscaler: 'AppType' = 2312, 'wanscaler', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAPP (Inter Access Point Protocol)
    #: - [UDP] IAPP (Inter Access Point Protocol)
    iapp: 'AppType' = 2313, 'iapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CR WebSystems
    #: - [UDP] CR WebSystems
    cr_websystems: 'AppType' = 2314, 'cr-websystems', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Precise Sft.
    #: - [UDP] Precise Sft.
    precise_sft: 'AppType' = 2315, 'precise-sft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SENT License Manager
    #: - [UDP] SENT License Manager
    sent_lm: 'AppType' = 2316, 'sent-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Attachmate G32
    #: - [UDP] Attachmate G32
    attachmate_g32: 'AppType' = 2317, 'attachmate-g32', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cadence Control
    #: - [UDP] Cadence Control
    cadencecontrol: 'AppType' = 2318, 'cadencecontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InfoLibria
    #: - [UDP] InfoLibria
    infolibria: 'AppType' = 2319, 'infolibria', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Siebel NS
    #: - [UDP] Siebel NS
    siebel_ns: 'AppType' = 2320, 'siebel-ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RDLAP
    #: - [UDP] RDLAP
    rdlap: 'AppType' = 2321, 'rdlap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ofsd
    #: - [UDP] ofsd
    ofsd: 'AppType' = 2322, 'ofsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3d-nfsd
    #: - [UDP] 3d-nfsd
    AppType_3d_nfsd: 'AppType' = 2323, '3d-nfsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cosmocall
    #: - [UDP] Cosmocall
    cosmocall: 'AppType' = 2324, 'cosmocall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ANSYS Licensing Interconnect
    #: - [UDP] ANSYS Licensing Interconnect
    ansysli: 'AppType' = 2325, 'ansysli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDCP
    #: - [UDP] IDCP
    idcp: 'AppType' = 2326, 'idcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xingcsm
    #: - [UDP] xingcsm
    xingcsm: 'AppType' = 2327, 'xingcsm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netrix SFTM
    #: - [UDP] Netrix SFTM
    netrix_sftm: 'AppType' = 2328, 'netrix-sftm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NVD User
    #: - [UDP] NVD User
    nvd_2184: 'AppType' = 2184, 'nvd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NVD
    #: - [UDP] NVD
    nvd_2329: 'AppType' = 2329, 'nvd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TSCCHAT
    #: - [UDP] TSCCHAT
    tscchat: 'AppType' = 2330, 'tscchat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AGENTVIEW
    #: - [UDP] AGENTVIEW
    agentview: 'AppType' = 2331, 'agentview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RCC Host
    #: - [UDP] RCC Host
    rcc_host: 'AppType' = 2332, 'rcc-host', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNAPP
    #: - [UDP] SNAPP
    snapp: 'AppType' = 2333, 'snapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACE Client Auth
    #: - [UDP] ACE Client Auth
    ace_client: 'AppType' = 2334, 'ace-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACE Proxy
    #: - [UDP] ACE Proxy
    ace_proxy: 'AppType' = 2335, 'ace-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple UG Control
    #: - [UDP] Apple UG Control
    appleugcontrol: 'AppType' = 2336, 'appleugcontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ideesrv
    #: - [UDP] ideesrv
    ideesrv: 'AppType' = 2337, 'ideesrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Norton Lambert
    #: - [UDP] Norton Lambert
    norton_lambert: 'AppType' = 2338, 'norton-lambert', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com WebView
    #: - [UDP] 3Com WebView
    AppType_3com_webview: 'AppType' = 2339, '3com-webview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WRS Registry IANA assigned this well-formed service name as a
    #:   replacement for "wrs_registry".
    #: - [TCP] WRS Registry
    #: - [UDP] WRS Registry IANA assigned this well-formed service name as a
    #:   replacement for "wrs_registry".
    #: - [UDP] WRS Registry
    wrs_registry: 'AppType' = 2340, 'wrs-registry', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] XIO Status
    #: - [UDP] XIO Status
    xiostatus: 'AppType' = 2341, 'xiostatus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Seagate Manage Exec
    #: - [UDP] Seagate Manage Exec
    manage_exec: 'AppType' = 2342, 'manage-exec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nati logos
    #: - [UDP] nati logos
    nati_logos: 'AppType' = 2343, 'nati-logos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcmsys
    #: - [UDP] fcmsys
    fcmsys: 'AppType' = 2344, 'fcmsys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dbm
    #: - [UDP] dbm
    dbm: 'AppType' = 2345, 'dbm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Game Connection Port IANA assigned this well-formed service name as a
    #:   replacement for "redstorm_join".
    #: - [TCP] Game Connection Port
    #: - [UDP] Game Connection Port IANA assigned this well-formed service name as a
    #:   replacement for "redstorm_join".
    #: - [UDP] Game Connection Port
    redstorm_join: 'AppType' = 2346, 'redstorm-join', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Game Announcement and Location IANA assigned this well-formed service
    #:   name as a replacement for "redstorm_find".
    #: - [TCP] Game Announcement and Location
    #: - [UDP] Game Announcement and Location IANA assigned this well-formed service
    #:   name as a replacement for "redstorm_find".
    #: - [UDP] Game Announcement and Location
    redstorm_find: 'AppType' = 2347, 'redstorm-find', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Information to query for game status IANA assigned this well-formed
    #:   service name as a replacement for "redstorm_info".
    #: - [TCP] Information to query for game status
    #: - [UDP] Information to query for game status IANA assigned this well-formed
    #:   service name as a replacement for "redstorm_info".
    #: - [UDP] Information to query for game status
    redstorm_info: 'AppType' = 2348, 'redstorm-info', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Diagnostics Port IANA assigned this well-formed service name as a
    #:   replacement for "redstorm_diag".
    #: - [TCP] Diagnostics Port
    #: - [UDP] Diagnostics Port IANA assigned this well-formed service name as a
    #:   replacement for "redstorm_diag".
    #: - [UDP] Diagnostics Port
    redstorm_diag: 'AppType' = 2349, 'redstorm-diag', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Pharos Booking Server
    #: - [UDP] Pharos Booking Server
    psbserver: 'AppType' = 2350, 'psbserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] psrserver
    #: - [UDP] psrserver
    psrserver: 'AppType' = 2351, 'psrserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pslserver
    #: - [UDP] pslserver
    pslserver: 'AppType' = 2352, 'pslserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pspserver
    #: - [UDP] pspserver
    pspserver: 'AppType' = 2353, 'pspserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] psprserver
    #: - [UDP] psprserver
    psprserver: 'AppType' = 2354, 'psprserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] psdbserver
    #: - [UDP] psdbserver
    psdbserver: 'AppType' = 2355, 'psdbserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GXT License Managemant
    #: - [UDP] GXT License Managemant
    gxtelmd: 'AppType' = 2356, 'gxtelmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniHub Server
    #: - [UDP] UniHub Server
    unihub_server: 'AppType' = 2357, 'unihub-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Futrix
    #: - [UDP] Futrix
    futrix: 'AppType' = 2358, 'futrix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FlukeServer
    #: - [UDP] FlukeServer
    flukeserver: 'AppType' = 2359, 'flukeserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NexstorIndLtd
    #: - [UDP] NexstorIndLtd
    nexstorindltd: 'AppType' = 2360, 'nexstorindltd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TL1
    #: - [UDP] TL1
    tl1: 'AppType' = 2361, 'tl1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] digiman
    #: - [UDP] digiman
    digiman: 'AppType' = 2362, 'digiman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Media Central NFSD
    #: - [UDP] Media Central NFSD
    mediacntrlnfsd: 'AppType' = 2363, 'mediacntrlnfsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OI-2000
    #: - [UDP] OI-2000
    oi_2000: 'AppType' = 2364, 'oi-2000', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dbref
    #: - [UDP] dbref
    dbref: 'AppType' = 2365, 'dbref', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qip-login
    #: - [UDP] qip-login
    qip_login: 'AppType' = 2366, 'qip-login', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Control
    #: - [UDP] Service Control
    service_ctrl: 'AppType' = 2367, 'service-ctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenTable
    #: - [UDP] OpenTable
    opentable: 'AppType' = 2368, 'opentable', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blockchain Identifier InFrastructure P2P
    #: - [UDP] Blockchain Identifier InFrastructure P2P
    bif_p2p: 'AppType' = 2369, 'bif-p2p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] L3-HBMon
    #: - [UDP] L3-HBMon
    l3_hbmon: 'AppType' = 2370, 'l3-hbmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RDA
    #: - [UDP] RDA
    rda_630: 'AppType' = 630, 'rda', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Remote Device Access
    rda_2371: 'AppType' = 2371, 'rda', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2371: 'AppType' = 2371, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] LanMessenger
    #: - [UDP] LanMessenger
    lanmessenger: 'AppType' = 2372, 'lanmessenger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Remograph License Manager
    remographlm: 'AppType' = 2373, 'remographlm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2373: 'AppType' = 2373, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Hydra RPC
    hydra: 'AppType' = 2374, 'hydra', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2374: 'AppType' = 2374, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Docker REST API (plain text)
    docker: 'AppType' = 2375, 'docker', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2375: 'AppType' = 2375, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Docker REST API (ssl)
    docker_s: 'AppType' = 2376, 'docker-s', TransportProtocol.get('tcp')

    #: [TCP] RPC interface for Docker Swarm
    swarm: 'AppType' = 2377, 'swarm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2377: 'AppType' = 2377, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_2378: 'AppType' = 2378, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] DALI lighting control
    dali: 'AppType' = 2378, 'dali', TransportProtocol.get('udp')

    #: [TCP] etcd client communication
    etcd_client: 'AppType' = 2379, 'etcd-client', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2379: 'AppType' = 2379, 'reserved', TransportProtocol.get('udp')

    #: [TCP] etcd server to server communication
    etcd_server: 'AppType' = 2380, 'etcd-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2380: 'AppType' = 2380, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Compaq HTTPS
    #: - [UDP] Compaq HTTPS
    compaq_https: 'AppType' = 2381, 'compaq-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft OLAP
    #: - [UDP] Microsoft OLAP
    ms_olap3: 'AppType' = 2382, 'ms-olap3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft OLAP
    #: - [UDP] Microsoft OLAP
    ms_olap4: 'AppType' = 2383, 'ms-olap4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SD-REQUEST
    sd_request: 'AppType' = 2384, 'sd-request', TransportProtocol.get('tcp')

    #: [UDP] SD-CAPACITY
    sd_capacity: 'AppType' = 2384, 'sd-capacity', TransportProtocol.get('udp')

    #: - [TCP] SD-DATA
    #: - [UDP] SD-DATA
    sd_data: 'AppType' = 2385, 'sd-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Tape
    #: - [UDP] Virtual Tape
    virtualtape: 'AppType' = 2386, 'virtualtape', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSAM Redirector
    #: - [UDP] VSAM Redirector
    vsamredirector: 'AppType' = 2387, 'vsamredirector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MYNAH AutoStart
    #: - [UDP] MYNAH AutoStart
    mynahautostart: 'AppType' = 2388, 'mynahautostart', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView Session Mgr
    #: - [UDP] OpenView Session Mgr
    ovsessionmgr: 'AppType' = 2389, 'ovsessionmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSMTP
    #: - [UDP] RSMTP
    rsmtp: 'AppType' = 2390, 'rsmtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3COM Net Management
    #: - [UDP] 3COM Net Management
    AppType_3com_net_mgmt: 'AppType' = 2391, '3com-net-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tactical Auth
    #: - [UDP] Tactical Auth
    tacticalauth: 'AppType' = 2392, 'tacticalauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS OLAP 1
    #: - [UDP] MS OLAP 1
    ms_olap1: 'AppType' = 2393, 'ms-olap1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS OLAP 2
    #: - [UDP] MS OLAP 2
    ms_olap2: 'AppType' = 2394, 'ms-olap2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LAN900 Remote IANA assigned this well-formed service name as a
    #:   replacement for "lan900_remote".
    #: - [TCP] LAN900 Remote
    #: - [UDP] LAN900 Remote IANA assigned this well-formed service name as a
    #:   replacement for "lan900_remote".
    #: - [UDP] LAN900 Remote
    lan900_remote: 'AppType' = 2395, 'lan900-remote', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Wusage
    #: - [UDP] Wusage
    wusage: 'AppType' = 2396, 'wusage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NCL
    #: - [UDP] NCL
    ncl: 'AppType' = 2397, 'ncl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orbiter
    #: - [UDP] Orbiter
    orbiter: 'AppType' = 2398, 'orbiter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileMaker, Inc. - Data Access Layer
    #: - [UDP] FileMaker, Inc. - Data Access Layer
    fmpro_fdal: 'AppType' = 2399, 'fmpro-fdal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpEquus Server
    #: - [UDP] OpEquus Server
    opequus_server: 'AppType' = 2400, 'opequus-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cvspserver
    #: - [UDP] cvspserver
    cvspserver: 'AppType' = 2401, 'cvspserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TaskMaster 2000 Server
    #: - [UDP] TaskMaster 2000 Server
    taskmaster2000_2402: 'AppType' = 2402, 'taskmaster2000', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TaskMaster 2000 Web
    #: - [UDP] TaskMaster 2000 Web
    taskmaster2000_2403: 'AppType' = 2403, 'taskmaster2000', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IEC 60870-5-104 process control over IP
    #: - [UDP] IEC 60870-5-104 process control over IP
    iec_104: 'AppType' = 2404, 'iec-104', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRC Netpoll
    #: - [UDP] TRC Netpoll
    trc_netpoll: 'AppType' = 2405, 'trc-netpoll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JediServer
    #: - [UDP] JediServer
    jediserver: 'AppType' = 2406, 'jediserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orion
    #: - [UDP] Orion
    orion: 'AppType' = 2407, 'orion', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CloudFlare Railgun Web Acceleration Protocol
    railgun_webaccl: 'AppType' = 2408, 'railgun-webaccl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2408: 'AppType' = 2408, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SNS Protocol
    #: - [UDP] SNS Protocol
    sns_protocol: 'AppType' = 2409, 'sns-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VRTS Registry
    #: - [UDP] VRTS Registry
    vrts_registry: 'AppType' = 2410, 'vrts-registry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netwave AP Management
    #: - [UDP] Netwave AP Management
    netwave_ap_mgmt: 'AppType' = 2411, 'netwave-ap-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CDN
    #: - [UDP] CDN
    cdn: 'AppType' = 2412, 'cdn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] orion-rmi-reg
    #: - [UDP] orion-rmi-reg
    orion_rmi_reg: 'AppType' = 2413, 'orion-rmi-reg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Beeyond
    #: - [UDP] Beeyond
    beeyond: 'AppType' = 2414, 'beeyond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Codima Remote Transaction Protocol
    #: - [UDP] Codima Remote Transaction Protocol
    codima_rtp: 'AppType' = 2415, 'codima-rtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMT Server
    #: - [UDP] RMT Server
    rmtserver: 'AppType' = 2416, 'rmtserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Composit Server
    #: - [UDP] Composit Server
    composit_server: 'AppType' = 2417, 'composit-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cas
    #: - [UDP] cas
    cas: 'AppType' = 2418, 'cas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Attachmate S2S
    #: - [UDP] Attachmate S2S
    attachmate_s2s: 'AppType' = 2419, 'attachmate-s2s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DSL Remote Management
    #: - [UDP] DSL Remote Management
    dslremote_mgmt: 'AppType' = 2420, 'dslremote-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] G-Talk
    #: - [UDP] G-Talk
    g_talk: 'AppType' = 2421, 'g-talk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CRMSBITS
    #: - [UDP] CRMSBITS
    crmsbits: 'AppType' = 2422, 'crmsbits', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RNRP
    #: - [UDP] RNRP
    rnrp: 'AppType' = 2423, 'rnrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KOFAX-SVR
    #: - [UDP] KOFAX-SVR
    kofax_svr: 'AppType' = 2424, 'kofax-svr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu App Manager
    #: - [UDP] Fujitsu App Manager
    fjitsuappmgr: 'AppType' = 2425, 'fjitsuappmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VeloCloud MultiPath Protocol
    #: - [UDP] VeloCloud MultiPath Protocol
    vcmp: 'AppType' = 2426, 'vcmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Media Gateway Control Protocol Gateway
    #: - [UDP] Media Gateway Control Protocol Gateway
    mgcp_gateway: 'AppType' = 2427, 'mgcp-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] One Way Trip Time
    #: - [UDP] One Way Trip Time
    ott: 'AppType' = 2428, 'ott', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FT-ROLE
    #: - [UDP] FT-ROLE
    ft_role: 'AppType' = 2429, 'ft-role', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] venus
    #: - [UDP] venus
    venus: 'AppType' = 2430, 'venus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] venus-se
    #: - [UDP] venus-se
    venus_se: 'AppType' = 2431, 'venus-se', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] codasrv
    #: - [UDP] codasrv
    codasrv: 'AppType' = 2432, 'codasrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] codasrv-se
    #: - [UDP] codasrv-se
    codasrv_se: 'AppType' = 2433, 'codasrv-se', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-epmap
    #: - [UDP] pxc-epmap
    pxc_epmap: 'AppType' = 2434, 'pxc-epmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OptiLogic
    #: - [UDP] OptiLogic
    optilogic: 'AppType' = 2435, 'optilogic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TOP/X
    #: - [UDP] TOP/X
    topx: 'AppType' = 2436, 'topx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Send Protocol (historic)
    #: - [UDP] Message Send Protocol (historic)
    msp_18: 'AppType' = 18, 'msp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSP
    #: - [UDP] MSP
    msp_2438: 'AppType' = 2438, 'msp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SybaseDBSynch
    #: - [UDP] SybaseDBSynch
    sybasedbsynch: 'AppType' = 2439, 'sybasedbsynch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spearway Lockers
    #: - [UDP] Spearway Lockers
    spearway: 'AppType' = 2440, 'spearway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pervasive I*net Data Server
    #: - [UDP] Pervasive I*net Data Server
    pvsw_inet: 'AppType' = 2441, 'pvsw-inet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netangel
    #: - [UDP] Netangel
    netangel: 'AppType' = 2442, 'netangel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PowerClient Central Storage Facility
    #: - [UDP] PowerClient Central Storage Facility
    powerclientcsf: 'AppType' = 2443, 'powerclientcsf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BT PP2 Sectrans
    #: - [UDP] BT PP2 Sectrans
    btpp2sectrans: 'AppType' = 2444, 'btpp2sectrans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTN1
    #: - [UDP] DTN1
    dtn1: 'AppType' = 2445, 'dtn1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bues_service IANA assigned this well-formed service name as a
    #:   replacement for "bues_service".
    #: - [TCP] bues_service
    #: - [UDP] bues_service IANA assigned this well-formed service name as a
    #:   replacement for "bues_service".
    #: - [UDP] bues_service
    bues_service: 'AppType' = 2446, 'bues-service', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView NNM daemon
    #: - [UDP] OpenView NNM daemon
    ovwdb: 'AppType' = 2447, 'ovwdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hpppsvr
    #: - [UDP] hpppsvr
    hpppssvr: 'AppType' = 2448, 'hpppssvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RATL
    #: - [UDP] RATL
    ratl: 'AppType' = 2449, 'ratl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netadmin
    #: - [UDP] netadmin
    netadmin: 'AppType' = 2450, 'netadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netchat
    #: - [UDP] netchat
    netchat: 'AppType' = 2451, 'netchat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SnifferClient
    #: - [UDP] SnifferClient
    snifferclient: 'AppType' = 2452, 'snifferclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] madge ltd
    #: - [UDP] madge ltd
    madge_ltd: 'AppType' = 2453, 'madge-ltd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IndX-DDS
    #: - [UDP] IndX-DDS
    indx_dds: 'AppType' = 2454, 'indx-dds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WAGO-IO-SYSTEM
    #: - [UDP] WAGO-IO-SYSTEM
    wago_io_system: 'AppType' = 2455, 'wago-io-system', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] altav-remmgt
    #: - [UDP] altav-remmgt
    altav_remmgt: 'AppType' = 2456, 'altav-remmgt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rapido_IP
    #: - [UDP] Rapido_IP
    rapido_ip: 'AppType' = 2457, 'rapido-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] griffin
    #: - [UDP] griffin
    griffin: 'AppType' = 2458, 'griffin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Community
    #: - [UDP] Community
    xrpl: 'AppType' = 2459, 'xrpl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ms-theater
    #: - [UDP] ms-theater
    ms_theater: 'AppType' = 2460, 'ms-theater', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qadmifoper
    #: - [UDP] qadmifoper
    qadmifoper: 'AppType' = 2461, 'qadmifoper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qadmifevent
    #: - [UDP] qadmifevent
    qadmifevent: 'AppType' = 2462, 'qadmifevent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LSI RAID Management
    #: - [UDP] LSI RAID Management
    lsi_raid_mgmt: 'AppType' = 2463, 'lsi-raid-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirecPC SI
    #: - [UDP] DirecPC SI
    direcpc_si: 'AppType' = 2464, 'direcpc-si', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Load Balance Management
    #: - [UDP] Load Balance Management
    lbm: 'AppType' = 2465, 'lbm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Load Balance Forwarding
    #: - [UDP] Load Balance Forwarding
    lbf: 'AppType' = 2466, 'lbf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] High Criteria
    #: - [UDP] High Criteria
    high_criteria: 'AppType' = 2467, 'high-criteria', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qip_msgd
    #: - [UDP] qip_msgd
    qip_msgd: 'AppType' = 2468, 'qip-msgd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MTI-TCS-COMM
    #: - [UDP] MTI-TCS-COMM
    mti_tcs_comm: 'AppType' = 2469, 'mti-tcs-comm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] taskman port
    #: - [UDP] taskman port
    taskman_port: 'AppType' = 2470, 'taskman-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SeaODBC
    #: - [UDP] SeaODBC
    seaodbc: 'AppType' = 2471, 'seaodbc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] C3
    #: - [UDP] C3
    c3: 'AppType' = 2472, 'c3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aker-cdp
    #: - [UDP] Aker-cdp
    aker_cdp: 'AppType' = 2473, 'aker-cdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vital Analysis
    #: - [UDP] Vital Analysis
    vitalanalysis: 'AppType' = 2474, 'vitalanalysis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACE Server
    #: - [UDP] ACE Server
    ace_server: 'AppType' = 2475, 'ace-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACE Server Propagation
    #: - [UDP] ACE Server Propagation
    ace_svr_prop: 'AppType' = 2476, 'ace-svr-prop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SecurSight Certificate Valifation Service
    #: - [UDP] SecurSight Certificate Valifation Service
    ssm_cvs: 'AppType' = 2477, 'ssm-cvs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SecurSight Authentication Server (SSL)
    #: - [UDP] SecurSight Authentication Server (SSL)
    ssm_cssps: 'AppType' = 2478, 'ssm-cssps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SecurSight Event Logging Server (SSL)
    #: - [UDP] SecurSight Event Logging Server (SSL)
    ssm_els: 'AppType' = 2479, 'ssm-els', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Informatica PowerExchange Listener
    #: - [UDP] Informatica PowerExchange Listener
    powerexchange: 'AppType' = 2480, 'powerexchange', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle GIOP
    #: - [UDP] Oracle GIOP
    giop: 'AppType' = 2481, 'giop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle GIOP SSL
    #: - [UDP] Oracle GIOP SSL
    giop_ssl: 'AppType' = 2482, 'giop-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle TTC
    #: - [UDP] Oracle TTC
    ttc: 'AppType' = 2483, 'ttc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle TTC SSL
    #: - [UDP] Oracle TTC SSL
    ttc_ssl: 'AppType' = 2484, 'ttc-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Objects1
    #: - [UDP] Net Objects1
    netobjects1: 'AppType' = 2485, 'netobjects1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Objects2
    #: - [UDP] Net Objects2
    netobjects2: 'AppType' = 2486, 'netobjects2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Policy Notice Service
    #: - [UDP] Policy Notice Service
    pns: 'AppType' = 2487, 'pns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Moy Corporation
    #: - [UDP] Moy Corporation
    moy_corp: 'AppType' = 2488, 'moy-corp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TSILB
    #: - [UDP] TSILB
    tsilb: 'AppType' = 2489, 'tsilb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qip_qdhcp
    #: - [UDP] qip_qdhcp
    qip_qdhcp: 'AppType' = 2490, 'qip-qdhcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Conclave CPP
    #: - [UDP] Conclave CPP
    conclave_cpp: 'AppType' = 2491, 'conclave-cpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GROOVE
    #: - [UDP] GROOVE
    groove: 'AppType' = 2492, 'groove', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian MQS
    #: - [UDP] Talarian MQS
    talarian_mqs: 'AppType' = 2493, 'talarian-mqs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC AR
    #: - [UDP] BMC AR
    bmc_ar: 'AppType' = 2494, 'bmc-ar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fast Remote Services
    #: - [UDP] Fast Remote Services
    fast_rem_serv: 'AppType' = 2495, 'fast-rem-serv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIRGIS
    #: - [UDP] DIRGIS
    dirgis: 'AppType' = 2496, 'dirgis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quad DB
    #: - [UDP] Quad DB
    quaddb: 'AppType' = 2497, 'quaddb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ODN-CasTraq
    #: - [UDP] ODN-CasTraq
    odn_castraq: 'AppType' = 2498, 'odn-castraq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniControl
    #: - [UDP] UniControl
    unicontrol_2437: 'AppType' = 2437, 'unicontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniControl
    #: - [UDP] UniControl
    unicontrol_2499: 'AppType' = 2499, 'unicontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Resource Tracking system server
    #: - [UDP] Resource Tracking system server
    rtsserv: 'AppType' = 2500, 'rtsserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Resource Tracking system client
    #: - [UDP] Resource Tracking system client
    rtsclient: 'AppType' = 2501, 'rtsclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kentrox Protocol
    #: - [UDP] Kentrox Protocol
    kentrox_prot: 'AppType' = 2502, 'kentrox-prot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMS-DPNSS
    #: - [UDP] NMS-DPNSS
    nms_dpnss: 'AppType' = 2503, 'nms-dpnss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WLBS
    #: - [UDP] WLBS
    wlbs: 'AppType' = 2504, 'wlbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PowerPlay Control
    #: - [UDP] PowerPlay Control
    ppcontrol: 'AppType' = 2505, 'ppcontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jbroker
    #: - [UDP] jbroker
    jbroker: 'AppType' = 2506, 'jbroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] spock
    #: - [UDP] spock
    spock: 'AppType' = 2507, 'spock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JDataStore
    #: - [UDP] JDataStore
    jdatastore: 'AppType' = 2508, 'jdatastore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fjmpss
    #: - [UDP] fjmpss
    fjmpss: 'AppType' = 2509, 'fjmpss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fjappmgrbulk
    #: - [UDP] fjappmgrbulk
    fjappmgrbulk: 'AppType' = 2510, 'fjappmgrbulk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Metastorm
    #: - [UDP] Metastorm
    metastorm: 'AppType' = 2511, 'metastorm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citrix IMA
    #: - [UDP] Citrix IMA
    citrixima: 'AppType' = 2512, 'citrixima', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citrix ADMIN
    #: - [UDP] Citrix ADMIN
    citrixadmin: 'AppType' = 2513, 'citrixadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Facsys NTP
    #: - [UDP] Facsys NTP
    facsys_ntp: 'AppType' = 2514, 'facsys-ntp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Facsys Router
    #: - [UDP] Facsys Router
    facsys_router: 'AppType' = 2515, 'facsys-router', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Main Control
    #: - [UDP] Main Control
    maincontrol: 'AppType' = 2516, 'maincontrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.323 Annex E Call Control Signalling Transport
    #: - [UDP] H.323 Annex E Call Control Signalling Transport
    call_sig_trans: 'AppType' = 2517, 'call-sig-trans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Willy
    #: - [UDP] Willy
    willy: 'AppType' = 2518, 'willy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] globmsgsvc
    #: - [UDP] globmsgsvc
    globmsgsvc: 'AppType' = 2519, 'globmsgsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pervasive Listener
    #: - [UDP] Pervasive Listener
    pvsw: 'AppType' = 2520, 'pvsw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adaptec Manager
    #: - [UDP] Adaptec Manager
    adaptecmgr: 'AppType' = 2521, 'adaptecmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WinDb
    #: - [UDP] WinDb
    windb: 'AppType' = 2522, 'windb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Qke LLC V.3
    #: - [UDP] Qke LLC V.3
    qke_llc_v3: 'AppType' = 2523, 'qke-llc-v3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Optiwave License Management
    #: - [UDP] Optiwave License Management
    optiwave_lm: 'AppType' = 2524, 'optiwave-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS V-Worlds
    #: - [UDP] MS V-Worlds
    ms_v_worlds: 'AppType' = 2525, 'ms-v-worlds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMA License Manager
    #: - [UDP] EMA License Manager
    ema_sent_lm: 'AppType' = 2526, 'ema-sent-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IQ Server
    #: - [UDP] IQ Server
    iqserver: 'AppType' = 2527, 'iqserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NCR CCL IANA assigned this well-formed service name as a replacement
    #:   for "ncr_ccl".
    #: - [TCP] NCR CCL
    #: - [UDP] NCR CCL IANA assigned this well-formed service name as a replacement
    #:   for "ncr_ccl".
    #: - [UDP] NCR CCL
    ncr_ccl: 'AppType' = 2528, 'ncr-ccl', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] UTS FTP
    #: - [UDP] UTS FTP
    utsftp: 'AppType' = 2529, 'utsftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VR Commerce
    #: - [UDP] VR Commerce
    vrcommerce: 'AppType' = 2530, 'vrcommerce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITO-E GUI
    #: - [UDP] ITO-E GUI
    ito_e_gui: 'AppType' = 2531, 'ito-e-gui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OVTOPMD
    #: - [UDP] OVTOPMD
    ovtopmd: 'AppType' = 2532, 'ovtopmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SnifferServer
    #: - [UDP] SnifferServer
    snifferserver: 'AppType' = 2533, 'snifferserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Combox Web Access
    #: - [UDP] Combox Web Access
    combox_web_acc: 'AppType' = 2534, 'combox-web-acc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MADCAP
    #: - [UDP] MADCAP
    madcap: 'AppType' = 2535, 'madcap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] btpp2audctr1
    #: - [UDP] btpp2audctr1
    btpp2audctr1: 'AppType' = 2536, 'btpp2audctr1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Upgrade Protocol
    #: - [UDP] Upgrade Protocol
    upgrade: 'AppType' = 2537, 'upgrade', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vnwk-prapi
    #: - [UDP] vnwk-prapi
    vnwk_prapi: 'AppType' = 2538, 'vnwk-prapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSI Admin
    #: - [UDP] VSI Admin
    vsiadmin: 'AppType' = 2539, 'vsiadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LonWorks2
    #: - [UDP] LonWorks2
    lonworks2: 'AppType' = 2541, 'lonworks2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] uDraw(Graph)
    #: - [UDP] uDraw(Graph)
    udrawgraph: 'AppType' = 2542, 'udrawgraph', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] REFTEK
    #: - [UDP] REFTEK
    reftek: 'AppType' = 2543, 'reftek', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Management Daemon Refresh
    #: - [UDP] Management Daemon Refresh
    novell_zen: 'AppType' = 2544, 'novell-zen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sis-emt
    #: - [UDP] sis-emt
    sis_emt: 'AppType' = 2545, 'sis-emt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vytalvaultbrtp
    #: - [UDP] vytalvaultbrtp
    vytalvaultbrtp: 'AppType' = 2546, 'vytalvaultbrtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vytalvaultvsmp
    #: - [UDP] vytalvaultvsmp
    vytalvaultvsmp: 'AppType' = 2547, 'vytalvaultvsmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vytalvaultpipe
    #: - [UDP] vytalvaultpipe
    vytalvaultpipe: 'AppType' = 2548, 'vytalvaultpipe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPASS
    #: - [UDP] IPASS
    ipass: 'AppType' = 2549, 'ipass', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ADS
    #: - [UDP] ADS
    ads: 'AppType' = 2550, 'ads', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISG UDA Server
    #: - [UDP] ISG UDA Server
    isg_uda_server: 'AppType' = 2551, 'isg-uda-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Call Logging
    #: - [UDP] Call Logging
    call_logging: 'AppType' = 2552, 'call-logging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] efidiningport
    #: - [UDP] efidiningport
    efidiningport: 'AppType' = 2553, 'efidiningport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VCnet-Link v10
    #: - [UDP] VCnet-Link v10
    vcnet_link_v10: 'AppType' = 2554, 'vcnet-link-v10', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq WCP
    #: - [UDP] Compaq WCP
    compaq_wcp: 'AppType' = 2555, 'compaq-wcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nicetec-nmsvc
    #: - [UDP] nicetec-nmsvc
    nicetec_nmsvc: 'AppType' = 2556, 'nicetec-nmsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nicetec-mgmt
    #: - [UDP] nicetec-mgmt
    nicetec_mgmt: 'AppType' = 2557, 'nicetec-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCLE Multi Media
    #: - [UDP] PCLE Multi Media
    pclemultimedia: 'AppType' = 2558, 'pclemultimedia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LSTP
    #: - [UDP] LSTP
    lstp: 'AppType' = 2559, 'lstp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] labrat
    #: - [UDP] labrat
    labrat: 'AppType' = 2560, 'labrat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MosaixCC
    #: - [UDP] MosaixCC
    mosaixcc: 'AppType' = 2561, 'mosaixcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Delibo
    #: - [UDP] Delibo
    delibo: 'AppType' = 2562, 'delibo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CTI Redwood
    #: - [UDP] CTI Redwood
    cti_redwood: 'AppType' = 2563, 'cti-redwood', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP 3000 NS/VT block mode telnet
    #: - [UDP] HP 3000 NS/VT block mode telnet
    hp_3000_telnet: 'AppType' = 2564, 'hp-3000-telnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Coordinator Server
    #: - [UDP] Coordinator Server
    coord_svr: 'AppType' = 2565, 'coord-svr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pcs-pcw
    #: - [UDP] pcs-pcw
    pcs_pcw: 'AppType' = 2566, 'pcs-pcw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco Line Protocol
    #: - [UDP] Cisco Line Protocol
    clp: 'AppType' = 2567, 'clp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPAM TRAP
    #: - [UDP] SPAM TRAP
    spamtrap: 'AppType' = 2568, 'spamtrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sonus Call Signal
    #: - [UDP] Sonus Call Signal
    sonuscallsig: 'AppType' = 2569, 'sonuscallsig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HS Port
    #: - [UDP] HS Port
    hs_port: 'AppType' = 2570, 'hs-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CECSVC
    #: - [UDP] CECSVC
    cecsvc: 'AppType' = 2571, 'cecsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBP
    #: - [UDP] IBP
    ibp: 'AppType' = 2572, 'ibp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trust Establish
    #: - [UDP] Trust Establish
    trustestablish: 'AppType' = 2573, 'trustestablish', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blockade BPSP
    #: - [UDP] Blockade BPSP
    blockade_bpsp: 'AppType' = 2574, 'blockade-bpsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HL7
    #: - [UDP] HL7
    hl7: 'AppType' = 2575, 'hl7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCL Pro Debugger
    #: - [UDP] TCL Pro Debugger
    tclprodebugger: 'AppType' = 2576, 'tclprodebugger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scriptics Lsrvr
    #: - [UDP] Scriptics Lsrvr
    scipticslsrvr: 'AppType' = 2577, 'scipticslsrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RVS ISDN DCP
    #: - [UDP] RVS ISDN DCP
    rvs_isdn_dcp: 'AppType' = 2578, 'rvs-isdn-dcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpfoncl
    #: - [UDP] mpfoncl
    mpfoncl: 'AppType' = 2579, 'mpfoncl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tributary
    #: - [UDP] Tributary
    tributary: 'AppType' = 2580, 'tributary', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARGIS TE
    #: - [UDP] ARGIS TE
    argis_te: 'AppType' = 2581, 'argis-te', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARGIS DS
    #: - [UDP] ARGIS DS
    argis_ds: 'AppType' = 2582, 'argis-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cyaserv
    #: - [UDP] cyaserv
    cyaserv: 'AppType' = 2584, 'cyaserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETX Server
    #: - [UDP] NETX Server
    netx_server: 'AppType' = 2585, 'netx-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETX Agent
    #: - [UDP] NETX Agent
    netx_agent: 'AppType' = 2586, 'netx-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MASC
    #: - [UDP] MASC
    masc: 'AppType' = 2587, 'masc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Privilege
    #: - [UDP] Privilege
    privilege: 'AppType' = 2588, 'privilege', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] quartus tcl
    #: - [UDP] quartus tcl
    quartus_tcl: 'AppType' = 2589, 'quartus-tcl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] idotdist
    #: - [UDP] idotdist
    idotdist: 'AppType' = 2590, 'idotdist', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Maytag Shuffle
    #: - [UDP] Maytag Shuffle
    maytagshuffle: 'AppType' = 2591, 'maytagshuffle', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netrek
    #: - [UDP] netrek
    netrek: 'AppType' = 2592, 'netrek', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MNS Mail Notice Service
    #: - [UDP] MNS Mail Notice Service
    mns_mail: 'AppType' = 2593, 'mns-mail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Base Server
    #: - [UDP] Data Base Server
    dts: 'AppType' = 2594, 'dts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] World Fusion 1
    #: - [UDP] World Fusion 1
    worldfusion1: 'AppType' = 2595, 'worldfusion1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] World Fusion 2
    #: - [UDP] World Fusion 2
    worldfusion2: 'AppType' = 2596, 'worldfusion2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Homestead Glory
    #: - [UDP] Homestead Glory
    homesteadglory: 'AppType' = 2597, 'homesteadglory', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citrix MA Client
    #: - [UDP] Citrix MA Client
    citriximaclient: 'AppType' = 2598, 'citriximaclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Snap Discovery
    #: - [UDP] Snap Discovery
    snapd: 'AppType' = 2599, 'snapd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPSTGMGR
    #: - [UDP] HPSTGMGR
    hpstgmgr: 'AppType' = 2600, 'hpstgmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] discp client
    #: - [UDP] discp client
    discp_client: 'AppType' = 2601, 'discp-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] discp server
    #: - [UDP] discp server
    discp_server: 'AppType' = 2602, 'discp-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Meter
    #: - [UDP] Service Meter
    servicemeter: 'AppType' = 2603, 'servicemeter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSC CCS
    #: - [UDP] NSC CCS
    nsc_ccs: 'AppType' = 2604, 'nsc-ccs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSC POSA
    #: - [UDP] NSC POSA
    nsc_posa: 'AppType' = 2605, 'nsc-posa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dell Netmon
    #: - [UDP] Dell Netmon
    netmon: 'AppType' = 2606, 'netmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dell Connection
    #: - [UDP] Dell Connection
    connection: 'AppType' = 2607, 'connection', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wag Service
    #: - [UDP] Wag Service
    wag_service: 'AppType' = 2608, 'wag-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] System Monitor
    #: - [UDP] System Monitor
    system_monitor: 'AppType' = 2609, 'system-monitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VersaTek
    #: - [UDP] VersaTek
    versa_tek: 'AppType' = 2610, 'versa-tek', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LIONHEAD
    #: - [UDP] LIONHEAD
    lionhead: 'AppType' = 2611, 'lionhead', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Qpasa Agent
    #: - [UDP] Qpasa Agent
    qpasa_agent: 'AppType' = 2612, 'qpasa-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMNTUBootstrap
    #: - [UDP] SMNTUBootstrap
    smntubootstrap: 'AppType' = 2613, 'smntubootstrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Never Offline
    #: - [UDP] Never Offline
    neveroffline: 'AppType' = 2614, 'neveroffline', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] firepower
    #: - [UDP] firepower
    firepower: 'AppType' = 2615, 'firepower', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appswitch-emp
    #: - [UDP] appswitch-emp
    appswitch_emp: 'AppType' = 2616, 'appswitch-emp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Clinical Context Managers
    #: - [UDP] Clinical Context Managers
    cmadmin: 'AppType' = 2617, 'cmadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Priority E-Com
    #: - [UDP] Priority E-Com
    priority_e_com: 'AppType' = 2618, 'priority-e-com', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bruce
    #: - [UDP] bruce
    bruce: 'AppType' = 2619, 'bruce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LPSRecommender
    #: - [UDP] LPSRecommender
    lpsrecommender: 'AppType' = 2620, 'lpsrecommender', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Miles Apart Jukebox Server
    #: - [UDP] Miles Apart Jukebox Server
    miles_apart: 'AppType' = 2621, 'miles-apart', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetricaDBC
    #: - [UDP] MetricaDBC
    metricadbc: 'AppType' = 2622, 'metricadbc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LMDP
    #: - [UDP] LMDP
    lmdp: 'AppType' = 2623, 'lmdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aria
    #: - [UDP] Aria
    aria: 'AppType' = 2624, 'aria', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blwnkl Port
    #: - [UDP] Blwnkl Port
    blwnkl_port: 'AppType' = 2625, 'blwnkl-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gbjd816
    #: - [UDP] gbjd816
    gbjd816: 'AppType' = 2626, 'gbjd816', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Moshe Beeri
    #: - [UDP] Moshe Beeri
    moshebeeri: 'AppType' = 2627, 'moshebeeri', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DICT
    #: - [UDP] DICT
    dict: 'AppType' = 2628, 'dict', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sitara Server
    #: - [UDP] Sitara Server
    sitaraserver: 'AppType' = 2629, 'sitaraserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sitara Management
    #: - [UDP] Sitara Management
    sitaramgmt: 'AppType' = 2630, 'sitaramgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sitara Dir
    #: - [UDP] Sitara Dir
    sitaradir: 'AppType' = 2631, 'sitaradir', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IRdg Post
    #: - [UDP] IRdg Post
    irdg_post: 'AppType' = 2632, 'irdg-post', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InterIntelli
    #: - [UDP] InterIntelli
    interintelli: 'AppType' = 2633, 'interintelli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PK Electronics
    #: - [UDP] PK Electronics
    pk_electronics: 'AppType' = 2634, 'pk-electronics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Back Burner
    #: - [UDP] Back Burner
    backburner: 'AppType' = 2635, 'backburner', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Solve
    #: - [UDP] Solve
    solve: 'AppType' = 2636, 'solve', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Import Document Service
    #: - [UDP] Import Document Service
    imdocsvc: 'AppType' = 2637, 'imdocsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sybase Anywhere
    #: - [UDP] Sybase Anywhere
    sybaseanywhere: 'AppType' = 2638, 'sybaseanywhere', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMInet
    #: - [UDP] AMInet
    aminet: 'AppType' = 2639, 'aminet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alcorn McBride Inc protocol used for device control
    #: - [UDP] Alcorn McBride Inc protocol used for device control
    ami_control: 'AppType' = 2640, 'ami-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HDL Server
    #: - [UDP] HDL Server
    hdl_srv: 'AppType' = 2641, 'hdl-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tragic
    #: - [UDP] Tragic
    tragic: 'AppType' = 2642, 'tragic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GTE-SAMP
    #: - [UDP] GTE-SAMP
    gte_samp: 'AppType' = 2643, 'gte-samp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Travsoft IPX Tunnel
    #: - [UDP] Travsoft IPX Tunnel
    travsoft_ipx_t: 'AppType' = 2644, 'travsoft-ipx-t', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novell IPX CMD
    #: - [UDP] Novell IPX CMD
    novell_ipx_cmd: 'AppType' = 2645, 'novell-ipx-cmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AND License Manager
    #: - [UDP] AND License Manager
    and_lm: 'AppType' = 2646, 'and-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SyncServer
    #: - [UDP] SyncServer
    syncserver: 'AppType' = 2647, 'syncserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Upsnotifyprot
    #: - [UDP] Upsnotifyprot
    upsnotifyprot: 'AppType' = 2648, 'upsnotifyprot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VPSIPPORT
    #: - [UDP] VPSIPPORT
    vpsipport: 'AppType' = 2649, 'vpsipport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eristwoguns
    #: - [UDP] eristwoguns
    eristwoguns: 'AppType' = 2650, 'eristwoguns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EBInSite
    #: - [UDP] EBInSite
    ebinsite: 'AppType' = 2651, 'ebinsite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InterPathPanel
    #: - [UDP] InterPathPanel
    interpathpanel: 'AppType' = 2652, 'interpathpanel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sonus
    #: - [UDP] Sonus
    sonus: 'AppType' = 2653, 'sonus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Corel VNC Admin IANA assigned this well-formed service name as a
    #:   replacement for "corel_vncadmin".
    #: - [TCP] Corel VNC Admin
    #: - [UDP] Corel VNC Admin IANA assigned this well-formed service name as a
    #:   replacement for "corel_vncadmin".
    #: - [UDP] Corel VNC Admin
    corel_vncadmin: 'AppType' = 2654, 'corel-vncadmin', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] UNIX Nt Glue
    #: - [UDP] UNIX Nt Glue
    unglue: 'AppType' = 2655, 'unglue', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kana
    #: - [UDP] Kana
    kana: 'AppType' = 2656, 'kana', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Dispatcher
    #: - [UDP] SNS Dispatcher
    sns_dispatcher: 'AppType' = 2657, 'sns-dispatcher', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Admin
    #: - [UDP] SNS Admin
    sns_admin: 'AppType' = 2658, 'sns-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Query
    #: - [UDP] SNS Query
    sns_query: 'AppType' = 2659, 'sns-query', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GC Monitor
    #: - [UDP] GC Monitor
    gcmonitor: 'AppType' = 2660, 'gcmonitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OLHOST
    #: - [UDP] OLHOST
    olhost: 'AppType' = 2661, 'olhost', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BinTec-CAPI
    #: - [UDP] BinTec-CAPI
    bintec_capi: 'AppType' = 2662, 'bintec-capi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BinTec-TAPI
    #: - [UDP] BinTec-TAPI
    bintec_tapi: 'AppType' = 2663, 'bintec-tapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Patrol for MQ GM
    #: - [UDP] Patrol for MQ GM
    patrol_mq_gm: 'AppType' = 2664, 'patrol-mq-gm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Patrol for MQ NM
    #: - [UDP] Patrol for MQ NM
    patrol_mq_nm: 'AppType' = 2665, 'patrol-mq-nm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] extensis
    #: - [UDP] extensis
    extensis: 'AppType' = 2666, 'extensis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alarm Clock Server
    #: - [UDP] Alarm Clock Server
    alarm_clock_s: 'AppType' = 2667, 'alarm-clock-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alarm Clock Client
    #: - [UDP] Alarm Clock Client
    alarm_clock_c: 'AppType' = 2668, 'alarm-clock-c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TOAD
    #: - [UDP] TOAD
    toad: 'AppType' = 2669, 'toad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TVE Announce
    #: - [UDP] TVE Announce
    tve_announce: 'AppType' = 2670, 'tve-announce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] newlixreg
    #: - [UDP] newlixreg
    newlixreg: 'AppType' = 2671, 'newlixreg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nhserver
    #: - [UDP] nhserver
    nhserver: 'AppType' = 2672, 'nhserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] First Call 42
    #: - [UDP] First Call 42
    firstcall42: 'AppType' = 2673, 'firstcall42', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ewnn
    #: - [UDP] ewnn
    ewnn: 'AppType' = 2674, 'ewnn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTC ETAP
    #: - [UDP] TTC ETAP
    ttc_etap: 'AppType' = 2675, 'ttc-etap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SIMSLink
    #: - [UDP] SIMSLink
    simslink: 'AppType' = 2676, 'simslink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gadget Gate 1 Way
    #: - [UDP] Gadget Gate 1 Way
    gadgetgate1way: 'AppType' = 2677, 'gadgetgate1way', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gadget Gate 2 Way
    #: - [UDP] Gadget Gate 2 Way
    gadgetgate2way: 'AppType' = 2678, 'gadgetgate2way', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sync Server SSL
    #: - [UDP] Sync Server SSL
    syncserverssl: 'AppType' = 2679, 'syncserverssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-sapxom
    #: - [UDP] pxc-sapxom
    pxc_sapxom: 'AppType' = 2680, 'pxc-sapxom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpnjsomb
    #: - [UDP] mpnjsomb
    mpnjsomb: 'AppType' = 2681, 'mpnjsomb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Removed
    removed_659: 'AppType' = 659, 'removed', TransportProtocol.get('undefined')

    #: [N/A] Removed
    removed_2682: 'AppType' = 2682, 'removed', TransportProtocol.get('undefined')

    #: - [TCP] NCDLoadBalance
    #: - [UDP] NCDLoadBalance
    ncdloadbalance: 'AppType' = 2683, 'ncdloadbalance', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpnjsosv
    #: - [UDP] mpnjsosv
    mpnjsosv: 'AppType' = 2684, 'mpnjsosv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpnjsocl
    #: - [UDP] mpnjsocl
    mpnjsocl: 'AppType' = 2685, 'mpnjsocl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mpnjsomg
    #: - [UDP] mpnjsomg
    mpnjsomg: 'AppType' = 2686, 'mpnjsomg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pq-lic-mgmt
    #: - [UDP] pq-lic-mgmt
    pq_lic_mgmt: 'AppType' = 2687, 'pq-lic-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] md-cf-http
    #: - [UDP] md-cf-http
    md_cg_http: 'AppType' = 2688, 'md-cg-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FastLynx
    #: - [UDP] FastLynx
    fastlynx: 'AppType' = 2689, 'fastlynx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP NNM Embedded Database
    #: - [UDP] HP NNM Embedded Database
    hp_nnm_data: 'AppType' = 2690, 'hp-nnm-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITInternet ISM Server
    #: - [UDP] ITInternet ISM Server
    itinternet: 'AppType' = 2691, 'itinternet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Admins LMS
    #: - [UDP] Admins LMS
    admins_lms: 'AppType' = 2692, 'admins-lms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unassigned
    #: - [UDP] Unassigned
    unassigned_2693: 'AppType' = 2693, 'unassigned', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pwrsevent
    #: - [UDP] pwrsevent
    pwrsevent: 'AppType' = 2694, 'pwrsevent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSPREAD
    #: - [UDP] VSPREAD
    vspread: 'AppType' = 2695, 'vspread', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify Admin
    #: - [UDP] Unify Admin
    unifyadmin: 'AppType' = 2696, 'unifyadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oce SNMP Trap Port
    #: - [UDP] Oce SNMP Trap Port
    oce_snmp_trap: 'AppType' = 2697, 'oce-snmp-trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCK-IVPIP
    #: - [UDP] MCK-IVPIP
    mck_ivpip: 'AppType' = 2698, 'mck-ivpip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Csoft Plus Client
    #: - [UDP] Csoft Plus Client
    csoft_plusclnt: 'AppType' = 2699, 'csoft-plusclnt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tqdata
    #: - [UDP] tqdata
    tqdata: 'AppType' = 2700, 'tqdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMS RCINFO
    #: - [UDP] SMS RCINFO
    sms_rcinfo: 'AppType' = 2701, 'sms-rcinfo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMS XFER
    #: - [UDP] SMS XFER
    sms_xfer: 'AppType' = 2702, 'sms-xfer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMS CHAT
    #: - [UDP] SMS CHAT
    sms_chat: 'AppType' = 2703, 'sms-chat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMS REMCTRL
    #: - [UDP] SMS REMCTRL
    sms_remctrl: 'AppType' = 2704, 'sms-remctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDS Admin
    #: - [UDP] SDS Admin
    sds_admin: 'AppType' = 2705, 'sds-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NCD Mirroring
    #: - [UDP] NCD Mirroring
    ncdmirroring: 'AppType' = 2706, 'ncdmirroring', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMCSYMAPIPORT
    #: - [UDP] EMCSYMAPIPORT
    emcsymapiport: 'AppType' = 2707, 'emcsymapiport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Banyan-Net
    #: - [UDP] Banyan-Net
    banyan_net: 'AppType' = 2708, 'banyan-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Supermon
    #: - [UDP] Supermon
    supermon: 'AppType' = 2709, 'supermon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSO Service
    #: - [UDP] SSO Service
    sso_service: 'AppType' = 2710, 'sso-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSO Control
    #: - [UDP] SSO Control
    sso_control: 'AppType' = 2711, 'sso-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Axapta Object Communication Protocol
    #: - [UDP] Axapta Object Communication Protocol
    aocp: 'AppType' = 2712, 'aocp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Raven Trinity Broker Service
    #: - [UDP] Raven Trinity Broker Service
    raventbs: 'AppType' = 2713, 'raventbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Raven Trinity Data Mover
    #: - [UDP] Raven Trinity Data Mover
    raventdm: 'AppType' = 2714, 'raventdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPSTGMGR2
    #: - [UDP] HPSTGMGR2
    hpstgmgr2: 'AppType' = 2715, 'hpstgmgr2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Inova IP Disco
    #: - [UDP] Inova IP Disco
    inova_ip_disco: 'AppType' = 2716, 'inova-ip-disco', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PN REQUESTER
    #: - [UDP] PN REQUESTER
    pn_requester: 'AppType' = 2717, 'pn-requester', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PN REQUESTER 2
    #: - [UDP] PN REQUESTER 2
    pn_requester2: 'AppType' = 2718, 'pn-requester2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scan & Change
    #: - [UDP] Scan & Change
    scan_change: 'AppType' = 2719, 'scan-change', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] wkars
    #: - [UDP] wkars
    wkars: 'AppType' = 2720, 'wkars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Smart Diagnose
    #: - [UDP] Smart Diagnose
    smart_diagnose: 'AppType' = 2721, 'smart-diagnose', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proactive Server
    #: - [UDP] Proactive Server
    proactivesrvr: 'AppType' = 2722, 'proactivesrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WatchDog NT Protocol
    #: - [UDP] WatchDog NT Protocol
    watchdog_nt: 'AppType' = 2723, 'watchdog-nt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qotps
    #: - [UDP] qotps
    qotps: 'AppType' = 2724, 'qotps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSOLAP PTP2
    #: - [UDP] MSOLAP PTP2
    msolap_ptp2: 'AppType' = 2725, 'msolap-ptp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TAMS
    #: - [UDP] TAMS
    tams: 'AppType' = 2726, 'tams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Media Gateway Control Protocol Call Agent
    #: - [UDP] Media Gateway Control Protocol Call Agent
    mgcp_callagent: 'AppType' = 2727, 'mgcp-callagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SQDR
    #: - [UDP] SQDR
    sqdr: 'AppType' = 2728, 'sqdr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TCIM Control
    #: - [UDP] TCIM Control
    tcim_control: 'AppType' = 2729, 'tcim-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NEC RaidPlus
    #: - [UDP] NEC RaidPlus
    nec_raidplus: 'AppType' = 2730, 'nec-raidplus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fyre Messanger
    #: - [UDP] Fyre Messanger
    fyre_messanger: 'AppType' = 2731, 'fyre-messanger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] G5M
    #: - [UDP] G5M
    g5m: 'AppType' = 2732, 'g5m', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Signet CTF
    #: - [UDP] Signet CTF
    signet_ctf: 'AppType' = 2733, 'signet-ctf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCS Software
    #: - [UDP] CCS Software
    ccs_software: 'AppType' = 2734, 'ccs-software', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetIQ Monitor Console
    #: - [UDP] NetIQ Monitor Console
    netiq_mc: 'AppType' = 2735, 'netiq-mc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADWIZ NMS SRV
    #: - [UDP] RADWIZ NMS SRV
    radwiz_nms_srv: 'AppType' = 2736, 'radwiz-nms-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SRP Feedback
    #: - [UDP] SRP Feedback
    srp_feedback: 'AppType' = 2737, 'srp-feedback', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDL TCP-OSI Gateway
    #: - [UDP] NDL TCP-OSI Gateway
    ndl_tcp_ois_gw: 'AppType' = 2738, 'ndl-tcp-ois-gw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TN Timing
    #: - [UDP] TN Timing
    tn_timing: 'AppType' = 2739, 'tn-timing', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alarm
    #: - [UDP] Alarm
    alarm: 'AppType' = 2740, 'alarm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TSB
    #: - [UDP] TSB
    tsb: 'AppType' = 2741, 'tsb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TSB2
    #: - [UDP] TSB2
    tsb2: 'AppType' = 2742, 'tsb2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] murx
    #: - [UDP] murx
    murx: 'AppType' = 2743, 'murx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] honyaku
    #: - [UDP] honyaku
    honyaku: 'AppType' = 2744, 'honyaku', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] URBISNET
    #: - [UDP] URBISNET
    urbisnet: 'AppType' = 2745, 'urbisnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPUDPENCAP
    #: - [UDP] CPUDPENCAP
    cpudpencap: 'AppType' = 2746, 'cpudpencap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    fjippol_swrly: 'AppType' = 2747, 'fjippol-swrly', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    fjippol_polsvr: 'AppType' = 2748, 'fjippol-polsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    fjippol_cnsl: 'AppType' = 2749, 'fjippol-cnsl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    fjippol_port1: 'AppType' = 2750, 'fjippol-port1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    fjippol_port2: 'AppType' = 2751, 'fjippol-port2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSISYS ACCESS
    #: - [UDP] RSISYS ACCESS
    rsisysaccess: 'AppType' = 2752, 'rsisysaccess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] de-spot
    #: - [UDP] de-spot
    de_spot: 'AppType' = 2753, 'de-spot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APOLLO CC
    #: - [UDP] APOLLO CC
    apollo_cc: 'AppType' = 2754, 'apollo-cc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Express Pay
    #: - [UDP] Express Pay
    expresspay: 'AppType' = 2755, 'expresspay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simplement-tie
    #: - [UDP] simplement-tie
    simplement_tie: 'AppType' = 2756, 'simplement-tie', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CNRP
    #: - [UDP] CNRP
    cnrp: 'AppType' = 2757, 'cnrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APOLLO Status
    #: - [UDP] APOLLO Status
    apollo_status: 'AppType' = 2758, 'apollo-status', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APOLLO GMS
    #: - [UDP] APOLLO GMS
    apollo_gms: 'AppType' = 2759, 'apollo-gms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Saba MS
    #: - [UDP] Saba MS
    sabams: 'AppType' = 2760, 'sabams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DICOM ISCL
    #: - [UDP] DICOM ISCL
    dicom_iscl: 'AppType' = 2761, 'dicom-iscl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DICOM TLS
    #: - [UDP] DICOM TLS
    dicom_tls: 'AppType' = 2762, 'dicom-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Desktop DNA
    #: - [UDP] Desktop DNA
    desktop_dna: 'AppType' = 2763, 'desktop-dna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Insurance
    #: - [UDP] Data Insurance
    data_insurance: 'AppType' = 2764, 'data-insurance', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qip-audup
    #: - [UDP] qip-audup
    qip_audup: 'AppType' = 2765, 'qip-audup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq SCP
    #: - [UDP] Compaq SCP
    compaq_scp: 'AppType' = 2766, 'compaq-scp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UADTC
    #: - [UDP] UADTC
    uadtc: 'AppType' = 2767, 'uadtc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UACS
    #: - [UDP] UACS
    uacs: 'AppType' = 2768, 'uacs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eXcE
    #: - [UDP] eXcE
    exce: 'AppType' = 2769, 'exce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Veronica
    #: - [UDP] Veronica
    veronica: 'AppType' = 2770, 'veronica', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vergence CM
    #: - [UDP] Vergence CM
    vergencecm: 'AppType' = 2771, 'vergencecm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] auris
    #: - [UDP] auris
    auris: 'AppType' = 2772, 'auris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RBackup Remote Backup
    #: - [UDP] RBackup Remote Backup
    rbakcup1: 'AppType' = 2773, 'rbakcup1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RBackup Remote Backup
    #: - [UDP] RBackup Remote Backup
    rbakcup2: 'AppType' = 2774, 'rbakcup2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMPP
    #: - [UDP] SMPP
    smpp: 'AppType' = 2775, 'smpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ridgeway Systems & Software
    #: - [UDP] Ridgeway Systems & Software
    ridgeway1: 'AppType' = 2776, 'ridgeway1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ridgeway Systems & Software
    #: - [UDP] Ridgeway Systems & Software
    ridgeway2: 'AppType' = 2777, 'ridgeway2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gwen-Sonya
    #: - [UDP] Gwen-Sonya
    gwen_sonya: 'AppType' = 2778, 'gwen-sonya', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LBC Sync
    #: - [UDP] LBC Sync
    lbc_sync: 'AppType' = 2779, 'lbc-sync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LBC Control
    #: - [UDP] LBC Control
    lbc_control: 'AppType' = 2780, 'lbc-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] whosells
    #: - [UDP] whosells
    whosells: 'AppType' = 2781, 'whosells', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] everydayrc
    #: - [UDP] everydayrc
    everydayrc: 'AppType' = 2782, 'everydayrc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AISES
    #: - [UDP] AISES
    aises: 'AppType' = 2783, 'aises', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] world wide web - development
    #: - [UDP] world wide web - development
    www_dev: 'AppType' = 2784, 'www-dev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aic-np
    #: - [UDP] aic-np
    aic_np: 'AppType' = 2785, 'aic-np', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aic-oncrpc - Destiny MCD database
    #: - [UDP] aic-oncrpc - Destiny MCD database
    aic_oncrpc: 'AppType' = 2786, 'aic-oncrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] piccolo - Cornerstone Software
    #: - [UDP] piccolo - Cornerstone Software
    piccolo: 'AppType' = 2787, 'piccolo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetWare Loadable Module - Seagate Software
    #: - [UDP] NetWare Loadable Module - Seagate Software
    fryeserv: 'AppType' = 2788, 'fryeserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Media Agent
    #: - [UDP] Media Agent
    media_agent: 'AppType' = 2789, 'media-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PLG Proxy
    #: - [UDP] PLG Proxy
    plgproxy: 'AppType' = 2790, 'plgproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MT Port Registrator
    #: - [UDP] MT Port Registrator
    mtport_regist: 'AppType' = 2791, 'mtport-regist', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] f5-globalsite
    #: - [UDP] f5-globalsite
    f5_globalsite: 'AppType' = 2792, 'f5-globalsite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] initlsmsad
    #: - [UDP] initlsmsad
    initlsmsad: 'AppType' = 2793, 'initlsmsad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_2794: 'AppType' = 2794, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] LiveStats
    #: - [UDP] LiveStats
    livestats: 'AppType' = 2795, 'livestats', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ac-tech
    #: - [UDP] ac-tech
    ac_tech: 'AppType' = 2796, 'ac-tech', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] esp-encap
    #: - [UDP] esp-encap
    esp_encap: 'AppType' = 2797, 'esp-encap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TMESIS-UPShot
    #: - [UDP] TMESIS-UPShot
    tmesis_upshot: 'AppType' = 2798, 'tmesis-upshot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICON Discover
    #: - [UDP] ICON Discover
    icon_discover: 'AppType' = 2799, 'icon-discover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACC RAID
    #: - [UDP] ACC RAID
    acc_raid: 'AppType' = 2800, 'acc-raid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IGCP
    #: - [UDP] IGCP
    igcp: 'AppType' = 2801, 'igcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Veritas TCP1
    veritas_tcp1: 'AppType' = 2802, 'veritas-tcp1', TransportProtocol.get('tcp')

    #: [UDP] Veritas UDP1
    veritas_udp1: 'AppType' = 2802, 'veritas-udp1', TransportProtocol.get('udp')

    #: - [TCP] btprjctrl
    #: - [UDP] btprjctrl
    btprjctrl: 'AppType' = 2803, 'btprjctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] March Networks Digital Video Recorders and Enterprise Service Manager
    #:   products
    #: - [UDP] March Networks Digital Video Recorders and Enterprise Service Manager
    #:   products
    dvr_esm: 'AppType' = 2804, 'dvr-esm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WTA WSP-S
    #: - [UDP] WTA WSP-S
    wta_wsp_s: 'AppType' = 2805, 'wta-wsp-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cspuni
    #: - [UDP] cspuni
    cspuni: 'AppType' = 2806, 'cspuni', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cspmulti
    #: - [UDP] cspmulti
    cspmulti: 'AppType' = 2807, 'cspmulti', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] J-LAN-P
    #: - [UDP] J-LAN-P
    j_lan_p: 'AppType' = 2808, 'j-lan-p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CORBA LOC
    #: - [UDP] CORBA LOC
    corbaloc: 'AppType' = 2809, 'corbaloc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active Net Steward
    #: - [UDP] Active Net Steward
    netsteward: 'AppType' = 2810, 'netsteward', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GSI FTP
    #: - [UDP] GSI FTP
    gsiftp: 'AppType' = 2811, 'gsiftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] atmtcp
    #: - [UDP] atmtcp
    atmtcp: 'AppType' = 2812, 'atmtcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] llm-pass
    #: - [UDP] llm-pass
    llm_pass: 'AppType' = 2813, 'llm-pass', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] llm-csv
    #: - [UDP] llm-csv
    llm_csv: 'AppType' = 2814, 'llm-csv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LBC Measurement
    #: - [UDP] LBC Measurement
    lbc_measure: 'AppType' = 2815, 'lbc-measure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LBC Watchdog
    #: - [UDP] LBC Watchdog
    lbc_watchdog: 'AppType' = 2816, 'lbc-watchdog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rmlnk
    #: - [UDP] rmlnk
    rmlnk: 'AppType' = 2818, 'rmlnk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FC Fault Notification
    #: - [UDP] FC Fault Notification
    fc_faultnotify: 'AppType' = 2819, 'fc-faultnotify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UniVision
    #: - [UDP] UniVision
    univision: 'AppType' = 2820, 'univision', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS Authentication Service
    #: - [UDP] VERITAS Authentication Service
    vrts_at_port: 'AppType' = 2821, 'vrts-at-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ka0wuc
    #: - [UDP] ka0wuc
    ka0wuc: 'AppType' = 2822, 'ka0wuc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CQG Net/LAN
    #: - [UDP] CQG Net/LAN
    cqg_netlan: 'AppType' = 2823, 'cqg-netlan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CQG Net/LAN 1
    #: - [UDP] CQG Net/Lan 1
    cqg_netlan_1: 'AppType' = 2824, 'cqg-netlan-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] (unassigned) Possibly assigned
    possibly_assigned: 'AppType' = 2825, 'possibly_assigned', TransportProtocol.get('undefined')

    #: - [TCP] slc systemlog
    #: - [UDP] slc systemlog
    slc_systemlog: 'AppType' = 2826, 'slc-systemlog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] slc ctrlrloops
    #: - [UDP] slc ctrlrloops
    slc_ctrlrloops: 'AppType' = 2827, 'slc-ctrlrloops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITM License Manager
    #: - [UDP] ITM License Manager
    itm_lm: 'AppType' = 2828, 'itm-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] silkp1
    #: - [UDP] silkp1
    silkp1: 'AppType' = 2829, 'silkp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] silkp2
    #: - [UDP] silkp2
    silkp2: 'AppType' = 2830, 'silkp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] silkp3
    #: - [UDP] silkp3
    silkp3: 'AppType' = 2831, 'silkp3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] silkp4
    #: - [UDP] silkp4
    silkp4: 'AppType' = 2832, 'silkp4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] glishd
    #: - [UDP] glishd
    glishd: 'AppType' = 2833, 'glishd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EVTP
    #: - [UDP] EVTP
    evtp: 'AppType' = 2834, 'evtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EVTP-DATA
    #: - [UDP] EVTP-DATA
    evtp_data: 'AppType' = 2835, 'evtp-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] catalyst
    #: - [UDP] catalyst
    catalyst: 'AppType' = 2836, 'catalyst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Repliweb
    #: - [UDP] Repliweb
    repliweb: 'AppType' = 2837, 'repliweb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Starbot
    #: - [UDP] Starbot
    starbot: 'AppType' = 2838, 'starbot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMSig Port
    #: - [UDP] NMSig Port
    nmsigport_2817: 'AppType' = 2817, 'nmsigport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NMSigPort
    #: - [UDP] NMSigPort
    nmsigport_2839: 'AppType' = 2839, 'nmsigport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] l3-exprt
    #: - [UDP] l3-exprt
    l3_exprt: 'AppType' = 2840, 'l3-exprt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] l3-ranger
    #: - [UDP] l3-ranger
    l3_ranger: 'AppType' = 2841, 'l3-ranger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] l3-hawk
    #: - [UDP] l3-hawk
    l3_hawk: 'AppType' = 2842, 'l3-hawk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDnet
    #: - [UDP] PDnet
    pdnet: 'AppType' = 2843, 'pdnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BPCP POLL
    #: - [UDP] BPCP POLL
    bpcp_poll: 'AppType' = 2844, 'bpcp-poll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BPCP TRAP
    #: - [UDP] BPCP TRAP
    bpcp_trap: 'AppType' = 2845, 'bpcp-trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AIMPP Hello
    #: - [UDP] AIMPP Hello
    aimpp_hello: 'AppType' = 2846, 'aimpp-hello', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AIMPP Port Req
    #: - [UDP] AIMPP Port Req
    aimpp_port_req: 'AppType' = 2847, 'aimpp-port-req', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMT-BLC-PORT
    #: - [UDP] AMT-BLC-PORT
    amt_blc_port: 'AppType' = 2848, 'amt-blc-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FXP Communication
    #: - [UDP] FXP Communication
    fxp_286: 'AppType' = 286, 'fxp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FXP
    #: - [UDP] FXP
    fxp_2849: 'AppType' = 2849, 'fxp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaConsole
    #: - [UDP] MetaConsole
    metaconsole: 'AppType' = 2850, 'metaconsole', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] webemshttp
    #: - [UDP] webemshttp
    webemshttp: 'AppType' = 2851, 'webemshttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bears-01
    #: - [UDP] bears-01
    bears_01: 'AppType' = 2852, 'bears-01', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISPipes
    #: - [UDP] ISPipes
    ispipes: 'AppType' = 2853, 'ispipes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InfoMover
    #: - [UDP] InfoMover
    infomover: 'AppType' = 2854, 'infomover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] MSRP over TCP [:rfc:`4976`]
    msrp: 'AppType' = 2855, 'msrp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_2855: 'AppType' = 2855, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] cesdinv
    #: - [UDP] cesdinv
    cesdinv: 'AppType' = 2856, 'cesdinv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SimCtIP
    #: - [UDP] SimCtIP
    simctlp: 'AppType' = 2857, 'simctlp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ECNP
    #: - [UDP] ECNP
    ecnp: 'AppType' = 2858, 'ecnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active Memory
    #: - [UDP] Active Memory
    activememory: 'AppType' = 2859, 'activememory', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dialpad Voice 1
    #: - [UDP] Dialpad Voice 1
    dialpad_voice1: 'AppType' = 2860, 'dialpad-voice1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dialpad Voice 2
    #: - [UDP] Dialpad Voice 2
    dialpad_voice2: 'AppType' = 2861, 'dialpad-voice2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTG Protocol
    #: - [UDP] TTG Protocol
    ttg_protocol: 'AppType' = 2862, 'ttg-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sonar Data
    #: - [UDP] Sonar Data
    sonardata: 'AppType' = 2863, 'sonardata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] main 5001 cmd
    #: - [UDP] main 5001 cmd
    astronova_main: 'AppType' = 2864, 'astronova-main', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pit-vpn
    #: - [UDP] pit-vpn
    pit_vpn: 'AppType' = 2865, 'pit-vpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iwlistener
    #: - [UDP] iwlistener
    iwlistener: 'AppType' = 2866, 'iwlistener', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] esps-portal
    #: - [UDP] esps-portal
    esps_portal: 'AppType' = 2867, 'esps-portal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Norman Proprietaqry Events Protocol
    #: - [UDP] Norman Proprietaqry Events Protocol
    npep_messaging: 'AppType' = 2868, 'npep-messaging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICSLAP
    #: - [UDP] ICSLAP
    icslap: 'AppType' = 2869, 'icslap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] daishi
    #: - [UDP] daishi
    daishi: 'AppType' = 2870, 'daishi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSI Select Play
    #: - [UDP] MSI Select Play
    msi_selectplay: 'AppType' = 2871, 'msi-selectplay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADIX
    #: - [UDP] RADIX
    radix: 'AppType' = 2872, 'radix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PubSub Realtime Telemetry Protocol
    #: - [UDP] PubSub Realtime Telemetry Protocol
    psrt: 'AppType' = 2873, 'psrt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DX Message Base Transport Protocol
    #: - [UDP] DX Message Base Transport Protocol
    dxmessagebase1: 'AppType' = 2874, 'dxmessagebase1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DX Message Base Transport Protocol
    #: - [UDP] DX Message Base Transport Protocol
    dxmessagebase2: 'AppType' = 2875, 'dxmessagebase2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPS Tunnel
    #: - [UDP] SPS Tunnel
    sps_tunnel: 'AppType' = 2876, 'sps-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BLUELANCE
    #: - [UDP] BLUELANCE
    bluelance: 'AppType' = 2877, 'bluelance', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AAP
    #: - [UDP] AAP
    aap: 'AppType' = 2878, 'aap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ucentric-ds
    #: - [UDP] ucentric-ds
    ucentric_ds: 'AppType' = 2879, 'ucentric-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synapse Transport
    #: - [UDP] Synapse Transport
    synapse: 'AppType' = 2880, 'synapse', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDSP
    #: - [UDP] NDSP
    ndsp: 'AppType' = 2881, 'ndsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDTP
    #: - [UDP] NDTP
    ndtp: 'AppType' = 2882, 'ndtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDNP
    #: - [UDP] NDNP
    ndnp: 'AppType' = 2883, 'ndnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Flash Msg
    #: - [UDP] Flash Msg
    flashmsg: 'AppType' = 2884, 'flashmsg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TopFlow
    #: - [UDP] TopFlow
    topflow: 'AppType' = 2885, 'topflow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RESPONSELOGIC
    #: - [UDP] RESPONSELOGIC
    responselogic: 'AppType' = 2886, 'responselogic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aironet
    #: - [UDP] aironet
    aironetddp: 'AppType' = 2887, 'aironetddp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPCSDLOBBY
    #: - [UDP] SPCSDLOBBY
    spcsdlobby: 'AppType' = 2888, 'spcsdlobby', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSOM
    #: - [UDP] RSOM
    rsom: 'AppType' = 2889, 'rsom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSPCLMULTI
    #: - [UDP] CSPCLMULTI
    cspclmulti: 'AppType' = 2890, 'cspclmulti', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CINEGRFX-ELMD License Manager
    #: - [UDP] CINEGRFX-ELMD License Manager
    cinegrfx_elmd: 'AppType' = 2891, 'cinegrfx-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNIFFERDATA
    #: - [UDP] SNIFFERDATA
    snifferdata: 'AppType' = 2892, 'snifferdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSECONNECTOR
    #: - [UDP] VSECONNECTOR
    vseconnector: 'AppType' = 2893, 'vseconnector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ABACUS-REMOTE
    #: - [UDP] ABACUS-REMOTE
    abacus_remote: 'AppType' = 2894, 'abacus-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NATUS LINK
    #: - [UDP] NATUS LINK
    natuslink: 'AppType' = 2895, 'natuslink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ECOVISIONG6-1
    #: - [UDP] ECOVISIONG6-1
    ecovisiong6_1: 'AppType' = 2896, 'ecovisiong6-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citrix RTMP
    #: - [UDP] Citrix RTMP
    citrix_rtmp: 'AppType' = 2897, 'citrix-rtmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APPLIANCE-CFG
    #: - [UDP] APPLIANCE-CFG
    appliance_cfg: 'AppType' = 2898, 'appliance-cfg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] POWERGEMPLUS
    #: - [UDP] POWERGEMPLUS
    powergemplus: 'AppType' = 2899, 'powergemplus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QUICKSUITE
    #: - [UDP] QUICKSUITE
    quicksuite: 'AppType' = 2900, 'quicksuite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ALLSTORCNS
    #: - [UDP] ALLSTORCNS
    allstorcns: 'AppType' = 2901, 'allstorcns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NET ASPI
    #: - [UDP] NET ASPI
    netaspi: 'AppType' = 2902, 'netaspi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SUITCASE
    #: - [UDP] SUITCASE
    suitcase: 'AppType' = 2903, 'suitcase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] M2UA
    #: - [UDP] M2UA
    #: - [SCTP] M2UA
    m2ua: 'AppType' = 2904, 'm2ua', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] M3UA [:rfc:`4666`]
    #: - [SCTP] M3UA [:rfc:`4666`]
    m3ua: 'AppType' = 2905, 'm3ua', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [N/A] De-registered
    de_registered_585: 'AppType' = 585, 'de_registered', TransportProtocol.get('undefined')

    #: [UDP] De-registered
    de_registered_2905: 'AppType' = 2905, 'de_registered', TransportProtocol.get('udp')

    #: - [TCP] CALLER9
    #: - [UDP] CALLER9
    caller9: 'AppType' = 2906, 'caller9', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WEBMETHODS B2B
    #: - [UDP] WEBMETHODS B2B
    webmethods_b2b: 'AppType' = 2907, 'webmethods-b2b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mao
    #: - [UDP] mao
    mao: 'AppType' = 2908, 'mao', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Funk Dialout
    #: - [UDP] Funk Dialout
    funk_dialout: 'AppType' = 2909, 'funk-dialout', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TDAccess
    #: - [UDP] TDAccess
    tdaccess: 'AppType' = 2910, 'tdaccess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Blockade
    #: - [UDP] Blockade
    blockade: 'AppType' = 2911, 'blockade', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Epicon
    #: - [UDP] Epicon
    epicon: 'AppType' = 2912, 'epicon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Booster Ware
    #: - [UDP] Booster Ware
    boosterware: 'AppType' = 2913, 'boosterware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Game Lobby
    #: - [UDP] Game Lobby
    gamelobby: 'AppType' = 2914, 'gamelobby', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TK Socket
    #: - [UDP] TK Socket
    tksocket: 'AppType' = 2915, 'tksocket', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Elvin Server IANA assigned this well-formed service name as a
    #:   replacement for "elvin_server".
    #: - [TCP] Elvin Server
    #: - [UDP] Elvin Server IANA assigned this well-formed service name as a
    #:   replacement for "elvin_server".
    #: - [UDP] Elvin Server
    elvin_server: 'AppType' = 2916, 'elvin-server', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Elvin Client IANA assigned this well-formed service name as a
    #:   replacement for "elvin_client".
    #: - [TCP] Elvin Client
    #: - [UDP] Elvin Client IANA assigned this well-formed service name as a
    #:   replacement for "elvin_client".
    #: - [UDP] Elvin Client
    elvin_client: 'AppType' = 2917, 'elvin-client', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Kasten Chase Pad
    #: - [UDP] Kasten Chase Pad
    kastenchasepad: 'AppType' = 2918, 'kastenchasepad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] roboER
    #: - [UDP] roboER
    roboer: 'AppType' = 2919, 'roboer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] roboEDA
    #: - [UDP] roboEDA
    roboeda: 'AppType' = 2920, 'roboeda', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CESD Contents Delivery Management
    #: - [UDP] CESD Contents Delivery Management
    cesdcdman: 'AppType' = 2921, 'cesdcdman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CESD Contents Delivery Data Transfer
    #: - [UDP] CESD Contents Delivery Data Transfer
    cesdcdtrn: 'AppType' = 2922, 'cesdcdtrn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WTA-WSP-WTP-S
    #: - [UDP] WTA-WSP-WTP-S
    wta_wsp_wtp_s: 'AppType' = 2923, 'wta-wsp-wtp-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PRECISE-VIP
    #: - [UDP] PRECISE-VIP
    precise_vip: 'AppType' = 2924, 'precise-vip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned (FRP-Released 12/7/00)
    unassigned_2925: 'AppType' = 2925, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] MOBILE-FILE-DL
    #: - [UDP] MOBILE-FILE-DL
    mobile_file_dl: 'AppType' = 2926, 'mobile-file-dl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UNIMOBILECTRL
    #: - [UDP] UNIMOBILECTRL
    unimobilectrl: 'AppType' = 2927, 'unimobilectrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] REDSTONE-CPSS
    #: - [UDP] REDSTONE-CPSS
    redstone_cpss: 'AppType' = 2928, 'redstone-cpss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMX-WEBADMIN
    #: - [UDP] AMX-WEBADMIN
    amx_webadmin: 'AppType' = 2929, 'amx-webadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMX-WEBLINX
    #: - [UDP] AMX-WEBLINX
    amx_weblinx: 'AppType' = 2930, 'amx-weblinx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Circle-X
    #: - [UDP] Circle-X
    circle_x: 'AppType' = 2931, 'circle-x', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INCP
    #: - [UDP] INCP
    incp: 'AppType' = 2932, 'incp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 4-TIER OPM GW
    #: - [UDP] 4-TIER OPM GW
    AppType_4_tieropmgw: 'AppType' = 2933, '4-tieropmgw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 4-TIER OPM CLI
    #: - [UDP] 4-TIER OPM CLI
    AppType_4_tieropmcli: 'AppType' = 2934, '4-tieropmcli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QTP
    #: - [UDP] QTP
    qtp: 'AppType' = 2935, 'qtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OTPatch
    #: - [UDP] OTPatch
    otpatch: 'AppType' = 2936, 'otpatch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PNACONSULT-LM
    #: - [UDP] PNACONSULT-LM
    pnaconsult_lm: 'AppType' = 2937, 'pnaconsult-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SM-PAS-1
    #: - [UDP] SM-PAS-1
    sm_pas_1: 'AppType' = 2938, 'sm-pas-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SM-PAS-2
    #: - [UDP] SM-PAS-2
    sm_pas_2: 'AppType' = 2939, 'sm-pas-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SM-PAS-3
    #: - [UDP] SM-PAS-3
    sm_pas_3: 'AppType' = 2940, 'sm-pas-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SM-PAS-4
    #: - [UDP] SM-PAS-4
    sm_pas_4: 'AppType' = 2941, 'sm-pas-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SM-PAS-5
    #: - [UDP] SM-PAS-5
    sm_pas_5: 'AppType' = 2942, 'sm-pas-5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTNRepository
    #: - [UDP] TTNRepository
    ttnrepository: 'AppType' = 2943, 'ttnrepository', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Megaco H-248
    #: - [UDP] Megaco H-248
    #: - [SCTP] Megaco-H.248 text
    megaco_h248: 'AppType' = 2944, 'megaco-h248', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] H248 Binary
    #: - [UDP] H248 Binary
    #: - [SCTP] Megaco/H.248 binary
    h248_binary: 'AppType' = 2945, 'h248-binary', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] FJSVmpor
    #: - [UDP] FJSVmpor
    fjsvmpor: 'AppType' = 2946, 'fjsvmpor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GPS Daemon request/response protocol
    #: - [UDP] GPS Daemon request/response protocol
    gpsd: 'AppType' = 2947, 'gpsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WAP PUSH
    #: - [UDP] WAP PUSH
    wap_push: 'AppType' = 2948, 'wap-push', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WAP PUSH SECURE
    #: - [UDP] WAP PUSH SECURE
    wap_pushsecure: 'AppType' = 2949, 'wap-pushsecure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESIP
    #: - [UDP] ESIP
    esip: 'AppType' = 2950, 'esip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OTTP
    #: - [UDP] OTTP
    ottp: 'AppType' = 2951, 'ottp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPFWSAS
    #: - [UDP] MPFWSAS
    mpfwsas: 'AppType' = 2952, 'mpfwsas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OVALARMSRV
    #: - [UDP] OVALARMSRV
    ovalarmsrv: 'AppType' = 2953, 'ovalarmsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OVALARMSRV-CMD
    #: - [UDP] OVALARMSRV-CMD
    ovalarmsrv_cmd: 'AppType' = 2954, 'ovalarmsrv-cmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSNOTIFY
    #: - [UDP] CSNOTIFY
    csnotify: 'AppType' = 2955, 'csnotify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OVRIMOSDBMAN
    #: - [UDP] OVRIMOSDBMAN
    ovrimosdbman: 'AppType' = 2956, 'ovrimosdbman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JAMCT5
    #: - [UDP] JAMCT5
    jmact5: 'AppType' = 2957, 'jmact5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JAMCT6
    #: - [UDP] JAMCT6
    jmact6: 'AppType' = 2958, 'jmact6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMOPAGT
    #: - [UDP] RMOPAGT
    rmopagt: 'AppType' = 2959, 'rmopagt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DFOXSERVER
    #: - [UDP] DFOXSERVER
    dfoxserver: 'AppType' = 2960, 'dfoxserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BOLDSOFT-LM
    #: - [UDP] BOLDSOFT-LM
    boldsoft_lm: 'AppType' = 2961, 'boldsoft-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPH-POLICY-CLI
    #: - [UDP] IPH-POLICY-CLI
    iph_policy_cli: 'AppType' = 2962, 'iph-policy-cli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPH-POLICY-ADM
    #: - [UDP] IPH-POLICY-ADM
    iph_policy_adm: 'AppType' = 2963, 'iph-policy-adm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BULLANT SRAP
    #: - [UDP] BULLANT SRAP
    bullant_srap: 'AppType' = 2964, 'bullant-srap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BULLANT RAP
    #: - [UDP] BULLANT RAP
    bullant_rap: 'AppType' = 2965, 'bullant-rap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDP-INFOTRIEVE
    #: - [UDP] IDP-INFOTRIEVE
    idp_infotrieve: 'AppType' = 2966, 'idp-infotrieve', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSC-AGENT
    #: - [UDP] SSC-AGENT
    ssc_agent: 'AppType' = 2967, 'ssc-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ENPP
    #: - [UDP] ENPP
    enpp: 'AppType' = 2968, 'enpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESSP
    #: - [UDP] ESSP
    essp: 'AppType' = 2969, 'essp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INDEX-NET
    #: - [UDP] INDEX-NET
    index_net: 'AppType' = 2970, 'index-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetClip clipboard daemon
    #: - [UDP] NetClip clipboard daemon
    netclip: 'AppType' = 2971, 'netclip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PMSM Webrctl
    #: - [UDP] PMSM Webrctl
    pmsm_webrctl: 'AppType' = 2972, 'pmsm-webrctl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SV Networks
    #: - [UDP] SV Networks
    svnetworks: 'AppType' = 2973, 'svnetworks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Signal
    #: - [UDP] Signal
    signal: 'AppType' = 2974, 'signal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Configuration Management Service
    #: - [UDP] Fujitsu Configuration Management Service
    fjmpcm: 'AppType' = 2975, 'fjmpcm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CNS Server Port
    #: - [UDP] CNS Server Port
    cns_srv_port: 'AppType' = 2976, 'cns-srv-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTCs Enterprise Test Access Protocol - NS
    #: - [UDP] TTCs Enterprise Test Access Protocol - NS
    ttc_etap_ns: 'AppType' = 2977, 'ttc-etap-ns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTCs Enterprise Test Access Protocol - DS
    #: - [UDP] TTCs Enterprise Test Access Protocol - DS
    ttc_etap_ds: 'AppType' = 2978, 'ttc-etap-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H.263 Video Streaming
    #: - [UDP] H.263 Video Streaming
    h263_video: 'AppType' = 2979, 'h263-video', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instant Messaging Service
    #: - [UDP] Instant Messaging Service
    wimd: 'AppType' = 2980, 'wimd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MYLXAMPORT
    #: - [UDP] MYLXAMPORT
    mylxamport: 'AppType' = 2981, 'mylxamport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IWB-WHITEBOARD
    #: - [UDP] IWB-WHITEBOARD
    iwb_whiteboard: 'AppType' = 2982, 'iwb-whiteboard', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETPLAN
    #: - [UDP] NETPLAN
    netplan: 'AppType' = 2983, 'netplan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPIDSADMIN
    #: - [UDP] HPIDSADMIN
    hpidsadmin: 'AppType' = 2984, 'hpidsadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPIDSAGENT
    #: - [UDP] HPIDSAGENT
    hpidsagent: 'AppType' = 2985, 'hpidsagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STONEFALLS
    #: - [UDP] STONEFALLS
    stonefalls: 'AppType' = 2986, 'stonefalls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] identify
    #: - [UDP] identify
    identify: 'AppType' = 2987, 'identify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HIPPA Reporting Protocol
    #: - [UDP] HIPPA Reporting Protocol
    hippad: 'AppType' = 2988, 'hippad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ZARKOV Intelligent Agent Communication
    #: - [UDP] ZARKOV Intelligent Agent Communication
    zarkov: 'AppType' = 2989, 'zarkov', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BOSCAP
    #: - [UDP] BOSCAP
    boscap: 'AppType' = 2990, 'boscap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WKSTN-MON
    #: - [UDP] WKSTN-MON
    wkstn_mon: 'AppType' = 2991, 'wkstn-mon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avenyo Server
    #: - [UDP] Avenyo Server
    avenyo: 'AppType' = 2992, 'avenyo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS VIS1
    #: - [UDP] VERITAS VIS1
    veritas_vis1: 'AppType' = 2993, 'veritas-vis1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS VIS2
    #: - [UDP] VERITAS VIS2
    veritas_vis2: 'AppType' = 2994, 'veritas-vis2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDRS
    #: - [UDP] IDRS
    idrs: 'AppType' = 2995, 'idrs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vsixml
    #: - [UDP] vsixml
    vsixml: 'AppType' = 2996, 'vsixml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] REBOL
    #: - [UDP] REBOL
    rebol: 'AppType' = 2997, 'rebol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real Secure
    #: - [UDP] Real Secure
    realsecure: 'AppType' = 2998, 'realsecure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RemoteWare Unassigned
    #: - [UDP] RemoteWare Unassigned
    remoteware_un: 'AppType' = 2999, 'remoteware-un', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HBCI
    #: - [UDP] HBCI
    hbci: 'AppType' = 3000, 'hbci', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RemoteWare Client
    #: - [UDP] RemoteWare Client
    remoteware_cl: 'AppType' = 3000, 'remoteware-cl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] OrigoDB Server Native Interface
    origo_native: 'AppType' = 3001, 'origo-native', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3001: 'AppType' = 3001, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] EXLM Agent
    #: - [UDP] EXLM Agent
    exlm_agent: 'AppType' = 3002, 'exlm-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RemoteWare Server
    #: - [UDP] RemoteWare Server
    remoteware_srv: 'AppType' = 3002, 'remoteware-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CGMS
    #: - [UDP] CGMS
    cgms: 'AppType' = 3003, 'cgms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Csoft Agent
    #: - [UDP] Csoft Agent
    csoftragent: 'AppType' = 3004, 'csoftragent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Genius License Manager
    #: - [UDP] Genius License Manager
    geniuslm: 'AppType' = 3005, 'geniuslm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instant Internet Admin
    #: - [UDP] Instant Internet Admin
    ii_admin: 'AppType' = 3006, 'ii-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lotus Mail Tracking Agent Protocol
    #: - [UDP] Lotus Mail Tracking Agent Protocol
    lotusmtap: 'AppType' = 3007, 'lotusmtap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Midnight Technologies
    #: - [UDP] Midnight Technologies
    midnight_tech: 'AppType' = 3008, 'midnight-tech', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PXC-NTFY
    #: - [UDP] PXC-NTFY
    pxc_ntfy: 'AppType' = 3009, 'pxc-ntfy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Telerate Workstation
    gw: 'AppType' = 3010, 'gw', TransportProtocol.get('tcp')

    #: [UDP] Telerate Workstation
    ping_pong: 'AppType' = 3010, 'ping-pong', TransportProtocol.get('udp')

    #: - [TCP] Trusted Web
    #: - [UDP] Trusted Web
    trusted_web: 'AppType' = 3011, 'trusted-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trusted Web Client
    #: - [UDP] Trusted Web Client
    twsdss: 'AppType' = 3012, 'twsdss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gilat Sky Surfer
    #: - [UDP] Gilat Sky Surfer
    gilatskysurfer: 'AppType' = 3013, 'gilatskysurfer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Broker Service IANA assigned this well-formed service name as a
    #:   replacement for "broker_service".
    #: - [TCP] Broker Service
    #: - [UDP] Broker Service IANA assigned this well-formed service name as a
    #:   replacement for "broker_service".
    #: - [UDP] Broker Service
    broker_service: 'AppType' = 3014, 'broker-service', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] NATI DSTP
    #: - [UDP] NATI DSTP
    nati_dstp: 'AppType' = 3015, 'nati-dstp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Notify Server IANA assigned this well-formed service name as a
    #:   replacement for "notify_srvr".
    #: - [TCP] Notify Server
    #: - [UDP] Notify Server IANA assigned this well-formed service name as a
    #:   replacement for "notify_srvr".
    #: - [UDP] Notify Server
    notify_srvr: 'AppType' = 3016, 'notify-srvr', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Event Listener IANA assigned this well-formed service name as a
    #:   replacement for "event_listener".
    #: - [TCP] Event Listener
    #: - [UDP] Event Listener IANA assigned this well-formed service name as a
    #:   replacement for "event_listener".
    #: - [UDP] Event Listener
    event_listener: 'AppType' = 3017, 'event-listener', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry IANA assigned this well-formed service name as a
    #:   replacement for "srvc_registry".
    #: - [TCP] Service Registry
    #: - [UDP] Service Registry IANA assigned this well-formed service name as a
    #:   replacement for "srvc_registry".
    #: - [UDP] Service Registry
    srvc_registry: 'AppType' = 3018, 'srvc-registry', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Resource Manager IANA assigned this well-formed service name as a
    #:   replacement for "resource_mgr".
    #: - [TCP] Resource Manager
    #: - [UDP] Resource Manager IANA assigned this well-formed service name as a
    #:   replacement for "resource_mgr".
    #: - [UDP] Resource Manager
    resource_mgr: 'AppType' = 3019, 'resource-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] CIFS
    #: - [UDP] CIFS
    cifs: 'AppType' = 3020, 'cifs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AGRI Server
    #: - [UDP] AGRI Server
    agriserver: 'AppType' = 3021, 'agriserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSREGAGENT
    #: - [UDP] CSREGAGENT
    csregagent: 'AppType' = 3022, 'csregagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] magicnotes
    #: - [UDP] magicnotes
    magicnotes: 'AppType' = 3023, 'magicnotes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDS_SSO IANA assigned this well-formed service name as a replacement
    #:   for "nds_sso".
    #: - [TCP] NDS_SSO
    #: - [UDP] NDS_SSO IANA assigned this well-formed service name as a replacement
    #:   for "nds_sso".
    #: - [UDP] NDS_SSO
    nds_sso: 'AppType' = 3024, 'nds-sso', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Arepa Raft
    #: - [UDP] Arepa Raft
    arepa_raft: 'AppType' = 3025, 'arepa-raft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AGRI Gateway
    #: - [UDP] AGRI Gateway
    agri_gateway: 'AppType' = 3026, 'agri-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LiebDevMgmt_C IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_C".
    #: - [TCP] LiebDevMgmt_C
    #: - [UDP] LiebDevMgmt_C IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_C".
    #: - [UDP] LiebDevMgmt_C
    liebdevmgmt_c: 'AppType' = 3027, 'liebdevmgmt-c', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] LiebDevMgmt_DM IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_DM".
    #: - [TCP] LiebDevMgmt_DM
    #: - [UDP] LiebDevMgmt_DM IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_DM".
    #: - [UDP] LiebDevMgmt_DM
    liebdevmgmt_dm: 'AppType' = 3028, 'liebdevmgmt-dm', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] LiebDevMgmt_A IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_A".
    #: - [TCP] LiebDevMgmt_A
    #: - [UDP] LiebDevMgmt_A IANA assigned this well-formed service name as a
    #:   replacement for "LiebDevMgmt_A".
    #: - [UDP] LiebDevMgmt_A
    liebdevmgmt_a: 'AppType' = 3029, 'liebdevmgmt-a', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Arepa Cas
    #: - [UDP] Arepa Cas
    arepa_cas: 'AppType' = 3030, 'arepa-cas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote AppleEvents/PPC Toolbox
    #: - [UDP] Remote AppleEvents/PPC Toolbox
    eppc: 'AppType' = 3031, 'eppc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Redwood Chat
    #: - [UDP] Redwood Chat
    redwood_chat: 'AppType' = 3032, 'redwood-chat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDB
    #: - [UDP] PDB
    pdb: 'AppType' = 3033, 'pdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Osmosis / Helix (R) AEEA Port
    #: - [UDP] Osmosis / Helix (R) AEEA Port
    osmosis_aeea: 'AppType' = 3034, 'osmosis-aeea', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FJSV gssagt
    #: - [UDP] FJSV gssagt
    fjsv_gssagt: 'AppType' = 3035, 'fjsv-gssagt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hagel DUMP
    #: - [UDP] Hagel DUMP
    hagel_dump: 'AppType' = 3036, 'hagel-dump', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP SAN Mgmt
    #: - [UDP] HP SAN Mgmt
    hp_san_mgmt: 'AppType' = 3037, 'hp-san-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Santak UPS
    #: - [UDP] Santak UPS
    santak_ups: 'AppType' = 3038, 'santak-ups', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cogitate, Inc.
    #: - [UDP] Cogitate, Inc.
    cogitate: 'AppType' = 3039, 'cogitate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tomato Springs
    #: - [UDP] Tomato Springs
    tomato_springs: 'AppType' = 3040, 'tomato-springs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] di-traceware
    #: - [UDP] di-traceware
    di_traceware: 'AppType' = 3041, 'di-traceware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] journee
    #: - [UDP] journee
    journee: 'AppType' = 3042, 'journee', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Broadcast Routing Protocol
    #: - [UDP] Broadcast Routing Protocol
    brp: 'AppType' = 3043, 'brp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Extensible Provisioning Protocol [:rfc:`5734`]
    #: - [UDP] Extensible Provisioning Protocol [:rfc:`5734`]
    epp_700: 'AppType' = 700, 'epp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EndPoint Protocol
    #: - [UDP] EndPoint Protocol
    epp_3044: 'AppType' = 3044, 'epp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ResponseNet
    #: - [UDP] ResponseNet
    responsenet: 'AppType' = 3045, 'responsenet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] di-ase
    #: - [UDP] di-ase
    di_ase: 'AppType' = 3046, 'di-ase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fast Security HL Server
    #: - [UDP] Fast Security HL Server
    hlserver: 'AppType' = 3047, 'hlserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sierra Net PC Trader
    #: - [UDP] Sierra Net PC Trader
    pctrader: 'AppType' = 3048, 'pctrader', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSWS
    #: - [UDP] NSWS
    nsws: 'AppType' = 3049, 'nsws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gds_db IANA assigned this well-formed service name as a replacement
    #:   for "gds_db".
    #: - [TCP] gds_db
    #: - [UDP] gds_db IANA assigned this well-formed service name as a replacement
    #:   for "gds_db".
    #: - [UDP] gds_db
    gds_db: 'AppType' = 3050, 'gds-db', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Galaxy Server
    #: - [UDP] Galaxy Server
    galaxy_server: 'AppType' = 3051, 'galaxy-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 3052
    #: - [UDP] APC 3052
    apc_3052: 'AppType' = 3052, 'apc-3052', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dsom-server
    #: - [UDP] dsom-server
    dsom_server: 'AppType' = 3053, 'dsom-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMT CNF PROT
    #: - [UDP] AMT CNF PROT
    amt_cnf_prot: 'AppType' = 3054, 'amt-cnf-prot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Policy Server
    #: - [UDP] Policy Server
    policyserver: 'AppType' = 3055, 'policyserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CDL Server
    #: - [UDP] CDL Server
    cdl_server: 'AppType' = 3056, 'cdl-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GoAhead FldUp
    #: - [UDP] GoAhead FldUp
    goahead_fldup: 'AppType' = 3057, 'goahead-fldup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] videobeans
    #: - [UDP] videobeans
    videobeans: 'AppType' = 3058, 'videobeans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qsoft
    #: - [UDP] qsoft
    qsoft: 'AppType' = 3059, 'qsoft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] interserver
    #: - [UDP] interserver
    interserver: 'AppType' = 3060, 'interserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cautcpd
    #: - [UDP] cautcpd
    cautcpd: 'AppType' = 3061, 'cautcpd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncacn-ip-tcp
    #: - [UDP] ncacn-ip-tcp
    ncacn_ip_tcp: 'AppType' = 3062, 'ncacn-ip-tcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ncadg-ip-udp
    #: - [UDP] ncadg-ip-udp
    ncadg_ip_udp: 'AppType' = 3063, 'ncadg-ip-udp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Port Redirector
    #: - [UDP] Remote Port Redirector
    rprt: 'AppType' = 3064, 'rprt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] slinterbase
    #: - [UDP] slinterbase
    slinterbase: 'AppType' = 3065, 'slinterbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETATTACHSDMP
    #: - [UDP] NETATTACHSDMP
    netattachsdmp: 'AppType' = 3066, 'netattachsdmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FJHPJP
    #: - [UDP] FJHPJP
    fjhpjp: 'AppType' = 3067, 'fjhpjp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ls3 Broadcast
    #: - [UDP] ls3 Broadcast
    ls3bcast: 'AppType' = 3068, 'ls3bcast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ls3
    #: - [UDP] ls3
    ls3: 'AppType' = 3069, 'ls3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MGXSWITCH
    #: - [UDP] MGXSWITCH
    mgxswitch: 'AppType' = 3070, 'mgxswitch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Crossplatform replication protocol
    xplat_replicate: 'AppType' = 3071, 'xplat-replicate', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3071: 'AppType' = 3071, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ContinuStor Monitor Port
    #: - [UDP] ContinuStor Monitor Port
    csd_monitor: 'AppType' = 3072, 'csd-monitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Very simple chatroom prot
    #: - [UDP] Very simple chatroom prot
    vcrp: 'AppType' = 3073, 'vcrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xbox game port
    #: - [UDP] Xbox game port
    xbox: 'AppType' = 3074, 'xbox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orbix 2000 Locator
    #: - [UDP] Orbix 2000 Locator
    orbix_locator: 'AppType' = 3075, 'orbix-locator', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orbix 2000 Config
    #: - [UDP] Orbix 2000 Config
    orbix_config: 'AppType' = 3076, 'orbix-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orbix 2000 Locator SSL
    #: - [UDP] Orbix 2000 Locator SSL
    orbix_loc_ssl: 'AppType' = 3077, 'orbix-loc-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Orbix 2000 Locator SSL
    #: - [UDP] Orbix 2000 Locator SSL
    orbix_cfg_ssl: 'AppType' = 3078, 'orbix-cfg-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LV Front Panel
    #: - [UDP] LV Front Panel
    lv_frontpanel: 'AppType' = 3079, 'lv-frontpanel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] stm_pproc IANA assigned this well-formed service name as a replacement
    #:   for "stm_pproc".
    #: - [TCP] stm_pproc
    #: - [UDP] stm_pproc IANA assigned this well-formed service name as a replacement
    #:   for "stm_pproc".
    #: - [UDP] stm_pproc
    stm_pproc: 'AppType' = 3080, 'stm-pproc', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] TL1-LV
    #: - [UDP] TL1-LV
    tl1_lv: 'AppType' = 3081, 'tl1-lv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TL1-RAW
    #: - [UDP] TL1-RAW
    tl1_raw: 'AppType' = 3082, 'tl1-raw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TL1-TELNET
    #: - [UDP] TL1-TELNET
    tl1_telnet: 'AppType' = 3083, 'tl1-telnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITM-MCCS
    #: - [UDP] ITM-MCCS
    itm_mccs: 'AppType' = 3084, 'itm-mccs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCIHReq
    #: - [UDP] PCIHReq
    pcihreq: 'AppType' = 3085, 'pcihreq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JDL-DBKitchen
    #: - [UDP] JDL-DBKitchen
    jdl_dbkitchen: 'AppType' = 3086, 'jdl-dbkitchen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Asoki SMA
    #: - [UDP] Asoki SMA
    asoki_sma: 'AppType' = 3087, 'asoki-sma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eXtensible Data Transfer Protocol
    #: - [UDP] eXtensible Data Transfer Protocol
    xdtp: 'AppType' = 3088, 'xdtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ParaTek Agent Linking
    #: - [UDP] ParaTek Agent Linking
    ptk_alink: 'AppType' = 3089, 'ptk-alink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senforce Session Services
    #: - [UDP] Senforce Session Services
    stss: 'AppType' = 3090, 'stss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 1Ci Server Management
    #: - [UDP] 1Ci Server Management
    AppType_1ci_smcs: 'AppType' = 3091, '1ci-smcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_3092: 'AppType' = 3092, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Jiiva RapidMQ Center
    #: - [UDP] Jiiva RapidMQ Center
    rapidmq_center: 'AppType' = 3093, 'rapidmq-center', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jiiva RapidMQ Registry
    #: - [UDP] Jiiva RapidMQ Registry
    rapidmq_reg: 'AppType' = 3094, 'rapidmq-reg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Panasas rendezvous port
    #: - [UDP] Panasas rendezvous port
    panasas: 'AppType' = 3095, 'panasas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active Print Server Port
    #: - [UDP] Active Print Server Port
    ndl_aps: 'AppType' = 3096, 'ndl-aps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_3097: 'AppType' = 3097, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] ITU-T Q.1902.1/Q.2150.3
    itu_bicc_stc: 'AppType' = 3097, 'itu-bicc-stc', TransportProtocol.get('sctp')

    #: - [TCP] Universal Message Manager
    #: - [UDP] Universal Message Manager
    umm_port: 'AppType' = 3098, 'umm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CHIPSY Machine Daemon
    #: - [UDP] CHIPSY Machine Daemon
    chmd: 'AppType' = 3099, 'chmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpCon/xps
    #: - [UDP] OpCon/xps
    opcon_xps: 'AppType' = 3100, 'opcon-xps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP PolicyXpert PIB Server
    #: - [UDP] HP PolicyXpert PIB Server
    hp_pxpib: 'AppType' = 3101, 'hp-pxpib', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SoftlinK Slave Mon Port
    #: - [UDP] SoftlinK Slave Mon Port
    slslavemon: 'AppType' = 3102, 'slslavemon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Autocue SMI Protocol
    #: - [UDP] Autocue SMI Protocol
    autocuesmi: 'AppType' = 3103, 'autocuesmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Autocue Logger Protocol
    autocuelog: 'AppType' = 3104, 'autocuelog', TransportProtocol.get('tcp')

    #: [UDP] Autocue Time Service
    autocuetime: 'AppType' = 3104, 'autocuetime', TransportProtocol.get('udp')

    #: - [TCP] Cardbox
    #: - [UDP] Cardbox
    cardbox: 'AppType' = 3105, 'cardbox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cardbox HTTP
    #: - [UDP] Cardbox HTTP
    cardbox_http: 'AppType' = 3106, 'cardbox-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Business protocol
    #: - [UDP] Business protocol
    business: 'AppType' = 3107, 'business', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Geolocate protocol
    #: - [UDP] Geolocate protocol
    geolocate: 'AppType' = 3108, 'geolocate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Personnel protocol
    #: - [UDP] Personnel protocol
    personnel: 'AppType' = 3109, 'personnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simulator control port
    #: - [UDP] simulator control port
    sim_control: 'AppType' = 3110, 'sim-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Web Synchronous Services
    #: - [UDP] Web Synchronous Services
    wsynch: 'AppType' = 3111, 'wsynch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KDE System Guard
    #: - [UDP] KDE System Guard
    ksysguard: 'AppType' = 3112, 'ksysguard', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CS-Authenticate Svr Port
    #: - [UDP] CS-Authenticate Svr Port
    cs_auth_svr: 'AppType' = 3113, 'cs-auth-svr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCM AutoDiscover
    #: - [UDP] CCM AutoDiscover
    ccmad: 'AppType' = 3114, 'ccmad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCTET Master
    #: - [UDP] MCTET Master
    mctet_master: 'AppType' = 3115, 'mctet-master', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCTET Gateway
    #: - [UDP] MCTET Gateway
    mctet_gateway: 'AppType' = 3116, 'mctet-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCTET Jserv
    #: - [UDP] MCTET Jserv
    mctet_jserv: 'AppType' = 3117, 'mctet-jserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PKAgent
    #: - [UDP] PKAgent
    pkagent: 'AppType' = 3118, 'pkagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2000 Kernel Port
    #: - [UDP] D2000 Kernel Port
    d2000kernel: 'AppType' = 3119, 'd2000kernel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2000 Webserver Port
    #: - [UDP] D2000 Webserver Port
    d2000webserver: 'AppType' = 3120, 'd2000webserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] The pacemaker remote (pcmk-remote) service extends high availability
    #: functionality outside of the Linux cluster into remote nodes.
    pcmk_remote: 'AppType' = 3121, 'pcmk-remote', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3121: 'AppType' = 3121, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] MTI VTR Emulator port
    #: - [UDP] MTI VTR Emulator port
    vtr_emulator: 'AppType' = 3122, 'vtr-emulator', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDI Translation Protocol
    #: - [UDP] EDI Translation Protocol
    edix: 'AppType' = 3123, 'edix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Beacon Port
    #: - [UDP] Beacon Port
    beacon_port: 'AppType' = 3124, 'beacon-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A13-AN Interface
    #: - [UDP] A13-AN Interface
    a13_an: 'AppType' = 3125, 'a13-an', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_3126: 'AppType' = 3126, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] CTX Bridge Port
    #: - [UDP] CTX Bridge Port
    ctx_bridge: 'AppType' = 3127, 'ctx-bridge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active API Server Port
    #: - [UDP] Active API Server Port
    ndl_aas: 'AppType' = 3128, 'ndl-aas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetPort Discovery Port
    #: - [UDP] NetPort Discovery Port
    netport_id: 'AppType' = 3129, 'netport-id', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICPv2
    #: - [UDP] ICPv2
    icpv2: 'AppType' = 3130, 'icpv2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Book Mark
    #: - [UDP] Net Book Mark
    netbookmark: 'AppType' = 3131, 'netbookmark', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Business Rule Engine Update Service
    #: - [UDP] Microsoft Business Rule Engine Update Service
    ms_rule_engine: 'AppType' = 3132, 'ms-rule-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prism Deploy User Port
    #: - [UDP] Prism Deploy User Port
    prism_deploy: 'AppType' = 3133, 'prism-deploy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Extensible Code Protocol
    #: - [UDP] Extensible Code Protocol
    ecp: 'AppType' = 3134, 'ecp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PeerBook Port
    #: - [UDP] PeerBook Port
    peerbook_port: 'AppType' = 3135, 'peerbook-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Grub Server Port
    #: - [UDP] Grub Server Port
    grubd: 'AppType' = 3136, 'grubd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rtnt-1 data packets
    #: - [UDP] rtnt-1 data packets
    rtnt_1: 'AppType' = 3137, 'rtnt-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rtnt-2 data packets
    #: - [UDP] rtnt-2 data packets
    rtnt_2: 'AppType' = 3138, 'rtnt-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Incognito Rendez-Vous
    #: - [UDP] Incognito Rendez-Vous
    incognitorv: 'AppType' = 3139, 'incognitorv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Arilia Multiplexor
    #: - [UDP] Arilia Multiplexor
    ariliamulti: 'AppType' = 3140, 'ariliamulti', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VMODEM
    #: - [UDP] VMODEM
    vmodem: 'AppType' = 3141, 'vmodem', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RDC WH EOS
    #: - [UDP] RDC WH EOS
    rdc_wh_eos: 'AppType' = 3142, 'rdc-wh-eos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sea View
    #: - [UDP] Sea View
    seaview: 'AppType' = 3143, 'seaview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tarantella
    #: - [UDP] Tarantella
    tarantella: 'AppType' = 3144, 'tarantella', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSI-LFAP
    #: - [UDP] CSI-LFAP
    csi_lfap: 'AppType' = 3145, 'csi-lfap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bears-02
    #: - [UDP] bears-02
    bears_02: 'AppType' = 3146, 'bears-02', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RFIO
    #: - [UDP] RFIO
    rfio: 'AppType' = 3147, 'rfio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetMike Game Administrator
    #: - [UDP] NetMike Game Administrator
    nm_game_admin: 'AppType' = 3148, 'nm-game-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetMike Game Server
    #: - [UDP] NetMike Game Server
    nm_game_server: 'AppType' = 3149, 'nm-game-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetMike Assessor Administrator
    #: - [UDP] NetMike Assessor Administrator
    nm_asses_admin: 'AppType' = 3150, 'nm-asses-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetMike Assessor
    #: - [UDP] NetMike Assessor
    nm_assessor: 'AppType' = 3151, 'nm-assessor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FeiTian Port
    #: - [UDP] FeiTian Port
    feitianrockey: 'AppType' = 3152, 'feitianrockey', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] S8Cargo Client Port
    #: - [UDP] S8Cargo Client Port
    s8_client_port: 'AppType' = 3153, 's8-client-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ON RMI Registry
    #: - [UDP] ON RMI Registry
    ccmrmi: 'AppType' = 3154, 'ccmrmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JpegMpeg Port
    #: - [UDP] JpegMpeg Port
    jpegmpeg: 'AppType' = 3155, 'jpegmpeg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Indura Collector
    #: - [UDP] Indura Collector
    indura: 'AppType' = 3156, 'indura', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LSA Communicator
    #: - [UDP] LSA Communicator
    lsa_comm: 'AppType' = 3157, 'lsa-comm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SmashTV Protocol
    #: - [UDP] SmashTV Protocol
    stvp: 'AppType' = 3158, 'stvp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NavegaWeb Tarification
    #: - [UDP] NavegaWeb Tarification
    navegaweb_port: 'AppType' = 3159, 'navegaweb-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TIP Application Server
    #: - [UDP] TIP Application Server
    tip_app_server: 'AppType' = 3160, 'tip-app-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DOC1 License Manager
    #: - [UDP] DOC1 License Manager
    doc1lm: 'AppType' = 3161, 'doc1lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SFLM
    #: - [UDP] SFLM
    sflm: 'AppType' = 3162, 'sflm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RES-SAP
    #: - [UDP] RES-SAP
    res_sap: 'AppType' = 3163, 'res-sap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IMPRS
    #: - [UDP] IMPRS
    imprs: 'AppType' = 3164, 'imprs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Newgenpay Engine Service
    #: - [UDP] Newgenpay Engine Service
    newgenpay: 'AppType' = 3165, 'newgenpay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quest Spotlight Out-Of-Process Collector
    #: - [UDP] Quest Spotlight Out-Of-Process Collector
    sossecollector: 'AppType' = 3166, 'sossecollector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Now Contact Public Server
    #: - [UDP] Now Contact Public Server
    nowcontact: 'AppType' = 3167, 'nowcontact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Now Up-to-Date Public Server
    #: - [UDP] Now Up-to-Date Public Server
    poweronnud: 'AppType' = 3168, 'poweronnud', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SERVERVIEW-AS
    #: - [UDP] SERVERVIEW-AS
    serverview_as: 'AppType' = 3169, 'serverview-as', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SERVERVIEW-ASN
    #: - [UDP] SERVERVIEW-ASN
    serverview_asn: 'AppType' = 3170, 'serverview-asn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SERVERVIEW-GF
    #: - [UDP] SERVERVIEW-GF
    serverview_gf: 'AppType' = 3171, 'serverview-gf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SERVERVIEW-RM
    #: - [UDP] SERVERVIEW-RM
    serverview_rm: 'AppType' = 3172, 'serverview-rm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SERVERVIEW-ICC
    #: - [UDP] SERVERVIEW-ICC
    serverview_icc: 'AppType' = 3173, 'serverview-icc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARMI Server
    #: - [UDP] ARMI Server
    armi_server: 'AppType' = 3174, 'armi-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] T1_E1_Over_IP
    #: - [UDP] T1_E1_Over_IP
    t1_e1_over_ip: 'AppType' = 3175, 't1-e1-over-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARS Master
    #: - [UDP] ARS Master
    ars_master: 'AppType' = 3176, 'ars-master', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Phonex Protocol
    #: - [UDP] Phonex Protocol
    phonex_port: 'AppType' = 3177, 'phonex-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Radiance UltraEdge Port
    #: - [UDP] Radiance UltraEdge Port
    radclientport: 'AppType' = 3178, 'radclientport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] H2GF W.2m Handover prot.
    #: - [UDP] H2GF W.2m Handover prot.
    h2gf_w_2m: 'AppType' = 3179, 'h2gf-w-2m', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Millicent Broker Server
    #: - [UDP] Millicent Broker Server
    mc_brk_srv: 'AppType' = 3180, 'mc-brk-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC Patrol Agent
    #: - [UDP] BMC Patrol Agent
    bmcpatrolagent: 'AppType' = 3181, 'bmcpatrolagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC Patrol Rendezvous
    #: - [UDP] BMC Patrol Rendezvous
    bmcpatrolrnvu: 'AppType' = 3182, 'bmcpatrolrnvu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] COPS/TLS
    #: - [UDP] COPS/TLS
    cops_tls: 'AppType' = 3183, 'cops-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ApogeeX Port
    #: - [UDP] ApogeeX Port
    apogeex_port: 'AppType' = 3184, 'apogeex-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SuSE Meta PPPD
    #: - [UDP] SuSE Meta PPPD
    smpppd: 'AppType' = 3185, 'smpppd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IIW Monitor User Port
    #: - [UDP] IIW Monitor User Port
    iiw_port: 'AppType' = 3186, 'iiw-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Open Design Listen Port
    #: - [UDP] Open Design Listen Port
    odi_port: 'AppType' = 3187, 'odi-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Broadcom Port
    #: - [UDP] Broadcom Port
    brcm_comm_port: 'AppType' = 3188, 'brcm-comm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pinnacle Sys InfEx Port
    #: - [UDP] Pinnacle Sys InfEx Port
    pcle_infex: 'AppType' = 3189, 'pcle-infex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConServR Proxy
    #: - [UDP] ConServR Proxy
    csvr_proxy: 'AppType' = 3190, 'csvr-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConServR SSL Proxy
    #: - [UDP] ConServR SSL Proxy
    csvr_sslproxy: 'AppType' = 3191, 'csvr-sslproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FireMon Revision Control
    #: - [UDP] FireMon Revision Control
    firemonrcc: 'AppType' = 3192, 'firemonrcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SpanDataPort
    #: - [UDP] SpanDataPort
    spandataport: 'AppType' = 3193, 'spandataport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rockstorm MAG protocol
    #: - [UDP] Rockstorm MAG protocol
    magbind: 'AppType' = 3194, 'magbind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Control Unit
    #: - [UDP] Network Control Unit
    ncu_1: 'AppType' = 3195, 'ncu-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Control Unit
    #: - [UDP] Network Control Unit
    ncu_2: 'AppType' = 3196, 'ncu-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Embrace Device Protocol Server
    #: - [UDP] Embrace Device Protocol Server
    embrace_dp_s: 'AppType' = 3197, 'embrace-dp-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Embrace Device Protocol Client
    #: - [UDP] Embrace Device Protocol Client
    embrace_dp_c: 'AppType' = 3198, 'embrace-dp-c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DMOD WorkSpace
    #: - [UDP] DMOD WorkSpace
    dmod_workspace: 'AppType' = 3199, 'dmod-workspace', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Press-sense Tick Port
    #: - [UDP] Press-sense Tick Port
    tick_port: 'AppType' = 3200, 'tick-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPQ-TaskSmart
    #: - [UDP] CPQ-TaskSmart
    cpq_tasksmart: 'AppType' = 3201, 'cpq-tasksmart', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IntraIntra
    #: - [UDP] IntraIntra
    intraintra: 'AppType' = 3202, 'intraintra', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Watcher Monitor
    #: - [UDP] Network Watcher Monitor
    netwatcher_mon: 'AppType' = 3203, 'netwatcher-mon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Watcher DB Access
    #: - [UDP] Network Watcher DB Access
    netwatcher_db: 'AppType' = 3204, 'netwatcher-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iSNS Server Port [:rfc:`4171`]
    #: - [UDP] iSNS Server Port [:rfc:`4171`]
    isns: 'AppType' = 3205, 'isns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IronMail POP Proxy
    #: - [UDP] IronMail POP Proxy
    ironmail: 'AppType' = 3206, 'ironmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Veritas Authentication Port
    #: - [UDP] Veritas Authentication Port
    vx_auth_port: 'AppType' = 3207, 'vx-auth-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PFU PR Callback
    #: - [UDP] PFU PR Callback
    pfu_prcallback: 'AppType' = 3208, 'pfu-prcallback', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP OpenView Network Path Engine Server
    #: - [UDP] HP OpenView Network Path Engine Server
    netwkpathengine: 'AppType' = 3209, 'netwkpathengine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Flamenco Networks Proxy
    #: - [UDP] Flamenco Networks Proxy
    flamenco_proxy: 'AppType' = 3210, 'flamenco-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avocent Secure Management
    #: - [UDP] Avocent Secure Management
    avsecuremgmt: 'AppType' = 3211, 'avsecuremgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Survey Instrument
    #: - [UDP] Survey Instrument
    surveyinst: 'AppType' = 3212, 'surveyinst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NEON 24X7 Mission Control
    #: - [UDP] NEON 24X7 Mission Control
    neon24x7: 'AppType' = 3213, 'neon24x7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JMQ Daemon Port 1
    #: - [UDP] JMQ Daemon Port 1
    jmq_daemon_1: 'AppType' = 3214, 'jmq-daemon-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JMQ Daemon Port 2
    #: - [UDP] JMQ Daemon Port 2
    jmq_daemon_2: 'AppType' = 3215, 'jmq-daemon-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ferrari electronic FOAM
    #: - [UDP] Ferrari electronic FOAM
    ferrari_foam: 'AppType' = 3216, 'ferrari-foam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unified IP & Telecom Environment
    #: - [UDP] Unified IP & Telecom Environment
    unite: 'AppType' = 3217, 'unite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMC SmartPackets
    #: - [UDP] EMC SmartPackets
    smartpackets: 'AppType' = 3218, 'smartpackets', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WMS Messenger
    #: - [UDP] WMS Messenger
    wms_messenger: 'AppType' = 3219, 'wms-messenger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XML NM over SSL
    #: - [UDP] XML NM over SSL
    xnm_ssl: 'AppType' = 3220, 'xnm-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XML NM over TCP
    #: - [UDP] XML NM over TCP
    xnm_clear_text: 'AppType' = 3221, 'xnm-clear-text', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Gateway Load Balancing Pr
    #: - [UDP] Gateway Load Balancing Pr
    glbp: 'AppType' = 3222, 'glbp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIGIVOTE (R) Vote-Server
    #: - [UDP] DIGIVOTE (R) Vote-Server
    digivote: 'AppType' = 3223, 'digivote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AES Discovery Port
    #: - [UDP] AES Discovery Port
    aes_discovery: 'AppType' = 3224, 'aes-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FCIP [:rfc:`3821`]
    #: - [UDP] FCIP [:rfc:`3821`]
    fcip_port: 'AppType' = 3225, 'fcip-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISI Industry Software IRP
    #: - [UDP] ISI Industry Software IRP
    isi_irp: 'AppType' = 3226, 'isi-irp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DiamondWave NMS Server
    #: - [UDP] DiamondWave NMS Server
    dwnmshttp: 'AppType' = 3227, 'dwnmshttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DiamondWave MSG Server
    #: - [UDP] DiamondWave MSG Server
    dwmsgserver: 'AppType' = 3228, 'dwmsgserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Global CD Port
    #: - [UDP] Global CD Port
    global_cd_port: 'AppType' = 3229, 'global-cd-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Software Distributor Port
    #: - [UDP] Software Distributor Port
    sftdst_port: 'AppType' = 3230, 'sftdst-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VidiGo communication (previous was: Delta Solutions Direct)
    #: - [UDP] VidiGo communication (previous was: Delta Solutions Direct)
    vidigo: 'AppType' = 3231, 'vidigo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MDT port [:rfc:`6513`]
    #: - [UDP] MDT port [:rfc:`6513`]
    mdtp: 'AppType' = 3232, 'mdtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WhiskerControl main port
    #: - [UDP] WhiskerControl main port
    whisker: 'AppType' = 3233, 'whisker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alchemy Server
    #: - [UDP] Alchemy Server
    alchemy: 'AppType' = 3234, 'alchemy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MDAP port
    #: - [UDP] MDAP Port
    mdap_port: 'AppType' = 3235, 'mdap-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appareNet Test Server
    #: - [UDP] appareNet Test Server
    apparenet_ts: 'AppType' = 3236, 'apparenet-ts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appareNet Test Packet Sequencer
    #: - [UDP] appareNet Test Packet Sequencer
    apparenet_tps: 'AppType' = 3237, 'apparenet-tps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appareNet Analysis Server
    #: - [UDP] appareNet Analysis Server
    apparenet_as: 'AppType' = 3238, 'apparenet-as', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appareNet User Interface
    #: - [UDP] appareNet User Interface
    apparenet_ui: 'AppType' = 3239, 'apparenet-ui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trio Motion Control Port
    #: - [UDP] Trio Motion Control Port
    triomotion: 'AppType' = 3240, 'triomotion', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SysOrb Monitoring Server
    #: - [UDP] SysOrb Monitoring Server
    sysorb: 'AppType' = 3241, 'sysorb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Session Description ID
    #: - [UDP] Session Description ID
    sdp_id_port: 'AppType' = 3242, 'sdp-id-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Timelot Port
    #: - [UDP] Timelot Port
    timelot: 'AppType' = 3243, 'timelot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OneSAF
    #: - [UDP] OneSAF
    onesaf: 'AppType' = 3244, 'onesaf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VIEO Fabric Executive
    #: - [UDP] VIEO Fabric Executive
    vieo_fe: 'AppType' = 3245, 'vieo-fe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DVT SYSTEM PORT
    #: - [UDP] DVT SYSTEM PORT
    dvt_system: 'AppType' = 3246, 'dvt-system', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DVT DATA LINK
    #: - [UDP] DVT DATA LINK
    dvt_data: 'AppType' = 3247, 'dvt-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PROCOS LM
    #: - [UDP] PROCOS LM
    procos_lm: 'AppType' = 3248, 'procos-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] State Sync Protocol
    #: - [UDP] State Sync Protocol
    ssp: 'AppType' = 3249, 'ssp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HMS hicp port
    #: - [UDP] HMS hicp port
    hicp: 'AppType' = 3250, 'hicp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sys Scanner
    #: - [UDP] Sys Scanner
    sysscanner: 'AppType' = 3251, 'sysscanner', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DHE port
    #: - [UDP] DHE port
    dhe: 'AppType' = 3252, 'dhe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDA Data
    #: - [UDP] PDA Data
    pda_data: 'AppType' = 3253, 'pda-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDA System
    #: - [UDP] PDA System
    pda_sys: 'AppType' = 3254, 'pda-sys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Semaphore Connection Port
    #: - [UDP] Semaphore Connection Port
    semaphore: 'AppType' = 3255, 'semaphore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq RPM Agent Port
    #: - [UDP] Compaq RPM Agent Port
    cpqrpm_agent: 'AppType' = 3256, 'cpqrpm-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compaq RPM Server Port
    #: - [UDP] Compaq RPM Server Port
    cpqrpm_server: 'AppType' = 3257, 'cpqrpm-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ivecon Server Port
    #: - [UDP] Ivecon Server Port
    ivecon_port: 'AppType' = 3258, 'ivecon-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Epson Network Common Devi
    #: - [UDP] Epson Network Common Devi
    epncdp2: 'AppType' = 3259, 'epncdp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iSCSI port [:rfc:`7143`]
    #: - [UDP] iSCSI port [:rfc:`7143`]
    iscsi_target: 'AppType' = 3260, 'iscsi-target', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] winShadow
    #: - [UDP] winShadow
    winshadow: 'AppType' = 3261, 'winshadow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NECP
    #: - [UDP] NECP
    necp: 'AppType' = 3262, 'necp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] E-Color Enterprise Imager
    #: - [UDP] E-Color Enterprise Imager
    ecolor_imager: 'AppType' = 3263, 'ecolor-imager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cc:mail/lotus
    #: - [UDP] cc:mail/lotus
    ccmail: 'AppType' = 3264, 'ccmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Altav Tunnel
    #: - [UDP] Altav Tunnel
    altav_tunnel: 'AppType' = 3265, 'altav-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NS CFG Server
    #: - [UDP] NS CFG Server
    ns_cfg_server: 'AppType' = 3266, 'ns-cfg-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Dial Out
    #: - [UDP] IBM Dial Out
    ibm_dial_out: 'AppType' = 3267, 'ibm-dial-out', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Global Catalog
    #: - [UDP] Microsoft Global Catalog
    msft_gc: 'AppType' = 3268, 'msft-gc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Global Catalog with LDAP/SSL
    #: - [UDP] Microsoft Global Catalog with LDAP/SSL
    msft_gc_ssl: 'AppType' = 3269, 'msft-gc-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Verismart
    #: - [UDP] Verismart
    verismart: 'AppType' = 3270, 'verismart', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSoft Prev Port
    #: - [UDP] CSoft Prev Port
    csoft_prev: 'AppType' = 3271, 'csoft-prev', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu User Manager
    #: - [UDP] Fujitsu User Manager
    user_manager: 'AppType' = 3272, 'user-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Extensible Multiplexed Protocol
    #: - [UDP] Simple Extensible Multiplexed Protocol
    sxmp: 'AppType' = 3273, 'sxmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ordinox Server
    #: - [UDP] Ordinox Server
    ordinox_server: 'AppType' = 3274, 'ordinox-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAMD
    #: - [UDP] SAMD
    samd: 'AppType' = 3275, 'samd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Maxim ASICs
    #: - [UDP] Maxim ASICs
    maxim_asics: 'AppType' = 3276, 'maxim-asics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AWG Proxy
    #: - [UDP] AWG Proxy
    awg_proxy: 'AppType' = 3277, 'awg-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LKCM Server
    #: - [UDP] LKCM Server
    lkcmserver: 'AppType' = 3278, 'lkcmserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VS Server
    #: - [UDP] VS Server
    vs_server: 'AppType' = 3280, 'vs-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SYSOPT
    #: - [UDP] SYSOPT
    sysopt: 'AppType' = 3281, 'sysopt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Datusorb
    #: - [UDP] Datusorb
    datusorb: 'AppType' = 3282, 'datusorb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Assistant
    #: - [UDP] Net Assistant
    apple_remote_desktop: 'AppType' = 3283, 'apple remote desktop (net assistant)', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 4Talk
    #: - [UDP] 4Talk
    AppType_4talk: 'AppType' = 3284, '4talk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Plato
    #: - [UDP] Plato
    plato: 'AppType' = 3285, 'plato', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] E-Net
    #: - [UDP] E-Net
    e_net: 'AppType' = 3286, 'e-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIRECTVDATA
    #: - [UDP] DIRECTVDATA
    directvdata: 'AppType' = 3287, 'directvdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] COPS
    #: - [UDP] COPS
    cops: 'AppType' = 3288, 'cops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ENPC
    #: - [UDP] ENPC
    enpc: 'AppType' = 3289, 'enpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAPS LOGISTICS TOOLKIT - LM
    #: - [UDP] CAPS LOGISTICS TOOLKIT - LM
    caps_lm: 'AppType' = 3290, 'caps-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] S A Holditch & Associates - LM
    #: - [UDP] S A Holditch & Associates - LM
    sah_lm: 'AppType' = 3291, 'sah-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cart O Rama
    #: - [UDP] Cart O Rama
    cart_o_rama: 'AppType' = 3292, 'cart-o-rama', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fg-fps
    #: - [UDP] fg-fps
    fg_fps: 'AppType' = 3293, 'fg-fps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fg-gip
    #: - [UDP] fg-gip
    fg_gip: 'AppType' = 3294, 'fg-gip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic IP Lookup
    #: - [UDP] Dynamic IP Lookup
    dyniplookup: 'AppType' = 3295, 'dyniplookup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rib License Manager
    #: - [UDP] Rib License Manager
    rib_slm: 'AppType' = 3296, 'rib-slm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cytel License Manager
    #: - [UDP] Cytel License Manager
    cytel_lm: 'AppType' = 3297, 'cytel-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DeskView
    #: - [UDP] DeskView
    deskview: 'AppType' = 3298, 'deskview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pdrncs
    #: - [UDP] pdrncs
    pdrncs: 'AppType' = 3299, 'pdrncs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Ceph monitor
    ceph: 'AppType' = 3300, 'ceph', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3300: 'AppType' = 3300, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Tarantool in-memory computing platform
    #: - [UDP] Tarantool in-memory computing platform
    tarantool: 'AppType' = 3301, 'tarantool', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCS Fastmail
    #: - [UDP] MCS Fastmail
    mcs_fastmail: 'AppType' = 3302, 'mcs-fastmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OP Session Client
    #: - [UDP] OP Session Client
    opsession_clnt: 'AppType' = 3303, 'opsession-clnt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OP Session Server
    #: - [UDP] OP Session Server
    opsession_srvr: 'AppType' = 3304, 'opsession-srvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ODETTE-FTP [:rfc:`5024`]
    #: - [UDP] ODETTE-FTP [:rfc:`5024`]
    odette_ftp: 'AppType' = 3305, 'odette-ftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MySQL
    #: - [UDP] MySQL
    mysql: 'AppType' = 3306, 'mysql', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OP Session Proxy
    #: - [UDP] OP Session Proxy
    opsession_prxy: 'AppType' = 3307, 'opsession-prxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNS Server
    #: - [UDP] TNS Server
    tns_server: 'AppType' = 3308, 'tns-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNS ADV
    #: - [UDP] TNS ADV
    tns_adv: 'AppType' = 3309, 'tns-adv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dyna Access
    #: - [UDP] Dyna Access
    dyna_access: 'AppType' = 3310, 'dyna-access', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCNS Tel Ret
    #: - [UDP] MCNS Tel Ret
    mcns_tel_ret: 'AppType' = 3311, 'mcns-tel-ret', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Application Management Server
    #: - [UDP] Application Management Server
    appman_server: 'AppType' = 3312, 'appman-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify Object Broker
    #: - [UDP] Unify Object Broker
    uorb: 'AppType' = 3313, 'uorb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify Object Host
    #: - [UDP] Unify Object Host
    uohost: 'AppType' = 3314, 'uohost', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CDID
    #: - [UDP] CDID
    cdid: 'AppType' = 3315, 'cdid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AICC/CMI
    #: - [UDP] AICC/CMI
    aicc_cmi: 'AppType' = 3316, 'aicc-cmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSAI PORT
    #: - [UDP] VSAI PORT
    vsaiport: 'AppType' = 3317, 'vsaiport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Swith to Swith Routing Information Protocol
    #: - [UDP] Swith to Swith Routing Information Protocol
    ssrip: 'AppType' = 3318, 'ssrip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDT License Manager
    #: - [UDP] SDT License Manager
    sdt_lmd: 'AppType' = 3319, 'sdt-lmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Office Link 2000
    #: - [UDP] Office Link 2000
    officelink2000: 'AppType' = 3320, 'officelink2000', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VNSSTR
    #: - [UDP] VNSSTR
    vnsstr: 'AppType' = 3321, 'vnsstr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SFTU
    #: - [UDP] SFTU
    sftu: 'AppType' = 3326, 'sftu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BBARS
    #: - [UDP] BBARS
    bbars: 'AppType' = 3327, 'bbars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eaglepoint License Manager
    #: - [UDP] Eaglepoint License Manager
    egptlm: 'AppType' = 3328, 'egptlm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Device Disc
    #: - [UDP] HP Device Disc
    hp_device_disc: 'AppType' = 3329, 'hp-device-disc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCS Calypso ICF
    #: - [UDP] MCS Calypso ICF
    mcs_calypsoicf: 'AppType' = 3330, 'mcs-calypsoicf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCS Messaging
    #: - [UDP] MCS Messaging
    mcs_messaging: 'AppType' = 3331, 'mcs-messaging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCS Mail Server
    #: - [UDP] MCS Mail Server
    mcs_mailsvr: 'AppType' = 3332, 'mcs-mailsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DEC Notes
    #: - [UDP] DEC Notes
    dec_notes: 'AppType' = 3333, 'dec-notes', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct TV Webcasting
    #: - [UDP] Direct TV Webcasting
    directv_web: 'AppType' = 3334, 'directv-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct TV Software Updates
    #: - [UDP] Direct TV Software Updates
    directv_soft: 'AppType' = 3335, 'directv-soft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct TV Tickers
    #: - [UDP] Direct TV Tickers
    directv_tick: 'AppType' = 3336, 'directv-tick', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct TV Data Catalog
    #: - [UDP] Direct TV Data Catalog
    directv_catlg: 'AppType' = 3337, 'directv-catlg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMF data b
    #: - [UDP] OMF data b
    anet_b: 'AppType' = 3338, 'anet-b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMF data l
    #: - [UDP] OMF data l
    anet_l: 'AppType' = 3339, 'anet-l', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMF data m
    #: - [UDP] OMF data m
    anet_m: 'AppType' = 3340, 'anet-m', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMF data h
    #: - [UDP] OMF data h
    anet_h: 'AppType' = 3341, 'anet-h', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WebTIE
    #: - [UDP] WebTIE
    webtie: 'AppType' = 3342, 'webtie', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Cluster Net
    #: - [UDP] MS Cluster Net
    ms_cluster_net: 'AppType' = 3343, 'ms-cluster-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BNT Manager
    #: - [UDP] BNT Manager
    bnt_manager: 'AppType' = 3344, 'bnt-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Influence
    #: - [UDP] Influence
    influence: 'AppType' = 3345, 'influence', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trnsprnt Proxy
    #: - [UDP] Trnsprnt Proxy
    trnsprntproxy: 'AppType' = 3346, 'trnsprntproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Phoenix RPC
    #: - [UDP] Phoenix RPC
    phoenix_rpc: 'AppType' = 3347, 'phoenix-rpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pangolin Laser
    #: - [UDP] Pangolin Laser
    pangolin_laser: 'AppType' = 3348, 'pangolin-laser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Chevin Services
    #: - [UDP] Chevin Services
    chevinservices: 'AppType' = 3349, 'chevinservices', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FINDVIATV
    #: - [UDP] FINDVIATV
    findviatv: 'AppType' = 3350, 'findviatv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Btrieve port
    #: - [UDP] Btrieve port
    btrieve: 'AppType' = 3351, 'btrieve', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scalable SQL
    #: - [UDP] Scalable SQL
    ssql: 'AppType' = 3352, 'ssql', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FATPIPE
    #: - [UDP] FATPIPE
    fatpipe: 'AppType' = 3353, 'fatpipe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SUITJD
    #: - [UDP] SUITJD
    suitjd: 'AppType' = 3354, 'suitjd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ordinox Dbase
    #: - [UDP] Ordinox Dbase
    ordinox_dbase: 'AppType' = 3355, 'ordinox-dbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UPNOTIFYPS
    #: - [UDP] UPNOTIFYPS
    upnotifyps: 'AppType' = 3356, 'upnotifyps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adtech Test IP
    #: - [UDP] Adtech Test IP
    adtech_test: 'AppType' = 3357, 'adtech-test', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mp Sys Rmsvr
    #: - [UDP] Mp Sys Rmsvr
    mpsysrmsvr: 'AppType' = 3358, 'mpsysrmsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WG NetForce
    #: - [UDP] WG NetForce
    wg_netforce: 'AppType' = 3359, 'wg-netforce', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KV Server
    #: - [UDP] KV Server
    kv_server: 'AppType' = 3360, 'kv-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KV Agent
    #: - [UDP] KV Agent
    kv_agent: 'AppType' = 3361, 'kv-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DJ ILM
    #: - [UDP] DJ ILM
    dj_ilm: 'AppType' = 3362, 'dj-ilm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NATI Vi Server
    #: - [UDP] NATI Vi Server
    nati_vi_server: 'AppType' = 3363, 'nati-vi-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CreativeServer
    #: - [UDP] CreativeServer
    creativeserver_453: 'AppType' = 453, 'creativeserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Creative Server
    #: - [UDP] Creative Server
    creativeserver_3364: 'AppType' = 3364, 'creativeserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ContentServer
    #: - [UDP] ContentServer
    contentserver_454: 'AppType' = 454, 'contentserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Content Server
    #: - [UDP] Content Server
    contentserver_3365: 'AppType' = 3365, 'contentserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CreativePartnr
    #: - [UDP] CreativePartnr
    creativepartnr_455: 'AppType' = 455, 'creativepartnr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Creative Partner
    #: - [UDP] Creative Partner
    creativepartnr_3366: 'AppType' = 3366, 'creativepartnr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TIP 2
    #: - [UDP] TIP 2
    tip2: 'AppType' = 3372, 'tip2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lavenir License Manager
    #: - [UDP] Lavenir License Manager
    lavenir_lm: 'AppType' = 3373, 'lavenir-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cluster Disc
    #: - [UDP] Cluster Disc
    cluster_disc: 'AppType' = 3374, 'cluster-disc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSNM Agent
    #: - [UDP] VSNM Agent
    vsnm_agent: 'AppType' = 3375, 'vsnm-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CD Broker
    #: - [UDP] CD Broker
    cdbroker: 'AppType' = 3376, 'cdbroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cogsys Network License Manager
    #: - [UDP] Cogsys Network License Manager
    cogsys_lm: 'AppType' = 3377, 'cogsys-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WSICOPY
    #: - [UDP] WSICOPY
    wsicopy: 'AppType' = 3378, 'wsicopy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SOCORFS
    #: - [UDP] SOCORFS
    socorfs: 'AppType' = 3379, 'socorfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Channels
    #: - [UDP] SNS Channels
    sns_channels: 'AppType' = 3380, 'sns-channels', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Geneous
    #: - [UDP] Geneous
    geneous: 'AppType' = 3381, 'geneous', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Network Enhanced Antitheft function
    #: - [UDP] Fujitsu Network Enhanced Antitheft function
    fujitsu_neat: 'AppType' = 3382, 'fujitsu-neat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Software Products License Manager
    #: - [UDP] Enterprise Software Products License Manager
    esp_lm: 'AppType' = 3383, 'esp-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cluster Management Services
    #: - [UDP] Hardware Management
    hp_clic: 'AppType' = 3384, 'hp-clic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qnxnetman
    #: - [UDP] qnxnetman
    qnxnetman: 'AppType' = 3385, 'qnxnetman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] GPRS Data
    gprs_data: 'AppType' = 3386, 'gprs-data', TransportProtocol.get('tcp')

    #: [UDP] GPRS SIG
    gprs_sig: 'AppType' = 3386, 'gprs-sig', TransportProtocol.get('udp')

    #: - [TCP] Back Room Net
    #: - [UDP] Back Room Net
    backroomnet: 'AppType' = 3387, 'backroomnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CB Server
    #: - [UDP] CB Server
    cbserver: 'AppType' = 3388, 'cbserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS WBT Server
    #: - [UDP] MS WBT Server
    ms_wbt_server: 'AppType' = 3389, 'ms-wbt-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Distributed Service Coordinator
    #: - [UDP] Distributed Service Coordinator
    dsc: 'AppType' = 3390, 'dsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAVANT
    #: - [UDP] SAVANT
    savant: 'AppType' = 3391, 'savant', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EFI License Management
    #: - [UDP] EFI License Management
    efi_lm: 'AppType' = 3392, 'efi-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2K Tapestry Client to Server
    #: - [UDP] D2K Tapestry Client to Server
    d2k_tapestry1: 'AppType' = 3393, 'd2k-tapestry1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D2K Tapestry Server to Server
    #: - [UDP] D2K Tapestry Server to Server
    d2k_tapestry2: 'AppType' = 3394, 'd2k-tapestry2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dyna License Manager (Elam)
    #: - [UDP] Dyna License Manager (Elam)
    dyna_lm: 'AppType' = 3395, 'dyna-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Printer Agent IANA assigned this well-formed service name as a
    #:   replacement for "printer_agent".
    #: - [TCP] Printer Agent
    #: - [UDP] Printer Agent IANA assigned this well-formed service name as a
    #:   replacement for "printer_agent".
    #: - [UDP] Printer Agent
    printer_agent: 'AppType' = 3396, 'printer-agent', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Cloanto License Manager
    #: - [UDP] Cloanto License Manager
    cloanto_lm: 'AppType' = 3397, 'cloanto-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mercantile
    #: - [UDP] Mercantile
    mercantile: 'AppType' = 3398, 'mercantile', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSMS
    #: - [UDP] CSMS
    csms: 'AppType' = 3399, 'csms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSMS2
    #: - [UDP] CSMS2
    csms2: 'AppType' = 3400, 'csms2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] filecast
    #: - [UDP] filecast
    filecast: 'AppType' = 3401, 'filecast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FXa Engine Network Port
    #: - [UDP] FXa Engine Network Port
    fxaengine_net: 'AppType' = 3402, 'fxaengine-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Removed
    removed: 'AppType' = 3404, 'removed', TransportProtocol.get('undefined')

    #: - [TCP] Nokia Announcement ch 1
    #: - [UDP] Nokia Announcement ch 1
    nokia_ann_ch1: 'AppType' = 3405, 'nokia-ann-ch1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nokia Announcement ch 2
    #: - [UDP] Nokia Announcement ch 2
    nokia_ann_ch2: 'AppType' = 3406, 'nokia-ann-ch2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LDAP admin server port
    #: - [UDP] LDAP admin server port
    ldap_admin: 'AppType' = 3407, 'ldap-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BES Api Port
    #: - [UDP] BES Api Port
    besapi: 'AppType' = 3408, 'besapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetworkLens Event Port
    #: - [UDP] NetworkLens Event Port
    networklens: 'AppType' = 3409, 'networklens', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetworkLens SSL Event
    #: - [UDP] NetworkLens SSL Event
    networklenss: 'AppType' = 3410, 'networklenss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BioLink Authenteon server
    #: - [UDP] BioLink Authenteon server
    biolink_auth: 'AppType' = 3411, 'biolink-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xmlBlaster
    #: - [UDP] xmlBlaster
    xmlblaster: 'AppType' = 3412, 'xmlblaster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SpecView Networking
    #: - [UDP] SpecView Networking
    svnet: 'AppType' = 3413, 'svnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BroadCloud WIP Port
    #: - [UDP] BroadCloud WIP Port
    wip_port: 'AppType' = 3414, 'wip-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BCI Name Service
    #: - [UDP] BCI Name Service
    bcinameservice: 'AppType' = 3415, 'bcinameservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AirMobile IS Command Port
    #: - [UDP] AirMobile IS Command Port
    commandport: 'AppType' = 3416, 'commandport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConServR file translation
    #: - [UDP] ConServR file translation
    csvr: 'AppType' = 3417, 'csvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote nmap
    #: - [UDP] Remote nmap
    rnmap: 'AppType' = 3418, 'rnmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Isogon SoftAudit
    #: - [UDP] ISogon SoftAudit
    softaudit: 'AppType' = 3419, 'softaudit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iFCP User Port [:rfc:`4172`]
    #: - [UDP] iFCP User Port [:rfc:`4172`]
    ifcp_port: 'AppType' = 3420, 'ifcp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bull Apprise portmapper
    #: - [UDP] Bull Apprise portmapper
    bmap: 'AppType' = 3421, 'bmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote USB System Port
    #: - [UDP] Remote USB System Port
    rusb_sys_port: 'AppType' = 3422, 'rusb-sys-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xTrade Reliable Messaging
    #: - [UDP] xTrade Reliable Messaging
    xtrm: 'AppType' = 3423, 'xtrm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xTrade over TLS/SSL
    #: - [UDP] xTrade over TLS/SSL
    xtrms: 'AppType' = 3424, 'xtrms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AGPS Access Port
    #: - [UDP] AGPS Access Port
    agps_port: 'AppType' = 3425, 'agps-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Arkivio Storage Protocol
    #: - [UDP] Arkivio Storage Protocol
    arkivio: 'AppType' = 3426, 'arkivio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WebSphere SNMP
    #: - [UDP] WebSphere SNMP
    websphere_snmp: 'AppType' = 3427, 'websphere-snmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 2Wire CSS
    #: - [UDP] 2Wire CSS
    twcss: 'AppType' = 3428, 'twcss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GCSP user port
    #: - [UDP] GCSP user port
    gcsp: 'AppType' = 3429, 'gcsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scott Studios Dispatch
    #: - [UDP] Scott Studios Dispatch
    ssdispatch: 'AppType' = 3430, 'ssdispatch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Active License Server Port
    #: - [UDP] Active License Server Port
    ndl_als: 'AppType' = 3431, 'ndl-als', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Device Protocol
    #: - [UDP] Secure Device Protocol
    osdcp: 'AppType' = 3432, 'osdcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OPNET Service Management Platform
    #: - [UDP] OPNET Service Management Platform
    opnet_smp: 'AppType' = 3433, 'opnet-smp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenCM Server
    #: - [UDP] OpenCM Server
    opencm: 'AppType' = 3434, 'opencm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pacom Security User Port
    #: - [UDP] Pacom Security User Port
    pacom: 'AppType' = 3435, 'pacom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GuardControl Exchange Protocol
    #: - [UDP] GuardControl Exchange Protocol
    gc_config: 'AppType' = 3436, 'gc-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Autocue Directory Service
    #: - [UDP] Autocue Directory Service
    autocueds: 'AppType' = 3437, 'autocueds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spiralcraft Admin
    #: - [UDP] Spiralcraft Admin
    spiral_admin: 'AppType' = 3438, 'spiral-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HRI Interface Port
    #: - [UDP] HRI Interface Port
    hri_port: 'AppType' = 3439, 'hri-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Steward Mgmt Console
    #: - [UDP] Net Steward Mgmt Console
    ans_console: 'AppType' = 3440, 'ans-console', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OC Connect Client
    #: - [UDP] OC Connect Client
    connect_client: 'AppType' = 3441, 'connect-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OC Connect Server
    #: - [UDP] OC Connect Server
    connect_server: 'AppType' = 3442, 'connect-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView Network Node Manager WEB Server
    #: - [UDP] OpenView Network Node Manager WEB Server
    ov_nnm_websrv: 'AppType' = 3443, 'ov-nnm-websrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Denali Server
    #: - [UDP] Denali Server
    denali_server: 'AppType' = 3444, 'denali-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Media Object Network Protocol
    #: - [UDP] Media Object Network Protocol
    monp: 'AppType' = 3445, 'monp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com FAX RPC port
    #: - [UDP] 3Com FAX RPC port
    AppType_3comfaxrpc: 'AppType' = 3446, '3comfaxrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirectNet IM System
    #: - [UDP] DirectNet IM System
    directnet: 'AppType' = 3447, 'directnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Discovery and Net Config
    #: - [UDP] Discovery and Net Config
    dnc_port: 'AppType' = 3448, 'dnc-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HotU Chat
    #: - [UDP] HotU Chat
    hotu_chat: 'AppType' = 3449, 'hotu-chat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAStorProxy
    #: - [UDP] CAStorProxy
    castorproxy: 'AppType' = 3450, 'castorproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASAM Services
    #: - [UDP] ASAM Services
    asam: 'AppType' = 3451, 'asam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SABP-Signalling Protocol
    #: - [UDP] SABP-Signalling Protocol
    sabp_signal: 'AppType' = 3452, 'sabp-signal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PSC Update
    #: - [UDP] PSC Update
    pscupd: 'AppType' = 3453, 'pscupd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple Remote Access Protocol
    #: - [UDP] Apple Remote Access Protocol
    mira: 'AppType' = 3454, 'mira', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSVP Port
    #: - [UDP] RSVP Port
    prsvp: 'AppType' = 3455, 'prsvp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VAT default data
    #: - [UDP] VAT default data
    vat: 'AppType' = 3456, 'vat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VAT default control
    #: - [UDP] VAT default control
    vat_control: 'AppType' = 3457, 'vat-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D3WinOSFI
    #: - [UDP] D3WinOSFI
    d3winosfi: 'AppType' = 3458, 'd3winosfi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TIP Integral
    #: - [UDP] TIP Integral
    integral: 'AppType' = 3459, 'integral', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM Manger
    #: - [UDP] EDM Manger
    edm_manager: 'AppType' = 3460, 'edm-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM Stager
    #: - [UDP] EDM Stager
    edm_stager: 'AppType' = 3461, 'edm-stager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM STD Notify
    #: - [UDP] EDM STD Notify
    edm_std_notify: 'AppType' = 3462, 'edm-std-notify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM ADM Notify
    #: - [UDP] EDM ADM Notify
    edm_adm_notify: 'AppType' = 3463, 'edm-adm-notify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM MGR Sync
    #: - [UDP] EDM MGR Sync
    edm_mgr_sync: 'AppType' = 3464, 'edm-mgr-sync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EDM MGR Cntrl
    #: - [UDP] EDM MGR Cntrl
    edm_mgr_cntrl: 'AppType' = 3465, 'edm-mgr-cntrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WORKFLOW
    #: - [UDP] WORKFLOW
    workflow: 'AppType' = 3466, 'workflow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RCST
    #: - [UDP] RCST
    rcst: 'AppType' = 3467, 'rcst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTCM Remote Controll
    #: - [UDP] TTCM Remote Controll
    ttcmremotectrl: 'AppType' = 3468, 'ttcmremotectrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pluribus
    #: - [UDP] Pluribus
    pluribus: 'AppType' = 3469, 'pluribus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jt400
    #: - [UDP] jt400
    jt400: 'AppType' = 3470, 'jt400', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jt400-ssl
    #: - [UDP] jt400-ssl
    jt400_ssl: 'AppType' = 3471, 'jt400-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JAUGS N-G Remotec 1
    #: - [UDP] JAUGS N-G Remotec 1
    jaugsremotec_1: 'AppType' = 3472, 'jaugsremotec-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JAUGS N-G Remotec 2
    #: - [UDP] JAUGS N-G Remotec 2
    jaugsremotec_2: 'AppType' = 3473, 'jaugsremotec-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TSP Automation
    #: - [UDP] TSP Automation
    ttntspauto: 'AppType' = 3474, 'ttntspauto', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Genisar Comm Port
    #: - [UDP] Genisar Comm Port
    genisar_port: 'AppType' = 3475, 'genisar-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NVIDIA Mgmt Protocol
    #: - [UDP] NVIDIA Mgmt Protocol
    nppmp: 'AppType' = 3476, 'nppmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eComm link port
    #: - [UDP] eComm link port
    ecomm: 'AppType' = 3477, 'ecomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Session Traversal Utilities for NAT (STUN) port [:rfc:`8489`]
    #: - [UDP] Session Traversal Utilities for NAT (STUN) port [:rfc:`8489`]
    stun: 'AppType' = 3478, 'stun', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TURN over TCP [:rfc:`8656`]
    #: - [UDP] TURN over UDP [:rfc:`8656`]
    turn: 'AppType' = 3478, 'turn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STUN Behavior Discovery over TCP [:rfc:`5780`]
    #: - [UDP] STUN Behavior Discovery over UDP [:rfc:`5780`]
    stun_behavior: 'AppType' = 3478, 'stun-behavior', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 2Wire RPC
    #: - [UDP] 2Wire RPC
    twrpc: 'AppType' = 3479, 'twrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Virtual Workspace
    #: - [UDP] Secure Virtual Workspace
    plethora: 'AppType' = 3480, 'plethora', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CleanerLive remote ctrl
    #: - [UDP] CleanerLive remote ctrl
    cleanerliverc: 'AppType' = 3481, 'cleanerliverc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vulture Monitoring System
    #: - [UDP] Vulture Monitoring System
    vulture: 'AppType' = 3482, 'vulture', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Slim Devices Protocol
    #: - [UDP] Slim Devices Protocol
    slim_devices: 'AppType' = 3483, 'slim-devices', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GBS SnapTalk Protocol
    #: - [UDP] GBS SnapTalk Protocol
    gbs_stp: 'AppType' = 3484, 'gbs-stp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CelaTalk
    #: - [UDP] CelaTalk
    celatalk: 'AppType' = 3485, 'celatalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IFSF Heartbeat Port
    #: - [UDP] IFSF Heartbeat Port
    ifsf_hb_port: 'AppType' = 3486, 'ifsf-hb-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] LISA TCP Transfer Channel
    ltctcp: 'AppType' = 3487, 'ltctcp', TransportProtocol.get('tcp')

    #: [UDP] LISA UDP Transfer Channel
    ltcudp: 'AppType' = 3487, 'ltcudp', TransportProtocol.get('udp')

    #: - [TCP] FS Remote Host Server
    #: - [UDP] FS Remote Host Server
    fs_rh_srv: 'AppType' = 3488, 'fs-rh-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTP/DIA
    #: - [UDP] DTP/DIA
    dtp_dia: 'AppType' = 3489, 'dtp-dia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Colubris Management Port
    #: - [UDP] Colubris Management Port
    colubris: 'AppType' = 3490, 'colubris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SWR Port
    #: - [UDP] SWR Port
    swr_port: 'AppType' = 3491, 'swr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TVDUM Tray Port
    #: - [UDP] TVDUM Tray Port
    tvdumtray_port: 'AppType' = 3492, 'tvdumtray-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network UPS Tools [:rfc:`9271`]
    #: - [UDP] Network UPS Tools
    nut: 'AppType' = 3493, 'nut', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM 3494
    #: - [UDP] IBM 3494
    ibm3494: 'AppType' = 3494, 'ibm3494', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] securitylayer over tcp
    #: - [UDP] securitylayer over tcp
    seclayer_tcp: 'AppType' = 3495, 'seclayer-tcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] securitylayer over tls
    #: - [UDP] securitylayer over tls
    seclayer_tls: 'AppType' = 3496, 'seclayer-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ipEther232Port
    #: - [UDP] ipEther232Port
    ipether232port: 'AppType' = 3497, 'ipether232port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DASHPAS user port
    #: - [UDP] DASHPAS user port
    dashpas_port: 'AppType' = 3498, 'dashpas-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SccIP Media
    #: - [UDP] SccIP Media
    sccip_media: 'AppType' = 3499, 'sccip-media', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTMP Port
    #: - [UDP] RTMP Port
    rtmp_port: 'AppType' = 3500, 'rtmp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iSoft-P2P
    #: - [UDP] iSoft-P2P
    isoft_p2p: 'AppType' = 3501, 'isoft-p2p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avocent Install Discovery
    #: - [UDP] Avocent Install Discovery
    avinstalldisc: 'AppType' = 3502, 'avinstalldisc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPLS LSP-echo Port [:rfc:`8029`]
    #: - [UDP] MPLS LSP-echo Port [:rfc:`8029`]
    lsp_ping: 'AppType' = 3503, 'lsp-ping', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IronStorm game server
    #: - [UDP] IronStorm game server
    ironstorm: 'AppType' = 3504, 'ironstorm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCM communications port
    #: - [UDP] CCM communications port
    ccmcomm: 'AppType' = 3505, 'ccmcomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 3506
    #: - [UDP] APC 3506
    apc_3506: 'AppType' = 3506, 'apc-3506', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nesh Broker Port
    #: - [UDP] Nesh Broker Port
    nesh_broker: 'AppType' = 3507, 'nesh-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interaction Web
    #: - [UDP] Interaction Web
    interactionweb: 'AppType' = 3508, 'interactionweb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Token SSL Port
    #: - [UDP] Virtual Token SSL Port
    vt_ssl: 'AppType' = 3509, 'vt-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XSS Port
    #: - [UDP] XSS Port
    xss_port: 'AppType' = 3510, 'xss-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WebMail/2
    #: - [UDP] WebMail/2
    webmail_2: 'AppType' = 3511, 'webmail-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aztec Distribution Port
    #: - [UDP] Aztec Distribution Port
    aztec: 'AppType' = 3512, 'aztec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adaptec Remote Protocol
    #: - [UDP] Adaptec Remote Protocol
    arcpd: 'AppType' = 3513, 'arcpd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MUST Peer to Peer
    #: - [UDP] MUST Peer to Peer
    must_p2p: 'AppType' = 3514, 'must-p2p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MUST Backplane
    #: - [UDP] MUST Backplane
    must_backplane: 'AppType' = 3515, 'must-backplane', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Smartcard Port
    #: - [UDP] Smartcard Port
    smartcard_port: 'AppType' = 3516, 'smartcard-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IEEE 802.11 WLANs WG IAPP
    #: - [UDP] IEEE 802.11 WLANs WG IAPP
    AppType_802_11_iapp: 'AppType' = 3517, '802-11-iapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Artifact Message Server
    #: - [UDP] Artifact Message Server
    artifact_msg: 'AppType' = 3518, 'artifact-msg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Netvion Messenger Port
    nvmsgd: 'AppType' = 3519, 'nvmsgd', TransportProtocol.get('tcp')

    #: [UDP] Netvion Galileo Port
    galileo: 'AppType' = 3519, 'galileo', TransportProtocol.get('udp')

    #: - [TCP] Netvion Galileo Log Port
    #: - [UDP] Netvion Galileo Log Port
    galileolog: 'AppType' = 3520, 'galileolog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Telequip Labs MC3SS
    #: - [UDP] Telequip Labs MC3SS
    mc3ss: 'AppType' = 3521, 'mc3ss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DO over NSSocketPort
    #: - [UDP] DO over NSSocketPort
    nssocketport: 'AppType' = 3522, 'nssocketport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Odeum Serverlink
    #: - [UDP] Odeum Serverlink
    odeumservlink: 'AppType' = 3523, 'odeumservlink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ECM Server port
    #: - [UDP] ECM Server port
    ecmport: 'AppType' = 3524, 'ecmport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EIS Server port
    #: - [UDP] EIS Server port
    eisport: 'AppType' = 3525, 'eisport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] starQuiz Port
    #: - [UDP] starQuiz Port
    starquiz_port: 'AppType' = 3526, 'starquiz-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS Backup Exec Server
    #: - [UDP] VERITAS Backup Exec Server
    beserver_msg_q: 'AppType' = 3527, 'beserver-msg-q', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JBoss IIOP
    #: - [UDP] JBoss IIOP
    jboss_iiop: 'AppType' = 3528, 'jboss-iiop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JBoss IIOP/SSL
    #: - [UDP] JBoss IIOP/SSL
    jboss_iiop_ssl: 'AppType' = 3529, 'jboss-iiop-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Grid Friendly
    #: - [UDP] Grid Friendly
    gf: 'AppType' = 3530, 'gf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Joltid
    #: - [UDP] Joltid
    joltid: 'AppType' = 3531, 'joltid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Raven Remote Management Control
    #: - [UDP] Raven Remote Management Control
    raven_rmp: 'AppType' = 3532, 'raven-rmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Raven Remote Management Data
    #: - [UDP] Raven Remote Management Data
    raven_rdp: 'AppType' = 3533, 'raven-rdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] URL Daemon Port
    #: - [UDP] URL Daemon Port
    urld_port: 'AppType' = 3534, 'urld-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS-LA
    #: - [UDP] MS-LA
    ms_la: 'AppType' = 3535, 'ms-la', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNAC
    #: - [UDP] SNAC
    snac: 'AppType' = 3536, 'snac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote NI-VISA port
    #: - [UDP] Remote NI-VISA port
    ni_visa_remote: 'AppType' = 3537, 'ni-visa-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Directory Server
    #: - [UDP] IBM Directory Server
    ibm_diradm: 'AppType' = 3538, 'ibm-diradm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Directory Server SSL
    #: - [UDP] IBM Directory Server SSL
    ibm_diradm_ssl: 'AppType' = 3539, 'ibm-diradm-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PNRP User Port
    #: - [UDP] PNRP User Port
    pnrp_port: 'AppType' = 3540, 'pnrp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VoiSpeed Port
    #: - [UDP] VoiSpeed Port
    voispeed_port: 'AppType' = 3541, 'voispeed-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA cluster monitor
    #: - [UDP] HA cluster monitor
    hacl_monitor: 'AppType' = 3542, 'hacl-monitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] qftest Lookup Port
    #: - [UDP] qftest Lookup Port
    qftest_lookup: 'AppType' = 3543, 'qftest-lookup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Teredo Port [:rfc:`4380`]
    #: - [UDP] Teredo Port [:rfc:`4380`]
    teredo: 'AppType' = 3544, 'teredo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CAMAC equipment
    #: - [UDP] CAMAC equipment
    camac: 'AppType' = 3545, 'camac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_3546: 'AppType' = 3546, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Symantec SIM
    #: - [UDP] Symantec SIM
    symantec_sim: 'AppType' = 3547, 'symantec-sim', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interworld
    #: - [UDP] Interworld
    interworld: 'AppType' = 3548, 'interworld', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tellumat MDR NMS
    #: - [UDP] Tellumat MDR NMS
    tellumat_nms: 'AppType' = 3549, 'tellumat-nms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure SMPP
    #: - [UDP] Secure SMPP
    ssmpp: 'AppType' = 3550, 'ssmpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apcupsd Information Port
    #: - [UDP] Apcupsd Information Port
    apcupsd: 'AppType' = 3551, 'apcupsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TeamAgenda Server Port
    #: - [UDP] TeamAgenda Server Port
    taserver: 'AppType' = 3552, 'taserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Red Box Recorder ADP
    #: - [UDP] Red Box Recorder ADP
    rbr_discovery: 'AppType' = 3553, 'rbr-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quest Notification Server
    #: - [UDP] Quest Notification Server
    questnotify: 'AppType' = 3554, 'questnotify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vipul's Razor
    #: - [UDP] Vipul's Razor
    razor: 'AppType' = 3555, 'razor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sky Transport Protocol
    #: - [UDP] Sky Transport Protocol
    sky_transport: 'AppType' = 3556, 'sky-transport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PersonalOS Comm Port
    #: - [UDP] PersonalOS Comm Port
    personalos_001: 'AppType' = 3557, 'personalos-001', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCP user port
    #: - [UDP] MCP user port
    mcp_port: 'AppType' = 3558, 'mcp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCTV control port
    #: - [UDP] CCTV control port
    cctv_port: 'AppType' = 3559, 'cctv-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INIServe port
    #: - [UDP] INIServe port
    iniserve_port: 'AppType' = 3560, 'iniserve-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC-OneKey
    #: - [UDP] BMC-OneKey
    bmc_onekey: 'AppType' = 3561, 'bmc-onekey', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDBProxy
    #: - [UDP] SDBProxy
    sdbproxy: 'AppType' = 3562, 'sdbproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Watcom Debug
    #: - [UDP] Watcom Debug
    watcomdebug: 'AppType' = 3563, 'watcomdebug', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Electromed SIM port
    #: - [UDP] Electromed SIM port
    esimport: 'AppType' = 3564, 'esimport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] M2PA [:rfc:`4165`]
    #: - [SCTP] M2PA [:rfc:`4165`]
    m2pa: 'AppType' = 3565, 'm2pa', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_3565: 'AppType' = 3565, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Quest Data Hub
    quest_data_hub: 'AppType' = 3566, 'quest-data-hub', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3566: 'AppType' = 3566, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] DOF Protocol Stack
    #: - [UDP] DOF Protocol Stack
    dof_eps: 'AppType' = 3567, 'dof-eps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DOF Secure Tunnel
    #: - [UDP] DOF Secure Tunnel
    dof_tunnel_sec: 'AppType' = 3568, 'dof-tunnel-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Meinberg Control Service
    #: - [UDP] Meinberg Control Service
    mbg_ctrl: 'AppType' = 3569, 'mbg-ctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCC Web Server Port
    #: - [UDP] MCC Web Server Port
    mccwebsvr_port: 'AppType' = 3570, 'mccwebsvr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MegaRAID Server Port
    #: - [UDP] MegaRAID Server Port
    megardsvr_port: 'AppType' = 3571, 'megardsvr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Registration Server Port
    #: - [UDP] Registration Server Port
    megaregsvrport: 'AppType' = 3572, 'megaregsvrport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Advantage Group UPS Suite
    #: - [UDP] Advantage Group UPS Suite
    tag_ups_1: 'AppType' = 3573, 'tag-ups-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DMAF Server
    dmaf_server: 'AppType' = 3574, 'dmaf-server', TransportProtocol.get('tcp')

    #: [UDP] DMAF Caster
    dmaf_caster: 'AppType' = 3574, 'dmaf-caster', TransportProtocol.get('udp')

    #: - [TCP] Coalsere CCM Port
    #: - [UDP] Coalsere CCM Port
    ccm_port: 'AppType' = 3575, 'ccm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Coalsere CMC Port
    #: - [UDP] Coalsere CMC Port
    cmc_port: 'AppType' = 3576, 'cmc-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Configuration Port
    #: - [UDP] Configuration Port
    config_port: 'AppType' = 3577, 'config-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Port
    #: - [UDP] Data Port
    data_port: 'AppType' = 3578, 'data-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tarantella Load Balancing
    #: - [UDP] Tarantella Load Balancing
    ttat3lb: 'AppType' = 3579, 'ttat3lb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NATI-ServiceLocator
    #: - [UDP] NATI-ServiceLocator
    nati_svrloc: 'AppType' = 3580, 'nati-svrloc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ascent Capture Licensing
    #: - [UDP] Ascent Capture Licensing
    kfxaclicensing: 'AppType' = 3581, 'kfxaclicensing', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PEG PRESS Server
    #: - [UDP] PEG PRESS Server
    press: 'AppType' = 3582, 'press', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CANEX Watch System
    #: - [UDP] CANEX Watch System
    canex_watch: 'AppType' = 3583, 'canex-watch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] U-DBase Access Protocol
    #: - [UDP] U-DBase Access Protocol
    u_dbap: 'AppType' = 3584, 'u-dbap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Emprise License Server
    #: - [UDP] Emprise License Server
    emprise_lls: 'AppType' = 3585, 'emprise-lls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] License Server Console
    #: - [UDP] License Server Console
    emprise_lsc: 'AppType' = 3586, 'emprise-lsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Peer to Peer Grouping
    #: - [UDP] Peer to Peer Grouping
    p2pgroup: 'AppType' = 3587, 'p2pgroup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sentinel Server
    #: - [UDP] Sentinel Server
    sentinel: 'AppType' = 3588, 'sentinel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] isomair
    #: - [UDP] isomair
    isomair: 'AppType' = 3589, 'isomair', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WV CSP SMS Binding
    #: - [UDP] WV CSP SMS Binding
    wv_csp_sms: 'AppType' = 3590, 'wv-csp-sms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LOCANIS G-TRACK Server
    #: - [UDP] LOCANIS G-TRACK Server
    gtrack_server: 'AppType' = 3591, 'gtrack-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LOCANIS G-TRACK NE Port
    #: - [UDP] LOCANIS G-TRACK NE Port
    gtrack_ne: 'AppType' = 3592, 'gtrack-ne', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BP Model Debugger
    #: - [UDP] BP Model Debugger
    bpmd: 'AppType' = 3593, 'bpmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MediaSpace
    #: - [UDP] MediaSpace
    mediaspace: 'AppType' = 3594, 'mediaspace', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ShareApp
    #: - [UDP] ShareApp
    shareapp: 'AppType' = 3595, 'shareapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Illusion Wireless MMOG
    #: - [UDP] Illusion Wireless MMOG
    iw_mmogame: 'AppType' = 3596, 'iw-mmogame', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A14 (AN-to-SC/MM)
    #: - [UDP] A14 (AN-to-SC/MM)
    a14: 'AppType' = 3597, 'a14', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A15 (AN-to-AN)
    #: - [UDP] A15 (AN-to-AN)
    a15: 'AppType' = 3598, 'a15', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quasar Accounting Server
    #: - [UDP] Quasar Accounting Server
    quasar_server: 'AppType' = 3599, 'quasar-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] text relay-answer
    #: - [UDP] text relay-answer
    trap_daemon: 'AppType' = 3600, 'trap-daemon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Visinet Gui
    #: - [UDP] Visinet Gui
    visinet_gui: 'AppType' = 3601, 'visinet-gui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] InfiniSwitch Mgr Client
    #: - [UDP] InfiniSwitch Mgr Client
    infiniswitchcl: 'AppType' = 3602, 'infiniswitchcl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Integrated Rcvr Control
    #: - [UDP] Integrated Rcvr Control
    int_rcv_cntrl: 'AppType' = 3603, 'int-rcv-cntrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC JMX Port
    #: - [UDP] BMC JMX Port
    bmc_jmx_port: 'AppType' = 3604, 'bmc-jmx-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ComCam IO Port
    #: - [UDP] ComCam IO Port
    comcam_io: 'AppType' = 3605, 'comcam-io', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Splitlock Server
    #: - [UDP] Splitlock Server
    splitlock: 'AppType' = 3606, 'splitlock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Precise I3
    #: - [UDP] Precise I3
    precise_i3: 'AppType' = 3607, 'precise-i3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trendchip control protocol
    #: - [UDP] Trendchip control protocol
    trendchip_dcp: 'AppType' = 3608, 'trendchip-dcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CPDI PIDAS Connection Mon
    #: - [UDP] CPDI PIDAS Connection Mon
    cpdi_pidas_cm: 'AppType' = 3609, 'cpdi-pidas-cm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ECHONET
    #: - [UDP] ECHONET
    echonet: 'AppType' = 3610, 'echonet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Six Degrees Port
    #: - [UDP] Six Degrees Port
    six_degrees: 'AppType' = 3611, 'six-degrees', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Micro Focus Data Protector
    #: - [UDP] Micro Focus Data Protector
    dataprotector: 'AppType' = 3612, 'dataprotector', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alaris Device Discovery
    #: - [UDP] Alaris Device Discovery
    alaris_disc: 'AppType' = 3613, 'alaris-disc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Satchwell Sigma
    #: - [UDP] Satchwell Sigma
    sigma_port: 'AppType' = 3614, 'sigma-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Start Messaging Network
    #: - [UDP] Start Messaging Network
    start_network: 'AppType' = 3615, 'start-network', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] cd3o Control Protocol
    #: - [UDP] cd3o Control Protocol
    cd3o_protocol: 'AppType' = 3616, 'cd3o-protocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ATI SHARP Logic Engine
    #: - [UDP] ATI SHARP Logic Engine
    sharp_server: 'AppType' = 3617, 'sharp-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AAIR-Network 1
    #: - [UDP] AAIR-Network 1
    aairnet_1: 'AppType' = 3618, 'aairnet-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AAIR-Network 2
    #: - [UDP] AAIR-Network 2
    aairnet_2: 'AppType' = 3619, 'aairnet-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPSON Projector Control Port
    #: - [UDP] EPSON Projector Control Port
    ep_pcp: 'AppType' = 3620, 'ep-pcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPSON Network Screen Port
    #: - [UDP] EPSON Network Screen Port
    ep_nsp: 'AppType' = 3621, 'ep-nsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FF LAN Redundancy Port
    #: - [UDP] FF LAN Redundancy Port
    ff_lr_port: 'AppType' = 3622, 'ff-lr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HAIPIS Dynamic Discovery
    #: - [UDP] HAIPIS Dynamic Discovery
    haipe_discover: 'AppType' = 3623, 'haipe-discover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Distributed Upgrade Port
    #: - [UDP] Distributed Upgrade Port
    dist_upgrade: 'AppType' = 3624, 'dist-upgrade', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Volley
    #: - [UDP] Volley
    volley: 'AppType' = 3625, 'volley', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bvControl Daemon
    #: - [UDP] bvControl Daemon
    bvcdaemon_port: 'AppType' = 3626, 'bvcdaemon-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jam Server Port
    #: - [UDP] Jam Server Port
    jamserverport: 'AppType' = 3627, 'jamserverport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPT Machine Interface
    #: - [UDP] EPT Machine Interface
    ept_machine: 'AppType' = 3628, 'ept-machine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESC/VP.net
    #: - [UDP] ESC/VP.net
    escvpnet: 'AppType' = 3629, 'escvpnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] C&S Remote Database Port
    #: - [UDP] C&S Remote Database Port
    cs_remote_db: 'AppType' = 3630, 'cs-remote-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] C&S Web Services Port
    #: - [UDP] C&S Web Services Port
    cs_services: 'AppType' = 3631, 'cs-services', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] distributed compiler
    #: - [UDP] distributed compiler
    distcc: 'AppType' = 3632, 'distcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wyrnix AIS port
    #: - [UDP] Wyrnix AIS port
    wacp: 'AppType' = 3633, 'wacp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hNTSP Library Manager
    #: - [UDP] hNTSP Library Manager
    hlibmgr: 'AppType' = 3634, 'hlibmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Distributed Objects
    #: - [UDP] Simple Distributed Objects
    sdo: 'AppType' = 3635, 'sdo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SerVistaITSM
    #: - [UDP] SerVistaITSM
    servistaitsm: 'AppType' = 3636, 'servistaitsm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Customer Service Port
    #: - [UDP] Customer Service Port
    scservp: 'AppType' = 3637, 'scservp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EHP Backup Protocol
    #: - [UDP] EHP Backup Protocol
    ehp_backup: 'AppType' = 3638, 'ehp-backup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Extensible Automation
    #: - [UDP] Extensible Automation
    xap_ha: 'AppType' = 3639, 'xap-ha', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netplay Port 1
    #: - [UDP] Netplay Port 1
    netplay_port1: 'AppType' = 3640, 'netplay-port1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netplay Port 2
    #: - [UDP] Netplay Port 2
    netplay_port2: 'AppType' = 3641, 'netplay-port2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Juxml Replication port
    #: - [UDP] Juxml Replication port
    juxml_port: 'AppType' = 3642, 'juxml-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AudioJuggler
    #: - [UDP] AudioJuggler
    audiojuggler: 'AppType' = 3643, 'audiojuggler', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ssowatch
    #: - [UDP] ssowatch
    ssowatch: 'AppType' = 3644, 'ssowatch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cyc
    #: - [UDP] Cyc
    cyc: 'AppType' = 3645, 'cyc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XSS Server Port
    #: - [UDP] XSS Server Port
    xss_srv_port: 'AppType' = 3646, 'xss-srv-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Splitlock Gateway
    #: - [UDP] Splitlock Gateway
    splitlock_gw: 'AppType' = 3647, 'splitlock-gw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fujitsu Cooperation Port
    #: - [UDP] Fujitsu Cooperation Port
    fjcp: 'AppType' = 3648, 'fjcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nishioka Miyuki Msg Protocol
    #: - [UDP] Nishioka Miyuki Msg Protocol
    nmmp: 'AppType' = 3649, 'nmmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PRISMIQ VOD plug-in
    #: - [UDP] PRISMIQ VOD plug-in
    prismiq_plugin: 'AppType' = 3650, 'prismiq-plugin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XRPC Registry
    #: - [UDP] XRPC Registry
    xrpc_registry: 'AppType' = 3651, 'xrpc-registry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VxCR NBU Default Port
    #: - [UDP] VxCR NBU Default Port
    vxcrnbuport: 'AppType' = 3652, 'vxcrnbuport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tunnel Setup Protocol [:rfc:`5572`]
    #: - [UDP] Tunnel Setup Protocol [:rfc:`5572`]
    tsp: 'AppType' = 3653, 'tsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VAP RealTime Messenger
    #: - [UDP] VAP RealTime Messenger
    vaprtm: 'AppType' = 3654, 'vaprtm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ActiveBatch Exec Agent
    #: - [UDP] ActiveBatch Exec Agent
    abatemgr: 'AppType' = 3655, 'abatemgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ActiveBatch Job Scheduler
    #: - [UDP] ActiveBatch Job Scheduler
    abatjss: 'AppType' = 3656, 'abatjss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ImmediaNet Beacon
    #: - [UDP] ImmediaNet Beacon
    immedianet_bcn: 'AppType' = 3657, 'immedianet-bcn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PlayStation AMS (Secure)
    #: - [UDP] PlayStation AMS (Secure)
    ps_ams: 'AppType' = 3658, 'ps-ams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple SASL
    #: - [UDP] Apple SASL
    apple_sasl: 'AppType' = 3659, 'apple-sasl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Tivoli Directory Service using SSL
    #: - [UDP] IBM Tivoli Directory Service using SSL
    can_nds_ssl: 'AppType' = 3660, 'can-nds-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Tivoli Directory Service using SSL
    #: - [UDP] IBM Tivoli Directory Service using SSL
    can_ferret_ssl: 'AppType' = 3661, 'can-ferret-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pserver
    #: - [UDP] pserver
    pserver: 'AppType' = 3662, 'pserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIRECWAY Tunnel Protocol
    #: - [UDP] DIRECWAY Tunnel Protocol
    dtp: 'AppType' = 3663, 'dtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UPS Engine Port
    #: - [UDP] UPS Engine Port
    ups_engine: 'AppType' = 3664, 'ups-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Engine Port
    #: - [UDP] Enterprise Engine Port
    ent_engine: 'AppType' = 3665, 'ent-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM eServer PAP
    #: - [UDP] IBM EServer PAP
    eserver_pap: 'AppType' = 3666, 'eserver-pap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Information Exchange
    #: - [UDP] IBM Information Exchange
    infoexch: 'AppType' = 3667, 'infoexch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dell Remote Management
    #: - [UDP] Dell Remote Management
    dell_rm_port: 'AppType' = 3668, 'dell-rm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA SAN Switch Management
    #: - [UDP] CA SAN Switch Management
    casanswmgmt: 'AppType' = 3669, 'casanswmgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMILE TCP/UDP Interface
    #: - [UDP] SMILE TCP/UDP Interface
    smile: 'AppType' = 3670, 'smile', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] e Field Control (EIBnet)
    #: - [UDP] e Field Control (EIBnet)
    efcp: 'AppType' = 3671, 'efcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LispWorks ORB
    #: - [UDP] LispWorks ORB
    lispworks_orb: 'AppType' = 3672, 'lispworks-orb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Openview Media Vault GUI
    #: - [UDP] Openview Media Vault GUI
    mediavault_gui: 'AppType' = 3673, 'mediavault-gui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WinINSTALL IPC Port
    #: - [UDP] WinINSTALL IPC Port
    wininstall_ipc: 'AppType' = 3674, 'wininstall-ipc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CallTrax Data Port
    #: - [UDP] CallTrax Data Port
    calltrax: 'AppType' = 3675, 'calltrax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VisualAge Pacbase server
    #: - [UDP] VisualAge Pacbase server
    va_pacbase: 'AppType' = 3676, 'va-pacbase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RoverLog IPC
    #: - [UDP] RoverLog IPC
    roverlog: 'AppType' = 3677, 'roverlog', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataGuardianLT
    #: - [UDP] DataGuardianLT
    ipr_dglt: 'AppType' = 3678, 'ipr-dglt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Newton Dock
    #: - [UDP] Newton Dock
    escale: 'AppType' = 3679, 'escale (newton dock)', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NPDS Tracker
    #: - [UDP] NPDS Tracker
    npds_tracker: 'AppType' = 3680, 'npds-tracker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BTS X73 Port
    #: - [UDP] BTS X73 Port
    bts_x73: 'AppType' = 3681, 'bts-x73', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMC SmartPackets-MAPI
    #: - [UDP] EMC SmartPackets-MAPI
    cas_mapi: 'AppType' = 3682, 'cas-mapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC EDV/EA
    #: - [UDP] BMC EDV/EA
    bmc_ea: 'AppType' = 3683, 'bmc-ea', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FAXstfX
    #: - [UDP] FAXstfX
    faxstfx_port: 'AppType' = 3684, 'faxstfx-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DS Expert Agent
    #: - [UDP] DS Expert Agent
    dsx_agent: 'AppType' = 3685, 'dsx-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trivial Network Management
    #: - [UDP] Trivial Network Management
    tnmpv2: 'AppType' = 3686, 'tnmpv2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simple-push
    #: - [UDP] simple-push
    simple_push: 'AppType' = 3687, 'simple-push', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] simple-push Secure
    #: - [UDP] simple-push Secure
    simple_push_s: 'AppType' = 3688, 'simple-push-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Digital Audio Access Protocol (iTunes)
    #: - [UDP] Digital Audio Access Protocol (iTunes)
    daap: 'AppType' = 3689, 'daap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Magaya Network Port
    #: - [UDP] Magaya Network Port
    magaya_network: 'AppType' = 3691, 'magaya-network', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Brimstone IntelSync
    #: - [UDP] Brimstone IntelSync
    intelsync: 'AppType' = 3692, 'intelsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Emergency Automatic Structure Lockdown System
    easl: 'AppType' = 3693, 'easl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3693: 'AppType' = 3693, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_3694: 'AppType' = 3694, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] BMC Data Collection
    #: - [UDP] BMC Data Collection
    bmc_data_coll: 'AppType' = 3695, 'bmc-data-coll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Telnet Com Port Control
    #: - [UDP] Telnet Com Port Control
    telnetcpcd: 'AppType' = 3696, 'telnetcpcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NavisWorks License System
    #: - [UDP] NavisWorks License System
    nw_license: 'AppType' = 3697, 'nw-license', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAGECTLPANEL
    #: - [UDP] SAGECTLPANEL
    sagectlpanel: 'AppType' = 3698, 'sagectlpanel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Internet Call Waiting
    #: - [UDP] Internet Call Waiting
    kpn_icw: 'AppType' = 3699, 'kpn-icw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LRS NetPage
    #: - [UDP] LRS NetPage
    lrs_paging: 'AppType' = 3700, 'lrs-paging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetCelera
    #: - [UDP] NetCelera
    netcelera: 'AppType' = 3701, 'netcelera', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Web Service Discovery
    #: - [UDP] Web Service Discovery
    ws_discovery: 'AppType' = 3702, 'ws-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adobe Server 3
    #: - [UDP] Adobe Server 3
    adobeserver_3: 'AppType' = 3703, 'adobeserver-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adobe Server 4
    #: - [UDP] Adobe Server 4
    adobeserver_4: 'AppType' = 3704, 'adobeserver-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adobe Server 5
    #: - [UDP] Adobe Server 5
    adobeserver_5: 'AppType' = 3705, 'adobeserver-5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real-Time Event Port
    #: - [UDP] Real-Time Event Port
    rt_event: 'AppType' = 3706, 'rt-event', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real-Time Event Secure Port
    #: - [UDP] Real-Time Event Secure Port
    rt_event_s: 'AppType' = 3707, 'rt-event-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun App Svr - Naming
    #: - [UDP] Sun App Svr - Naming
    sun_as_iiops: 'AppType' = 3708, 'sun-as-iiops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA-IDMS Server
    #: - [UDP] CA-IDMS Server
    ca_idms: 'AppType' = 3709, 'ca-idms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PortGate Authentication
    #: - [UDP] PortGate Authentication
    portgate_auth: 'AppType' = 3710, 'portgate-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EBD Server 2
    #: - [UDP] EBD Server 2
    edb_server2: 'AppType' = 3711, 'edb-server2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sentinel Enterprise
    #: - [UDP] Sentinel Enterprise
    sentinel_ent: 'AppType' = 3712, 'sentinel-ent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TFTP over TLS
    #: - [UDP] TFTP over TLS
    tftps: 'AppType' = 3713, 'tftps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DELOS Direct Messaging
    #: - [UDP] DELOS Direct Messaging
    delos_dms: 'AppType' = 3714, 'delos-dms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anoto Rendezvous Port
    #: - [UDP] Anoto Rendezvous Port
    anoto_rendezv: 'AppType' = 3715, 'anoto-rendezv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WV CSP SMS CIR Channel
    #: - [UDP] WV CSP SMS CIR Channel
    wv_csp_sms_cir: 'AppType' = 3716, 'wv-csp-sms-cir', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WV CSP UDP/IP CIR Channel
    #: - [UDP] WV CSP UDP/IP CIR Channel
    wv_csp_udp_cir: 'AppType' = 3717, 'wv-csp-udp-cir', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OPUS Server Port
    #: - [UDP] OPUS Server Port
    opus_services: 'AppType' = 3718, 'opus-services', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iTel Server Port
    #: - [UDP] iTel Server Port
    itelserverport: 'AppType' = 3719, 'itelserverport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UF Astro. Instr. Services
    #: - [UDP] UF Astro. Instr. Services
    ufastro_instr: 'AppType' = 3720, 'ufastro-instr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xsync
    #: - [UDP] Xsync
    xsync: 'AppType' = 3721, 'xsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xserve RAID
    #: - [UDP] Xserve RAID
    xserveraid: 'AppType' = 3722, 'xserveraid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sychron Service Daemon
    #: - [UDP] Sychron Service Daemon
    sychrond: 'AppType' = 3723, 'sychrond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] World of Warcraft
    #: - [UDP] World of Warcraft
    blizwow: 'AppType' = 3724, 'blizwow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netia NA-ER Port
    #: - [UDP] Netia NA-ER Port
    na_er_tip: 'AppType' = 3725, 'na-er-tip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xyratex Array Manager
    #: - [UDP] Xyratex Array Manager
    array_manager: 'AppType' = 3726, 'array-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ericsson Mobile Data Unit
    #: - [UDP] Ericsson Mobile Data Unit
    e_mdu: 'AppType' = 3727, 'e-mdu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ericsson Web on Air
    #: - [UDP] Ericsson Web on Air
    e_woa: 'AppType' = 3728, 'e-woa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fireking Audit Port
    #: - [UDP] Fireking Audit Port
    fksp_audit: 'AppType' = 3729, 'fksp-audit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Client Control
    #: - [UDP] Client Control
    client_ctrl: 'AppType' = 3730, 'client-ctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Manager
    #: - [UDP] Service Manager
    smap: 'AppType' = 3731, 'smap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mobile Wnn
    #: - [UDP] Mobile Wnn
    m_wnn: 'AppType' = 3732, 'm-wnn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multipuesto Msg Port
    #: - [UDP] Multipuesto Msg Port
    multip_msg: 'AppType' = 3733, 'multip-msg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synel Data Collection Port
    #: - [UDP] Synel Data Collection Port
    synel_data: 'AppType' = 3734, 'synel-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Password Distribution
    #: - [UDP] Password Distribution
    pwdis: 'AppType' = 3735, 'pwdis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RealSpace RMI
    #: - [UDP] RealSpace RMI
    rs_rmi: 'AppType' = 3736, 'rs-rmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] XPanel Daemon
    xpanel: 'AppType' = 3737, 'xpanel', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3737: 'AppType' = 3737, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] versaTalk Server Port
    #: - [UDP] versaTalk Server Port
    versatalk: 'AppType' = 3738, 'versatalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Launchbird LicenseManager
    #: - [UDP] Launchbird LicenseManager
    launchbird_lm: 'AppType' = 3739, 'launchbird-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Heartbeat Protocol
    #: - [UDP] Heartbeat Protocol
    heartbeat: 'AppType' = 3740, 'heartbeat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WysDM Agent
    #: - [UDP] WysDM Agent
    wysdma: 'AppType' = 3741, 'wysdma', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CST - Configuration & Service Tracker
    #: - [UDP] CST - Configuration & Service Tracker
    cst_port: 'AppType' = 3742, 'cst-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IP Control Systems Ltd.
    #: - [UDP] IP Control Systems Ltd.
    ipcs_command: 'AppType' = 3743, 'ipcs-command', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SASG
    #: - [UDP] SASG
    sasg: 'AppType' = 3744, 'sasg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GWRTC Call Port
    #: - [UDP] GWRTC Call Port
    gw_call_port: 'AppType' = 3745, 'gw-call-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LXPRO.COM LinkTest
    #: - [UDP] LXPRO.COM LinkTest
    linktest: 'AppType' = 3746, 'linktest', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LXPRO.COM LinkTest SSL
    #: - [UDP] LXPRO.COM LinkTest SSL
    linktest_s: 'AppType' = 3747, 'linktest-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] webData
    #: - [UDP] webData
    webdata: 'AppType' = 3748, 'webdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CimTrak
    #: - [UDP] CimTrak
    cimtrak: 'AppType' = 3749, 'cimtrak', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CBOS/IP ncapsalation port
    #: - [UDP] CBOS/IP ncapsalation port
    cbos_ip_port: 'AppType' = 3750, 'cbos-ip-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CommLinx GPRS Cube
    #: - [UDP] CommLinx GPRS Cube
    gprs_cube: 'AppType' = 3751, 'gprs-cube', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vigil-IP RemoteAgent
    #: - [UDP] Vigil-IP RemoteAgent
    vipremoteagent: 'AppType' = 3752, 'vipremoteagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NattyServer Port
    #: - [UDP] NattyServer Port
    nattyserver: 'AppType' = 3753, 'nattyserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TimesTen Broker Port
    #: - [UDP] TimesTen Broker Port
    timestenbroker: 'AppType' = 3754, 'timestenbroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAS Remote Help Server
    #: - [UDP] SAS Remote Help Server
    sas_remote_hlp: 'AppType' = 3755, 'sas-remote-hlp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Canon CAPT Port
    #: - [UDP] Canon CAPT Port
    canon_capt: 'AppType' = 3756, 'canon-capt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GRF Server Port
    #: - [UDP] GRF Server Port
    grf_port: 'AppType' = 3757, 'grf-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] apw RMI registry
    #: - [UDP] apw RMI registry
    apw_registry: 'AppType' = 3758, 'apw-registry', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Exapt License Manager
    #: - [UDP] Exapt License Manager
    exapt_lmgr: 'AppType' = 3759, 'exapt-lmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] adTempus Client
    #: - [UDP] adTEmpus Client
    adtempusclient: 'AppType' = 3760, 'adtempusclient', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gsakmp port [:rfc:`4535`]
    #: - [UDP] gsakmp port [:rfc:`4535`]
    gsakmp: 'AppType' = 3761, 'gsakmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GBS SnapMail Protocol
    #: - [UDP] GBS SnapMail Protocol
    gbs_smp: 'AppType' = 3762, 'gbs-smp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XO Wave Control Port
    #: - [UDP] XO Wave Control Port
    xo_wave: 'AppType' = 3763, 'xo-wave', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MNI Protected Routing
    #: - [UDP] MNI Protected Routing
    mni_prot_rout: 'AppType' = 3764, 'mni-prot-rout', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Traceroute
    #: - [UDP] Remote Traceroute
    rtraceroute: 'AppType' = 3765, 'rtraceroute', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SSL e-watch sitewatch server
    sitewatch_s: 'AppType' = 3766, 'sitewatch-s', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3766: 'AppType' = 3766, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ListMGR Port
    #: - [UDP] ListMGR Port
    listmgr_port: 'AppType' = 3767, 'listmgr-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] rblcheckd server daemon
    #: - [UDP] rblcheckd server daemon
    rblcheckd: 'AppType' = 3768, 'rblcheckd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HAIPE Network Keying
    #: - [UDP] HAIPE Network Keying
    haipe_otnk: 'AppType' = 3769, 'haipe-otnk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cinderella Collaboration
    #: - [UDP] Cinderella Collaboration
    cindycollab: 'AppType' = 3770, 'cindycollab', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTP Paging Port
    #: - [UDP] RTP Paging Port
    paging_port: 'AppType' = 3771, 'paging-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Chantry Tunnel Protocol
    #: - [UDP] Chantry Tunnel Protocol
    ctp: 'AppType' = 3772, 'ctp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ctdhercules
    #: - [UDP] ctdhercules
    ctdhercules: 'AppType' = 3773, 'ctdhercules', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ZICOM
    #: - [UDP] ZICOM
    zicom: 'AppType' = 3774, 'zicom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISPM Manager Port
    #: - [UDP] ISPM Manager Port
    ispmmgr: 'AppType' = 3775, 'ispmmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Device Provisioning Port
    #: - [UDP] Device Provisioning Port
    dvcprov_port: 'AppType' = 3776, 'dvcprov-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jibe EdgeBurst
    #: - [UDP] Jibe EdgeBurst
    jibe_eb: 'AppType' = 3777, 'jibe-eb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cutler-Hammer IT Port
    #: - [UDP] Cutler-Hammer IT Port
    c_h_it_port: 'AppType' = 3778, 'c-h-it-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cognima Replication
    #: - [UDP] Cognima Replication
    cognima: 'AppType' = 3779, 'cognima', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nuzzler Network Protocol
    #: - [UDP] Nuzzler Network Protocol
    nnp: 'AppType' = 3780, 'nnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ABCvoice server port
    #: - [UDP] ABCvoice server port
    abcvoice_port: 'AppType' = 3781, 'abcvoice-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure ISO TP0 port
    #: - [UDP] Secure ISO TP0 port
    iso_tp0s: 'AppType' = 3782, 'iso-tp0s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Impact Mgr./PEM Gateway
    #: - [UDP] Impact Mgr./PEM Gateway
    bim_pem: 'AppType' = 3783, 'bim-pem', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BFD Control Protocol [:rfc:`5881`]
    #: - [UDP] BFD Control Protocol [:rfc:`5881`]
    bfd_control: 'AppType' = 3784, 'bfd-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BFD Echo Protocol [:rfc:`5881`]
    #: - [UDP] BFD Echo Protocol [:rfc:`5881`]
    bfd_echo: 'AppType' = 3785, 'bfd-echo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSW Upstrigger port
    #: - [UDP] VSW Upstrigger port
    upstriggervsw: 'AppType' = 3786, 'upstriggervsw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fintrx
    #: - [UDP] Fintrx
    fintrx: 'AppType' = 3787, 'fintrx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPACEWAY Routing port
    #: - [UDP] SPACEWAY Routing port
    isrp_port: 'AppType' = 3788, 'isrp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RemoteDeploy Administration Port [July 2003]
    #: - [UDP] RemoteDeploy Administration Port [July 2003]
    remotedeploy: 'AppType' = 3789, 'remotedeploy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QuickBooks RDS
    #: - [UDP] QuickBooks RDS
    quickbooksrds: 'AppType' = 3790, 'quickbooksrds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TV NetworkVideo Data port
    #: - [UDP] TV NetworkVideo Data port
    tvnetworkvideo: 'AppType' = 3791, 'tvnetworkvideo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] e-Watch Corporation SiteWatch
    #: - [UDP] e-Watch Corporation SiteWatch
    sitewatch: 'AppType' = 3792, 'sitewatch', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DataCore Software
    #: - [UDP] DataCore Software
    dcsoftware: 'AppType' = 3793, 'dcsoftware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JAUS Robots
    #: - [UDP] JAUS Robots
    jaus: 'AppType' = 3794, 'jaus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] myBLAST Mekentosj port
    #: - [UDP] myBLAST Mekentosj port
    myblast: 'AppType' = 3795, 'myblast', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spaceway Dialer
    #: - [UDP] Spaceway Dialer
    spw_dialer: 'AppType' = 3796, 'spw-dialer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] idps
    #: - [UDP] idps
    idps: 'AppType' = 3797, 'idps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Minilock
    #: - [UDP] Minilock
    minilock: 'AppType' = 3798, 'minilock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADIUS Dynamic Authorization [:rfc:`3576`]
    #: - [UDP] RADIUS Dynamic Authorization [:rfc:`3576`]
    radius_dynauth: 'AppType' = 3799, 'radius-dynauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Print Services Interface
    #: - [UDP] Print Services Interface
    pwgpsi: 'AppType' = 3800, 'pwgpsi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ibm manager service
    #: - [UDP] ibm manager service
    ibm_mgr: 'AppType' = 3801, 'ibm-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VHD
    #: - [UDP] VHD
    vhd: 'AppType' = 3802, 'vhd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SoniqSync
    #: - [UDP] SoniqSync
    soniqsync: 'AppType' = 3803, 'soniqsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Harman IQNet Port
    #: - [UDP] Harman IQNet Port
    iqnet_port: 'AppType' = 3804, 'iqnet-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ThorGuard Server Port
    #: - [UDP] ThorGuard Server Port
    tcpdataserver: 'AppType' = 3805, 'tcpdataserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote System Manager
    #: - [UDP] Remote System Manager
    wsmlb: 'AppType' = 3806, 'wsmlb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SpuGNA Communication Port
    #: - [UDP] SpuGNA Communication Port
    spugna: 'AppType' = 3807, 'spugna', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun App Svr-IIOPClntAuth
    #: - [UDP] Sun App Svr-IIOPClntAuth
    sun_as_iiops_ca: 'AppType' = 3808, 'sun-as-iiops-ca', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Java Desktop System Configuration Agent
    #: - [UDP] Java Desktop System Configuration Agent
    apocd: 'AppType' = 3809, 'apocd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WLAN AS server
    #: - [UDP] WLAN AS server
    wlanauth: 'AppType' = 3810, 'wlanauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMP
    #: - [UDP] AMP
    amp: 'AppType' = 3811, 'amp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netO WOL Server
    #: - [UDP] netO WOL Server
    neto_wol_server: 'AppType' = 3812, 'neto-wol-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rhapsody Interface Protocol
    #: - [UDP] Rhapsody Interface Protocol
    rap_ip: 'AppType' = 3813, 'rap-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] netO DCS
    #: - [UDP] netO DCS
    neto_dcs: 'AppType' = 3814, 'neto-dcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LANsurveyor XML
    #: - [UDP] LANsurveyor XML
    lansurveyorxml: 'AppType' = 3815, 'lansurveyorxml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Local Patch Server
    #: - [UDP] Sun Local Patch Server
    sunlps_http: 'AppType' = 3816, 'sunlps-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Yosemite Tech Tapeware
    #: - [UDP] Yosemite Tech Tapeware
    tapeware: 'AppType' = 3817, 'tapeware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Crinis Heartbeat
    #: - [UDP] Crinis Heartbeat
    crinis_hb: 'AppType' = 3818, 'crinis-hb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPL Sequ Layer Protocol
    #: - [UDP] EPL Sequ Layer Protocol
    epl_slp: 'AppType' = 3819, 'epl-slp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Siemens AuD SCP
    #: - [UDP] Siemens AuD SCP
    scp: 'AppType' = 3820, 'scp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ATSC PMCP Standard
    #: - [UDP] ATSC PMCP Standard
    pmcp: 'AppType' = 3821, 'pmcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compute Pool Discovery
    #: - [UDP] Compute Pool Discovery
    acp_discovery: 'AppType' = 3822, 'acp-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compute Pool Conduit
    #: - [UDP] Compute Pool Conduit
    acp_conduit: 'AppType' = 3823, 'acp-conduit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Compute Pool Policy
    #: - [UDP] Compute Pool Policy
    acp_policy: 'AppType' = 3824, 'acp-policy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Antera FlowFusion Process Simulation
    #: - [UDP] Antera FlowFusion Process Simulation
    ffserver: 'AppType' = 3825, 'ffserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WarMUX game server
    #: - [UDP] WarMUX game server
    warmux: 'AppType' = 3826, 'warmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netadmin Systems MPI service
    #: - [UDP] Netadmin Systems MPI service
    netmpi: 'AppType' = 3827, 'netmpi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netadmin Systems Event Handler
    #: - [UDP] Netadmin Systems Event Handler
    neteh: 'AppType' = 3828, 'neteh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netadmin Systems Event Handler External
    #: - [UDP] Netadmin Systems Event Handler External
    neteh_ext: 'AppType' = 3829, 'neteh-ext', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cerner System Management Agent
    #: - [UDP] Cerner System Management Agent
    cernsysmgmtagt: 'AppType' = 3830, 'cernsysmgmtagt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Docsvault Application Service
    #: - [UDP] Docsvault Application Service
    dvapps: 'AppType' = 3831, 'dvapps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xxNETserver
    #: - [UDP] xxNETserver
    xxnetserver: 'AppType' = 3832, 'xxnetserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AIPN LS Authentication
    #: - [UDP] AIPN LS Authentication
    aipn_auth: 'AppType' = 3833, 'aipn-auth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spectar Data Stream Service
    #: - [UDP] Spectar Data Stream Service
    spectardata: 'AppType' = 3834, 'spectardata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spectar Database Rights Service
    #: - [UDP] Spectar Database Rights Service
    spectardb: 'AppType' = 3835, 'spectardb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MARKEM NEXTGEN DCP
    #: - [UDP] MARKEM NEXTGEN DCP
    markem_dcp: 'AppType' = 3836, 'markem-dcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MARKEM Auto-Discovery
    #: - [UDP] MARKEM Auto-Discovery
    mkm_discovery: 'AppType' = 3837, 'mkm-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Scito Object Server
    #: - [UDP] Scito Object Server
    sos: 'AppType' = 3838, 'sos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMX Resource Management Suite
    #: - [UDP] AMX Resource Management Suite
    amx_rms: 'AppType' = 3839, 'amx-rms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] www.FlirtMitMir.de
    #: - [UDP] www.FlirtMitMir.de
    flirtmitmir: 'AppType' = 3840, 'flirtmitmir', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] ShipRush Database Server
    shiprush_db_svr: 'AppType' = 3841, 'shiprush-db-svr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3841: 'AppType' = 3841, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] NHCI status port
    #: - [UDP] NHCI status port
    nhci: 'AppType' = 3842, 'nhci', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quest Common Agent
    #: - [UDP] Quest Common Agent
    quest_agent: 'AppType' = 3843, 'quest-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RNM
    #: - [UDP] RNM
    rnm: 'AppType' = 3844, 'rnm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] V-ONE Single Port Proxy
    #: - [UDP] V-ONE Single Port Proxy
    v_one_spp: 'AppType' = 3845, 'v-one-spp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Astare Network PCP
    #: - [UDP] Astare Network PCP
    an_pcp: 'AppType' = 3846, 'an-pcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS Firewall Control
    #: - [UDP] MS Firewall Control
    msfw_control: 'AppType' = 3847, 'msfw-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IT Environmental Monitor
    #: - [UDP] IT Environmental Monitor
    item: 'AppType' = 3848, 'item', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPACEWAY DNS Preload
    #: - [UDP] SPACEWAY DNS Preload
    spw_dnspreload: 'AppType' = 3849, 'spw-dnspreload', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QTMS Bootstrap Protocol
    #: - [UDP] QTMS Bootstrap Protocol
    qtms_bootstrap: 'AppType' = 3850, 'qtms-bootstrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SpectraTalk Port
    #: - [UDP] SpectraTalk Port
    spectraport: 'AppType' = 3851, 'spectraport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSE App Configuration
    #: - [UDP] SSE App Configuration
    sse_app_config: 'AppType' = 3852, 'sse-app-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SONY scanning protocol
    #: - [UDP] SONY scanning protocol
    sscan: 'AppType' = 3853, 'sscan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Stryker Comm Port
    #: - [UDP] Stryker Comm Port
    stryker_com: 'AppType' = 3854, 'stryker-com', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenTRAC
    #: - [UDP] OpenTRAC
    opentrac: 'AppType' = 3855, 'opentrac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INFORMER
    #: - [UDP] INFORMER
    informer: 'AppType' = 3856, 'informer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trap Port
    #: - [UDP] Trap Port
    trap_port: 'AppType' = 3857, 'trap-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trap Port MOM
    #: - [UDP] Trap Port MOM
    trap_port_mom: 'AppType' = 3858, 'trap-port-mom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Navini Port
    #: - [UDP] Navini Port
    nav_port: 'AppType' = 3859, 'nav-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Server/Application State Protocol (SASP)
    #: - [UDP] Server/Application State Protocol (SASP)
    sasp: 'AppType' = 3860, 'sasp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] winShadow Host Discovery
    #: - [UDP] winShadow Host Discovery
    winshadow_hd: 'AppType' = 3861, 'winshadow-hd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GIGA-POCKET
    #: - [UDP] GIGA-POCKET
    giga_pocket: 'AppType' = 3862, 'giga-pocket', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] asap tcp port
    asap_tcp: 'AppType' = 3863, 'asap-tcp', TransportProtocol.get('tcp')

    #: [UDP] asap udp port [:rfc:`5352`]
    asap_udp: 'AppType' = 3863, 'asap-udp', TransportProtocol.get('udp')

    #: [SCTP] asap sctp [:rfc:`5352`]
    asap_sctp: 'AppType' = 3863, 'asap-sctp', TransportProtocol.get('sctp')

    #: [TCP] asap/tls tcp port [:rfc:`5352`]
    asap_tcp_tls: 'AppType' = 3864, 'asap-tcp-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_3864: 'AppType' = 3864, 'reserved', TransportProtocol.get('udp')

    #: [SCTP] asap-sctp/tls [:rfc:`5352`]
    asap_sctp_tls: 'AppType' = 3864, 'asap-sctp-tls', TransportProtocol.get('sctp')

    #: - [TCP] xpl automation protocol
    #: - [UDP] xpl automation protocol
    xpl: 'AppType' = 3865, 'xpl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun SDViz DZDAEMON Port
    #: - [UDP] Sun SDViz DZDAEMON Port
    dzdaemon: 'AppType' = 3866, 'dzdaemon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun SDViz DZOGLSERVER Port
    #: - [UDP] Sun SDViz DZOGLSERVER Port
    dzoglserver: 'AppType' = 3867, 'dzoglserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIAMETER
    #: - [SCTP] DIAMETER [:rfc:`3588`]
    diameter: 'AppType' = 3868, 'diameter', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_3868: 'AppType' = 3868, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] hp OVSAM MgmtServer Disco
    #: - [UDP] hp OVSAM MgmtServer Disco
    ovsam_mgmt: 'AppType' = 3869, 'ovsam-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] hp OVSAM HostAgent Disco
    #: - [UDP] hp OVSAM HostAgent Disco
    ovsam_d_agent: 'AppType' = 3870, 'ovsam-d-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avocent DS Authorization
    #: - [UDP] Avocent DS Authorization
    avocent_adsap: 'AppType' = 3871, 'avocent-adsap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OEM Agent
    #: - [UDP] OEM Agent
    oem_agent: 'AppType' = 3872, 'oem-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fagordnc
    #: - [UDP] fagordnc
    fagordnc: 'AppType' = 3873, 'fagordnc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SixXS Configuration
    #: - [UDP] SixXS Configuration
    sixxsconfig: 'AppType' = 3874, 'sixxsconfig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PNBSCADA
    #: - [UDP] PNBSCADA
    pnbscada: 'AppType' = 3875, 'pnbscada', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirectoryLockdown Agent IANA assigned this well-formed service name as
    #:   a replacement for "dl_agent".
    #: - [TCP] DirectoryLockdown Agent
    #: - [UDP] DirectoryLockdown Agent IANA assigned this well-formed service name as
    #:   a replacement for "dl_agent".
    #: - [UDP] DirectoryLockdown Agent
    dl_agent: 'AppType' = 3876, 'dl-agent', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] XMPCR Interface Port
    #: - [UDP] XMPCR Interface Port
    xmpcr_interface: 'AppType' = 3877, 'xmpcr-interface', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FotoG CAD interface
    #: - [UDP] FotoG CAD interface
    fotogcad: 'AppType' = 3878, 'fotogcad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] appss license manager
    #: - [UDP] appss license manager
    appss_lm: 'AppType' = 3879, 'appss-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IGRS
    #: - [UDP] IGRS
    igrs: 'AppType' = 3880, 'igrs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Acquisition and Control
    #: - [UDP] Data Acquisition and Control
    idac: 'AppType' = 3881, 'idac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTS Service Port
    #: - [UDP] DTS Service Port
    msdts1: 'AppType' = 3882, 'msdts1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VR Peripheral Network
    #: - [UDP] VR Peripheral Network
    vrpn: 'AppType' = 3883, 'vrpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SofTrack Metering
    #: - [UDP] SofTrack Metering
    softrack_meter: 'AppType' = 3884, 'softrack-meter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TopFlow SSL
    #: - [UDP] TopFlow SSL
    topflow_ssl: 'AppType' = 3885, 'topflow-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NEI management port
    #: - [UDP] NEI management port
    nei_management: 'AppType' = 3886, 'nei-management', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ciphire Data Transport
    #: - [UDP] Ciphire Data Transport
    ciphire_data: 'AppType' = 3887, 'ciphire-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ciphire Services
    #: - [UDP] Ciphire Services
    ciphire_serv: 'AppType' = 3888, 'ciphire-serv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] D and V Tester Control Port
    #: - [UDP] D and V Tester Control Port
    dandv_tester: 'AppType' = 3889, 'dandv-tester', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Niche Data Server Connect
    #: - [UDP] Niche Data Server Connect
    ndsconnect: 'AppType' = 3890, 'ndsconnect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle RTC-PM port
    #: - [UDP] Oracle RTC-PM port
    rtc_pm_port: 'AppType' = 3891, 'rtc-pm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCC-image-port
    #: - [UDP] PCC-image-port
    pcc_image_port: 'AppType' = 3892, 'pcc-image-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CGI StarAPI Server
    #: - [UDP] CGI StarAPI Server
    cgi_starapi: 'AppType' = 3893, 'cgi-starapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SyAM Agent Port
    #: - [UDP] SyAM Agent Port
    syam_agent: 'AppType' = 3894, 'syam-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SyAm SMC Service Port
    #: - [UDP] SyAm SMC Service Port
    syam_smc: 'AppType' = 3895, 'syam-smc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Distributed Objects over TLS
    #: - [UDP] Simple Distributed Objects over TLS
    sdo_tls: 'AppType' = 3896, 'sdo-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Distributed Objects over SSH
    #: - [UDP] Simple Distributed Objects over SSH
    sdo_ssh: 'AppType' = 3897, 'sdo-ssh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS, Inc. SmartEye NET Internet Protocol
    #: - [UDP] IAS, Inc. SmartEye NET Internet Protocol
    senip: 'AppType' = 3898, 'senip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITV Port
    #: - [UDP] ITV Port
    itv_control: 'AppType' = 3899, 'itv-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] udt_os IANA assigned this well-formed service name as a replacement
    #:   for "udt_os".
    #: - [TCP] udt_os
    #: - [UDP] udt_os IANA assigned this well-formed service name as a replacement
    #:   for "udt_os".
    #: - [UDP] udt_os
    udt_os_1382: 'AppType' = 1382, 'udt-os', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Unidata UDT OS IANA assigned this well-formed service name as a
    #:   replacement for "udt_os".
    #: - [TCP] Unidata UDT OS
    #: - [UDP] Unidata UDT OS IANA assigned this well-formed service name as a
    #:   replacement for "udt_os".
    #: - [UDP] Unidata UDT OS
    udt_os_3900: 'AppType' = 3900, 'udt-os', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] NIM Service Handler
    #: - [UDP] NIM Service Handler
    nimsh: 'AppType' = 3901, 'nimsh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NIMsh Auxiliary Port
    #: - [UDP] NIMsh Auxiliary Port
    nimaux: 'AppType' = 3902, 'nimaux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CharsetMGR
    #: - [UDP] CharsetMGR
    charsetmgr: 'AppType' = 3903, 'charsetmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Arnet Omnilink Port
    #: - [UDP] Arnet Omnilink Port
    omnilink_port: 'AppType' = 3904, 'omnilink-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mailbox Update (MUPDATE) protocol [:rfc:`3656`]
    #: - [UDP] Mailbox Update (MUPDATE) protocol [:rfc:`3656`]
    mupdate: 'AppType' = 3905, 'mupdate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TopoVista elevation data
    #: - [UDP] TopoVista elevation data
    topovista_data: 'AppType' = 3906, 'topovista-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Imoguia Port
    #: - [UDP] Imoguia Port
    imoguia_port: 'AppType' = 3907, 'imoguia-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Procurve NetManagement
    #: - [UDP] HP Procurve NetManagement
    hppronetman: 'AppType' = 3908, 'hppronetman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SurfControl CPA
    #: - [UDP] SurfControl CPA
    surfcontrolcpa: 'AppType' = 3909, 'surfcontrolcpa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Printer Request Port
    #: - [UDP] Printer Request Port
    prnrequest: 'AppType' = 3910, 'prnrequest', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Printer Status Port
    #: - [UDP] Printer Status Port
    prnstatus: 'AppType' = 3911, 'prnstatus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Global Maintech Stars
    #: - [UDP] Global Maintech Stars
    gbmt_stars: 'AppType' = 3912, 'gbmt-stars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ListCREATOR Port
    #: - [UDP] ListCREATOR Port
    listcrt_port: 'AppType' = 3913, 'listcrt-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ListCREATOR Port 2
    #: - [UDP] ListCREATOR Port 2
    listcrt_port_2: 'AppType' = 3914, 'listcrt-port-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Auto-Graphics Cataloging
    #: - [UDP] Auto-Graphics Cataloging
    agcat: 'AppType' = 3915, 'agcat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WysDM Controller
    #: - [UDP] WysDM Controller
    wysdmc: 'AppType' = 3916, 'wysdmc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AFT multiplex port
    #: - [UDP] AFT multiplex port
    aftmux: 'AppType' = 3917, 'aftmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PacketCableMultimediaCOPS
    #: - [UDP] PacketCableMultimediaCOPS
    pktcablemmcops: 'AppType' = 3918, 'pktcablemmcops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HyperIP
    #: - [UDP] HyperIP
    hyperip: 'AppType' = 3919, 'hyperip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Exasoft IP Port
    #: - [UDP] Exasoft IP Port
    exasoftport1: 'AppType' = 3920, 'exasoftport1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Herodotus Net
    #: - [UDP] Herodotus Net
    herodotus_net: 'AppType' = 3921, 'herodotus-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Soronti Update Port
    #: - [UDP] Soronti Update Port
    sor_update: 'AppType' = 3922, 'sor-update', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Symbian Service Broker
    #: - [UDP] Symbian Service Broker
    symb_sb_port: 'AppType' = 3923, 'symb-sb-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPL_GPRS_PORT
    #: - [UDP] MPL_GPRS_Port
    mpl_gprs_port: 'AppType' = 3924, 'mpl-gprs-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zoran Media Port
    #: - [UDP] Zoran Media Port
    zmp: 'AppType' = 3925, 'zmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WINPort
    #: - [UDP] WINPort
    winport: 'AppType' = 3926, 'winport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ScsTsr
    #: - [UDP] ScsTsr
    natdataservice: 'AppType' = 3927, 'natdataservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PXE NetBoot Manager
    #: - [UDP] PXE NetBoot Manager
    netboot_pxe: 'AppType' = 3928, 'netboot-pxe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMS Port
    #: - [UDP] AMS Port
    smauth_port: 'AppType' = 3929, 'smauth-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Syam Web Server Port
    #: - [UDP] Syam Web Server Port
    syam_webserver: 'AppType' = 3930, 'syam-webserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MSR Plugin Port
    #: - [UDP] MSR Plugin Port
    msr_plugin_port: 'AppType' = 3931, 'msr-plugin-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic Site System
    #: - [UDP] Dynamic Site System
    dyn_site: 'AppType' = 3932, 'dyn-site', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PL/B App Server User Port
    #: - [UDP] PL/B App Server User Port
    plbserve_port: 'AppType' = 3933, 'plbserve-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PL/B File Manager Port
    #: - [UDP] PL/B File Manager Port
    sunfm_port: 'AppType' = 3934, 'sunfm-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDP Port Mapper Protocol
    #: - [UDP] SDP Port Mapper Protocol
    sdp_portmapper: 'AppType' = 3935, 'sdp-portmapper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mailprox
    #: - [UDP] Mailprox
    mailprox: 'AppType' = 3936, 'mailprox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DVB Service Discovery
    #: - [UDP] DVB Service Discovery
    dvbservdsc: 'AppType' = 3937, 'dvbservdsc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle dbControl Agent po IANA assigned this well-formed service name
    #:   as a replacement for "dbcontrol_agent".
    #: - [TCP] Oracle dbControl Agent po
    #: - [UDP] Oracle dbControl Agent po IANA assigned this well-formed service name
    #:   as a replacement for "dbcontrol_agent".
    #: - [UDP] Oracle dbControl Agent po
    dbcontrol_agent: 'AppType' = 3938, 'dbcontrol-agent', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Anti-virus Application Management Port
    #: - [UDP] Anti-virus Application Management Port
    aamp: 'AppType' = 3939, 'aamp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XeCP Node Service
    #: - [UDP] XeCP Node Service
    xecp_node: 'AppType' = 3940, 'xecp-node', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Home Portal Web Server
    #: - [UDP] Home Portal Web Server
    homeportal_web: 'AppType' = 3941, 'homeportal-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] satellite distribution
    #: - [UDP] satellite distribution
    srdp: 'AppType' = 3942, 'srdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TetraNode Ip Gateway
    #: - [UDP] TetraNode Ip Gateway
    tig: 'AppType' = 3943, 'tig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] S-Ops Management
    #: - [UDP] S-Ops Management
    sops: 'AppType' = 3944, 'sops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EMCADS Server Port
    #: - [UDP] EMCADS Server Port
    emcads: 'AppType' = 3945, 'emcads', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BackupEDGE Server
    #: - [UDP] BackupEDGE Server
    backupedge: 'AppType' = 3946, 'backupedge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Connect and Control Protocol for Consumer, Commercial, and Industrial
    #:   Electronic Devices
    #: - [UDP] Connect and Control Protocol for Consumer, Commercial, and Industrial
    #:   Electronic Devices
    ccp: 'AppType' = 3947, 'ccp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anton Paar Device Administration Protocol
    #: - [UDP] Anton Paar Device Administration Protocol
    apdap: 'AppType' = 3948, 'apdap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dynamic Routing Information Protocol
    #: - [UDP] Dynamic Routing Information Protocol
    drip: 'AppType' = 3949, 'drip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Name Munging
    #: - [UDP] Name Munging
    namemunge: 'AppType' = 3950, 'namemunge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PWG IPP Facsimile
    #: - [UDP] PWG IPP Facsimile
    pwgippfax: 'AppType' = 3951, 'pwgippfax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] I3 Session Manager
    #: - [UDP] I3 Session Manager
    i3_sessionmgr: 'AppType' = 3952, 'i3-sessionmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Eydeas XMLink Connect
    #: - [UDP] Eydeas XMLink Connect
    xmlink_connect: 'AppType' = 3953, 'xmlink-connect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AD Replication RPC
    #: - [UDP] AD Replication RPC
    adrep: 'AppType' = 3954, 'adrep', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] p2pCommunity
    #: - [UDP] p2pCommunity
    p2pcommunity: 'AppType' = 3955, 'p2pcommunity', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GigE Vision Control
    #: - [UDP] GigE Vision Control
    gvcp: 'AppType' = 3956, 'gvcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MQEnterprise Broker
    #: - [UDP] MQEnterprise Broker
    mqe_broker: 'AppType' = 3957, 'mqe-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MQEnterprise Agent
    #: - [UDP] MQEnterprise Agent
    mqe_agent: 'AppType' = 3958, 'mqe-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tree Hopper Networking
    #: - [UDP] Tree Hopper Networking
    treehopper: 'AppType' = 3959, 'treehopper', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bess Peer Assessment
    #: - [UDP] Bess Peer Assessment
    bess: 'AppType' = 3960, 'bess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ProAxess Server
    #: - [UDP] ProAxess Server
    proaxess: 'AppType' = 3961, 'proaxess', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SBI Agent Protocol
    #: - [UDP] SBI Agent Protocol
    sbi_agent: 'AppType' = 3962, 'sbi-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Teran Hybrid Routing Protocol
    #: - [UDP] Teran Hybrid Routing Protocol
    thrp: 'AppType' = 3963, 'thrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SASG GPRS
    #: - [UDP] SASG GPRS
    sasggprs: 'AppType' = 3964, 'sasggprs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avanti IP to NCPE API
    #: - [UDP] Avanti IP to NCPE API
    ati_ip_to_ncpe: 'AppType' = 3965, 'ati-ip-to-ncpe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BuildForge Lock Manager
    #: - [UDP] BuildForge Lock Manager
    bflckmgr: 'AppType' = 3966, 'bflckmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PPS Message Service
    #: - [UDP] PPS Message Service
    ppsms: 'AppType' = 3967, 'ppsms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iAnywhere DBNS
    #: - [UDP] iAnywhere DBNS
    ianywhere_dbns: 'AppType' = 3968, 'ianywhere-dbns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Landmark Messages
    #: - [UDP] Landmark Messages
    landmarks: 'AppType' = 3969, 'landmarks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LANrev Agent
    #: - [UDP] LANrev Agent
    lanrevagent: 'AppType' = 3970, 'lanrevagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LANrev Server
    #: - [UDP] LANrev Server
    lanrevserver: 'AppType' = 3971, 'lanrevserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ict-control Protocol
    #: - [UDP] ict-control Protocol
    iconp: 'AppType' = 3972, 'iconp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ConnectShip Progistics
    #: - [UDP] ConnectShip Progistics
    progistics: 'AppType' = 3973, 'progistics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Applicant Tracking Service
    #: - [UDP] Remote Applicant Tracking Service
    xk22: 'AppType' = 3974, 'xk22', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Air Shot
    #: - [UDP] Air Shot
    airshot: 'AppType' = 3975, 'airshot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Server Automation Agent
    #: - [UDP] Server Automation Agent
    opswagent: 'AppType' = 3976, 'opswagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Opsware Manager
    #: - [UDP] Opsware Manager
    opswmanager: 'AppType' = 3977, 'opswmanager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secured Configuration Server
    #: - [UDP] Secured Configuration Server
    secure_cfg_svr: 'AppType' = 3978, 'secure-cfg-svr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Smith Micro Wide Area Network Service
    #: - [UDP] Smith Micro Wide Area Network Service
    smwan: 'AppType' = 3979, 'smwan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_3980: 'AppType' = 3980, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Starfish System Admin
    #: - [UDP] Starfish System Admin
    starfish: 'AppType' = 3981, 'starfish', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESRI Image Server
    #: - [UDP] ESRI Image Server
    eis: 'AppType' = 3982, 'eis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESRI Image Service
    #: - [UDP] ESRI Image Service
    eisp: 'AppType' = 3983, 'eisp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MAPPER network node manager
    #: - [UDP] MAPPER network node manager
    mapper_nodemgr: 'AppType' = 3984, 'mapper-nodemgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MAPPER TCP/IP server
    #: - [UDP] MAPPER TCP/IP server
    mapper_mapethd: 'AppType' = 3985, 'mapper-mapethd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MAPPER workstation server IANA assigned this well-formed service name
    #:   as a replacement for "mapper-ws_ethd".
    #: - [TCP] MAPPER workstation server
    #: - [UDP] MAPPER workstation server IANA assigned this well-formed service name
    #:   as a replacement for "mapper-ws_ethd".
    #: - [UDP] MAPPER workstation server
    mapper_ws_ethd: 'AppType' = 3986, 'mapper-ws-ethd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Centerline
    #: - [UDP] Centerline
    centerline: 'AppType' = 3987, 'centerline', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DCS Configuration Port
    #: - [UDP] DCS Configuration Port
    dcs_config: 'AppType' = 3988, 'dcs-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BindView-Query Engine
    #: - [UDP] BindView-Query Engine
    bv_queryengine: 'AppType' = 3989, 'bv-queryengine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BindView-IS
    #: - [UDP] BindView-IS
    bv_is: 'AppType' = 3990, 'bv-is', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BindView-SMCServer
    #: - [UDP] BindView-SMCServer
    bv_smcsrv: 'AppType' = 3991, 'bv-smcsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BindView-DirectoryServer
    #: - [UDP] BindView-DirectoryServer
    bv_ds: 'AppType' = 3992, 'bv-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BindView-Agent
    #: - [UDP] BindView-Agent
    bv_agent: 'AppType' = 3993, 'bv-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_3994: 'AppType' = 3994, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] ISS Management Svcs SSL
    #: - [UDP] ISS Management Svcs SSL
    iss_mgmt_ssl: 'AppType' = 3995, 'iss-mgmt-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] abcsoftware-01
    #: - [UDP] abcsoftware-01
    abcsoftware: 'AppType' = 3996, 'abcsoftware', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] aes_db
    #: - [UDP] aes_db
    agentsease_db: 'AppType' = 3997, 'agentsease-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Distributed Nagios Executor Service
    #: - [UDP] Distributed Nagios Executor Service
    dnx: 'AppType' = 3998, 'dnx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Norman distributes scanning service
    #: - [UDP] Norman distributes scanning service
    nvcnet: 'AppType' = 3999, 'nvcnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Terabase
    #: - [UDP] Terabase
    terabase: 'AppType' = 4000, 'terabase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NewOak
    #: - [UDP] NewOak
    newoak: 'AppType' = 4001, 'newoak', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-spvr-ft
    #: - [UDP] pxc-spvr-ft
    pxc_spvr_ft: 'AppType' = 4002, 'pxc-spvr-ft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-splr-ft
    #: - [UDP] pxc-splr-ft
    pxc_splr_ft: 'AppType' = 4003, 'pxc-splr-ft', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-roid
    #: - [UDP] pxc-roid
    pxc_roid: 'AppType' = 4004, 'pxc-roid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-pin
    #: - [UDP] pxc-pin
    pxc_pin: 'AppType' = 4005, 'pxc-pin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-spvr
    #: - [UDP] pxc-spvr
    pxc_spvr: 'AppType' = 4006, 'pxc-spvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pxc-splr
    #: - [UDP] pxc-splr
    pxc_splr: 'AppType' = 4007, 'pxc-splr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetCheque accounting
    #: - [UDP] NetCheque accounting
    netcheque: 'AppType' = 4008, 'netcheque', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Chimera HWM
    #: - [UDP] Chimera HWM
    chimera_hwm: 'AppType' = 4009, 'chimera-hwm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Samsung Unidex
    #: - [UDP] Samsung Unidex
    samsung_unidex: 'AppType' = 4010, 'samsung-unidex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alternate Service Boot
    #: - [UDP] Alternate Service Boot
    altserviceboot: 'AppType' = 4011, 'altserviceboot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDA Gate
    #: - [UDP] PDA Gate
    pda_gate: 'AppType' = 4012, 'pda-gate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACL Manager
    #: - [UDP] ACL Manager
    acl_manager: 'AppType' = 4013, 'acl-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TAICLOCK
    #: - [UDP] TAICLOCK
    taiclock: 'AppType' = 4014, 'taiclock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian Mcast
    #: - [UDP] Talarian Mcast
    talarian_mcast1: 'AppType' = 4015, 'talarian-mcast1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian Mcast
    #: - [UDP] Talarian Mcast
    talarian_mcast2: 'AppType' = 4016, 'talarian-mcast2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian Mcast
    #: - [UDP] Talarian Mcast
    talarian_mcast3: 'AppType' = 4017, 'talarian-mcast3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian Mcast
    #: - [UDP] Talarian Mcast
    talarian_mcast4: 'AppType' = 4018, 'talarian-mcast4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talarian Mcast
    #: - [UDP] Talarian Mcast
    talarian_mcast5: 'AppType' = 4019, 'talarian-mcast5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRAP Port
    #: - [UDP] TRAP Port
    trap: 'AppType' = 4020, 'trap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nexus Portal
    #: - [UDP] Nexus Portal
    nexus_portal: 'AppType' = 4021, 'nexus-portal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DNOX
    #: - [UDP] DNOX
    dnox: 'AppType' = 4022, 'dnox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESNM Zoning Port
    #: - [UDP] ESNM Zoning Port
    esnm_zoning: 'AppType' = 4023, 'esnm-zoning', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNP1 User Port
    #: - [UDP] TNP1 User Port
    tnp1_port: 'AppType' = 4024, 'tnp1-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Partition Image Port
    #: - [UDP] Partition Image Port
    partimage: 'AppType' = 4025, 'partimage', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Graphical Debug Server
    #: - [UDP] Graphical Debug Server
    as_debug: 'AppType' = 4026, 'as-debug', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bitxpress
    #: - [UDP] bitxpress
    bxp: 'AppType' = 4027, 'bxp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTServer Port
    #: - [UDP] DTServer Port
    dtserver_port: 'AppType' = 4028, 'dtserver-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IP Q signaling protocol
    #: - [UDP] IP Q signaling protocol
    ip_qsig: 'AppType' = 4029, 'ip-qsig', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Accell/JSP Daemon Port
    #: - [UDP] Accell/JSP Daemon Port
    jdmn_port: 'AppType' = 4030, 'jdmn-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UUCP over SSL
    #: - [UDP] UUCP over SSL
    suucp: 'AppType' = 4031, 'suucp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VERITAS Authorization Service
    #: - [UDP] VERITAS Authorization Service
    vrts_auth_port: 'AppType' = 4032, 'vrts-auth-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SANavigator Peer Port
    #: - [UDP] SANavigator Peer Port
    sanavigator: 'AppType' = 4033, 'sanavigator', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ubiquinox Daemon
    #: - [UDP] Ubiquinox Daemon
    ubxd: 'AppType' = 4034, 'ubxd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WAP Push OTA-HTTP port
    #: - [UDP] WAP Push OTA-HTTP port
    wap_push_http: 'AppType' = 4035, 'wap-push-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WAP Push OTA-HTTP secure
    #: - [UDP] WAP Push OTA-HTTP secure
    wap_push_https: 'AppType' = 4036, 'wap-push-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RaveHD network control
    #: - [UDP] RaveHD network control
    ravehd: 'AppType' = 4037, 'ravehd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fazzt Point-To-Point
    #: - [UDP] Fazzt Point-To-Point
    fazzt_ptp: 'AppType' = 4038, 'fazzt-ptp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fazzt Administration
    #: - [UDP] Fazzt Administration
    fazzt_admin: 'AppType' = 4039, 'fazzt-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Yo.net main service
    #: - [UDP] Yo.net main service
    yo_main: 'AppType' = 4040, 'yo-main', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rocketeer-Houston
    #: - [UDP] Rocketeer-Houston
    houston: 'AppType' = 4041, 'houston', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LDXP
    #: - [UDP] LDXP
    ldxp: 'AppType' = 4042, 'ldxp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Neighbour Identity Resolution
    #: - [UDP] Neighbour Identity Resolution
    nirp: 'AppType' = 4043, 'nirp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Location Tracking Protocol
    #: - [UDP] Location Tracking Protocol
    ltp: 'AppType' = 4044, 'ltp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Printing Protocol
    #: - [UDP] Network Printing Protocol
    npp_92: 'AppType' = 92, 'npp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Paging Protocol
    #: - [UDP] Network Paging Protocol
    npp_4045: 'AppType' = 4045, 'npp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Accounting Protocol
    #: - [UDP] Accounting Protocol
    acp_proto: 'AppType' = 4046, 'acp-proto', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Context Transfer Protocol
    #: - [UDP] Context Transfer Protocol
    ctp_state: 'AppType' = 4047, 'ctp-state', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4048: 'AppType' = 4048, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Wide Area File Services
    #: - [UDP] Wide Area File Services
    wafs: 'AppType' = 4049, 'wafs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wide Area File Services
    #: - [UDP] Wide Area File Services
    cisco_wafs: 'AppType' = 4050, 'cisco-wafs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco Peer to Peer Distribution Protocol
    #: - [UDP] Cisco Peer to Peer Distribution Protocol
    cppdp: 'AppType' = 4051, 'cppdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VoiceConnect Interact
    #: - [UDP] VoiceConnect Interact
    interact: 'AppType' = 4052, 'interact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CosmoCall Universe Communications Port 1
    #: - [UDP] CosmoCall Universe Communications Port 1
    ccu_comm_1: 'AppType' = 4053, 'ccu-comm-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CosmoCall Universe Communications Port 2
    #: - [UDP] CosmoCall Universe Communications Port 2
    ccu_comm_2: 'AppType' = 4054, 'ccu-comm-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CosmoCall Universe Communications Port 3
    #: - [UDP] CosmoCall Universe Communications Port 3
    ccu_comm_3: 'AppType' = 4055, 'ccu-comm-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Location Message Service
    #: - [UDP] Location Message Service
    lms: 'AppType' = 4056, 'lms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Servigistics WFM server
    #: - [UDP] Servigistics WFM server
    wfm: 'AppType' = 4057, 'wfm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kingfisher protocol
    #: - [UDP] Kingfisher protocol
    kingfisher: 'AppType' = 4058, 'kingfisher', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DLMS/COSEM
    #: - [UDP] DLMS/COSEM
    dlms_cosem: 'AppType' = 4059, 'dlms-cosem', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DSMETER Inter-Agent Transfer Channel IANA assigned this well-formed
    #:   service name as a replacement for "dsmeter_iatc".
    #: - [TCP] DSMETER Inter-Agent Transfer Channel
    #: - [UDP] DSMETER Inter-Agent Transfer Channel IANA assigned this well-formed
    #:   service name as a replacement for "dsmeter_iatc".
    #: - [UDP] DSMETER Inter-Agent Transfer Channel
    dsmeter_iatc: 'AppType' = 4060, 'dsmeter-iatc', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Ice Location Service (TCP)
    #: - [UDP] Ice Location Service (TCP)
    ice_location: 'AppType' = 4061, 'ice-location', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ice Location Service (SSL)
    #: - [UDP] Ice Location Service (SSL)
    ice_slocation: 'AppType' = 4062, 'ice-slocation', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ice Firewall Traversal Service (TCP)
    #: - [UDP] Ice Firewall Traversal Service (TCP)
    ice_router: 'AppType' = 4063, 'ice-router', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ice Firewall Traversal Service (SSL)
    #: - [UDP] Ice Firewall Traversal Service (SSL)
    ice_srouter: 'AppType' = 4064, 'ice-srouter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Avanti Common Data IANA assigned this well-formed service name as a
    #:   replacement for "avanti_cdp".
    #: - [TCP] Avanti Common Data
    #: - [UDP] Avanti Common Data IANA assigned this well-formed service name as a
    #:   replacement for "avanti_cdp".
    #: - [UDP] Avanti Common Data
    avanti_cdp: 'AppType' = 4065, 'avanti-cdp', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Performance Measurement and Analysis
    #: - [UDP] Performance Measurement and Analysis
    pmas: 'AppType' = 4066, 'pmas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Information Distribution Protocol
    #: - [UDP] Information Distribution Protocol
    idp: 'AppType' = 4067, 'idp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IP Fleet Broadcast
    #: - [UDP] IP Fleet Broadcast
    ipfltbcst: 'AppType' = 4068, 'ipfltbcst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Minger Email Address Validation Service
    #: - [UDP] Minger Email Address Validation Service
    minger: 'AppType' = 4069, 'minger', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trivial IP Encryption (TrIPE)
    #: - [UDP] Trivial IP Encryption (TrIPE)
    tripe: 'AppType' = 4070, 'tripe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Automatically Incremental Backup
    #: - [UDP] Automatically Incremental Backup
    aibkup: 'AppType' = 4071, 'aibkup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Zieto Socket Communications
    #: - [UDP] Zieto Socket Communications
    zieto_sock: 'AppType' = 4072, 'zieto-sock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interactive Remote Application Pairing Protocol
    #: - [UDP] Interactive Remote Application Pairing Protocol
    irapp: 'AppType' = 4073, 'irapp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cequint City ID UI trigger
    #: - [UDP] Cequint City ID UI trigger
    cequint_cityid: 'AppType' = 4074, 'cequint-cityid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ISC Alarm Message Service
    #: - [UDP] ISC Alarm Message Service
    perimlan: 'AppType' = 4075, 'perimlan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Seraph DCS
    #: - [UDP] Seraph DCS
    seraph: 'AppType' = 4076, 'seraph', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4077: 'AppType' = 4077, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Ascom IP Alarming
    ascomalarm: 'AppType' = 4077, 'ascomalarm', TransportProtocol.get('udp')

    #: [TCP] Coordinated Security Service Protocol
    cssp: 'AppType' = 4078, 'cssp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4078: 'AppType' = 4078, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SANtools Diagnostic Server
    #: - [UDP] SANtools Diagnostic Server
    santools: 'AppType' = 4079, 'santools', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lorica inside facing
    #: - [UDP] Lorica inside facing
    lorica_in: 'AppType' = 4080, 'lorica-in', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lorica inside facing (SSL)
    #: - [UDP] Lorica inside facing (SSL)
    lorica_in_sec: 'AppType' = 4081, 'lorica-in-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lorica outside facing
    #: - [UDP] Lorica outside facing
    lorica_out: 'AppType' = 4082, 'lorica-out', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lorica outside facing (SSL)
    #: - [UDP] Lorica outside facing (SSL)
    lorica_out_sec: 'AppType' = 4083, 'lorica-out-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4084: 'AppType' = 4084, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Fortisphere VM Service
    fortisphere_vm: 'AppType' = 4084, 'fortisphere-vm', TransportProtocol.get('udp')

    #: [TCP] EZNews Newsroom Message Service
    ezmessagesrv: 'AppType' = 4085, 'ezmessagesrv', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4085: 'AppType' = 4085, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4086: 'AppType' = 4086, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Firewall/NAT state table synchronization
    ftsync: 'AppType' = 4086, 'ftsync', TransportProtocol.get('udp')

    #: [TCP] APplus Service
    applusservice: 'AppType' = 4087, 'applusservice', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4087: 'AppType' = 4087, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Noah Printing Service Protocol
    npsp: 'AppType' = 4088, 'npsp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4088: 'AppType' = 4088, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] OpenCORE Remote Control Service
    #: - [UDP] OpenCORE Remote Control Service
    opencore: 'AppType' = 4089, 'opencore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA BCAST Service Guide
    #: - [UDP] OMA BCAST Service Guide
    omasgport: 'AppType' = 4090, 'omasgport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EminentWare Installer
    #: - [UDP] EminentWare Installer
    ewinstaller: 'AppType' = 4091, 'ewinstaller', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EminentWare DGS
    #: - [UDP] EminentWare DGS
    ewdgs: 'AppType' = 4092, 'ewdgs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pvx Plus CS Host
    #: - [UDP] Pvx Plus CS Host
    pvxpluscs: 'AppType' = 4093, 'pvxpluscs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sysrq daemon
    #: - [UDP] sysrq daemon
    sysrqd: 'AppType' = 4094, 'sysrqd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xtgui information service
    #: - [UDP] xtgui information service
    xtgui: 'AppType' = 4095, 'xtgui', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BRE (Bridge Relay Element)
    #: - [UDP] BRE (Bridge Relay Element)
    bre: 'AppType' = 4096, 'bre', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Patrol View
    #: - [UDP] Patrol View
    patrolview: 'AppType' = 4097, 'patrolview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] drmsfsd
    #: - [UDP] drmsfsd
    drmsfsd: 'AppType' = 4098, 'drmsfsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DPCP
    #: - [UDP] DPCP
    dpcp: 'AppType' = 4099, 'dpcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IGo Incognito Data Port
    #: - [UDP] IGo Incognito Data Port
    igo_incognito: 'AppType' = 4100, 'igo-incognito', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Braille protocol
    #: - [UDP] Braille protocol
    brlp_0: 'AppType' = 4101, 'brlp-0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Braille protocol
    #: - [UDP] Braille protocol
    brlp_1: 'AppType' = 4102, 'brlp-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Braille protocol
    #: - [UDP] Braille protocol
    brlp_2: 'AppType' = 4103, 'brlp-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Braille protocol
    #: - [UDP] Braille protocol
    brlp_3: 'AppType' = 4104, 'brlp-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Shofar
    #: - [UDP] Shofar
    shofar: 'AppType' = 4105, 'shofar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synchronite
    #: - [UDP] Synchronite
    synchronite: 'AppType' = 4106, 'synchronite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JDL Accounting LAN Service
    #: - [UDP] JDL Accounting LAN Service
    j_ac: 'AppType' = 4107, 'j-ac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACCEL
    #: - [UDP] ACCEL
    accel: 'AppType' = 4108, 'accel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instantiated Zero-control Messaging
    #: - [UDP] Instantiated Zero-control Messaging
    izm: 'AppType' = 4109, 'izm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] G2 RFID Tag Telemetry Data
    #: - [UDP] G2 RFID Tag Telemetry Data
    g2tag: 'AppType' = 4110, 'g2tag', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xgrid
    #: - [UDP] Xgrid
    xgrid: 'AppType' = 4111, 'xgrid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple VPN Server Reporting Protocol
    #: - [UDP] Apple VPN Server Reporting Protocol
    apple_vpns_rp: 'AppType' = 4112, 'apple-vpns-rp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AIPN LS Registration
    #: - [UDP] AIPN LS Registration
    aipn_reg: 'AppType' = 4113, 'aipn-reg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JomaMQMonitor
    #: - [UDP] JomaMQMonitor
    jomamqmonitor: 'AppType' = 4114, 'jomamqmonitor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CDS Transfer Agent
    #: - [UDP] CDS Transfer Agent
    cds: 'AppType' = 4115, 'cds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] smartcard-TLS
    #: - [UDP] smartcard-TLS
    smartcard_tls: 'AppType' = 4116, 'smartcard-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hillr Connection Manager
    #: - [UDP] Hillr Connection Manager
    hillrserv: 'AppType' = 4117, 'hillrserv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netadmin Systems NETscript service
    #: - [UDP] Netadmin Systems NETscript service
    netscript: 'AppType' = 4118, 'netscript', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Assuria Log Manager
    #: - [UDP] Assuria Log Manager
    assuria_slm: 'AppType' = 4119, 'assuria-slm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] MiniRem Remote Telemetry and Control
    minirem: 'AppType' = 4120, 'minirem', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4120: 'AppType' = 4120, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] e-Builder Application Communication
    #: - [UDP] e-Builder Application Communication
    e_builder: 'AppType' = 4121, 'e-builder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fiber Patrol Alarm Service
    #: - [UDP] Fiber Patrol Alarm Service
    fprams: 'AppType' = 4122, 'fprams', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rohill TetraNode Ip Gateway v2
    #: - [UDP] Rohill TetraNode Ip Gateway v2
    tigv2: 'AppType' = 4124, 'tigv2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Opsview Envoy
    #: - [UDP] Opsview Envoy
    opsview_envoy: 'AppType' = 4125, 'opsview-envoy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Data Domain Replication Service
    #: - [UDP] Data Domain Replication Service
    ddrepl: 'AppType' = 4126, 'ddrepl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetUniKeyServer
    #: - [UDP] NetUniKeyServer
    unikeypro: 'AppType' = 4127, 'unikeypro', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NuFW decision delegation protocol
    #: - [UDP] NuFW decision delegation protocol
    nufw: 'AppType' = 4128, 'nufw', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NuFW authentication protocol
    #: - [UDP] NuFW authentication protocol
    nuauth: 'AppType' = 4129, 'nuauth', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FRONET message protocol
    #: - [UDP] FRONET message protocol
    fronet: 'AppType' = 4130, 'fronet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Global Maintech Stars
    #: - [UDP] Global Maintech Stars
    stars: 'AppType' = 4131, 'stars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NUTS Daemon IANA assigned this well-formed service name as a
    #:   replacement for "nuts_dem".
    #: - [TCP] NUTS Daemon
    #: - [UDP] NUTS Daemon IANA assigned this well-formed service name as a
    #:   replacement for "nuts_dem".
    #: - [UDP] NUTS Daemon
    nuts_dem: 'AppType' = 4132, 'nuts-dem', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] NUTS Bootp Server IANA assigned this well-formed service name as a
    #:   replacement for "nuts_bootp".
    #: - [TCP] NUTS Bootp Server
    #: - [UDP] NUTS Bootp Server IANA assigned this well-formed service name as a
    #:   replacement for "nuts_bootp".
    #: - [UDP] NUTS Bootp Server
    nuts_bootp: 'AppType' = 4133, 'nuts-bootp', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] NIFTY-Serve HMI protocol
    #: - [UDP] NIFTY-Serve HMI protocol
    nifty_hmi: 'AppType' = 4134, 'nifty-hmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Classic Line Database Server Attach
    #: - [UDP] Classic Line Database Server Attach
    cl_db_attach: 'AppType' = 4135, 'cl-db-attach', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Classic Line Database Server Request
    #: - [UDP] Classic Line Database Server Request
    cl_db_request: 'AppType' = 4136, 'cl-db-request', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Classic Line Database Server Remote
    #: - [UDP] Classic Line Database Server Remote
    cl_db_remote: 'AppType' = 4137, 'cl-db-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nettest
    #: - [UDP] nettest
    nettest: 'AppType' = 4138, 'nettest', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Imperfect Networks Server
    #: - [UDP] Imperfect Networks Server
    thrtx: 'AppType' = 4139, 'thrtx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cedros Fraud Detection System IANA assigned this well-formed service
    #:   name as a replacement for "cedros_fds".
    #: - [TCP] Cedros Fraud Detection System
    #: - [UDP] Cedros Fraud Detection System IANA assigned this well-formed service
    #:   name as a replacement for "cedros_fds".
    #: - [UDP] Cedros Fraud Detection System
    cedros_fds: 'AppType' = 4140, 'cedros-fds', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Workflow Server
    #: - [UDP] Workflow Server
    oirtgsvc: 'AppType' = 4141, 'oirtgsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Document Server
    #: - [UDP] Document Server
    oidocsvc: 'AppType' = 4142, 'oidocsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Document Replication
    #: - [UDP] Document Replication
    oidsr: 'AppType' = 4143, 'oidsr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4144: 'AppType' = 4144, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] VVR Control
    #: - [UDP] VVR Control
    vvr_control: 'AppType' = 4145, 'vvr-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TGCConnect Beacon
    #: - [UDP] TGCConnect Beacon
    tgcconnect: 'AppType' = 4146, 'tgcconnect', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multum Service Manager
    #: - [UDP] Multum Service Manager
    vrxpservman: 'AppType' = 4147, 'vrxpservman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HHB Handheld Client
    #: - [UDP] HHB Handheld Client
    hhb_handheld: 'AppType' = 4148, 'hhb-handheld', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A10 GSLB Service
    #: - [UDP] A10 GSLB Service
    agslb: 'AppType' = 4149, 'agslb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PowerAlert Network Shutdown Agent
    #: - [UDP] PowerAlert Network Shutdown Agent
    poweralert_nsa: 'AppType' = 4150, 'poweralert-nsa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Men & Mice Remote Control IANA assigned this well-formed service name
    #:   as a replacement for "menandmice_noh".
    #: - [TCP] Men & Mice Remote Control
    #: - [UDP] Men & Mice Remote Control IANA assigned this well-formed service name
    #:   as a replacement for "menandmice_noh".
    #: - [UDP] Men & Mice Remote Control
    menandmice_noh: 'AppType' = 4151, 'menandmice-noh', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] iDigTech Multiplex IANA assigned this well-formed service name as a
    #:   replacement for "idig_mux".
    #: - [TCP] iDigTech Multiplex
    #: - [UDP] iDigTech Multiplex IANA assigned this well-formed service name as a
    #:   replacement for "idig_mux".
    #: - [UDP] iDigTech Multiplex
    idig_mux: 'AppType' = 4152, 'idig-mux', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] MBL Remote Battery Monitoring
    #: - [UDP] MBL Remote Battery Monitoring
    mbl_battd: 'AppType' = 4153, 'mbl-battd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] atlinks device discovery
    #: - [UDP] atlinks device discovery
    atlinks: 'AppType' = 4154, 'atlinks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bazaar version control system
    #: - [UDP] Bazaar version control system
    bzr: 'AppType' = 4155, 'bzr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STAT Results
    #: - [UDP] STAT Results
    stat_results: 'AppType' = 4156, 'stat-results', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STAT Scanner Control
    #: - [UDP] STAT Scanner Control
    stat_scanner: 'AppType' = 4157, 'stat-scanner', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STAT Command Center
    #: - [UDP] STAT Command Center
    stat_cc: 'AppType' = 4158, 'stat-cc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Security Service
    #: - [UDP] Network Security Service
    nss: 'AppType' = 4159, 'nss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jini Discovery
    #: - [UDP] Jini Discovery
    jini_discovery: 'AppType' = 4160, 'jini-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMS Contact
    #: - [UDP] OMS Contact
    omscontact: 'AppType' = 4161, 'omscontact', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMS Topology
    #: - [UDP] OMS Topology
    omstopology: 'AppType' = 4162, 'omstopology', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Silver Peak Peer Protocol
    #: - [UDP] Silver Peak Peer Protocol
    silverpeakpeer: 'AppType' = 4163, 'silverpeakpeer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Silver Peak Communication Protocol
    #: - [UDP] Silver Peak Communication Protocol
    silverpeakcomm: 'AppType' = 4164, 'silverpeakcomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ArcLink over Ethernet
    #: - [UDP] ArcLink over Ethernet
    altcp: 'AppType' = 4165, 'altcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Joost Peer to Peer Protocol
    #: - [UDP] Joost Peer to Peer Protocol
    joost: 'AppType' = 4166, 'joost', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DeskDirect Global Network
    #: - [UDP] DeskDirect Global Network
    ddgn: 'AppType' = 4167, 'ddgn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PrintSoft License Server
    #: - [UDP] PrintSoft License Server
    pslicser: 'AppType' = 4168, 'pslicser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Automation Drive Interface Transport
    iadt: 'AppType' = 4169, 'iadt', TransportProtocol.get('tcp')

    #: [UDP] Internet ADT Discovery Protocol
    iadt_disc: 'AppType' = 4169, 'iadt-disc', TransportProtocol.get('udp')

    #: [TCP] SMPTE Content Synchonization Protocol
    d_cinema_csp: 'AppType' = 4170, 'd-cinema-csp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4170: 'AppType' = 4170, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Maxlogic Supervisor Communication
    ml_svnet: 'AppType' = 4171, 'ml-svnet', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4171: 'AppType' = 4171, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] PC over IP
    #: - [UDP] PC over IP
    pcoip: 'AppType' = 4172, 'pcoip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4173: 'AppType' = 4173, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] MMA Device Discovery
    mma_discovery: 'AppType' = 4173, 'mma-discovery', TransportProtocol.get('udp')

    #: [TCP] StorMagic Cluster Services
    smcluster: 'AppType' = 4174, 'smcluster', TransportProtocol.get('tcp')

    #: [UDP] StorMagic Discovery
    sm_disc: 'AppType' = 4174, 'sm-disc', TransportProtocol.get('udp')

    #: [TCP] Brocade Cluster Communication Protocol
    bccp: 'AppType' = 4175, 'bccp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4175: 'AppType' = 4175, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Translattice Cluster IPC Proxy
    tl_ipcproxy: 'AppType' = 4176, 'tl-ipcproxy', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4176: 'AppType' = 4176, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Wello P2P pubsub service
    #: - [UDP] Wello P2P pubsub service
    wello: 'AppType' = 4177, 'wello', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] StorMan
    #: - [UDP] StorMan
    storman: 'AppType' = 4178, 'storman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Maxum Services
    #: - [UDP] Maxum Services
    maxumsp: 'AppType' = 4179, 'maxumsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTTPX
    #: - [UDP] HTTPX
    httpx: 'AppType' = 4180, 'httpx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MacBak
    #: - [UDP] MacBak
    macbak: 'AppType' = 4181, 'macbak', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Production Company Pro TCP Service
    #: - [UDP] Production Company Pro TCP Service
    pcptcpservice: 'AppType' = 4182, 'pcptcpservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CyborgNet communications protocol
    #: - [UDP] CyborgNet communications protocol
    cyborgnet: 'AppType' = 4183, 'cyborgnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UNIVERSE SUITE MESSAGE SERVICE IANA assigned this well-formed service
    #:   name as a replacement for "universe_suite".
    #: - [TCP] UNIVERSE SUITE MESSAGE SERVICE
    #: - [UDP] UNIVERSE SUITE MESSAGE SERVICE IANA assigned this well-formed service
    #:   name as a replacement for "universe_suite".
    #: - [UDP] UNIVERSE SUITE MESSAGE SERVICE
    universe_suite: 'AppType' = 4184, 'universe-suite', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Woven Control Plane Protocol
    #: - [UDP] Woven Control Plane Protocol
    wcpp: 'AppType' = 4185, 'wcpp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Box Backup Store Service
    boxbackupstore: 'AppType' = 4186, 'boxbackupstore', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4186: 'AppType' = 4186, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Cascade Proxy IANA assigned this well-formed service name as a
    #:   replacement for "csc_proxy".
    #: - [TCP] Cascade Proxy
    csc_proxy: 'AppType' = 4187, 'csc-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4187: 'AppType' = 4187, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Vatata Peer to Peer Protocol
    #: - [UDP] Vatata Peer to Peer Protocol
    vatata: 'AppType' = 4188, 'vatata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Path Computation Element Communication Protocol
    pcep: 'AppType' = 4189, 'pcep', TransportProtocol.get('tcp')

    #: [UDP] Reserved [:rfc:`5440`]
    reserved_4189: 'AppType' = 4189, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ManageSieve Protocol
    sieve: 'AppType' = 4190, 'sieve', TransportProtocol.get('tcp')

    #: [UDP] Reserved [:rfc:`5804`]
    reserved_4190: 'AppType' = 4190, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4191: 'AppType' = 4191, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Dual Stack MIPv6 NAT Traversal [:rfc:`5555`]
    dsmipv6: 'AppType' = 4191, 'dsmipv6', TransportProtocol.get('udp')

    #: [TCP] Azeti Agent Service
    azeti: 'AppType' = 4192, 'azeti', TransportProtocol.get('tcp')

    #: [UDP] azeti blinddate
    azeti_bd: 'AppType' = 4192, 'azeti-bd', TransportProtocol.get('udp')

    #: [TCP] PxPlus remote file srvr
    pvxplusio: 'AppType' = 4193, 'pvxplusio', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4193: 'AppType' = 4193, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Security Protocol and Data Model
    spdm: 'AppType' = 4194, 'spdm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4194: 'AppType' = 4194, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] AWS protocol for cloud remoting solution
    #: - [UDP] AWS protocol for cloud remoting solution
    #: - [SCTP] AWS protocol for cloud remoting solution
    #: - [DCCP] AWS protocol for cloud remoting solution
    aws_wsp: 'AppType' = 4195, 'aws-wsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp') | TransportProtocol.get('dccp')

    #: [N/A] Unassigned
    unassigned_4196: 'AppType' = 4196, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Harman HControl Protocol
    #: - [UDP] Harman HControl Protocol
    hctl: 'AppType' = 4197, 'hctl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4198: 'AppType' = 4198, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] EIMS ADMIN
    #: - [UDP] EIMS ADMIN
    eims_admin: 'AppType' = 4199, 'eims-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Corel CCam
    #: - [UDP] Corel CCam
    corelccam: 'AppType' = 4300, 'corelccam', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Diagnostic Data
    #: - [UDP] Diagnostic Data
    d_data: 'AppType' = 4301, 'd-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Diagnostic Data Control
    #: - [UDP] Diagnostic Data Control
    d_data_control: 'AppType' = 4302, 'd-data-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Railroad Command Protocol
    #: - [UDP] Simple Railroad Command Protocol
    srcp: 'AppType' = 4303, 'srcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] One-Wire Filesystem Server
    #: - [UDP] One-Wire Filesystem Server
    owserver: 'AppType' = 4304, 'owserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] better approach to mobile ad-hoc networking
    #: - [UDP] better approach to mobile ad-hoc networking
    batman: 'AppType' = 4305, 'batman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Hellgate London
    #: - [UDP] Hellgate London
    pinghgl: 'AppType' = 4306, 'pinghgl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TrueConf Videoconference Service
    #: - [UDP] TrueConf Videoconference Service
    trueconf: 'AppType' = 4307, 'trueconf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CompX-LockView
    #: - [UDP] CompX-LockView
    compx_lockview: 'AppType' = 4308, 'compx-lockview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Exsequi Appliance Discovery
    #: - [UDP] Exsequi Appliance Discovery
    dserver: 'AppType' = 4309, 'dserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mir-RT exchange service
    #: - [UDP] Mir-RT exchange service
    mirrtex: 'AppType' = 4310, 'mirrtex', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] P6R Secure Server Management Console
    p6ssmc: 'AppType' = 4311, 'p6ssmc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4311: 'AppType' = 4311, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Parascale Membership Manager
    pscl_mgt: 'AppType' = 4312, 'pscl-mgt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4312: 'AppType' = 4312, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PERRLA User Services
    perrla: 'AppType' = 4313, 'perrla', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4313: 'AppType' = 4313, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ChoiceView Agent
    choiceview_agt: 'AppType' = 4314, 'choiceview-agt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4314: 'AppType' = 4314, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4315: 'AppType' = 4315, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] ChoiceView Client
    choiceview_clt: 'AppType' = 4316, 'choiceview-clt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4316: 'AppType' = 4316, 'reserved', TransportProtocol.get('udp')

    #: [TCP] OpenTelemetry Protocol
    opentelemetry: 'AppType' = 4317, 'opentelemetry', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4317: 'AppType' = 4317, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4318: 'AppType' = 4318, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Fox SkyTale encrypted communication
    #: - [UDP] Fox SkyTale encrypted communication
    fox_skytale: 'AppType' = 4319, 'fox-skytale', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FDT Remote Categorization Protocol
    #: - [UDP] FDT Remote Categorization Protocol
    fdt_rcatp: 'AppType' = 4320, 'fdt-rcatp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Who Is [:rfc:`2167`]
    #: - [UDP] Remote Who Is [:rfc:`2167`]
    rwhois: 'AppType' = 4321, 'rwhois', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIM Event Service
    #: - [UDP] TRIM Event Service
    trim_event: 'AppType' = 4322, 'trim-event', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIM ICE Service
    #: - [UDP] TRIM ICE Service
    trim_ice: 'AppType' = 4323, 'trim-ice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Reserved
    reserved_4324: 'AppType' = 4324, 'reserved', TransportProtocol.get('undefined')

    #: [TCP] Cadcorp GeognoSIS Administrator
    geognosisadmin: 'AppType' = 4325, 'geognosisadmin', TransportProtocol.get('tcp')

    #: [UDP] Cadcorp GeognoSIS Administrator
    geognosisman: 'AppType' = 4325, 'geognosisman', TransportProtocol.get('udp')

    #: - [TCP] Cadcorp GeognoSIS
    #: - [UDP] Cadcorp GeognoSIS
    geognosis: 'AppType' = 4326, 'geognosis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jaxer Web Protocol
    #: - [UDP] Jaxer Web Protocol
    jaxer_web: 'AppType' = 4327, 'jaxer-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Jaxer Manager Command Protocol
    #: - [UDP] Jaxer Manager Command Protocol
    jaxer_manager: 'AppType' = 4328, 'jaxer-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PubliQare Distributed Environment Synchronisation Engine
    publiqare_sync: 'AppType' = 4329, 'publiqare-sync', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4329: 'AppType' = 4329, 'reserved', TransportProtocol.get('udp')

    #: [TCP] DEY Storage Administration REST API
    dey_sapi: 'AppType' = 4330, 'dey-sapi', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4330: 'AppType' = 4330, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ktickets REST API for event management and ticketing systems (embedded
    #: POS devices)
    ktickets_rest: 'AppType' = 4331, 'ktickets-rest', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4331: 'AppType' = 4331, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Getty Images FOCUS service
    getty_focus: 'AppType' = 4332, 'getty-focus', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4332: 'AppType' = 4332, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ArrowHead Service Protocol (AHSP)
    #: - [UDP] ArrowHead Service Protocol (AHSP)
    #: - [SCTP] ArrowHead Service Protocol (AHSP)
    ahsp: 'AppType' = 4333, 'ahsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: [TCP] NETCONF Call Home (SSH) [:rfc:`8071`]
    netconf_ch_ssh: 'AppType' = 4334, 'netconf-ch-ssh', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4334: 'AppType' = 4334, 'reserved', TransportProtocol.get('udp')

    #: [TCP] NETCONF Call Home (TLS) [:rfc:`8071`]
    netconf_ch_tls: 'AppType' = 4335, 'netconf-ch-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4335: 'AppType' = 4335, 'reserved', TransportProtocol.get('udp')

    #: [TCP] RESTCONF Call Home (TLS) [:rfc:`8071`]
    restconf_ch_tls: 'AppType' = 4336, 'restconf-ch-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4336: 'AppType' = 4336, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Gaia Connector Protocol
    #: - [UDP] Gaia Connector Protocol
    gaia: 'AppType' = 4340, 'gaia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4341: 'AppType' = 4341, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] LISP Data Packets [:rfc:`9300`]
    lisp_data: 'AppType' = 4341, 'lisp-data', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4342: 'AppType' = 4342, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] LISP Control Packets [:rfc:`9301`]
    lisp_control: 'AppType' = 4342, 'lisp-control', TransportProtocol.get('udp')

    #: - [TCP] UNICALL
    #: - [UDP] UNICALL
    unicall: 'AppType' = 4343, 'unicall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VinaInstall
    #: - [UDP] VinaInstall
    vinainstall: 'AppType' = 4344, 'vinainstall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Macro 4 Network AS
    #: - [UDP] Macro 4 Network AS
    m4_network_as: 'AppType' = 4345, 'm4-network-as', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ELAN LM
    #: - [UDP] ELAN LM
    elanlm: 'AppType' = 4346, 'elanlm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LAN Surveyor
    #: - [UDP] LAN Surveyor
    lansurveyor: 'AppType' = 4347, 'lansurveyor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITOSE
    #: - [UDP] ITOSE
    itose: 'AppType' = 4348, 'itose', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] File System Port Map
    #: - [UDP] File System Port Map
    fsportmap: 'AppType' = 4349, 'fsportmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Device
    #: - [UDP] Net Device
    net_device: 'AppType' = 4350, 'net-device', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PLCY Net Services
    #: - [UDP] PLCY Net Services
    plcy_net_svcs: 'AppType' = 4351, 'plcy-net-svcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Projector Link
    #: - [UDP] Projector Link
    pjlink: 'AppType' = 4352, 'pjlink', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] F5 iQuery
    #: - [UDP] F5 iQuery
    f5_iquery: 'AppType' = 4353, 'f5-iquery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSNet Transmitter
    #: - [UDP] QSNet Transmitter
    qsnet_trans: 'AppType' = 4354, 'qsnet-trans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSNet Workstation
    #: - [UDP] QSNet Workstation
    qsnet_workst: 'AppType' = 4355, 'qsnet-workst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSNet Assistant
    #: - [UDP] QSNet Assistant
    qsnet_assist: 'AppType' = 4356, 'qsnet-assist', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSNet Conductor
    #: - [UDP] QSNet Conductor
    qsnet_cond: 'AppType' = 4357, 'qsnet-cond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QSNet Nucleus
    #: - [UDP] QSNet Nucleus
    qsnet_nucl: 'AppType' = 4358, 'qsnet-nucl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA BCAST Long-Term Key Messages
    #: - [UDP] OMA BCAST Long-Term Key Messages
    omabcastltkm: 'AppType' = 4359, 'omabcastltkm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Matrix VNet Communication Protocol IANA assigned this well-formed
    #:   service name as a replacement for "matrix_vnet".
    #: - [TCP] Matrix VNet Communication Protocol
    matrix_vnet: 'AppType' = 4360, 'matrix-vnet', TransportProtocol.get('tcp') | TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4360: 'AppType' = 4360, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4361: 'AppType' = 4361, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] NavCom Discovery and Control Port
    nacnl: 'AppType' = 4361, 'nacnl', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4362: 'AppType' = 4362, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] AFORE vNode Discovery protocol
    afore_vdp_disc: 'AppType' = 4362, 'afore-vdp-disc', TransportProtocol.get('udp')

    #: [UDP] ShadowStream System
    shadowstream: 'AppType' = 4366, 'shadowstream', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4366: 'AppType' = 4366, 'reserved', TransportProtocol.get('tcp')

    #: [N/A] Unassigned
    unassigned_4367: 'AppType' = 4367, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] WeatherBrief Direct
    #: - [UDP] WeatherBrief Direct
    wxbrief: 'AppType' = 4368, 'wxbrief', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Erlang Port Mapper Daemon
    #: - [UDP] Erlang Port Mapper Daemon
    epmd: 'AppType' = 4369, 'epmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ELPRO V2 Protocol Tunnel IANA assigned this well-formed service name
    #:   as a replacement for "elpro_tunnel".
    #: - [TCP] ELPRO V2 Protocol Tunnel
    #: - [UDP] ELPRO V2 Protocol Tunnel IANA assigned this well-formed service name
    #:   as a replacement for "elpro_tunnel".
    #: - [UDP] ELPRO V2 Protocol Tunnel
    elpro_tunnel: 'AppType' = 4370, 'elpro-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [TCP] LAN2CAN Control
    l2c_control: 'AppType' = 4371, 'l2c-control', TransportProtocol.get('tcp')

    #: [UDP] LAN2CAN Discovery
    l2c_disc: 'AppType' = 4371, 'l2c-disc', TransportProtocol.get('udp')

    #: - [TCP] LAN2CAN Data
    #: - [UDP] LAN2CAN Data
    l2c_data: 'AppType' = 4372, 'l2c-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Authenticated Command Service
    #: - [UDP] Remote Authenticated Command Service
    remctl: 'AppType' = 4373, 'remctl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PSI Push-to-Talk Protocol
    psi_ptt: 'AppType' = 4374, 'psi-ptt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4374: 'AppType' = 4374, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Toltec EasyShare
    #: - [UDP] Toltec EasyShare
    tolteces: 'AppType' = 4375, 'tolteces', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BioAPI Interworking
    #: - [UDP] BioAPI Interworking
    bip: 'AppType' = 4376, 'bip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cambridge Pixel SPx Server
    #: - [UDP] Cambridge Pixel SPx Server
    cp_spxsvr: 'AppType' = 4377, 'cp-spxsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cambridge Pixel SPx Display
    #: - [UDP] Cambridge Pixel SPx Display
    cp_spxdpy: 'AppType' = 4378, 'cp-spxdpy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CTDB
    #: - [UDP] CTDB
    ctdb: 'AppType' = 4379, 'ctdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xandros Community Management Service
    #: - [UDP] Xandros Community Management Service
    xandros_cms: 'AppType' = 4389, 'xandros-cms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Physical Access Control
    #: - [UDP] Physical Access Control
    wiegand: 'AppType' = 4390, 'wiegand', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] American Printware IMServer Protocol
    apwi_imserver: 'AppType' = 4391, 'apwi-imserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4391: 'AppType' = 4391, 'reserved', TransportProtocol.get('udp')

    #: [TCP] American Printware RXServer Protocol
    apwi_rxserver: 'AppType' = 4392, 'apwi-rxserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4392: 'AppType' = 4392, 'reserved', TransportProtocol.get('udp')

    #: [TCP] American Printware RXSpooler Protocol
    apwi_rxspooler: 'AppType' = 4393, 'apwi-rxspooler', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4393: 'AppType' = 4393, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4394: 'AppType' = 4394, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] American Printware Discovery
    apwi_disc: 'AppType' = 4394, 'apwi-disc', TransportProtocol.get('udp')

    #: - [TCP] OmniVision communication for Virtual environments
    #: - [UDP] OmniVision communication for Virtual environments
    omnivisionesx: 'AppType' = 4395, 'omnivisionesx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Fly Object Space
    fly: 'AppType' = 4396, 'fly', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4396: 'AppType' = 4396, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Services
    #: - [UDP] ASIGRA Services
    ds_srv: 'AppType' = 4400, 'ds-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting DS-System Service
    #: - [UDP] ASIGRA Televaulting DS-System Service
    ds_srvr: 'AppType' = 4401, 'ds-srvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting DS-Client Service
    #: - [UDP] ASIGRA Televaulting DS-Client Service
    ds_clnt: 'AppType' = 4402, 'ds-clnt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting DS-Client Monitoring/Management
    #: - [UDP] ASIGRA Televaulting DS-Client Monitoring/Management
    ds_user: 'AppType' = 4403, 'ds-user', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting DS-System Monitoring/Management
    #: - [UDP] ASIGRA Televaulting DS-System Monitoring/Management
    ds_admin: 'AppType' = 4404, 'ds-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting Message Level Restore service
    #: - [UDP] ASIGRA Televaulting Message Level Restore service
    ds_mail: 'AppType' = 4405, 'ds-mail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASIGRA Televaulting DS-Sleeper Service
    #: - [UDP] ASIGRA Televaulting DS-Sleeper Service
    ds_slp: 'AppType' = 4406, 'ds-slp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Network Access Control Agent
    nacagent: 'AppType' = 4407, 'nacagent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4407: 'AppType' = 4407, 'reserved', TransportProtocol.get('udp')

    #: [TCP] SLS Technology Control Centre
    slscc: 'AppType' = 4408, 'slscc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4408: 'AppType' = 4408, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Net-Cabinet comunication
    netcabinet_com: 'AppType' = 4409, 'netcabinet-com', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4409: 'AppType' = 4409, 'reserved', TransportProtocol.get('udp')

    #: [TCP] RIB iTWO Application Server
    itwo_server: 'AppType' = 4410, 'itwo-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4410: 'AppType' = 4410, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Found Messaging Protocol
    found: 'AppType' = 4411, 'found', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4411: 'AppType' = 4411, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4412: 'AppType' = 4412, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] SmallChat
    smallchat: 'AppType' = 4412, 'smallchat', TransportProtocol.get('udp')

    #: [TCP] AVI Systems NMS
    avi_nms: 'AppType' = 4413, 'avi-nms', TransportProtocol.get('tcp')

    #: [UDP] AVI Systems NMS
    avi_nms_disc: 'AppType' = 4413, 'avi-nms-disc', TransportProtocol.get('udp')

    #: [TCP] Updog Monitoring and Status Framework
    updog: 'AppType' = 4414, 'updog', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4414: 'AppType' = 4414, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Brocade Virtual Router Request
    brcd_vr_req: 'AppType' = 4415, 'brcd-vr-req', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4415: 'AppType' = 4415, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PJJ Media Player
    pjj_player: 'AppType' = 4416, 'pjj-player', TransportProtocol.get('tcp')

    #: [UDP] PJJ Media Player discovery
    pjj_player_disc: 'AppType' = 4416, 'pjj-player-disc', TransportProtocol.get('udp')

    #: [TCP] Workflow Director Communication
    workflowdir: 'AppType' = 4417, 'workflowdir', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4417: 'AppType' = 4417, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4418: 'AppType' = 4418, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] AXYS communication protocol
    axysbridge: 'AppType' = 4418, 'axysbridge', TransportProtocol.get('udp')

    #: [TCP] Colnod Binary Protocol
    cbp: 'AppType' = 4419, 'cbp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4419: 'AppType' = 4419, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] NVM Express over Fabrics storage access
    #: - [UDP] NVM Express over Fabrics storage access
    nvme: 'AppType' = 4420, 'nvme', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Multi-Platform Remote Management for Cloud Infrastructure
    scaleft: 'AppType' = 4421, 'scaleft', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4421: 'AppType' = 4421, 'reserved', TransportProtocol.get('udp')

    #: [TCP] TSEP Installation Service Protocol
    tsepisp: 'AppType' = 4422, 'tsepisp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4422: 'AppType' = 4422, 'reserved', TransportProtocol.get('udp')

    #: [TCP] thingkit secure mesh
    thingkit: 'AppType' = 4423, 'thingkit', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4423: 'AppType' = 4423, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4424: 'AppType' = 4424, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] NetROCKEY6 SMART Plus Service
    #: - [UDP] NetROCKEY6 SMART Plus Service
    netrockey6: 'AppType' = 4425, 'netrockey6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMARTS Beacon Port
    #: - [UDP] SMARTS Beacon Port
    beacon_port_2: 'AppType' = 4426, 'beacon-port-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Drizzle database server
    drizzle: 'AppType' = 4427, 'drizzle', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4427: 'AppType' = 4427, 'reserved', TransportProtocol.get('udp')

    #: [TCP] OMV-Investigation Server-Client
    omviserver: 'AppType' = 4428, 'omviserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4428: 'AppType' = 4428, 'reserved', TransportProtocol.get('udp')

    #: [TCP] OMV Investigation Agent-Server
    omviagent: 'AppType' = 4429, 'omviagent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4429: 'AppType' = 4429, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] REAL SQL Server
    #: - [UDP] REAL SQL Server
    rsqlserver: 'AppType' = 4430, 'rsqlserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] adWISE Pipe
    wspipe: 'AppType' = 4431, 'wspipe', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4431: 'AppType' = 4431, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] L-ACOUSTICS management
    #: - [UDP] L-ACOUSTICS management
    l_acoustics: 'AppType' = 4432, 'l-acoustics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Versile Object Protocol
    vop: 'AppType' = 4433, 'vop', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4433: 'AppType' = 4433, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4441: 'AppType' = 4441, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Netblox Protocol
    netblox: 'AppType' = 4441, 'netblox', TransportProtocol.get('udp')

    #: - [TCP] Saris
    #: - [UDP] Saris
    saris: 'AppType' = 4442, 'saris', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pharos
    #: - [UDP] Pharos
    pharos: 'AppType' = 4443, 'pharos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] KRB524
    #: - [UDP] KRB524
    krb524: 'AppType' = 4444, 'krb524', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NV Video default
    #: - [UDP] NV Video default
    nv_video: 'AppType' = 4444, 'nv-video', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] UPNOTIFYP
    #: - [UDP] UPNOTIFYP
    upnotifyp: 'AppType' = 4445, 'upnotifyp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] N1-FWP
    #: - [UDP] N1-FWP
    n1_fwp: 'AppType' = 4446, 'n1-fwp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] N1-RMGMT
    #: - [UDP] N1-RMGMT
    n1_rmgmt: 'AppType' = 4447, 'n1-rmgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ASC Licence Manager
    #: - [UDP] ASC Licence Manager
    asc_slmd: 'AppType' = 4448, 'asc-slmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PrivateWire
    #: - [UDP] PrivateWire
    privatewire: 'AppType' = 4449, 'privatewire', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Common ASCII Messaging Protocol
    #: - [UDP] Common ASCII Messaging Protocol
    camp: 'AppType' = 4450, 'camp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CTI System Msg
    #: - [UDP] CTI System Msg
    ctisystemmsg: 'AppType' = 4451, 'ctisystemmsg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CTI Program Load
    #: - [UDP] CTI Program Load
    ctiprogramload: 'AppType' = 4452, 'ctiprogramload', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSS Alert Manager
    #: - [UDP] NSS Alert Manager
    nssalertmgr: 'AppType' = 4453, 'nssalertmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NSS Agent Manager
    #: - [UDP] NSS Agent Manager
    nssagentmgr: 'AppType' = 4454, 'nssagentmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PR Chat User
    #: - [UDP] PR Chat User
    prchat_user: 'AppType' = 4455, 'prchat-user', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PR Chat Server
    #: - [UDP] PR Chat Server
    prchat_server: 'AppType' = 4456, 'prchat-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PR Register
    #: - [UDP] PR Register
    prregister: 'AppType' = 4457, 'prregister', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Matrix Configuration Protocol
    #: - [UDP] Matrix Configuration Protocol
    mcp: 'AppType' = 4458, 'mcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4459: 'AppType' = 4459, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Network Time Security Key Establishment [:rfc:`8915`]
    ntske: 'AppType' = 4460, 'ntske', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4460: 'AppType' = 4460, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] hpssmgmt service
    #: - [UDP] hpssmgmt service
    hpssmgmt: 'AppType' = 4484, 'hpssmgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Assyst Data Repository Service
    assyst_dr: 'AppType' = 4485, 'assyst-dr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4485: 'AppType' = 4485, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Integrated Client Message Service
    #: - [UDP] Integrated Client Message Service
    icms: 'AppType' = 4486, 'icms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Protocol for Remote Execution over TCP
    prex_tcp: 'AppType' = 4487, 'prex-tcp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4487: 'AppType' = 4487, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Apple Wide Area Connectivity Service ICE Bootstrap
    #: - [UDP] Apple Wide Area Connectivity Service ICE Bootstrap
    awacs_ice: 'AppType' = 4488, 'awacs-ice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IPsec NAT-Traversal [:rfc:`9329`]
    #: - [UDP] IPsec NAT-Traversal [:rfc:`3948`][:rfc:`7296`]
    ipsec_nat_t: 'AppType' = 4500, 'ipsec-nat-t', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_4501: 'AppType' = 4501, 'unassigned', TransportProtocol.get('undefined')

    #: [SCTP] A25 (FAP-FGW)
    a25_fap_fgw: 'AppType' = 4502, 'a25-fap-fgw', TransportProtocol.get('sctp')

    #: [TCP] Reserved
    reserved_4534: 'AppType' = 4534, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Armagetron Advanced Game Server
    armagetronad: 'AppType' = 4534, 'armagetronad', TransportProtocol.get('udp')

    #: - [TCP] Event Heap Server
    #: - [UDP] Event Heap Server
    ehs: 'AppType' = 4535, 'ehs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Event Heap Server SSL
    #: - [UDP] Event Heap Server SSL
    ehs_ssl: 'AppType' = 4536, 'ehs-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WSS Security Service
    #: - [UDP] WSS Security Service
    wssauthsvc: 'AppType' = 4537, 'wssauthsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Software Data Exchange Gateway
    #: - [UDP] Software Data Exchange Gateway
    swx_gate: 'AppType' = 4538, 'swx-gate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WorldScores
    #: - [UDP] WorldScores
    worldscores: 'AppType' = 4545, 'worldscores', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SF License Manager (Sentinel)
    #: - [UDP] SF License Manager (Sentinel)
    sf_lm: 'AppType' = 4546, 'sf-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lanner License Manager
    #: - [UDP] Lanner License Manager
    lanner_lm: 'AppType' = 4547, 'lanner-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synchromesh
    #: - [UDP] Synchromesh
    synchromesh: 'AppType' = 4548, 'synchromesh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aegate PMR Service
    #: - [UDP] Aegate PMR Service
    aegate: 'AppType' = 4549, 'aegate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Perman I Interbase Server
    #: - [UDP] Perman I Interbase Server
    gds_adppiw_db: 'AppType' = 4550, 'gds-adppiw-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIH Services [:rfc:`5677`]
    #: - [UDP] MIH Services [:rfc:`5677`]
    ieee_mih: 'AppType' = 4551, 'ieee-mih', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Men and Mice Monitoring
    #: - [UDP] Men and Mice Monitoring
    menandmice_mon: 'AppType' = 4552, 'menandmice-mon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] ICS host services
    icshostsvc: 'AppType' = 4553, 'icshostsvc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4553: 'AppType' = 4553, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] MS FRS Replication
    #: - [UDP] MS FRS Replication
    msfrs: 'AppType' = 4554, 'msfrs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RSIP Port [:rfc:`3103`]
    #: - [UDP] RSIP Port [:rfc:`3103`]
    rsip: 'AppType' = 4555, 'rsip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DTN Bundle TCP CL Protocol [:rfc:`9174`]
    #: - [UDP] DTN Bundle UDP CL Protocol [:rfc:`7122`]
    #: - [DCCP] DTN Bundle DCCP CL Protocol [:rfc:`7122`]
    dtn_bundle: 'AppType' = 4556, 'dtn-bundle', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('dccp')

    #: [TCP] Reserved
    reserved_4557: 'AppType' = 4557, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Marathon everRun Quorum Service Server
    mtcevrunqss: 'AppType' = 4557, 'mtcevrunqss', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4558: 'AppType' = 4558, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Marathon everRun Quorum Service Manager
    mtcevrunqman: 'AppType' = 4558, 'mtcevrunqman', TransportProtocol.get('udp')

    #: - [TCP] HylaFAX
    #: - [UDP] HylaFAX
    hylafax: 'AppType' = 4559, 'hylafax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Amahi Anywhere
    amahi_anywhere: 'AppType' = 4563, 'amahi-anywhere', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4563: 'AppType' = 4563, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Kids Watch Time Control Service
    #: - [UDP] Kids Watch Time Control Service
    kwtc: 'AppType' = 4566, 'kwtc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRAM
    #: - [UDP] TRAM
    tram: 'AppType' = 4567, 'tram', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC Reporting
    #: - [UDP] BMC Reporting
    bmc_reporting: 'AppType' = 4568, 'bmc-reporting', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Inter-Asterisk eXchange [:rfc:`5456`]
    #: - [UDP] Inter-Asterisk eXchange [:rfc:`5456`]
    iax: 'AppType' = 4569, 'iax', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Service to distribute and update within a site deployment information
    #: for Oracle Communications Suite
    deploymentmap: 'AppType' = 4570, 'deploymentmap', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4570: 'AppType' = 4570, 'reserved', TransportProtocol.get('udp')

    #: [TCP] A port for communication between a server and client for a custom
    #: backup system
    cardifftec_back: 'AppType' = 4573, 'cardifftec-back', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4573: 'AppType' = 4573, 'reserved', TransportProtocol.get('udp')

    #: [TCP] RID over HTTP/TLS [:rfc:`6546`]
    rid: 'AppType' = 4590, 'rid', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4590: 'AppType' = 4590, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] HRPD L3T (AT-AN)
    #: - [UDP] HRPD L3T (AT-AN)
    l3t_at_an: 'AppType' = 4591, 'l3t-at-an', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4592: 'AppType' = 4592, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] HRPD-ITH (AT-AN)
    hrpd_ith_at_an: 'AppType' = 4592, 'hrpd-ith-at-an', TransportProtocol.get('udp')

    #: - [TCP] IPT (ANRI-ANRI)
    #: - [UDP] IPT (ANRI-ANRI)
    ipt_anri_anri: 'AppType' = 4593, 'ipt-anri-anri', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-Session (ANRI-ANRI)
    #: - [UDP] IAS-Session (ANRI-ANRI)
    ias_session: 'AppType' = 4594, 'ias-session', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-Paging (ANRI-ANRI)
    #: - [UDP] IAS-Paging (ANRI-ANRI)
    ias_paging: 'AppType' = 4595, 'ias-paging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IAS-Neighbor (ANRI-ANRI)
    #: - [UDP] IAS-Neighbor (ANRI-ANRI)
    ias_neighbor: 'AppType' = 4596, 'ias-neighbor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A21 (AN-1xBS)
    #: - [UDP] A21 (AN-1xBS)
    a21_an_1xbs: 'AppType' = 4597, 'a21-an-1xbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A16 (AN-AN)
    #: - [UDP] A16 (AN-AN)
    a16_an_an: 'AppType' = 4598, 'a16-an-an', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A17 (AN-AN)
    #: - [UDP] A17 (AN-AN)
    a17_an_an: 'AppType' = 4599, 'a17-an-an', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Piranha1
    #: - [UDP] Piranha1
    piranha1: 'AppType' = 4600, 'piranha1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Piranha2
    #: - [UDP] Piranha2
    piranha2: 'AppType' = 4601, 'piranha2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] EAX MTS Server
    mtsserver: 'AppType' = 4602, 'mtsserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4602: 'AppType' = 4602, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Men & Mice Upgrade Agent
    menandmice_upg: 'AppType' = 4603, 'menandmice-upg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4603: 'AppType' = 4603, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Identity Registration Protocol
    irp: 'AppType' = 4604, 'irp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4604: 'AppType' = 4604, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Direct End to End Secure Chat Protocol
    sixchat: 'AppType' = 4605, 'sixchat', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4605: 'AppType' = 4605, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Secure ID to IP registration and lookup
    sixid: 'AppType' = 4606, 'sixid', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4606: 'AppType' = 4606, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4621: 'AppType' = 4621, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Bidirectional single port remote radio VOIP and Control stream
    ventoso: 'AppType' = 4621, 'ventoso', TransportProtocol.get('udp')

    #: - [TCP] Distributed Denial-of-Service Open Threat Signaling (DOTS) Signal
    #:   Channel Protocol. The service name is used to construct the SRV service
    #:   names "_dots-signal._udp" and "_dots-signal._tcp" for discovering DOTS
    #:   servers used to establish DOTS signal channel. [:rfc:`8973`][:rfc:`9132`]
    #: - [UDP] Distributed Denial-of-Service Open Threat Signaling (DOTS) Signal
    #:   Channel Protocol. The service name is used to construct the SRV service
    #:   names "_dots-signal._udp" and "_dots-signal._tcp" for discovering DOTS
    #:   servers used to establish DOTS signal channel. [:rfc:`8973`][:rfc:`9132`]
    dots_signal: 'AppType' = 4646, 'dots-signal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PlayStation2 App Port
    #: - [UDP] PlayStation2 App Port
    playsta2_app: 'AppType' = 4658, 'playsta2-app', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PlayStation2 Lobby Port
    #: - [UDP] PlayStation2 Lobby Port
    playsta2_lob: 'AppType' = 4659, 'playsta2-lob', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] smaclmgr
    #: - [UDP] smaclmgr
    smaclmgr: 'AppType' = 4660, 'smaclmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kar2ouche Peer location service
    #: - [UDP] Kar2ouche Peer location service
    kar2ouche: 'AppType' = 4661, 'kar2ouche', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OrbitNet Message Service
    #: - [UDP] OrbitNet Message Service
    oms: 'AppType' = 4662, 'oms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Note It! Message Service
    #: - [UDP] Note It! Message Service
    noteit: 'AppType' = 4663, 'noteit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Rimage Messaging Server
    #: - [UDP] Rimage Messaging Server
    ems: 'AppType' = 4664, 'ems', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Container Client Message Service
    #: - [UDP] Container Client Message Service
    contclientms: 'AppType' = 4665, 'contclientms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] E-Port Message Service
    #: - [UDP] E-Port Message Service
    eportcomm: 'AppType' = 4666, 'eportcomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MMA Comm Services
    #: - [UDP] MMA Comm Services
    mmacomm: 'AppType' = 4667, 'mmacomm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MMA EDS Service
    #: - [UDP] MMA EDS Service
    mmaeds: 'AppType' = 4668, 'mmaeds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] E-Port Data Service
    #: - [UDP] E-Port Data Service
    eportcommdata: 'AppType' = 4669, 'eportcommdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Light packets transfer protocol
    #: - [UDP] Light packets transfer protocol
    light: 'AppType' = 4670, 'light', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bull RSF action server
    #: - [UDP] Bull RSF action server
    acter: 'AppType' = 4671, 'acter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] remote file access server
    #: - [UDP] remote file access server
    rfa: 'AppType' = 4672, 'rfa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CXWS Operations
    #: - [UDP] CXWS Operations
    cxws: 'AppType' = 4673, 'cxws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AppIQ Agent Management
    #: - [UDP] AppIQ Agent Management
    appiq_mgmt: 'AppType' = 4674, 'appiq-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BIAP Device Status
    #: - [UDP] BIAP Device Status
    dhct_status: 'AppType' = 4675, 'dhct-status', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BIAP Generic Alert
    #: - [UDP] BIAP Generic Alert
    dhct_alerts: 'AppType' = 4676, 'dhct-alerts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Business Continuity Servi
    #: - [UDP] Business Continuity Servi
    bcs: 'AppType' = 4677, 'bcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] boundary traversal
    #: - [UDP] boundary traversal
    traversal: 'AppType' = 4678, 'traversal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MGE UPS Supervision
    #: - [UDP] MGE UPS Supervision
    mgesupervision: 'AppType' = 4679, 'mgesupervision', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MGE UPS Management
    #: - [UDP] MGE UPS Management
    mgemanagement: 'AppType' = 4680, 'mgemanagement', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Parliant Telephony System
    #: - [UDP] Parliant Telephony System
    parliant: 'AppType' = 4681, 'parliant', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] finisar
    #: - [UDP] finisar
    finisar: 'AppType' = 4682, 'finisar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Spike Clipboard Service
    #: - [UDP] Spike Clipboard Service
    spike: 'AppType' = 4683, 'spike', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RFID Reader Protocol 1.0
    #: - [UDP] RFID Reader Protocol 1.0
    rfid_rp1: 'AppType' = 4684, 'rfid-rp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Autopac Protocol
    #: - [UDP] Autopac Protocol
    autopac: 'AppType' = 4685, 'autopac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Manina Service Protocol
    #: - [UDP] Manina Service Protocol
    msp_os: 'AppType' = 4686, 'msp-os', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Scanner Tool FTP
    #: - [UDP] Network Scanner Tool FTP
    nst: 'AppType' = 4687, 'nst', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mobile P2P Service
    #: - [UDP] Mobile P2P Service
    mobile_p2p: 'AppType' = 4688, 'mobile-p2p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Altova DatabaseCentral
    #: - [UDP] Altova DatabaseCentral
    altovacentral: 'AppType' = 4689, 'altovacentral', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prelude IDS message proto
    #: - [UDP] Prelude IDS message proto
    prelude: 'AppType' = 4690, 'prelude', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] monotone Netsync Protocol
    #: - [UDP] monotone Netsync Protocol
    mtn: 'AppType' = 4691, 'mtn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Conspiracy messaging
    #: - [UDP] Conspiracy messaging
    conspiracy: 'AppType' = 4692, 'conspiracy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetXMS Agent
    #: - [UDP] NetXMS Agent
    netxms_agent: 'AppType' = 4700, 'netxms-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetXMS Management
    #: - [UDP] NetXMS Management
    netxms_mgmt: 'AppType' = 4701, 'netxms-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetXMS Server Synchronization
    #: - [UDP] NetXMS Server Synchronization
    netxms_sync: 'AppType' = 4702, 'netxms-sync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Network Performance Quality Evaluation System Test Service
    npqes_test: 'AppType' = 4703, 'npqes-test', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4703: 'AppType' = 4703, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Assuria Insider
    assuria_ins: 'AppType' = 4704, 'assuria-ins', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4704: 'AppType' = 4704, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Trinity Trust Network Node Communication
    #: - [UDP] Trinity Trust Network Node Communication
    #: - [SCTP] Trinity Trust Network Node Communication
    trinity_dist: 'AppType' = 4711, 'trinity-dist', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] TruckStar Service
    #: - [UDP] TruckStar Service
    truckstar: 'AppType' = 4725, 'truckstar', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4726: 'AppType' = 4726, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] A26 (FAP-FGW)
    a26_fap_fgw: 'AppType' = 4726, 'a26-fap-fgw', TransportProtocol.get('udp')

    #: [TCP] F-Link Client Information Service
    fcis: 'AppType' = 4727, 'fcis', TransportProtocol.get('tcp')

    #: [UDP] F-Link Client Information Service Discovery
    fcis_disc: 'AppType' = 4727, 'fcis-disc', TransportProtocol.get('udp')

    #: - [TCP] CA Port Multiplexer
    #: - [UDP] CA Port Multiplexer
    capmux: 'AppType' = 4728, 'capmux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4729: 'AppType' = 4729, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] GSM Interface Tap
    gsmtap: 'AppType' = 4729, 'gsmtap', TransportProtocol.get('udp')

    #: - [TCP] Gearman Job Queue System
    #: - [UDP] Gearman Job Queue System
    gearman: 'AppType' = 4730, 'gearman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Remote Capture Protocol
    remcap: 'AppType' = 4731, 'remcap', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4731: 'AppType' = 4731, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4732: 'AppType' = 4732, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] OHM server trigger
    ohmtrigger: 'AppType' = 4732, 'ohmtrigger', TransportProtocol.get('udp')

    #: [TCP] RES Orchestration Catalog Services
    resorcs: 'AppType' = 4733, 'resorcs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4733: 'AppType' = 4733, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] IPDR/SP
    #: - [UDP] IPDR/SP
    ipdr_sp: 'AppType' = 4737, 'ipdr-sp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SoleraTec Locator
    #: - [UDP] SoleraTec Locator
    solera_lpn: 'AppType' = 4738, 'solera-lpn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IP Flow Info Export
    #: - [UDP] IP Flow Info Export
    #: - [SCTP] IP Flow Info Export
    ipfix: 'AppType' = 4739, 'ipfix', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] ipfix protocol over TLS
    #: - [SCTP] ipfix protocol over DTLS
    #: - [UDP] ipfix protocol over DTLS
    ipfixs: 'AppType' = 4740, 'ipfixs', TransportProtocol.get('tcp') | TransportProtocol.get('sctp') | TransportProtocol.get('udp')

    #: - [TCP] Luminizer Manager
    #: - [UDP] Luminizer Manager
    lumimgrd: 'AppType' = 4741, 'lumimgrd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SICCT
    sicct: 'AppType' = 4742, 'sicct', TransportProtocol.get('tcp')

    #: [UDP] SICCT Service Discovery Protocol
    sicct_sdp: 'AppType' = 4742, 'sicct-sdp', TransportProtocol.get('udp')

    #: - [TCP] openhpi HPI service
    #: - [UDP] openhpi HPI service
    openhpid: 'AppType' = 4743, 'openhpid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Internet File Synchronization Protocol
    #: - [UDP] Internet File Synchronization Protocol
    ifsp: 'AppType' = 4744, 'ifsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Funambol Mobile Push
    #: - [UDP] Funambol Mobile Push
    fmp: 'AppType' = 4745, 'fmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4746: 'AppType' = 4746, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] IntelliAdmin Discovery
    intelliadm_disc: 'AppType' = 4746, 'intelliadm-disc', TransportProtocol.get('udp')

    #: [UDP] peer-to-peer file exchange protocol
    buschtrommel: 'AppType' = 4747, 'buschtrommel', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4747: 'AppType' = 4747, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] Simple Service Auto Discovery
    #: - [UDP] Simple Service Auto Discovery
    ssad: 'AppType' = 4750, 'ssad', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Policy Control Protocol
    #: - [UDP] Simple Policy Control Protocol
    spocp: 'AppType' = 4751, 'spocp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simple Network Audio Protocol
    #: - [UDP] Simple Network Audio Protocol
    snap: 'AppType' = 4752, 'snap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Simple Invocation of Methods Over Network (SIMON)
    simon: 'AppType' = 4753, 'simon', TransportProtocol.get('tcp')

    #: [UDP] Simple Invocation of Methods Over Network (SIMON) Discovery
    simon_disc: 'AppType' = 4753, 'simon-disc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4754: 'AppType' = 4754, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] GRE-in-UDP Encapsulation [:rfc:`8086`]
    gre_in_udp: 'AppType' = 4754, 'gre-in-udp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4755: 'AppType' = 4755, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] GRE-in-UDP Encapsulation with DTLS [:rfc:`8086`]
    gre_udp_dtls: 'AppType' = 4755, 'gre-udp-dtls', TransportProtocol.get('udp')

    #: [TCP] Reticle Decision Center
    rdcenter: 'AppType' = 4756, 'rdcenter', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4756: 'AppType' = 4756, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Converge RPC
    converge: 'AppType' = 4774, 'converge', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4774: 'AppType' = 4774, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] BFD Multihop Control
    #: - [UDP] BFD Multihop Control
    bfd_multi_ctl: 'AppType' = 4784, 'bfd-multi-ctl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4785: 'AppType' = 4785, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cisco Nexus Control Protocol
    cncp: 'AppType' = 4785, 'cncp', TransportProtocol.get('udp')

    #: [TCP] Smart Install Service
    smart_install: 'AppType' = 4786, 'smart-install', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4786: 'AppType' = 4786, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Service Insertion Architecture (SIA) Control-Plane
    sia_ctrl_plane: 'AppType' = 4787, 'sia-ctrl-plane', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4787: 'AppType' = 4787, 'reserved', TransportProtocol.get('udp')

    #: [TCP] eXtensible Messaging Client Protocol
    xmcp: 'AppType' = 4788, 'xmcp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4788: 'AppType' = 4788, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Virtual eXtensible Local Area Network (VXLAN) [:rfc:`7348`]
    vxlan: 'AppType' = 4789, 'vxlan', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4789: 'AppType' = 4789, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Generic Protocol Extension for Virtual eXtensible Local Area Network
    #: (VXLAN)
    vxlan_gpe: 'AppType' = 4790, 'vxlan-gpe', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4790: 'AppType' = 4790, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] IP Routable RocE
    roce: 'AppType' = 4791, 'roce', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4791: 'AppType' = 4791, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] IP Routable Unified Bus
    #: - [UDP] IP Routable Unified Bus
    unified_bus: 'AppType' = 4792, 'unified-bus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Icona Instant Messenging System
    #: - [UDP] Icona Instant Messenging System
    iims: 'AppType' = 4800, 'iims', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Icona Web Embedded Chat
    #: - [UDP] Icona Web Embedded Chat
    iwec: 'AppType' = 4801, 'iwec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Icona License System Server
    #: - [UDP] Icona License System Server
    ilss: 'AppType' = 4802, 'ilss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Notateit Messaging
    notateit: 'AppType' = 4803, 'notateit', TransportProtocol.get('tcp')

    #: [UDP] Notateit Messaging Discovery
    notateit_disc: 'AppType' = 4803, 'notateit-disc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4804: 'AppType' = 4804, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] AJA ntv4 Video System Discovery
    aja_ntv4_disc: 'AppType' = 4804, 'aja-ntv4-disc', TransportProtocol.get('udp')

    #: - [TCP] HTCP
    #: - [UDP] HTCP
    htcp: 'AppType' = 4827, 'htcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Varadero-0
    #: - [UDP] Varadero-0
    varadero_0: 'AppType' = 4837, 'varadero-0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Varadero-1
    #: - [UDP] Varadero-1
    varadero_1: 'AppType' = 4838, 'varadero-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Varadero-2
    #: - [UDP] Varadero-2
    varadero_2: 'AppType' = 4839, 'varadero-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] OPC UA Connection Protocol
    opcua_tcp: 'AppType' = 4840, 'opcua-tcp', TransportProtocol.get('tcp')

    #: [UDP] OPC UA Multicast Datagram Protocol
    opcua_udp: 'AppType' = 4840, 'opcua-udp', TransportProtocol.get('udp')

    #: - [TCP] QUOSA Virtual Library Service
    #: - [UDP] QUOSA Virtual Library Service
    quosa: 'AppType' = 4841, 'quosa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nCode ICE-flow Library AppServer
    #: - [UDP] nCode ICE-flow Library AppServer
    gw_asv: 'AppType' = 4842, 'gw-asv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OPC UA TCP Protocol over TLS/SSL
    #: - [UDP] OPC UA TCP Protocol over TLS/SSL
    opcua_tls: 'AppType' = 4843, 'opcua-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nCode ICE-flow Library LogServer
    #: - [UDP] nCode ICE-flow Library LogServer
    gw_log: 'AppType' = 4844, 'gw-log', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WordCruncher Remote Library Service
    #: - [UDP] WordCruncher Remote Library Service
    wcr_remlib: 'AppType' = 4845, 'wcr-remlib', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Contamac ICM Service IANA assigned this well-formed service name as a
    #:   replacement for "contamac_icm".
    #: - [TCP] Contamac ICM Service
    #: - [UDP] Contamac ICM Service IANA assigned this well-formed service name as a
    #:   replacement for "contamac_icm".
    #: - [UDP] Contamac ICM Service
    contamac_icm: 'AppType' = 4846, 'contamac-icm', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Web Fresh Communication
    #: - [UDP] Web Fresh Communication
    wfc: 'AppType' = 4847, 'wfc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] App Server - Admin HTTP
    #: - [UDP] App Server - Admin HTTP
    appserv_http: 'AppType' = 4848, 'appserv-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] App Server - Admin HTTPS
    #: - [UDP] App Server - Admin HTTPS
    appserv_https: 'AppType' = 4849, 'appserv-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun App Server - NA
    #: - [UDP] Sun App Server - NA
    sun_as_nodeagt: 'AppType' = 4850, 'sun-as-nodeagt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apache Derby Replication
    #: - [UDP] Apache Derby Replication
    derby_repli: 'AppType' = 4851, 'derby-repli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Unify Debugger
    #: - [UDP] Unify Debugger
    unify_debug: 'AppType' = 4867, 'unify-debug', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Photon Relay
    #: - [UDP] Photon Relay
    phrelay: 'AppType' = 4868, 'phrelay', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Photon Relay Debug
    #: - [UDP] Photon Relay Debug
    phrelaydbg: 'AppType' = 4869, 'phrelaydbg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citcom Tracking Service
    #: - [UDP] Citcom Tracking Service
    cc_tracking: 'AppType' = 4870, 'cc-tracking', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wired
    #: - [UDP] Wired
    wired: 'AppType' = 4871, 'wired', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tritium CAN Bus Bridge Service
    #: - [UDP] Tritium CAN Bus Bridge Service
    tritium_can: 'AppType' = 4876, 'tritium-can', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lighting Management Control System
    #: - [UDP] Lighting Management Control System
    lmcs: 'AppType' = 4877, 'lmcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4878: 'AppType' = 4878, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Agilent Instrument Discovery
    inst_discovery: 'AppType' = 4878, 'inst-discovery', TransportProtocol.get('udp')

    #: [TCP] WSDL Event Receiver
    wsdl_event: 'AppType' = 4879, 'wsdl-event', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4879: 'AppType' = 4879, 'reserved', TransportProtocol.get('udp')

    #: [TCP] IVI High-Speed LAN Instrument Protocol
    hislip: 'AppType' = 4880, 'hislip', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4880: 'AppType' = 4880, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4881: 'AppType' = 4881, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] SOCP Time Synchronization Protocol
    socp_t: 'AppType' = 4881, 'socp-t', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4882: 'AppType' = 4882, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] SOCP Control Protocol
    socp_c: 'AppType' = 4882, 'socp-c', TransportProtocol.get('udp')

    #: [TCP] Meier-Phelps License Server
    wmlserver: 'AppType' = 4883, 'wmlserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4883: 'AppType' = 4883, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] HiveStor Distributed File System
    #: - [UDP] HiveStor Distributed File System
    hivestor: 'AppType' = 4884, 'hivestor', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ABBS
    #: - [UDP] ABBS
    abbs: 'AppType' = 4885, 'abbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] xcap code analysis portal public user access
    xcap_portal: 'AppType' = 4888, 'xcap-portal', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4888: 'AppType' = 4888, 'reserved', TransportProtocol.get('udp')

    #: [TCP] xcap code analysis portal cluster control and administration
    xcap_control: 'AppType' = 4889, 'xcap-control', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4889: 'AppType' = 4889, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] LysKOM Protocol A
    #: - [UDP] LysKOM Protocol A
    lyskom: 'AppType' = 4894, 'lyskom', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RAdmin Port
    #: - [UDP] RAdmin Port
    radmin_port: 'AppType' = 4899, 'radmin-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HFSQL Client/Server Database Engine
    #: - [UDP] HFSQL Client/Server Database Engine
    hfcs: 'AppType' = 4900, 'hfcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileLocator Remote Search Agent IANA assigned this well-formed service
    #:   name as a replacement for "flr_agent".
    #: - [TCP] FileLocator Remote Search Agent
    flr_agent: 'AppType' = 4901, 'flr-agent', TransportProtocol.get('tcp') | TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4901: 'AppType' = 4901, 'reserved', TransportProtocol.get('udp')

    #: [TCP] magicCONROL RF and Data Interface
    magiccontrol: 'AppType' = 4902, 'magiccontrol', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4902: 'AppType' = 4902, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Technicolor LUT Access Protocol
    lutap: 'AppType' = 4912, 'lutap', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4912: 'AppType' = 4912, 'reserved', TransportProtocol.get('udp')

    #: [TCP] LUTher Control Protocol
    lutcp: 'AppType' = 4913, 'lutcp', TransportProtocol.get('tcp')

    #: - [TCP] Bones Remote Control
    #: - [UDP] Bones Remote Control
    bones: 'AppType' = 4914, 'bones', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Fibics Remote Control Service
    frcs: 'AppType' = 4915, 'frcs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4915: 'AppType' = 4915, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Signal protocol port for autonomic networking
    an_signaling: 'AppType' = 4936, 'an-signaling', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4936: 'AppType' = 4936, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] Reserved
    reserved_4937: 'AppType' = 4937, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] ATSC-M/H Service Signaling Channel
    atsc_mh_ssc: 'AppType' = 4937, 'atsc-mh-ssc', TransportProtocol.get('udp')

    #: - [TCP] Equitrac Office
    #: - [UDP] Equitrac Office
    eq_office_4940: 'AppType' = 4940, 'eq-office-4940', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Equitrac Office
    #: - [UDP] Equitrac Office
    eq_office_4941: 'AppType' = 4941, 'eq-office-4941', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Equitrac Office
    #: - [UDP] Equitrac Office
    eq_office_4942: 'AppType' = 4942, 'eq-office-4942', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Munin Graphing Framework
    #: - [UDP] Munin Graphing Framework
    munin: 'AppType' = 4949, 'munin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sybase Server Monitor
    #: - [UDP] Sybase Server Monitor
    sybasesrvmon: 'AppType' = 4950, 'sybasesrvmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PWG WIMS
    #: - [UDP] PWG WIMS
    pwgwims: 'AppType' = 4951, 'pwgwims', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SAG Directory Server
    #: - [UDP] SAG Directory Server
    sagxtsds: 'AppType' = 4952, 'sagxtsds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Synchronization Arbiter
    dbsyncarbiter: 'AppType' = 4953, 'dbsyncarbiter', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4953: 'AppType' = 4953, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] CCSS QMessageMonitor
    #: - [UDP] CCSS QMessageMonitor
    ccss_qmm: 'AppType' = 4969, 'ccss-qmm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CCSS QSystemMonitor
    #: - [UDP] CCSS QSystemMonitor
    ccss_qsm: 'AppType' = 4970, 'ccss-qsm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] BackUp and Restore Program
    burp: 'AppType' = 4971, 'burp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4971: 'AppType' = 4971, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Citrix Virtual Path
    ctxs_vpp: 'AppType' = 4980, 'ctxs-vpp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_4980: 'AppType' = 4980, 'reserved', TransportProtocol.get('tcp')

    #: [N/A] Unassigned
    unassigned_4983: 'AppType' = 4983, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] WebYast
    webyast: 'AppType' = 4984, 'webyast', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4984: 'AppType' = 4984, 'reserved', TransportProtocol.get('udp')

    #: [TCP] GER HC Standard
    gerhcs: 'AppType' = 4985, 'gerhcs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_4985: 'AppType' = 4985, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Model Railway Interface Program
    #: - [UDP] Model Railway Interface Program
    mrip: 'AppType' = 4986, 'mrip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMAR Ethernet Port 1
    #: - [UDP] SMAR Ethernet Port 1
    smar_se_port1: 'AppType' = 4987, 'smar-se-port1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMAR Ethernet Port 2
    #: - [UDP] SMAR Ethernet Port 2
    smar_se_port2: 'AppType' = 4988, 'smar-se-port2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Parallel for GAUSS (tm)
    #: - [UDP] Parallel for GAUSS (tm)
    parallel: 'AppType' = 4989, 'parallel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BusySync Calendar Synch. Protocol
    #: - [UDP] BusySync Calendar Synch. Protocol
    busycal: 'AppType' = 4990, 'busycal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VITA Radio Transport
    #: - [UDP] VITA Radio Transport
    vrt: 'AppType' = 4991, 'vrt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HFSQL Client/Server Database Engine Manager
    #: - [UDP] HFSQL Client/Server Database Engine Manager
    hfcs_manager: 'AppType' = 4999, 'hfcs-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    commplex_main: 'AppType' = 5000, 'commplex-main', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    commplex_link: 'AppType' = 5001, 'commplex-link', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] radio free ethernet
    #: - [UDP] radio free ethernet
    rfe: 'AppType' = 5002, 'rfe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileMaker, Inc. - Proprietary transport
    #: - [UDP] FileMaker, Inc. - Proprietary name binding
    fmpro_internal: 'AppType' = 5003, 'fmpro-internal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTP media data [:rfc:`3551`][:rfc:`4571`]
    #: - [UDP] RTP media data [:rfc:`3551`]
    #: - [DCCP] RTP media data [:rfc:`3551`][:rfc:`5762`]
    avt_profile_1: 'AppType' = 5004, 'avt-profile-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('dccp')

    #: - [TCP] RTP control protocol [:rfc:`3551`][:rfc:`4571`]
    #: - [UDP] RTP control protocol [:rfc:`3551`]
    #: - [DCCP] RTP control protocol [:rfc:`3551`][:rfc:`5762`]
    avt_profile_2: 'AppType' = 5005, 'avt-profile-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('dccp')

    #: - [TCP] wsm server
    #: - [UDP] wsm server
    wsm_server: 'AppType' = 5006, 'wsm-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] wsm server ssl
    #: - [UDP] wsm server ssl
    wsm_server_ssl: 'AppType' = 5007, 'wsm-server-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synapsis EDGE
    #: - [UDP] Synapsis EDGE
    synapsis_edge: 'AppType' = 5008, 'synapsis-edge', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Windows Filesystem
    #: - [UDP] Microsoft Windows Filesystem
    winfs: 'AppType' = 5009, 'winfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TelepathStart
    #: - [UDP] TelepathStart
    telelpathstart: 'AppType' = 5010, 'telelpathstart', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TelepathAttack
    #: - [UDP] TelepathAttack
    telelpathattack: 'AppType' = 5011, 'telelpathattack', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetOnTap Service
    #: - [UDP] NetOnTap Service
    nsp: 'AppType' = 5012, 'nsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileMaker, Inc. - Proprietary transport
    #: - [UDP] FileMaker, Inc. - Proprietary transport
    fmpro_v6: 'AppType' = 5013, 'fmpro-v6', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5014: 'AppType' = 5014, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Overlay Network Protocol
    onpsocket: 'AppType' = 5014, 'onpsocket', TransportProtocol.get('udp')

    #: [TCP] FileMaker, Inc. - Web publishing
    fmwp: 'AppType' = 5015, 'fmwp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5015: 'AppType' = 5015, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] zenginkyo-1
    #: - [UDP] zenginkyo-1
    zenginkyo_1: 'AppType' = 5020, 'zenginkyo-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] zenginkyo-2
    #: - [UDP] zenginkyo-2
    zenginkyo_2: 'AppType' = 5021, 'zenginkyo-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mice server
    #: - [UDP] mice server
    mice: 'AppType' = 5022, 'mice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Htuil Server for PLD2
    #: - [UDP] Htuil Server for PLD2
    htuilsrv: 'AppType' = 5023, 'htuilsrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Storix I/O daemon (data)
    #: - [UDP] Storix I/O daemon (data)
    strexec_d: 'AppType' = 5026, 'strexec-d', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Storix I/O daemon (stat)
    #: - [UDP] Storix I/O daemon (stat)
    strexec_s: 'AppType' = 5027, 'strexec-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Quiqum Virtual Relais
    qvr: 'AppType' = 5028, 'qvr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5028: 'AppType' = 5028, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Infobright Database Server
    #: - [UDP] Infobright Database Server
    infobright: 'AppType' = 5029, 'infobright', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_5030: 'AppType' = 5030, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5031: 'AppType' = 5031, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Direct Message Protocol
    dmp: 'AppType' = 5031, 'dmp', TransportProtocol.get('udp')

    #: [TCP] SignaCert Enterprise Trust Server Agent
    signacert_agent: 'AppType' = 5032, 'signacert-agent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5032: 'AppType' = 5032, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Janstor Secure Data
    jtnetd_server: 'AppType' = 5033, 'jtnetd-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5033: 'AppType' = 5033, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Janstor Status
    jtnetd_status: 'AppType' = 5034, 'jtnetd-status', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5034: 'AppType' = 5034, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] asnaacceler8db
    #: - [UDP] asnaacceler8db
    asnaacceler8db: 'AppType' = 5042, 'asnaacceler8db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ShopWorX Administration
    #: - [UDP] ShopWorX Administration
    swxadmin: 'AppType' = 5043, 'swxadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LXI Event Service
    #: - [UDP] LXI Event Service
    lxi_evntsvc: 'AppType' = 5044, 'lxi-evntsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Open Settlement Protocol
    osp: 'AppType' = 5045, 'osp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5045: 'AppType' = 5045, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5046: 'AppType' = 5046, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Vishay PM UDP Service
    vpm_udp: 'AppType' = 5046, 'vpm-udp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5047: 'AppType' = 5047, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] iSCAPE Data Broadcasting
    iscape: 'AppType' = 5047, 'iscape', TransportProtocol.get('udp')

    #: [TCP] Texai Message Service
    texai: 'AppType' = 5048, 'texai', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5048: 'AppType' = 5048, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] iVocalize Web Conference
    #: - [UDP] iVocalize Web Conference
    ivocalize: 'AppType' = 5049, 'ivocalize', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] multimedia conference control tool
    #: - [UDP] multimedia conference control tool
    mmcc: 'AppType' = 5050, 'mmcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITA Agent
    #: - [UDP] ITA Agent
    ita_agent: 'AppType' = 5051, 'ita-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITA Manager
    #: - [UDP] ITA Manager
    ita_manager: 'AppType' = 5052, 'ita-manager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] RLM License Server
    rlm: 'AppType' = 5053, 'rlm', TransportProtocol.get('tcp')

    #: [UDP] RLM Discovery Server
    rlm_disc: 'AppType' = 5053, 'rlm-disc', TransportProtocol.get('udp')

    #: [TCP] RLM administrative interface
    rlm_admin: 'AppType' = 5054, 'rlm-admin', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5054: 'AppType' = 5054, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] UNOT
    #: - [UDP] UNOT
    unot: 'AppType' = 5055, 'unot', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intecom Pointspan 1
    #: - [UDP] Intecom Pointspan 1
    intecom_ps1: 'AppType' = 5056, 'intecom-ps1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intecom Pointspan 2
    #: - [UDP] Intecom Pointspan 2
    intecom_ps2: 'AppType' = 5057, 'intecom-ps2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5058: 'AppType' = 5058, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Locus Discovery
    locus_disc: 'AppType' = 5058, 'locus-disc', TransportProtocol.get('udp')

    #: - [TCP] SIP Directory Services
    #: - [UDP] SIP Directory Services
    sds: 'AppType' = 5059, 'sds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SIP [:rfc:`3263`]
    #: - [UDP] SIP [:rfc:`3263`]
    #: - [SCTP] SIP [:rfc:`4168`]
    sip: 'AppType' = 5060, 'sip', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] SIP-TLS [:rfc:`3263`]
    #: - [UDP] SIP-TLS [:rfc:`3263`]
    #: - [SCTP] SIP-TLS [:rfc:`4168`]
    sips: 'AppType' = 5061, 'sips', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Localisation access
    #: - [UDP] Localisation access
    na_localise: 'AppType' = 5062, 'na-localise', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] centrify secure RPC
    csrpc: 'AppType' = 5063, 'csrpc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5063: 'AppType' = 5063, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Channel Access 1
    #: - [UDP] Channel Access 1
    ca_1: 'AppType' = 5064, 'ca-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Channel Access 2
    #: - [UDP] Channel Access 2
    ca_2: 'AppType' = 5065, 'ca-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STANAG-5066-SUBNET-INTF
    #: - [UDP] STANAG-5066-SUBNET-INTF
    stanag_5066: 'AppType' = 5066, 'stanag-5066', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Authentx Service
    #: - [UDP] Authentx Service
    authentx: 'AppType' = 5067, 'authentx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Bitforest Data Service
    bitforestsrv: 'AppType' = 5068, 'bitforestsrv', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5068: 'AppType' = 5068, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] I/Net 2000-NPR
    #: - [UDP] I/Net 2000-NPR
    i_net_2000_npr: 'AppType' = 5069, 'i-net-2000-npr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VersaTrans Server Agent Service
    #: - [UDP] VersaTrans Server Agent Service
    vtsas: 'AppType' = 5070, 'vtsas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PowerSchool
    #: - [UDP] PowerSchool
    powerschool: 'AppType' = 5071, 'powerschool', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Anything In Anything
    #: - [UDP] Anything In Anything
    ayiya: 'AppType' = 5072, 'ayiya', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Advantage Group Port Mgr
    #: - [UDP] Advantage Group Port Mgr
    tag_pm: 'AppType' = 5073, 'tag-pm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ALES Query
    #: - [UDP] ALES Query
    alesquery: 'AppType' = 5074, 'alesquery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Reserved
    reserved_5075: 'AppType' = 5075, 'reserved', TransportProtocol.get('udp')

    #: [UDP] PixelPusher pixel data
    pixelpusher: 'AppType' = 5078, 'pixelpusher', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5078: 'AppType' = 5078, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] Reserved
    reserved_5079: 'AppType' = 5079, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cambridge Pixel SPx Reports
    cp_spxrpts: 'AppType' = 5079, 'cp-spxrpts', TransportProtocol.get('udp')

    #: - [TCP] OnScreen Data Collection Service
    #: - [UDP] OnScreen Data Collection Service
    onscreen: 'AppType' = 5080, 'onscreen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SDL - Ent Trans Server
    #: - [UDP] SDL - Ent Trans Server
    sdl_ets: 'AppType' = 5081, 'sdl-ets', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Qpur Communication Protocol
    #: - [UDP] Qpur Communication Protocol
    qcp: 'AppType' = 5082, 'qcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Qpur File Protocol
    #: - [UDP] Qpur File Protocol
    qfp: 'AppType' = 5083, 'qfp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPCglobal Low-Level Reader Protocol
    #: - [UDP] EPCglobal Low-Level Reader Protocol
    llrp: 'AppType' = 5084, 'llrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EPCglobal Encrypted LLRP
    #: - [UDP] EPCglobal Encrypted LLRP
    encrypted_llrp: 'AppType' = 5085, 'encrypted-llrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Aprigo Collection Service
    aprigo_cs: 'AppType' = 5086, 'aprigo-cs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5086: 'AppType' = 5086, 'reserved', TransportProtocol.get('udp')

    #: [TCP] BIOTIC - Binary Internet of Things Interoperable Communication
    biotic: 'AppType' = 5087, 'biotic', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5087: 'AppType' = 5087, 'reserved', TransportProtocol.get('udp')

    #: [SCTP] Candidate AR
    car: 'AppType' = 5090, 'car', TransportProtocol.get('sctp')

    #: [SCTP] Context Transfer Protocol [:rfc:`4065`]
    cxtp: 'AppType' = 5091, 'cxtp', TransportProtocol.get('sctp')

    #: [TCP] Reserved
    reserved_5092: 'AppType' = 5092, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Magpie Binary
    magpie: 'AppType' = 5092, 'magpie', TransportProtocol.get('udp')

    #: - [TCP] Sentinel LM
    #: - [UDP] Sentinel LM
    sentinel_lm: 'AppType' = 5093, 'sentinel-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HART-IP
    #: - [UDP] HART-IP
    hart_ip: 'AppType' = 5094, 'hart-ip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SentLM Srv2Srv
    #: - [UDP] SentLM Srv2Srv
    sentlm_srv2srv: 'AppType' = 5099, 'sentlm-srv2srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Socalia service mux
    #: - [UDP] Socalia service mux
    socalia: 'AppType' = 5100, 'socalia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Talarian_TCP
    talarian_tcp: 'AppType' = 5101, 'talarian-tcp', TransportProtocol.get('tcp')

    #: [UDP] Talarian_UDP
    talarian_udp: 'AppType' = 5101, 'talarian-udp', TransportProtocol.get('udp')

    #: - [TCP] Oracle OMS non-secure
    #: - [UDP] Oracle OMS non-secure
    oms_nonsecure: 'AppType' = 5102, 'oms-nonsecure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Actifio C2C
    actifio_c2c: 'AppType' = 5103, 'actifio-c2c', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5103: 'AppType' = 5103, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5104: 'AppType' = 5104, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] TinyMessage
    tinymessage: 'AppType' = 5104, 'tinymessage', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5105: 'AppType' = 5105, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Hughes Association Protocol
    hughes_ap: 'AppType' = 5105, 'hughes-ap', TransportProtocol.get('udp')

    #: [TCP] Actifio UDS Agent
    actifioudsagent: 'AppType' = 5106, 'actifioudsagent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5106: 'AppType' = 5106, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Disk to Disk replication between Actifio Clusters
    actifioreplic: 'AppType' = 5107, 'actifioreplic', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5107: 'AppType' = 5107, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] TAEP AS service
    #: - [UDP] TAEP AS service
    taep_as_svc: 'AppType' = 5111, 'taep-as-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PeerMe Msg Cmd Service
    #: - [UDP] PeerMe Msg Cmd Service
    pm_cmdsvr: 'AppType' = 5112, 'pm-cmdsvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5113: 'AppType' = 5113, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Enterprise Vault Services
    ev_services: 'AppType' = 5114, 'ev-services', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5114: 'AppType' = 5114, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Symantec Autobuild Service
    autobuild: 'AppType' = 5115, 'autobuild', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5115: 'AppType' = 5115, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5116: 'AppType' = 5116, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] EPSON Projecter Image Transfer
    emb_proj_cmd: 'AppType' = 5116, 'emb-proj-cmd', TransportProtocol.get('udp')

    #: [TCP] GradeCam Image Processing
    gradecam: 'AppType' = 5117, 'gradecam', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5117: 'AppType' = 5117, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Barracuda Backup Protocol
    #: - [UDP] Barracuda Backup Protocol
    barracuda_bbs: 'AppType' = 5120, 'barracuda-bbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Policy Commander
    #: - [UDP] Policy Commander
    nbt_pc: 'AppType' = 5133, 'nbt-pc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PP ActivationServer
    ppactivation: 'AppType' = 5134, 'ppactivation', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5134: 'AppType' = 5134, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ERP-Scale
    erp_scale: 'AppType' = 5135, 'erp-scale', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5135: 'AppType' = 5135, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5136: 'AppType' = 5136, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Minotaur SA
    minotaur_sa: 'AppType' = 5136, 'minotaur-sa', TransportProtocol.get('udp')

    #: - [TCP] MyCTS server port
    #: - [UDP] MyCTS server port
    ctsd: 'AppType' = 5137, 'ctsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RMONITOR SECURE IANA assigned this well-formed service name as a
    #:   replacement for "rmonitor_secure".
    #: - [TCP] RMONITOR SECURE
    #: - [UDP] RMONITOR SECURE IANA assigned this well-formed service name as a
    #:   replacement for "rmonitor_secure".
    #: - [UDP] RMONITOR SECURE
    rmonitor_secure: 'AppType' = 5145, 'rmonitor-secure', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [TCP] Social Alarm Service
    social_alarm: 'AppType' = 5146, 'social-alarm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5146: 'AppType' = 5146, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Ascend Tunnel Management Protocol
    #: - [UDP] Ascend Tunnel Management Protocol
    atmp: 'AppType' = 5150, 'atmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ESRI SDE Instance IANA assigned this well-formed service name as a
    #:   replacement for "esri_sde".
    #: - [TCP] ESRI SDE Instance
    #: - [UDP] ESRI SDE Remote Start IANA assigned this well-formed service name as a
    #:   replacement for "esri_sde".
    #: - [UDP] ESRI SDE Remote Start
    esri_sde: 'AppType' = 5151, 'esri-sde', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] ESRI SDE Instance Discovery
    #: - [UDP] ESRI SDE Instance Discovery
    sde_discovery: 'AppType' = 5152, 'sde-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_5153: 'AppType' = 5153, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BZFlag game server
    #: - [UDP] BZFlag game server
    bzflag: 'AppType' = 5154, 'bzflag', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle asControl Agent
    #: - [UDP] Oracle asControl Agent
    asctrl_agent: 'AppType' = 5155, 'asctrl-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Russian Online Game
    rugameonline: 'AppType' = 5156, 'rugameonline', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5156: 'AppType' = 5156, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Mediat Remote Object Exchange
    mediat: 'AppType' = 5157, 'mediat', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5157: 'AppType' = 5157, 'reserved', TransportProtocol.get('udp')

    #: [TCP] SNMP over SSH Transport Model [:rfc:`5592`]
    snmpssh: 'AppType' = 5161, 'snmpssh', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5161: 'AppType' = 5161, 'reserved', TransportProtocol.get('udp')

    #: [TCP] SNMP Notification over SSH Transport Model [:rfc:`5592`]
    snmpssh_trap: 'AppType' = 5162, 'snmpssh-trap', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5162: 'AppType' = 5162, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Shadow Backup
    sbackup: 'AppType' = 5163, 'sbackup', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5163: 'AppType' = 5163, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Virtual Protocol Adapter
    vpa: 'AppType' = 5164, 'vpa', TransportProtocol.get('tcp')

    #: [UDP] Virtual Protocol Adapter Discovery
    vpa_disc: 'AppType' = 5164, 'vpa-disc', TransportProtocol.get('udp')

    #: - [TCP] ife_1corp IANA assigned this well-formed service name as a replacement
    #:   for "ife_icorp".
    #: - [TCP] ife_1corp
    #: - [UDP] ife_1corp IANA assigned this well-formed service name as a replacement
    #:   for "ife_icorp".
    #: - [UDP] ife_1corp
    ife_icorp: 'AppType' = 5165, 'ife-icorp', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] WinPCS Service Connection
    #: - [UDP] WinPCS Service Connection
    winpcs: 'AppType' = 5166, 'winpcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCTE104 Connection
    #: - [UDP] SCTE104 Connection
    scte104: 'AppType' = 5167, 'scte104', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCTE30 Connection
    #: - [UDP] SCTE30 Connection
    scte30: 'AppType' = 5168, 'scte30', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PC over IP Endpoint Management
    pcoip_mgmt: 'AppType' = 5172, 'pcoip-mgmt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5172: 'AppType' = 5172, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] America-Online
    #: - [UDP] America-Online
    aol: 'AppType' = 5190, 'aol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AmericaOnline1
    #: - [UDP] AmericaOnline1
    aol_1: 'AppType' = 5191, 'aol-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AmericaOnline2
    #: - [UDP] AmericaOnline2
    aol_2: 'AppType' = 5192, 'aol-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AmericaOnline3
    #: - [UDP] AmericaOnline3
    aol_3: 'AppType' = 5193, 'aol-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CipherPoint Config Service
    cpscomm: 'AppType' = 5194, 'cpscomm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5194: 'AppType' = 5194, 'reserved', TransportProtocol.get('udp')

    #: [TCP] The protocol is used by a license server and client programs to
    #: control use of program licenses that float to networked machines
    ampl_lic: 'AppType' = 5195, 'ampl-lic', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5195: 'AppType' = 5195, 'reserved', TransportProtocol.get('udp')

    #: [TCP] The protocol is used by two programs that exchange "table" data used
    #: in the AMPL modeling language
    ampl_tableproxy: 'AppType' = 5196, 'ampl-tableproxy', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5196: 'AppType' = 5196, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Tunstall Lone worker device interface
    tunstall_lwp: 'AppType' = 5197, 'tunstall-lwp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5197: 'AppType' = 5197, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] TARGUS GetData
    #: - [UDP] TARGUS GetData
    targus_getdata: 'AppType' = 5200, 'targus-getdata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TARGUS GetData 1
    #: - [UDP] TARGUS GetData 1
    targus_getdata1: 'AppType' = 5201, 'targus-getdata1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TARGUS GetData 2
    #: - [UDP] TARGUS GetData 2
    targus_getdata2: 'AppType' = 5202, 'targus-getdata2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TARGUS GetData 3
    #: - [UDP] TARGUS GetData 3
    targus_getdata3: 'AppType' = 5203, 'targus-getdata3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Nomad Device Video Transfer
    nomad: 'AppType' = 5209, 'nomad', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5209: 'AppType' = 5209, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] NOTEZA Data Safety Service
    #: - [SCTP] NOTEZA Data Safety Service
    noteza: 'AppType' = 5215, 'noteza', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_5215: 'AppType' = 5215, 'reserved', TransportProtocol.get('udp')

    #: [TCP] 3eTI Extensible Management Protocol for OAMP
    AppType_3exmp: 'AppType' = 5221, '3exmp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5221: 'AppType' = 5221, 'reserved', TransportProtocol.get('udp')

    #: [TCP] XMPP Client Connection [:rfc:`6120`]
    xmpp_client: 'AppType' = 5222, 'xmpp-client', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5222: 'AppType' = 5222, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] HP Virtual Machine Group Management
    #: - [UDP] HP Virtual Machine Group Management
    hpvirtgrp: 'AppType' = 5223, 'hpvirtgrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Virtual Machine Console Operations
    #: - [UDP] HP Virtual Machine Console Operations
    hpvirtctrl: 'AppType' = 5224, 'hpvirtctrl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Server
    #: - [UDP] HP Server
    hp_server: 'AppType' = 5225, 'hp-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Status
    #: - [UDP] HP Status
    hp_status: 'AppType' = 5226, 'hp-status', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP System Performance Metric Service
    #: - [UDP] HP System Performance Metric Service
    perfd: 'AppType' = 5227, 'perfd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] HP Virtual Room Service
    hpvroom: 'AppType' = 5228, 'hpvroom', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5228: 'AppType' = 5228, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Netflow/IPFIX/sFlow Collector and Forwarder Management
    jaxflow: 'AppType' = 5229, 'jaxflow', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5229: 'AppType' = 5229, 'reserved', TransportProtocol.get('udp')

    #: [TCP] JaxMP RealFlow application and protocol data
    jaxflow_data: 'AppType' = 5230, 'jaxflow-data', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5230: 'AppType' = 5230, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Remote Control of Scan Software for Cruse Scanners
    crusecontrol: 'AppType' = 5231, 'crusecontrol', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5231: 'AppType' = 5231, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Cruse Scanning System Service
    csedaemon: 'AppType' = 5232, 'csedaemon', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5232: 'AppType' = 5232, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Etinnae Network File Service
    enfs: 'AppType' = 5233, 'enfs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5233: 'AppType' = 5233, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] EEnet communications
    #: - [UDP] EEnet communications
    eenet: 'AppType' = 5234, 'eenet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Galaxy Network Service
    #: - [UDP] Galaxy Network Service
    galaxy_network: 'AppType' = 5235, 'galaxy-network', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP]
    #: - [UDP]
    padl2sim: 'AppType' = 5236, 'padl2sim', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] m-net discovery
    #: - [UDP] m-net discovery
    mnet_discovery: 'AppType' = 5237, 'mnet-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] ATTUne API
    attune: 'AppType' = 5242, 'attune', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5242: 'AppType' = 5242, 'reserved', TransportProtocol.get('udp')

    #: [TCP] xyClient Status API and rendevous point
    xycstatus: 'AppType' = 5243, 'xycstatus', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5243: 'AppType' = 5243, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5244: 'AppType' = 5244, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] DownTools Control Protocol
    downtools: 'AppType' = 5245, 'downtools', TransportProtocol.get('tcp')

    #: [UDP] DownTools Discovery Protocol
    downtools_disc: 'AppType' = 5245, 'downtools-disc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5246: 'AppType' = 5246, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] CAPWAP Control Protocol [:rfc:`5415`]
    capwap_control: 'AppType' = 5246, 'capwap-control', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5247: 'AppType' = 5247, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] CAPWAP Data Protocol [:rfc:`5415`]
    capwap_data: 'AppType' = 5247, 'capwap-data', TransportProtocol.get('udp')

    #: - [TCP] CA Access Control Web Service
    #: - [UDP] CA Access Control Web Service
    caacws: 'AppType' = 5248, 'caacws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA AC Lang Service
    #: - [UDP] CA AC Lang Service
    caaclang2: 'AppType' = 5249, 'caaclang2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] soaGateway
    #: - [UDP] soaGateway
    soagateway: 'AppType' = 5250, 'soagateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA eTrust VM Service
    #: - [UDP] CA eTrust VM Service
    caevms: 'AppType' = 5251, 'caevms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Movaz SSC
    #: - [UDP] Movaz SSC
    movaz_ssc: 'AppType' = 5252, 'movaz-ssc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Kohler Power Device Protocol
    kpdp: 'AppType' = 5253, 'kpdp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5253: 'AppType' = 5253, 'reserved', TransportProtocol.get('udp')

    #: [TCP] LogCabin storage service
    logcabin: 'AppType' = 5254, 'logcabin', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5254: 'AppType' = 5254, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] 3Com Network Jack Port 1
    #: - [UDP] 3Com Network Jack Port 1
    AppType_3com_njack_1: 'AppType' = 5264, '3com-njack-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3Com Network Jack Port 2
    #: - [UDP] 3Com Network Jack Port 2
    AppType_3com_njack_2: 'AppType' = 5265, '3com-njack-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] XMPP Server Connection [:rfc:`6120`]
    xmpp_server: 'AppType' = 5269, 'xmpp-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5269: 'AppType' = 5269, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Cartographer XMP
    #: - [UDP] Cartographer XMP
    cartographerxmp: 'AppType' = 5270, 'cartographerxmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] StageSoft CueLink messaging
    cuelink: 'AppType' = 5271, 'cuelink', TransportProtocol.get('tcp')

    #: [UDP] StageSoft CueLink discovery
    cuelink_disc: 'AppType' = 5271, 'cuelink-disc', TransportProtocol.get('udp')

    #: - [TCP] PK
    #: - [UDP] PK
    pk: 'AppType' = 5272, 'pk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Bidirectional-streams Over Synchronous HTTP (BOSH)
    xmpp_bosh: 'AppType' = 5280, 'xmpp-bosh', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5280: 'AppType' = 5280, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Undo License Manager
    undo_lm: 'AppType' = 5281, 'undo-lm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5281: 'AppType' = 5281, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Marimba Transmitter Port
    #: - [UDP] Marimba Transmitter Port
    transmit_port: 'AppType' = 5282, 'transmit-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NLG Data Service
    #: - [UDP] NLG Data Service
    nlg_data: 'AppType' = 5299, 'nlg-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA cluster heartbeat
    #: - [UDP] HA cluster heartbeat
    hacl_hb: 'AppType' = 5300, 'hacl-hb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA cluster general services
    #: - [UDP] HA cluster general services
    hacl_gs: 'AppType' = 5301, 'hacl-gs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA cluster configuration
    #: - [UDP] HA cluster configuration
    hacl_cfg: 'AppType' = 5302, 'hacl-cfg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA cluster probing
    #: - [UDP] HA cluster probing
    hacl_probe: 'AppType' = 5303, 'hacl-probe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA Cluster Commands
    #: - [UDP] HA Cluster Commands
    hacl_local: 'AppType' = 5304, 'hacl-local', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA Cluster Test
    #: - [UDP] HA Cluster Test
    hacl_test: 'AppType' = 5305, 'hacl-test', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun MC Group
    #: - [UDP] Sun MC Group
    sun_mc_grp: 'AppType' = 5306, 'sun-mc-grp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO AIP
    #: - [UDP] SCO AIP
    sco_aip: 'AppType' = 5307, 'sco-aip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CFengine
    #: - [UDP] CFengine
    cfengine: 'AppType' = 5308, 'cfengine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] J Printer
    #: - [UDP] J Printer
    jprinter: 'AppType' = 5309, 'jprinter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Outlaws
    #: - [UDP] Outlaws
    outlaws: 'AppType' = 5310, 'outlaws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5311: 'AppType' = 5311, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Permabit Client-Server
    #: - [UDP] Permabit Client-Server
    permabit_cs: 'AppType' = 5312, 'permabit-cs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Real-time & Reliable Data
    #: - [UDP] Real-time & Reliable Data
    rrdp: 'AppType' = 5313, 'rrdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] opalis-rbt-ipc
    #: - [UDP] opalis-rbt-ipc
    opalis_rbt_ipc: 'AppType' = 5314, 'opalis-rbt-ipc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HA Cluster UDP Polling
    #: - [UDP] HA Cluster UDP Polling
    hacl_poll: 'AppType' = 5315, 'hacl-poll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] HPBladeSystem Monitor Service
    hpbladems: 'AppType' = 5316, 'hpbladems', TransportProtocol.get('tcp')

    #: [UDP] Unassigned
    unassigned_5316: 'AppType' = 5316, 'unassigned', TransportProtocol.get('udp')

    #: [TCP] HP Device Monitor Service
    hpdevms: 'AppType' = 5317, 'hpdevms', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5317: 'AppType' = 5317, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PKIX Certificate Management using CMS (CMC) [:rfc:`6402`]
    pkix_cmc: 'AppType' = 5318, 'pkix-cmc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5318: 'AppType' = 5318, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5319: 'AppType' = 5319, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Webservices-based Zn interface of BSF
    bsfserver_zn: 'AppType' = 5320, 'bsfserver-zn', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5320: 'AppType' = 5320, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Webservices-based Zn interface of BSF over SSL
    bsfsvr_zn_ssl: 'AppType' = 5321, 'bsfsvr-zn-ssl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5321: 'AppType' = 5321, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Sculptor Database Server
    #: - [UDP] Sculptor Database Server
    kfserver: 'AppType' = 5343, 'kfserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] xkoto DRCP
    #: - [UDP] xkoto DRCP
    xkotodrcp: 'AppType' = 5344, 'xkotodrcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Session Traversal Utilities for NAT (STUN) port [:rfc:`8489`]
    #: - [UDP] STUN over DTLS [:rfc:`7350`]
    stuns: 'AppType' = 5349, 'stuns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TURN over TLS [:rfc:`8656`]
    #: - [UDP] TURN over DTLS [:rfc:`7350`]
    turns: 'AppType' = 5349, 'turns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] STUN Behavior Discovery over TLS [:rfc:`5780`]
    #: - [UDP] Reserved for a future enhancement of STUN-BEHAVIOR [:rfc:`5780`]
    stun_behaviors: 'AppType' = 5349, 'stun-behaviors', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5350: 'AppType' = 5350, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Port Control Protocol Multicast [:rfc:`6887`]
    pcp_multicast: 'AppType' = 5350, 'pcp-multicast', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5351: 'AppType' = 5351, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Port Control Protocol [:rfc:`6887`]
    pcp: 'AppType' = 5351, 'pcp', TransportProtocol.get('udp')

    #: - [TCP] DNS Long-Lived Queries [:rfc:`8764`]
    #: - [UDP] DNS Long-Lived Queries [:rfc:`8764`]
    dns_llq: 'AppType' = 5352, 'dns-llq', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multicast DNS [:rfc:`6762`]
    #: - [UDP] Multicast DNS [:rfc:`6762`]
    mdns: 'AppType' = 5353, 'mdns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multicast DNS Responder IPC
    #: - [UDP] Multicast DNS Responder IPC
    mdnsresponder: 'AppType' = 5354, 'mdnsresponder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LLMNR
    #: - [UDP] LLMNR
    llmnr: 'AppType' = 5355, 'llmnr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Small Business
    #: - [UDP] Microsoft Small Business
    ms_smlbiz: 'AppType' = 5356, 'ms-smlbiz', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Web Services for Devices
    #: - [UDP] Web Services for Devices
    wsdapi: 'AppType' = 5357, 'wsdapi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WS for Devices Secured
    #: - [UDP] WS for Devices Secured
    wsdapi_s: 'AppType' = 5358, 'wsdapi-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Alerter
    #: - [UDP] Microsoft Alerter
    ms_alerter: 'AppType' = 5359, 'ms-alerter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Protocol for Windows SideShow
    #: - [UDP] Protocol for Windows SideShow
    ms_sideshow: 'AppType' = 5360, 'ms-sideshow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Protocol for Windows SideShow
    #: - [UDP] Secure Protocol for Windows SideShow
    ms_s_sideshow: 'AppType' = 5361, 'ms-s-sideshow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Windows Server WSD2 Service
    #: - [UDP] Microsoft Windows Server WSD2 Service
    serverwsd2: 'AppType' = 5362, 'serverwsd2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Windows Network Projection
    #: - [UDP] Windows Network Projection
    net_projection: 'AppType' = 5363, 'net-projection', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Microsoft Kernel Debugger
    kdnet: 'AppType' = 5364, 'kdnet', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5364: 'AppType' = 5364, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] StressTester(tm) Injector
    #: - [UDP] StressTester(tm) Injector
    stresstester: 'AppType' = 5397, 'stresstester', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Elektron Administration
    #: - [UDP] Elektron Administration
    elektron_admin: 'AppType' = 5398, 'elektron-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SecurityChase
    #: - [UDP] SecurityChase
    securitychase: 'AppType' = 5399, 'securitychase', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Excerpt Search
    #: - [UDP] Excerpt Search
    excerpt: 'AppType' = 5400, 'excerpt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Excerpt Search Secure
    #: - [UDP] Excerpt Search Secure
    excerpts: 'AppType' = 5401, 'excerpts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mftp
    #: - [UDP] mftp
    mftp_349: 'AppType' = 349, 'mftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OmniCast MFTP
    #: - [UDP] OmniCast MFTP
    mftp_5402: 'AppType' = 5402, 'mftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPOMS-CI-LSTN
    #: - [UDP] HPOMS-CI-LSTN
    hpoms_ci_lstn: 'AppType' = 5403, 'hpoms-ci-lstn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HPOMS-DPS-LSTN
    #: - [UDP] HPOMS-DPS-LSTN
    hpoms_dps_lstn: 'AppType' = 5404, 'hpoms-dps-lstn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetSupport
    #: - [UDP] NetSupport
    netsupport: 'AppType' = 5405, 'netsupport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Systemics Sox
    #: - [UDP] Systemics Sox
    systemics_sox: 'AppType' = 5406, 'systemics-sox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Foresyte-Clear
    #: - [UDP] Foresyte-Clear
    foresyte_clear: 'AppType' = 5407, 'foresyte-clear', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Foresyte-Sec
    #: - [UDP] Foresyte-Sec
    foresyte_sec: 'AppType' = 5408, 'foresyte-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Salient Data Server
    #: - [UDP] Salient Data Server
    salient_dtasrv: 'AppType' = 5409, 'salient-dtasrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Salient User Manager
    #: - [UDP] Salient User Manager
    salient_usrmgr: 'AppType' = 5410, 'salient-usrmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ActNet
    #: - [UDP] ActNet
    actnet: 'AppType' = 5411, 'actnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Continuus
    #: - [UDP] Continuus
    continuus: 'AppType' = 5412, 'continuus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WWIOTALK
    #: - [UDP] WWIOTALK
    wwiotalk: 'AppType' = 5413, 'wwiotalk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] StatusD
    #: - [UDP] StatusD
    statusd: 'AppType' = 5414, 'statusd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NS Server
    #: - [UDP] NS Server
    ns_server: 'AppType' = 5415, 'ns-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Gateway
    #: - [UDP] SNS Gateway
    sns_gateway: 'AppType' = 5416, 'sns-gateway', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SNS Agent
    #: - [UDP] SNS Agent
    sns_agent: 'AppType' = 5417, 'sns-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCNTP
    #: - [UDP] MCNTP
    mcntp: 'AppType' = 5418, 'mcntp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DJ-ICE
    #: - [UDP] DJ-ICE
    dj_ice: 'AppType' = 5419, 'dj-ice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cylink-C
    #: - [UDP] Cylink-C
    cylink_c: 'AppType' = 5420, 'cylink-c', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net Support 2
    #: - [UDP] Net Support 2
    netsupport2: 'AppType' = 5421, 'netsupport2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Salient MUX
    #: - [UDP] Salient MUX
    salient_mux: 'AppType' = 5422, 'salient-mux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VIRTUALUSER
    #: - [UDP] VIRTUALUSER
    virtualuser: 'AppType' = 5423, 'virtualuser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Beyond Remote
    #: - [UDP] Beyond Remote
    beyond_remote: 'AppType' = 5424, 'beyond-remote', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Beyond Remote Command Channel
    #: - [UDP] Beyond Remote Command Channel
    br_channel: 'AppType' = 5425, 'br-channel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DEVBASIC
    #: - [UDP] DEVBASIC
    devbasic: 'AppType' = 5426, 'devbasic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCO-PEER-TTA
    #: - [UDP] SCO-PEER-TTA
    sco_peer_tta: 'AppType' = 5427, 'sco-peer-tta', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TELACONSOLE
    #: - [UDP] TELACONSOLE
    telaconsole: 'AppType' = 5428, 'telaconsole', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Billing and Accounting System Exchange
    #: - [UDP] Billing and Accounting System Exchange
    base: 'AppType' = 5429, 'base', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RADEC CORP
    #: - [UDP] RADEC CORP
    radec_corp: 'AppType' = 5430, 'radec-corp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PARK AGENT
    #: - [UDP] PARK AGENT
    park_agent: 'AppType' = 5431, 'park-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PostgreSQL Database
    #: - [UDP] PostgreSQL Database
    postgresql: 'AppType' = 5432, 'postgresql', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pyrrho DBMS
    #: - [UDP] Pyrrho DBMS
    pyrrho: 'AppType' = 5433, 'pyrrho', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SGI Array Services Daemon
    #: - [UDP] SGI Array Services Daemon
    sgi_arrayd: 'AppType' = 5434, 'sgi-arrayd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SCEANICS situation and action notification
    #: - [UDP] SCEANICS situation and action notification
    sceanics: 'AppType' = 5435, 'sceanics', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5436: 'AppType' = 5436, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] pmip6-cntl [:rfc:`5844`]
    pmip6_cntl: 'AppType' = 5436, 'pmip6-cntl', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5437: 'AppType' = 5437, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] pmip6-data [:rfc:`5844`]
    pmip6_data: 'AppType' = 5437, 'pmip6-data', TransportProtocol.get('udp')

    #: - [TCP] Pearson HTTPS
    #: - [UDP] Pearson HTTPS
    spss: 'AppType' = 5443, 'spss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5444: 'AppType' = 5444, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Server Message Block over Remote Direct Memory Access
    #: - [SCTP] Server Message Block over Remote Direct Memory Access
    smbdirect: 'AppType' = 5445, 'smbdirect', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_5445: 'AppType' = 5445, 'reserved', TransportProtocol.get('udp')

    #: [TCP] TiePie engineering data acquisition
    tiepie: 'AppType' = 5450, 'tiepie', TransportProtocol.get('tcp')

    #: [UDP] TiePie engineering data acquisition (discovery)
    tiepie_disc: 'AppType' = 5450, 'tiepie-disc', TransportProtocol.get('udp')

    #: - [TCP] SureBox
    #: - [UDP] SureBox
    surebox: 'AppType' = 5453, 'surebox', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 5454
    #: - [UDP] APC 5454
    apc_5454: 'AppType' = 5454, 'apc-5454', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 5455
    #: - [UDP] APC 5455
    apc_5455: 'AppType' = 5455, 'apc-5455', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 5456
    #: - [UDP] APC 5456
    apc_5456: 'AppType' = 5456, 'apc-5456', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SILKMETER
    #: - [UDP] SILKMETER
    silkmeter: 'AppType' = 5461, 'silkmeter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTL Publisher
    #: - [UDP] TTL Publisher
    ttl_publisher: 'AppType' = 5462, 'ttl-publisher', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TTL Price Proxy
    #: - [UDP] TTL Price Proxy
    ttlpriceproxy: 'AppType' = 5463, 'ttlpriceproxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quail Networks Object Broker
    #: - [UDP] Quail Networks Object Broker
    quailnet: 'AppType' = 5464, 'quailnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NETOPS-BROKER
    #: - [UDP] NETOPS-BROKER
    netops_broker: 'AppType' = 5465, 'netops-broker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] The Apsolab company's data collection protocol (native api)
    apsolab_col: 'AppType' = 5470, 'apsolab-col', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5470: 'AppType' = 5470, 'reserved', TransportProtocol.get('udp')

    #: [TCP] The Apsolab company's secure data collection protocol (native api)
    apsolab_cols: 'AppType' = 5471, 'apsolab-cols', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5471: 'AppType' = 5471, 'reserved', TransportProtocol.get('udp')

    #: [TCP] The Apsolab company's dynamic tag protocol
    apsolab_tag: 'AppType' = 5472, 'apsolab-tag', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5472: 'AppType' = 5472, 'reserved', TransportProtocol.get('udp')

    #: [TCP] The Apsolab company's secure dynamic tag protocol
    apsolab_tags: 'AppType' = 5473, 'apsolab-tags', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5473: 'AppType' = 5473, 'reserved', TransportProtocol.get('udp')

    #: [UDP] The Apsolab company's status query protocol
    apsolab_rpc: 'AppType' = 5474, 'apsolab-rpc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5474: 'AppType' = 5474, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] The Apsolab company's data retrieval protocol
    apsolab_data: 'AppType' = 5475, 'apsolab-data', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5475: 'AppType' = 5475, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] fcp-addr-srvr1
    #: - [UDP] fcp-addr-srvr1
    fcp_addr_srvr1: 'AppType' = 5500, 'fcp-addr-srvr1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcp-addr-srvr2
    #: - [UDP] fcp-addr-srvr2
    fcp_addr_srvr2: 'AppType' = 5501, 'fcp-addr-srvr2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcp-srvr-inst1
    #: - [UDP] fcp-srvr-inst1
    fcp_srvr_inst1: 'AppType' = 5502, 'fcp-srvr-inst1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcp-srvr-inst2
    #: - [UDP] fcp-srvr-inst2
    fcp_srvr_inst2: 'AppType' = 5503, 'fcp-srvr-inst2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcp-cics-gw1
    #: - [UDP] fcp-cics-gw1
    fcp_cics_gw1: 'AppType' = 5504, 'fcp-cics-gw1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Checkout Database
    #: - [UDP] Checkout Database
    checkoutdb: 'AppType' = 5505, 'checkoutdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Amcom Mobile Connect
    #: - [UDP] Amcom Mobile Connect
    amc: 'AppType' = 5506, 'amc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PowerSysLab Electrical Management
    psl_management: 'AppType' = 5507, 'psl-management', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5507: 'AppType' = 5507, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Matter Operational Discovery and Communi
    #: - [UDP] Matter Operational Discovery and Communi
    matter: 'AppType' = 5540, 'matter', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] QF-Test License Server
    qftest_licserve: 'AppType' = 5543, 'qftest-licserve', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5543: 'AppType' = 5543, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Model Railway control using the CBUS message protocol
    cbus: 'AppType' = 5550, 'cbus', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5550: 'AppType' = 5550, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SGI Eventmond Port
    #: - [UDP] SGI Eventmond Port
    sgi_eventmond: 'AppType' = 5553, 'sgi-eventmond', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SGI ESP HTTP
    #: - [UDP] SGI ESP HTTP
    sgi_esphttp: 'AppType' = 5554, 'sgi-esphttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Personal Agent
    #: - [UDP] Personal Agent
    personal_agent: 'AppType' = 5555, 'personal-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Freeciv gameplay
    #: - [UDP] Freeciv gameplay
    freeciv: 'AppType' = 5556, 'freeciv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Sandlab FARENET
    farenet: 'AppType' = 5557, 'farenet', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5557: 'AppType' = 5557, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Data Protector BURA
    dp_bura: 'AppType' = 5565, 'dp-bura', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5565: 'AppType' = 5565, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Westec Connect
    westec_connect: 'AppType' = 5566, 'westec-connect', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5566: 'AppType' = 5566, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] DOF Protocol Stack Multicast/Secure Transport
    #: - [UDP] DOF Protocol Stack Multicast/Secure Transport
    dof_dps_mc_sec: 'AppType' = 5567, 'dof-dps-mc-sec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Session Data Transport Multicast
    #: - [UDP] Session Data Transport Multicast
    sdt: 'AppType' = 5568, 'sdt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PLASA E1.33, Remote Device Management (RDM) controller status
    #: notifications
    rdmnet_ctrl: 'AppType' = 5569, 'rdmnet-ctrl', TransportProtocol.get('tcp')

    #: [UDP] PLASA E1.33, Remote Device Management (RDM) messages
    rdmnet_device: 'AppType' = 5569, 'rdmnet-device', TransportProtocol.get('udp')

    #: - [TCP] SAS Domain Management Messaging Protocol
    #: - [UDP] SAS Domain Management Messaging Protocol
    sdmmp: 'AppType' = 5573, 'sdmmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SAS IO Forwarding
    lsi_bobcat: 'AppType' = 5574, 'lsi-bobcat', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5574: 'AppType' = 5574, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Oracle Access Protocol
    ora_oap: 'AppType' = 5575, 'ora-oap', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5575: 'AppType' = 5575, 'reserved', TransportProtocol.get('udp')

    #: [TCP] FleetDisplay Tracking Service
    fdtracks: 'AppType' = 5579, 'fdtracks', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5579: 'AppType' = 5579, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] T-Mobile SMS Protocol Message 0
    #: - [UDP] T-Mobile SMS Protocol Message 0
    tmosms0: 'AppType' = 5580, 'tmosms0', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] T-Mobile SMS Protocol Message 1
    #: - [UDP] T-Mobile SMS Protocol Message 1
    tmosms1: 'AppType' = 5581, 'tmosms1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] T-Mobile SMS Protocol Message 3
    #: - [UDP] T-Mobile SMS Protocol Message 3
    fac_restore: 'AppType' = 5582, 'fac-restore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] T-Mobile SMS Protocol Message 2
    #: - [UDP] T-Mobile SMS Protocol Message 2
    tmo_icon_sync: 'AppType' = 5583, 'tmo-icon-sync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BeInSync-Web
    #: - [UDP] BeInSync-Web
    bis_web: 'AppType' = 5584, 'bis-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BeInSync-sync
    #: - [UDP] BeInSync-sync
    bis_sync: 'AppType' = 5585, 'bis-sync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Planning to send mobile terminated SMS to the specific port so that
    #: the SMS is not visible to the client
    att_mt_sms: 'AppType' = 5586, 'att-mt-sms', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5586: 'AppType' = 5586, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] inin secure messaging
    #: - [UDP] inin secure messaging
    ininmessaging: 'AppType' = 5597, 'ininmessaging', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCT Market Data Feed
    #: - [UDP] MCT Market Data Feed
    mctfeed: 'AppType' = 5598, 'mctfeed', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Security Remote Install
    #: - [UDP] Enterprise Security Remote Install
    esinstall: 'AppType' = 5599, 'esinstall', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Security Manager
    #: - [UDP] Enterprise Security Manager
    esmmanager: 'AppType' = 5600, 'esmmanager', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Enterprise Security Agent
    #: - [UDP] Enterprise Security Agent
    esmagent: 'AppType' = 5601, 'esmagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A1-MSC
    #: - [UDP] A1-MSC
    a1_msc: 'AppType' = 5602, 'a1-msc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A1-BS
    #: - [UDP] A1-BS
    a1_bs: 'AppType' = 5603, 'a1-bs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A3-SDUNode
    #: - [UDP] A3-SDUNode
    a3_sdunode: 'AppType' = 5604, 'a3-sdunode', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] A4-SDUNode
    #: - [UDP] A4-SDUNode
    a4_sdunode: 'AppType' = 5605, 'a4-sdunode', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Fiscal Registering Protocol
    efr: 'AppType' = 5618, 'efr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5618: 'AppType' = 5618, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Node Initiated Network Association Forma
    #: - [UDP] Node Initiated Network Association Forma
    ninaf: 'AppType' = 5627, 'ninaf', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTrust API
    #: - [UDP] HTrust API
    htrust: 'AppType' = 5628, 'htrust', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Symantec Storage Foundation for Database
    #: - [UDP] Symantec Storage Foundation for Database
    symantec_sfdb: 'AppType' = 5629, 'symantec-sfdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PreciseCommunication
    #: - [UDP] PreciseCommunication
    precise_comm: 'AppType' = 5630, 'precise-comm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pcANYWHEREdata
    #: - [UDP] pcANYWHEREdata
    pcanywheredata: 'AppType' = 5631, 'pcanywheredata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pcANYWHEREstat
    #: - [UDP] pcANYWHEREstat
    pcanywherestat: 'AppType' = 5632, 'pcanywherestat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BE Operations Request Listener
    #: - [UDP] BE Operations Request Listener
    beorl: 'AppType' = 5633, 'beorl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SF Message Service
    #: - [UDP] SF Message Service
    xprtld: 'AppType' = 5634, 'xprtld', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SFM Authentication Subsystem
    sfmsso: 'AppType' = 5635, 'sfmsso', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5635: 'AppType' = 5635, 'reserved', TransportProtocol.get('udp')

    #: [TCP] SFMdb - SFM DB server
    sfm_db_server: 'AppType' = 5636, 'sfm-db-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5636: 'AppType' = 5636, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Symantec CSSC
    cssc: 'AppType' = 5637, 'cssc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5637: 'AppType' = 5637, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Symantec Fingerprint Lookup and Container Reference Service
    flcrs: 'AppType' = 5638, 'flcrs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5638: 'AppType' = 5638, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Symantec Integrity Checking Service
    ics: 'AppType' = 5639, 'ics', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5639: 'AppType' = 5639, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Ventureforth Mobile
    vfmobile: 'AppType' = 5646, 'vfmobile', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5646: 'AppType' = 5646, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Nagios Remote Plugin Executor
    nrpe: 'AppType' = 5666, 'nrpe', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5666: 'AppType' = 5666, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ZeroMQ file publish-subscribe protocol
    filemq: 'AppType' = 5670, 'filemq', TransportProtocol.get('tcp')

    #: [UDP] Local area discovery and messaging over ZeroMQ
    zre_disc: 'AppType' = 5670, 'zre-disc', TransportProtocol.get('udp')

    #: - [TCP] amqp protocol over TLS/SSL
    #: - [UDP] amqp protocol over TLS/SSL
    amqps: 'AppType' = 5671, 'amqps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AMQP
    #: - [UDP] AMQP
    #: - [SCTP] AMQP
    amqp: 'AppType' = 5672, 'amqp', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] JACL Message Server
    #: - [UDP] JACL Message Server
    jms: 'AppType' = 5673, 'jms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HyperSCSI Port
    #: - [UDP] HyperSCSI Port
    hyperscsi_port: 'AppType' = 5674, 'hyperscsi-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] V5UA application port [:rfc:`3807`]
    #: - [UDP] V5UA application port [:rfc:`3807`]
    #: - [SCTP] V5UA application port [:rfc:`3807`]
    v5ua: 'AppType' = 5675, 'v5ua', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] RA Administration
    #: - [UDP] RA Administration
    raadmin: 'AppType' = 5676, 'raadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quest Central DB2 Launchr
    #: - [UDP] Quest Central DB2 Launchr
    questdb2_lnchr: 'AppType' = 5677, 'questdb2-lnchr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Remote Replication Agent Connection
    #: - [UDP] Remote Replication Agent Connection
    rrac: 'AppType' = 5678, 'rrac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Direct Cable Connect Manager
    #: - [UDP] Direct Cable Connect Manager
    dccm: 'AppType' = 5679, 'dccm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Auriga Router Service
    #: - [UDP] Auriga Router Service
    auriga_router: 'AppType' = 5680, 'auriga-router', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Net-coneX Control Protocol
    #: - [UDP] Net-coneX Control Protocol
    ncxcp: 'AppType' = 5681, 'ncxcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5682: 'AppType' = 5682, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] BrightCore control & data transfer exchange
    brightcore: 'AppType' = 5682, 'brightcore', TransportProtocol.get('udp')

    #: - [TCP] Constrained Application Protocol (CoAP) [:rfc:`8323`]
    #: - [UDP] Constrained Application Protocol [:rfc:`7252`]
    coap: 'AppType' = 5683, 'coap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Constrained Application Protocol (CoAP) [:rfc:`7301`][:rfc:`8323`]
    #: - [UDP] DTLS-secured CoAP [:rfc:`7252`]
    coaps: 'AppType' = 5684, 'coaps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] GOG multiplayer game protocol
    gog_multiplayer: 'AppType' = 5687, 'gog-multiplayer', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5687: 'AppType' = 5687, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] GGZ Gaming Zone
    #: - [UDP] GGZ Gaming Zone
    ggz: 'AppType' = 5688, 'ggz', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QM video network management protocol
    #: - [UDP] QM video network management protocol
    qmvideo: 'AppType' = 5689, 'qmvideo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Robert Bosch Data Transfer
    rbsystem: 'AppType' = 5693, 'rbsystem', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5693: 'AppType' = 5693, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Key Management Interoperability Protocol
    kmip: 'AppType' = 5696, 'kmip', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5696: 'AppType' = 5696, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Dell SupportAssist data center management
    supportassist: 'AppType' = 5700, 'supportassist', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5700: 'AppType' = 5700, 'reserved', TransportProtocol.get('udp')

    #: [TCP] StorageOS REST API
    storageos: 'AppType' = 5705, 'storageos', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5705: 'AppType' = 5705, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] proshare conf audio
    #: - [UDP] proshare conf audio
    proshareaudio: 'AppType' = 5713, 'proshareaudio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proshare conf video
    #: - [UDP] proshare conf video
    prosharevideo: 'AppType' = 5714, 'prosharevideo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proshare conf data
    #: - [UDP] proshare conf data
    prosharedata: 'AppType' = 5715, 'prosharedata', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proshare conf request
    #: - [UDP] proshare conf request
    prosharerequest: 'AppType' = 5716, 'prosharerequest', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] proshare conf notify
    #: - [UDP] proshare conf notify
    prosharenotify: 'AppType' = 5717, 'prosharenotify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DPM Communication Server
    #: - [UDP] DPM Communication Server
    dpm: 'AppType' = 5718, 'dpm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DPM Agent Coordinator
    #: - [UDP] DPM Agent Coordinator
    dpm_agent: 'AppType' = 5719, 'dpm-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MS-Licensing
    #: - [UDP] MS-Licensing
    ms_licensing: 'AppType' = 5720, 'ms-licensing', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Desktop Passthru Service
    #: - [UDP] Desktop Passthru Service
    dtpt: 'AppType' = 5721, 'dtpt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft DFS Replication Service
    #: - [UDP] Microsoft DFS Replication Service
    msdfsr: 'AppType' = 5722, 'msdfsr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Operations Manager - Health Service
    #: - [UDP] Operations Manager - Health Service
    omhs: 'AppType' = 5723, 'omhs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Operations Manager - SDK Service
    #: - [UDP] Operations Manager - SDK Service
    omsdk: 'AppType' = 5724, 'omsdk', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Microsoft Identity Lifecycle Manager
    ms_ilm: 'AppType' = 5725, 'ms-ilm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5725: 'AppType' = 5725, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Microsoft Lifecycle Manager Secure Token Service
    ms_ilm_sts: 'AppType' = 5726, 'ms-ilm-sts', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5726: 'AppType' = 5726, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ASG Event Notification Framework
    asgenf: 'AppType' = 5727, 'asgenf', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5727: 'AppType' = 5727, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Dist. I/O Comm. Service Data and Control
    io_dist_data: 'AppType' = 5728, 'io-dist-data', TransportProtocol.get('tcp')

    #: [UDP] Dist. I/O Comm. Service Group Membership
    io_dist_group: 'AppType' = 5728, 'io-dist-group', TransportProtocol.get('udp')

    #: - [TCP] Openmail User Agent Layer
    #: - [UDP] Openmail User Agent Layer
    openmail: 'AppType' = 5729, 'openmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Steltor's calendar access
    #: - [UDP] Steltor's calendar access
    unieng: 'AppType' = 5730, 'unieng', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDA Discover Port 1
    #: - [UDP] IDA Discover Port 1
    ida_discover1: 'AppType' = 5741, 'ida-discover1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IDA Discover Port 2
    #: - [UDP] IDA Discover Port 2
    ida_discover2: 'AppType' = 5742, 'ida-discover2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Watchdoc NetPOD Protocol
    #: - [UDP] Watchdoc NetPOD Protocol
    watchdoc_pod: 'AppType' = 5743, 'watchdoc-pod', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Watchdoc Server
    #: - [UDP] Watchdoc Server
    watchdoc: 'AppType' = 5744, 'watchdoc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcopy-server
    #: - [UDP] fcopy-server
    fcopy_server: 'AppType' = 5745, 'fcopy-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fcopys-server
    #: - [UDP] fcopys-server
    fcopys_server: 'AppType' = 5746, 'fcopys-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wildbits Tunatic
    #: - [UDP] Wildbits Tunatic
    tunatic: 'AppType' = 5747, 'tunatic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Wildbits Tunalyzer
    #: - [UDP] Wildbits Tunalyzer
    tunalyzer: 'AppType' = 5748, 'tunalyzer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5749: 'AppType' = 5749, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Bladelogic Agent Service
    #: - [UDP] Bladelogic Agent Service
    rscd: 'AppType' = 5750, 'rscd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenMail Desk Gateway server
    #: - [UDP] OpenMail Desk Gateway server
    openmailg: 'AppType' = 5755, 'openmailg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_5756: 'AppType' = 5756, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] OpenMail X.500 Directory Server
    #: - [UDP] OpenMail X.500 Directory Server
    x500ms: 'AppType' = 5757, 'x500ms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenMail NewMail Server
    #: - [UDP] OpenMail NewMail Server
    openmailns: 'AppType' = 5766, 'openmailns', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenMail Suer Agent Layer (Secure)
    #: - [UDP] OpenMail Suer Agent Layer (Secure)
    s_openmail: 'AppType' = 5767, 's-openmail', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenMail CMTS Server
    #: - [UDP] OpenMail CMTS Server
    openmailpxy: 'AppType' = 5768, 'openmailpxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] x509solutions Internal CA
    #: - [UDP] x509solutions Internal CA
    spramsca: 'AppType' = 5769, 'spramsca', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] x509solutions Secure Data
    #: - [UDP] x509solutions Secure Data
    spramsd: 'AppType' = 5770, 'spramsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NetAgent
    #: - [UDP] NetAgent
    netagent: 'AppType' = 5771, 'netagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Control commands and responses
    #: - [UDP] Control commands and responses
    starfield_io: 'AppType' = 5777, 'starfield-io', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Visual Tag System RPC
    vts_rpc: 'AppType' = 5780, 'vts-rpc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5780: 'AppType' = 5780, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] 3PAR Event Reporting Service
    #: - [UDP] 3PAR Event Reporting Service
    AppType_3par_evts: 'AppType' = 5781, '3par-evts', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3PAR Management Service
    #: - [UDP] 3PAR Management Service
    AppType_3par_mgmt: 'AppType' = 5782, '3par-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] 3PAR Management Service with SSL
    #: - [UDP] 3PAR Management Service with SSL
    AppType_3par_mgmt_ssl: 'AppType' = 5783, '3par-mgmt-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5784: 'AppType' = 5784, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cisco Interbox Application Redundancy
    ibar: 'AppType' = 5784, 'ibar', TransportProtocol.get('udp')

    #: - [TCP] 3PAR Inform Remote Copy
    #: - [UDP] 3PAR Inform Remote Copy
    AppType_3par_rcopy: 'AppType' = 5785, '3par-rcopy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5786: 'AppType' = 5786, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] redundancy notification
    cisco_redu: 'AppType' = 5786, 'cisco-redu', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5787: 'AppType' = 5787, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cisco WAAS Cluster Protocol
    waascluster: 'AppType' = 5787, 'waascluster', TransportProtocol.get('udp')

    #: - [TCP] XtreamX Supervised Peer message
    #: - [UDP] XtreamX Supervised Peer message
    xtreamx: 'AppType' = 5793, 'xtreamx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_5794: 'AppType' = 5794, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Simple Peered Discovery Protocol
    spdp: 'AppType' = 5794, 'spdp', TransportProtocol.get('udp')

    #: [TCP] Proprietary Website deployment service
    enlabel_dpl: 'AppType' = 5798, 'enlabel-dpl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5798: 'AppType' = 5798, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ICMPD
    #: - [UDP] ICMPD
    icmpd: 'AppType' = 5813, 'icmpd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Support Automation
    #: - [UDP] Support Automation
    spt_automation: 'AppType' = 5814, 'spt-automation', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] AutoPass licensing
    autopassdaemon: 'AppType' = 5820, 'autopassdaemon', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5820: 'AppType' = 5820, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Z-firm ShipRush interface for web access and bidirectional data
    shiprush_d_ch: 'AppType' = 5841, 'shiprush-d-ch', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5841: 'AppType' = 5841, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reversion Backup/Restore
    reversion: 'AppType' = 5842, 'reversion', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5842: 'AppType' = 5842, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] WHEREHOO
    #: - [UDP] WHEREHOO
    wherehoo: 'AppType' = 5859, 'wherehoo', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PlanetPress Suite Messeng
    #: - [UDP] PlanetPress Suite Messeng
    ppsuitemsg: 'AppType' = 5863, 'ppsuitemsg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Diameter over TLS/TCP [:rfc:`6733`]
    #: - [SCTP] Diameter over DTLS/SCTP [:rfc:`6733`]
    diameters: 'AppType' = 5868, 'diameters', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_5868: 'AppType' = 5868, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Javascript Unit Test Environment
    jute: 'AppType' = 5883, 'jute', TransportProtocol.get('tcp')

    #: - [TCP] Remote Framebuffer [:rfc:`6143`]
    #: - [UDP] Remote Framebuffer [:rfc:`6143`]
    rfb: 'AppType' = 5900, 'rfb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Flight & Flow Info for Collaborative Env
    #: - [UDP] Flight & Flow Info for Collaborative Env
    #: - [SCTP] Flight & Flow Info for Collaborative Env
    ff_ice: 'AppType' = 5903, 'ff-ice', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Air-Ground SWIM
    #: - [UDP] Air-Ground SWIM
    #: - [SCTP] Air-Ground SWIM
    ag_swim: 'AppType' = 5904, 'ag-swim', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Adv Surface Mvmnt and Guidance Cont Sys
    #: - [UDP] Adv Surface Mvmnt and Guidance Cont Sys
    #: - [SCTP] Adv Surface Mvmnt and Guidance Cont Sys
    asmgcs: 'AppType' = 5905, 'asmgcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Remotely Piloted Vehicle C&C
    #: - [UDP] Remotely Piloted Vehicle C&C
    #: - [SCTP] Remotely Piloted Vehicle C&C
    rpas_c2: 'AppType' = 5906, 'rpas-c2', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Distress and Safety Data App
    #: - [UDP] Distress and Safety Data App
    #: - [SCTP] Distress and Safety Data App
    dsd: 'AppType' = 5907, 'dsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] IPS Management Application
    #: - [UDP] IPS Management Application
    #: - [SCTP] IPS Management Application
    ipsma: 'AppType' = 5908, 'ipsma', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Air-ground media advisory
    #: - [UDP] Air-ground media advisory
    #: - [SCTP] Air-ground media advisory
    agma: 'AppType' = 5909, 'agma', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Air Traffic Services applications using ATN
    #: - [UDP] Air Traffic Services applications using ATN
    ats_atn: 'AppType' = 5910, 'ats-atn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Context Management
    cm: 'AppType' = 5910, 'cm', TransportProtocol.get('sctp')

    #: - [TCP] Air Traffic Services applications using ACARS
    #: - [UDP] Air Traffic Services applications using ACARS
    ats_acars: 'AppType' = 5911, 'ats-acars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Controller Pilot Data Link Communication
    cpdlc: 'AppType' = 5911, 'cpdlc', TransportProtocol.get('sctp')

    #: - [TCP] Aeronautical Information Service/Meteorological applications using
    #:   ACARS
    #: - [UDP] Aeronautical Information Service/Meteorological applications using
    #:   ACARS
    ais_met: 'AppType' = 5912, 'ais-met', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Flight Information Services
    fis: 'AppType' = 5912, 'fis', TransportProtocol.get('sctp')

    #: - [TCP] Airline operational communications applications using ACARS
    #: - [UDP] Airline operational communications applications using ACARS
    aoc_acars: 'AppType' = 5913, 'aoc-acars', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Automatic Dependent Surveillance
    ads_c: 'AppType' = 5913, 'ads-c', TransportProtocol.get('sctp')

    #: - [TCP] Indy Application Server
    #: - [UDP] Indy Application Server
    indy: 'AppType' = 5963, 'indy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mppolicy-v5
    #: - [UDP] mppolicy-v5
    mppolicy_v5: 'AppType' = 5968, 'mppolicy-v5', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mppolicy-mgr
    #: - [UDP] mppolicy-mgr
    mppolicy_mgr: 'AppType' = 5969, 'mppolicy-mgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CouchDB
    #: - [UDP] CouchDB
    couchdb: 'AppType' = 5984, 'couchdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM WS-Management HTTP
    #: - [UDP] WBEM WS-Management HTTP
    wsman: 'AppType' = 5985, 'wsman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM WS-Management HTTP over TLS/SSL
    #: - [UDP] WBEM WS-Management HTTP over TLS/SSL
    wsmans: 'AppType' = 5986, 'wsmans', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM RMI
    #: - [UDP] WBEM RMI
    wbem_rmi: 'AppType' = 5987, 'wbem-rmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM CIM-XML (HTTP)
    #: - [UDP] WBEM CIM-XML (HTTP)
    wbem_http: 'AppType' = 5988, 'wbem-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM CIM-XML (HTTPS)
    #: - [UDP] WBEM CIM-XML (HTTPS)
    wbem_https: 'AppType' = 5989, 'wbem-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WBEM Export HTTPS
    #: - [UDP] WBEM Export HTTPS
    wbem_exp_https: 'AppType' = 5990, 'wbem-exp-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NUXSL
    #: - [UDP] NUXSL
    nuxsl: 'AppType' = 5991, 'nuxsl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Consul InSight Security
    #: - [UDP] Consul InSight Security
    consul_insight: 'AppType' = 5992, 'consul-insight', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DMTF WBEM CIM REST
    cim_rs: 'AppType' = 5993, 'cim-rs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5993: 'AppType' = 5993, 'reserved', TransportProtocol.get('udp')

    #: [TCP] RMS Agent Listening Service
    rms_agent: 'AppType' = 5994, 'rms-agent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_5994: 'AppType' = 5994, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] CVSup
    #: - [UDP] CVSup
    cvsup: 'AppType' = 5999, 'cvsup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NDL-AHP-SVC
    #: - [UDP] NDL-AHP-SVC
    ndl_ahp_svc: 'AppType' = 6064, 'ndl-ahp-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WinPharaoh
    #: - [UDP] WinPharaoh
    winpharaoh: 'AppType' = 6065, 'winpharaoh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EWCTSP
    #: - [UDP] EWCTSP
    ewctsp: 'AppType' = 6066, 'ewctsp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6067: 'AppType' = 6067, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] GSMP/ANCP [:rfc:`6320`]
    gsmp_ancp: 'AppType' = 6068, 'gsmp-ancp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6068: 'AppType' = 6068, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] TRIP
    #: - [UDP] TRIP
    trip: 'AppType' = 6069, 'trip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Messageasap
    #: - [UDP] Messageasap
    messageasap: 'AppType' = 6070, 'messageasap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SSDTP
    #: - [UDP] SSDTP
    ssdtp: 'AppType' = 6071, 'ssdtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DIAGNOSE-PROC
    #: - [UDP] DIAGNOSE-PROC
    diagnose_proc: 'AppType' = 6072, 'diagnose-proc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DirectPlay8
    #: - [UDP] DirectPlay8
    directplay8: 'AppType' = 6073, 'directplay8', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Max
    #: - [UDP] Microsoft Max
    max: 'AppType' = 6074, 'max', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Microsoft DPM Access Control Manager
    dpm_acm: 'AppType' = 6075, 'dpm-acm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6075: 'AppType' = 6075, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Microsoft DPM WCF Certificates
    msft_dpm_cert: 'AppType' = 6076, 'msft-dpm-cert', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6076: 'AppType' = 6076, 'reserved', TransportProtocol.get('udp')

    #: [TCP] iConstruct Server
    iconstructsrv: 'AppType' = 6077, 'iconstructsrv', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6077: 'AppType' = 6077, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Generic UDP Encapsulation [draft-herbert-gue]
    gue: 'AppType' = 6080, 'gue', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6080: 'AppType' = 6080, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Generic Network Virtualization Encapsulation (Geneve) [:rfc:`8926`]
    geneve: 'AppType' = 6081, 'geneve', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6081: 'AppType' = 6081, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] Reserved
    reserved_6082: 'AppType' = 6082, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] APCO Project 25 Common Air Interface - UDP encapsulation
    p25cai: 'AppType' = 6082, 'p25cai', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6083: 'AppType' = 6083, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] telecomsoftware miami broadcast
    miami_bcast: 'AppType' = 6083, 'miami-bcast', TransportProtocol.get('udp')

    #: [TCP] Peer to Peer Infrastructure Configuration [:rfc:`6940`]
    reload_config: 'AppType' = 6084, 'reload-config', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6084: 'AppType' = 6084, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] konspire2b p2p network
    #: - [UDP] konspire2b p2p network
    konspire2b: 'AppType' = 6085, 'konspire2b', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PDTP P2P
    #: - [UDP] PDTP P2P
    pdtp: 'AppType' = 6086, 'pdtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Local Download Sharing Service
    #: - [UDP] Local Download Sharing Service
    ldss: 'AppType' = 6087, 'ldss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SuperDog License Manager
    doglms: 'AppType' = 6088, 'doglms', TransportProtocol.get('tcp')

    #: [UDP] SuperDog License Manager Notifier
    doglms_notify: 'AppType' = 6088, 'doglms-notify', TransportProtocol.get('udp')

    #: [TCP] RAXA Management
    raxa_mgmt: 'AppType' = 6099, 'raxa-mgmt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6099: 'AppType' = 6099, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SynchroNet-db
    #: - [UDP] SynchroNet-db
    synchronet_db: 'AppType' = 6100, 'synchronet-db', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SynchroNet-rtc
    #: - [UDP] SynchroNet-rtc
    synchronet_rtc: 'AppType' = 6101, 'synchronet-rtc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SynchroNet-upd
    #: - [UDP] SynchroNet-upd
    synchronet_upd: 'AppType' = 6102, 'synchronet-upd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RETS
    #: - [UDP] RETS
    rets: 'AppType' = 6103, 'rets', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DBDB
    #: - [UDP] DBDB
    dbdb: 'AppType' = 6104, 'dbdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Prima Server
    #: - [UDP] Prima Server
    primaserver: 'AppType' = 6105, 'primaserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MPS Server
    #: - [UDP] MPS Server
    mpsserver: 'AppType' = 6106, 'mpsserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ETC Control
    #: - [UDP] ETC Control
    etc_control: 'AppType' = 6107, 'etc-control', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sercomm-SCAdmin
    #: - [UDP] Sercomm-SCAdmin
    sercomm_scadmin: 'AppType' = 6108, 'sercomm-scadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GLOBECAST-ID
    #: - [UDP] GLOBECAST-ID
    globecast_id: 'AppType' = 6109, 'globecast-id', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP SoftBench CM
    #: - [UDP] HP SoftBench CM
    softcm: 'AppType' = 6110, 'softcm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP SoftBench Sub-Process Control
    #: - [UDP] HP SoftBench Sub-Process Control
    spc: 'AppType' = 6111, 'spc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Desk-Top Sub-Process Control Daemon
    #: - [UDP] Desk-Top Sub-Process Control Daemon
    dtspcd: 'AppType' = 6112, 'dtspcd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Daylite Server
    dayliteserver: 'AppType' = 6113, 'dayliteserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6113: 'AppType' = 6113, 'reserved', TransportProtocol.get('udp')

    #: [TCP] WRspice IPC Service
    wrspice: 'AppType' = 6114, 'wrspice', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6114: 'AppType' = 6114, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Xic IPC Service
    xic: 'AppType' = 6115, 'xic', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6115: 'AppType' = 6115, 'reserved', TransportProtocol.get('udp')

    #: [TCP] XicTools License Manager Service
    xtlserv: 'AppType' = 6116, 'xtlserv', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6116: 'AppType' = 6116, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Daylite Touch Sync
    daylitetouch: 'AppType' = 6117, 'daylitetouch', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6117: 'AppType' = 6117, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Transparent Inter Process Communication
    tipc: 'AppType' = 6118, 'tipc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6118: 'AppType' = 6118, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] SPDY for a faster web
    spdy: 'AppType' = 6121, 'spdy', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6121: 'AppType' = 6121, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Backup Express Web Server
    #: - [UDP] Backup Express Web Server
    bex_webadmin: 'AppType' = 6122, 'bex-webadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Backup Express
    #: - [UDP] Backup Express
    backup_express: 'AppType' = 6123, 'backup-express', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Phlexible Network Backup Service
    #: - [UDP] Phlexible Network Backup Service
    pnbs: 'AppType' = 6124, 'pnbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] The DameWare Mobile Gateway Service
    damewaremobgtwy: 'AppType' = 6130, 'damewaremobgtwy', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6130: 'AppType' = 6130, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] New Boundary Tech WOL
    #: - [UDP] New Boundary Tech WOL
    nbt_wol: 'AppType' = 6133, 'nbt-wol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pulsonix Network License Service
    #: - [UDP] Pulsonix Network License Service
    pulsonixnls: 'AppType' = 6140, 'pulsonixnls', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Meta Corporation License Manager
    #: - [UDP] Meta Corporation License Manager
    meta_corp: 'AppType' = 6141, 'meta-corp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aspen Technology License Manager
    #: - [UDP] Aspen Technology License Manager
    aspentec_lm: 'AppType' = 6142, 'aspentec-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Watershed License Manager
    #: - [UDP] Watershed License Manager
    watershed_lm: 'AppType' = 6143, 'watershed-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] StatSci License Manager - 1
    #: - [UDP] StatSci License Manager - 1
    statsci1_lm: 'AppType' = 6144, 'statsci1-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] StatSci License Manager - 2
    #: - [UDP] StatSci License Manager - 2
    statsci2_lm: 'AppType' = 6145, 'statsci2-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Lone Wolf Systems License Manager
    #: - [UDP] Lone Wolf Systems License Manager
    lonewolf_lm: 'AppType' = 6146, 'lonewolf-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Montage License Manager
    #: - [UDP] Montage License Manager
    montage_lm: 'AppType' = 6147, 'montage-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ricardo North America License Manager
    #: - [UDP] Ricardo North America License Manager
    ricardo_lm_1522: 'AppType' = 1522, 'ricardo-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ricardo North America License Manager
    #: - [UDP] Ricardo North America License Manager
    ricardo_lm_6148: 'AppType' = 6148, 'ricardo-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] tal-pod
    #: - [UDP] tal-pod
    tal_pod: 'AppType' = 6149, 'tal-pod', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] EFB Application Control Interface
    efb_aci: 'AppType' = 6159, 'efb-aci', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6159: 'AppType' = 6159, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Emerson Extensible Control and Management Protocol
    ecmp: 'AppType' = 6160, 'ecmp', TransportProtocol.get('tcp')

    #: [UDP] Emerson Extensible Control and Management Protocol Data
    ecmp_data: 'AppType' = 6160, 'ecmp-data', TransportProtocol.get('udp')

    #: - [TCP] PATROL Internet Srv Mgr
    #: - [UDP] PATROL Internet Srv Mgr
    patrol_ism: 'AppType' = 6161, 'patrol-ism', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PATROL Collector
    #: - [UDP] PATROL Collector
    patrol_coll: 'AppType' = 6162, 'patrol-coll', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Precision Scribe Cnx Port
    #: - [UDP] Precision Scribe Cnx Port
    pscribe: 'AppType' = 6163, 'pscribe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LM-X License Manager by X-Formation
    #: - [UDP] LM-X License Manager by X-Formation
    lm_x: 'AppType' = 6200, 'lm-x', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6201: 'AppType' = 6201, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Management of service nodes in a processing grid for thermodynamic
    #: calculations
    thermo_calc: 'AppType' = 6201, 'thermo-calc', TransportProtocol.get('udp')

    #: - [TCP] QMTP over TLS
    #: - [UDP] QMTP over TLS
    qmtps: 'AppType' = 6209, 'qmtps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Radmind Access Protocol
    #: - [UDP] Radmind Access Protocol
    radmind: 'AppType' = 6222, 'radmind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] JEOL Network Services Data Transport Protocol 1
    jeol_nsdtp_1: 'AppType' = 6241, 'jeol-nsdtp-1', TransportProtocol.get('tcp')

    #: [UDP] JEOL Network Services Dynamic Discovery Protocol 1
    jeol_nsddp_1: 'AppType' = 6241, 'jeol-nsddp-1', TransportProtocol.get('udp')

    #: [TCP] JEOL Network Services Data Transport Protocol 2
    jeol_nsdtp_2: 'AppType' = 6242, 'jeol-nsdtp-2', TransportProtocol.get('tcp')

    #: [UDP] JEOL Network Services Dynamic Discovery Protocol 2
    jeol_nsddp_2: 'AppType' = 6242, 'jeol-nsddp-2', TransportProtocol.get('udp')

    #: [TCP] JEOL Network Services Data Transport Protocol 3
    jeol_nsdtp_3: 'AppType' = 6243, 'jeol-nsdtp-3', TransportProtocol.get('tcp')

    #: [UDP] JEOL Network Services Dynamic Discovery Protocol 3
    jeol_nsddp_3: 'AppType' = 6243, 'jeol-nsddp-3', TransportProtocol.get('udp')

    #: [TCP] JEOL Network Services Data Transport Protocol 4
    jeol_nsdtp_4: 'AppType' = 6244, 'jeol-nsdtp-4', TransportProtocol.get('tcp')

    #: [UDP] JEOL Network Services Dynamic Discovery Protocol 4
    jeol_nsddp_4: 'AppType' = 6244, 'jeol-nsddp-4', TransportProtocol.get('udp')

    #: - [TCP] TL1 Raw Over SSL/TLS
    #: - [UDP] TL1 Raw Over SSL/TLS
    tl1_raw_ssl: 'AppType' = 6251, 'tl1-raw-ssl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TL1 over SSH
    #: - [UDP] TL1 over SSH
    tl1_ssh: 'AppType' = 6252, 'tl1-ssh', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CRIP
    #: - [UDP] CRIP
    crip: 'AppType' = 6253, 'crip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] GridLAB-D User Interface
    gld: 'AppType' = 6267, 'gld', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6267: 'AppType' = 6267, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Grid Authentication
    #: - [UDP] Grid Authentication
    grid: 'AppType' = 6268, 'grid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Grid Authentication Alt
    #: - [UDP] Grid Authentication Alt
    grid_alt: 'AppType' = 6269, 'grid-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC GRX
    #: - [UDP] BMC GRX
    bmc_grx: 'AppType' = 6300, 'bmc-grx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC CONTROL-D LDAP SERVER IANA assigned this well-formed service name
    #:   as a replacement for "bmc_ctd_ldap".
    #: - [TCP] BMC CONTROL-D LDAP SERVER
    #: - [UDP] BMC CONTROL-D LDAP SERVER IANA assigned this well-formed service name
    #:   as a replacement for "bmc_ctd_ldap".
    #: - [UDP] BMC CONTROL-D LDAP SERVER
    bmc_ctd_ldap: 'AppType' = 6301, 'bmc-ctd-ldap', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] Unified Fabric Management Protocol
    #: - [UDP] Unified Fabric Management Protocol
    ufmp: 'AppType' = 6306, 'ufmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Sensor Control Unit Protocol
    scup: 'AppType' = 6315, 'scup', TransportProtocol.get('tcp')

    #: [UDP] Sensor Control Unit Protocol Discovery Protocol
    scup_disc: 'AppType' = 6315, 'scup-disc', TransportProtocol.get('udp')

    #: - [TCP] Ethernet Sensor Communications Protocol
    #: - [UDP] Ethernet Sensor Communications Protocol
    abb_escp: 'AppType' = 6316, 'abb-escp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Navtech Radar Sensor Data Command
    nav_data_cmd: 'AppType' = 6317, 'nav-data-cmd', TransportProtocol.get('tcp')

    #: [UDP] Navtech Radar Sensor Data
    nav_data: 'AppType' = 6317, 'nav-data', TransportProtocol.get('udp')

    #: - [TCP] Double-Take Replication Service
    #: - [UDP] Double-Take Replication Service
    repsvc: 'AppType' = 6320, 'repsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Empress Software Connectivity Server 1
    #: - [UDP] Empress Software Connectivity Server 1
    emp_server1: 'AppType' = 6321, 'emp-server1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Empress Software Connectivity Server 2
    #: - [UDP] Empress Software Connectivity Server 2
    emp_server2: 'AppType' = 6322, 'emp-server2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6323: 'AppType' = 6323, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] HR Device Network Configuration Service
    hrd_ncs: 'AppType' = 6324, 'hrd-ncs', TransportProtocol.get('tcp')

    #: [UDP] HR Device Network service
    hrd_ns_disc: 'AppType' = 6324, 'hrd-ns-disc', TransportProtocol.get('udp')

    #: [TCP] Double-Take Management Service
    dt_mgmtsvc: 'AppType' = 6325, 'dt-mgmtsvc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6325: 'AppType' = 6325, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Double-Take Virtual Recovery Assistant
    dt_vra: 'AppType' = 6326, 'dt-vra', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6326: 'AppType' = 6326, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] sFlow traffic monitoring
    #: - [UDP] sFlow traffic monitoring
    sflow: 'AppType' = 6343, 'sflow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Argus-Spectr security and fire-prevention systems service
    streletz: 'AppType' = 6344, 'streletz', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6344: 'AppType' = 6344, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] gnutella-svc
    #: - [UDP] gnutella-svc
    gnutella_svc: 'AppType' = 6346, 'gnutella-svc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] gnutella-rtr
    #: - [UDP] gnutella-rtr
    gnutella_rtr: 'AppType' = 6347, 'gnutella-rtr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] App Discovery and Access Protocol
    #: - [UDP] App Discovery and Access Protocol
    adap: 'AppType' = 6350, 'adap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PMCS applications
    #: - [UDP] PMCS applications
    pmcs: 'AppType' = 6355, 'pmcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaEdit+ Multi-User
    #: - [UDP] MetaEdit+ Multi-User
    metaedit_mu: 'AppType' = 6360, 'metaedit-mu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Named Data Networking
    ndn: 'AppType' = 6363, 'ndn', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6363: 'AppType' = 6363, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] MetaEdit+ Server Administration
    #: - [UDP] MetaEdit+ Server Administration
    metaedit_se: 'AppType' = 6370, 'metaedit-se', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] An advanced key-value cache and store
    redis: 'AppType' = 6379, 'redis', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6379: 'AppType' = 6379, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Metatude Dialogue Server
    #: - [UDP] Metatude Dialogue Server
    metatude_mds: 'AppType' = 6382, 'metatude-mds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] clariion-evr01
    #: - [UDP] clariion-evr01
    clariion_evr01: 'AppType' = 6389, 'clariion-evr01', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MetaEdit+ WebService API
    #: - [UDP] MetaEdit+ WebService API
    metaedit_ws: 'AppType' = 6390, 'metaedit-ws', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Business Objects CMS contact port
    boe_cms: 'AppType' = 6400, 'boe-cms', TransportProtocol.get('undefined')

    #: [N/A] boe-was
    boe_was: 'AppType' = 6401, 'boe-was', TransportProtocol.get('undefined')

    #: [N/A] boe-eventsrv
    boe_eventsrv: 'AppType' = 6402, 'boe-eventsrv', TransportProtocol.get('undefined')

    #: [N/A] boe-cachesvr
    boe_cachesvr: 'AppType' = 6403, 'boe-cachesvr', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_filesvr: 'AppType' = 6404, 'boe-filesvr', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_pagesvr: 'AppType' = 6405, 'boe-pagesvr', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_processsvr: 'AppType' = 6406, 'boe-processsvr', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_resssvr1: 'AppType' = 6407, 'boe-resssvr1', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_resssvr2: 'AppType' = 6408, 'boe-resssvr2', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_resssvr3: 'AppType' = 6409, 'boe-resssvr3', TransportProtocol.get('undefined')

    #: [N/A] Business Objects Enterprise internal server
    boe_resssvr4: 'AppType' = 6410, 'boe-resssvr4', TransportProtocol.get('undefined')

    #: - [TCP] Faxcom Message Service
    #: - [UDP] Faxcom Message Service
    faxcomservice: 'AppType' = 6417, 'faxcomservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SYserver remote commands
    syserverremote: 'AppType' = 6418, 'syserverremote', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6418: 'AppType' = 6418, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Simple VDR Protocol
    svdrp: 'AppType' = 6419, 'svdrp', TransportProtocol.get('tcp')

    #: [UDP] Simple VDR Protocol Discovery
    svdrp_disc: 'AppType' = 6419, 'svdrp-disc', TransportProtocol.get('udp')

    #: - [TCP] NIM_VDRShell
    #: - [UDP] NIM_VDRShell
    nim_vdrshell: 'AppType' = 6420, 'nim-vdrshell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] NIM_WAN
    #: - [UDP] NIM_WAN
    nim_wan: 'AppType' = 6421, 'nim-wan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PgBouncer
    pgbouncer: 'AppType' = 6432, 'pgbouncer', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6432: 'AppType' = 6432, 'reserved', TransportProtocol.get('udp')

    #: [TCP] heliosd daemon
    heliosd: 'AppType' = 6440, 'heliosd', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6440: 'AppType' = 6440, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6441: 'AppType' = 6441, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Transitory Application Request Protocol
    tarp: 'AppType' = 6442, 'tarp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6442: 'AppType' = 6442, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default HTTPS Domain
    #: - [UDP] Service Registry Default HTTPS Domain
    sun_sr_https: 'AppType' = 6443, 'sun-sr-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Grid Engine Execution Service IANA assigned this well-formed service
    #:   name as a replacement for "sge_execd".
    #: - [TCP] Grid Engine Execution Service
    #: - [UDP] Grid Engine Execution Service IANA assigned this well-formed service
    #:   name as a replacement for "sge_execd".
    #: - [UDP] Grid Engine Execution Service
    sge_execd: 'AppType' = 6445, 'sge-execd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] MySQL Proxy
    #: - [UDP] MySQL Proxy
    mysql_proxy: 'AppType' = 6446, 'mysql-proxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SKIP Certificate Receive
    #: - [UDP] SKIP Certificate Receive
    skip_cert_recv: 'AppType' = 6455, 'skip-cert-recv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SKIP Certificate Send
    #: - [UDP] SKIP Certificate Send
    skip_cert_send: 'AppType' = 6456, 'skip-cert-send', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Port assignment for medical device communication in accordance to IEEE
    #:   11073-20701
    #: - [UDP] Port assignment for medical device communication in accordance to IEEE
    #:   11073-20701
    ieee11073_20701: 'AppType' = 6464, 'ieee11073-20701', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LVision License Manager
    #: - [UDP] LVision License Manager
    lvision_lm: 'AppType' = 6471, 'lvision-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default HTTP Domain
    #: - [UDP] Service Registry Default HTTP Domain
    sun_sr_http: 'AppType' = 6480, 'sun-sr-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Tags
    #: - [UDP] Service Tags
    servicetags: 'AppType' = 6481, 'servicetags', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Logical Domains Management Interface
    #: - [UDP] Logical Domains Management Interface
    ldoms_mgmt: 'AppType' = 6482, 'ldoms-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SunVTS RMI
    #: - [UDP] SunVTS RMI
    sunvts_rmi: 'AppType' = 6483, 'sunvts-rmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default JMS Domain
    #: - [UDP] Service Registry Default JMS Domain
    sun_sr_jms: 'AppType' = 6484, 'sun-sr-jms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default IIOP Domain
    #: - [UDP] Service Registry Default IIOP Domain
    sun_sr_iiop: 'AppType' = 6485, 'sun-sr-iiop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default IIOPS Domain
    #: - [UDP] Service Registry Default IIOPS Domain
    sun_sr_iiops: 'AppType' = 6486, 'sun-sr-iiops', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default IIOPAuth Domain
    #: - [UDP] Service Registry Default IIOPAuth Domain
    sun_sr_iiop_aut: 'AppType' = 6487, 'sun-sr-iiop-aut', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default JMX Domain
    #: - [UDP] Service Registry Default JMX Domain
    sun_sr_jmx: 'AppType' = 6488, 'sun-sr-jmx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Service Registry Default Admin Domain
    #: - [UDP] Service Registry Default Admin Domain
    sun_sr_admin: 'AppType' = 6489, 'sun-sr-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Master
    #: - [UDP] BoKS Master
    boks: 'AppType' = 6500, 'boks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Servc IANA assigned this well-formed service name as a
    #:   replacement for "boks_servc".
    #: - [TCP] BoKS Servc
    #: - [UDP] BoKS Servc IANA assigned this well-formed service name as a
    #:   replacement for "boks_servc".
    #: - [UDP] BoKS Servc
    boks_servc: 'AppType' = 6501, 'boks-servc', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Servm IANA assigned this well-formed service name as a
    #:   replacement for "boks_servm".
    #: - [TCP] BoKS Servm
    #: - [UDP] BoKS Servm IANA assigned this well-formed service name as a
    #:   replacement for "boks_servm".
    #: - [UDP] BoKS Servm
    boks_servm: 'AppType' = 6502, 'boks-servm', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Clntd IANA assigned this well-formed service name as a
    #:   replacement for "boks_clntd".
    #: - [TCP] BoKS Clntd
    #: - [UDP] BoKS Clntd IANA assigned this well-formed service name as a
    #:   replacement for "boks_clntd".
    #: - [UDP] BoKS Clntd
    boks_clntd: 'AppType' = 6503, 'boks-clntd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6504: 'AppType' = 6504, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] BoKS Admin Private Port IANA assigned this well-formed service name as
    #:   a replacement for "badm_priv".
    #: - [TCP] BoKS Admin Private Port
    #: - [UDP] BoKS Admin Private Port IANA assigned this well-formed service name as
    #:   a replacement for "badm_priv".
    #: - [UDP] BoKS Admin Private Port
    badm_priv: 'AppType' = 6505, 'badm-priv', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Admin Public Port IANA assigned this well-formed service name as
    #:   a replacement for "badm_pub".
    #: - [TCP] BoKS Admin Public Port
    #: - [UDP] BoKS Admin Public Port IANA assigned this well-formed service name as
    #:   a replacement for "badm_pub".
    #: - [UDP] BoKS Admin Public Port
    badm_pub: 'AppType' = 6506, 'badm-pub', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Dir Server, Private Port IANA assigned this well-formed service
    #:   name as a replacement for "bdir_priv".
    #: - [TCP] BoKS Dir Server, Private Port
    #: - [UDP] BoKS Dir Server, Private Port IANA assigned this well-formed service
    #:   name as a replacement for "bdir_priv".
    #: - [UDP] BoKS Dir Server, Private Port
    bdir_priv: 'AppType' = 6507, 'bdir-priv', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] BoKS Dir Server, Public Port IANA assigned this well-formed service
    #:   name as a replacement for "bdir_pub".
    #: - [TCP] BoKS Dir Server, Public Port
    #: - [UDP] BoKS Dir Server, Public Port IANA assigned this well-formed service
    #:   name as a replacement for "bdir_pub".
    #: - [UDP] BoKS Dir Server, Public Port
    bdir_pub: 'AppType' = 6508, 'bdir-pub', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] MGCS-MFP Port
    #: - [UDP] MGCS-MFP Port
    mgcs_mfp_port: 'AppType' = 6509, 'mgcs-mfp-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MCER Port
    #: - [UDP] MCER Port
    mcer_port: 'AppType' = 6510, 'mcer-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6511: 'AppType' = 6511, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Datagram Congestion Control Protocol Encapsulation for NAT Traversal
    #: [:rfc:`6773`]
    dccp_udp: 'AppType' = 6511, 'dccp-udp', TransportProtocol.get('udp')

    #: [TCP] NETCONF over TLS [:rfc:`7589`][RFC-ietf-netconf-over-tls13-04]
    netconf_tls: 'AppType' = 6513, 'netconf-tls', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6513: 'AppType' = 6513, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Syslog over TLS [:rfc:`5425`]
    #: - [UDP] syslog over DTLS [:rfc:`6012`]
    #: - [DCCP] syslog over DTLS [:rfc:`6012`]
    syslog_tls: 'AppType' = 6514, 'syslog-tls', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('dccp')

    #: - [TCP] Elipse RPC Protocol
    #: - [UDP] Elipse RPC Protocol
    elipse_rec: 'AppType' = 6515, 'elipse-rec', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] lds_distrib
    #: - [UDP] lds_distrib
    lds_distrib: 'AppType' = 6543, 'lds-distrib', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LDS Dump Service
    #: - [UDP] LDS Dump Service
    lds_dump: 'AppType' = 6544, 'lds-dump', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 6547
    #: - [UDP] APC 6547
    apc_6547: 'AppType' = 6547, 'apc-6547', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 6548
    #: - [UDP] APC 6548
    apc_6548: 'AppType' = 6548, 'apc-6548', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 6549
    #: - [UDP] APC 6549
    apc_6549: 'AppType' = 6549, 'apc-6549', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] fg-sysupdate
    #: - [UDP] fg-sysupdate
    fg_sysupdate: 'AppType' = 6550, 'fg-sysupdate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Software Update Manager
    #: - [UDP] Software Update Manager
    sum: 'AppType' = 6551, 'sum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Checkmk Monitoring Agent
    checkmk_agent: 'AppType' = 6556, 'checkmk-agent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6556: 'AppType' = 6556, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6557: 'AppType' = 6557, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP]
    #: - [UDP]
    xdsxdm: 'AppType' = 6558, 'xdsxdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SANE Control Port
    #: - [UDP] SANE Control Port
    sane_port: 'AppType' = 6566, 'sane-port', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Reserved
    reserved_6567: 'AppType' = 6567, 'reserved', TransportProtocol.get('undefined')

    #: - [TCP] CanIt Storage Manager IANA assigned this well-formed service name as a
    #:   replacement for "canit_store".
    #: - [TCP] CanIt Storage Manager
    canit_store: 'AppType' = 6568, 'canit-store', TransportProtocol.get('tcp') | TransportProtocol.get('tcp')

    #: [UDP] Roaring Penguin IP Address Reputation Collection
    rp_reputation: 'AppType' = 6568, 'rp-reputation', TransportProtocol.get('udp')

    #: - [TCP] Affiliate
    #: - [UDP] Affiliate
    affiliate: 'AppType' = 6579, 'affiliate', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Parsec Masterserver
    #: - [UDP] Parsec Masterserver
    parsec_master: 'AppType' = 6580, 'parsec-master', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Parsec Peer-to-Peer
    #: - [UDP] Parsec Peer-to-Peer
    parsec_peer: 'AppType' = 6581, 'parsec-peer', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Parsec Gameserver
    #: - [UDP] Parsec Gameserver
    parsec_game: 'AppType' = 6582, 'parsec-game', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JOA Jewel Suite
    #: - [UDP] JOA Jewel Suite
    joajewelsuite: 'AppType' = 6583, 'joajewelsuite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6588: 'AppType' = 6588, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Microsoft Hyper-V Live Migration
    mshvlm: 'AppType' = 6600, 'mshvlm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6600: 'AppType' = 6600, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Microsoft Threat Management Gateway SSTP
    mstmg_sstp: 'AppType' = 6601, 'mstmg-sstp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6601: 'AppType' = 6601, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Windows WSS Communication Framework
    wsscomfrmwk: 'AppType' = 6602, 'wsscomfrmwk', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6602: 'AppType' = 6602, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ODETTE-FTP over TLS/SSL [:rfc:`5024`]
    #: - [UDP] ODETTE-FTP over TLS/SSL [:rfc:`5024`]
    odette_ftps: 'AppType' = 6619, 'odette-ftps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kerberos V5 FTP Data
    #: - [UDP] Kerberos V5 FTP Data
    kftp_data: 'AppType' = 6620, 'kftp-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kerberos V5 FTP Control
    #: - [UDP] Kerberos V5 FTP Control
    kftp: 'AppType' = 6621, 'kftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multicast FTP
    #: - [UDP] Multicast FTP
    mcftp: 'AppType' = 6622, 'mcftp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Kerberos V5 Telnet
    #: - [UDP] Kerberos V5 Telnet
    ktelnet: 'AppType' = 6623, 'ktelnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DataScaler database
    datascaler_db: 'AppType' = 6624, 'datascaler-db', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6624: 'AppType' = 6624, 'reserved', TransportProtocol.get('udp')

    #: [TCP] DataScaler control
    datascaler_ctl: 'AppType' = 6625, 'datascaler-ctl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6625: 'AppType' = 6625, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] WAGO Service and Update
    #: - [UDP] WAGO Service and Update
    wago_service: 'AppType' = 6626, 'wago-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Allied Electronics NeXGen
    #: - [UDP] Allied Electronics NeXGen
    nexgen: 'AppType' = 6627, 'nexgen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AFE Stock Channel M/C
    #: - [UDP] AFE Stock Channel M/C
    afesc_mc: 'AppType' = 6628, 'afesc-mc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secondary, (non ANDI) multi-protocol multi-function interface to the
    #:   Allied ANDI-based family of forecourt controllers
    #: - [UDP] Secondary, (non ANDI) multi-protocol multi-function interface to the
    #:   Allied ANDI-based family of forecourt controllers
    nexgen_aux: 'AppType' = 6629, 'nexgen-aux', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6630: 'AppType' = 6630, 'unassigned', TransportProtocol.get('undefined')

    #: [N/A] Unassigned
    unassigned_6631: 'AppType' = 6631, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] eGenix mxODBC Connect
    mxodbc_connect: 'AppType' = 6632, 'mxodbc-connect', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6632: 'AppType' = 6632, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6633: 'AppType' = 6633, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cisco vPath Services Overlay
    cisco_vpath_tun: 'AppType' = 6633, 'cisco-vpath-tun', TransportProtocol.get('udp')

    #: [UDP] MPLS Performance Measurement out-of-band response
    mpls_pm: 'AppType' = 6634, 'mpls-pm', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6634: 'AppType' = 6634, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] Reserved
    reserved_6635: 'AppType' = 6635, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Encapsulate MPLS packets in UDP tunnels. [:rfc:`7510`]
    mpls_udp: 'AppType' = 6635, 'mpls-udp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6636: 'AppType' = 6636, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Encapsulate MPLS packets in UDP tunnels with DTLS. [:rfc:`7510`]
    mpls_udp_dtls: 'AppType' = 6636, 'mpls-udp-dtls', TransportProtocol.get('udp')

    #: [TCP] Open vSwitch Database protocol [:rfc:`7047`]
    ovsdb: 'AppType' = 6640, 'ovsdb', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6640: 'AppType' = 6640, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] OpenFlow
    #: - [UDP] OpenFlow
    openflow: 'AppType' = 6653, 'openflow', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6654: 'AppType' = 6654, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] PC SOFT - Software factory UI/manager
    pcs_sf_ui_man: 'AppType' = 6655, 'pcs-sf-ui-man', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6655: 'AppType' = 6655, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Emergency Message Control Service
    emgmsg: 'AppType' = 6656, 'emgmsg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6656: 'AppType' = 6656, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6657: 'AppType' = 6657, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] PalCom Discovery
    palcom_disc: 'AppType' = 6657, 'palcom-disc', TransportProtocol.get('udp')

    #: - [TCP] Vocaltec Global Online Directory
    #: - [UDP] Vocaltec Global Online Directory
    vocaltec_gold: 'AppType' = 6670, 'vocaltec-gold', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] P4P Portal Service
    #: - [UDP] P4P Portal Service
    p4p_portal: 'AppType' = 6671, 'p4p-portal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] vision_server IANA assigned this well-formed service name as a
    #:   replacement for "vision_server".
    #: - [TCP] vision_server
    #: - [UDP] vision_server IANA assigned this well-formed service name as a
    #:   replacement for "vision_server".
    #: - [UDP] vision_server
    vision_server: 'AppType' = 6672, 'vision-server', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: - [TCP] vision_elmd IANA assigned this well-formed service name as a
    #:   replacement for "vision_elmd".
    #: - [TCP] vision_elmd
    #: - [UDP] vision_elmd IANA assigned this well-formed service name as a
    #:   replacement for "vision_elmd".
    #: - [UDP] vision_elmd
    vision_elmd: 'AppType' = 6673, 'vision-elmd', TransportProtocol.get('tcp') | TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('udp')

    #: [TCP] Viscount Freedom Bridge Protocol
    vfbp: 'AppType' = 6678, 'vfbp', TransportProtocol.get('tcp')

    #: [UDP] Viscount Freedom Bridge Discovery
    vfbp_disc: 'AppType' = 6678, 'vfbp-disc', TransportProtocol.get('udp')

    #: - [TCP] Osorno Automation
    #: - [UDP] Osorno Automation
    osaut: 'AppType' = 6679, 'osaut', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CleverView for cTrace Message Service
    clever_ctrace: 'AppType' = 6687, 'clever-ctrace', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6687: 'AppType' = 6687, 'reserved', TransportProtocol.get('udp')

    #: [TCP] CleverView for TCP/IP Message Service
    clever_tcpip: 'AppType' = 6688, 'clever-tcpip', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6688: 'AppType' = 6688, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Tofino Security Appliance
    #: - [UDP] Tofino Security Appliance
    tsa: 'AppType' = 6689, 'tsa', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CLEVERDetect Message Service
    cleverdetect: 'AppType' = 6690, 'cleverdetect', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6690: 'AppType' = 6690, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6696: 'AppType' = 6696, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Babel Routing Protocol [:rfc:`8966`]
    babel: 'AppType' = 6696, 'babel', TransportProtocol.get('udp')

    #: [TCP] Internet Relay Chat via TLS/SSL [:rfc:`7194`]
    ircs_u: 'AppType' = 6697, 'ircs-u', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6697: 'AppType' = 6697, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6698: 'AppType' = 6698, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Reserved
    reserved_6699: 'AppType' = 6699, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Babel Routing Protocol over DTLS [:rfc:`8968`]
    babel_dtls: 'AppType' = 6699, 'babel-dtls', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_6700: 'AppType' = 6700, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] KTI/ICAD Nameserver
    #: - [UDP] KTI/ICAD Nameserver
    kti_icad_srvr: 'AppType' = 6701, 'kti-icad-srvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Unassigned
    unassigned_6701: 'AppType' = 6701, 'unassigned', TransportProtocol.get('sctp')

    #: - [TCP] e-Design network
    #: - [UDP] e-Design network
    e_design_net: 'AppType' = 6702, 'e-design-net', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] Unassigned
    unassigned_6702: 'AppType' = 6702, 'unassigned', TransportProtocol.get('sctp')

    #: - [TCP] e-Design web
    #: - [UDP] e-Design web
    e_design_web: 'AppType' = 6703, 'e-design-web', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [UDP] Reserved
    #: - [TCP] Reserved
    reserved_6704: 'AppType' = 6704, 'reserved', TransportProtocol.get('udp') | TransportProtocol.get('tcp')

    #: [SCTP] ForCES HP (High Priority) channel [:rfc:`5811`]
    frc_hp: 'AppType' = 6704, 'frc-hp', TransportProtocol.get('sctp')

    #: - [UDP] Reserved
    #: - [TCP] Reserved
    reserved_6705: 'AppType' = 6705, 'reserved', TransportProtocol.get('udp') | TransportProtocol.get('tcp')

    #: [SCTP] ForCES MP (Medium Priority) channel [:rfc:`5811`]
    frc_mp: 'AppType' = 6705, 'frc-mp', TransportProtocol.get('sctp')

    #: - [UDP] Reserved
    #: - [TCP] Reserved
    reserved_6706: 'AppType' = 6706, 'reserved', TransportProtocol.get('udp') | TransportProtocol.get('tcp')

    #: [SCTP] ForCES LP (Low priority) channel [:rfc:`5811`]
    frc_lp: 'AppType' = 6706, 'frc-lp', TransportProtocol.get('sctp')

    #: - [TCP] Internet Backplane Protocol
    #: - [UDP] Internet Backplane Protocol
    ibprotocol: 'AppType' = 6714, 'ibprotocol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Fibotrader Communications
    #: - [UDP] Fibotrader Communications
    fibotrader_com: 'AppType' = 6715, 'fibotrader-com', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Princity Agent
    princity_agent: 'AppType' = 6716, 'princity-agent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6716: 'AppType' = 6716, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] BMC PERFORM AGENT
    #: - [UDP] BMC PERFORM AGENT
    bmc_perf_agent: 'AppType' = 6767, 'bmc-perf-agent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] BMC PERFORM MGRD
    #: - [UDP] BMC PERFORM MGRD
    bmc_perf_mgrd: 'AppType' = 6768, 'bmc-perf-mgrd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ADInstruments GxP Server
    #: - [UDP] ADInstruments GxP Server
    adi_gxp_srvprt: 'AppType' = 6769, 'adi-gxp-srvprt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PolyServe http
    #: - [UDP] PolyServe http
    plysrv_http: 'AppType' = 6770, 'plysrv-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PolyServe https
    #: - [UDP] PolyServe https
    plysrv_https: 'AppType' = 6771, 'plysrv-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] netTsunami Tracker
    ntz_tracker: 'AppType' = 6777, 'ntz-tracker', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6777: 'AppType' = 6777, 'reserved', TransportProtocol.get('udp')

    #: [TCP] netTsunami p2p storage system
    ntz_p2p_storage: 'AppType' = 6778, 'ntz-p2p-storage', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6778: 'AppType' = 6778, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_6784: 'AppType' = 6784, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Bidirectional Forwarding Detection (BFD) on Link Aggregation Group
    #: (LAG) Interfaces [:rfc:`7130`]
    bfd_lag: 'AppType' = 6784, 'bfd-lag', TransportProtocol.get('udp')

    #: - [TCP] DGPF Individual Exchange
    #: - [UDP] DGPF Individual Exchange
    dgpf_exchg: 'AppType' = 6785, 'dgpf-exchg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Java Web Console JMX
    #: - [UDP] Sun Java Web Console JMX
    smc_jmx: 'AppType' = 6786, 'smc-jmx', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Web Console Admin
    #: - [UDP] Sun Web Console Admin
    smc_admin: 'AppType' = 6787, 'smc-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SMC-HTTP
    #: - [UDP] SMC-HTTP
    smc_http: 'AppType' = 6788, 'smc-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] GSS-API for the Oracle Remote Administration Daemon
    radg: 'AppType' = 6789, 'radg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6789: 'AppType' = 6789, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] HNMP
    #: - [UDP] HNMP
    hnmp: 'AppType' = 6790, 'hnmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Halcyon Network Manager
    #: - [UDP] Halcyon Network Manager
    hnm: 'AppType' = 6791, 'hnm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ACNET Control System Protocol
    #: - [UDP] ACNET Control System Protocol
    acnet: 'AppType' = 6801, 'acnet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] PenTBox Secure IM Protocol
    pentbox_sim: 'AppType' = 6817, 'pentbox-sim', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6817: 'AppType' = 6817, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ambit-lm
    #: - [UDP] ambit-lm
    ambit_lm: 'AppType' = 6831, 'ambit-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netmo Default
    #: - [UDP] Netmo Default
    netmo_default: 'AppType' = 6841, 'netmo-default', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Netmo HTTP
    #: - [UDP] Netmo HTTP
    netmo_http: 'AppType' = 6842, 'netmo-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ICCRUSHMORE
    #: - [UDP] ICCRUSHMORE
    iccrushmore: 'AppType' = 6850, 'iccrushmore', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Acctopus Command Channel
    acctopus_cc: 'AppType' = 6868, 'acctopus-cc', TransportProtocol.get('tcp')

    #: [UDP] Acctopus Status
    acctopus_st: 'AppType' = 6868, 'acctopus-st', TransportProtocol.get('udp')

    #: - [TCP] MUSE
    #: - [UDP] MUSE
    muse: 'AppType' = 6888, 'muse', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] R*TIME Viewer Data Interface
    rtimeviewer: 'AppType' = 6900, 'rtimeviewer', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6900: 'AppType' = 6900, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Novell Jetstream messaging protocol
    jetstream: 'AppType' = 6901, 'jetstream', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6901: 'AppType' = 6901, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Ping with RX/TX latency/loss split
    #: - [UDP] Ping with RX/TX latency/loss split
    split_ping: 'AppType' = 6924, 'split-ping', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] EthoScan Service
    #: - [UDP] EthoScan Service
    ethoscan: 'AppType' = 6935, 'ethoscan', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] XenSource Management Service
    #: - [UDP] XenSource Management Service
    xsmsvc: 'AppType' = 6936, 'xsmsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Biometrics Server
    #: - [UDP] Biometrics Server
    bioserver: 'AppType' = 6946, 'bioserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OTLP
    #: - [UDP] OTLP
    otlp: 'AppType' = 6951, 'otlp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JMACT3
    #: - [UDP] JMACT3
    jmact3: 'AppType' = 6961, 'jmact3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] jmevt2
    #: - [UDP] jmevt2
    jmevt2: 'AppType' = 6962, 'jmevt2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swismgr1
    #: - [UDP] swismgr1
    swismgr1: 'AppType' = 6963, 'swismgr1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swismgr2
    #: - [UDP] swismgr2
    swismgr2: 'AppType' = 6964, 'swismgr2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swistrap
    #: - [UDP] swistrap
    swistrap: 'AppType' = 6965, 'swistrap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] swispol
    #: - [UDP] swispol
    swispol: 'AppType' = 6966, 'swispol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] acmsoda
    #: - [UDP] acmsoda
    acmsoda: 'AppType' = 6969, 'acmsoda', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Conductor test coordination protocol
    conductor: 'AppType' = 6970, 'conductor', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_6970: 'AppType' = 6970, 'reserved', TransportProtocol.get('udp')

    #: [SCTP] conductor for multiplex
    conductor_mpx: 'AppType' = 6970, 'conductor-mpx', TransportProtocol.get('sctp')

    #: [TCP] Reserved
    reserved_6980: 'AppType' = 6980, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] QoS-extended OLSR protocol
    qolyester: 'AppType' = 6980, 'qolyester', TransportProtocol.get('udp')

    #: - [TCP] Mobility XE Protocol
    #: - [UDP] Mobility XE Protocol
    mobilitysrv: 'AppType' = 6997, 'mobilitysrv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IATP-highPri
    #: - [UDP] IATP-highPri
    iatp_highpri: 'AppType' = 6998, 'iatp-highpri', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IATP-normalPri
    #: - [UDP] IATP-normalPri
    iatp_normalpri: 'AppType' = 6999, 'iatp-normalpri', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] file server itself
    #: - [UDP] file server itself
    afs3_fileserver: 'AppType' = 7000, 'afs3-fileserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] callbacks to cache managers
    #: - [UDP] callbacks to cache managers
    afs3_callback: 'AppType' = 7001, 'afs3-callback', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] users & groups database
    #: - [UDP] users & groups database
    afs3_prserver: 'AppType' = 7002, 'afs3-prserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] volume location database
    #: - [UDP] volume location database
    afs3_vlserver: 'AppType' = 7003, 'afs3-vlserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AFS/Kerberos authentication service
    #: - [UDP] AFS/Kerberos authentication service
    afs3_kaserver: 'AppType' = 7004, 'afs3-kaserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] volume managment server
    #: - [UDP] volume managment server
    afs3_volser: 'AppType' = 7005, 'afs3-volser', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] error interpretation service
    #: - [UDP] error interpretation service
    afs3_errors: 'AppType' = 7006, 'afs3-errors', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] basic overseer process
    #: - [UDP] basic overseer process
    afs3_bos: 'AppType' = 7007, 'afs3-bos', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] server-to-server updater
    #: - [UDP] server-to-server updater
    afs3_update: 'AppType' = 7008, 'afs3-update', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] remote cache manager service
    #: - [UDP] remote cache manager service
    afs3_rmtsys: 'AppType' = 7009, 'afs3-rmtsys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] onlinet uninterruptable power supplies
    #: - [UDP] onlinet uninterruptable power supplies
    ups_onlinet: 'AppType' = 7010, 'ups-onlinet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talon Discovery Port
    #: - [UDP] Talon Discovery Port
    talon_disc: 'AppType' = 7011, 'talon-disc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talon Engine
    #: - [UDP] Talon Engine
    talon_engine: 'AppType' = 7012, 'talon-engine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microtalon Discovery
    #: - [UDP] Microtalon Discovery
    microtalon_dis: 'AppType' = 7013, 'microtalon-dis', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microtalon Communications
    #: - [UDP] Microtalon Communications
    microtalon_com: 'AppType' = 7014, 'microtalon-com', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Talon Webserver
    #: - [UDP] Talon Webserver
    talon_webserver: 'AppType' = 7015, 'talon-webserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SPG Controls Carrier
    #: - [UDP] SPG Controls Carrier
    spg: 'AppType' = 7016, 'spg', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] GeneRic Autonomic Signaling Protocol [:rfc:`8990`]
    #: - [UDP] GeneRic Autonomic Signaling Protocol [:rfc:`8990`]
    grasp: 'AppType' = 7017, 'grasp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] FISA Service
    fisa_svc: 'AppType' = 7018, 'fisa-svc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7018: 'AppType' = 7018, 'reserved', TransportProtocol.get('udp')

    #: [TCP] doceri drawing service control
    doceri_ctl: 'AppType' = 7019, 'doceri-ctl', TransportProtocol.get('tcp')

    #: [UDP] doceri drawing service screen view
    doceri_view: 'AppType' = 7019, 'doceri-view', TransportProtocol.get('udp')

    #: - [TCP] DP Serve
    #: - [UDP] DP Serve
    dpserve: 'AppType' = 7020, 'dpserve', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DP Serve Admin
    #: - [UDP] DP Serve Admin
    dpserveadmin: 'AppType' = 7021, 'dpserveadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CT Discovery Protocol
    #: - [UDP] CT Discovery Protocol
    ctdp: 'AppType' = 7022, 'ctdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Comtech T2 NMCS
    #: - [UDP] Comtech T2 NMCS
    ct2nmcs: 'AppType' = 7023, 'ct2nmcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vormetric service
    #: - [UDP] Vormetric service
    vmsvc: 'AppType' = 7024, 'vmsvc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vormetric Service II
    #: - [UDP] Vormetric Service II
    vmsvc_2: 'AppType' = 7025, 'vmsvc-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Loreji Webhosting Panel
    loreji_panel: 'AppType' = 7026, 'loreji-panel', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7026: 'AppType' = 7026, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] ObjectPlanet probe
    #: - [UDP] ObjectPlanet probe
    op_probe: 'AppType' = 7030, 'op-probe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] IPOSPLANET retailing multi devices protocol
    iposplanet: 'AppType' = 7031, 'iposplanet', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7031: 'AppType' = 7031, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7040: 'AppType' = 7040, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Quest application level network service discovery
    quest_disc: 'AppType' = 7040, 'quest-disc', TransportProtocol.get('udp')

    #: - [TCP] ARCP
    #: - [UDP] ARCP
    arcp: 'AppType' = 7070, 'arcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IWGADTS Aircraft Housekeeping Message
    #: - [UDP] IWGADTS Aircraft Housekeeping Message
    iwg1: 'AppType' = 7071, 'iwg1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] iba Device Configuration Protocol
    iba_cfg: 'AppType' = 7072, 'iba-cfg', TransportProtocol.get('tcp')

    #: [UDP] iba Device Configuration Protocol
    iba_cfg_disc: 'AppType' = 7072, 'iba-cfg-disc', TransportProtocol.get('udp')

    #: [TCP] MarTalk protocol
    martalk: 'AppType' = 7073, 'martalk', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7073: 'AppType' = 7073, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] EmpowerID Communication
    #: - [UDP] EmpowerID Communication
    empowerid: 'AppType' = 7080, 'empowerid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7088: 'AppType' = 7088, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Zixi live video transport protocol
    zixi_transport: 'AppType' = 7088, 'zixi-transport', TransportProtocol.get('udp')

    #: [UDP] Java Discovery Protocol
    jdp_disc: 'AppType' = 7095, 'jdp-disc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7095: 'AppType' = 7095, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] lazy-ptop
    #: - [UDP] lazy-ptop
    lazy_ptop: 'AppType' = 7099, 'lazy-ptop', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] X Font Service
    #: - [UDP] X Font Service
    font_service: 'AppType' = 7100, 'font-service', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Embedded Light Control Network
    #: - [UDP] Embedded Light Control Network
    elcn: 'AppType' = 7101, 'elcn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7107: 'AppType' = 7107, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] AES-X170
    aes_x170: 'AppType' = 7107, 'aes-x170', TransportProtocol.get('udp')

    #: [TCP] Encrypted chat and file transfer service
    rothaga: 'AppType' = 7117, 'rothaga', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7117: 'AppType' = 7117, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Virtual Prototypes License Manager
    #: - [UDP] Virtual Prototypes License Manager
    virprot_lm: 'AppType' = 7121, 'virprot-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7122: 'AppType' = 7122, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] End-to-end TLS Relay Control Connection
    snif: 'AppType' = 7123, 'snif', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7123: 'AppType' = 7123, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] intelligent data manager
    #: - [UDP] intelligent data manager
    scenidm: 'AppType' = 7128, 'scenidm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Catalog Content Search
    #: - [UDP] Catalog Content Search
    scenccs: 'AppType' = 7129, 'scenccs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA BSM Comm
    #: - [UDP] CA BSM Comm
    cabsm_comm: 'AppType' = 7161, 'cabsm-comm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA Storage Manager
    #: - [UDP] CA Storage Manager
    caistoragemgr: 'AppType' = 7162, 'caistoragemgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA Connection Broker
    #: - [UDP] CA Connection Broker
    cacsambroker: 'AppType' = 7163, 'cacsambroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] File System Repository Agent
    #: - [UDP] File System Repository Agent
    fsr: 'AppType' = 7164, 'fsr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Document WCF Server
    #: - [UDP] Document WCF Server
    doc_server: 'AppType' = 7165, 'doc-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Aruba eDiscovery Server
    #: - [UDP] Aruba eDiscovery Server
    aruba_server: 'AppType' = 7166, 'aruba-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CA SRM Agent
    casrmagent: 'AppType' = 7167, 'casrmagent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7167: 'AppType' = 7167, 'reserved', TransportProtocol.get('udp')

    #: [TCP] cncKadServer DB & Inventory Services
    cnckadserver: 'AppType' = 7168, 'cnckadserver', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7168: 'AppType' = 7168, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Consequor Consulting Process Integration Bridge
    #: - [UDP] Consequor Consulting Process Integration Bridge
    ccag_pib: 'AppType' = 7169, 'ccag-pib', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Adaptive Name/Service Resolution
    #: - [UDP] Adaptive Name/Service Resolution
    nsrp: 'AppType' = 7170, 'nsrp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Discovery and Retention Mgt Production
    #: - [UDP] Discovery and Retention Mgt Production
    drm_production: 'AppType' = 7171, 'drm-production', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Port used for MetalBend programmable interface
    metalbend: 'AppType' = 7172, 'metalbend', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7172: 'AppType' = 7172, 'reserved', TransportProtocol.get('udp')

    #: [TCP] zSecure Server
    zsecure: 'AppType' = 7173, 'zsecure', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7173: 'AppType' = 7173, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Clutild
    #: - [UDP] Clutild
    clutild: 'AppType' = 7174, 'clutild', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Janus Guidewire Enterprise Discovery Service Bus
    janus_disc: 'AppType' = 7181, 'janus-disc', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7181: 'AppType' = 7181, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] FODMS FLIP
    #: - [UDP] FODMS FLIP
    fodms: 'AppType' = 7200, 'fodms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DLIP
    #: - [UDP] DLIP
    dlip: 'AppType' = 7201, 'dlip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Inter-Channel Termination Protocol (ICTP) for multi-wavelength PON
    #: (Passive Optical Network) systems [TR-352][Inter-Channel-Termination
    #: protocol (ICTP) is a peer-to-peer protocol with a binary message format that
    #: is used by a set of optical line terminal (OLT) channel terminations (CT)
    #: comprising a 40-Gigabit-capable next-generation passive optical network (NG-
    #: PON2) system to exchange channel profile configuration and status
    #: information and to coordinate the operation of tunable optical network units
    #: (ONUs) in a multi-wavelength channel environment.]
    pon_ictp: 'AppType' = 7202, 'pon-ictp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7202: 'AppType' = 7202, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Communication ports for PaperStream Server services
    ps_server: 'AppType' = 7215, 'ps-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7215: 'AppType' = 7215, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PaperStream Capture Professional
    ps_capture_pro: 'AppType' = 7216, 'ps-capture-pro', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7216: 'AppType' = 7216, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Registry A & M Protocol
    #: - [UDP] Registry A & M Protocol
    ramp: 'AppType' = 7227, 'ramp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Citrix Universal Printing Port
    citrixupp: 'AppType' = 7228, 'citrixupp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7228: 'AppType' = 7228, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Citrix UPP Gateway
    citrixuppg: 'AppType' = 7229, 'citrixuppg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7229: 'AppType' = 7229, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Traffic forwarding for Okta cloud infra
    asa_gateways: 'AppType' = 7234, 'asa-gateways', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7234: 'AppType' = 7234, 'reserved', TransportProtocol.get('udp')

    #: [UDP] ASP Coordination Protocol
    aspcoordination: 'AppType' = 7235, 'aspcoordination', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7235: 'AppType' = 7235, 'reserved', TransportProtocol.get('tcp')

    #: [TCP] Wi-Fi Alliance Wi-Fi Display Protocol
    display: 'AppType' = 7236, 'display', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7236: 'AppType' = 7236, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PADS (Public Area Display System) Server
    pads: 'AppType' = 7237, 'pads', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7237: 'AppType' = 7237, 'reserved', TransportProtocol.get('udp')

    #: [TCP] FrontRow Calypso Human Interface Control Protocol
    frc_hicp: 'AppType' = 7244, 'frc-hicp', TransportProtocol.get('tcp')

    #: [UDP] FrontRow Calypso Human Interface Control Protocol
    frc_hicp_disc: 'AppType' = 7244, 'frc-hicp-disc', TransportProtocol.get('udp')

    #: - [TCP] Calypso Network Access Protocol
    #: - [UDP] Calypso Network Access Protocol
    cnap: 'AppType' = 7262, 'cnap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] WatchMe Monitoring 7272
    #: - [UDP] WatchMe Monitoring 7272
    watchme_7272: 'AppType' = 7272, 'watchme-7272', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA Roaming Location
    #: - [UDP] OMA Roaming Location
    oma_rlp: 'AppType' = 7273, 'oma-rlp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA Roaming Location SEC
    #: - [UDP] OMA Roaming Location SEC
    oma_rlp_s: 'AppType' = 7274, 'oma-rlp-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA UserPlane Location
    #: - [UDP] OMA UserPlane Location
    oma_ulp: 'AppType' = 7275, 'oma-ulp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA Internal Location Protocol
    #: - [UDP] OMA Internal Location Protocol
    oma_ilp: 'AppType' = 7276, 'oma-ilp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA Internal Location Secure Protocol
    #: - [UDP] OMA Internal Location Secure Protocol
    oma_ilp_s: 'AppType' = 7277, 'oma-ilp-s', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OMA Dynamic Content Delivery over CBS
    #: - [UDP] OMA Dynamic Content Delivery over CBS
    oma_dcdocbs: 'AppType' = 7278, 'oma-dcdocbs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Citrix Licensing
    #: - [UDP] Citrix Licensing
    ctxlic: 'AppType' = 7279, 'ctxlic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITACTIONSERVER 1
    #: - [UDP] ITACTIONSERVER 1
    itactionserver1: 'AppType' = 7280, 'itactionserver1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ITACTIONSERVER 2
    #: - [UDP] ITACTIONSERVER 2
    itactionserver2: 'AppType' = 7281, 'itactionserver2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] eventACTION/ussACTION (MZCA) server
    mzca_action: 'AppType' = 7282, 'mzca-action', TransportProtocol.get('tcp')

    #: [UDP] eventACTION/ussACTION (MZCA) alert
    mzca_alert: 'AppType' = 7282, 'mzca-alert', TransportProtocol.get('udp')

    #: [TCP] General Statistics Rendezvous Protocol
    genstat: 'AppType' = 7283, 'genstat', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7283: 'AppType' = 7283, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] LifeKeeper Communications
    #: - [UDP] LifeKeeper Communications
    lcm_server: 'AppType' = 7365, 'lcm-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mind-file system server
    #: - [UDP] mind-file system server
    mindfilesys: 'AppType' = 7391, 'mindfilesys', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] mrss-rendezvous server
    #: - [UDP] mrss-rendezvous server
    mrssrendezvous: 'AppType' = 7392, 'mrssrendezvous', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] nFoldMan Remote Publish
    #: - [UDP] nFoldMan Remote Publish
    nfoldman: 'AppType' = 7393, 'nfoldman', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] File system export of backup images
    #: - [UDP] File system export of backup images
    fse: 'AppType' = 7394, 'fse', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] winqedit
    #: - [UDP] winqedit
    winqedit: 'AppType' = 7395, 'winqedit', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7396: 'AppType' = 7396, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Hexarc Command Language
    #: - [UDP] Hexarc Command Language
    hexarc: 'AppType' = 7397, 'hexarc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTPS Discovery
    #: - [UDP] RTPS Discovery
    rtps_discovery: 'AppType' = 7400, 'rtps-discovery', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTPS Data-Distribution User-Traffic
    #: - [UDP] RTPS Data-Distribution User-Traffic
    rtps_dd_ut: 'AppType' = 7401, 'rtps-dd-ut', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RTPS Data-Distribution Meta-Traffic
    #: - [UDP] RTPS Data-Distribution Meta-Traffic
    rtps_dd_mt: 'AppType' = 7402, 'rtps-dd-mt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ionix Network Monitor
    #: - [UDP] Ionix Network Monitor
    ionixnetmon: 'AppType' = 7410, 'ionixnetmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Streaming of measurement data
    #: - [UDP] Streaming of measurement data
    daqstream: 'AppType' = 7411, 'daqstream', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7420: 'AppType' = 7420, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Multichannel real-time lighting control
    ipluminary: 'AppType' = 7420, 'ipluminary', TransportProtocol.get('udp')

    #: - [TCP] Matisse Port Monitor
    #: - [UDP] Matisse Port Monitor
    mtportmon: 'AppType' = 7421, 'mtportmon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM Postmaster Manager
    #: - [UDP] OpenView DM Postmaster Manager
    pmdmgr: 'AppType' = 7426, 'pmdmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM Event Agent Manager
    #: - [UDP] OpenView DM Event Agent Manager
    oveadmgr: 'AppType' = 7427, 'oveadmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM Log Agent Manager
    #: - [UDP] OpenView DM Log Agent Manager
    ovladmgr: 'AppType' = 7428, 'ovladmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM rqt communication
    #: - [UDP] OpenView DM rqt communication
    opi_sock: 'AppType' = 7429, 'opi-sock', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM xmpv7 api pipe
    #: - [UDP] OpenView DM xmpv7 api pipe
    xmpv7: 'AppType' = 7430, 'xmpv7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenView DM ovc/xmpv3 api pipe
    #: - [UDP] OpenView DM ovc/xmpv3 api pipe
    pmd: 'AppType' = 7431, 'pmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Faximum
    #: - [UDP] Faximum
    faximum: 'AppType' = 7437, 'faximum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Oracle Application Server HTTPS
    #: - [UDP] Oracle Application Server HTTPS
    oracleas_https: 'AppType' = 7443, 'oracleas-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Stateless Transport Tunneling Protocol
    sttunnel: 'AppType' = 7471, 'sttunnel', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7471: 'AppType' = 7471, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7472: 'AppType' = 7472, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Rise: The Vieneo Province
    #: - [UDP] Rise: The Vieneo Province
    rise: 'AppType' = 7473, 'rise', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Neo4j Graph Database
    neo4j: 'AppType' = 7474, 'neo4j', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7474: 'AppType' = 7474, 'reserved', TransportProtocol.get('udp')

    #: [TCP] IT Asset Management
    openit: 'AppType' = 7478, 'openit', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7478: 'AppType' = 7478, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] telops-lmd
    #: - [UDP] telops-lmd
    telops_lmd: 'AppType' = 7491, 'telops-lmd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Silhouette User
    #: - [UDP] Silhouette User
    silhouette: 'AppType' = 7500, 'silhouette', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP OpenView Bus Daemon
    #: - [UDP] HP OpenView Bus Daemon
    ovbus: 'AppType' = 7501, 'ovbus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Automation Device Configuration Protocol
    adcp: 'AppType' = 7508, 'adcp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7508: 'AppType' = 7508, 'reserved', TransportProtocol.get('udp')

    #: [TCP] ACPLT - process automation service
    acplt: 'AppType' = 7509, 'acplt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7509: 'AppType' = 7509, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] HP OpenView Application Server
    #: - [UDP] HP OpenView Application Server
    ovhpas: 'AppType' = 7510, 'ovhpas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] pafec-lm
    #: - [UDP] pafec-lm
    pafec_lm: 'AppType' = 7511, 'pafec-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Saratoga Transfer Protocol
    #: - [UDP] Saratoga Transfer Protocol
    saratoga: 'AppType' = 7542, 'saratoga', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] atul server
    #: - [UDP] atul server
    atul: 'AppType' = 7543, 'atul', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FlowAnalyzer DisplayServer
    #: - [UDP] FlowAnalyzer DisplayServer
    nta_ds: 'AppType' = 7544, 'nta-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FlowAnalyzer UtilityServer
    #: - [UDP] FlowAnalyzer UtilityServer
    nta_us: 'AppType' = 7545, 'nta-us', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cisco Fabric service
    #: - [UDP] Cisco Fabric service
    cfs: 'AppType' = 7546, 'cfs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Broadband Forum CWMP
    #: - [UDP] Broadband Forum CWMP
    cwmp: 'AppType' = 7547, 'cwmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Threat Information Distribution Protocol
    #: - [UDP] Threat Information Distribution Protocol
    tidp: 'AppType' = 7548, 'tidp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network Layer Signaling Transport Layer
    #: - [UDP] Network Layer Signaling Transport Layer
    nls_tl: 'AppType' = 7549, 'nls-tl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7550: 'AppType' = 7550, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cloud Signaling Service
    cloudsignaling: 'AppType' = 7550, 'cloudsignaling', TransportProtocol.get('udp')

    #: [TCP] ControlONE Console signaling
    controlone_con: 'AppType' = 7551, 'controlone-con', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7551: 'AppType' = 7551, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Sniffer Command Protocol
    #: - [UDP] Sniffer Command Protocol
    sncp: 'AppType' = 7560, 'sncp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Control Framework [:rfc:`6230`]
    cfw: 'AppType' = 7563, 'cfw', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7563: 'AppType' = 7563, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] VSI Omega
    #: - [UDP] VSI Omega
    vsi_omega: 'AppType' = 7566, 'vsi-omega', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Dell EqualLogic Host Group Management
    dell_eql_asm: 'AppType' = 7569, 'dell-eql-asm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7569: 'AppType' = 7569, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Aries Kfinder
    #: - [UDP] Aries Kfinder
    aries_kfinder: 'AppType' = 7570, 'aries-kfinder', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Oracle Coherence Cluster Service
    coherence: 'AppType' = 7574, 'coherence', TransportProtocol.get('tcp')

    #: [UDP] Oracle Coherence Cluster discovery service
    coherence_disc: 'AppType' = 7574, 'coherence-disc', TransportProtocol.get('udp')

    #: - [TCP] Sun License Manager
    #: - [UDP] Sun License Manager
    sun_lm: 'AppType' = 7588, 'sun-lm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MIPI Alliance Debug
    #: - [UDP] MIPI Alliance Debug
    mipi_debug: 'AppType' = 7606, 'mipi-debug', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Instrument Neutral Distributed Interface
    #: - [UDP] Instrument Neutral Distributed Interface
    indi: 'AppType' = 7624, 'indi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7625: 'AppType' = 7625, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] SImple Middlebox COnfiguration (SIMCO) Server [:rfc:`4540`]
    #: - [SCTP] SImple Middlebox COnfiguration (SIMCO)
    simco: 'AppType' = 7626, 'simco', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [N/A] De-registered
    de_registered_3403: 'AppType' = 3403, 'de_registered', TransportProtocol.get('undefined')

    #: [UDP] De-registered
    de_registered_7626: 'AppType' = 7626, 'de_registered', TransportProtocol.get('udp')

    #: - [TCP] SOAP Service Port
    #: - [UDP] SOAP Service Port
    soap_http: 'AppType' = 7627, 'soap-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Primary Agent Work Notification
    #: - [UDP] Primary Agent Work Notification
    zen_pawn: 'AppType' = 7628, 'zen-pawn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OpenXDAS Wire Protocol
    #: - [UDP] OpenXDAS Wire Protocol
    xdas: 'AppType' = 7629, 'xdas', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] HA Web Konsole
    hawk: 'AppType' = 7630, 'hawk', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7630: 'AppType' = 7630, 'reserved', TransportProtocol.get('udp')

    #: [TCP] TESLA System Messaging
    tesla_sys_msg: 'AppType' = 7631, 'tesla-sys-msg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7631: 'AppType' = 7631, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7632: 'AppType' = 7632, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] PMDF Management
    #: - [UDP] PMDF Management
    pmdfmgt: 'AppType' = 7633, 'pmdfmgt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] bonjour-cuseeme
    #: - [UDP] bonjour-cuseeme
    cuseeme: 'AppType' = 7648, 'cuseeme', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Proprietary immutable distributed data storage
    #: - [UDP] Proprietary immutable distributed data storage
    rome: 'AppType' = 7663, 'rome', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] iMQ STOMP Server
    imqstomp: 'AppType' = 7672, 'imqstomp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7672: 'AppType' = 7672, 'reserved', TransportProtocol.get('udp')

    #: [TCP] iMQ STOMP Server over SSL
    imqstomps: 'AppType' = 7673, 'imqstomps', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7673: 'AppType' = 7673, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] iMQ SSL tunnel
    #: - [UDP] iMQ SSL tunnel
    imqtunnels: 'AppType' = 7674, 'imqtunnels', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iMQ Tunnel
    #: - [UDP] iMQ Tunnel
    imqtunnel: 'AppType' = 7675, 'imqtunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iMQ Broker Rendezvous
    #: - [UDP] iMQ Broker Rendezvous
    imqbrokerd: 'AppType' = 7676, 'imqbrokerd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun App Server - HTTPS
    #: - [UDP] Sun App Server - HTTPS
    sun_user_https: 'AppType' = 7677, 'sun-user-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Microsoft Delivery Optimization Peer-to-Peer
    #: - [UDP] Microsoft Delivery Optimization Peer-to-Peer
    ms_do: 'AppType' = 7680, 'ms-do', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Cleondris DMT
    dmt: 'AppType' = 7683, 'dmt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7683: 'AppType' = 7683, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Bolt database connection
    bolt: 'AppType' = 7687, 'bolt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7687: 'AppType' = 7687, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7688: 'AppType' = 7688, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Collaber Network Service
    #: - [UDP] Collaber Network Service
    collaber: 'AppType' = 7689, 'collaber', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Service-Oriented Vehicle Diagnostics
    sovd: 'AppType' = 7690, 'sovd', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7690: 'AppType' = 7690, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] KLIO communications
    #: - [UDP] KLIO communications
    klio: 'AppType' = 7697, 'klio', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] EM7 Secure Communications
    em7_secom: 'AppType' = 7700, 'em7-secom', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7700: 'AppType' = 7700, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_7701: 'AppType' = 7701, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [SCTP] SCF nFAPI defining MAC/PHY split
    nfapi: 'AppType' = 7701, 'nfapi', TransportProtocol.get('sctp')

    #: - [TCP] EM7 Dynamic Updates
    #: - [UDP] EM7 Dynamic Updates
    sync_em7: 'AppType' = 7707, 'sync-em7', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] scientia.net
    #: - [UDP] scientia.net
    scinet: 'AppType' = 7708, 'scinet', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MedImage Portal
    #: - [UDP] MedImage Portal
    medimageportal: 'AppType' = 7720, 'medimageportal', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Novell Snap-in Deep Freeze Control
    #: - [UDP] Novell Snap-in Deep Freeze Control
    nsdeepfreezectl: 'AppType' = 7724, 'nsdeepfreezectl', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Nitrogen Service
    #: - [UDP] Nitrogen Service
    nitrogen: 'AppType' = 7725, 'nitrogen', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FreezeX Console Service
    #: - [UDP] FreezeX Console Service
    freezexservice: 'AppType' = 7726, 'freezexservice', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Trident Systems Data
    #: - [UDP] Trident Systems Data
    trident_data: 'AppType' = 7727, 'trident-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Open-Source Virtual Reality
    #: - [UDP] Open-Source Virtual Reality
    #: - [SCTP] Open-Source Virtual Reality
    osvr: 'AppType' = 7728, 'osvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] Smith Protocol over IP
    #: - [UDP] Smith Protocol over IP
    smip: 'AppType' = 7734, 'smip', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HP Enterprise Discovery Agent
    #: - [UDP] HP Enterprise Discovery Agent
    aiagent: 'AppType' = 7738, 'aiagent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ScriptView Network
    #: - [UDP] ScriptView Network
    scriptview: 'AppType' = 7741, 'scriptview', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Mugginsoft Script Server Service
    msss: 'AppType' = 7742, 'msss', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7742: 'AppType' = 7742, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Sakura Script Transfer Protocol
    #: - [UDP] Sakura Script Transfer Protocol
    sstp_1: 'AppType' = 7743, 'sstp-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] RAQMON PDU [:rfc:`4712`]
    #: - [UDP] RAQMON PDU [:rfc:`4712`]
    raqmon_pdu: 'AppType' = 7744, 'raqmon-pdu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Put/Run/Get Protocol
    #: - [UDP] Put/Run/Get Protocol
    prgp: 'AppType' = 7747, 'prgp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] A File System using TLS over a wide area network
    inetfs: 'AppType' = 7775, 'inetfs', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7775: 'AppType' = 7775, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7776: 'AppType' = 7776, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] cbt
    #: - [UDP] cbt
    cbt: 'AppType' = 7777, 'cbt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Interwise
    #: - [UDP] Interwise
    interwise: 'AppType' = 7778, 'interwise', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VSTAT
    #: - [UDP] VSTAT
    vstat: 'AppType' = 7779, 'vstat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7780: 'AppType' = 7780, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] accu-lmgr
    #: - [UDP] accu-lmgr
    accu_lmgr: 'AppType' = 7781, 'accu-lmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Seamless Bidirectional Forwarding Detection (S-BFD) [:rfc:`7881`]
    s_bfd: 'AppType' = 7784, 's-bfd', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7784: 'AppType' = 7784, 'reserved', TransportProtocol.get('tcp')

    #: [N/A] Unassigned
    unassigned_7785: 'AppType' = 7785, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] MINIVEND
    #: - [UDP] MINIVEND
    minivend: 'AppType' = 7786, 'minivend', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Popup Reminders Receive
    #: - [UDP] Popup Reminders Receive
    popup_reminders: 'AppType' = 7787, 'popup-reminders', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7788: 'AppType' = 7788, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Office Tools Pro Receive
    #: - [UDP] Office Tools Pro Receive
    office_tools: 'AppType' = 7789, 'office-tools', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Q3ADE Cluster Service
    #: - [UDP] Q3ADE Cluster Service
    q3ade: 'AppType' = 7794, 'q3ade', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Propel Connector port
    #: - [UDP] Propel Connector port
    pnet_conn: 'AppType' = 7797, 'pnet-conn', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Propel Encoder port
    #: - [UDP] Propel Encoder port
    pnet_enc: 'AppType' = 7798, 'pnet-enc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Alternate BSDP Service
    #: - [UDP] Alternate BSDP Service
    altbsdp: 'AppType' = 7799, 'altbsdp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apple Software Restore
    #: - [UDP] Apple Software Restore
    asr: 'AppType' = 7800, 'asr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Server Protocol - client
    #: - [UDP] Secure Server Protocol - client
    ssp_client: 'AppType' = 7801, 'ssp-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Virtualized Network Services Tunnel Protocol
    vns_tp: 'AppType' = 7802, 'vns-tp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7802: 'AppType' = 7802, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] Riverbed WAN Optimization Protocol
    #: - [UDP] Riverbed WAN Optimization Protocol
    rbt_wanopt: 'AppType' = 7810, 'rbt-wanopt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 7845
    #: - [UDP] APC 7845
    apc_7845: 'AppType' = 7845, 'apc-7845', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] APC 7846
    #: - [UDP] APC 7846
    apc_7846: 'AppType' = 7846, 'apc-7846', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] A product key authentication protocol made by CSO
    csoauth: 'AppType' = 7847, 'csoauth', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7847: 'AppType' = 7847, 'reserved', TransportProtocol.get('udp')

    #: [TCP] MobileAnalyzer& MobileMonitor
    mobileanalyzer: 'AppType' = 7869, 'mobileanalyzer', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7869: 'AppType' = 7869, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Riverbed Steelhead Mobile Service
    rbt_smc: 'AppType' = 7870, 'rbt-smc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7870: 'AppType' = 7870, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Mobile Device Management
    mdm: 'AppType' = 7871, 'mdm', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7871: 'AppType' = 7871, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7872: 'AppType' = 7872, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] TLS-based Mobile IPv6 Security [:rfc:`6618`]
    mipv6tls: 'AppType' = 7872, 'mipv6tls', TransportProtocol.get('udp')

    #: [TCP] Opswise Message Service
    owms: 'AppType' = 7878, 'owms', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7878: 'AppType' = 7878, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_7879: 'AppType' = 7879, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Pearson
    #: - [UDP] Pearson
    pss: 'AppType' = 7880, 'pss', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Universal Broker
    #: - [UDP] Universal Broker
    ubroker: 'AppType' = 7887, 'ubroker', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Multicast Event
    #: - [UDP] Multicast Event
    mevent: 'AppType' = 7900, 'mevent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNOS Service Protocol
    #: - [UDP] TNOS Service Protocol
    tnos_sp: 'AppType' = 7901, 'tnos-sp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNOS shell Protocol
    #: - [UDP] TNOS shell Protocol
    tnos_dp: 'AppType' = 7902, 'tnos-dp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TNOS Secure DiaguardProtocol
    #: - [UDP] TNOS Secure DiaguardProtocol
    tnos_dps: 'AppType' = 7903, 'tnos-dps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] QuickObjects secure port
    #: - [UDP] QuickObjects secure port
    qo_secure: 'AppType' = 7913, 'qo-secure', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tier 2 Data Resource Manager
    #: - [UDP] Tier 2 Data Resource Manager
    t2_drm: 'AppType' = 7932, 't2-drm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Tier 2 Business Rules Manager
    #: - [UDP] Tier 2 Business Rules Manager
    t2_brm: 'AppType' = 7933, 't2-brm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Encrypted, extendable, general-purpose synchronization protocol
    #: - [UDP] Encrypted, extendable, general-purpose synchronization protocol
    generalsync: 'AppType' = 7962, 'generalsync', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Supercell
    #: - [UDP] Supercell
    supercell: 'AppType' = 7967, 'supercell', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Micromuse-ncps
    #: - [UDP] Micromuse-ncps
    micromuse_ncps: 'AppType' = 7979, 'micromuse-ncps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Quest Vista
    #: - [UDP] Quest Vista
    quest_vista: 'AppType' = 7980, 'quest-vista', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Spotlight on SQL Server Desktop Collect
    sossd_collect: 'AppType' = 7981, 'sossd-collect', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7981: 'AppType' = 7981, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Spotlight on SQL Server Desktop Agent
    sossd_agent: 'AppType' = 7982, 'sossd-agent', TransportProtocol.get('tcp')

    #: [UDP] Spotlight on SQL Server Desktop Agent Discovery
    sossd_disc: 'AppType' = 7982, 'sossd-disc', TransportProtocol.get('udp')

    #: [TCP] PUSH Notification Service
    pushns: 'AppType' = 7997, 'pushns', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_7997: 'AppType' = 7997, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_7998: 'AppType' = 7998, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] USI Content Push Service
    usicontentpush: 'AppType' = 7998, 'usicontentpush', TransportProtocol.get('udp')

    #: - [TCP] iRDMI2
    #: - [UDP] iRDMI2
    irdmi2: 'AppType' = 7999, 'irdmi2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] iRDMI
    #: - [UDP] iRDMI
    irdmi: 'AppType' = 8000, 'irdmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VCOM Tunnel
    #: - [UDP] VCOM Tunnel
    vcom_tunnel: 'AppType' = 8001, 'vcom-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Teradata ORDBMS
    #: - [UDP] Teradata ORDBMS
    teradataordbms: 'AppType' = 8002, 'teradataordbms', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Mulberry Connect Reporting Service
    #: - [UDP] Mulberry Connect Reporting Service
    mcreport: 'AppType' = 8003, 'mcreport', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Opensource Evolv Enterprise Platform P2P Network Node Connection
    #: Protocol
    p2pevolvenet: 'AppType' = 8004, 'p2pevolvenet', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8004: 'AppType' = 8004, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] MXI Generation II for z/OS
    #: - [UDP] MXI Generation II for z/OS
    mxi: 'AppType' = 8005, 'mxi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] World Programming analytics
    wpl_analytics: 'AppType' = 8006, 'wpl-analytics', TransportProtocol.get('tcp')

    #: [UDP] World Programming analytics discovery
    wpl_disc: 'AppType' = 8006, 'wpl-disc', TransportProtocol.get('udp')

    #: - [TCP] I/O oriented cluster computing software
    #: - [UDP] I/O oriented cluster computing software
    warppipe: 'AppType' = 8007, 'warppipe', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] FileMaker, Inc. - HTTP Alternate (see Port 80)
    #: - [UDP] FileMaker, Inc. - HTTP Alternate (see Port 80)
    http_alt_591: 'AppType' = 591, 'http-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTTP Alternate
    #: - [UDP] HTTP Alternate
    http_alt_8008: 'AppType' = 8008, 'http-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] NVMe over Fabrics Discovery Service
    nvme_disc: 'AppType' = 8009, 'nvme-disc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8009: 'AppType' = 8009, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Reserved
    reserved_8010: 'AppType' = 8010, 'reserved', TransportProtocol.get('undefined')

    #: [TCP] Configuration Cloud Service
    cfg_cloud: 'AppType' = 8015, 'cfg-cloud', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8015: 'AppType' = 8015, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Beckhoff Automation Device Specification
    ads_s: 'AppType' = 8016, 'ads-s', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8016: 'AppType' = 8016, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8017: 'AppType' = 8017, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Cisco Cloudsec Dataplane Port Number
    cisco_cloudsec: 'AppType' = 8017, 'cisco-cloudsec', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8018: 'AppType' = 8018, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] QB DB Dynamic Port
    #: - [UDP] QB DB Dynamic Port
    qbdb: 'AppType' = 8019, 'qbdb', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intuit Entitlement Service and Discovery
    #: - [UDP] Intuit Entitlement Service and Discovery
    intu_ec_svcdisc: 'AppType' = 8020, 'intu-ec-svcdisc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Intuit Entitlement Client
    #: - [UDP] Intuit Entitlement Client
    intu_ec_client: 'AppType' = 8021, 'intu-ec-client', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] oa-system
    #: - [UDP] oa-system
    oa_system: 'AppType' = 8022, 'oa-system', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ARCATrust vault API
    #: - [UDP] ARCATrust vault API
    arca_api: 'AppType' = 8023, 'arca-api', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8024: 'AppType' = 8024, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] CA Audit Distribution Agent
    #: - [UDP] CA Audit Distribution Agent
    ca_audit_da: 'AppType' = 8025, 'ca-audit-da', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CA Audit Distribution Server
    #: - [UDP] CA Audit Distribution Server
    ca_audit_ds: 'AppType' = 8026, 'ca-audit-ds', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] peer tracker and data relay service
    #: - [UDP] peer tracker and data relay service
    papachi_p2p_srv: 'AppType' = 8027, 'papachi-p2p-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ProEd
    #: - [UDP] ProEd
    pro_ed: 'AppType' = 8032, 'pro-ed', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] MindPrint
    #: - [UDP] MindPrint
    mindprint: 'AppType' = 8033, 'mindprint', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] .vantronix Management
    #: - [UDP] .vantronix Management
    vantronix_mgmt: 'AppType' = 8034, 'vantronix-mgmt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ampify Messaging Protocol
    #: - [UDP] Ampify Messaging Protocol
    ampify: 'AppType' = 8040, 'ampify', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xcorpeon ASIC Carrier Ethernet Transport
    #: - [UDP] Xcorpeon ASIC Carrier Ethernet Transport
    enguity_xccetp: 'AppType' = 8041, 'enguity-xccetp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] FireScope Agent
    fs_agent: 'AppType' = 8042, 'fs-agent', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8042: 'AppType' = 8042, 'reserved', TransportProtocol.get('udp')

    #: [TCP] FireScope Server
    fs_server: 'AppType' = 8043, 'fs-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8043: 'AppType' = 8043, 'reserved', TransportProtocol.get('udp')

    #: [TCP] FireScope Management Interface
    fs_mgmt: 'AppType' = 8044, 'fs-mgmt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8044: 'AppType' = 8044, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Rocrail Client Service
    rocrail: 'AppType' = 8051, 'rocrail', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8051: 'AppType' = 8051, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Server
    #: - [UDP] Senomix Timesheets Server
    senomix01: 'AppType' = 8052, 'senomix01', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Client [1 year assignment]
    #: - [UDP] Senomix Timesheets Client [1 year assignment]
    senomix02: 'AppType' = 8053, 'senomix02', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Server [1 year assignment]
    #: - [UDP] Senomix Timesheets Server [1 year assignment]
    senomix03: 'AppType' = 8054, 'senomix03', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Server [1 year assignment]
    #: - [UDP] Senomix Timesheets Server [1 year assignment]
    senomix04: 'AppType' = 8055, 'senomix04', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Server [1 year assignment]
    #: - [UDP] Senomix Timesheets Server [1 year assignment]
    senomix05: 'AppType' = 8056, 'senomix05', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Client [1 year assignment]
    #: - [UDP] Senomix Timesheets Client [1 year assignment]
    senomix06: 'AppType' = 8057, 'senomix06', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Client [1 year assignment]
    #: - [UDP] Senomix Timesheets Client [1 year assignment]
    senomix07: 'AppType' = 8058, 'senomix07', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Senomix Timesheets Client [1 year assignment]
    #: - [UDP] Senomix Timesheets Client [1 year assignment]
    senomix08: 'AppType' = 8059, 'senomix08', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8060: 'AppType' = 8060, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Asymmetric Extended Route Optimization (AERO) [:rfc:`6706`]
    aero: 'AppType' = 8060, 'aero', TransportProtocol.get('udp')

    #: [TCP] Nikatron Device Protocol
    nikatron_dev: 'AppType' = 8061, 'nikatron-dev', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8061: 'AppType' = 8061, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Toad BI Application Server
    toad_bi_appsrvr: 'AppType' = 8066, 'toad-bi-appsrvr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8066: 'AppType' = 8066, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Infinidat async replication
    infi_async: 'AppType' = 8067, 'infi-async', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8067: 'AppType' = 8067, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Oracle Unified Communication Suite's Indexed Search Converter
    ucs_isc: 'AppType' = 8070, 'ucs-isc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8070: 'AppType' = 8070, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Gadu-Gadu
    #: - [UDP] Gadu-Gadu
    gadugadu: 'AppType' = 8074, 'gadugadu', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Reserved
    #: - [UDP] Reserved
    reserved_8077: 'AppType' = 8077, 'reserved', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] HTTP Alternate (see port 80)
    #: - [UDP] HTTP Alternate (see port 80)
    http_alt: 'AppType' = 8080, 'http-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Proxy Admin Service
    #: - [UDP] Sun Proxy Admin Service
    sunproxyadmin: 'AppType' = 8081, 'sunproxyadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Utilistor (Client)
    #: - [UDP] Utilistor (Client)
    us_cli: 'AppType' = 8082, 'us-cli', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Utilistor (Server)
    #: - [UDP] Utilistor (Server)
    us_srv: 'AppType' = 8083, 'us-srv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Snarl Network Protocol over HTTP
    websnp: 'AppType' = 8084, 'websnp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8084: 'AppType' = 8084, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8085: 'AppType' = 8085, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Distributed SCADA Networking Rendezvous Port
    #: - [UDP] Distributed SCADA Networking Rendezvous Port
    d_s_n: 'AppType' = 8086, 'd-s-n', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Simplify Media SPP Protocol
    #: - [UDP] Simplify Media SPP Protocol
    simplifymedia: 'AppType' = 8087, 'simplifymedia', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Radan HTTP
    #: - [UDP] Radan HTTP
    radan_http: 'AppType' = 8088, 'radan-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8089: 'AppType' = 8089, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Vehicle to station messaging
    opsmessaging: 'AppType' = 8090, 'opsmessaging', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8090: 'AppType' = 8090, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Jam Link Framework
    jamlink: 'AppType' = 8091, 'jamlink', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8091: 'AppType' = 8091, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SAC Port Id
    #: - [UDP] SAC Port Id
    sac: 'AppType' = 8097, 'sac', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Xprint Server
    #: - [UDP] Xprint Server
    xprint_server: 'AppType' = 8100, 'xprint-server', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Logical Domains Migration
    ldoms_migr: 'AppType' = 8101, 'ldoms-migr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8101: 'AppType' = 8101, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Oracle Kernel zones migration server
    kz_migr: 'AppType' = 8102, 'kz-migr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8102: 'AppType' = 8102, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8111: 'AppType' = 8111, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Skynetflow network services
    skynetflow: 'AppType' = 8111, 'skynetflow', TransportProtocol.get('udp')

    #: - [TCP] MTL8000 Matrix
    #: - [UDP] MTL8000 Matrix
    mtl8000_matrix: 'AppType' = 8115, 'mtl8000-matrix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Check Point Clustering
    #: - [UDP] Check Point Clustering
    cp_cluster: 'AppType' = 8116, 'cp-cluster', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Purity replication clustering and remote management
    purityrpc: 'AppType' = 8117, 'purityrpc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8117: 'AppType' = 8117, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Privoxy HTTP proxy
    #: - [UDP] Privoxy HTTP proxy
    privoxy: 'AppType' = 8118, 'privoxy', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apollo Data Port
    #: - [UDP] Apollo Data Port
    apollo_data: 'AppType' = 8121, 'apollo-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Apollo Admin Port
    #: - [UDP] Apollo Admin Port
    apollo_admin: 'AppType' = 8122, 'apollo-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PayCash Online Protocol
    #: - [UDP] PayCash Online Protocol
    paycash_online: 'AppType' = 8128, 'paycash-online', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PayCash Wallet-Browser
    #: - [UDP] PayCash Wallet-Browser
    paycash_wbp: 'AppType' = 8129, 'paycash-wbp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INDIGO-VRMI
    #: - [UDP] INDIGO-VRMI
    indigo_vrmi: 'AppType' = 8130, 'indigo-vrmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] INDIGO-VBCP
    #: - [UDP] INDIGO-VBCP
    indigo_vbcp: 'AppType' = 8131, 'indigo-vbcp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] dbabble
    #: - [UDP] dbabble
    dbabble: 'AppType' = 8132, 'dbabble', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] The Puppet master service
    puppet: 'AppType' = 8140, 'puppet', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8140: 'AppType' = 8140, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] i-SDD file transfer
    #: - [UDP] i-SDD file transfer
    isdd: 'AppType' = 8148, 'isdd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8149: 'AppType' = 8149, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Edge of Reality game data
    eor_game: 'AppType' = 8149, 'eor-game', TransportProtocol.get('udp')

    #: [TCP] QuantaStor Management Interface
    quantastor: 'AppType' = 8153, 'quantastor', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8153: 'AppType' = 8153, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Patrol
    #: - [UDP] Patrol
    patrol: 'AppType' = 8160, 'patrol', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Patrol SNMP
    #: - [UDP] Patrol SNMP
    patrol_snmp: 'AppType' = 8161, 'patrol-snmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] LPAR2RRD client server communication
    lpar2rrd: 'AppType' = 8162, 'lpar2rrd', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8162: 'AppType' = 8162, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Intermapper network management system
    intermapper: 'AppType' = 8181, 'intermapper', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8181: 'AppType' = 8181, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] VMware Fault Domain Manager
    #: - [UDP] VMware Fault Domain Manager
    vmware_fdm: 'AppType' = 8182, 'vmware-fdm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] ProRemote
    proremote: 'AppType' = 8183, 'proremote', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8183: 'AppType' = 8183, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Remote iTach Connection
    #: - [UDP] Remote iTach Connection
    itach: 'AppType' = 8184, 'itach', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Generic control plane for RPHY
    gcp_rphy: 'AppType' = 8190, 'gcp-rphy', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8190: 'AppType' = 8190, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Limner Pressure
    limnerpressure: 'AppType' = 8191, 'limnerpressure', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8191: 'AppType' = 8191, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] SpyTech Phone Service
    #: - [UDP] SpyTech Phone Service
    spytechphone: 'AppType' = 8192, 'spytechphone', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8193: 'AppType' = 8193, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Bloomberg data API
    #: - [UDP] Bloomberg data API
    blp1: 'AppType' = 8194, 'blp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Bloomberg feed
    #: - [UDP] Bloomberg feed
    blp2: 'AppType' = 8195, 'blp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] VVR DATA
    #: - [UDP] VVR DATA
    vvr_data: 'AppType' = 8199, 'vvr-data', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIVNET
    #: - [UDP] TRIVNET
    trivnet1: 'AppType' = 8200, 'trivnet1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] TRIVNET
    #: - [UDP] TRIVNET
    trivnet2: 'AppType' = 8201, 'trivnet2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8202: 'AppType' = 8202, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Audio+Ethernet Standard Open Protocol
    aesop: 'AppType' = 8202, 'aesop', TransportProtocol.get('udp')

    #: - [TCP] LM Perfworks
    #: - [UDP] LM Perfworks
    lm_perfworks: 'AppType' = 8204, 'lm-perfworks', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LM Instmgr
    #: - [UDP] LM Instmgr
    lm_instmgr: 'AppType' = 8205, 'lm-instmgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LM Dta
    #: - [UDP] LM Dta
    lm_dta: 'AppType' = 8206, 'lm-dta', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LM SServer
    #: - [UDP] LM SServer
    lm_sserver: 'AppType' = 8207, 'lm-sserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] LM Webwatcher
    #: - [UDP] LM Webwatcher
    lm_webwatcher: 'AppType' = 8208, 'lm-webwatcher', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8211: 'AppType' = 8211, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Aruba Networks AP management
    aruba_papi: 'AppType' = 8211, 'aruba-papi', TransportProtocol.get('udp')

    #: - [TCP] RexecJ Server
    #: - [UDP] RexecJ Server
    rexecj: 'AppType' = 8230, 'rexecj', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8231: 'AppType' = 8231, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] HNCP [:rfc:`7788`]
    hncp_udp_port: 'AppType' = 8231, 'hncp-udp-port', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8232: 'AppType' = 8232, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] HNCP over DTLS [:rfc:`7788`]
    hncp_dtls_port: 'AppType' = 8232, 'hncp-dtls-port', TransportProtocol.get('udp')

    #: - [TCP] Synapse Non Blocking HTTPS
    #: - [UDP] Synapse Non Blocking HTTPS
    synapse_nhttps: 'AppType' = 8243, 'synapse-nhttps', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8266: 'AppType' = 8266, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] ESPeasy peer-2-peer communication
    espeasy_p2p: 'AppType' = 8266, 'espeasy-p2p', TransportProtocol.get('udp')

    #: [TCP] Robot Framework Remote Library Interface
    robot_remote: 'AppType' = 8270, 'robot-remote', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8270: 'AppType' = 8270, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Microsoft Connected Cache
    #: - [UDP] Microsoft Connected Cache
    ms_mcc: 'AppType' = 8276, 'ms-mcc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Synapse Non Blocking HTTP
    #: - [UDP] Synapse Non Blocking HTTP
    synapse_nhttp: 'AppType' = 8280, 'synapse-nhttp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8281: 'AppType' = 8281, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Libelle EnterpriseBus
    libelle: 'AppType' = 8282, 'libelle', TransportProtocol.get('tcp')

    #: [UDP] Libelle EnterpriseBus discovery
    libelle_disc: 'AppType' = 8282, 'libelle-disc', TransportProtocol.get('udp')

    #: - [SCTP] Reserved
    #: - [DCCP] Reserved
    reserved_8282: 'AppType' = 8282, 'reserved', TransportProtocol.get('sctp') | TransportProtocol.get('dccp')

    #: - [TCP] Bloomberg professional
    #: - [UDP] Bloomberg professional
    blp3: 'AppType' = 8292, 'blp3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Hiperscan Identification Service
    hiperscan_id: 'AppType' = 8293, 'hiperscan-id', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8293: 'AppType' = 8293, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Bloomberg intelligent client
    #: - [UDP] Bloomberg intelligent client
    blp4: 'AppType' = 8294, 'blp4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Transport Management Interface
    #: - [UDP] Transport Management Interface
    tmi: 'AppType' = 8300, 'tmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Amberon PPC/PPS
    #: - [UDP] Amberon PPC/PPS
    amberon: 'AppType' = 8301, 'amberon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Hub Open Network
    hub_open_net: 'AppType' = 8313, 'hub-open-net', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8313: 'AppType' = 8313, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Thin(ium) Network Protocol
    #: - [UDP] Thin(ium) Network Protocol
    tnp_discover: 'AppType' = 8320, 'tnp-discover', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Thin(ium) Network Protocol
    #: - [UDP] Thin(ium) Network Protocol
    tnp: 'AppType' = 8321, 'tnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Garmin Marine
    #: - [UDP] Garmin Marine
    garmin_marine: 'AppType' = 8322, 'garmin-marine', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Server Find
    #: - [UDP] Server Find
    server_find: 'AppType' = 8351, 'server-find', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cruise ENUM
    #: - [UDP] Cruise ENUM
    cruise_enum: 'AppType' = 8376, 'cruise-enum', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cruise SWROUTE
    #: - [UDP] Cruise SWROUTE
    cruise_swroute: 'AppType' = 8377, 'cruise-swroute', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cruise CONFIG
    #: - [UDP] Cruise CONFIG
    cruise_config: 'AppType' = 8378, 'cruise-config', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cruise DIAGS
    #: - [UDP] Cruise DIAGS
    cruise_diags: 'AppType' = 8379, 'cruise-diags', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Cruise UPDATE
    #: - [UDP] Cruise UPDATE
    cruise_update: 'AppType' = 8380, 'cruise-update', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] M2m Services
    #: - [UDP] M2m Services
    m2mservices: 'AppType' = 8383, 'm2mservices', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Marathon Transport Protocol
    marathontp: 'AppType' = 8384, 'marathontp', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8384: 'AppType' = 8384, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] cvd
    #: - [UDP] cvd
    cvd: 'AppType' = 8400, 'cvd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] sabarsd
    #: - [UDP] sabarsd
    sabarsd: 'AppType' = 8401, 'sabarsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] abarsd
    #: - [UDP] abarsd
    abarsd: 'AppType' = 8402, 'abarsd', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] admind
    #: - [UDP] admind
    admind_3279: 'AppType' = 3279, 'admind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] admind
    #: - [UDP] admind
    admind_8403: 'AppType' = 8403, 'admind', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] SuperVault Cloud
    svcloud: 'AppType' = 8404, 'svcloud', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8404: 'AppType' = 8404, 'reserved', TransportProtocol.get('udp')

    #: [TCP] SuperVault Backup
    svbackup: 'AppType' = 8405, 'svbackup', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8405: 'AppType' = 8405, 'reserved', TransportProtocol.get('udp')

    #: [UDP] Reserved
    reserved_8415: 'AppType' = 8415, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] eSpeech Session Protocol
    #: - [UDP] eSpeech Session Protocol
    espeech: 'AppType' = 8416, 'espeech', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] eSpeech RTP Protocol
    #: - [UDP] eSpeech RTP Protocol
    espeech_rtp: 'AppType' = 8417, 'espeech-rtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Aristech text-to-speech server
    aritts: 'AppType' = 8423, 'aritts', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8423: 'AppType' = 8423, 'reserved', TransportProtocol.get('udp')

    #: [TCP] PostgreSQL Backup
    pgbackrest: 'AppType' = 8432, 'pgbackrest', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8432: 'AppType' = 8432, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8433: 'AppType' = 8433, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Non Persistent Desktop and Application Streaming
    aws_as2: 'AppType' = 8433, 'aws-as2', TransportProtocol.get('udp')

    #: - [TCP] CyBro A-bus Protocol
    #: - [UDP] CyBro A-bus Protocol
    cybro_a_bus: 'AppType' = 8442, 'cybro-a-bus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCsync HTTPS
    #: - [UDP] PCsync HTTPS
    pcsync_https: 'AppType' = 8443, 'pcsync-https', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PCsync HTTP
    #: - [UDP] PCsync HTTP
    pcsync_http: 'AppType' = 8444, 'pcsync-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Port for copy peer sync feature
    copy: 'AppType' = 8445, 'copy', TransportProtocol.get('tcp')

    #: [UDP] Port for copy discovery
    copy_disc: 'AppType' = 8445, 'copy-disc', TransportProtocol.get('udp')

    #: [TCP] Matrix Federation Protocol
    matrix_fed: 'AppType' = 8448, 'matrix-fed', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8448: 'AppType' = 8448, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8449: 'AppType' = 8449, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] npmp
    #: - [UDP] npmp
    npmp: 'AppType' = 8450, 'npmp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Nexenta Management GUI
    nexentamv: 'AppType' = 8457, 'nexentamv', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8457: 'AppType' = 8457, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Cisco Address Validation Protocol
    cisco_avp: 'AppType' = 8470, 'cisco-avp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8470: 'AppType' = 8470, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] PIM over Reliable Transport [:rfc:`6559`]
    #: - [SCTP] PIM over Reliable Transport [:rfc:`6559`]
    pim_port: 'AppType' = 8471, 'pim-port', TransportProtocol.get('tcp') | TransportProtocol.get('sctp')

    #: [UDP] Reserved
    reserved_8471: 'AppType' = 8471, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Overlay Transport Virtualization (OTV)
    #: - [UDP] Overlay Transport Virtualization (OTV)
    otv: 'AppType' = 8472, 'otv', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Virtual Point to Point
    #: - [UDP] Virtual Point to Point
    vp2p: 'AppType' = 8473, 'vp2p', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] AquaMinds NoteShare
    #: - [UDP] AquaMinds NoteShare
    noteshare: 'AppType' = 8474, 'noteshare', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Flight Message Transfer Protocol
    #: - [UDP] Flight Message Transfer Protocol
    fmtp: 'AppType' = 8500, 'fmtp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] CYTEL Message Transfer Management
    cmtp_mgt: 'AppType' = 8501, 'cmtp-mgt', TransportProtocol.get('tcp')

    #: [UDP] CYTEL Message Transfer Audio and Video
    cmtp_av: 'AppType' = 8501, 'cmtp-av', TransportProtocol.get('udp')

    #: [TCP] FTN Message Transfer Protocol
    ftnmtp: 'AppType' = 8502, 'ftnmtp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8502: 'AppType' = 8502, 'reserved', TransportProtocol.get('udp')

    #: [UDP] MPLS LSP Self-Ping [:rfc:`7746`]
    lsp_self_ping: 'AppType' = 8503, 'lsp-self-ping', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8503: 'AppType' = 8503, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] RTSP Alternate (see port 554)
    #: - [UDP] RTSP Alternate (see port 554)
    rtsp_alt: 'AppType' = 8554, 'rtsp-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] SYMAX D-FENCE
    #: - [UDP] SYMAX D-FENCE
    d_fence: 'AppType' = 8555, 'd-fence', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DOF Tunneling Protocol
    #: - [UDP] DOF Tunneling Protocol
    dof_tunnel: 'AppType' = 8567, 'dof-tunnel', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Surveillance Data
    #: - [UDP] Surveillance Data
    asterix: 'AppType' = 8600, 'asterix', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8609: 'AppType' = 8609, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Canon Compact Printer Protocol Discovery
    canon_cpp_disc: 'AppType' = 8609, 'canon-cpp-disc', TransportProtocol.get('udp')

    #: - [TCP] Canon MFNP Service
    #: - [UDP] Canon MFNP Service
    canon_mfnp: 'AppType' = 8610, 'canon-mfnp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Canon BJNP Port 1
    #: - [UDP] Canon BJNP Port 1
    canon_bjnp1: 'AppType' = 8611, 'canon-bjnp1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Canon BJNP Port 2
    #: - [UDP] Canon BJNP Port 2
    canon_bjnp2: 'AppType' = 8612, 'canon-bjnp2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Canon BJNP Port 3
    #: - [UDP] Canon BJNP Port 3
    canon_bjnp3: 'AppType' = 8613, 'canon-bjnp3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Canon BJNP Port 4
    #: - [UDP] Canon BJNP Port 4
    canon_bjnp4: 'AppType' = 8614, 'canon-bjnp4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Imink Service Control
    imink: 'AppType' = 8615, 'imink', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8615: 'AppType' = 8615, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Monetra
    monetra: 'AppType' = 8665, 'monetra', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8665: 'AppType' = 8665, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Monetra Administrative Access
    monetra_admin: 'AppType' = 8666, 'monetra-admin', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8666: 'AppType' = 8666, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8667: 'AppType' = 8667, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Spartan management
    spartan: 'AppType' = 8668, 'spartan', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8668: 'AppType' = 8668, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Motorola Solutions Customer Programming Software for Radio Management
    msi_cps_rm: 'AppType' = 8675, 'msi-cps-rm', TransportProtocol.get('tcp')

    #: [UDP] Motorola Solutions Customer Programming Software for Radio Management
    #: Discovery
    msi_cps_rm_disc: 'AppType' = 8675, 'msi-cps-rm-disc', TransportProtocol.get('udp')

    #: - [TCP] Sun App Server - JMX/RMI
    #: - [UDP] Sun App Server - JMX/RMI
    sun_as_jmxrmi: 'AppType' = 8686, 'sun-as-jmxrmi', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8687: 'AppType' = 8687, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] OpenRemote Controller HTTP/REST
    openremote_ctrl: 'AppType' = 8688, 'openremote-ctrl', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8688: 'AppType' = 8688, 'reserved', TransportProtocol.get('udp')

    #: [TCP] VNYX Primary Port
    vnyx: 'AppType' = 8699, 'vnyx', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8699: 'AppType' = 8699, 'reserved', TransportProtocol.get('udp')

    #: [TCP] gRPC for SEMI Standards implementations
    semi_grpc: 'AppType' = 8710, 'semi-grpc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8710: 'AppType' = 8710, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Nuance Voice Control
    nvc: 'AppType' = 8711, 'nvc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8711: 'AppType' = 8711, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8732: 'AppType' = 8732, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] DASGIP Net Services
    dtp_net: 'AppType' = 8732, 'dtp-net', TransportProtocol.get('udp')

    #: - [TCP] iBus
    #: - [UDP] iBus
    ibus: 'AppType' = 8733, 'ibus', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] DEY Storage Key Negotiation
    dey_keyneg: 'AppType' = 8750, 'dey-keyneg', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8750: 'AppType' = 8750, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] MC-APPSERVER
    #: - [UDP] MC-APPSERVER
    mc_appserver: 'AppType' = 8763, 'mc-appserver', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] OPENQUEUE
    #: - [UDP] OPENQUEUE
    openqueue: 'AppType' = 8764, 'openqueue', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Ultraseek HTTP
    #: - [UDP] Ultraseek HTTP
    ultraseek_http: 'AppType' = 8765, 'ultraseek-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Agilent Connectivity Service
    #: - [UDP] Agilent Connectivity Service
    amcs: 'AppType' = 8766, 'amcs', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Online mobile multiplayer game
    core_of_source: 'AppType' = 8767, 'core-of-source', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8767: 'AppType' = 8767, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Sandpolis Server
    sandpolis: 'AppType' = 8768, 'sandpolis', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8768: 'AppType' = 8768, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Okta MultiPlatform Access Mgmt for Cloud Svcs
    oktaauthenticat: 'AppType' = 8769, 'oktaauthenticat', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8769: 'AppType' = 8769, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Digital Photo Access Protocol (iPhoto)
    #: - [UDP] Digital Photo Access Protocol (iPhoto)
    dpap: 'AppType' = 8770, 'dpap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Stonebranch Universal Enterprise Controller
    uec: 'AppType' = 8778, 'uec', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8778: 'AppType' = 8778, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Message Client
    #: - [UDP] Message Client
    msgclnt: 'AppType' = 8786, 'msgclnt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Message Server
    #: - [UDP] Message Server
    msgsrvr: 'AppType' = 8787, 'msgsrvr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Accedian Performance Measurement
    #: - [UDP] Accedian Performance Measurement
    acd_pm: 'AppType' = 8793, 'acd-pm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Sun Web Server Admin Service
    #: - [UDP] Sun Web Server Admin Service
    sunwebadmin: 'AppType' = 8800, 'sunwebadmin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] truecm
    #: - [UDP] truecm
    truecm: 'AppType' = 8804, 'truecm', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8805: 'AppType' = 8805, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Destination Port number for PFCP
    pfcp: 'AppType' = 8805, 'pfcp', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8806: 'AppType' = 8806, 'unassigned', TransportProtocol.get('undefined')

    #: [TCP] Reserved
    reserved_8807: 'AppType' = 8807, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] HES-CLIP Interoperability protocol
    hes_clip: 'AppType' = 8807, 'hes-clip', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8808: 'AppType' = 8808, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] STATSports Broadcast Service
    ssports_bcast: 'AppType' = 8808, 'ssports-bcast', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8809: 'AppType' = 8809, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] MCPTT Off-Network Protocol (MONP)
    AppType_3gpp_monp: 'AppType' = 8809, '3gpp-monp', TransportProtocol.get('udp')

    #: - [TCP] dxspider linking protocol
    #: - [UDP] dxspider linking protocol
    dxspider: 'AppType' = 8873, 'dxspider', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CDDBP
    #: - [UDP] CDDBP
    cddbp_alt: 'AppType' = 8880, 'cddbp-alt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Galaxy4D Online Game Engine
    galaxy4d: 'AppType' = 8881, 'galaxy4d', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8881: 'AppType' = 8881, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8882: 'AppType' = 8882, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] Secure MQTT
    #: - [UDP] Secure MQTT
    secure_mqtt: 'AppType' = 8883, 'secure-mqtt', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] NewsEDGE server TCP (TCP 1)
    ddi_tcp_1: 'AppType' = 8888, 'ddi-tcp-1', TransportProtocol.get('tcp')

    #: [UDP] NewsEDGE server UDP (UDP 1)
    ddi_udp_1: 'AppType' = 8888, 'ddi-udp-1', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 1
    ddi_tcp_2: 'AppType' = 8889, 'ddi-tcp-2', TransportProtocol.get('tcp')

    #: [UDP] NewsEDGE server broadcast
    ddi_udp_2: 'AppType' = 8889, 'ddi-udp-2', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 2
    ddi_tcp_3: 'AppType' = 8890, 'ddi-tcp-3', TransportProtocol.get('tcp')

    #: [UDP] NewsEDGE client broadcast
    ddi_udp_3: 'AppType' = 8890, 'ddi-udp-3', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 3: NESS application
    ddi_tcp_4: 'AppType' = 8891, 'ddi-tcp-4', TransportProtocol.get('tcp')

    #: [UDP] Desktop Data UDP 3: NESS application
    ddi_udp_4: 'AppType' = 8891, 'ddi-udp-4', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 4: FARM product
    ddi_tcp_5: 'AppType' = 8892, 'ddi-tcp-5', TransportProtocol.get('tcp')

    #: [UDP] Desktop Data UDP 4: FARM product
    ddi_udp_5: 'AppType' = 8892, 'ddi-udp-5', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 5: NewsEDGE/Web application
    ddi_tcp_6: 'AppType' = 8893, 'ddi-tcp-6', TransportProtocol.get('tcp')

    #: [UDP] Desktop Data UDP 5: NewsEDGE/Web application
    ddi_udp_6: 'AppType' = 8893, 'ddi-udp-6', TransportProtocol.get('udp')

    #: [TCP] Desktop Data TCP 6: COAL application
    ddi_tcp_7: 'AppType' = 8894, 'ddi-tcp-7', TransportProtocol.get('tcp')

    #: [UDP] Desktop Data UDP 6: COAL application
    ddi_udp_7: 'AppType' = 8894, 'ddi-udp-7', TransportProtocol.get('udp')

    #: - [TCP] ospf-lite
    #: - [UDP] ospf-lite
    ospf_lite: 'AppType' = 8899, 'ospf-lite', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JMB-CDS 1
    #: - [UDP] JMB-CDS 1
    jmb_cds1: 'AppType' = 8900, 'jmb-cds1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] JMB-CDS 2
    #: - [UDP] JMB-CDS 2
    jmb_cds2: 'AppType' = 8901, 'jmb-cds2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] WFA Device Provisioning Protocol
    dpp: 'AppType' = 8908, 'dpp', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8908: 'AppType' = 8908, 'reserved', TransportProtocol.get('udp')

    #: [N/A] Unassigned
    unassigned_8909: 'AppType' = 8909, 'unassigned', TransportProtocol.get('undefined')

    #: - [TCP] manyone-http
    #: - [UDP] manyone-http
    manyone_http: 'AppType' = 8910, 'manyone-http', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] manyone-xml
    #: - [UDP] manyone-xml
    manyone_xml: 'AppType' = 8911, 'manyone-xml', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Windows Client Backup
    #: - [UDP] Windows Client Backup
    wcbackup: 'AppType' = 8912, 'wcbackup', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Dragonfly System Service
    #: - [UDP] Dragonfly System Service
    dragonfly: 'AppType' = 8913, 'dragonfly', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Transaction Warehouse Data Service
    twds: 'AppType' = 8937, 'twds', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8937: 'AppType' = 8937, 'reserved', TransportProtocol.get('udp')

    #: [TCP] unbound dns nameserver control
    ub_dns_control: 'AppType' = 8953, 'ub-dns-control', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8953: 'AppType' = 8953, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Cumulus Admin Port
    #: - [UDP] Cumulus Admin Port
    cumulus_admin: 'AppType' = 8954, 'cumulus-admin', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Network of Devices Provider
    #: - [UDP] Network of Devices Provider
    nod_provider: 'AppType' = 8980, 'nod-provider', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [UDP] Network of Devices Client
    nod_client: 'AppType' = 8981, 'nod-client', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_8981: 'AppType' = 8981, 'reserved', TransportProtocol.get('tcp')

    #: - [TCP] Sun Web Server SSL Admin Service
    #: - [UDP] Sun Web Server SSL Admin Service
    sunwebadmins: 'AppType' = 8989, 'sunwebadmins', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] webmail HTTP service
    #: - [UDP] webmail HTTP service
    http_wmap: 'AppType' = 8990, 'http-wmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] webmail HTTPS service
    #: - [UDP] webmail HTTPS service
    https_wmap: 'AppType' = 8991, 'https-wmap', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Oracle Messaging Server Event Notification Service
    oracle_ms_ens: 'AppType' = 8997, 'oracle-ms-ens', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8997: 'AppType' = 8997, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Canto RoboFlow Control
    canto_roboflow: 'AppType' = 8998, 'canto-roboflow', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_8998: 'AppType' = 8998, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Brodos Crypto Trade Protocol
    #: - [UDP] Brodos Crypto Trade Protocol
    bctp: 'AppType' = 8999, 'bctp', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] CSlistener
    #: - [UDP] CSlistener
    cslistener: 'AppType' = 9000, 'cslistener', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] ETL Service Manager
    #: - [UDP] ETL Service Manager
    etlservicemgr: 'AppType' = 9001, 'etlservicemgr', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] DynamID authentication
    #: - [UDP] DynamID authentication
    dynamid: 'AppType' = 9002, 'dynamid', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Golem Inter-System RPC
    golem: 'AppType' = 9005, 'golem', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9005: 'AppType' = 9005, 'reserved', TransportProtocol.get('udp')

    #: [N/A] De-Commissioned Port
    de_commissioned_port: 'AppType' = 9006, 'de_commissioned_port', TransportProtocol.get('undefined')

    #: [TCP] Reserved
    reserved_9007: 'AppType' = 9007, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Open Grid Services Client
    ogs_client: 'AppType' = 9007, 'ogs-client', TransportProtocol.get('udp')

    #: [TCP] Open Grid Services Server
    ogs_server: 'AppType' = 9008, 'ogs-server', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9008: 'AppType' = 9008, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] Pichat Server
    #: - [UDP] Pichat Server
    pichat: 'AppType' = 9009, 'pichat', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Secure Data Replicator Protocol
    sdr: 'AppType' = 9010, 'sdr', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9010: 'AppType' = 9010, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_9011: 'AppType' = 9011, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] D-Star Routing digital voice+data for amateur radio
    d_star: 'AppType' = 9011, 'd-star', TransportProtocol.get('udp')

    #: - [TCP] TAMBORA
    #: - [UDP] TAMBORA
    tambora: 'AppType' = 9020, 'tambora', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Pangolin Identification
    #: - [UDP] Pangolin Identification
    panagolin_ident: 'AppType' = 9021, 'panagolin-ident', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] PrivateArk Remote Agent
    #: - [UDP] PrivateArk Remote Agent
    paragent: 'AppType' = 9022, 'paragent', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Web Access - 1
    #: - [UDP] Secure Web Access - 1
    swa_1: 'AppType' = 9023, 'swa-1', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Web Access - 2
    #: - [UDP] Secure Web Access - 2
    swa_2: 'AppType' = 9024, 'swa-2', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Web Access - 3
    #: - [UDP] Secure Web Access - 3
    swa_3: 'AppType' = 9025, 'swa-3', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Secure Web Access - 4
    #: - [UDP] Secure Web Access - 4
    swa_4: 'AppType' = 9026, 'swa-4', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Versiera Agent Listener
    versiera: 'AppType' = 9050, 'versiera', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9050: 'AppType' = 9050, 'reserved', TransportProtocol.get('udp')

    #: [TCP] Fusion-io Central Manager Service
    fio_cmgmt: 'AppType' = 9051, 'fio-cmgmt', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9051: 'AppType' = 9051, 'reserved', TransportProtocol.get('udp')

    #: [TCP] CardWeb request-response I/O exchange
    cardweb_io: 'AppType' = 9060, 'cardweb-io', TransportProtocol.get('tcp')

    #: [UDP] CardWeb realtime device data
    cardweb_rt: 'AppType' = 9060, 'cardweb-rt', TransportProtocol.get('udp')

    #: - [TCP] Groove GLRPC
    #: - [UDP] Groove GLRPC
    glrpc: 'AppType' = 9080, 'glrpc', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: [TCP] Reserved
    reserved_9081: 'AppType' = 9081, 'reserved', TransportProtocol.get('tcp')

    #: [UDP] Required for Adaptive Quality of Service
    cisco_aqos: 'AppType' = 9081, 'cisco-aqos', TransportProtocol.get('udp')

    #: [SCTP] LCS Application Protocol
    lcs_ap: 'AppType' = 9082, 'lcs-ap', TransportProtocol.get('sctp')

    #: [TCP] EMC PowerPath Mgmt Service
    emc_pp_mgmtsvc: 'AppType' = 9083, 'emc-pp-mgmtsvc', TransportProtocol.get('tcp')

    #: [UDP] Reserved
    reserved_9083: 'AppType' = 9083, 'reserved', TransportProtocol.get('udp')

    #: - [TCP] IBM AURORA Performance Visualizer
    #: - [UDP] IBM AURORA Performance Visualizer
    #: - [SCTP] IBM AURORA Performance Visualizer
    aurora: 'AppType' = 9084, 'aurora', TransportProtocol.get('tcp') | TransportProtocol.get('udp') | TransportProtocol.get('sctp')

    #: - [TCP] IBM Remote System Console
    #: - [UDP] IBM Remote System Console
    ibm_rsyscon: 'AppType' = 9085, 'ibm-rsyscon', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Vesa Net2Display
    #: - [UDP] Vesa Net2Display
    net2display: 'AppType' = 9086, 'net2display', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] Classic Data Server
    #: - [UDP] Classic Data Server
    classic: 'AppType' = 9087, 'classic', TransportProtocol.get('tcp') | TransportProtocol.get('udp')

    #: - [TCP] IBM Informix SQL Interface
    #: - [UDP] IBM Informix SQL Interface
    sqlexec: 'AppType' = 9088, 'sqlexec', TransportProtocol.get('tcp') | TransportProtocol.get('udp'