"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloneRepository = exports.Ec2Environment = exports.ConnectionType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const cloud9_generated_1 = require("../lib/cloud9.generated");
/**
 * The connection type used for connecting to an Amazon EC2 environment.
 */
var ConnectionType;
(function (ConnectionType) {
    /**
     * Conect through SSH
     */
    ConnectionType["CONNECT_SSH"] = "CONNECT_SSH";
    /**
     * Connect through AWS Systems Manager
     */
    ConnectionType["CONNECT_SSM"] = "CONNECT_SSM";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
/**
 * A Cloud9 Environment with Amazon EC2
 * @resource AWS::Cloud9::EnvironmentEC2
 */
class Ec2Environment extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloud9_Ec2EnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Ec2Environment);
            }
            throw error;
        }
        this.vpc = props.vpc;
        if (!props.subnetSelection && this.vpc.publicSubnets.length === 0) {
            throw new Error('no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection');
        }
        const vpcSubnets = props.subnetSelection ?? { subnetType: ec2.SubnetType.PUBLIC };
        const c9env = new cloud9_generated_1.CfnEnvironmentEC2(this, 'Resource', {
            name: props.ec2EnvironmentName,
            description: props.description,
            instanceType: props.instanceType?.toString() ?? ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO).toString(),
            subnetId: this.vpc.selectSubnets(vpcSubnets).subnetIds[0],
            repositories: props.clonedRepositories ? props.clonedRepositories.map(r => ({
                repositoryUrl: r.repositoryUrl,
                pathComponent: r.pathComponent,
            })) : undefined,
            connectionType: props.connectionType ?? ConnectionType.CONNECT_SSH,
        });
        this.environmentId = c9env.ref;
        this.ec2EnvironmentArn = c9env.getAtt('Arn').toString();
        this.ec2EnvironmentName = c9env.getAtt('Name').toString();
        this.ideUrl = `https://${this.env.region}.console.aws.amazon.com/cloud9/ide/${this.environmentId}`;
    }
    /**
     * import from EnvironmentEc2Name
     */
    static fromEc2EnvironmentName(scope, id, ec2EnvironmentName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.ec2EnvironmentName = ec2EnvironmentName;
                this.ec2EnvironmentArn = cdk.Stack.of(this).formatArn({
                    service: 'cloud9',
                    resource: 'environment',
                    resourceName: this.ec2EnvironmentName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.Ec2Environment = Ec2Environment;
_a = JSII_RTTI_SYMBOL_1;
Ec2Environment[_a] = { fqn: "@aws-cdk/aws-cloud9.Ec2Environment", version: "1.162.0" };
/**
 * The class for different repository providers
 */
class CloneRepository {
    constructor(repositoryUrl, pathComponent) {
        this.repositoryUrl = repositoryUrl;
        this.pathComponent = pathComponent;
    }
    /**
     * import repository to cloud9 environment from AWS CodeCommit
     *
     * @param repository the codecommit repository to clone from
     * @param path  the target path in cloud9 environment
     */
    static fromCodeCommit(repository, path) {
        return {
            repositoryUrl: repository.repositoryCloneUrlHttp,
            pathComponent: path,
        };
    }
}
exports.CloneRepository = CloneRepository;
_b = JSII_RTTI_SYMBOL_1;
CloneRepository[_b] = { fqn: "@aws-cdk/aws-cloud9.CloneRepository", version: "1.162.0" };
//# sourceMappingURL=data:application/json;base64,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