"""
Test the parsing of full manifests
"""
import os

from pytest import mark
from mpd_parser.parser import Parser

from tests.conftest import touch_attributes, MANIFESTS_DIR


@mark.parametrize("input_file", [
    "./../manifests/bigBuckBunny-onDemend.mpd",
])
def test_mpd_tag(input_file):
    """ Test that parser works and create MPD object """
    with open(input_file, mode="r", encoding='UTF-8') as manifest_file:
        mpd_string = manifest_file.read()
        mpd = Parser.from_string(mpd_string)
        assert mpd.id is None
        assert mpd.type == "static"
        assert mpd.min_buffer_time == "PT1.500000S"
        assert mpd.media_presentation_duration == "PT0H9M55.46S"
        assert mpd.profiles == "urn:mpeg:dash:profile:isoff-on-demand:2011"
        assert len(mpd.program_informations) == 1
        assert mpd.program_informations[0].titles[0].text == \
               "dashed/BigBuckBunny_1s_onDemand_2014_05_09.mpd generated by GPAC"
        assert mpd.program_informations[0].more_info_url == "http://gpac.sourceforge.net"
        assert len(mpd.periods) == 1
        assert mpd.periods[0].id is None
        assert mpd.periods[0].duration == "PT0H9M55.46S"


@mark.parametrize("input_file", [
    "./../manifests/bigBuckBunny-onDemend.mpd",
])
def test_parse_from_file_mpd_tag(input_file):
    """ Test that parser works and create MPD object """
    mpd = Parser.from_file(input_file)
    assert mpd.id is None
    assert mpd.type == "static"
    assert mpd.min_buffer_time == "PT1.500000S"
    assert mpd.media_presentation_duration == "PT0H9M55.46S"
    assert mpd.profiles == "urn:mpeg:dash:profile:isoff-on-demand:2011"
    assert len(mpd.program_informations) == 1
    assert mpd.program_informations[0].titles[0].text == \
           "dashed/BigBuckBunny_1s_onDemand_2014_05_09.mpd generated by GPAC"
    assert mpd.program_informations[0].more_info_url == "http://gpac.sourceforge.net"
    assert len(mpd.periods) == 1
    assert mpd.periods[0].id is None
    assert mpd.periods[0].duration == "PT0H9M55.46S"

@mark.parametrize("url", [
    "https://bitmovin-a.akamaihd.net/content/MI201109210084_1/mpds/f08e80da-bf1d-4e3d-8899-f0f6155f6efa.mpd",
])
def test_parse_from_url_mpd_tag(url):
    """ Test that parser works and create MPD object from URL target """
    mpd = Parser.from_url(url)
    assert mpd.id == "f08e80da-bf1d-4e3d-8899-f0f6155f6efa"
    assert mpd.type == "static"
    assert mpd.min_buffer_time == "P0Y0M0DT0H0M1.000S"
    assert mpd.media_presentation_duration == "P0Y0M0DT0H3M30.000S"
    assert mpd.profiles == "urn:mpeg:dash:profile:isoff-main:2011"
    assert mpd.program_informations == []
    assert len(mpd.periods) == 1
    assert mpd.periods[0].id is None
    assert mpd.periods[0].duration is None
    assert mpd.publish_time == "2015-08-04T10:47:32.000Z"
    assert len(mpd.periods[0].adaptation_sets) == 2
    assert mpd.periods[0].adaptation_sets[0].mime_type == "video/mp4"
    assert mpd.periods[0].adaptation_sets[0].codecs == "avc1.42c00d"


@mark.parametrize("parsing_type", [Parser.from_file, Parser.from_string])
@mark.parametrize("input_file", [f"{MANIFESTS_DIR}{name}" for name in os.listdir(MANIFESTS_DIR)])
def test_touch_all_manifest_properties(input_file, parsing_type):
    """
        Test each manifest by walking over it's xml tree.
    Does not verify values.
    """
    if parsing_type == Parser.from_file:
        mpd = parsing_type(input_file)
        touch_attributes(mpd)
        return

    with open(input_file, mode="r", encoding='UTF-8') as manifest_file:
        mpd_string = manifest_file.read()
        mpd = parsing_type(mpd_string)
    touch_attributes(mpd)


@mark.parametrize("input_file", [
    "./../manifests/bigBuckBunny-onDemend.mpd",
])
def test_to_string(input_file):
    """ test the converter from MPD object to string xml """
    with open(input_file, mode="r") as manifest_file:
        mpd_string = manifest_file.read()
        orig_mpd = Parser.from_string(mpd_string)
        mpd_result = Parser.to_string(orig_mpd)
        transformed_mpd = Parser.from_string(mpd_result)
        assert orig_mpd.program_informations[0].more_info_url == transformed_mpd.program_informations[0].more_info_url
        assert orig_mpd.periods[0].adaptation_sets[0].representations[0].segment_bases[0].initializations[
                   0].range == \
               transformed_mpd.periods[0].adaptation_sets[0].representations[0].segment_bases[0].initializations[
                   0].range
