"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_apigatewayv2_1 = require("aws-cdk-lib/aws-apigatewayv2");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(method, path) {
        this.method = method ?? HttpMethod.ANY;
        this.path = path;
        this.key = path ? `${method} ${path}` : '$default';
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_HttpMethod(method);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.with);
            }
            throw error;
        }
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('A route path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(method, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
_a = JSII_RTTI_SYMBOL_1;
HttpRouteKey[_a] = { fqn: "@aws-cdk/aws-apigatewayv2-alpha.HttpRouteKey", version: "2.65.0-alpha.0" };
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey();
/**
 * Supported Route Authorizer types
 */
var HttpRouteAuthorizationType;
(function (HttpRouteAuthorizationType) {
    /** AWS IAM */
    HttpRouteAuthorizationType["AWS_IAM"] = "AWS_IAM";
    /** JSON Web Tokens */
    HttpRouteAuthorizationType["JWT"] = "JWT";
    /** Lambda Authorizer */
    HttpRouteAuthorizationType["CUSTOM"] = "CUSTOM";
    /** No authorizer */
    HttpRouteAuthorizationType["NONE"] = "NONE";
})(HttpRouteAuthorizationType || (HttpRouteAuthorizationType = {}));
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_HttpRouteProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, HttpRoute);
            }
            throw error;
        }
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        this.method = props.routeKey.method;
        this.routeArn = this.produceRouteArn(props.routeKey.method);
        const config = props.integration._bindToRoute({
            route: this,
            scope: this,
        });
        this.authBindResult = props.authorizer?.bind({
            route: this,
            scope: this.httpApi instanceof constructs_1.Construct ? this.httpApi : this,
        });
        if (this.authBindResult && !(this.authBindResult.authorizationType in HttpRouteAuthorizationType)) {
            throw new Error(`authorizationType should either be AWS_IAM, JWT, CUSTOM, or NONE but was '${this.authBindResult.authorizationType}'`);
        }
        let authorizationScopes = this.authBindResult?.authorizationScopes;
        if (this.authBindResult && props.authorizationScopes) {
            authorizationScopes = Array.from(new Set([
                ...authorizationScopes ?? [],
                ...props.authorizationScopes,
            ]));
        }
        if (authorizationScopes?.length === 0) {
            authorizationScopes = undefined;
        }
        const routeProps = {
            apiId: props.httpApi.apiId,
            routeKey: props.routeKey.key,
            target: `integrations/${config.integrationId}`,
            authorizerId: this.authBindResult?.authorizerId,
            authorizationType: this.authBindResult?.authorizationType ?? 'NONE',
            authorizationScopes,
        };
        const route = new aws_apigatewayv2_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
    produceRouteArn(httpMethod) {
        const stage = '*';
        const iamHttpMethod = httpMethod === HttpMethod.ANY ? '*' : httpMethod;
        const path = this.path ?? '/';
        // When the user has provided a path with path variables, we replace the
        // path variable and all that follows with a wildcard.
        const iamPath = path.replace(/\{.*?\}.*/, '*');
        return `arn:aws:execute-api:${this.env.region}:${this.env.account}:${this.httpApi.apiId}/${stage}/${iamHttpMethod}${iamPath}`;
    }
    grantInvoke(grantee, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_GrantInvokeOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantInvoke);
            }
            throw error;
        }
        if (!this.authBindResult || this.authBindResult.authorizationType !== HttpRouteAuthorizationType.AWS_IAM) {
            throw new Error('To use grantInvoke, you must use IAM authorization');
        }
        const httpMethods = Array.from(new Set(options.httpMethods ?? [this.method]));
        if (this.method !== HttpMethod.ANY && httpMethods.some(method => method !== this.method)) {
            throw new Error('This route does not support granting invoke for all requested http methods');
        }
        const resourceArns = httpMethods.map(httpMethod => {
            return this.produceRouteArn(httpMethod);
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['execute-api:Invoke'],
            resourceArns: resourceArns,
        });
    }
}
exports.HttpRoute = HttpRoute;
_b = JSII_RTTI_SYMBOL_1;
HttpRoute[_b] = { fqn: "@aws-cdk/aws-apigatewayv2-alpha.HttpRoute", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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