# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/implementation/01_validators.ipynb.

# %% auto 0
__all__ = ['AnyArg', 'ArgumentValidator', 'ArgumentFunctionValidator', 'argument_validator_from_argument', 'AnyInt']

# %% ../nbs/implementation/01_validators.ipynb 2
from typing import Protocol, Any, runtime_checkable
from collections.abc import Callable
from .core import AnyCallable

# %% ../nbs/implementation/01_validators.ipynb 8
@runtime_checkable
class ArgumentValidator(Protocol):
    "Interface for all argument validators."
    
    @property
    def name() -> str:
        "Name of the argument in the signature"
        
    @property
    def position() -> int:
        "Position of the argument in the signature"
        
    def is_valid(self, argument: Any) -> bool:
        ...

# %% ../nbs/implementation/01_validators.ipynb 10
class ArgumentFunctionValidator:
    "Validate an argument by evaluating an arbitrary function"
    def __init__(self, func: AnyCallable[bool], name: str, position: int, display:str|None = None):
        self._func = func
        self._name = name
        self._position = position
        self._display = display
        
    @property
    def name(self) -> str:
        return self._name
    
    @property
    def position(self) -> int:
        return self._position
        
    def is_valid(self, argument: Any) -> bool:
        return self._func(argument)
    
    def __str__(self):
        if self._display is None:
            return f'ArgumentFunctionValidator(name:{self.name}, position={self.position})'
        return f'ArgumentFunctionValidator(argument_name:{self.name}, position={self.position}): {self._display}'
    def __repr__(self): return str(self)
    
assert isinstance(ArgumentFunctionValidator, ArgumentValidator), "ArgumentFunctionValidator does not implement the ArgumentValidator-Protocol"

# %% ../nbs/implementation/01_validators.ipynb 16
def argument_validator_from_argument(argument: Any, name:str, position: int, verbose:bool=False) -> ArgumentValidator:
    if verbose: print(f"Constructing ArgumentValidatorFrom {argument}")
    match argument:
        case ArgumentValidator():
            return argument
        case type():
            return ArgumentFunctionValidator(lambda v: isinstance(v, argument), name=name, position=position, display=f'any_{argument.__name__}')
        case Callable():
            if hasattr(argument, 'display'):
                display = argument.display
            else:
                display = 'callable()'
            
            return ArgumentFunctionValidator(argument, name=name, position=position, display=display)
    
    return ArgumentFunctionValidator(lambda v: v==argument, name=name, position=position, display=f'== {argument}')

# %% ../nbs/implementation/01_validators.ipynb 33
AnyArg = lambda: lambda v: True
AnyArg.display = 'any()'

# %% ../nbs/implementation/01_validators.ipynb 34
class AnyInt:
    "Special validator that provides methods for integers"
    def __init__(self,name: str, position: int, display:str|None = None):
        self._name = name
        self._position = position
        self._display = display
        
        self._validators: list[ArgumentValidator] = [argument_validator_from_argument(int, name=name, position=position)]
        self._validator_names: list[str] = ['AnyInt()']
    
    @property
    def name(self) -> str:
        return self._name
    
    @property
    def position(self) -> int:
        return self._position
    
    def greather_than(self, lower: int) -> "AnyInt":
        greather_func = lambda value: lower<value
        self._validator_names.append(f"greather_than({lower})")
        
        self._validators.append(argument_validator_from_argument(greather_func, name=self._name, position=self._position))
        return self
    
    def greather_than_or_equal(self, lower: int) -> "AnyInt":
        greather_func = lambda value: lower<=value
        self._validator_names.append(f"greather_than_or_equal({lower})")
        
        self._validators.append(argument_validator_from_argument(greather_func, name=self._name, position=self._position))
        return self
    
    def less_than(self, upper: int) -> "AnyInt":
        less_func = lambda value: value<upper
        self._validator_names.append(f"less_than({upper})")
        
        self._validators.append(argument_validator_from_argument(less_func, name=self._name, position=self._position))
        return self
    
    def less_than_or_equal(self, upper: int) -> "AnyInt":
        less_func = lambda value: value<=upper
        self._validator_names.append(f"less_than_or_equal({upper})")
        
        self._validators.append(argument_validator_from_argument(less_func, name=self._name, position=self._position))
        return self
        
    def is_valid(self, argument: Any) -> bool:
        return all(validator.is_valid(argument) for validator in self._validators)
    
    def __str__(self):
        return '.'.join(self._validator_names)
    
    def __repr__(self): return str(self)
