"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const sinon = require("sinon");
const backoff_generator_1 = require("../../lib/backoff-generator");
const dynamodb_1 = require("../../lib/dynamodb");
const x509_certs_1 = require("../../lib/x509-certs");
const acm_handlers_1 = require("../acm-handlers");
describe('AcmCertificateImporter', () => {
    const physicalId = 'physicalId';
    const certArn = 'certArn';
    const oldEnv = process.env;
    let consoleWarnSpy;
    beforeAll(() => {
        jest.spyOn(global.console, 'log').mockImplementation(() => { });
        jest.spyOn(global.console, 'error').mockImplementation(() => { });
        consoleWarnSpy = jest.spyOn(global.console, 'warn').mockImplementation(() => { });
    });
    afterAll(() => {
        jest.restoreAllMocks();
    });
    beforeEach(() => {
        process.env.DATABASE = 'database';
        AWSMock.setSDKInstance(AWS);
    });
    afterEach(() => {
        process.env = oldEnv;
        sinon.restore();
        AWSMock.restore();
    });
    describe('doCreate', () => {
        const doCreateProps = {
            Tags: [],
            X509CertificatePem: {
                Cert: 'cert',
                CertChain: 'certChain',
                Key: 'key',
                Passphrase: 'passphrase',
            },
        };
        beforeEach(() => {
            sinon.stub(x509_certs_1.Certificate, 'decryptKey').returns(Promise.resolve('key'));
            // Mock out the API call in getSecretString
            AWSMock.mock('SecretsManager', 'getSecretValue', sinon.fake.resolves({ SecretString: 'secret' }));
        });
        test('throws when a secret does not have SecretString', async () => {
            // GIVEN
            const getSecretValueFake = sinon.fake.resolves({});
            AWSMock.remock('SecretsManager', 'getSecretValue', getSecretValueFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: new MockCompositeStringIndexTable(),
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .rejects.toThrow(/Secret .* did not contain a SecretString as expected/);
            expect(getSecretValueFake.calledOnce).toBe(true);
        });
        test('retries importing certificate', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
            const putItemStub = sinon.stub(resourceTable, 'putItem').resolves(true);
            const importCertificateStub = sinon.stub()
                .onFirstCall().rejects('Rate exceeded')
                .onSecondCall().rejects('Rate exceeded')
                .onThirdCall().resolves({ CertificateArn: certArn });
            AWSMock.mock('ACM', 'importCertificate', importCertificateStub);
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff').resolves(true);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .resolves.toEqual({ CertificateArn: certArn });
            expect(getItemStub.calledOnce).toBe(true);
            expect(putItemStub.calledOnce).toBe(true);
            expect(importCertificateStub.calledThrice).toBe(true);
            expect(backoffStub.callCount).toEqual(2);
        });
        test('throws after max import retries', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
            const attempts = 10;
            const importCertificateStub = sinon.stub();
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff');
            for (let i = 0; i < attempts; i++) {
                importCertificateStub.onCall(i).rejects('Rate exceeded');
                backoffStub.onCall(i).resolves(i < attempts - 1);
            }
            AWSMock.mock('ACM', 'importCertificate', importCertificateStub);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .rejects.toThrow(/Failed to import certificate .* after [0-9]+ attempts\./);
            expect(getItemStub.calledOnce).toBe(true);
            expect(importCertificateStub.callCount).toBe(attempts);
            expect(backoffStub.callCount).toEqual(attempts);
        });
        describe('existing', () => {
            test('throws if item ARN is missing', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({});
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toEqual(new Error("Database Item missing 'ARN' attribute"));
                expect(getItemStub.calledOnce).toBe(true);
            });
            test('throws if certificate not found in ACM', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({ ARN: certArn });
                const getCertificateFake = sinon.fake.rejects({});
                AWSMock.mock('ACM', 'getCertificate', getCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toThrow(new RegExp(`Database entry ${certArn} could not be found in ACM:`));
                expect(getItemStub.calledOnce).toBe(true);
                expect(getCertificateFake.calledOnce).toBe(true);
            });
            test('imports certificate', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({ ARN: certArn });
                const getCertificateFake = sinon.fake.resolves({ Certificate: 'cert' });
                AWSMock.mock('ACM', 'getCertificate', getCertificateFake);
                const importCertificateFake = sinon.fake.resolves({});
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .resolves.toEqual({ CertificateArn: certArn });
                expect(getItemStub.calledOnce).toBe(true);
                expect(getCertificateFake.calledOnce).toBe(true);
                // Verify that we import the existing certificate to support replacing/updating of it (e.g. to rotate certs)
                expect(importCertificateFake.calledOnce).toBe(true);
            });
        });
        describe('new', () => {
            test('throws if CertificateArn not populated', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
                const importCertificateFake = sinon.fake.resolves({});
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toThrow(/CertificateArn was not properly populated after attempt to import .*$/);
                expect(getItemStub.calledOnce).toBe(true);
                expect(importCertificateFake.calledOnce).toBe(true);
            });
            test('imports certificate', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
                const putItemStub = sinon.stub(resourceTable, 'putItem').resolves(true);
                const importCertificateFake = sinon.fake.resolves({ CertificateArn: certArn });
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .resolves.toEqual({ CertificateArn: certArn });
                expect(getItemStub.calledOnce).toBe(true);
                expect(putItemStub.calledOnce).toBe(true);
                expect(importCertificateFake.calledOnce).toBe(true);
            });
        });
    });
    describe('doDelete', () => {
        test('throws if describeCertificate is in use after max attempts', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: ['something'] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            // This is hardcoded in the code being tested
            const maxAttempts = 10;
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff').resolves();
            const shouldContinueStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'shouldContinue')
                .returns(true)
                .onCall(maxAttempts - 1).returns(false);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`));
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.callCount).toEqual(maxAttempts);
            expect(backoffStub.callCount).toEqual(maxAttempts);
            expect(shouldContinueStub.callCount).toEqual(maxAttempts);
        });
        test('throws when deleting certificate from ACM fails', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const error = new Error('error');
            const deleteCertificateFake = sinon.fake.rejects(error);
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(error);
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
        });
        test('warns when deleting certificate from ACM fails with AccessDeniedException', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const error = new Error('AccessDeniedException');
            const deleteCertificateFake = sinon.fake.rejects(error);
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(error);
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
            expect(consoleWarnSpy.mock.calls.length).toBeGreaterThanOrEqual(1);
            expect(consoleWarnSpy.mock.calls.map(args => args[0]).join('\n')).toMatch(new RegExp(`Could not delete Certificate ${certArn}. Please ensure it has been deleted.`));
        });
        test('deletes the certificate', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({ key: { ARN: certArn } });
            const deleteItemStub = sinon.stub(resourceTable, 'deleteItem').resolves(true);
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const deleteCertificateFake = sinon.fake.resolves({});
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .resolves.not.toThrow();
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
            expect(deleteItemStub.calledOnce).toBe(true);
        });
    });
});
/**
 * Specialization of AcmCertificateImporter that overrides methods inherited from
 * DynamoBackedResource so that no API calls are made.
 *
 * This allows the testing code above to focus on the testing the AcmCertificateImporter
 * class without having to deal with mocking out API calls from its parent class.
 */
class TestAcmCertificateImporter extends acm_handlers_1.AcmCertificateImporter {
    constructor(props) {
        var _a;
        super(props.acm, props.dynamoDb, props.secretsManager);
        this.resourceTableOverride = (_a = props.resourceTableOverride) !== null && _a !== void 0 ? _a : new MockCompositeStringIndexTable();
    }
    async databasePermissionsCheck() {
        // Do nothing
        return;
    }
    async getResourceTable() {
        return this.resourceTableOverride;
    }
}
/**
 * Mock implementation of CompositeStringIndexTable that does not make API calls.
 *
 * This allows the test code above to instantiate a CompositeStringIndexTable object
 * that can be mocked.
 */
class MockCompositeStringIndexTable extends dynamodb_1.CompositeStringIndexTable {
    constructor() {
        super(new AWS.DynamoDB(), '', '', '');
    }
    async deleteTable() { }
    async putItem(_props) {
        return true;
    }
    async getItem(_props) {
        return {};
    }
    async deleteItem(_props) {
        return true;
    }
    async query(_primaryKeyValue, _pageLimit) {
        return {};
    }
}
//# sourceMappingURL=data:application/json;base64,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