"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.VersionProviderResource = void 0;
const custom_resource_1 = require("../lib/custom-resource");
const version_provider_1 = require("../lib/version-provider");
/**
 * This custom resource will parse and return the S3 URI's of the Deadline installers and Docker recipes for use in
 * any constructs that need to install Deadline.
 */
class VersionProviderResource extends custom_resource_1.SimpleCustomResource {
    constructor() {
        super();
        this.versionProvider = new version_provider_1.VersionProvider();
    }
    /**
     * @inheritdoc
     */
    validateInput(data) {
        return this.implementsIVersionProviderResourceProperties(data);
    }
    /**
     * @inheritdoc
     */
    // @ts-ignore  -- we do not use the physicalId
    async doCreate(physicalId, resourceProperties) {
        const deadlinePlatFormVersionedUris = await this.versionProvider.getVersionUris({
            versionString: resourceProperties.versionString,
            platform: version_provider_1.Platform.linux,
            product: version_provider_1.Product.deadline,
        });
        const deadlineLinux = deadlinePlatFormVersionedUris.get(version_provider_1.Platform.linux);
        const deadlineLinuxUris = deadlineLinux.Uris;
        const s3Bucket = this.parseS3BucketName(deadlineLinuxUris.bundle);
        const linuxRepoObjectKey = this.parseS3ObjectKey(deadlineLinuxUris.repositoryInstaller);
        const linuxClientObjectKey = this.parseS3ObjectKey(deadlineLinuxUris.clientInstaller);
        return {
            S3Bucket: s3Bucket,
            MajorVersion: deadlineLinux.MajorVersion,
            MinorVersion: deadlineLinux.MinorVersion,
            ReleaseVersion: deadlineLinux.ReleaseVersion,
            LinuxPatchVersion: deadlineLinux.PatchVersion,
            LinuxRepositoryInstaller: linuxRepoObjectKey,
            LinuxClientInstaller: linuxClientObjectKey,
        };
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    async doDelete(physicalId, resourceProperties) {
        // Nothing to do -- we don't modify anything.
        return;
    }
    implementsIVersionProviderResourceProperties(value) {
        if (!value || typeof (value) !== 'object') {
            return false;
        }
        if (value.versionString) {
            if (!version_provider_1.Version.validateVersionString(value.versionString)) {
                console.log(`Failed to validate the version string: ${value.versionString}`);
                return false;
            }
        }
        if (value.forceRun && typeof (value.forceRun) !== 'string') {
            return false;
        }
        return true;
    }
    /**
     * Parses the S3 bucket name from an S3 URI.
     */
    parseS3BucketName(uri) {
        let bucketName;
        try {
            bucketName = this.findRegex(uri, /^s3:\/\/([A-Za-z0-9\-]+)\//)[1];
        }
        catch (e) {
            throw new Error(`Could not parse S3 bucket name from ${uri}`);
        }
        return bucketName;
    }
    /**
     * Parses the S3 object key from an S3 URI.
     */
    parseS3ObjectKey(uri) {
        let objectKey;
        try {
            objectKey = this.findRegex(uri, /^s3:\/\/[A-Za-z0-9\-]+\/([A-Za-z0-9\-\/\.]+)$/)[1];
        }
        catch (e) {
            throw new Error(`Could not parse S3 object key from ${uri}`);
        }
        return objectKey;
    }
    // Assumes a single capture is in the regex
    findRegex(str, re) {
        const found = str.match(re);
        if (found === null) {
            throw new Error(`Couldn't find regular expression ${re} in ${str}`);
        }
        return found;
    }
}
exports.VersionProviderResource = VersionProviderResource;
/**
 * The handler used to provide the installer links for the requested version
 */
async function handler(event, context) {
    const versionProvider = new VersionProviderResource();
    return await versionProvider.handler(event, context);
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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