/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { LambdaContext } from '../lib/aws-lambda';
/**
 * Provided solely for the purpose of testing to shrink the default file size from 1GiB
 * to something smaller.
 * @param filesize Desired filesize in MiB. Must be a multiple of 10
 */
export declare function setDefaultFilesize(filesize: number): void;
/**
 * Add numbered files (e.g. 00000, 00001) of a given size to a filesystem.
 * Note: exported so that we can test it.
 * @param numFilesToAdd How many numbered files to add.
 * @param filesize Size, in MiB, of the files to create.
 * @param mountPoint Directory in which to create the directory.
 */
export declare function growFilesystem(numFilesToAdd: number, filesize: number, mountPoint: string): Promise<void>;
/**
 * Delete a given number of numbered files from the given filesystem.
 * Note: exported so that we can test it.
 * @param numFilesToRemove How many files to remove from the directory.
 * @param mountPoint Directory from which to delete files.
 */
export declare function shrinkFilesystem(numFilesToRemove: number, mountPoint: string): Promise<void>;
/**
 * Lambda handler. Expected to be invoked from a step function.
 * Calculates the disk size under the given directory. This is equivalent to calling:
 * du -sh -BG <directory>
 * @param event { "mountPoint": <string> }
 * @param context
 * @returns Disk usage in GiB
 */
export declare function getDiskUsage(event: {
    [key: string]: string;
}, context: LambdaContext): Promise<number>;
/**
 * Lambda handler. Expected to be invoked from a step function.
 * Adds or removes files from a given EFS filesystem to pad the filesystem with data.
 *
 * If the filesystem is padded to less than the number of desired GiB then files are added as numbered
 * files 1GiB in size to the given 'mountPoint'; adding at most 20 on each invocation.
 *
 * If the filesystem is padded with more than the desired GiB then numbered files are removed from the
 * given filesystem. Each numbered file is assumed to be exactly 1GiB in size.
 * @param event {
 *    "desiredPadding": "<integer number of GiB>",
 *    "mountPoint": "<string>"
 * }
 * @param context
 * @returns Nothing
 */
export declare function padFilesystem(event: {
    [key: string]: string;
}, context: LambdaContext): Promise<void>;
