"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const read_certificate_1 = require("../read-certificate");
const secretPartialArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('readCertificateData', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const certData = 'BEGIN CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const data = await read_certificate_1.readCertificateData(secretPartialArn, client);
        // THEN
        expect(data).toStrictEqual(certData);
    });
    test('not a certificate', async () => {
        // GIVEN
        const certData = 'NOT A CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const promise = read_certificate_1.readCertificateData(secretPartialArn, client);
        // THEN
        await expect(promise).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
    test('binary data', async () => {
        // GIVEN
        const certData = Buffer.from('BEGIN CERTIFICATE', 'utf-8');
        const secretContents = {
            SecretBinary: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const promise = read_certificate_1.readCertificateData(secretPartialArn, client);
        // THEN
        await expect(promise).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
});
//# sourceMappingURL=data:application/json;base64,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