"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const handler_1 = require("../handler");
jest.mock('../../lib/ecr-provider', () => {
    class ThinkboxEcrProviderMock {
        constructor(indexFilePath) {
            ThinkboxEcrProviderMock.mocks.constructor(indexFilePath);
        }
        getGlobalEcrBaseURI() {
            return ThinkboxEcrProviderMock.mocks.getGlobalEcrBaseURI();
        }
    }
    ThinkboxEcrProviderMock.mocks = {
        constructor: jest.fn(),
        getGlobalEcrBaseURI: jest.fn(() => {
            return Promise.resolve('public.ecr.aws/deadline/');
        }),
    };
    return {
        ThinkboxEcrProvider: ThinkboxEcrProviderMock,
    };
});
jest.mock('https');
describe('ThinkboxEcrProviderResource', () => {
    let ecrProviderResource;
    beforeAll(() => {
        // Suppress console output during tests
        jest.spyOn(console, 'log').mockReturnValue(undefined);
        jest.spyOn(console, 'info').mockReturnValue(undefined);
        jest.spyOn(console, 'warn').mockReturnValue(undefined);
        jest.spyOn(console, 'error').mockReturnValue(undefined);
    });
    beforeEach(() => {
        ecrProviderResource = new handler_1.ThinkboxEcrProviderResource();
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    afterAll(() => {
        jest.clearAllMocks();
    });
    describe('.validateInput()', () => {
        // Valid configurations
        describe('should return true if', () => {
            test.each([
                'testValue',
                undefined,
            ])('{ForceRun=%s}', async (forceRun) => {
                // GIVEN
                const input = {
                    forceRun,
                };
                // WHEN
                const returnValue = ecrProviderResource.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
        });
        // Invalid configurations
        const invalidConfigs = [
            { ForceRun: 1 },
            { ForceRun: [1] },
            { ForceRun: { a: 1 } },
            [],
            'abc',
            1,
        ];
        describe('should return false if', () => {
            test.each(invalidConfigs.map((config) => {
                return [
                    JSON.stringify(config),
                    config,
                ];
            }))('%s', async (_str, config) => {
                expect(ecrProviderResource.validateInput(config)).toBeFalsy();
            });
        });
    });
    describe('uses ThinkboxEcrProvider', () => {
        test('global', async () => {
            // GIVEN
            const mockBaseURI = 'baseURI';
            const ThinkboxEcrProvider = jest.requireMock('../../lib/ecr-provider').ThinkboxEcrProvider;
            ThinkboxEcrProvider.mocks.getGlobalEcrBaseURI.mockReturnValue(Promise.resolve(mockBaseURI));
            // WHEN
            const promise = ecrProviderResource.doCreate('someID', {
                ForceRun: 'forceRun',
            });
            const result = await promise;
            // THEN
            expect(ThinkboxEcrProvider.mocks.constructor).toHaveBeenCalledTimes(1);
            expect(ThinkboxEcrProvider.mocks.getGlobalEcrBaseURI).toHaveBeenCalledTimes(1);
            expect(result).toEqual({
                EcrURIPrefix: mockBaseURI,
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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