"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateStringOptional = exports.validateString = exports.convertToBoolean = exports.convertToInt = exports.validateLaunchTemplateConfigs = exports.validateLaunchTemplateOverrides = exports.validateLaunchTemplateSpecification = exports.convertTagSpecifications = exports.isValidTagSpecification = exports.validateProperty = exports.validateArray = exports.convertSpotEventPluginSettings = exports.convertSpotFleetRequestConfiguration = void 0;
/**
 * Convert the configuration we received from ConfigureSpotEventPlugin construct to the fromat expected by the Spot Event Plugin.
 * boolean and number properties get converted into strings when passed to the Lambda,
 * so we need to restore the original types.
 */
function convertSpotFleetRequestConfiguration(spotFleetRequestConfigs) {
    const convertedSpotFleetRequestConfigs = {};
    for (const [group_name, sfrConfigs] of Object.entries(spotFleetRequestConfigs)) {
        const convertedSpotFleetRequestProps = {
            AllocationStrategy: validateString(sfrConfigs.AllocationStrategy, `${group_name}.AllocationStrategy`),
            IamFleetRole: validateString(sfrConfigs.IamFleetRole, `${group_name}.IamFleetRole`),
            // Empty array needed for compatibility with SEP since it expects an array for the LaunchSpecifications property
            LaunchSpecifications: [],
            LaunchTemplateConfigs: sfrConfigs.LaunchTemplateConfigs ? validateLaunchTemplateConfigs(sfrConfigs.LaunchTemplateConfigs, `${group_name}.LaunchTemplateConfigs`) : undefined,
            ReplaceUnhealthyInstances: convertToBoolean(sfrConfigs.ReplaceUnhealthyInstances, `${group_name}.ReplaceUnhealthyInstances`),
            TargetCapacity: convertToInt(sfrConfigs.TargetCapacity, `${group_name}.TargetCapacity`),
            TerminateInstancesWithExpiration: convertToBoolean(sfrConfigs.TerminateInstancesWithExpiration, `${group_name}.TerminateInstancesWithExpiration`),
            Type: validateString(sfrConfigs.Type, `${group_name}.Type`),
            ValidUntil: validateStringOptional(sfrConfigs.ValidUntil, `${group_name}.ValidUntil`),
            TagSpecifications: convertTagSpecifications(sfrConfigs.TagSpecifications, `${group_name}.TagSpecifications`),
        };
        convertedSpotFleetRequestConfigs[group_name] = convertedSpotFleetRequestProps;
    }
    return convertedSpotFleetRequestConfigs;
}
exports.convertSpotFleetRequestConfiguration = convertSpotFleetRequestConfiguration;
/**
 * Convert the configuration we received from ConfigureSpotEventPlugin construct to the fromat expected by the Spot Event Plugin.
 * boolean and number properties get converted into strings when passed to the Lambda,
 * so we need to restore the original types.
 */
function convertSpotEventPluginSettings(pluginOptions) {
    return {
        AWSInstanceStatus: validateString(pluginOptions.AWSInstanceStatus, 'AWSInstanceStatus'),
        DeleteInterruptedSlaves: convertToBoolean(pluginOptions.DeleteInterruptedSlaves, 'DeleteInterruptedSlaves'),
        DeleteTerminatedSlaves: convertToBoolean(pluginOptions.DeleteTerminatedSlaves, 'DeleteTerminatedSlaves'),
        IdleShutdown: convertToInt(pluginOptions.IdleShutdown, 'IdleShutdown'),
        Logging: validateString(pluginOptions.Logging, 'Logging'),
        PreJobTaskMode: validateString(pluginOptions.PreJobTaskMode, 'PreJobTaskMode'),
        Region: validateString(pluginOptions.Region, 'Region'),
        ResourceTracker: convertToBoolean(pluginOptions.ResourceTracker, 'ResourceTracker'),
        StaggerInstances: convertToInt(pluginOptions.StaggerInstances, 'StaggerInstances'),
        State: validateString(pluginOptions.State, 'State'),
        StrictHardCap: convertToBoolean(pluginOptions.StrictHardCap, 'StrictHardCap'),
    };
}
exports.convertSpotEventPluginSettings = convertSpotEventPluginSettings;
function validateArray(input, propertyName) {
    if (!input || !Array.isArray(input) || input.length === 0) {
        throw new Error(`${propertyName} should be an array with at least one element.`);
    }
}
exports.validateArray = validateArray;
function validateProperty(isValid, property, propertyName) {
    if (!isValid(property)) {
        throw new Error(`${propertyName} type is not valid.`);
    }
}
exports.validateProperty = validateProperty;
function isValidTagSpecification(tagSpecification) {
    if (!tagSpecification || typeof (tagSpecification) !== 'object' || Array.isArray(tagSpecification)) {
        return false;
    }
    // We also verify resourceType with validateString later
    if (!tagSpecification.ResourceType || typeof (tagSpecification.ResourceType) !== 'string') {
        return false;
    }
    if (!tagSpecification.Tags || !Array.isArray(tagSpecification.Tags)) {
        return false;
    }
    for (let element of tagSpecification.Tags) {
        if (!element || typeof (element) !== 'object') {
            return false;
        }
        ;
        if (!element.Key || typeof (element.Key) !== 'string') {
            return false;
        }
        if (!element.Value || typeof (element.Value) !== 'string') {
            return false;
        }
    }
    return true;
}
exports.isValidTagSpecification = isValidTagSpecification;
function convertTagSpecifications(tagSpecifications, propertyName) {
    validateArray(tagSpecifications, propertyName);
    const convertedTagSpecifications = tagSpecifications.map(tagSpecification => {
        validateProperty(isValidTagSpecification, tagSpecification, propertyName);
        const convertedTagSpecification = {
            ResourceType: validateString(tagSpecification.ResourceType, `${propertyName}.ResourceType`),
            Tags: tagSpecification.Tags,
        };
        return convertedTagSpecification;
    });
    return convertedTagSpecifications;
}
exports.convertTagSpecifications = convertTagSpecifications;
function validateLaunchTemplateSpecification(launchTemplateSpecification, propertyName) {
    const id = validateStringOptional(launchTemplateSpecification.LaunchTemplateId, `${propertyName}.LaunchTemplateId`);
    const name = validateStringOptional(launchTemplateSpecification.LaunchTemplateName, `${propertyName}.LaunchTemplateName`);
    if ((id === undefined && name === undefined) || (id !== undefined && name !== undefined)) {
        throw new Error(`Exactly one of ${propertyName}.LaunchTemplateId or ${propertyName}.LaunchTemplateName must be specified, but got: ${id} and ${name} respectively`);
    }
    validateString(launchTemplateSpecification.Version, `${propertyName}.Version`);
}
exports.validateLaunchTemplateSpecification = validateLaunchTemplateSpecification;
function validateLaunchTemplateOverrides(launchTemplateOverrides, propertyName) {
    validateStringOptional(launchTemplateOverrides.AvailabilityZone, `${propertyName}.AvailabilityZone`);
    validateStringOptional(launchTemplateOverrides.InstanceType, `${propertyName}.InstanceType`);
    validateStringOptional(launchTemplateOverrides.SpotPrice, `${propertyName}.SpotPrice`);
    validateStringOptional(launchTemplateOverrides.SubnetId, `${propertyName}.SubnetId`);
    validateProperty(num => num === undefined || typeof num === 'number', launchTemplateOverrides.WeightedCapacity, `${propertyName}.WeightedCapacity`);
}
exports.validateLaunchTemplateOverrides = validateLaunchTemplateOverrides;
function validateLaunchTemplateConfigs(launchTemplateConfigs, propertyName) {
    validateArray(launchTemplateConfigs, propertyName);
    launchTemplateConfigs.forEach((ltc, i) => {
        const ltcPropertyName = `${propertyName}[${i}]`;
        validateProperty(input => typeof input === 'object' && !Array.isArray(input), ltc.LaunchTemplateSpecification, `${ltcPropertyName}.LaunchTemplateSpecification`);
        validateLaunchTemplateSpecification(ltc.LaunchTemplateSpecification, `${ltcPropertyName}.LaunchTemplateSpecification`);
        validateProperty(input => Array.isArray(input), ltc.Overrides, `${ltcPropertyName}.Overrides`);
        ltc.Overrides.forEach((override, idx) => validateLaunchTemplateOverrides(override, `${ltcPropertyName}.Overrides[${idx}]`));
    });
    return launchTemplateConfigs;
}
exports.validateLaunchTemplateConfigs = validateLaunchTemplateConfigs;
function convertToInt(value, propertyName) {
    if (typeof (value) === 'number') {
        if (Number.isInteger(value)) {
            return value;
        }
    }
    if (typeof (value) === 'string') {
        const result = Number.parseFloat(value);
        if (Number.isInteger(result)) {
            return result;
        }
    }
    throw new Error(`The value of ${propertyName} should be an integer. Received: ${value} of type ${typeof (value)}`);
}
exports.convertToInt = convertToInt;
function convertToBoolean(value, propertyName) {
    if (typeof (value) === 'boolean') {
        return value;
    }
    if (typeof (value) === 'string') {
        if (value === 'true') {
            return true;
        }
        if (value === 'false') {
            return false;
        }
    }
    throw new Error(`The value of ${propertyName} should be a boolean. Received: ${value} of type ${typeof (value)}`);
}
exports.convertToBoolean = convertToBoolean;
function validateString(value, propertyName) {
    if (typeof (value) === 'string') {
        return value;
    }
    throw new Error(`The value of ${propertyName} should be a string. Received: ${value} of type ${typeof (value)}`);
}
exports.validateString = validateString;
function validateStringOptional(value, propertyName) {
    if (value === undefined) {
        return undefined;
    }
    return validateString(value, propertyName);
}
exports.validateStringOptional = validateStringOptional;
//# sourceMappingURL=data:application/json;base64,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