"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let vpc;
let workerFleet;
const DEFAULT_CONSTRUCT_ID = 'UBL';
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        jest.spyOn(renderQueue, 'configureSecretsManagementAutoRegistration');
        stack = new core_1.Stack(app, 'Stack', { env });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(stack, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        licenses = [lib_1.UsageBasedLicense.forMaya()];
    });
    function createUbl(props) {
        return new lib_1.UsageBasedLicensing(stack, DEFAULT_CONSTRUCT_ID, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            ...props,
        });
    }
    test('vpcSubnets specified => does not emit warnings', () => {
        // GIVEN
        const vpcSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        };
        // WHEN
        const ubl = createUbl({
            vpcSubnets,
        });
        // THEN
        expect(ubl.node.metadataEntry).not.toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: expect.stringMatching(/dedicated subnet/i),
        }));
    });
    test('vpcSubnets not specified => emits warning about dedicated subnets', () => {
        // WHEN
        const ubl = createUbl();
        // THEN
        expect(ubl.node.metadataEntry).toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
        }));
    });
    describe('configures auto registration', () => {
        test('default to private subnets', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
        test.each([
            [{
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                }],
        ])('%s', (vpcSubnets) => {
            // WHEN
            const ubl = createUbl({
                vpcSubnets,
            });
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets,
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
    });
    test('creates an ECS cluster', () => {
        // WHEN
        createUbl();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            createUbl({
                desiredCount: 2,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const logGroup = ubl.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
        test('uses the supplied security group', () => {
            // GIVEN
            const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'UblSecurityGroup', {
                vpc,
            });
            // WHEN
            createUbl({ securityGroup });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // WHEN
                createUbl();
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                // WHEN
                createUbl({ desiredCount });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: certificateSecret.secretArn,
                    }),
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        // WHEN
        createUbl({
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = DEFAULT_CONSTRUCT_ID;
        // WHEN
        createUbl({
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // WHEN
            createUbl({
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const ubl = createUbl();
            const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
            workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(workerStack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                }),
            });
            // WHEN
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                assert_1.expect(workerStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(ubl).stackName}:ExportsOutputFnGetAttUBLClusterASGInstanceSecurityGroup*`),
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                createUbl({ licenses: [] });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule from UsageBasedLicensing ASG to RenderQueue ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.connections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 4433,
                ToPort: 4433,
                GroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(ublSecurityGroup.securityGroupId),
            }));
        });
        test('adds ingress rule from RenderQueue ASG to UsageBasedLicensing ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.backendConnections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 17004,
                ToPort: 17004,
                GroupId: stack.resolve(ublSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: stack.resolve(`${renderQueue.endpoint.applicationProtocol.toLowerCase()}://${renderQueue.endpoint.socketAddress}`),
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                createUbl();
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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