"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Stage', () => {
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    /**
     * A sub-class of Stage that makes the constructor public for testing purposes.
     *
     * This is to be able to specify a manifest rather than load it from the file-system via {@link Stage.fromDirectory}.
     */
    class StageWithPulicConstructor extends lib_1.Stage {
        constructor(props) {
            super(props);
        }
    }
    beforeEach(() => {
        jest.resetModules();
    });
    describe('.fromDirectory', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {},
        };
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        let stage;
        beforeAll(() => {
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            stage = require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        });
        afterAll(() => {
            readFileSync.mockRestore();
            jest.resetModules();
        });
        test('has dirPath', () => {
            // THEN
            expect(stage.dirPath).toBe(STAGE_PATH);
        });
        test('loads manifest.json from directory', () => {
            // THEN
            const expectedPath = path.join(STAGE_PATH, 'manifest.json');
            expect(readFileSync).toHaveBeenCalledWith(expectedPath, expect.objectContaining({
                encoding: 'utf8',
            }));
            expect(stage.manifest).toEqual(manifest);
        });
    });
    test('.fromDirectory not supported version failure', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.8.0',
            recipes: {},
        };
        // WHEN
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        readFileSync.mockReturnValue(JSON.stringify(manifest));
        // THEN
        expect(() => {
            require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        }).toThrow('Staged Deadline Version (10.1.8.0) is less than the minimum supported version (10.1.9.2)');
        readFileSync.mockRestore();
        jest.resetModules();
    });
    test('has manifest', () => {
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {
                a: {
                    title: 'a-title',
                    description: 'a-description',
                    buildArgs: {
                        argone: 'a-argone-value',
                        argtwo: 'a-argtwo-value',
                    },
                },
                b: {
                    title: 'b-title',
                    description: 'b-description',
                    buildArgs: {
                        argone: 'b-argone-value',
                        argtwo: 'b-argtwo-value',
                    },
                },
            },
        };
        const stage = new StageWithPulicConstructor({
            manifest,
            path: STAGE_PATH,
        });
        // THEN
        expect(stage.manifest).toEqual(manifest);
    });
    describe('manifest validation', () => {
        test.each([
            [
                'mainfest wrong type',
                {
                    manifest: 1,
                    expectedError: /Expected object but got/,
                },
            ],
            [
                'manifest version not string',
                {
                    manifest: {
                        version: 1,
                        recipes: {},
                    },
                },
            ],
            [
                'missing schema',
                {
                    manifest: {
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "schema" key/,
                },
            ],
            [
                'wrong schema type', {
                    manifest: {
                        schema: 'a',
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Expected a numeric "schema" but got:/,
                },
            ],
            [
                'missing version', {
                    manifest: {
                        schema: 1,
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "version" key/,
                },
            ],
            [
                'incorrect version format', {
                    manifest: {
                        schema: 1,
                        version: '10.1.',
                        recipes: {},
                    },
                    expectedError: /Invalid version format/,
                },
            ],
            [
                'version not supported', {
                    manifest: {
                        schema: 1,
                        version: '10.1.0.0',
                        recipes: {},
                    },
                    expectedError: 'Staged Deadline Version (10.1.0.0) is less than the minimum supported version (10.1.9.2)',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { manifest, expectedError } = testcase;
            // WHEN
            function when() {
                new StageWithPulicConstructor({
                    path: STAGE_PATH,
                    manifest,
                });
            }
            expect(when).toThrow(expectedError);
        });
    });
    describe('valid recipes', () => {
        test.each([
            [
                'build args and no target',
                {
                    title: 'Eugene\'s Favorite Cookies',
                    description: 'C is for cookie. That\'s good enough for me',
                    buildArgs: {
                        flour: '3 cups',
                        butter: '1 cup',
                        sugar: '1/2 cup',
                        egg: '2',
                        bakingpowder: '1 tsp',
                        bakingsoda: '1 tsb',
                        vanilla: '1 tsp',
                        salt: '1 pinch',
                    },
                },
            ],
            [
                'buildArgs and a target',
                {
                    title: 'Josh\'s S\'mores',
                    description: 'a secret family recipe',
                    buildArgs: {
                        grahamcracker: '2',
                        marshmellow: '1',
                        chocolate: '2',
                        campfire: '1',
                    },
                    target: 'simplicity',
                },
            ],
            [
                'target and no buildArgs',
                {
                    title: 'Jericho\s special brownie batch with a strawberry milkshake',
                    description: 'a slight improvement to the original recipe',
                    target: 'target',
                },
            ],
            [
                'no target or buildArgs',
                {
                    title: 'Yash\'s Tequila Lime Shot (TLS)',
                    description: 'A sure-fire way to brighten your day',
                },
            ],
        ])('%s', (_scenario, recipe) => {
            // GIVEN
            const manifest = {
                recipes: {
                    recipeName: recipe,
                },
                schema: 1,
                version: '10.1.9.2',
            };
            // WHEN
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // THEN
            expect(stage.manifest.recipes.recipeName).toEqual(recipe);
        });
    });
    describe('.imageFromRecipe()', () => {
        let app;
        let stack;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
        });
        test('creates DockerImageAssets from existing recipes', () => {
            jest.resetModules();
            const mockDockerImageAssetConstructor = jest.fn();
            jest.mock('@aws-cdk/aws-ecr-assets', () => {
                class DockerImageAsset {
                    constructor(...args) {
                        mockDockerImageAssetConstructor(...args);
                    }
                }
                return {
                    DockerImageAsset,
                };
            });
            // Import the class under test now that the ec2 mock is in place
            const ReloadedStage = jest.requireActual('../lib').Stage; // eslint-disable-line
            class ReloadedStageWithPublicConstructor extends ReloadedStage {
                constructor(props) {
                    super(props);
                }
            }
            // GIVEN
            const recipeName = 'myname';
            const recipe = {
                description: 'some description',
                title: 'some title',
                buildArgs: {
                    a: 'a',
                    b: 'b',
                },
            };
            const stage = new ReloadedStageWithPublicConstructor({
                path: STAGE_PATH,
                manifest: {
                    version: '10.1.9.2',
                    schema: 1,
                    recipes: {
                        [recipeName]: recipe,
                    },
                },
            });
            // WHEN
            stage.imageFromRecipe(stack, 'Image', recipeName);
            // THEN
            expect(mockDockerImageAssetConstructor).toHaveBeenCalledWith(stack, 'Image', {
                ...recipe,
                directory: STAGE_PATH,
            });
            expect(mockDockerImageAssetConstructor.mock.calls[0][2]).not.toMatchObject({
                target: expect.anything(),
            });
        });
        test('throws exception when recipe does not exist', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const invalidRecipeName = 'this-recipe-does-not-exist';
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            function when() {
                stage.imageFromRecipe(stack, 'Image', invalidRecipeName);
            }
            // THEN
            expect(when).toThrowError(`No such recipe: ${invalidRecipeName}`);
        });
    });
    describe('.getVersion()', () => {
        test('returns a version in the success case', () => {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            stage.getVersion(stack, 'Version');
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assert_1.stringLike('*'),
                versionString: '10.1.9',
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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