"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_1 = require("../lib");
let app;
let stack;
let spotFleetStack;
let vpc;
let renderQueue;
let rcsImage;
const groupName = 'Group_Name';
const deadlineGroups = [
    groupName,
];
const workerMachineImage = new aws_ec2_1.GenericLinuxImage({
    'us-east-1': 'ami-any',
});
const instanceTypes = [
    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
];
const maxCapacity = 1;
describe('SpotEventPluginFleet', () => {
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'infraStack', {
            env: {
                region: 'us-east-1',
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        spotFleetStack = new core_1.Stack(app, 'SpotFleetStack', {
            env: {
                region: 'us-east-1',
            },
        });
    });
    describe('created with default values', () => {
        test('creates a security group', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.securityGroups).toBeDefined();
            expect(fleet.securityGroups.length).toBe(1);
            assert_1.expect(spotFleetStack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
        });
        test('allows connection to the render queue', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
                SourceSecurityGroupId: spotFleetStack.resolve(fleet.connections.securityGroups[0].securityGroupId),
            }));
        });
        test('creates a spot fleet instance role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('creates an instance profile', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.instanceProfile).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                Roles: assert_1.arrayWith({
                    Ref: spotFleetStack.getLogicalId(fleet.fleetInstanceRole.node.defaultChild),
                }),
            }));
        });
        test('creates a spot fleet role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('adds group names to user data', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBeDefined();
            expect(renderedUserData).toMatch(groupName.toLocaleLowerCase());
        });
        test('adds RFDK tags', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            expect(fleet.tags).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: rfdkTag.name,
                    Value: rfdkTag.value,
                })),
            }));
        });
        test('uses default LogGroup prefix %s', () => {
            // GIVEN
            const id = 'SpotFleet';
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: '/renderfarm/' + id,
            }));
        });
        test('sets default allocation strategy', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(lib_1.SpotFleetAllocationStrategy.LOWEST_PRICE);
        });
        test('does not set valid until property', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.validUntil).toBeUndefined();
        });
        test('does not set valid block devices', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.blockDevices).toBeUndefined();
        });
        test('does not set ssh key', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.keyName).toBeUndefined();
        });
        test('creates launch template configs for each instance type', () => {
            // WHEN
            const moreInstanceTypes = [
                new aws_ec2_1.InstanceType('t2.small'),
                new aws_ec2_1.InstanceType('c5.large'),
            ];
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                instanceTypes: moreInstanceTypes,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(moreInstanceTypes.length);
            moreInstanceTypes.forEach(instanceType => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.InstanceType === instanceType.toString());
                })).toBeTruthy();
            });
        });
        test('creates launch template configs for each subnet id', () => {
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT });
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: subnets,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(subnets.subnets.length);
            subnets.subnetIds.forEach(subnetId => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.SubnetId === subnetId);
                })).toBeTruthy();
            });
        });
        test('add tag indicating resource tracker is enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: 'instance',
                        Tags: assert_1.arrayWith({
                            Key: 'DeadlineTrackedAWSResource',
                            Value: 'SpotEventPlugin',
                        }),
                    }),
                }),
            }));
        });
        test('adds multiple fleet security groups to launch template', () => {
            // GIVEN
            const securityGroups = [
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup1', { vpc }),
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup2', { vpc }),
            ];
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet2', {
                vpc,
                renderQueue,
                deadlineGroups: ['group2'],
                instanceTypes: [new aws_ec2_1.InstanceType('t2.micro')],
                workerMachineImage,
                maxCapacity: 1,
                securityGroups,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    SecurityGroupIds: securityGroups.map(sg => spotFleetStack.resolve(sg.securityGroupId)),
                }),
            }));
        });
        test('adds fleet tags to launch template', () => {
            // GIVEN
            const tag = {
                key: 'mykey',
                value: 'myvalue',
            };
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // WHEN
            core_1.Tags.of(fleet).add(tag.key, tag.value);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: lib_1.SpotFleetResourceType.INSTANCE.toString(),
                        Tags: assert_1.arrayWith({
                            Key: tag.key,
                            Value: tag.value,
                        }),
                    }),
                }),
            }));
        });
    });
    describe('created with custom values', () => {
        test('uses provided required properties', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const imageConfig = workerMachineImage.getImage(fleet);
            // THEN
            expect(fleet.deadlineGroups).toStrictEqual(deadlineGroups.map(group => group.toLocaleLowerCase()));
            expect(fleet.instanceTypes).toBe(instanceTypes);
            expect(fleet.osType).toBe(imageConfig.osType);
            expect(fleet.maxCapacity).toBe(maxCapacity);
        });
        test('uses provided security group', () => {
            // GIVEN
            const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(1);
            expect(fleet.securityGroups).toContainEqual(sg);
        });
        test('uses multiple provided security groups', () => {
            // GIVEN
            const sg1 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-123456789', {
                allowAllOutbound: false,
            });
            const sg2 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-987654321', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg1,
                    sg2,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(2);
            expect(fleet.securityGroups).toContainEqual(sg1);
            expect(fleet.securityGroups).toContainEqual(sg2);
        });
        test('adds to provided user data', () => {
            // GIVEN
            const originalCommands = 'original commands';
            const originalUserData = workerMachineImage.getImage(spotFleetStack).userData;
            originalUserData.addCommands(originalCommands);
            const renderedOriginalUser = originalUserData.render();
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userData: originalUserData,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBe(originalUserData);
            expect(renderedUserData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(originalCommands)));
            expect(renderedUserData).not.toEqual(renderedOriginalUser);
        });
        test('uses provided spot fleet instance role from the same stack', () => {
            // GIVEN
            const spotFleetInstanceRole = new aws_iam_1.Role(spotFleetStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                fleetInstanceRole: spotFleetInstanceRole,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBe(spotFleetInstanceRole);
            assert_1.expect(spotFleetStack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('throws if provided spot fleet instance role is not from the same stack', () => {
            // GIVEN
            const otherStack = new core_1.Stack(app, 'OtherStack', {
                env: { region: 'us-east-1' },
            });
            const spotFleetInstanceRole = new aws_iam_1.Role(otherStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            function createSpotEventPluginFleet() {
                new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    fleetInstanceRole: spotFleetInstanceRole,
                });
            }
            // THEN
            expect(createSpotEventPluginFleet).toThrowError('Fleet instance role should be created on the same stack as SpotEventPluginFleet to avoid circular dependencies.');
        });
        test('uses provided spot fleet role', () => {
            // GIVEN
            const fleetRole = new aws_iam_1.Role(stack, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                fleetRole: fleetRole,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBe(fleetRole);
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('tags resources deployed by CDK', () => {
            // GIVEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const tagName = 'name';
            const tagValue = 'tagValue';
            // WHEN
            core_1.Tags.of(fleet).add(tagName, tagValue);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: tagName,
                    Value: tagValue,
                })),
            }));
        });
        test('uses provided subnets', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            const expectedSubnetId = stack.resolve(vpc.privateSubnets[0].subnetId);
            // THEN
            expect(stack.resolve(fleet.subnets.subnetIds)).toContainEqual(expectedSubnetId);
        });
        test('.defaultSubnets is false when subnets provided', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            // THEN
            expect(fleet.defaultSubnets).toBeFalsy();
        });
        test('uses provided allocation strategy', () => {
            // GIVEN
            const allocationStartegy = lib_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                allocationStrategy: allocationStartegy,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(allocationStartegy);
        });
        test('adds deadline region to user data', () => {
            // GIVEN
            const deadlineRegion = 'someregion';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlineRegion: deadlineRegion,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(deadlineRegion);
        });
        test('adds deadline pools to user data', () => {
            // GIVEN
            const pool1 = 'Pool1';
            const pool2 = 'pool2';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlinePools: [
                    pool1,
                    pool2,
                ],
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(pool1.toLocaleLowerCase());
            expect(renderedUserData).toMatch(pool2.toLocaleLowerCase());
        });
        test('uses provided ssh key name', () => {
            // GIVEN
            const keyName = 'test-key-name';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                keyName: keyName,
            });
            // THEN
            expect(fleet.keyName).toEqual(keyName);
        });
        test('UserData is added by UserDataProvider', () => {
            // WHEN
            class UserDataProvider extends lib_1.InstanceUserDataProvider {
                preCloudWatchAgent(host) {
                    host.userData.addCommands('echo preCloudWatchAgent');
                }
                preRenderQueueConfiguration(host) {
                    host.userData.addCommands('echo preRenderQueueConfiguration');
                }
                preWorkerConfiguration(host) {
                    host.userData.addCommands('echo preWorkerConfiguration');
                }
                postWorkerLaunch(host) {
                    host.userData.addCommands('echo postWorkerLaunch');
                }
            }
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userDataProvider: new UserDataProvider(spotFleetStack, 'UserDataProvider'),
            });
            const userData = fleet.userData.render();
            // THEN
            expect(userData).toContain('echo preCloudWatchAgent');
            expect(userData).toContain('echo preRenderQueueConfiguration');
            expect(userData).toContain('echo preWorkerConfiguration');
            expect(userData).toContain('echo postWorkerLaunch');
        });
        test.each([
            'test-prefix/',
            '',
        ])('uses custom LogGroup prefix %s', (testPrefix) => {
            // GIVEN
            const id = 'SpotFleet';
            const logGroupProps = {
                logGroupPrefix: testPrefix,
            };
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                logGroupProps: logGroupProps,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: testPrefix + id,
            }));
        });
        test('adds tag indicating resource tracker is not enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                trackInstancesWithResourceTracker: false,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assert_1.objectLike({
                    TagSpecifications: assert_1.arrayWith({
                        ResourceType: 'instance',
                        Tags: assert_1.arrayWith({
                            Key: 'DeadlineResourceTracker',
                            Value: 'SpotEventPlugin',
                        }),
                    }),
                }),
            }));
        });
    });
    describe('allowing remote control', () => {
        test('from CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('to CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('from SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('from other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
    });
    describe('validation with', () => {
        describe('instance types', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyInstanceTypes = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: emptyInstanceTypes,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one instance type is required for a Spot Fleet Request Configuration/);
            });
            test('passes with at least one', () => {
                // GIVEN
                const oneInstanceType = [aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL)];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: oneInstanceType,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('subnets', () => {
            test('error if no subnets provided', () => {
                // GIVEN
                const invalidSubnets = {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
                    availabilityZones: ['dummy zone'],
                };
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    vpcSubnets: invalidSubnets,
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.ERROR);
                expect(fleet.node.metadataEntry[0].data).toMatch(/Did not find any subnets matching/);
            });
        });
        describe('groups', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyDeadlineGroups = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: emptyDeadlineGroups,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one Deadline Group is required for a Spot Fleet Request Configuration/);
            });
            test.each([
                'none',
                'with space',
                'group_*',
            ])('throws with %s', (invalidGroupName) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [invalidGroupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineGroups'/);
            });
            test('passes with valid group name', () => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [groupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('region', () => {
            test.each([
                'none',
                'all',
                'unrecognized',
                'none@123',
                'None',
            ])('throws with %s', (deadlineRegion) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineRegion'/);
            });
            test('passes with reserved name as substring', () => {
                // GIVEN
                const deadlineRegion = 'none123';
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('Block Device Tests', () => {
            test('Warning if no BlockDevices provided', () => {
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
            });
            test('No Warnings if Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: '/dev/xvda',
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('Warnings if non-Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('Warnings for BlockDevices without encryption specified', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('No warnings for Ephemeral blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('No warnings for Suppressed blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                        }],
                });
                //THEN
                expect(fleet.node.metadataEntry).toHaveLength(0);
            });
            test('throws if block devices without iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.IO1;
                // WHEN
                function createSpotEventPluginFleet() {
                    return new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups: [
                            groupName,
                        ],
                        instanceTypes: [
                            aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                        ],
                        workerMachineImage,
                        maxCapacity: 1,
                        blockDevices: [{
                                deviceName,
                                volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                    volumeType,
                                }),
                            }],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/iops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            });
            test('warning if block devices with iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const iops = 100;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups: [
                        groupName,
                    ],
                    instanceTypes: [
                        aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                    ],
                    workerMachineImage,
                    maxCapacity: 1,
                    blockDevices: [{
                            deviceName,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                iops,
                                volumeType,
                            }),
                        }],
                });
                // THEN
                expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadataEntry[0].data).toMatch('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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