"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const asset_constants_1 = require("../../deadline/test/asset-constants");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC', {
        subnetConfiguration: [
            {
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            {
                name: 'Isolated',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
            },
        ],
    });
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = {
                patchVersion: 0,
                repository: {
                    objectKey: 'testInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
                },
                client: {
                    objectKey: 'testClientInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxClientInstallerBucket'),
                },
            };
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    version = new MockVersion([10, 1, 19, 4]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: [
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const isolatedSubnetIds = ['IsolatedSubnet1', 'IsolatedSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        isolatedSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: isolatedSubnetIds,
    }));
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                ],
                Effect: 'Allow',
                Resource: {
                    Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                },
            }),
        },
    }));
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Effect: 'Allow',
                Resource: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            '/*',
                        ],
                    ],
                }),
            }),
        },
    }));
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    }));
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    }));
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadataEntry).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct',
        }),
    ]));
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadataEntry).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBInstance'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }));
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    expect(repo.node.metadataEntry).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: warningMsg,
        }),
    ]));
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new core_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' }));
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::DocDB::DBInstance', instanceCount, {
        AutoMinorVersionUpgrade: true,
    }));
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(period),
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    }));
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(20),
        },
    });
    // THEN
    expect(repo.node.metadataEntry).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Backup retention for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
        backup: {
            retention: core_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::FileSystem'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::MountTarget'));
    expect(repo.node.tryFindChild('PadEfsStorage')).toBeDefined();
    expect(repo.node.findChild('FileSystem').node.tryFindChild('PaddingAccessPoint')).toBeDefined();
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT30M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toEqual(expect.stringContaining(version.versionString));
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    }));
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const asgLogicalIdToken = escapeTokenRegex('${Token[Stack.repositoryInstaller.Installer.ASG.LogicalID.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nTOKEN=\\$\\(curl -X PUT "http:\\/\\/169\\.254\\.169\\.254\\/latest\\/api\\/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> \\/dev\\/null\\)\nINSTANCE="\\$\\(curl -s -H "X-aws-ec2-metadata-token: \\$TOKEN" http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id  2> \\/dev\\/null\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalIdToken} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: 'autoscaling:UpdateAutoScalingGroup',
                Condition: {
                    StringEquals: {
                        'autoscaling:ResourceTag/resourceLogicalId': core_1.Names.uniqueId(repo),
                    },
                },
                Effect: 'Allow',
                Resource: '*',
            }, {
                Action: 'ec2:DescribeTags',
                Effect: 'Allow',
                Resource: '*',
            }),
        },
    }));
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instanceRole = (instance
        .node.findChild('InstanceRole')
        .node.defaultChild);
    const db = (repo
        .node.findChild('DocumentDatabase'));
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository false rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    const regex = new RegExp(escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \\"/mnt/repository/DeadlineRepository\\"'));
    expect(userData).toMatch(regex);
    // Assert the IAM instance profile is given read access to the database credentials secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                ],
                Effect: 'Allow',
                Resource: stack.resolve(db.secret.secretArn),
            }),
        },
        Roles: [
            stack.resolve(instanceRole.ref),
        ],
    }));
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 2, {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Effect: 'Allow',
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Resource: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                            ],
                        ],
                    },
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                                '/*',
                            ],
                        ],
                    },
                ],
            }),
        },
    }));
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 2,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::EFS::MountTarget', numMountTargets, {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        assert_1.SynthUtils.synthesize(newStack);
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    }));
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_2.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
test('disable Secrets Management by default when Deadline version is old', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    const repository = new lib_1.Repository(newStack, 'Repo', {
        vpc,
        version: oldVersion,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeFalsy();
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws when Secrets Management is enabled but deadline version is too low', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version: oldVersion,
            vpc,
            secretsManagementSettings: {
                enabled: true,
            },
        });
    }
    // THEN
    expect(when).toThrow(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
});
test('imports repository settings', () => {
    // GIVEN
    const repositorySettings = new aws_s3_assets_1.Asset(stack, 'RepositorySettingsAsset', {
        path: __filename,
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        repositorySettings,
    });
    // THEN
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`aws s3 cp '${repositorySettings.s3ObjectUrl}'`);
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = true changes ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => true,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).toMatch('-o 1000:1000');
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = false does not change ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => false,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).not.toMatch('-o 1000:1000');
});
test('secret manager enabled', () => {
    // GIVEN
    const expectedCredentials = new aws_secretsmanager_1.Secret(stack, 'CustomSMAdminUser', {
        description: 'Custom admin credentials for the Secret Management',
        generateSecretString: {
            excludeCharacters: '\"$&\'()-/<>[\\]\`{|}',
            includeSpace: false,
            passwordLength: 24,
            requireEachIncludedType: true,
            generateStringKey: 'password',
            secretStringTemplate: JSON.stringify({ username: 'admin' }),
        },
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: true,
            credentials: expectedCredentials,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBe(expectedCredentials);
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`-r ${stack.region} -c ${expectedCredentials.secretArn}`);
});
test('secret manager is enabled by default', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeTruthy();
    expect(repository.secretsManagementSettings.credentials).toBeDefined();
});
test('credentials are undefined when secrets management is disabled', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: false,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws an error if credentials are undefined and database is imported', () => {
    // GIVEN
    const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
        vpc,
    });
    const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
    const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
        clusterEndpointAddress: '1.2.3.4',
        clusterIdentifier: 'foo',
        instanceEndpointAddresses: ['1.2.3.5'],
        instanceIdentifiers: ['i0'],
        port: 27001,
        readerEndpointAddress: '1.2.3.6',
        securityGroup: sg,
    });
    const databaseConnection = lib_1.DatabaseConnection.forDocDB({ database, login: secret });
    // WHEN
    function when() {
        new lib_1.Repository(stack, 'Repository', {
            vpc,
            version,
            database: databaseConnection,
        });
    }
    // THEN
    expect(when).toThrow('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVwb3NpdG9yeS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVwb3NpdG9yeS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsNENBUXlCO0FBRXpCLGtEQUFtRDtBQUNuRCw4Q0FjMEI7QUFDMUIsOENBSTBCO0FBRTFCLDRDQUF5QztBQUN6QywwREFBK0M7QUFDL0Msb0VBQXFEO0FBQ3JELHdDQU91QjtBQUV2QixxQ0FFb0I7QUFDcEIsNkRBRXFDO0FBQ3JDLHlFQUU2QztBQUM3QyxnQ0FPZ0I7QUFDaEIsdURBRTJCO0FBRTNCLElBQUksR0FBUSxDQUFDO0FBQ2IsSUFBSSxLQUFZLENBQUM7QUFDakIsSUFBSSxHQUFTLENBQUM7QUFDZCxJQUFJLE9BQWlCLENBQUM7QUFFdEIsU0FBUyxnQkFBZ0IsQ0FBQyxDQUFTO0lBQ2pDLDZDQUE2QztJQUM3QyxtRUFBbUU7SUFDbkUsbUNBQW1DO0lBQ25DLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLENBQUM7QUFDekMsQ0FBQztBQUVELFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUNoQixLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ2hDLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1FBQzFCLG1CQUFtQixFQUFFO1lBQ25CO2dCQUNFLElBQUksRUFBRSxRQUFRO2dCQUNkLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE1BQU07YUFDOUI7WUFDRDtnQkFDRSxJQUFJLEVBQUUsU0FBUztnQkFDZixVQUFVLEVBQUUsb0JBQVUsQ0FBQyxnQkFBZ0I7YUFDeEM7WUFDRDtnQkFDRSxJQUFJLEVBQUUsVUFBVTtnQkFDaEIsVUFBVSxFQUFFLG9CQUFVLENBQUMsZ0JBQWdCO2FBQ3hDO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFdBQVksU0FBUSxhQUFPO1FBQWpDOztZQUNXLG9CQUFlLEdBQXVCO2dCQUM3QyxZQUFZLEVBQUUsQ0FBQztnQkFDZixVQUFVLEVBQUU7b0JBQ1YsU0FBUyxFQUFFLGVBQWU7b0JBQzFCLFFBQVEsRUFBRSxJQUFJLGVBQU0sQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLENBQUM7aUJBQ3BEO2dCQUNELE1BQU0sRUFBRTtvQkFDTixTQUFTLEVBQUUscUJBQXFCO29CQUNoQyxRQUFRLEVBQUUsSUFBSSxlQUFNLENBQUMsS0FBSyxFQUFFLDRCQUE0QixDQUFDO2lCQUMxRDthQUNGLENBQUM7UUFLSixDQUFDO1FBSFEsc0JBQXNCO1lBQzNCLE9BQU8sSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3pCLENBQUM7S0FDRjtJQUVELE9BQU8sR0FBRyxJQUFJLFdBQVcsQ0FBQyxDQUFDLEVBQUUsRUFBQyxDQUFDLEVBQUMsRUFBRSxFQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDekMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO0lBQ3ZDLFFBQVE7SUFDUixJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtRQUM3QixHQUFHO1FBQ0gsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1FBQ1YsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7WUFDN0IsR0FBRztZQUNILE9BQU87U0FDUixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLENBQUM7QUFDbkIsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0RBQW9ELEVBQUUsR0FBRyxFQUFFO0lBQzlELE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7UUFDekUsVUFBVSxFQUFFO1lBQ1YsT0FBTyxFQUFFLEdBQUc7WUFDWixPQUFPLEVBQUUsR0FBRztZQUNaLGlCQUFpQixFQUFFO2dCQUNqQjtvQkFDRSxHQUFHLEVBQUUsaUNBQWlDO2lCQUN2QztnQkFDRDtvQkFDRSxHQUFHLEVBQUUsaUNBQWlDO2lCQUN2QzthQUNGO1NBQ0Y7UUFDRCxjQUFjLEVBQUU7WUFDZCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsT0FBTyxFQUFFLE9BQU87YUFDakI7U0FDRjtRQUNELFlBQVksRUFBRTtZQUNaLDBCQUEwQixFQUFFO2dCQUMxQixXQUFXLEVBQUUsSUFBSTthQUNsQjtZQUNELDBCQUEwQixFQUFFO2dCQUMxQixtQ0FBbUMsRUFBRSxJQUFJO2FBQzFDO1NBQ0Y7UUFDRCxTQUFTLEVBQUU7WUFDVCxzREFBc0Q7U0FDdkQ7S0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0lBQ3JDLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUNBQXVDLEVBQUU7UUFDNUUsWUFBWSxFQUFFLFVBQVU7S0FDekIsQ0FBQyxDQUFDLENBQUM7SUFFSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO1FBQ3JFLFVBQVUsRUFBRSxLQUFLO1FBQ2pCLFFBQVEsRUFBRSxJQUFJO1FBQ2QsTUFBTSxFQUFFLElBQUk7UUFDWixPQUFPLEVBQUU7WUFDUCxZQUFZLEVBQUU7Z0JBQ1osdURBQXVEO2dCQUN2RCxTQUFTO2FBQ1Y7U0FDRjtRQUNELHFCQUFxQixFQUFFO1lBQ3JCLFlBQVksRUFBRTtnQkFDWixrREFBa0Q7Z0JBQ2xELFNBQVM7YUFDVjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx1Q0FBdUMsRUFBRSxHQUFHLEVBQUU7SUFDakQsb0ZBQW9GO0lBQ3BGLG1CQUFtQjtJQUVuQixPQUFPO0lBQ1AsTUFBTSxpQkFBaUIsR0FBRyxDQUFFLGlCQUFpQixFQUFFLGlCQUFpQixDQUFFLENBQUM7SUFDbkUsTUFBTSxPQUFPLEdBQUcsYUFBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7UUFDdEQsaUJBQWlCLEVBQUUsQ0FBQyxZQUFZLEVBQUUsWUFBWSxDQUFDO1FBQy9DLEtBQUssRUFBRSxPQUFPO1FBQ2QsaUJBQWlCO0tBQ2xCLENBQUMsQ0FBQztJQUNILElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRyxFQUFFLE9BQU87UUFDWixPQUFPO1FBQ1AsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLG9CQUFVLENBQUMsZ0JBQWdCLEVBQUU7S0FDeEQsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7UUFDekUsaUJBQWlCLEVBQUUsaUJBQWlCO0tBQ3JDLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7UUFDckUsVUFBVSxFQUFFLEtBQUs7UUFDakIsUUFBUSxFQUFFLElBQUk7UUFDZCxNQUFNLEVBQUUsSUFBSTtRQUNaLE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRTtnQkFDWix1REFBdUQ7Z0JBQ3ZELFNBQVM7YUFDVjtTQUNGO1FBQ0QscUJBQXFCLEVBQUU7WUFDckIsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsU0FBUzthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztJQUNKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7UUFDckUsVUFBVSxFQUFFLEtBQUs7UUFDakIsUUFBUSxFQUFFO1lBQ1IsWUFBWSxFQUFFO2dCQUNaLDZDQUE2QztnQkFDN0MsTUFBTTthQUNQO1NBQ0Y7UUFDRCxNQUFNLEVBQUU7WUFDTixZQUFZLEVBQUU7Z0JBQ1osNkNBQTZDO2dCQUM3QyxNQUFNO2FBQ1A7U0FDRjtRQUNELE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRTtnQkFDWiwwREFBMEQ7Z0JBQzFELFNBQVM7YUFDVjtTQUNGO1FBQ0QscUJBQXFCLEVBQUU7WUFDckIsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsU0FBUzthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUg7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E0Qkc7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQUM7Z0JBQ25CLE1BQU0sRUFBRTtvQkFDTiwrQkFBK0I7b0JBQy9CLCtCQUErQjtpQkFDaEM7Z0JBQ0QsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFO29CQUNSLEdBQUcsRUFBRSw2REFBNkQ7aUJBQ25FO2FBQ0YsQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxREFBcUQsRUFBRSxHQUFHLEVBQUU7SUFDL0QsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUN2RCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUUsa0JBQVMsQ0FDbEI7Z0JBQ0UsTUFBTSxFQUFFO29CQUNOLGVBQWU7b0JBQ2YsZUFBZTtvQkFDZixVQUFVO2lCQUNYO2dCQUNELE1BQU0sRUFBRSxPQUFPO2dCQUNmLFFBQVEsRUFBRSxrQkFBUyxDQUNqQjtvQkFDRSxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxNQUFNOzRCQUNOO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELFFBQVE7NEJBQ1I7Z0NBQ0UsR0FBRyxFQUFFLGlDQUFlLENBQUMsTUFBTTs2QkFDNUI7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsRUFDRDtvQkFDRSxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxNQUFNOzRCQUNOO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELFFBQVE7NEJBQ1I7Z0NBQ0UsR0FBRyxFQUFFLGlDQUFlLENBQUMsTUFBTTs2QkFDNUI7NEJBQ0QsSUFBSTt5QkFDTDtxQkFDRjtpQkFDRixDQUNGO2FBQ0YsQ0FDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7SUFDcEUsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1FBQ3ZELGVBQWUsRUFBRSxDQUFDO1FBQ2xCLFlBQVksRUFBRSxpQ0FBaUM7S0FDaEQsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7SUFDeEQsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsMkdBQTJHO0lBQzNHLHNHQUFzRztJQUN0RyxpQ0FBaUM7SUFDakMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxxQkFBcUIsRUFBRTtRQUMxRCxJQUFJLEVBQUUsUUFBUTtRQUNkLEtBQUssRUFBRTtZQUNMLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLG9GQUFvRjtvQkFDcEYsRUFBRTtvQkFDRiw0SkFBNEo7b0JBQzVKLEVBQUU7b0JBQ0YsME9BQTBPO2lCQUMzTzthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEdBQUcsRUFBRTtJQUNuRCxRQUFRO0lBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLFFBQVEsR0FBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQWlDLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBRWhGLE9BQU87SUFDUCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLGdCQUFnQixDQUFDLHdEQUF3RCxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ25ILENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztJQUNSLENBQUMsb0JBQWEsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDO0lBQ2pDLENBQUMsb0JBQWEsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDO0lBQ2hDLENBQUMsb0JBQWEsQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDO0NBQ3JDLENBQUMsQ0FBQywrQ0FBK0MsRUFBRSxDQUFDLE1BQXFCLEVBQUUsUUFBZ0IsRUFBRSxFQUFFO0lBQzlGLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsUUFBUSxFQUFFLE1BQU07U0FDakI7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtRQUM1RCxjQUFjLEVBQUUsUUFBUTtLQUN6QixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztJQUNSLENBQUMsb0JBQWEsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDO0lBQ2pDLENBQUMsb0JBQWEsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDO0lBQ2hDLENBQUMsb0JBQWEsQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDO0NBQ3JDLENBQUMsQ0FBQyxpREFBaUQsRUFBRSxDQUFDLE1BQXFCLEVBQUUsUUFBZ0IsRUFBRSxFQUFFO0lBQ2hHLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsVUFBVSxFQUFFLE1BQU07U0FDbkI7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRTtRQUMzRCxjQUFjLEVBQUUsUUFBUTtLQUN6QixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRFQUE0RSxFQUFFLEdBQUcsRUFBRTtJQUN0RixRQUFRO0lBQ1IsTUFBTSxPQUFPLEdBQUcsSUFBSSxvQkFBYSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtRQUM1RCxHQUFHO0tBQ0osQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUN2RCxVQUFVLEVBQUUsT0FBTztLQUNwQixDQUFDLENBQUM7SUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLG1CQUFZLENBQUMsS0FBSyxFQUFFO1FBQ3JDLFVBQVUsRUFBRSxPQUFPO1FBQ25CLFdBQVcsRUFBRSxNQUFNO0tBQ3BCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxVQUFVLEVBQUUsTUFBTTtRQUNsQixPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsVUFBVSxFQUFFLG9CQUFhLENBQUMsT0FBTztTQUNsQztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxPQUFPLENBQ3JDLE1BQU0sQ0FBQyxlQUFlLENBQUM7UUFDckIsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3RCLElBQUksRUFBRSxpQkFBaUI7WUFDdkIsSUFBSSxFQUFFLDRHQUE0RztTQUNuSCxDQUFDO0tBQ0gsQ0FBQyxDQUNILENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3RUFBd0UsRUFBRSxHQUFHLEVBQUU7SUFDbEYsUUFBUTtJQUNSLE1BQU0sVUFBVSxHQUFHLElBQUksMkJBQWUsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQzdELFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxRQUFRO1NBQ25CO1FBQ0QsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO1FBQ0QsR0FBRztRQUNILFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLG9CQUFVLENBQUMsZ0JBQWdCO1NBQ3hDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILFFBQVEsRUFBRSx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxVQUFVLENBQUMsTUFBTyxFQUFFLENBQUM7UUFDMUYsT0FBTztRQUNQLGFBQWEsRUFBRTtZQUNiLFFBQVEsRUFBRSxvQkFBYSxDQUFDLE9BQU87U0FDaEM7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsT0FBTyxDQUNyQyxNQUFNLENBQUMsZUFBZSxDQUFDO1FBQ3JCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUN0QixJQUFJLEVBQUUsaUJBQWlCO1lBQ3ZCLElBQUksRUFBRSx3R0FBd0c7U0FDL0csQ0FBQztLQUNILENBQUMsQ0FDSCxDQUFDO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0RBQXNELEVBQUUsR0FBRyxFQUFFO0lBQ2hFLE1BQU0sT0FBTyxHQUFHLElBQUksb0JBQWEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDNUQsR0FBRztLQUNKLENBQUMsQ0FBQztJQUNILE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQVcsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFDdkQsVUFBVSxFQUFFLE9BQU87S0FDcEIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsSUFBSSxtQkFBWSxDQUFDLEtBQUssRUFBRTtRQUNyQyxVQUFVLEVBQUUsT0FBTztRQUNuQixXQUFXLEVBQUUsTUFBTTtLQUNwQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUMzQyxHQUFHO1FBQ0gsVUFBVSxFQUFFLE1BQU07UUFDbEIsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxDQUFDO0lBQzNELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLENBQUM7SUFDNUQsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtRQUM1RCwyQkFBMkIsRUFBRSxDQUFFLE9BQU8sQ0FBRTtLQUN6QyxFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztJQUM3QixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHFDQUFxQyxFQUFFO1FBQzFFLFVBQVUsRUFBRTtZQUNWLFVBQVUsRUFBRSxTQUFTO1NBQ3RCO0tBQ0YsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7SUFDN0IsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx3QkFBd0IsRUFBRTtRQUM3RCx1QkFBdUIsRUFBRSxJQUFJO0tBQzlCLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0RBQStELEVBQUUsR0FBRyxFQUFFO0lBQ3pFLE1BQU0sT0FBTyxHQUFHLElBQUksb0JBQWEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDNUQsR0FBRztLQUNKLENBQUMsQ0FBQztJQUNILE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQVcsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFDdkQsVUFBVSxFQUFFLE9BQU87S0FDcEIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsSUFBSSxtQkFBWSxDQUFDLEtBQUssRUFBRTtRQUNyQyxVQUFVLEVBQUUsT0FBTztRQUNuQixXQUFXLEVBQUUsTUFBTTtLQUNwQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUMzQyxHQUFHO1FBQ0gsVUFBVSxFQUFFLE1BQU07UUFDbEIsT0FBTztRQUNQLG9CQUFvQixFQUFFLEtBQUs7S0FDNUIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUM7SUFDM0QsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtRQUMvRCwyQkFBMkIsRUFBRSxDQUFFLE9BQU8sQ0FBRTtLQUN6QyxFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztJQUM3QixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLHlCQUFnQixDQUFDLHFDQUFxQyxFQUFFO1FBQzdFLFVBQVUsRUFBRTtZQUNWLFVBQVUsRUFBRSxTQUFTO1NBQ3RCO0tBQ0YsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7QUFDL0IsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNEVBQTRFLEVBQUUsR0FBRyxFQUFFO0lBQ3RGLFFBQVE7SUFDUixNQUFNLFVBQVUsR0FBRyxJQUFJLDJCQUFlLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUM3RCxVQUFVLEVBQUU7WUFDVixRQUFRLEVBQUUsUUFBUTtTQUNuQjtRQUNELFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FDM0IsdUJBQWEsQ0FBQyxFQUFFLEVBQ2hCLHNCQUFZLENBQUMsS0FBSyxDQUNuQjtRQUNELEdBQUc7UUFDSCxVQUFVLEVBQUU7WUFDVixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxvQkFBVSxDQUFDLGdCQUFnQjtTQUN4QztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsVUFBVSxFQUFFLG9CQUFhLENBQUMsT0FBTztTQUNsQztRQUNELFFBQVEsRUFBRSx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxVQUFVLENBQUMsTUFBTyxFQUFFLENBQUM7UUFDMUYsb0JBQW9CLEVBQUUsSUFBSTtLQUMzQixDQUFDLENBQUM7SUFFSCxNQUFNLFVBQVUsR0FBRywyR0FBMkc7UUFDNUgsbUVBQW1FLENBQUM7SUFFdEUsT0FBTztJQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE9BQU8sQ0FDckMsTUFBTSxDQUFDLGVBQWUsQ0FBQztRQUNyQixNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDdEIsSUFBSSxFQUFFLGlCQUFpQjtZQUN2QixJQUFJLEVBQUUsVUFBVTtTQUNqQixDQUFDO0tBQ0gsQ0FBQyxDQUNILENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsUUFBUTtJQUNSLE1BQU0sZUFBZSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztJQUNuRCxNQUFNLGFBQWEsR0FBRyxJQUFJLGFBQUcsQ0FBQyxlQUFlLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFFekQsTUFBTSxPQUFPLEdBQUc7UUFDZCxnQkFBTSxDQUFDLG9CQUFvQixDQUFDLGVBQWUsRUFBRSxTQUFTLEVBQUU7WUFDdEQsUUFBUSxFQUFFLFdBQVc7WUFDckIsZ0JBQWdCLEVBQUUsWUFBWTtTQUMvQixDQUFDO1FBQ0YsZ0JBQU0sQ0FBQyxvQkFBb0IsQ0FBQyxlQUFlLEVBQUUsU0FBUyxFQUFFO1lBQ3RELFFBQVEsRUFBRSxXQUFXO1lBQ3JCLGdCQUFnQixFQUFFLFlBQVk7U0FDL0IsQ0FBQztLQUNILENBQUM7SUFDRixNQUFNLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFFdEQsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxhQUFhLEVBQUUscUJBQXFCLEVBQUU7UUFDbkQsR0FBRyxFQUFFLGFBQWE7UUFDbEIsT0FBTztRQUNQLFVBQVUsRUFBRTtZQUNWLE9BQU87U0FDUjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDJCQUEyQixFQUFFO1FBQ3hFLFNBQVMsRUFBRTtZQUNULFdBQVc7WUFDWCxXQUFXO1NBQ1o7S0FDRixDQUFDLENBQUMsQ0FBQztJQUNKLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUUsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ2xHLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUUsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3BHLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJDQUEyQyxFQUFFLEdBQUcsRUFBRTtJQUNyRCxRQUFRO0lBQ1IsTUFBTSxhQUFhLEdBQUcsQ0FBQyxDQUFDO0lBRXhCLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsdUJBQXVCLEVBQUUsYUFBYTtLQUN2QyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQywyQkFBa0IsQ0FBQyx3QkFBd0IsRUFBRSxhQUFhLEVBQUU7UUFDOUUsdUJBQXVCLEVBQUUsSUFBSTtLQUM5QixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtJQUN2RCxRQUFRO0lBQ1IsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDO0lBRWxCLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsaUJBQWlCLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7U0FDekM7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtRQUM1RCxxQkFBcUIsRUFBRSxNQUFNO0tBQzlCLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0RBQXNELEVBQUUsR0FBRyxFQUFFO0lBQ2hFLFFBQVE7SUFDUixNQUFNLFVBQVUsR0FBRyxJQUFJLDJCQUFlLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUM3RCxVQUFVLEVBQUU7WUFDVixRQUFRLEVBQUUsUUFBUTtTQUNuQjtRQUNELFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FDM0IsdUJBQWEsQ0FBQyxFQUFFLEVBQ2hCLHNCQUFZLENBQUMsS0FBSyxDQUNuQjtRQUNELEdBQUc7UUFDSCxVQUFVLEVBQUU7WUFDVixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxvQkFBVSxDQUFDLGdCQUFnQjtTQUN4QztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxRQUFRLEVBQUUsd0JBQWtCLENBQUMsUUFBUSxDQUFDLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDLE1BQU8sRUFBRSxDQUFDO1FBQzFGLE9BQU87UUFDUCxhQUFhLEVBQUU7WUFDYixpQkFBaUIsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztTQUNyQztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxPQUFPLENBQ3JDLE1BQU0sQ0FBQyxlQUFlLENBQUM7UUFDckIsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3RCLElBQUksRUFBRSxpQkFBaUI7WUFDdkIsSUFBSSxFQUFFLDJHQUEyRztTQUNsSCxDQUFDO0tBQ0gsQ0FBQyxDQUNILENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFFMUQsTUFBTSxVQUFVLEdBQUcsSUFBSSwyQkFBZSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDN0QsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLFFBQVE7U0FDbkI7UUFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQzNCLHVCQUFhLENBQUMsRUFBRSxFQUNoQixzQkFBWSxDQUFDLEtBQUssQ0FDbkI7UUFDRCxHQUFHO1FBQ0gsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxnQkFBZ0I7U0FDeEM7UUFDRCxNQUFNLEVBQUU7WUFDTixTQUFTLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7U0FDN0I7S0FDRixDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRTtRQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7S0FDMUM7SUFFRCxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsUUFBUSxFQUFFLHdCQUFrQixDQUFDLFFBQVEsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUN6RixPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUM7SUFDMUQsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQztJQUMzRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM5RCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7QUFDbEcsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0ZBQW9GLEVBQUUsR0FBRyxFQUFFO0lBQzlGLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILE9BQU87UUFDUCw0QkFBNEIsRUFBRSxLQUFLO0tBQ3BDLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLE1BQU0sR0FBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQWlDLENBQUMsUUFBUSxDQUFDO0lBQ3JFLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsQ0FBQztBQUMxRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrR0FBK0csRUFBRSxHQUFHLEVBQUU7SUFDekgsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsT0FBTztRQUNQLDRCQUE0QixFQUFFLFFBQVE7S0FDdkMsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBaUMsQ0FBQyxRQUFRLENBQUM7SUFDckUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0FBQzFELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBEQUEwRCxFQUFFLEdBQUcsRUFBRTtJQUNwRSxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCw2QkFBNkIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuRCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxvQ0FBb0MsRUFBRTtRQUNyRSxjQUFjLEVBQUU7WUFDZCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsT0FBTyxFQUFFLE9BQU87YUFDakI7U0FDRjtLQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7QUFDdkMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0VBQW9FLEVBQUUsR0FBRyxFQUFFO0lBQzlFLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILDZCQUE2QixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBQ25ELE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxNQUFNLEdBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFpQyxDQUFDLFFBQVEsQ0FBQztJQUNyRSxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztBQUNsRixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxJQUFJLENBQUM7SUFDUixjQUFjO0lBQ2QsRUFBRTtDQUNILENBQUMsQ0FBQyxnRUFBZ0UsRUFBRSxDQUFDLFVBQWtCLEVBQUUsRUFBRTtJQUMxRixRQUFRO0lBQ1IsTUFBTSxFQUFFLEdBQUcscUJBQXFCLENBQUM7SUFFakMsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1FBQ3hCLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsY0FBYyxFQUFFLFVBQVU7U0FDM0I7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1FBQ3ZELFlBQVksRUFBRSxVQUFVLEdBQUcsRUFBRTtLQUM5QixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsNkJBQTZCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbkQsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLFdBQVcsR0FBRyxnQkFBZ0IsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQ2xFLE1BQU0saUJBQWlCLEdBQUcsZ0JBQWdCLENBQUMsa0VBQWtFLENBQUMsQ0FBQztJQUMvRyxNQUFNLGNBQWMsR0FBRyxrWUFBa1ksV0FBVyxzTEFBc0wsV0FBVyx5S0FBeUssS0FBSyxDQUFDLFNBQVMsZUFBZSxpQkFBaUIsYUFBYSxXQUFXLHlFQUF5RSxDQUFDO0lBQy81QixNQUFNLENBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFpQyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO0lBQzNHLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQ2xCO2dCQUNFLE1BQU0sRUFBRSxvQ0FBb0M7Z0JBQzVDLFNBQVMsRUFBRTtvQkFDVCxZQUFZLEVBQUU7d0JBQ1osMkNBQTJDLEVBQUUsWUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7cUJBQ2xFO2lCQUNGO2dCQUNELE1BQU0sRUFBRSxPQUFPO2dCQUNmLFFBQVEsRUFBRSxHQUFHO2FBQ2QsRUFDRDtnQkFDRSxNQUFNLEVBQUUsa0JBQWtCO2dCQUMxQixNQUFNLEVBQUUsT0FBTztnQkFDZixRQUFRLEVBQUUsR0FBRzthQUNkLENBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO0lBQ2hELFFBQVE7SUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCw2QkFBNkIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuRCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxRQUFRLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDL0MsR0FBRztRQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1FBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO0tBQ25HLENBQUMsQ0FBQztJQUNILE1BQU0sWUFBWSxHQUFHLENBQ25CLFFBQVE7U0FDTCxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQztTQUM5QixJQUFJLENBQUMsWUFBWSxDQUNWLENBQUM7SUFDYixNQUFNLEVBQUUsR0FBRyxDQUNULElBQUk7U0FDRCxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQ25CLENBQUM7SUFFckIsT0FBTztJQUNQLElBQUksQ0FBQyx1QkFBdUIsQ0FBQztRQUMzQixJQUFJLEVBQUUsUUFBUTtRQUNkLFVBQVUsRUFBRSxpQkFBaUI7S0FDOUIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztJQUU1QyxPQUFPO0lBQ1AseUZBQXlGO0lBQ3pGLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsMkRBQTJELENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFcEgsNENBQTRDO0lBQzVDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsMkNBQTJDLENBQUMsQ0FBQztJQUV0RSwyRkFBMkY7SUFDM0YsTUFBTSxLQUFLLEdBQUcsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsNEZBQTRGLENBQUMsQ0FBQyxDQUFDO0lBQ3pJLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFaEMsMEZBQTBGO0lBQzFGLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQUM7Z0JBQ25CLE1BQU0sRUFBRTtvQkFDTiwrQkFBK0I7b0JBQy9CLCtCQUErQjtpQkFDaEM7Z0JBQ0QsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLE1BQU8sQ0FBQyxTQUFTLENBQUM7YUFDOUMsQ0FBQztTQUNIO1FBQ0QsS0FBSyxFQUFFO1lBQ0wsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDO1NBQ2hDO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrREFBK0QsRUFBRSxHQUFHLEVBQUU7SUFDekUsZ0dBQWdHO0lBRWhHLFFBQVE7SUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCw2QkFBNkIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuRCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxTQUFTLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7UUFDakQsR0FBRztRQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1FBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO0tBQ25HLENBQUMsQ0FBQztJQUNILE1BQU0sU0FBUyxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1FBQ2pELEdBQUc7UUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztRQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztLQUNuRyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsSUFBSSxDQUFDLHVCQUF1QixDQUFDO1FBQzNCLElBQUksRUFBRSxTQUFTO1FBQ2YsVUFBVSxFQUFFLGlCQUFpQjtLQUM5QixDQUFDLENBQUM7SUFDSCxJQUFJLENBQUMsdUJBQXVCLENBQUM7UUFDM0IsSUFBSSxFQUFFLFNBQVM7UUFDZixVQUFVLEVBQUUsaUJBQWlCO0tBQzlCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCw4R0FBOEc7SUFDOUcsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQywyQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7UUFDNUQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQ2xCO2dCQUNFLE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRTtvQkFDTixlQUFlO29CQUNmLGVBQWU7b0JBQ2YsVUFBVTtpQkFDWDtnQkFDRCxRQUFRLEVBQUU7b0JBQ1I7d0JBQ0UsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsTUFBTTtnQ0FDTjtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCxRQUFRO2dDQUNSO29DQUNFLEdBQUcsRUFBRSwrQkFBYSxDQUFDLE1BQU07aUNBQzFCOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNEO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ047b0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtpQ0FDdEI7Z0NBQ0QsUUFBUTtnQ0FDUjtvQ0FDRSxHQUFHLEVBQUUsK0JBQWEsQ0FBQyxNQUFNO2lDQUMxQjtnQ0FDRCxJQUFJOzZCQUNMO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7SUFDOUQsUUFBUTtJQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILDZCQUE2QixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBQ25ELE9BQU87S0FDUixDQUFDLENBQUM7SUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtRQUMvQyxHQUFHO1FBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7UUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsYUFBYSxDQUFDLHdCQUFjLENBQUMscUNBQXFDLENBQUM7S0FDL0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLENBQUMsdUJBQXVCLENBQUM7WUFDM0IsSUFBSSxFQUFFLFFBQVE7WUFDZCxVQUFVLEVBQUUsTUFBTTtTQUNuQixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsNEVBQTRFLENBQUMsQ0FBQztBQUNoRyxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO0lBQ3ZCLCtCQUFpQixDQUFDO1FBQ2hCLGFBQWEsRUFBRSxZQUFZO1FBQzNCLGVBQWUsRUFBRSxHQUFHLEVBQUU7WUFDcEIsUUFBUTtZQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUN0RCxJQUFJLGdCQUFVLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRTtnQkFDMUMsR0FBRztnQkFDSCxPQUFPO2FBQ1IsQ0FBQyxDQUFDO1lBQ0gsT0FBTyxhQUFhLENBQUM7UUFDdkIsQ0FBQztRQUNELGtCQUFrQixFQUFFO1lBQ2xCLHlCQUF5QixFQUFFLENBQUM7WUFDNUIscUNBQXFDLEVBQUUsQ0FBQztZQUN4QywyQkFBMkIsRUFBRSxDQUFDO1lBQzlCLDZCQUE2QixFQUFFLENBQUM7WUFDaEMsdUJBQXVCLEVBQUUsQ0FBQztZQUMxQix3QkFBd0IsRUFBRSxDQUFDO1lBQzNCLGdCQUFnQixFQUFFLENBQUM7WUFDbkIsb0NBQW9DLEVBQUUsQ0FBQztZQUN2QyxxQkFBcUIsRUFBRSxDQUFDO1NBQ3pCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO0lBQy9CLElBQUksdUJBQXVDLENBQUM7SUFFNUMsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLHVCQUF1QixHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQ3pGLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUU7UUFFckIsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtZQUNoRCxPQUFPO1lBQ1AsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7Z0JBQ2xDLE9BQU87Z0JBQ1AsR0FBRztnQkFDSCxxQkFBcUIsRUFBRTtvQkFDckIsUUFBUSxFQUFFLHVCQUF1QjtpQkFDbEM7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtnQkFDNUQsbUJBQW1CLEVBQUUsa0JBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ3ZGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFO1FBRW5CLElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7WUFDaEQsT0FBTztZQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO2dCQUNsQyxPQUFPO2dCQUNQLEdBQUc7Z0JBQ0gscUJBQXFCLEVBQUU7b0JBQ3JCLFVBQVUsRUFBRSx1QkFBdUI7aUJBQ3BDO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGdIQUFnSDtZQUNoSCxNQUFNLGVBQWUsR0FBRyxHQUFHLENBQUMsYUFBYSxFQUFFLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztZQUMzRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLHVCQUF1QixFQUFFLGVBQWUsRUFBRTtnQkFDL0UsY0FBYyxFQUFFLGtCQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxlQUFlLENBQUMsQ0FBQzthQUNsRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRTtRQUV6QixJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1lBQ2hELE9BQU87WUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtnQkFDbEMsT0FBTztnQkFDUCxHQUFHO2dCQUNILHFCQUFxQixFQUFFO29CQUNyQixTQUFTLEVBQUUsdUJBQXVCO2lCQUNuQzthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVDQUF1QyxFQUFFO2dCQUM1RSxjQUFjLEVBQUUsa0JBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ2xGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFFTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtJQUM5RCxRQUFRO0lBQ1IsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQzVDLE9BQU8sR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzdDLElBQUksZ0JBQVUsQ0FBQyxRQUFRLEVBQUUsWUFBWSxFQUFFO1FBQ3JDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLFNBQVMsS0FBSztRQUNaLG1CQUFVLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQywyREFBMkQsQ0FBQyxDQUFDO0FBQ3JGLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJEQUEyRCxFQUFFLEdBQUcsRUFBRTtJQUNyRSxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDL0MsT0FBTztRQUNQLEdBQUc7S0FDSixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxXQUFXLEdBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFnQixDQUFDLElBQUksQ0FBQyxZQUE2QixDQUFDO0lBQ3pHLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtRQUN4RCxZQUFZLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDO1FBQzVDLFNBQVMsRUFBRTtZQUNULEdBQUcsRUFBRSxHQUFHO1lBQ1IsR0FBRyxFQUFFLEdBQUc7U0FDVDtRQUNELGFBQWEsRUFBRSxFQUFFO0tBQ2xCLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO0lBQzNFLFFBQVE7SUFDUixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDNUMsTUFBTSxFQUFFLEdBQUcsSUFBSSxvQkFBYSxDQUFDLFFBQVEsRUFBRSxZQUFZLEVBQUU7UUFDbkQsR0FBRztLQUNKLENBQUMsQ0FBQztJQUNILE1BQU0sV0FBVyxHQUFHLElBQUksbUJBQVksQ0FBQyxRQUFRLEVBQUU7UUFDN0MsVUFBVSxFQUFFLEVBQUU7S0FDZixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsU0FBUyxJQUFJO1FBQ1gsSUFBSSxnQkFBVSxDQUFDLFFBQVEsRUFBRSxNQUFNLEVBQUU7WUFDL0IsT0FBTztZQUNQLEdBQUc7WUFDSCxVQUFVLEVBQUUsV0FBVztTQUN4QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsT0FBTztJQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsMEVBQTBFLENBQUMsQ0FBQztBQUNuRyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvRUFBb0UsRUFBRSxHQUFHLEVBQUU7SUFDOUUsUUFBUTtJQUNSLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztJQUM1QyxNQUFNLFVBQVUsR0FBRyxJQUFJLGtCQUFZLENBQUMsUUFBUSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7SUFFN0YsT0FBTztJQUNQLE1BQU0sVUFBVSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxRQUFRLEVBQUUsTUFBTSxFQUFFO1FBQ2xELEdBQUc7UUFDSCxPQUFPLEVBQUUsVUFBVTtLQUNwQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUNqRSxNQUFNLENBQUMsVUFBVSxDQUFDLHlCQUF5QixDQUFDLFdBQVcsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDO0FBQzNFLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJFQUEyRSxFQUFFLEdBQUcsRUFBRTtJQUNyRixRQUFRO0lBQ1IsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQzVDLE1BQU0sVUFBVSxHQUFHLElBQUksa0JBQVksQ0FBQyxRQUFRLEVBQUUsb0JBQW9CLEVBQUUsRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztJQUU3RixPQUFPO0lBQ1AsU0FBUyxJQUFJO1FBQ1gsSUFBSSxnQkFBVSxDQUFDLFFBQVEsRUFBRSxNQUFNLEVBQUU7WUFDL0IsT0FBTyxFQUFFLFVBQVU7WUFDbkIsR0FBRztZQUNILHlCQUF5QixFQUFFO2dCQUN6QixPQUFPLEVBQUUsSUFBSTthQUNkO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELE9BQU87SUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLGtDQUFrQyxVQUFVLENBQUMsYUFBYSxvSEFBb0gsYUFBTyxDQUFDLGtDQUFrQyxDQUFDLGFBQWEsb0VBQW9FLENBQUMsQ0FBQztBQUNuVSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsUUFBUTtJQUNSLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxxQkFBSyxDQUFDLEtBQUssRUFBRSx5QkFBeUIsRUFBRTtRQUNyRSxJQUFJLEVBQUUsVUFBVTtLQUNqQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxVQUFVLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDckQsR0FBRztRQUNILE9BQU87UUFDUCxrQkFBa0I7S0FDbkIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sY0FBYyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBcUIsQ0FBQztJQUNyRixNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxjQUFjLGtCQUFrQixDQUFDLFdBQVcsR0FBRyxDQUFDLENBQUM7QUFDdEcsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbUdBQW1HLEVBQUUsR0FBRyxFQUFFO0lBQzdHLFFBQVE7SUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtRQUMvQyxPQUFPO1FBQ1AsR0FBRztRQUNILFVBQVUsRUFBRTtZQUNWLG9CQUFvQixFQUFFLENBQUMsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFLEdBQUUsQ0FBQztZQUM3Qyx3QkFBd0IsRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJO1NBQ3JDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBaUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7SUFFOUUsT0FBTztJQUNQLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7QUFDekMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNEdBQTRHLEVBQUUsR0FBRyxFQUFFO0lBQ3RILFFBQVE7SUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtRQUMvQyxPQUFPO1FBQ1AsR0FBRztRQUNILFVBQVUsRUFBRTtZQUNWLG9CQUFvQixFQUFFLENBQUMsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFLEdBQUUsQ0FBQztZQUM3Qyx3QkFBd0IsRUFBRSxHQUFHLEVBQUUsQ0FBQyxLQUFLO1NBQ3RDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBaUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7SUFFOUUsT0FBTztJQUNQLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0FBQzdDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdCQUF3QixFQUFFLEdBQUcsRUFBRTtJQUNsQyxRQUFRO0lBQ1IsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLDJCQUFNLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQ2pFLFdBQVcsRUFBRSxvREFBb0Q7UUFDakUsb0JBQW9CLEVBQUU7WUFDcEIsaUJBQWlCLEVBQUUsdUJBQXVCO1lBQzFDLFlBQVksRUFBRSxLQUFLO1lBQ25CLGNBQWMsRUFBRSxFQUFFO1lBQ2xCLHVCQUF1QixFQUFFLElBQUk7WUFDN0IsaUJBQWlCLEVBQUUsVUFBVTtZQUM3QixvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDO1NBQzVEO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sVUFBVSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1FBQ3JELEdBQUc7UUFDSCxPQUFPO1FBQ1AseUJBQXlCLEVBQUU7WUFDekIsT0FBTyxFQUFFLElBQUk7WUFDYixXQUFXLEVBQUUsbUJBQW1CO1NBQ2pDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsV0FBVyxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDbkYsTUFBTSxjQUFjLEdBQUcsVUFBVSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFxQixDQUFDO0lBQ3JGLE1BQU0sQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLE1BQU0sT0FBTyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDO0FBQy9HLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxPQUFPO0lBQ1AsTUFBTSxVQUFVLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDckQsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUNsRSxNQUFNLENBQUMsVUFBVSxDQUFDLHlCQUF5QixDQUFDLFdBQVcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0FBQ3pFLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtEQUErRCxFQUFFLEdBQUcsRUFBRTtJQUN6RSxPQUFPO0lBQ1AsTUFBTSxVQUFVLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDckQsR0FBRztRQUNILE9BQU87UUFDUCx5QkFBeUIsRUFBRTtZQUN6QixPQUFPLEVBQUUsS0FBSztTQUNmO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsV0FBVyxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7QUFDM0UsQ0FBQyxDQUFDLENBQUM7QUFHSCxJQUFJLENBQUMsdUVBQXVFLEVBQUUsR0FBRyxFQUFFO0lBQ2pGLFFBQVE7SUFDUixNQUFNLEVBQUUsR0FBRyxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUN4QyxHQUFHO0tBQ0osQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsSUFBSSwyQkFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUMzQyxNQUFNLFFBQVEsR0FBRywyQkFBZSxDQUFDLDZCQUE2QixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7UUFDakYsc0JBQXNCLEVBQUUsU0FBUztRQUNqQyxpQkFBaUIsRUFBRSxLQUFLO1FBQ3hCLHlCQUF5QixFQUFFLENBQUUsU0FBUyxDQUFFO1FBQ3hDLG1CQUFtQixFQUFFLENBQUUsSUFBSSxDQUFFO1FBQzdCLElBQUksRUFBRSxLQUFLO1FBQ1gscUJBQXFCLEVBQUUsU0FBUztRQUNoQyxhQUFhLEVBQUUsRUFBRTtLQUNsQixDQUFDLENBQUM7SUFDSCxNQUFNLGtCQUFrQixHQUFHLHdCQUFrQixDQUFDLFFBQVEsQ0FBQyxFQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFDLENBQUMsQ0FBQztJQUVsRixPQUFPO0lBQ1AsU0FBUyxJQUFJO1FBQ1gsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDbEMsR0FBRztZQUNILE9BQU87WUFDUCxRQUFRLEVBQUUsa0JBQWtCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRCxPQUFPO0lBQ1AsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxxU0FBcVMsQ0FBQyxDQUFDO0FBQzlULENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgYXJyYXlXaXRoLFxuICBjb3VudFJlc291cmNlc0xpa2UsXG4gIGV4cGVjdCBhcyBleHBlY3RDREssXG4gIGhhdmVSZXNvdXJjZSxcbiAgaGF2ZVJlc291cmNlTGlrZSxcbiAgUmVzb3VyY2VQYXJ0LFxuICBTeW50aFV0aWxzLFxufSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHtBdXRvU2NhbGluZ0dyb3VwfSBmcm9tICdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHtEYXRhYmFzZUNsdXN0ZXJ9IGZyb20gJ0Bhd3MtY2RrL2F3cy1kb2NkYic7XG5pbXBvcnQge1xuICBBbWF6b25MaW51eEdlbmVyYXRpb24sXG4gIEluc3RhbmNlLFxuICBJbnN0YW5jZUNsYXNzLFxuICBJbnN0YW5jZVNpemUsXG4gIEluc3RhbmNlVHlwZSxcbiAgSVNlY3VyaXR5R3JvdXAsXG4gIElWcGMsXG4gIE1hY2hpbmVJbWFnZSxcbiAgU2VjdXJpdHlHcm91cCxcbiAgU3VibmV0LFxuICBTdWJuZXRUeXBlLFxuICBWcGMsXG4gIFdpbmRvd3NWZXJzaW9uLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIEFjY2Vzc1BvaW50LFxuICBDZm5GaWxlU3lzdGVtLFxuICBGaWxlU3lzdGVtIGFzIEVmc0ZpbGVTeXN0ZW0sXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lZnMnO1xuaW1wb3J0IHsgQ2ZuUm9sZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IEFzc2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzLWFzc2V0cyc7XG5pbXBvcnQgeyBTZWNyZXQgfSBmcm9tICdAYXdzLWNkay9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHtcbiAgQXBwLFxuICBDZm5FbGVtZW50LFxuICBEdXJhdGlvbixcbiAgTmFtZXMsXG4gIFJlbW92YWxQb2xpY3ksXG4gIFN0YWNrLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgTW91bnRhYmxlRWZzLFxufSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCB7XG4gIHRlc3RDb25zdHJ1Y3RUYWdzLFxufSBmcm9tICcuLi8uLi9jb3JlL3Rlc3QvdGFnLWhlbHBlcnMnO1xuaW1wb3J0IHtcbiAgQ1dBX0FTU0VUX0xJTlVYLFxufSBmcm9tICcuLi8uLi9kZWFkbGluZS90ZXN0L2Fzc2V0LWNvbnN0YW50cyc7XG5pbXBvcnQge1xuICBEYXRhYmFzZUNvbm5lY3Rpb24sXG4gIElWZXJzaW9uLFxuICBSZXBvc2l0b3J5LFxuICBWZXJzaW9uUXVlcnksXG4gIFZlcnNpb24sXG4gIFBsYXRmb3JtSW5zdGFsbGVycyxcbn0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7XG4gIFJFUE9fRENfQVNTRVQsXG59IGZyb20gJy4vYXNzZXQtY29uc3RhbnRzJztcblxubGV0IGFwcDogQXBwO1xubGV0IHN0YWNrOiBTdGFjaztcbmxldCB2cGM6IElWcGM7XG5sZXQgdmVyc2lvbjogSVZlcnNpb247XG5cbmZ1bmN0aW9uIGVzY2FwZVRva2VuUmVnZXgoczogc3RyaW5nKTogc3RyaW5nIHtcbiAgLy8gQSBDREsgVG9rZW4gbG9va3MgbGlrZTogJHtUb2tlbltUT0tFTi4xMl19XG4gIC8vIFRoaXMgY29udGFpbnMgdGhlIHJlZ2V4IHNwZWNpYWwgY2hhcmFjdGVyczogLiwgJCwgeywgfSwgWywgYW5kIF1cbiAgLy8gRXNjYXBlIHRob3NlIGZvciB1c2UgaW4gYSByZWdleC5cbiAgcmV0dXJuIHMucmVwbGFjZSgvWy4ke31bXFxdXS9nLCAnXFxcXCQmJyk7XG59XG5cbmJlZm9yZUVhY2goKCkgPT4ge1xuICBhcHAgPSBuZXcgQXBwKCk7XG4gIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG4gIHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWUEMnLCB7XG4gICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAge1xuICAgICAgICBuYW1lOiAnUHVibGljJyxcbiAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiAnUHJpdmF0ZScsXG4gICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX05BVCxcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIG5hbWU6ICdJc29sYXRlZCcsXG4gICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCxcbiAgICAgIH0sXG4gICAgXSxcbiAgfSk7XG5cbiAgY2xhc3MgTW9ja1ZlcnNpb24gZXh0ZW5kcyBWZXJzaW9uIGltcGxlbWVudHMgSVZlcnNpb24ge1xuICAgIHJlYWRvbmx5IGxpbnV4SW5zdGFsbGVyczogUGxhdGZvcm1JbnN0YWxsZXJzID0ge1xuICAgICAgcGF0Y2hWZXJzaW9uOiAwLFxuICAgICAgcmVwb3NpdG9yeToge1xuICAgICAgICBvYmplY3RLZXk6ICd0ZXN0SW5zdGFsbGVyJyxcbiAgICAgICAgczNCdWNrZXQ6IG5ldyBCdWNrZXQoc3RhY2ssICdMaW51eEluc3RhbGxlckJ1Y2tldCcpLFxuICAgICAgfSxcbiAgICAgIGNsaWVudDoge1xuICAgICAgICBvYmplY3RLZXk6ICd0ZXN0Q2xpZW50SW5zdGFsbGVyJyxcbiAgICAgICAgczNCdWNrZXQ6IG5ldyBCdWNrZXQoc3RhY2ssICdMaW51eENsaWVudEluc3RhbGxlckJ1Y2tldCcpLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgcHVibGljIGxpbnV4RnVsbFZlcnNpb25TdHJpbmcoKSB7XG4gICAgICByZXR1cm4gdGhpcy50b1N0cmluZygpO1xuICAgIH1cbiAgfVxuXG4gIHZlcnNpb24gPSBuZXcgTW9ja1ZlcnNpb24oWzEwLDEsMTksNF0pO1xufSk7XG5cbnRlc3QoJ2NhbiBjcmVhdGUgdHdvIHJlcG9zaXRvcmllcycsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvMScsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QoKCkgPT4ge1xuICAgIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwbzInLCB7XG4gICAgICB2cGMsXG4gICAgICB2ZXJzaW9uLFxuICAgIH0pO1xuICB9KS5ub3QudG9UaHJvdygpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGluc3RhbmNlIGlzIGNyZWF0ZWQgY29ycmVjdGx5JywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnLCB7XG4gICAgUHJvcGVydGllczoge1xuICAgICAgTWF4U2l6ZTogJzEnLFxuICAgICAgTWluU2l6ZTogJzEnLFxuICAgICAgVlBDWm9uZUlkZW50aWZpZXI6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZjogJ1ZQQ1ByaXZhdGVTdWJuZXQxU3VibmV0OEJDQTEwRTAnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgUmVmOiAnVlBDUHJpdmF0ZVN1Ym5ldDJTdWJuZXRDRkNEQUE3QScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgQ3JlYXRpb25Qb2xpY3k6IHtcbiAgICAgIFJlc291cmNlU2lnbmFsOiB7XG4gICAgICAgIENvdW50OiAxLFxuICAgICAgICBUaW1lb3V0OiAnUFQxNU0nLFxuICAgICAgfSxcbiAgICB9LFxuICAgIFVwZGF0ZVBvbGljeToge1xuICAgICAgQXV0b1NjYWxpbmdSZXBsYWNpbmdVcGRhdGU6IHtcbiAgICAgICAgV2lsbFJlcGxhY2U6IHRydWUsXG4gICAgICB9LFxuICAgICAgQXV0b1NjYWxpbmdTY2hlZHVsZWRBY3Rpb246IHtcbiAgICAgICAgSWdub3JlVW5tb2RpZmllZEdyb3VwU2l6ZVByb3BlcnRpZXM6IHRydWUsXG4gICAgICB9LFxuICAgIH0sXG4gICAgRGVwZW5kc09uOiBbXG4gICAgICAncmVwb3NpdG9yeUluc3RhbGxlckRvY3VtZW50RGF0YWJhc2VJbnN0YW5jZTExQTZGOEM4RScsXG4gICAgXSxcbiAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkxhdW5jaENvbmZpZ3VyYXRpb24nLCB7XG4gICAgSW5zdGFuY2VUeXBlOiAndDMubGFyZ2UnLFxuICB9KSk7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgRnJvbVBvcnQ6IDIwNDksXG4gICAgVG9Qb3J0OiAyMDQ5LFxuICAgIEdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckZpbGVTeXN0ZW1FZnNTZWN1cml0eUdyb3VwMjg5RDA0M0MnLFxuICAgICAgICAnR3JvdXBJZCcsXG4gICAgICBdLFxuICAgIH0sXG4gICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgJ3JlcG9zaXRvcnlJbnN0YWxsZXJJbnN0YW5jZVNlY3VyaXR5R3JvdXAwQjU3MDVEMycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGhvbm9ycyB2cGNTdWJuZXQnLCAoKSA9PiB7XG4gIC8vIE5vdGU6IERlZmF1bHQgaXMgcHJpdmF0ZSBzdWJuZXRzLCBzbyBpdCdzIHN1ZmZpY2llbnQgdG8gdGVzdCBzb21ldGhpbmcgb3RoZXIgdGhhblxuICAvLyBwcml2YXRlIHN1Ym5ldHMuXG5cbiAgLy8gV0hFTlxuICBjb25zdCBpc29sYXRlZFN1Ym5ldElkcyA9IFsgJ0lzb2xhdGVkU3VibmV0MScsICdJc29sYXRlZFN1Ym5ldDInIF07XG4gIGNvbnN0IGF0dHJWcGMgPSBWcGMuZnJvbVZwY0F0dHJpYnV0ZXMoc3RhY2ssICdUZXN0VnBjJywge1xuICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBbJ3VzLWVhc3QtMWEnLCAndXMtZWFzdC0xYiddLFxuICAgIHZwY0lkOiAndnBjaWQnLFxuICAgIGlzb2xhdGVkU3VibmV0SWRzLFxuICB9KTtcbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYzogYXR0clZwYyxcbiAgICB2ZXJzaW9uLFxuICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX0lTT0xBVEVEIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgIFZQQ1pvbmVJZGVudGlmaWVyOiBpc29sYXRlZFN1Ym5ldElkcyxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIHNlY3VyaXR5IGdyb3VwcyBjcmVhdGVkIGNvcnJlY3RseScsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgRnJvbVBvcnQ6IDIwNDksXG4gICAgVG9Qb3J0OiAyMDQ5LFxuICAgIEdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckZpbGVTeXN0ZW1FZnNTZWN1cml0eUdyb3VwMjg5RDA0M0MnLFxuICAgICAgICAnR3JvdXBJZCcsXG4gICAgICBdLFxuICAgIH0sXG4gICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgJ3JlcG9zaXRvcnlJbnN0YWxsZXJJbnN0YW5jZVNlY3VyaXR5R3JvdXAwQjU3MDVEMycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICBGcm9tUG9ydDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRG9jdW1lbnREYXRhYmFzZUEzNkNFN0ZFJyxcbiAgICAgICAgJ1BvcnQnLFxuICAgICAgXSxcbiAgICB9LFxuICAgIFRvUG9ydDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRG9jdW1lbnREYXRhYmFzZUEzNkNFN0ZFJyxcbiAgICAgICAgJ1BvcnQnLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckRvY3VtZW50RGF0YWJhc2VTZWN1cml0eUdyb3VwQkVGREM1OEYnLFxuICAgICAgICAnR3JvdXBJZCcsXG4gICAgICBdLFxuICAgIH0sXG4gICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgJ3JlcG9zaXRvcnlJbnN0YWxsZXJJbnN0YW5jZVNlY3VyaXR5R3JvdXAwQjU3MDVEMycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbi8qXG5JQU0gUG9saWN5IGRvY3VtZW50IHRlc3RzLiBUaGUgcG9saWN5IGZvciB0aGUgaW5zdGFsbGVyIGluc3RhbmNlIGlzOlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7fSwgLy8gQ2xvdWRXYXRjaCBsb2cgZ3JvdXAgcHV0XG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IGluc3RhbGwgc2NyaXB0XG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IHN0cmluZyBwYXJhbWV0ZXJzXG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IGdldCBpbnN0YWxsZXIgcGVybWlzc2lvbnNcbiAgICAgICAge30sIC8vIGdwZyBnZXQgaW5zdGFsbGVyIHBlcm1pc3Npb25zXG4gICAgICAgIHt9LCAvLyBEb2NEQiBzZWNyZXQgZ2V0XG4gICAgICAgIHt9LCAvLyBmaWxlc3lzdGVtIG1vdW50IHNjcmlwdCBnZXRcbiAgICAgICAge30sIC8vIGluc3RhbGxlciBnZXRcbiAgICAgICAge30sIC8vIHJlcG9zaXRvcnkgaW5zdGFsbGF0aW9uIHNjcmlwdCBhc3NldCBnZXRcbiAgICAgICAge30sIC8vIHVwZGF0ZSBhdXRvc2NhbGluZyBwb2xpY3lcbiAgICAgICAge30sIC8vIGRlc2NyaWJlIHRhZ3NcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xuXG4gIFdlIG9ubHkgZXhwbGljaXRseSB0ZXN0IGZvciB0aGUgcGVybWlzc2lvbnMgd2UgZXhwbGljaXRseSBhZGQ6XG4gICAgLSBkb2NEQiBzZWNyZXQgZ2V0XG4gICAgLSBpbnN0YWxsZXIgZ2V0XG4gICAgLSBhdXRvc2NhbGluZyBwb2xpY3kgKGhhbmRsZWQgYnk6ICdhbGlkYXRlIGluc3RhbmNlIHNlbGYtdGVybWluYXRpb24nIHRlc3QpXG4gICAgLSBkZXNjcmliZSB0YWdzIChoYW5kbGVkIGJ5OiAnYWxpZGF0ZSBpbnN0YW5jZSBzZWxmLXRlcm1pbmF0aW9uJyB0ZXN0KVxuICBUaGUgb3RoZXJzIGFyZSBzaWRlLWVmZmVjdHMgb2YgYWRkaW5nIHRoZSBjbG91ZHdhdGNoIGFnZW50LCBhbmQgbW91bnRpbmcgdGhlIGZpbGVzeXN0ZW0uXG4gIFdlIG1ha2Ugc3VyZSB0aGF0IHdlIGRvIHRob3NlIHRoaW5ncyBpbiBvdGhlciB3YXlzLCBhbmQgdHJ1c3QgdG8gdGhlaXIgY29uc3RydWN0cyB0byBiZVxuICB2ZXJpZnlpbmcgcHJvcGVyIHBlcm1pc3Npb25zLlxuICovXG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGlhbSBwZXJtaXNzaW9uczogZGIgc2VjcmV0IGFjY2VzcycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aCh7XG4gICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgICAgXSxcbiAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgIFJlZjogJ3JlcG9zaXRvcnlJbnN0YWxsZXJEb2N1bWVudERhdGFiYXNlU2VjcmV0QXR0YWNobWVudDI5NzUzQjdDJyxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdyZXBvc2l0b3J5IGluc3RhbGxlciBpYW0gcGVybWlzc2lvbnM6IGluc3RhbGxlciBnZXQnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgoXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICdzMzpMaXN0KicsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6IGFycmF5V2l0aChcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6czM6OjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IENXQV9BU1NFVF9MSU5VWC5CdWNrZXQsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpzMzo6OicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogQ1dBX0FTU0VUX0xJTlVYLkJ1Y2tldCxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnLyonLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIH0sXG4gICAgICApLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdkZWZhdWx0IHJlcG9zaXRvcnkgaW5zdGFsbGVyIGxvZyBncm91cCBjcmVhdGVkIGNvcnJlY3RseScsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6TG9nUmV0ZW50aW9uJywge1xuICAgIFJldGVudGlvbkluRGF5czogMyxcbiAgICBMb2dHcm91cE5hbWU6ICcvcmVuZGVyZmFybS9yZXBvc2l0b3J5SW5zdGFsbGVyJyxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGxvZ3MgYWxsIHJlcXVpcmVkIGZpbGVzJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICAvLyBUaGUgQ2xvdWRXYXRjaEFnZW50IHN0b3JlcyB0aGUgYWdlbnQgY29uZmlndXJhdGlvbiBpbiBhbiBTU00gUGFyYW1ldGVyLiBDaGVjayBpdCBmb3IgdGhlIHJlcXVpcmVkIHNldHVwLlxuICAvLyBOb3RlOiBUaGlzIHdvdWxkIGJlIGJldHRlciBpbXBsZW1lbnRlZCB1c2luZyB0aGUgdG9vbHMgaW46IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzg0NDRcbiAgLy8gYnV0IHRoYXQgaXMgbm90IHlldCBhdmFpbGFibGUuXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpTU006OlBhcmFtZXRlcicsIHtcbiAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICBWYWx1ZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICd7XFxcImxvZ3NcXFwiOntcXFwibG9nc19jb2xsZWN0ZWRcXFwiOntcXFwiZmlsZXNcXFwiOntcXFwiY29sbGVjdF9saXN0XFxcIjpbe1xcXCJsb2dfZ3JvdXBfbmFtZVxcXCI6XFxcIicsXG4gICAgICAgICAge30sIC8vIGxvZyBncm91cCBuYW1lLiBjaGVja2VkIGluIGFub3RoZXIgdGVzdC5cbiAgICAgICAgICAnXFxcIixcXFwibG9nX3N0cmVhbV9uYW1lXFxcIjpcXFwiY2xvdWQtaW5pdC1vdXRwdXQte2luc3RhbmNlX2lkfVxcXCIsXFxcImZpbGVfcGF0aFxcXCI6XFxcIi92YXIvbG9nL2Nsb3VkLWluaXQtb3V0cHV0LmxvZ1xcXCIsXFxcInRpbWV6b25lXFxcIjpcXFwiTG9jYWxcXFwifSx7XFxcImxvZ19ncm91cF9uYW1lXFxcIjpcXFwiJyxcbiAgICAgICAgICB7fSwgLy8gbG9nIGdyb3VwIG5hbWUgYWdhaW4uXG4gICAgICAgICAgJ1xcXCIsXFxcImxvZ19zdHJlYW1fbmFtZVxcXCI6XFxcImRlYWRsaW5lUmVwb3NpdG9yeUluc3RhbGxhdGlvbkxvZ3Mte2luc3RhbmNlX2lkfVxcXCIsXFxcImZpbGVfcGF0aFxcXCI6XFxcIi90bXAvYml0cm9ja19pbnN0YWxsZXIubG9nXFxcIixcXFwidGltZXpvbmVcXFwiOlxcXCJMb2NhbFxcXCJ9XX19LFxcXCJsb2dfc3RyZWFtX25hbWVcXFwiOlxcXCJEZWZhdWx0TG9nU3RyZWFtLXtpbnN0YW5jZV9pZH1cXFwiLFxcXCJmb3JjZV9mbHVzaF9pbnRlcnZhbFxcXCI6MTV9fScsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdyZXBvc2l0b3J5IG1vdW50cyByZXBvc2l0b3J5IGZpbGVzeXN0ZW0nLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgdXNlckRhdGEgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ21vdW50RWZzLnNoICR7VG9rZW5bVE9LRU4uXFxcXGQrXX0gL21udC9lZnMvZnMxIGZhbHNlIHJ3JykpKTtcbn0pO1xuXG50ZXN0LmVhY2goW1xuICBbUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLCAnRGVsZXRlJ10sXG4gIFtSZW1vdmFsUG9saWN5LlJFVEFJTiwgJ1JldGFpbiddLFxuICBbUmVtb3ZhbFBvbGljeS5TTkFQU0hPVCwgJ1NuYXBzaG90J10sXG5dKSgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgcmVtb3ZhbCBwb2xpY3k6ICVwJywgKHBvbGljeTogUmVtb3ZhbFBvbGljeSwgZXhwZWN0ZWQ6IHN0cmluZykgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBkYXRhYmFzZTogcG9saWN5LFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInLCB7XG4gICAgRGVsZXRpb25Qb2xpY3k6IGV4cGVjdGVkLFxuICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG59KTtcblxudGVzdC5lYWNoKFtcbiAgW1JlbW92YWxQb2xpY3kuREVTVFJPWSwgJ0RlbGV0ZSddLFxuICBbUmVtb3ZhbFBvbGljeS5SRVRBSU4sICdSZXRhaW4nXSxcbiAgW1JlbW92YWxQb2xpY3kuU05BUFNIT1QsICdTbmFwc2hvdCddLFxuXSkoJ3JlcG9zaXRvcnkgaG9ub3JzIGZpbGVzeXN0ZW0gcmVtb3ZhbCBwb2xpY3k6ICVwJywgKHBvbGljeTogUmVtb3ZhbFBvbGljeSwgZXhwZWN0ZWQ6IHN0cmluZykgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBmaWxlc3lzdGVtOiBwb2xpY3ksXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUZTOjpGaWxlU3lzdGVtJywge1xuICAgIERlbGV0aW9uUG9saWN5OiBleHBlY3RlZCxcbiAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgd2FybnMgaWYgcmVtb3ZhbCBwb2xpY3kgZm9yIGZpbGVzeXN0ZW0gd2hlbiBmaWxlc3lzdGVtIHByb3ZpZGVkJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCB0ZXN0RUZTID0gbmV3IEVmc0ZpbGVTeXN0ZW0oc3RhY2ssICdUZXN0RWZzRmlsZVN5c3RlbScsIHtcbiAgICB2cGMsXG4gIH0pO1xuICBjb25zdCB0ZXN0QVAgPSBuZXcgQWNjZXNzUG9pbnQoc3RhY2ssICdUZXN0QWNjZXNzUG9pbnQnLCB7XG4gICAgZmlsZVN5c3RlbTogdGVzdEVGUyxcbiAgfSk7XG4gIGNvbnN0IHRlc3RGUyA9IG5ldyBNb3VudGFibGVFZnMoc3RhY2ssIHtcbiAgICBmaWxlc3lzdGVtOiB0ZXN0RUZTLFxuICAgIGFjY2Vzc1BvaW50OiB0ZXN0QVAsXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgZmlsZVN5c3RlbTogdGVzdEZTLFxuICAgIHZlcnNpb24sXG4gICAgcmVtb3ZhbFBvbGljeToge1xuICAgICAgZmlsZXN5c3RlbTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlcG8ubm9kZS5tZXRhZGF0YUVudHJ5KS50b0VxdWFsKFxuICAgIGV4cGVjdC5hcnJheUNvbnRhaW5pbmcoW1xuICAgICAgZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgICB0eXBlOiAnYXdzOmNkazp3YXJuaW5nJyxcbiAgICAgICAgZGF0YTogJ1JlbW92YWxQb2xpY3kgZm9yIGZpbGVzeXN0ZW0gd2lsbCBub3QgYmUgYXBwbGllZCBzaW5jZSBhIGZpbGVzeXN0ZW0gaXMgbm90IGJlaW5nIGNyZWF0ZWQgYnkgdGhpcyBjb25zdHJ1Y3QnLFxuICAgICAgfSksXG4gICAgXSksXG4gICk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSB3YXJucyBpZiByZW1vdmFsIHBvbGljeSBmb3IgZGF0YWJhc2Ugd2hlbiBkYXRhYmFzZSBwcm92aWRlZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZnNEYXRhYmFzZSA9IG5ldyBEYXRhYmFzZUNsdXN0ZXIoc3RhY2ssICdUZXN0RGJDbHVzdGVyJywge1xuICAgIG1hc3RlclVzZXI6IHtcbiAgICAgIHVzZXJuYW1lOiAnbWFzdGVyJyxcbiAgICB9LFxuICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICApLFxuICAgIHZwYyxcbiAgICB2cGNTdWJuZXRzOiB7XG4gICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX05BVCxcbiAgICB9LFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIGRhdGFiYXNlOiBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yRG9jREIoeyBkYXRhYmFzZTogZnNEYXRhYmFzZSwgbG9naW46IGZzRGF0YWJhc2Uuc2VjcmV0ISB9KSxcbiAgICB2ZXJzaW9uLFxuICAgIHJlbW92YWxQb2xpY3k6IHtcbiAgICAgIGRhdGFiYXNlOiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVwby5ub2RlLm1ldGFkYXRhRW50cnkpLnRvRXF1YWwoXG4gICAgZXhwZWN0LmFycmF5Q29udGFpbmluZyhbXG4gICAgICBleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgICAgIHR5cGU6ICdhd3M6Y2RrOndhcm5pbmcnLFxuICAgICAgICBkYXRhOiAnUmVtb3ZhbFBvbGljeSBmb3IgZGF0YWJhc2Ugd2lsbCBub3QgYmUgYXBwbGllZCBzaW5jZSBhIGRhdGFiYXNlIGlzIG5vdCBiZWluZyBjcmVhdGVkIGJ5IHRoaXMgY29uc3RydWN0JyxcbiAgICAgIH0pLFxuICAgIF0pLFxuICApO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgY3JlYXRlcyBkZWFkbGluZURhdGFiYXNlIGlmIG5vbmUgcHJvdmlkZWQnLCAoKSA9PiB7XG4gIGNvbnN0IHRlc3RFRlMgPSBuZXcgRWZzRmlsZVN5c3RlbShzdGFjaywgJ1Rlc3RFZnNGaWxlU3lzdGVtJywge1xuICAgIHZwYyxcbiAgfSk7XG4gIGNvbnN0IHRlc3RBUCA9IG5ldyBBY2Nlc3NQb2ludChzdGFjaywgJ1Rlc3RBY2Nlc3NQb2ludCcsIHtcbiAgICBmaWxlU3lzdGVtOiB0ZXN0RUZTLFxuICB9KTtcbiAgY29uc3QgdGVzdEZTID0gbmV3IE1vdW50YWJsZUVmcyhzdGFjaywge1xuICAgIGZpbGVzeXN0ZW06IHRlc3RFRlMsXG4gICAgYWNjZXNzUG9pbnQ6IHRlc3RBUCxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIGZpbGVTeXN0ZW06IHRlc3RGUyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkRvY0RCOjpEQkluc3RhbmNlJykpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlcicsIHtcbiAgICBFbmFibGVDbG91ZHdhdGNoTG9nc0V4cG9ydHM6IFsgJ2F1ZGl0JyBdLFxuICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlclBhcmFtZXRlckdyb3VwJywge1xuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIGF1ZGl0X2xvZ3M6ICdlbmFibGVkJyxcbiAgICB9LFxuICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RG9jREI6OkRCSW5zdGFuY2UnLCB7XG4gICAgQXV0b01pbm9yVmVyc2lvblVwZ3JhZGU6IHRydWUsXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdkaXNhYmxpbmcgQXVkaXQgbG9nZ2luZyBkb2VzIG5vdCBlbmFibGUgQ2xvdWR3YXRjaCBhdWRpdCBsb2dzJywgKCkgPT4ge1xuICBjb25zdCB0ZXN0RUZTID0gbmV3IEVmc0ZpbGVTeXN0ZW0oc3RhY2ssICdUZXN0RWZzRmlsZVN5c3RlbScsIHtcbiAgICB2cGMsXG4gIH0pO1xuICBjb25zdCB0ZXN0QVAgPSBuZXcgQWNjZXNzUG9pbnQoc3RhY2ssICdUZXN0QWNjZXNzUG9pbnQnLCB7XG4gICAgZmlsZVN5c3RlbTogdGVzdEVGUyxcbiAgfSk7XG4gIGNvbnN0IHRlc3RGUyA9IG5ldyBNb3VudGFibGVFZnMoc3RhY2ssIHtcbiAgICBmaWxlc3lzdGVtOiB0ZXN0RUZTLFxuICAgIGFjY2Vzc1BvaW50OiB0ZXN0QVAsXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICBmaWxlU3lzdGVtOiB0ZXN0RlMsXG4gICAgdmVyc2lvbixcbiAgICBkYXRhYmFzZUF1ZGl0TG9nZ2luZzogZmFsc2UsXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlcicpKTtcbiAgZXhwZWN0Q0RLKHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInLCB7XG4gICAgRW5hYmxlQ2xvdWR3YXRjaExvZ3NFeHBvcnRzOiBbICdhdWRpdCcgXSxcbiAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbiAgZXhwZWN0Q0RLKHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCcsIHtcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBhdWRpdF9sb2dzOiAnZW5hYmxlZCcsXG4gICAgfSxcbiAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbn0pO1xuXG50ZXN0KCdyZXBvc2l0b3J5IHdhcm5zIGlmIGRhdGFiYXNlQXVkaXRMb2dnaW5nIGRlZmluZWQgYW5kIGRhdGFiYXNlIGlzIHNwZWNpZmllZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZnNEYXRhYmFzZSA9IG5ldyBEYXRhYmFzZUNsdXN0ZXIoc3RhY2ssICdUZXN0RGJDbHVzdGVyJywge1xuICAgIG1hc3RlclVzZXI6IHtcbiAgICAgIHVzZXJuYW1lOiAnbWFzdGVyJyxcbiAgICB9LFxuICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICApLFxuICAgIHZwYyxcbiAgICB2cGNTdWJuZXRzOiB7XG4gICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX05BVCxcbiAgICB9LFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgcmVtb3ZhbFBvbGljeToge1xuICAgICAgZmlsZXN5c3RlbTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH0sXG4gICAgZGF0YWJhc2U6IERhdGFiYXNlQ29ubmVjdGlvbi5mb3JEb2NEQih7IGRhdGFiYXNlOiBmc0RhdGFiYXNlLCBsb2dpbjogZnNEYXRhYmFzZS5zZWNyZXQhIH0pLFxuICAgIGRhdGFiYXNlQXVkaXRMb2dnaW5nOiB0cnVlLFxuICB9KTtcblxuICBjb25zdCB3YXJuaW5nTXNnID0gJ1RoZSBwYXJhbWV0ZXIgZGF0YWJhc2VBdWRpdExvZ2dpbmcgb25seSBoYXMgYW4gZWZmZWN0IHdoZW4gdGhlIFJlcG9zaXRvcnkgaXMgY3JlYXRpbmcgaXRzIG93biBkYXRhYmFzZS5cXG4nICtcbiAgICAnUGxlYXNlIGVuc3VyZSB0aGF0IHRoZSBEYXRhYmFzZSBwcm92aWRlZCBpcyBjb25maWd1cmVkIGNvcnJlY3RseS4nO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlcG8ubm9kZS5tZXRhZGF0YUVudHJ5KS50b0VxdWFsKFxuICAgIGV4cGVjdC5hcnJheUNvbnRhaW5pbmcoW1xuICAgICAgZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgICB0eXBlOiAnYXdzOmNkazp3YXJuaW5nJyxcbiAgICAgICAgZGF0YTogd2FybmluZ01zZyxcbiAgICAgIH0pLFxuICAgIF0pLFxuICApO1xufSk7XG5cbnRlc3QoJ2hvbm9ycyBzdWJuZXQgc3BlY2lmaWNhdGlvbicsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZGVwZW5kZW5jeVN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0RlcFN0YWNrJyk7XG4gIGNvbnN0IGRlcGVuZGVuY3lWcGMgPSBuZXcgVnBjKGRlcGVuZGVuY3lTdGFjaywgJ0RlcFZwYycpO1xuXG4gIGNvbnN0IHN1Ym5ldHMgPSBbXG4gICAgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKGRlcGVuZGVuY3lTdGFjaywgJ1N1Ym5ldDEnLCB7XG4gICAgICBzdWJuZXRJZDogJ1N1Ym5ldElEMScsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiAndXMtd2VzdC0yYScsXG4gICAgfSksXG4gICAgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKGRlcGVuZGVuY3lTdGFjaywgJ1N1Ym5ldDInLCB7XG4gICAgICBzdWJuZXRJZDogJ1N1Ym5ldElEMicsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiAndXMtd2VzdC0yYicsXG4gICAgfSksXG4gIF07XG4gIGNvbnN0IGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoaXNvbGF0ZWRTdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjOiBkZXBlbmRlbmN5VnBjLFxuICAgIHZlcnNpb24sXG4gICAgdnBjU3VibmV0czoge1xuICAgICAgc3VibmV0cyxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQlN1Ym5ldEdyb3VwJywge1xuICAgIFN1Ym5ldElkczogW1xuICAgICAgJ1N1Ym5ldElEMScsXG4gICAgICAnU3VibmV0SUQyJyxcbiAgICBdLFxuICB9KSk7XG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVGUzo6TW91bnRUYXJnZXQnLCB7IFN1Ym5ldElkOiAnU3VibmV0SUQxJyB9KSk7XG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVGUzo6TW91bnRUYXJnZXQnLCB7IFN1Ym5ldElkOiAnU3VibmV0SUQyJyB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgaW5zdGFuY2UgY291bnQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGluc3RhbmNlQ291bnQgPSAyO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICAgIGRvY3VtZW50RGJJbnN0YW5jZUNvdW50OiBpbnN0YW5jZUNvdW50LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OkRvY0RCOjpEQkluc3RhbmNlJywgaW5zdGFuY2VDb3VudCwge1xuICAgIEF1dG9NaW5vclZlcnNpb25VcGdyYWRlOiB0cnVlLFxuICB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgcmV0ZW50aW9uIHBlcmlvZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgcGVyaW9kID0gMjA7XG5cbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgYmFja3VwT3B0aW9uczoge1xuICAgICAgZGF0YWJhc2VSZXRlbnRpb246IER1cmF0aW9uLmRheXMocGVyaW9kKSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyJywge1xuICAgIEJhY2t1cFJldGVudGlvblBlcmlvZDogcGVyaW9kLFxuICB9KSk7XG59KTtcblxudGVzdCgnd2FybnMgaWYgYm90aCByZXRlbnRpb24gcGVyaW9kIGFuZCBkYXRhYmFzZSBwcm92aWRlZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZnNEYXRhYmFzZSA9IG5ldyBEYXRhYmFzZUNsdXN0ZXIoc3RhY2ssICdUZXN0RGJDbHVzdGVyJywge1xuICAgIG1hc3RlclVzZXI6IHtcbiAgICAgIHVzZXJuYW1lOiAnbWFzdGVyJyxcbiAgICB9LFxuICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICApLFxuICAgIHZwYyxcbiAgICB2cGNTdWJuZXRzOiB7XG4gICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX05BVCxcbiAgICB9LFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIGRhdGFiYXNlOiBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yRG9jREIoeyBkYXRhYmFzZTogZnNEYXRhYmFzZSwgbG9naW46IGZzRGF0YWJhc2Uuc2VjcmV0ISB9KSxcbiAgICB2ZXJzaW9uLFxuICAgIGJhY2t1cE9wdGlvbnM6IHtcbiAgICAgIGRhdGFiYXNlUmV0ZW50aW9uOiBEdXJhdGlvbi5kYXlzKDIwKSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChyZXBvLm5vZGUubWV0YWRhdGFFbnRyeSkudG9FcXVhbChcbiAgICBleHBlY3QuYXJyYXlDb250YWluaW5nKFtcbiAgICAgIGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgICAgdHlwZTogJ2F3czpjZGs6d2FybmluZycsXG4gICAgICAgIGRhdGE6ICdCYWNrdXAgcmV0ZW50aW9uIGZvciBkYXRhYmFzZSB3aWxsIG5vdCBiZSBhcHBsaWVkIHNpbmNlIGEgZGF0YWJhc2UgaXMgbm90IGJlaW5nIGNyZWF0ZWQgYnkgdGhpcyBjb25zdHJ1Y3QnLFxuICAgICAgfSksXG4gICAgXSksXG4gICk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBjcmVhdGVzIGZpbGVzeXN0ZW0gaWYgbm9uZSBwcm92aWRlZCcsICgpID0+IHtcblxuICBjb25zdCBmc0RhdGFiYXNlID0gbmV3IERhdGFiYXNlQ2x1c3RlcihzdGFjaywgJ1Rlc3REYkNsdXN0ZXInLCB7XG4gICAgbWFzdGVyVXNlcjoge1xuICAgICAgdXNlcm5hbWU6ICdtYXN0ZXInLFxuICAgIH0sXG4gICAgaW5zdGFuY2VUeXBlOiBJbnN0YW5jZVR5cGUub2YoXG4gICAgICBJbnN0YW5jZUNsYXNzLlI0LFxuICAgICAgSW5zdGFuY2VTaXplLkxBUkdFLFxuICAgICksXG4gICAgdnBjLFxuICAgIHZwY1N1Ym5ldHM6IHtcbiAgICAgIG9uZVBlckF6OiB0cnVlLFxuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfTkFULFxuICAgIH0sXG4gICAgYmFja3VwOiB7XG4gICAgICByZXRlbnRpb246IER1cmF0aW9uLmRheXMoMTUpLFxuICAgIH0sXG4gIH0pO1xuXG4gIGlmICghZnNEYXRhYmFzZS5zZWNyZXQpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ3NlY3JldCBjYW5ub3QgYmUgbnVsbCcpO1xuICB9XG5cbiAgLy8gV0hFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICBkYXRhYmFzZTogRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHsgZGF0YWJhc2U6IGZzRGF0YWJhc2UsIGxvZ2luOiBmc0RhdGFiYXNlLnNlY3JldCB9KSxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVGUzo6RmlsZVN5c3RlbScpKTtcbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUZTOjpNb3VudFRhcmdldCcpKTtcbiAgZXhwZWN0KHJlcG8ubm9kZS50cnlGaW5kQ2hpbGQoJ1BhZEVmc1N0b3JhZ2UnKSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlcG8ubm9kZS5maW5kQ2hpbGQoJ0ZpbGVTeXN0ZW0nKS5ub2RlLnRyeUZpbmRDaGlsZCgnUGFkZGluZ0FjY2Vzc1BvaW50JykpLnRvQmVEZWZpbmVkKCk7XG59KTtcblxudGVzdCgnZGVmYXVsdCByZXBvc2l0b3J5IGluc3RhbmNlIGlzIGNyZWF0ZWQgdXNpbmcgdXNlciBkZWZpbmVkIGluc3RhbGxhdGlvbiBwYXRoIHByZWZpeCcsICgpID0+IHtcbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uUHJlZml4OiAneHl6JyxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBjb25zdCBzY3JpcHQgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YTtcbiAgZXhwZWN0KHNjcmlwdC5yZW5kZXIoKSkudG9NYXRjaCgvXFwvbW50XFwvZWZzXFwvZnMxXFwveHl6Lyk7XG59KTtcblxudGVzdCgnZGVmYXVsdCByZXBvc2l0b3J5IGluc3RhbmNlIGlzIGNyZWF0ZWQgdXNpbmcgdXNlciBkZWZpbmVkIGluc3RhbGxhdGlvbiBwYXRoIHByZWZpeCB3aXRoIGV4dHJhIHNsYXNoZXMgaW4gcGF0aCcsICgpID0+IHtcbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uUHJlZml4OiAnL3h5ei8vJyxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBjb25zdCBzY3JpcHQgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YTtcbiAgZXhwZWN0KHNjcmlwdC5yZW5kZXIoKSkudG9NYXRjaCgvXFwvbW50XFwvZWZzXFwvZnMxXFwveHl6Lyk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBpbnN0YW5jZSBpcyBjcmVhdGVkIHdpdGggdXNlciBkZWZpbmVkIHRpbWVvdXQnLCAoKSA9PiB7XG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMzApLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnLCB7XG4gICAgQ3JlYXRpb25Qb2xpY3k6IHtcbiAgICAgIFJlc291cmNlU2lnbmFsOiB7XG4gICAgICAgIENvdW50OiAxLFxuICAgICAgICBUaW1lb3V0OiAnUFQzME0nLFxuICAgICAgfSxcbiAgICB9LFxuICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBpbnN0YW5jZSBpcyBjcmVhdGVkIHdpdGggY29ycmVjdCBpbnN0YWxsZXIgcGF0aCB2ZXJzaW9uJywgKCkgPT4ge1xuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uVGltZW91dDogRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBjb25zdCBzY3JpcHQgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YTtcbiAgZXhwZWN0KHNjcmlwdC5yZW5kZXIoKSkudG9FcXVhbChleHBlY3Quc3RyaW5nQ29udGFpbmluZyh2ZXJzaW9uLnZlcnNpb25TdHJpbmcpKTtcbn0pO1xuXG50ZXN0LmVhY2goW1xuICAndGVzdC1wcmVmaXgvJyxcbiAgJycsXG5dKSgncmVwb3NpdG9yeSBpbnN0YW5jZSBpcyBjcmVhdGVkIHdpdGggY29ycmVjdCBMb2dHcm91cCBwcmVmaXggJXMnLCAodGVzdFByZWZpeDogc3RyaW5nKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGlkID0gJ3JlcG9zaXRvcnlJbnN0YWxsZXInO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssIGlkLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgbG9nR3JvdXBQcm9wczoge1xuICAgICAgbG9nR3JvdXBQcmVmaXg6IHRlc3RQcmVmaXgsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpMb2dSZXRlbnRpb24nLCB7XG4gICAgTG9nR3JvdXBOYW1lOiB0ZXN0UHJlZml4ICsgaWQsXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCd2YWxpZGF0ZSBpbnN0YW5jZSBzZWxmLXRlcm1pbmF0aW9uJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHJlcG9zaXRvcnlJbnN0YWxsYXRpb25UaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDMwKSxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGNvbnN0IHJlZ2lvblRva2VuID0gZXNjYXBlVG9rZW5SZWdleCgnJHtUb2tlbltBV1MuUmVnaW9uLlxcXFxkK119Jyk7XG4gIGNvbnN0IGFzZ0xvZ2ljYWxJZFRva2VuID0gZXNjYXBlVG9rZW5SZWdleCgnJHtUb2tlbltTdGFjay5yZXBvc2l0b3J5SW5zdGFsbGVyLkluc3RhbGxlci5BU0cuTG9naWNhbElELlxcXFxkK119Jyk7XG4gIGNvbnN0IGV4cGVjdGVkU3RyaW5nID0gYGZ1bmN0aW9uIGV4aXRUcmFwXFxcXChcXFxcKVxcXFx7XFxuZXhpdENvZGU9XFxcXCRcXFxcP1xcbnNsZWVwIDFtXFxuVE9LRU49XFxcXCRcXFxcKGN1cmwgLVggUFVUIFwiaHR0cDpcXFxcL1xcXFwvMTY5XFxcXC4yNTRcXFxcLjE2OVxcXFwuMjU0XFxcXC9sYXRlc3RcXFxcL2FwaVxcXFwvdG9rZW5cIiAtSCBcIlgtYXdzLWVjMi1tZXRhZGF0YS10b2tlbi10dGwtc2Vjb25kczogMzBcIiAyPiBcXFxcL2RldlxcXFwvbnVsbFxcXFwpXFxuSU5TVEFOQ0U9XCJcXFxcJFxcXFwoY3VybCAtcyAtSCBcIlgtYXdzLWVjMi1tZXRhZGF0YS10b2tlbjogXFxcXCRUT0tFTlwiIGh0dHA6XFxcXC9cXFxcLzE2OVxcXFwuMjU0XFxcXC4xNjlcXFxcLjI1NFxcXFwvbGF0ZXN0XFxcXC9tZXRhLWRhdGFcXFxcL2luc3RhbmNlLWlkICAyPiBcXFxcL2RldlxcXFwvbnVsbFxcXFwpXCJcXG5BU0c9XCJcXFxcJFxcXFwoYXdzIC0tcmVnaW9uICR7cmVnaW9uVG9rZW59IGVjMiBkZXNjcmliZS10YWdzIC0tZmlsdGVycyBcIk5hbWU9cmVzb3VyY2UtaWQsVmFsdWVzPVxcXFwkXFxcXHtJTlNUQU5DRVxcXFx9XCIgXCJOYW1lPWtleSxWYWx1ZXM9YXdzOmF1dG9zY2FsaW5nOmdyb3VwTmFtZVwiIC0tcXVlcnkgXCJUYWdzXFxcXFswXFxcXF1cXFxcLlZhbHVlXCIgLS1vdXRwdXQgdGV4dFxcXFwpXCJcXG5hd3MgLS1yZWdpb24gJHtyZWdpb25Ub2tlbn0gYXV0b3NjYWxpbmcgdXBkYXRlLWF1dG8tc2NhbGluZy1ncm91cCAtLWF1dG8tc2NhbGluZy1ncm91cC1uYW1lIFxcXFwkXFxcXHtBU0dcXFxcfSAtLW1pbi1zaXplIDAgLS1tYXgtc2l6ZSAwIC0tZGVzaXJlZC1jYXBhY2l0eSAwXFxuXFxcXC9vcHRcXFxcL2F3c1xcXFwvYmluXFxcXC9jZm4tc2lnbmFsIC0tc3RhY2sgJHtzdGFjay5zdGFja05hbWV9IC0tcmVzb3VyY2UgJHthc2dMb2dpY2FsSWRUb2tlbn0gLS1yZWdpb24gJHtyZWdpb25Ub2tlbn0gLWUgXFxcXCRleGl0Q29kZSBcXFxcfFxcXFx8IGVjaG8gJ0ZhaWxlZCB0byBzZW5kIENsb3VkZm9ybWF0aW9uIFNpZ25hbCdcXG5cXFxcfWA7XG4gIGV4cGVjdCgocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YS5yZW5kZXIoKSkudG9NYXRjaChuZXcgUmVnRXhwKGV4cGVjdGVkU3RyaW5nKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgoXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246ICdhdXRvc2NhbGluZzpVcGRhdGVBdXRvU2NhbGluZ0dyb3VwJyxcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICAnYXV0b3NjYWxpbmc6UmVzb3VyY2VUYWcvcmVzb3VyY2VMb2dpY2FsSWQnOiBOYW1lcy51bmlxdWVJZChyZXBvKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogJ2VjMjpEZXNjcmliZVRhZ3MnLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICB9LFxuICAgICAgKSxcbiAgICB9LFxuICB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBjb25maWd1cmUgY2xpZW50IGluc3RhbmNlJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uVGltZW91dDogRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgdmVyc2lvbixcbiAgfSk7XG4gIGNvbnN0IGluc3RhbmNlID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgdnBjLFxuICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgfSk7XG4gIGNvbnN0IGluc3RhbmNlUm9sZSA9IChcbiAgICBpbnN0YW5jZVxuICAgICAgLm5vZGUuZmluZENoaWxkKCdJbnN0YW5jZVJvbGUnKVxuICAgICAgLm5vZGUuZGVmYXVsdENoaWxkXG4gICkgYXMgQ2ZuUm9sZTtcbiAgY29uc3QgZGIgPSAoXG4gICAgcmVwb1xuICAgICAgLm5vZGUuZmluZENoaWxkKCdEb2N1bWVudERhdGFiYXNlJylcbiAgKSBhcyBEYXRhYmFzZUNsdXN0ZXI7XG5cbiAgLy8gV0hFTlxuICByZXBvLmNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHtcbiAgICBob3N0OiBpbnN0YW5jZSxcbiAgICBtb3VudFBvaW50OiAnL21udC9yZXBvc2l0b3J5JyxcbiAgfSk7XG4gIGNvbnN0IHVzZXJEYXRhID0gaW5zdGFuY2UudXNlckRhdGEucmVuZGVyKCk7XG5cbiAgLy8gVEhFTlxuICAvLyB3aGl0ZS1ib3ggdGVzdGluZy4gSWYgd2UgbW91bnQgdGhlIGZpbGVzeXN0ZW0sIHRoZW4gd2UndmUgY2FsbGVkOiBzZXR1cERpcmVjdENvbm5lY3QoKVxuICBleHBlY3QodXNlckRhdGEpLnRvTWF0Y2gobmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KCdtb3VudEVmcy5zaCAke1Rva2VuW1RPS0VOLlxcXFxkK119IC9tbnQvcmVwb3NpdG9yeSBmYWxzZSBydycpKSk7XG5cbiAgLy8gTWFrZSBzdXJlIHdlIGFkZGVkIHRoZSBEQiBjb25uZWN0aW9uIGFyZ3NcbiAgZXhwZWN0KHVzZXJEYXRhKS50b01hdGNoKC8uKmV4cG9ydCAtZiBjb25maWd1cmVfZGVhZGxpbmVfZGF0YWJhc2UuKi8pO1xuXG4gIC8vIE1ha2Ugc3VyZSB3ZSBjYWxsIHRoZSBjb25maWd1cmVSZXBvc2l0b3J5RGlyZWN0Q29ubmVjdCBzY3JpcHQgd2l0aCBhcHByb3ByaWF0ZSBhcmd1bWVudC5cbiAgY29uc3QgcmVnZXggPSBuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ1xcJy90bXAvJHtUb2tlbltUT0tFTi5cXFxcZCtdfSR7VG9rZW5bVE9LRU4uXFxcXGQrXX1cXCcgXFxcXFwiL21udC9yZXBvc2l0b3J5L0RlYWRsaW5lUmVwb3NpdG9yeVxcXFxcIicpKTtcbiAgZXhwZWN0KHVzZXJEYXRhKS50b01hdGNoKHJlZ2V4KTtcblxuICAvLyBBc3NlcnQgdGhlIElBTSBpbnN0YW5jZSBwcm9maWxlIGlzIGdpdmVuIHJlYWQgYWNjZXNzIHRvIHRoZSBkYXRhYmFzZSBjcmVkZW50aWFscyBzZWNyZXRcbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aCh7XG4gICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgICAgXSxcbiAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICBSZXNvdXJjZTogc3RhY2sucmVzb2x2ZShkYi5zZWNyZXQhLnNlY3JldEFybiksXG4gICAgICB9KSxcbiAgICB9LFxuICAgIFJvbGVzOiBbXG4gICAgICBzdGFjay5yZXNvbHZlKGluc3RhbmNlUm9sZS5yZWYpLFxuICAgIF0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdjb25maWd1cmVDbGllbnRJbnN0YW5jZSB1c2VzIHNpbmdsZXRvbiBmb3IgcmVwbyBjb25maWcgc2NyaXB0JywgKCkgPT4ge1xuICAvLyBOb3RlOiBJZiB0aGlzIHRlc3QgZmFpbHMsIHRoZW4gY2hlY2sgdGhlIGFzc2V0IGhhc2ggZm9yIFJFUE9fRENfQVNTRVQgLS0gaXQgbWF5IGhhdmUgY2hhbmdlZC5cblxuICAvLyBHSVZFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uVGltZW91dDogRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgdmVyc2lvbixcbiAgfSk7XG4gIGNvbnN0IGluc3RhbmNlMSA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlMScsIHtcbiAgICB2cGMsXG4gICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICB9KTtcbiAgY29uc3QgaW5zdGFuY2UyID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UyJywge1xuICAgIHZwYyxcbiAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgcmVwby5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgaG9zdDogaW5zdGFuY2UxLFxuICAgIG1vdW50UG9pbnQ6ICcvbW50L3JlcG9zaXRvcnknLFxuICB9KTtcbiAgcmVwby5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgaG9zdDogaW5zdGFuY2UyLFxuICAgIG1vdW50UG9pbnQ6ICcvbW50L3JlcG9zaXRvcnknLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIC8vIE1ha2Ugc3VyZSB0aGF0IGJvdGggaW5zdGFuY2VzIGhhdmUgYWNjZXNzIHRvIHRoZSBzYW1lIEFzc2V0IGZvciB0aGUgY29uZmlndXJlUmVwb3NpdG9yeURpcmVjdENvbm5lY3Qgc2NyaXB0XG4gIGV4cGVjdENESyhzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OklBTTo6UG9saWN5JywgMiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aChcbiAgICAgICAge1xuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICdzMzpMaXN0KicsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBSZXNvdXJjZTogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpzMzo6OicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogUkVQT19EQ19BU1NFVC5CdWNrZXQsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpzMzo6OicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogUkVQT19EQ19BU1NFVC5CdWNrZXQsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJy8qJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgKSxcbiAgICB9LFxuICB9KSk7XG59KTtcblxudGVzdCgnd2luZG93cyBjbGllbnQgY2Fubm90IGRpcmVjdCBjb25uZWN0IHRvIHJlcG9zaXRvcnknLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHJlcG9zaXRvcnlJbnN0YWxsYXRpb25UaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDMwKSxcbiAgICB2ZXJzaW9uLFxuICB9KTtcbiAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICB2cGMsXG4gICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdFdpbmRvd3MoV2luZG93c1ZlcnNpb24uV0lORE9XU19TRVJWRVJfMjAxOV9FTkdMSVNIX0ZVTExfQkFTRSksXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KCgpID0+IHtcbiAgICByZXBvLmNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHtcbiAgICAgIGhvc3Q6IGluc3RhbmNlLFxuICAgICAgbW91bnRQb2ludDogJ2Q6XFxcXCcsXG4gICAgfSk7XG4gIH0pLnRvVGhyb3dFcnJvcignRGVhZGxpbmUgZGlyZWN0IGNvbm5lY3Qgb24gV2luZG93cyBob3N0cyBpcyBub3QgeWV0IHN1cHBvcnRlZCBieSB0aGUgUkZESy4nKTtcbn0pO1xuXG5kZXNjcmliZSgndGFnZ2luZycsICgpID0+IHtcbiAgdGVzdENvbnN0cnVjdFRhZ3Moe1xuICAgIGNvbnN0cnVjdE5hbWU6ICdSZXBvc2l0b3J5JyxcbiAgICBjcmVhdGVDb25zdHJ1Y3Q6ICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcbiAgICAgIG5ldyBSZXBvc2l0b3J5KGlzb2xhdGVkU3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHZlcnNpb24sXG4gICAgICB9KTtcbiAgICAgIHJldHVybiBpc29sYXRlZFN0YWNrO1xuICAgIH0sXG4gICAgcmVzb3VyY2VUeXBlQ291bnRzOiB7XG4gICAgICAnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnOiAzLFxuICAgICAgJ0FXUzo6RG9jREI6OkRCQ2x1c3RlclBhcmFtZXRlckdyb3VwJzogMSxcbiAgICAgICdBV1M6OkRvY0RCOjpEQlN1Ym5ldEdyb3VwJzogMSxcbiAgICAgICdBV1M6OlNlY3JldHNNYW5hZ2VyOjpTZWNyZXQnOiAyLFxuICAgICAgJ0FXUzo6RG9jREI6OkRCQ2x1c3Rlcic6IDEsXG4gICAgICAnQVdTOjpEb2NEQjo6REJJbnN0YW5jZSc6IDEsXG4gICAgICAnQVdTOjpJQU06OlJvbGUnOiAxLFxuICAgICAgJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnOiAxLFxuICAgICAgJ0FXUzo6U1NNOjpQYXJhbWV0ZXInOiAxLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCdTZWN1cml0eSBHcm91cHMnLCAoKSA9PiB7XG4gIGxldCByZXBvc2l0b3J5U2VjdXJpdHlHcm91cDogSVNlY3VyaXR5R3JvdXA7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgcmVwb3NpdG9yeVNlY3VyaXR5R3JvdXAgPSBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgJ0FkZGl0aW9uYWxTZWN1cml0eUdyb3VwJywgeyB2cGMgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdEb2NEQicsICgpID0+IHtcblxuICAgIHRlc3QoJ2FkZHMgc2VjdXJpdHkgZ3JvdXBzIG9uIGNvbnN0cnVjdGlvbicsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgICAgdmVyc2lvbixcbiAgICAgICAgdnBjLFxuICAgICAgICBzZWN1cml0eUdyb3Vwc09wdGlvbnM6IHtcbiAgICAgICAgICBkYXRhYmFzZTogcmVwb3NpdG9yeVNlY3VyaXR5R3JvdXAsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInLCB7XG4gICAgICAgIFZwY1NlY3VyaXR5R3JvdXBJZHM6IGFycmF5V2l0aChzdGFjay5yZXNvbHZlKHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCkpLFxuICAgICAgfSkpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnRUZTJywgKCkgPT4ge1xuXG4gICAgdGVzdCgnYWRkcyBzZWN1cml0eSBncm91cHMgb24gY29uc3RydWN0aW9uJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgICAgICB2ZXJzaW9uLFxuICAgICAgICB2cGMsXG4gICAgICAgIHNlY3VyaXR5R3JvdXBzT3B0aW9uczoge1xuICAgICAgICAgIGZpbGVTeXN0ZW06IHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIC8vIFRoZSBFRlMgY29uc3RydWN0IGFkZHMgdGhlIHNlY3VyaXR5IGdyb3VwIHRvIGVhY2ggbW91bnQgdGFyZ2V0LCBhbmQgb25lIG1vdW50IHRhcmdldCBpcyBnZW5lcmF0ZWQgcGVyIHN1Ym5ldC5cbiAgICAgIGNvbnN0IG51bU1vdW50VGFyZ2V0cyA9IHZwYy5zZWxlY3RTdWJuZXRzKCkuc3VibmV0cy5sZW5ndGg7XG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzTGlrZSgnQVdTOjpFRlM6Ok1vdW50VGFyZ2V0JywgbnVtTW91bnRUYXJnZXRzLCB7XG4gICAgICAgIFNlY3VyaXR5R3JvdXBzOiBhcnJheVdpdGgoc3RhY2sucmVzb2x2ZShyZXBvc2l0b3J5U2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ0luc3RhbGxlcicsICgpID0+IHtcblxuICAgIHRlc3QoJ2FkZHMgc2VjdXJpdHkgZ3JvdXBzIG9uIGNvbnN0cnVjdGlvbicsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgICAgdmVyc2lvbixcbiAgICAgICAgdnBjLFxuICAgICAgICBzZWN1cml0eUdyb3Vwc09wdGlvbnM6IHtcbiAgICAgICAgICBpbnN0YWxsZXI6IHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6TGF1bmNoQ29uZmlndXJhdGlvbicsIHtcbiAgICAgICAgU2VjdXJpdHlHcm91cHM6IGFycmF5V2l0aChzdGFjay5yZXNvbHZlKHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCkpLFxuICAgICAgfSkpO1xuICAgIH0pO1xuXG4gIH0pO1xufSk7XG5cbnRlc3QoJ3ZhbGlkYXRlcyBWZXJzaW9uUXVlcnkgaXMgbm90IGluIGEgZGlmZmVyZW50IHN0YWNrJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBuZXdTdGFjayA9IG5ldyBTdGFjayhhcHAsICdOZXdTdGFjaycpO1xuICB2ZXJzaW9uID0gbmV3IFZlcnNpb25RdWVyeShzdGFjaywgJ1ZlcnNpb24nKTtcbiAgbmV3IFJlcG9zaXRvcnkobmV3U3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGZ1bmN0aW9uIHN5bnRoKCkge1xuICAgIFN5bnRoVXRpbHMuc3ludGhlc2l6ZShuZXdTdGFjayk7XG4gIH1cblxuICAvLyBUSEVOXG4gIGV4cGVjdChzeW50aCkudG9UaHJvdygnQSBWZXJzaW9uUXVlcnkgY2FuIG5vdCBiZSBzdXBwbGllZCBmcm9tIGEgZGlmZmVyZW50IHN0YWNrJyk7XG59KTtcblxudGVzdCgnY3JlYXRlcyBhbiBFRlMgQWNjZXNzUG9pbnQgd2hlbiBubyBmaWxlc3lzdGVtIGlzIHN1cHBsaWVkJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgdmVyc2lvbixcbiAgICB2cGMsXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgY29uc3QgZWZzUmVzb3VyY2UgPSAocmVwby5ub2RlLmZpbmRDaGlsZCgnRmlsZVN5c3RlbScpIGFzIENmbkVsZW1lbnQpLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkZpbGVTeXN0ZW07XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVGUzo6QWNjZXNzUG9pbnQnLCB7XG4gICAgRmlsZVN5c3RlbUlkOiBzdGFjay5yZXNvbHZlKGVmc1Jlc291cmNlLnJlZiksXG4gICAgUG9zaXhVc2VyOiB7XG4gICAgICBHaWQ6ICcwJyxcbiAgICAgIFVpZDogJzAnLFxuICAgIH0sXG4gICAgUm9vdERpcmVjdG9yeToge30sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCd0aHJvd3MgYW4gZXJyb3IgaWYgc3VwcGxpZWQgYSBNb3VudGFibGVFZnMgd2l0aCBubyBBY2Nlc3MgUG9pbnQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IG5ld1N0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ05ld1N0YWNrJyk7XG4gIGNvbnN0IGZzID0gbmV3IEVmc0ZpbGVTeXN0ZW0obmV3U3RhY2ssICdGaWxlU3lzdGVtJywge1xuICAgIHZwYyxcbiAgfSk7XG4gIGNvbnN0IG1vdW50YWJsZUZzID0gbmV3IE1vdW50YWJsZUVmcyhuZXdTdGFjaywge1xuICAgIGZpbGVzeXN0ZW06IGZzLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGZ1bmN0aW9uIHdoZW4oKSB7XG4gICAgbmV3IFJlcG9zaXRvcnkobmV3U3RhY2ssICdSZXBvJywge1xuICAgICAgdmVyc2lvbixcbiAgICAgIHZwYyxcbiAgICAgIGZpbGVTeXN0ZW06IG1vdW50YWJsZUZzLFxuICAgIH0pO1xuICB9XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qod2hlbikudG9UaHJvdygnV2hlbiB1c2luZyBFRlMgd2l0aCB0aGUgUmVwb3NpdG9yeSwgeW91IG11c3QgcHJvdmlkZSBhbiBFRlMgQWNjZXNzIFBvaW50Jyk7XG59KTtcblxudGVzdCgnZGlzYWJsZSBTZWNyZXRzIE1hbmFnZW1lbnQgYnkgZGVmYXVsdCB3aGVuIERlYWRsaW5lIHZlcnNpb24gaXMgb2xkJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBuZXdTdGFjayA9IG5ldyBTdGFjayhhcHAsICdOZXdTdGFjaycpO1xuICBjb25zdCBvbGRWZXJzaW9uID0gbmV3IFZlcnNpb25RdWVyeShuZXdTdGFjaywgJ09sZERlYWRsaW5lVmVyc2lvbicsIHsgdmVyc2lvbjogJzEwLjAuMC4wJyB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG9zaXRvcnkgPSBuZXcgUmVwb3NpdG9yeShuZXdTdGFjaywgJ1JlcG8nLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb246IG9sZFZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5lbmFibGVkKS50b0JlRmFsc3koKTtcbiAgZXhwZWN0KHJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5jcmVkZW50aWFscykudG9CZVVuZGVmaW5lZCgpO1xufSk7XG5cbnRlc3QoJ3Rocm93cyB3aGVuIFNlY3JldHMgTWFuYWdlbWVudCBpcyBlbmFibGVkIGJ1dCBkZWFkbGluZSB2ZXJzaW9uIGlzIHRvbyBsb3cnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IG5ld1N0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ05ld1N0YWNrJyk7XG4gIGNvbnN0IG9sZFZlcnNpb24gPSBuZXcgVmVyc2lvblF1ZXJ5KG5ld1N0YWNrLCAnT2xkRGVhZGxpbmVWZXJzaW9uJywgeyB2ZXJzaW9uOiAnMTAuMC4wLjAnIH0pO1xuXG4gIC8vIFdIRU5cbiAgZnVuY3Rpb24gd2hlbigpIHtcbiAgICBuZXcgUmVwb3NpdG9yeShuZXdTdGFjaywgJ1JlcG8nLCB7XG4gICAgICB2ZXJzaW9uOiBvbGRWZXJzaW9uLFxuICAgICAgdnBjLFxuICAgICAgc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5nczoge1xuICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHdoZW4pLnRvVGhyb3coYFRoZSBzdXBwbGllZCBEZWFkbGluZSB2ZXJzaW9uICgke29sZFZlcnNpb24udmVyc2lvblN0cmluZ30pIGRvZXMgbm90IHN1cHBvcnQgRGVhZGxpbmUgU2VjcmV0cyBNYW5hZ2VtZW50IGluIFJGREsuIEVpdGhlciB1cGdyYWRlIERlYWRsaW5lIHRvIHRoZSBtaW5pbXVtIHJlcXVpcmVkIHZlcnNpb24gKCR7VmVyc2lvbi5NSU5JTVVNX1NFQ1JFVFNfTUFOQUdFTUVOVF9WRVJTSU9OLnZlcnNpb25TdHJpbmd9KSBvciBkaXNhYmxlIHRoZSBmZWF0dXJlIGluIHRoZSBSZXBvc2l0b3J5J3MgY29uc3RydWN0IHByb3BlcnRpZXMuYCk7XG59KTtcblxudGVzdCgnaW1wb3J0cyByZXBvc2l0b3J5IHNldHRpbmdzJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCByZXBvc2l0b3J5U2V0dGluZ3MgPSBuZXcgQXNzZXQoc3RhY2ssICdSZXBvc2l0b3J5U2V0dGluZ3NBc3NldCcsIHtcbiAgICBwYXRoOiBfX2ZpbGVuYW1lLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG9zaXRvcnkgPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgcmVwb3NpdG9yeVNldHRpbmdzLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGNvbnN0IGluc3RhbGxlckdyb3VwID0gcmVwb3NpdG9yeS5ub2RlLnRyeUZpbmRDaGlsZCgnSW5zdGFsbGVyJykgYXMgQXV0b1NjYWxpbmdHcm91cDtcbiAgZXhwZWN0KGluc3RhbGxlckdyb3VwLnVzZXJEYXRhLnJlbmRlcigpKS50b0NvbnRhaW4oYGF3cyBzMyBjcCAnJHtyZXBvc2l0b3J5U2V0dGluZ3MuczNPYmplY3RVcmx9J2ApO1xufSk7XG5cbnRlc3QoJ0lNb3VudGFibGVMaW51eEZpbGVzeXN0ZW0udXNlc1VzZXJQb3NpeFBlcm1pc3Npb25zKCkgPSB0cnVlIGNoYW5nZXMgb3duZXJzaGlwIG9mIHJlcG9zaXRvcnkgZmlsZXMnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgdmVyc2lvbixcbiAgICB2cGMsXG4gICAgZmlsZVN5c3RlbToge1xuICAgICAgbW91bnRUb0xpbnV4SW5zdGFuY2U6IChfdGFyZ2V0LCBfbW91bnQpID0+IHt9LFxuICAgICAgdXNlc1VzZXJQb3NpeFBlcm1pc3Npb25zOiAoKSA9PiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3Qgc2NyaXB0ID0gKHJlcG8ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQXV0b1NjYWxpbmdHcm91cCkudXNlckRhdGEucmVuZGVyKCk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc2NyaXB0KS50b01hdGNoKCctbyAxMDAwOjEwMDAnKTtcbn0pO1xuXG50ZXN0KCdJTW91bnRhYmxlTGludXhGaWxlc3lzdGVtLnVzZXNVc2VyUG9zaXhQZXJtaXNzaW9ucygpID0gZmFsc2UgZG9lcyBub3QgY2hhbmdlIG93bmVyc2hpcCBvZiByZXBvc2l0b3J5IGZpbGVzJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgIHZlcnNpb24sXG4gICAgdnBjLFxuICAgIGZpbGVTeXN0ZW06IHtcbiAgICAgIG1vdW50VG9MaW51eEluc3RhbmNlOiAoX3RhcmdldCwgX21vdW50KSA9PiB7fSxcbiAgICAgIHVzZXNVc2VyUG9zaXhQZXJtaXNzaW9uczogKCkgPT4gZmFsc2UsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBjb25zdCBzY3JpcHQgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzY3JpcHQpLm5vdC50b01hdGNoKCctbyAxMDAwOjEwMDAnKTtcbn0pO1xuXG50ZXN0KCdzZWNyZXQgbWFuYWdlciBlbmFibGVkJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBleHBlY3RlZENyZWRlbnRpYWxzID0gbmV3IFNlY3JldChzdGFjaywgJ0N1c3RvbVNNQWRtaW5Vc2VyJywge1xuICAgIGRlc2NyaXB0aW9uOiAnQ3VzdG9tIGFkbWluIGNyZWRlbnRpYWxzIGZvciB0aGUgU2VjcmV0IE1hbmFnZW1lbnQnLFxuICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICBleGNsdWRlQ2hhcmFjdGVyczogJ1xcXCIkJlxcJygpLS88PltcXFxcXVxcYHt8fScsXG4gICAgICBpbmNsdWRlU3BhY2U6IGZhbHNlLFxuICAgICAgcGFzc3dvcmRMZW5ndGg6IDI0LFxuICAgICAgcmVxdWlyZUVhY2hJbmNsdWRlZFR5cGU6IHRydWUsXG4gICAgICBnZW5lcmF0ZVN0cmluZ0tleTogJ3Bhc3N3b3JkJyxcbiAgICAgIHNlY3JldFN0cmluZ1RlbXBsYXRlOiBKU09OLnN0cmluZ2lmeSh7IHVzZXJuYW1lOiAnYWRtaW4nIH0pLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwb3NpdG9yeSA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICBzZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzOiB7XG4gICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgY3JlZGVudGlhbHM6IGV4cGVjdGVkQ3JlZGVudGlhbHMsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmNyZWRlbnRpYWxzKS50b0JlKGV4cGVjdGVkQ3JlZGVudGlhbHMpO1xuICBjb25zdCBpbnN0YWxsZXJHcm91cCA9IHJlcG9zaXRvcnkubm9kZS50cnlGaW5kQ2hpbGQoJ0luc3RhbGxlcicpIGFzIEF1dG9TY2FsaW5nR3JvdXA7XG4gIGV4cGVjdChpbnN0YWxsZXJHcm91cC51c2VyRGF0YS5yZW5kZXIoKSkudG9Db250YWluKGAtciAke3N0YWNrLnJlZ2lvbn0gLWMgJHtleHBlY3RlZENyZWRlbnRpYWxzLnNlY3JldEFybn1gKTtcbn0pO1xuXG50ZXN0KCdzZWNyZXQgbWFuYWdlciBpcyBlbmFibGVkIGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwb3NpdG9yeSA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmVuYWJsZWQpLnRvQmVUcnV0aHkoKTtcbiAgZXhwZWN0KHJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5jcmVkZW50aWFscykudG9CZURlZmluZWQoKTtcbn0pO1xuXG50ZXN0KCdjcmVkZW50aWFscyBhcmUgdW5kZWZpbmVkIHdoZW4gc2VjcmV0cyBtYW5hZ2VtZW50IGlzIGRpc2FibGVkJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG9zaXRvcnkgPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5nczoge1xuICAgICAgZW5hYmxlZDogZmFsc2UsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmNyZWRlbnRpYWxzKS50b0JlVW5kZWZpbmVkKCk7XG59KTtcblxuXG50ZXN0KCd0aHJvd3MgYW4gZXJyb3IgaWYgY3JlZGVudGlhbHMgYXJlIHVuZGVmaW5lZCBhbmQgZGF0YWJhc2UgaXMgaW1wb3J0ZWQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHNnID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRycsIHtcbiAgICB2cGMsXG4gIH0pO1xuICBjb25zdCBzZWNyZXQgPSBuZXcgU2VjcmV0KHN0YWNrLCAnU2VjcmV0Jyk7XG4gIGNvbnN0IGRhdGFiYXNlID0gRGF0YWJhc2VDbHVzdGVyLmZyb21EYXRhYmFzZUNsdXN0ZXJBdHRyaWJ1dGVzKHN0YWNrLCAnRGJDbHVzdGVyJywge1xuICAgIGNsdXN0ZXJFbmRwb2ludEFkZHJlc3M6ICcxLjIuMy40JyxcbiAgICBjbHVzdGVySWRlbnRpZmllcjogJ2ZvbycsXG4gICAgaW5zdGFuY2VFbmRwb2ludEFkZHJlc3NlczogWyAnMS4yLjMuNScgXSxcbiAgICBpbnN0YW5jZUlkZW50aWZpZXJzOiBbICdpMCcgXSxcbiAgICBwb3J0OiAyNzAwMSxcbiAgICByZWFkZXJFbmRwb2ludEFkZHJlc3M6ICcxLjIuMy42JyxcbiAgICBzZWN1cml0eUdyb3VwOiBzZyxcbiAgfSk7XG4gIGNvbnN0IGRhdGFiYXNlQ29ubmVjdGlvbiA9IERhdGFiYXNlQ29ubmVjdGlvbi5mb3JEb2NEQih7ZGF0YWJhc2UsIGxvZ2luOiBzZWNyZXR9KTtcblxuICAvLyBXSEVOXG4gIGZ1bmN0aW9uIHdoZW4oKSB7XG4gICAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgICAgdnBjLFxuICAgICAgdmVyc2lvbixcbiAgICAgIGRhdGFiYXNlOiBkYXRhYmFzZUNvbm5lY3Rpb24sXG4gICAgfSk7XG4gIH1cblxuICAvLyBUSEVOXG4gIGV4cGVjdCh3aGVuKS50b1Rocm93KCdBZG1pbiBjcmVkZW50aWFscyBmb3IgRGVhZGxpbmUgU2VjcmV0cyBNYW5hZ2VtZW50IGNhbm5vdCBiZSBnZW5lcmF0ZWQgd2hlbiB1c2luZyBhbiBpbXBvcnRlZCBkYXRhYmFzZS4gRm9yIHNldHRpbmcgdXAgeW91ciBvd24gY3JlZGVudGlhbHMsIHBsZWFzZSByZWZlciB0byBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1yZmRrL3RyZWUvbWFpbmxpbmUvcGFja2FnZXMvYXdzLXJmZGsvbGliL2RlYWRsaW5lI2NvbmZpZ3VyaW5nLWRlYWRsaW5lLXNlY3JldHMtbWFuYWdlbWVudC1vbi10aGUtcmVwb3NpdG9yeS4nKTtcbn0pO1xuIl19