"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const lib_1 = require("../lib");
const spot_event_plugin_fleet_1 = require("../lib/spot-event-plugin-fleet");
describe('ConfigureSpotEventPlugin', () => {
    let stack;
    let vpc;
    let region;
    let renderQueue;
    let version;
    let app;
    let fleet;
    let groupName;
    const workerMachineImage = new aws_ec2_1.GenericWindowsImage({
        'us-east-1': 'ami-any',
    });
    beforeEach(() => {
        region = 'us-east-1';
        app = new core_1.App();
        stack = new core_1.Stack(app, 'stack', {
            env: {
                region,
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        groupName = 'group_name1';
        fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
            vpc,
            renderQueue: renderQueue,
            deadlineGroups: [
                groupName,
            ],
            instanceTypes: [
                aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
            ],
            workerMachineImage,
            maxCapacity: 1,
        });
    });
    describe('creates a custom resource', () => {
        test('with default spot event plugin properties', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'Disabled',
                    DeleteInterruptedSlaves: false,
                    DeleteTerminatedSlaves: false,
                    IdleShutdown: 10,
                    Logging: 'Standard',
                    PreJobTaskMode: 'Conservative',
                    Region: core_1.Stack.of(renderQueue).region,
                    ResourceTracker: true,
                    StaggerInstances: 50,
                    State: 'Global Enabled',
                    StrictHardCap: false,
                }),
            })));
        });
        test('with custom spot event plugin properties', () => {
            // GIVEN
            const configuration = {
                awsInstanceStatus: lib_1.SpotEventPluginDisplayInstanceStatus.EXTRA_INFO_0,
                deleteEC2SpotInterruptedWorkers: true,
                deleteSEPTerminatedWorkers: true,
                idleShutdown: core_1.Duration.minutes(20),
                loggingLevel: lib_1.SpotEventPluginLoggingLevel.VERBOSE,
                preJobTaskMode: lib_1.SpotEventPluginPreJobTaskMode.NORMAL,
                region: 'us-west-2',
                enableResourceTracker: false,
                maximumInstancesStartedPerCycle: 10,
                state: lib_1.SpotEventPluginState.DISABLED,
                strictHardCap: true,
            };
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'ExtraInfo0',
                    DeleteInterruptedSlaves: true,
                    DeleteTerminatedSlaves: true,
                    IdleShutdown: 20,
                    Logging: 'Verbose',
                    PreJobTaskMode: 'Normal',
                    Region: 'us-west-2',
                    ResourceTracker: false,
                    StaggerInstances: 10,
                    State: 'Disabled',
                    StrictHardCap: true,
                }),
            })));
        });
        test('without spot fleets', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: assert_1.ABSENT,
            }));
        });
        test('provides RQ connection parameters to custom resource', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueue.endpoint.hostname),
                    port: stack.resolve(renderQueue.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueue.endpoint.applicationProtocol.toString()),
                }),
            })));
        });
        test('with default spot fleet request configuration', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: {
                    [groupName]: {
                        AllocationStrategy: 'lowestPrice',
                        IamFleetRole: stack.resolve(fleet.fleetRole.roleArn),
                        LaunchTemplateConfigs: [
                            {
                                LaunchTemplateSpecification: {
                                    Version: '$Latest',
                                    LaunchTemplateId: stack.resolve(fleet.launchTemplate.launchTemplateId),
                                },
                            },
                        ],
                        TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            ResourceType: 'spot-fleet-request',
                            Tags: assert_1.arrayWith({
                                Key: rfdkTag.name,
                                Value: rfdkTag.value,
                            }),
                        })),
                    },
                },
            }));
        });
        test('adds policies to the render queue', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('adds resource tracker policy even if rt disabled', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration: {
                    enableResourceTracker: false,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
        });
        test.each([
            undefined,
            [],
        ])('without spot fleet', (noFleets) => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: noFleets,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.ABSENT,
            })));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('fleet with validUntil', () => {
            // GIVEN
            const validUntil = core_1.Expiration.atDate(new Date(2022, 11, 17));
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                validUntil,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        ValidUntil: validUntil.date.toISOString(),
                    }),
                }),
            })));
        });
    });
    test('only one object allowed per render queue', () => {
        // GIVEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/Only one ConfigureSpotEventPlugin construct is allowed per render queue./);
    });
    test('can create multiple objects with different render queues', () => {
        // GIVEN
        const renderQueue2 = new lib_1.RenderQueue(stack, 'RQ2', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository2', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
            vpc,
            renderQueue: renderQueue2,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('Custom::RFDK_ConfigureSpotEventPlugin', 2));
    });
    test('throws with not supported render queue', () => {
        // GIVEN
        const invalidRenderQueue = {};
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: invalidRenderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/The provided render queue is not an instance of RenderQueue class. Some functionality is not supported./);
    });
    test('tagSpecifications returns undefined if fleet does not have tags', () => {
        // GIVEN
        const mockFleet = {
            tags: {
                hasTags: jest.fn().mockReturnValue(false),
            },
        };
        const mockedFleet = mockFleet;
        const config = new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = stack.resolve(config['tagSpecifications'](mockedFleet, lib_1.SpotFleetResourceType.INSTANCE));
        // THEN
        expect(result).toBeUndefined();
    });
    describe('with TLS', () => {
        let renderQueueWithTls;
        let caCert;
        beforeEach(() => {
            const host = 'renderqueue';
            const zoneName = 'deadline-test.internal';
            caCert = new core_2.X509CertificatePem(stack, 'RootCA', {
                subject: {
                    cn: 'SampleRootCA',
                },
            });
            renderQueueWithTls = new lib_1.RenderQueue(stack, 'RQ with TLS', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository2', {
                    vpc,
                    version,
                }),
                version,
                hostname: {
                    zone: new aws_route53_1.PrivateHostedZone(stack, 'DnsZone', {
                        vpc,
                        zoneName: zoneName,
                    }),
                    hostname: host,
                },
                trafficEncryption: {
                    externalTLS: {
                        rfdkCertificate: new core_2.X509CertificatePem(stack, 'RQCert', {
                            subject: {
                                cn: `${host}.${zoneName}`,
                            },
                            signingCertificate: caCert,
                        }),
                    },
                },
            });
        });
        test('Lambda role can get the ca secret', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(renderQueueWithTls.certChain.secretArn),
                        },
                    ],
                },
                Roles: [
                    {
                        Ref: 'ConfigureSpotEventPluginConfiguratorServiceRole341B4735',
                    },
                ],
            }));
        });
        test('creates a custom resource with connection', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueueWithTls.endpoint.hostname),
                    port: stack.resolve(renderQueueWithTls.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueueWithTls.endpoint.applicationProtocol.toString()),
                    caCertificateArn: stack.resolve(renderQueueWithTls.certChain.secretArn),
                }),
            })));
        });
    });
    test('throws with the same group name', () => {
        // WHEN
        function createConfigureSpotEventPlugin() {
            const duplicateFleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'DuplicateSpotFleet', {
                vpc,
                renderQueue,
                workerMachineImage: fleet.machineImage,
                instanceTypes: fleet.instanceTypes,
                maxCapacity: fleet.maxCapacity,
                deadlineGroups: fleet.deadlineGroups,
            });
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                    duplicateFleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(`Bad Group Name: ${groupName}. Group names in Spot Fleet Request Configurations should be unique.`);
    });
    test('uses selected subnets', () => {
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'configure-spot-event-plugin.configureSEP',
            VpcConfig: {
                SubnetIds: [
                    stack.resolve(vpc.privateSubnets[0].subnetId),
                ],
            },
        }));
    });
    describe('throws with wrong deadline version', () => {
        test.each([
            ['10.1.9'],
            ['10.1.10'],
        ])('%s', (versionString) => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            version = new lib_1.VersionQuery(newStack, 'OldVersion', {
                version: versionString,
            });
            renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(newStack, 'Repository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                }),
                trafficEncryption: { externalTLS: { enabled: false } },
                version,
            });
            // WHEN
            function createConfigureSpotEventPlugin() {
                new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                    vpc,
                    renderQueue: renderQueue,
                    spotFleets: [
                        fleet,
                    ],
                });
            }
            // THEN
            expect(createConfigureSpotEventPlugin).toThrowError(`Minimum supported Deadline version for ConfigureSpotEventPlugin is 10.1.12.0. Received: ${versionString}.`);
        });
    });
    test('does not throw with min deadline version', () => {
        // GIVEN
        const versionString = '10.1.12';
        const newStack = new core_1.Stack(app, 'NewStack');
        version = new lib_1.VersionQuery(newStack, 'OldVersion', {
            version: versionString,
        });
        renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(newStack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(newStack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).not.toThrow();
    });
    describe('secrets management enabled', () => {
        beforeEach(() => {
            region = 'us-east-1';
            app = new core_1.App();
            stack = new core_1.Stack(app, 'stack', {
                env: {
                    region,
                },
            });
            vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
            version = new lib_1.VersionQuery(stack, 'Version');
            renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository', {
                    vpc,
                    version,
                }),
                version,
            });
            groupName = 'group_name1';
        });
        test('a fleet without vpcSubnets specified => warns about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
            }));
        });
        test('a fleet with vpcSubnets specified => does not warn about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleetWithSubnets', {
                vpc,
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
                },
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).not.toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(/dedicated subnet/i),
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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