"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 *
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 *
 * @stability stable
 */
class UsageBasedLicense {
    /**
     * @stability stable
     */
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4151), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5555), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
_a = JSII_RTTI_SYMBOL_1;
UsageBasedLicense[_a] = { fqn: "aws-rfdk.deadline.UsageBasedLicense", version: "0.41.0" };
/**
 * Constant used to signify unlimited overage.
 *
 * @stability stable
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL) (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html ) in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, only allows ingress from the
 * Render Queue (in order to register Workers for license forwarding).
 *
 * When a Deadline Worker requires access to licensing via `UsageBasedLicensing.grantPortAccess(...)`, then the RFDK
 * constructs will grant that worker’s security group ingress on TCP port 17004 as well as other ports as required by
 * the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * ![architecture diagram](/diagrams/deadline/UsageBasedLicensing.svg)
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *    License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *    Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *    Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 *
 * @stability stable
 */
class UsageBasedLicensing extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        if (!props.vpcSubnets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            core_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
        }
        const vpcSubnets = (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT };
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets,
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: (_d = props.desiredCount) !== null && _d !== void 0 ? _d : 1,
            maxCapacity: (_e = props.desiredCount) !== null && _e !== void 0 ? _e : 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroup,
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = (_g = (_f = props.logGroupProps) === null || _f === void 0 ? void 0 : _f.logGroupPrefix) !== null && _g !== void 0 ? _g : UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: (_h = props.desiredCount) !== null && _h !== void 0 ? _h : 1,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.renderQueue.configureSecretsManagementAutoRegistration({
                dependent: this.service.node.defaultChild,
                registrationStatus: _1.SecretsManagementRegistrationStatus.REGISTERED,
                role: _1.SecretsManagementRole.CLIENT,
                vpc: props.vpc,
                vpcSubnets,
            });
        }
        // Grant the render queue the ability to connect to the license forwarder to register workers
        this.asg.connections.allowFrom(props.renderQueue.backendConnections, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required.
     *
     * @param workerFleet - worker fleet.
     * @param licenses - UBL licenses.
     * @stability stable
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                workerFleet.connections.allowTo(this, port);
            });
        });
        workerFleet.connections.allowTo(this, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     *
     * @stability stable
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
_b = JSII_RTTI_SYMBOL_1;
UsageBasedLicensing[_b] = { fqn: "aws-rfdk.deadline.UsageBasedLicensing", version: "0.41.0" };
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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