"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const _1 = require("./");
/**
 * Class for interacting with the Deadline stage directory.
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 *
 * @stability stable
 */
class Stage {
    /**
     * Constructs a Stage instance.
     *
     * @stability stable
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path.
     *
     * @param manifestPath The path to the manifest JSON file.
     * @stability stable
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory.
     * @stability stable
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        /* istanbul ignore else */
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        // Do minimum supported deadline version check
        const stagedVersion = _1.Version.parse(version);
        if (stagedVersion.isLessThan(_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            throw new TypeError(`Staged Deadline Version (${version}) is less than the minimum supported version (${_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()})`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @stability stable
     */
    getVersion(scope, id) {
        const releaseVersion = this.getReleaseVersion(this.manifest.version);
        return new _1.VersionQuery(scope, id, { version: releaseVersion });
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage.
     *
     * @param scope The scope for the {@link DockerImageAsset}.
     * @param id The construct ID of the {@link DockerImageAsset}.
     * @param recipeName The name of the recipe.
     * @stability stable
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
    /**
     * This removes the patch version from a full version string. No validation is done as that is handled
     * in the constructor with the version check.
     */
    getReleaseVersion(fullVersion) {
        const versionComponents = fullVersion.split('.');
        return `${versionComponents[0]}.${versionComponents[1]}.${versionComponents[2]}`;
    }
    /**
     * @stability stable
     */
    get clientInstallerPath() {
        const INSTALLER_FILENAME_RE = /^DeadlineClient-(?<version>.+)-linux-x64-installer\.run$/;
        const listing = fs.readdirSync(path.join(this.dirPath, 'bin')).filter(filename => INSTALLER_FILENAME_RE.test(filename));
        /* istanbul ignore else */
        if (listing.length === 1) {
            const filename = listing[0];
            const match = INSTALLER_FILENAME_RE.exec(filename);
            const version = match.groups.version;
            const recipes = Array.from(Object.values(this.manifest.recipes));
            const aRecipeHasMatchingDlVersion = recipes.some((recipe) => {
                var _b;
                return ((_b = recipe.buildArgs) === null || _b === void 0 ? void 0 : _b.DL_VERSION) === version;
            });
            const installerPath = path.join(this.dirPath, 'bin', listing[0]);
            if (!aRecipeHasMatchingDlVersion) {
                throw new Error(`No stage recipes refer to the Deadline Client installer found (${installerPath})`);
            }
            return installerPath;
        }
        else if (listing.length === 0) {
            throw new Error(`No matching Client installer found in "${this.dirPath}"`);
        }
        else {
            throw new Error(`Multiple Client installers found: ${listing.join(',')}`);
        }
    }
}
exports.Stage = Stage;
_a = JSII_RTTI_SYMBOL_1;
Stage[_a] = { fqn: "aws-rfdk.deadline.Stage", version: "0.41.0" };
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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