"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 *
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 * - A security Group if security groups are not provided.
 * - An EC2 Launch Template for the Spot Fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class SpotEventPluginFleet extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.defaultSubnets = !props.vpcSubnets;
        this.deadlineGroups = props.deadlineGroups.map(group => group.toLocaleLowerCase());
        this.deadlinePools = (_b = props.deadlinePools) === null || _b === void 0 ? void 0 : _b.map(pool => pool.toLocaleLowerCase());
        this.validateProps(props);
        this.securityGroups = (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = (_d = props.fleetInstanceRole) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = (_e = props.fleetRole) !== null && _e !== void 0 ? _e : new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets((_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = (_g = props.allocationStrategy) !== null && _g !== void 0 ? _g : spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = (_h = props.userData) !== null && _h !== void 0 ? _h : imageConfig.userData;
        this.machineImage = props.workerMachineImage;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: this.deadlineGroups,
                pools: this.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        this.launchTemplate = this.createLaunchTemplate((_j = props.trackInstancesWithResourceTracker) !== null && _j !== void 0 ? _j : true);
        this._launchTemplateConfigs = this.createLaunchTemplateConfigs();
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowRemoteControlTo().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlFrom(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowRemoteControlFrom().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlTo(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    createLaunchTemplate(resourceTrackerEnabled) {
        const launchTemplate = new aws_ec2_1.LaunchTemplate(this, 'LaunchTemplate', {
            blockDevices: this.blockDevices,
            role: this.fleetInstanceRole,
            machineImage: this.machineImage,
            keyName: this.keyName,
            securityGroup: this.securityGroups[0],
            userData: this.userData,
        });
        if (this.securityGroups.length > 1) {
            launchTemplate.connections.addSecurityGroup(...this.securityGroups.slice(1));
        }
        core_1.Tags.of(launchTemplate).add(resourceTrackerEnabled ? 'DeadlineTrackedAWSResource' : 'DeadlineResourceTracker', 'SpotEventPlugin');
        return launchTemplate;
    }
    createLaunchTemplateConfigs() {
        const launchTemplateConfigs = [];
        // Create a launch template config for each instance type + subnet pair
        this.instanceTypes.forEach(instanceType => {
            this.subnets.subnetIds.forEach(subnetId => {
                launchTemplateConfigs.push({
                    LaunchTemplateSpecification: {
                        Version: aws_ec2_1.LaunchTemplateSpecialVersions.LATEST_VERSION,
                        LaunchTemplateId: this.launchTemplate.launchTemplateId,
                        LaunchTemplateName: this.launchTemplate.launchTemplateName,
                    },
                    Overrides: [{
                            InstanceType: instanceType.toString(),
                            SubnetId: subnetId,
                        }],
                });
            });
        });
        return launchTemplateConfigs;
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', this.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (core_1.Stack.of(fleetInstanceRole) !== core_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-z0-9-_]+$/g;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    core_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "0.41.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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