"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_1 = require("./version");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *    access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *    Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *    Repository construct instead.
 *
 * @stability stable
 */
class Repository extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        if (props.database && ((_b = props.backupOptions) === null || _b === void 0 ? void 0 : _b.databaseRetention)) {
            core_1.Annotations.of(this).addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_d = props.removalPolicy) === null || _d === void 0 ? void 0 : _d.database)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem instanceof core_2.MountableEfs && !props.fileSystem.accessPoint) {
            throw new Error('When using EFS with the Repository, you must provide an EFS Access Point');
        }
        if (((_f = (_e = props.secretsManagementSettings) === null || _e === void 0 ? void 0 : _e.enabled) !== null && _f !== void 0 ? _f : true) && props.database && !props.database.databaseConstruct) {
            throw new Error('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
        }
        this.version = props.version;
        const meetsMinSecretsVersion = !this.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION);
        const secretsManagementIsEnabled = (_h = (_g = props.secretsManagementSettings) === null || _g === void 0 ? void 0 : _g.enabled) !== null && _h !== void 0 ? _h : meetsMinSecretsVersion;
        if (secretsManagementIsEnabled && !meetsMinSecretsVersion) {
            throw new Error(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        }
        this.secretsManagementSettings = {
            enabled: secretsManagementIsEnabled,
            credentials: (_k = (_j = props.secretsManagementSettings) === null || _j === void 0 ? void 0 : _j.credentials) !== null && _k !== void 0 ? _k : (secretsManagementIsEnabled ? new aws_secretsmanager_1.Secret(((_l = props.database) === null || _l === void 0 ? void 0 : _l.databaseConstruct) ? core_1.Stack.of((_m = props.database) === null || _m === void 0 ? void 0 : _m.databaseConstruct) : this, 'SMAdminUser', {
                description: 'Admin credentials for Deadline Secrets Management',
                generateSecretString: {
                    excludeCharacters: '\"$&\'()/<>[\\]\`{|}',
                    includeSpace: false,
                    passwordLength: 24,
                    requireEachIncludedType: true,
                    generateStringKey: 'password',
                    secretStringTemplate: JSON.stringify({ username: Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME }),
                },
                removalPolicy: (_p = (_o = props.secretsManagementSettings) === null || _o === void 0 ? void 0 : _o.credentialsRemovalPolicy) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            }) : undefined),
        };
        this.fileSystem = (_q = props.fileSystem) !== null && _q !== void 0 ? _q : (() => {
            var _b, _c, _d, _e, _f;
            const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_d = (_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.RETAIN,
                securityGroup: (_e = props.securityGroupsOptions) === null || _e === void 0 ? void 0 : _e.fileSystem,
            });
            const paddingAccess = fs.addAccessPoint('PaddingAccessPoint', {
                createAcl: {
                    ownerGid: '0',
                    ownerUid: '0',
                    permissions: '744',
                },
                path: '/RFDK_PaddingFiles',
            });
            new core_2.PadEfsStorage(this, 'PadEfsStorage', {
                vpc: props.vpc,
                vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
                accessPoint: paddingAccess,
                desiredPadding: core_1.Size.gibibytes(40),
            });
            const accessPoint = fs.addAccessPoint('AccessPoint', {
                posixUser: {
                    uid: '0',
                    gid: '0',
                },
            });
            return new core_2.MountableEfs(this, {
                filesystem: fs,
                accessPoint,
            });
        })();
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                core_1.Annotations.of(this).addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_r = props.databaseAuditLogging) !== null && _r !== void 0 ? _r : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_s = props.documentDbInstanceCount) !== null && _s !== void 0 ? _s : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                vpc: props.vpc,
                vpcSubnets: (_t = props.vpcSubnets) !== null && _t !== void 0 ? _t : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT, onePerAz: true },
                securityGroup: (_u = props.securityGroupsOptions) === null || _u === void 0 ? void 0 : _u.database,
                instances,
                backup: {
                    retention: (_w = (_v = props.backupOptions) === null || _v === void 0 ? void 0 : _v.databaseRetention) !== null && _w !== void 0 ? _w : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_y = (_x = props.removalPolicy) === null || _x === void 0 ? void 0 : _x.database) !== null && _y !== void 0 ? _y : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_z = props.vpcSubnets) !== null && _z !== void 0 ? _z : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            minCapacity: 1,
            maxCapacity: 1,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            }),
            securityGroup: (_0 = props.securityGroupsOptions) === null || _0 === void 0 ? void 0 : _0.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version, props.repositorySettings, 
        // Change ownership of the Deadline repository files if-and-only-if the mounted file-system
        // uses the POSIX permissions based on the process' user UID/GID
        this.fileSystem.usesUserPosixPermissions() ? Repository.REPOSITORY_OWNER : undefined);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof version_query_1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _b, _c;
        const hostMountPoint = (_b = props.containerInstances.filesystemMountPoint) !== null && _b !== void 0 ? _b : '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = (_c = props.containers.filesystemMountPoint) !== null && _c !== void 0 ? _c : `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this.installerGroup);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _b;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this.installerGroup);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _b;
        const prefix = (_b = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _b !== void 0 ? _b : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.installerGroup).add(tagKey, core_1.Names.uniqueId(this));
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)');
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version, settings, owner) {
        var _b, _c;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        const installerArgs = [
            '-i', `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
            '-p', `"${installPath}"`,
            '-v', version.linuxFullVersionString(),
        ];
        if (this.secretsManagementSettings.enabled) {
            installerArgs.push('-r', core_1.Stack.of((_b = this.secretsManagementSettings.credentials) !== null && _b !== void 0 ? _b : this).region);
            this.secretsManagementSettings.credentials.grantRead(installerGroup);
            installerArgs.push('-c', (_c = this.secretsManagementSettings.credentials.secretArn) !== null && _c !== void 0 ? _c : '');
        }
        if (settings) {
            const repositorySettingsFilePath = installerGroup.userData.addS3DownloadCommand({
                bucket: settings.bucket,
                bucketKey: settings.s3ObjectKey,
            });
            installerArgs.push('-s', repositorySettingsFilePath);
        }
        // We can ignore this in test coverage because we always use Repository.REPOSITORY_OWNER
        /* istanbul ignore next */
        if (owner) {
            installerArgs.push('-o', `${owner.uid}:${owner.gid}`);
        }
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: installerArgs,
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "0.41.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
/**
 * The Repository owner is 1000:1000 (ec2-user on AL2).
 */
Repository.REPOSITORY_OWNER = { uid: 1000, gid: 1000 };
/**
 * Default username for auto generated admin credentials in Secret Manager.
 */
Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME = 'admin';
//# sourceMappingURL=data:application/json;base64,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