"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const secrets_management_1 = require("./secrets-management");
const version_1 = require("./version");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * ![architecture diagram](/diagrams/deadline/RenderQueue.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments and it is configured to be on by default.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        this.props = props;
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        this.repository = props.repository;
        this.renderQueueSize = (_b = props === null || props === void 0 ? void 0 : props.renderQueueSize) !== null && _b !== void 0 ? _b : { min: 1, max: 1 };
        if (props.version.isLessThan(RenderQueue.MINIMUM_LOAD_BALANCING_VERSION)) {
            // Deadline versions earlier than 10.1.10 do not support horizontal scaling behind a load-balancer, so we limit to at most one instance
            if (((_c = this.renderQueueSize.min) !== null && _c !== void 0 ? _c : 0) > 1) {
                throw new Error(`renderQueueSize.min for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.min}`);
            }
            if (((_d = this.renderQueueSize.desired) !== null && _d !== void 0 ? _d : 0) > 1) {
                throw new Error(`renderQueueSize.desired for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.desired}`);
            }
            if (((_e = this.renderQueueSize.max) !== null && _e !== void 0 ? _e : 0) > 1) {
                throw new Error(`renderQueueSize.max for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.max}`);
            }
        }
        this.version = props === null || props === void 0 ? void 0 : props.version;
        const externalProtocol = ((_g = (_f = props.trafficEncryption) === null || _f === void 0 ? void 0 : _f.externalTLS) === null || _g === void 0 ? void 0 : _g.enabled) === false ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        let loadBalancerFQDN;
        let domainZone;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            const tlsInfo = this.getOrCreateTlsInfo(props);
            this.certChain = tlsInfo.certChain;
            this.clientCert = tlsInfo.serverCert;
            loadBalancerFQDN = tlsInfo.fullyQualifiedDomainName;
            domainZone = tlsInfo.domainZone;
        }
        else {
            if (props.hostname) {
                loadBalancerFQDN = this.generateFullyQualifiedDomainName(props.hostname.zone, props.hostname.hostname);
                domainZone = props.hostname.zone;
            }
        }
        this.version = props.version;
        const internalProtocol = (_j = (_h = props.trafficEncryption) === null || _h === void 0 ? void 0 : _h.internalProtocol) !== null && _j !== void 0 ? _j : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_l = (_k = props.renderQueueSize) === null || _k === void 0 ? void 0 : _k.min) !== null && _l !== void 0 ? _l : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        const maxCapacity = (_m = this.renderQueueSize.max) !== null && _m !== void 0 ? _m : (_o = this.renderQueueSize) === null || _o === void 0 ? void 0 : _o.desired;
        if (((_p = this.renderQueueSize) === null || _p === void 0 ? void 0 : _p.desired) && maxCapacity && ((_q = this.renderQueueSize) === null || _q === void 0 ? void 0 : _q.desired) > maxCapacity) {
            throw new Error(`renderQueueSize.desired capacity cannot be more than ${maxCapacity}: got ${this.renderQueueSize.desired}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_r = props.vpcSubnets) !== null && _r !== void 0 ? _r : RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            instanceType: (_s = props.instanceType) !== null && _s !== void 0 ? _s : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_t = this.renderQueueSize) === null || _t === void 0 ? void 0 : _t.desired,
            maxCapacity,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: undefined,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: (_u = props.securityGroups) === null || _u === void 0 ? void 0 : _u.backend,
        });
        this.backendConnections = this.asg.connections;
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if ((_v = props.enableLocalFileCaching) !== null && _v !== void 0 ? _v : false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        if (props.repository.secretsManagementSettings.enabled) {
            const errors = [];
            if (props.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION)) {
                errors.push(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
            }
            if (props.repository.secretsManagementSettings.credentials === undefined) {
                errors.push('The Repository does not have Secrets Management credentials');
            }
            if (internalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('The internal protocol on the Render Queue is not HTTPS.');
            }
            if (externalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('External TLS on the Render Queue is not enabled.');
            }
            if (errors.length > 0) {
                throw new Error(`Deadline Secrets Management is enabled on the supplied Repository but cannot be enabled on the Render Queue for the following reasons:\n${errors.join('\n')}`);
            }
        }
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
            runAsUser: RenderQueue.RCS_USER,
            secretsManagementOptions: props.repository.secretsManagementSettings.enabled ? {
                credentials: props.repository.secretsManagementSettings.credentials,
                posixUsername: RenderQueue.RCS_USER.username,
            } : undefined,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_w = props.vpcSubnetsAlb) !== null && _w !== void 0 ? _w : RenderQueue.DEFAULT_VPC_SUBNETS_ALB,
            internetFacing: false,
            deletionProtection: (_x = props.deletionProtection) !== null && _x !== void 0 ? _x : true,
            securityGroup: (_y = props.securityGroups) === null || _y === void 0 ? void 0 : _y.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_z = this.renderQueueSize) === null || _z === void 0 ? void 0 : _z.desired,
            domainZone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN !== null && loadBalancerFQDN !== void 0 ? loadBalancerFQDN : this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        (_0 = props.repository.secretsManagementSettings.credentials) === null || _0 === void 0 ? void 0 : _0.grantRead(this);
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Configure a rule to automatically register all Deadline Secrets Management identities connecting from a given subnet to a specified role and status.
     *
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/secrets-management/deadline-secrets-management.html#identity-management-registration-settings-ref-label
     * for details.
     *
     * All RFDK constructs that require Deadline Secrets Management identity registration call this method internally.
     * End-users of RFDK should not need to use this method unless they have a special need and understand its inner
     * workings.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureSecretsManagementAutoRegistration(props) {
        if (!this.repository.secretsManagementSettings.enabled) {
            // Secrets management is not enabled, so do nothing
            return;
        }
        this.identityRegistrationSettings.addSubnetIdentityRegistrationSetting(props);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_2.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        if (props.secretsManagementOptions !== undefined) {
            environment.RCS_SM_CREDENTIALS_URI = props.secretsManagementOptions.credentials.secretArn;
        }
        // We can ignore this in test coverage because we always use RenderQueue.RCS_USER
        /* istanbul ignore next */
        const user = props.runAsUser ? `${props.runAsUser.uid}:${props.runAsUser.gid}` : undefined;
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
            user,
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        if (props.secretsManagementOptions !== undefined) {
            // Create volume to persist the RSA keypairs generated by Deadline between ECS tasks
            // This makes it so subsequent ECS tasks use the same initial Secrets Management identity
            const volumeName = 'deadline-user-keypairs';
            taskDefinition.addVolume({
                name: volumeName,
                dockerVolumeConfiguration: {
                    scope: aws_ecs_1.Scope.SHARED,
                    autoprovision: true,
                    driver: 'local',
                },
            });
            // Mount the volume into the container at the location where Deadline expects it
            containerDefinition.addMountPoints({
                readOnly: false,
                sourceVolume: volumeName,
                containerPath: `/home/${props.secretsManagementOptions.posixUsername}/.config/.mono/keypairs`,
            });
        }
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
    /**
     * Checks if the user supplied any certificate to use for TLS and uses them, or creates defaults to use.
     * @param props
     * @returns TlsInfo either based on input to the render queue, or the created defaults
     */
    getOrCreateTlsInfo(props) {
        var _b, _c, _d, _e;
        if ((((_c = (_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.externalTLS) === null || _c === void 0 ? void 0 : _c.acmCertificate) !== undefined) ||
            (((_e = (_d = props.trafficEncryption) === null || _d === void 0 ? void 0 : _d.externalTLS) === null || _e === void 0 ? void 0 : _e.rfdkCertificate) !== undefined)) {
            if (props.hostname === undefined) {
                throw new Error('The hostname for the render queue must be defined if supplying your own certificates.');
            }
            return this.getTlsInfoFromUserProps(props.trafficEncryption.externalTLS, props.hostname);
        }
        return this.createDefaultTlsInfo(props.vpc, props.hostname);
    }
    /**
     * Creates a default certificate to use for TLS and a PrivateHostedZone to put the load balancer in.
     * @param vpc
     * @param hostname
     * @returns default TlsInfo
     */
    createDefaultTlsInfo(vpc, hostname) {
        var _b;
        const domainZone = (_b = hostname === null || hostname === void 0 ? void 0 : hostname.zone) !== null && _b !== void 0 ? _b : new aws_route53_1.PrivateHostedZone(this, 'DnsZone', {
            vpc: vpc,
            zoneName: RenderQueue.DEFAULT_DOMAIN_NAME,
        });
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(domainZone, hostname === null || hostname === void 0 ? void 0 : hostname.hostname);
        const rootCa = new core_2.X509CertificatePem(this, 'RootCA', {
            subject: {
                cn: 'RenderQueueRootCA',
            },
        });
        const rfdkCert = new core_2.X509CertificatePem(this, 'RenderQueuePemCert', {
            subject: {
                cn: fullyQualifiedDomainName,
            },
            signingCertificate: rootCa,
        });
        const serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', rfdkCert);
        const certChain = rfdkCert.certChain;
        return {
            domainZone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Gets the certificate and PrivateHostedZone provided in the Render Queue's construct props.
     * @param externalTLS
     * @param hostname
     * @returns The provided certificate and domain info
     */
    getTlsInfoFromUserProps(externalTLS, hostname) {
        let serverCert;
        let certChain;
        if ((externalTLS.acmCertificate !== undefined) &&
            (externalTLS.rfdkCertificate !== undefined)) {
            throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
        }
        if (!hostname.hostname) {
            throw new Error('A hostname must be supplied if a certificate is supplied, '
                + 'with the common name of the certificate matching the hostname + domain name.');
        }
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(hostname.zone, hostname.hostname);
        if (externalTLS.acmCertificate) {
            if (externalTLS.acmCertificateChain === undefined) {
                throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
            }
            serverCert = externalTLS.acmCertificate;
            certChain = externalTLS.acmCertificateChain;
        }
        else { // Using externalTLS.rfdkCertificate
            if (externalTLS.rfdkCertificate.certChain === undefined) {
                throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
            }
            serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', externalTLS.rfdkCertificate);
            certChain = externalTLS.rfdkCertificate.certChain;
        }
        return {
            domainZone: hostname.zone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Helper method to create the fully qualified domain name for the given hostname and PrivateHostedZone.
     * @param hostname
     * @param zone
     * @returns The fully qualified domain name
     */
    generateFullyQualifiedDomainName(zone, hostname = RenderQueue.DEFAULT_HOSTNAME) {
        if (!RenderQueue.RE_VALID_HOSTNAME.test(hostname)) {
            throw new Error(`Invalid RenderQueue hostname: ${hostname}`);
        }
        return `${hostname}.${zone.zoneName}`;
    }
    /**
     * The instance that runs commands during the deployment.
     */
    get deploymentInstance() {
        var _b;
        const CONFIGURE_REPOSITORY_CONSTRUCT_ID = 'ConfigureRepository';
        const deploymentInstanceNode = this.node.tryFindChild(CONFIGURE_REPOSITORY_CONSTRUCT_ID);
        if (deploymentInstanceNode === undefined) {
            return new deployment_instance_1.DeploymentInstance(this, CONFIGURE_REPOSITORY_CONSTRUCT_ID, {
                securityGroup: this.backendConnections.securityGroups[0],
                vpc: this.props.vpc,
                vpcSubnets: (_b = this.props.vpcSubnets) !== null && _b !== void 0 ? _b : RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            });
        }
        else if (deploymentInstanceNode instanceof deployment_instance_1.DeploymentInstance) {
            return deploymentInstanceNode;
        }
        else {
            throw new Error(`Unexpected type for ${deploymentInstanceNode.node.path}. Expected ${deployment_instance_1.DeploymentInstance.name}, but found ${typeof (deploymentInstanceNode)}.`);
        }
    }
    /**
     * The construct that manages Deadline Secrets Management identity registration settings
     */
    get identityRegistrationSettings() {
        var _b;
        const IDENTITY_REGISTRATION_CONSTRUCT_ID = 'SecretsManagementIdentityRegistration';
        const secretsManagementIdentityRegistration = this.node.tryFindChild(IDENTITY_REGISTRATION_CONSTRUCT_ID);
        if (!secretsManagementIdentityRegistration) {
            return new secrets_management_1.SecretsManagementIdentityRegistration(this, IDENTITY_REGISTRATION_CONSTRUCT_ID, {
                deploymentInstance: this.deploymentInstance,
                repository: this.repository,
                renderQueueSubnets: this.props.vpc.selectSubnets((_b = this.props.vpcSubnetsAlb) !== null && _b !== void 0 ? _b : RenderQueue.DEFAULT_VPC_SUBNETS_ALB),
                version: this.props.version,
            });
        }
        else if (secretsManagementIdentityRegistration instanceof secrets_management_1.SecretsManagementIdentityRegistration) {
            return secretsManagementIdentityRegistration;
        }
        else {
            throw new Error(`Unexpected type for ${secretsManagementIdentityRegistration.node.path}. Expected ${secrets_management_1.SecretsManagementIdentityRegistration.name}, but found ${typeof (secretsManagementIdentityRegistration)}.`);
        }
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.41.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
RenderQueue.DEFAULT_HOSTNAME = 'renderqueue';
RenderQueue.DEFAULT_DOMAIN_NAME = 'aws-rfdk.com';
RenderQueue.DEFAULT_VPC_SUBNETS_ALB = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT, onePerAz: true };
RenderQueue.DEFAULT_VPC_SUBNETS_OTHER = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT };
/**
* The minimum Deadline version required for the Remote Connection Server to support load-balancing
*/
RenderQueue.MINIMUM_LOAD_BALANCING_VERSION = new version_1.Version([10, 1, 10, 0]);
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
/**
 * Information for the RCS user.
 */
RenderQueue.RCS_USER = { uid: 1000, gid: 1000, username: 'ec2-user' };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVuZGVyLXF1ZXVlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVuZGVyLXF1ZXVlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztHQUdHO0FBRUgsK0JBRWM7QUFDZCw4REFJa0M7QUFJbEMsOENBUzBCO0FBQzFCLDhDQVEwQjtBQUMxQixnRUFFbUM7QUFDbkMsb0ZBTTZDO0FBQzdDLDhDQU0wQjtBQUkxQixzREFBMEY7QUFJMUYsd0NBSXVCO0FBRXZCLHdCQVdXO0FBQ1gscUNBT29CO0FBRXBCLDRFQUF3RTtBQUN4RSw4REFFcUM7QUFDckMsbURBRXlCO0FBQ3pCLDZEQUU4QjtBQUM5Qix1Q0FBb0M7QUFDcEMsdUVBRW1DO0FBaURuQzs7R0FFRztBQUNILE1BQWUsZUFBZ0IsU0FBUSxnQkFBUztDQW9DL0M7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQUdELE1BQWEsV0FBWSxTQUFRLGVBQWU7Ozs7SUFrSDlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQXVCOztRQUNoRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRHdDLFVBQUssR0FBTCxLQUFLLENBQWtCO1FBbkRsRjs7V0FFRztRQUNLLHlCQUFvQixHQUFZLEtBQUssQ0FBQztRQUU5Qzs7V0FFRztRQUNLLHFDQUFnQyxHQUFZLEtBQUssQ0FBQztRQThDeEQsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxlQUFlLFNBQUcsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLGVBQWUsbUNBQUksRUFBQyxHQUFHLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUMsQ0FBQztRQUVsRSxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyw4QkFBOEIsQ0FBQyxFQUFFO1lBQ3hFLHVJQUF1STtZQUN2SSxJQUFJLE9BQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLG1DQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsV0FBVyxDQUFDLDhCQUE4QixDQUFDLFFBQVEsRUFBRSxtQ0FBbUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO2FBQzNMO1lBQ0QsSUFBSSxPQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxtQ0FBSSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELFdBQVcsQ0FBQyw4QkFBOEIsQ0FBQyxRQUFRLEVBQUUsbUNBQW1DLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQzthQUNuTTtZQUNELElBQUksT0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsbUNBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxXQUFXLENBQUMsOEJBQThCLENBQUMsUUFBUSxFQUFFLG1DQUFtQyxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7YUFDM0w7U0FDRjtRQUVELElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLE9BQU8sQ0FBQztRQUU5QixNQUFNLGdCQUFnQixHQUFHLGFBQUEsS0FBSyxDQUFDLGlCQUFpQiwwQ0FBRSxXQUFXLDBDQUFFLE9BQU8sTUFBSyxLQUFLLENBQUMsQ0FBQyxDQUFDLGdEQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsZ0RBQW1CLENBQUMsS0FBSyxDQUFDO1FBQ3hJLElBQUksZ0JBQW9DLENBQUM7UUFDekMsSUFBSSxVQUFtQyxDQUFDO1FBRXhDLElBQUssZ0JBQWdCLEtBQU0sZ0RBQW1CLENBQUMsS0FBSyxFQUFHO1lBQ3JELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUUvQyxJQUFJLENBQUMsU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7WUFDbkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO1lBQ3JDLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQztZQUNwRCxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztTQUNqQzthQUFNO1lBQ0wsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO2dCQUNsQixnQkFBZ0IsR0FBRyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDdkcsVUFBVSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO2FBQ2xDO1NBQ0Y7UUFFRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFFN0IsTUFBTSxnQkFBZ0IsZUFBRyxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLGdCQUFnQixtQ0FBSSxnREFBbUIsQ0FBQyxLQUFLLENBQUM7UUFFaEcsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMxQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsZUFBRyxLQUFLLENBQUMsZUFBZSwwQ0FBRSxHQUFHLG1DQUFJLENBQUMsQ0FBQztRQUNwRCxJQUFJLFdBQVcsR0FBRyxDQUFDLEVBQUU7WUFDbkIsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUN4RjtRQUNELE1BQU0sV0FBVyxTQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyx5Q0FBSSxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPLENBQUM7UUFDOUUsSUFBSSxPQUFBLElBQUksQ0FBQyxlQUFlLDBDQUFFLE9BQU8sS0FBSSxXQUFXLElBQUksT0FBQSxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPLElBQUcsV0FBVyxFQUFFO1lBQy9GLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELFdBQVcsU0FBUyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDN0g7UUFDRCxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRTtZQUNsRCxVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksV0FBVyxDQUFDLHlCQUF5QjtZQUNyRSxZQUFZLFFBQUUsS0FBSyxDQUFDLFlBQVksbUNBQUksSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztZQUNoRSxXQUFXO1lBQ1gsZUFBZSxRQUFFLElBQUksQ0FBQyxlQUFlLDBDQUFFLE9BQU87WUFDOUMsV0FBVztZQUNYLFlBQVksRUFBRSxDQUFDO29CQUNiLFVBQVUsRUFBRSxXQUFXO29CQUN2QiwrRkFBK0Y7b0JBQy9GLHNFQUFzRTtvQkFDdEUsTUFBTSxFQUFFLG1DQUFpQixDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ3ZELENBQUM7WUFDRixVQUFVLEVBQUUsU0FBUztZQUNyQixZQUFZLEVBQUUsOEJBQVksQ0FBQyxhQUFhLEVBQUU7WUFDMUMsZ0hBQWdIO1lBQ2hILGdEQUFnRDtZQUNoRCxhQUFhO1lBQ2IsYUFBYSxRQUFFLEtBQUssQ0FBQyxjQUFjLDBDQUFFLE9BQU87U0FDN0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDO1FBRS9DOzs7O1dBSUc7UUFDSCxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQzNCLDhCQUE4QixDQUMvQixDQUFDO1FBQ0YsVUFBSSxLQUFLLENBQUMsc0JBQXNCLG1DQUFJLEtBQUssRUFBRTtZQUN6QywrQ0FBK0M7WUFDL0MsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNsQztRQUVELE1BQU0sa0JBQWtCLEdBQUcsV0FBVyxDQUFDLGVBQWUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3pFLE1BQU0sa0JBQWtCLEdBQUcsV0FBVyxDQUFDLGVBQWUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBRXpFLElBQUksQ0FBQyxRQUFRLEdBQUcsc0JBQWUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsRUFBRTtZQUN6RSxjQUFjLEVBQUUsY0FBYztZQUM5QixHQUFHLEtBQUssQ0FBQyxhQUFhO1NBQ3ZCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVuQyxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsT0FBTyxFQUFFO1lBQ3RELE1BQU0sTUFBTSxHQUFHLEVBQUUsQ0FBQztZQUNsQixJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLGlCQUFPLENBQUMsa0NBQWtDLENBQUMsRUFBRTtnQkFDeEUsTUFBTSxDQUFDLElBQUksQ0FBQyxrQ0FBa0MsS0FBSyxDQUFDLE9BQU8sQ0FBQyxhQUFhLG9IQUFvSCxpQkFBTyxDQUFDLGtDQUFrQyxDQUFDLGFBQWEsb0VBQW9FLENBQUMsQ0FBQzthQUM1VDtZQUNELElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxXQUFXLEtBQUssU0FBUyxFQUFFO2dCQUN4RSxNQUFNLENBQUMsSUFBSSxDQUFDLDZEQUE2RCxDQUFDLENBQUM7YUFDNUU7WUFDRCxJQUFJLGdCQUFnQixLQUFLLGdEQUFtQixDQUFDLEtBQUssRUFBRTtnQkFDbEQsTUFBTSxDQUFDLElBQUksQ0FBQyx5REFBeUQsQ0FBQyxDQUFDO2FBQ3hFO1lBQ0QsSUFBSSxnQkFBZ0IsS0FBSyxnREFBbUIsQ0FBQyxLQUFLLEVBQUU7Z0JBQ2xELE1BQU0sQ0FBQyxJQUFJLENBQUMsa0RBQWtELENBQUMsQ0FBQzthQUNqRTtZQUNELElBQUksTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsMklBQTJJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ2pMO1NBQ0Y7UUFDRCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUM7WUFDL0MsS0FBSyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsc0JBQXNCO1lBQzFDLFVBQVUsRUFBRSxrQkFBa0I7WUFDOUIsUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsU0FBUyxFQUFFLFdBQVcsQ0FBQyxRQUFRO1lBQy9CLHdCQUF3QixFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztnQkFDN0UsV0FBVyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsV0FBWTtnQkFDcEUsYUFBYSxFQUFFLFdBQVcsQ0FBQyxRQUFRLENBQUMsUUFBUTthQUM3QyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ2QsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7UUFFckMscURBQXFEO1FBRXJELE1BQU0sWUFBWSxHQUFHLElBQUksb0RBQXVCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRTtZQUMzRCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHO1lBQ3JCLFVBQVUsUUFBRSxLQUFLLENBQUMsYUFBYSxtQ0FBSSxXQUFXLENBQUMsdUJBQXVCO1lBQ3RFLGNBQWMsRUFBRSxLQUFLO1lBQ3JCLGtCQUFrQixRQUFFLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksSUFBSTtZQUNwRCxhQUFhLFFBQUUsS0FBSyxDQUFDLGNBQWMsMENBQUUsUUFBUTtTQUM5QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksb0RBQWlDLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ2pGLFdBQVcsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUM1QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsWUFBWSxRQUFFLElBQUksQ0FBQyxlQUFlLDBDQUFFLE9BQU87WUFDM0MsVUFBVTtZQUNWLFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsWUFBWSxFQUFFLGtCQUFrQjtZQUNoQyxZQUFZO1lBQ1osUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixjQUFjO1lBQ2Qsa0hBQWtIO1lBQ2xILGlIQUFpSDtZQUNqSCxrSEFBa0g7WUFDbEgsMEJBQTBCO1lBQzFCLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsaUJBQWlCLEVBQUUsR0FBRztZQUN0QiwyR0FBMkc7WUFDM0csWUFBWSxFQUFFLEtBQUs7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsZ0ZBQWdGO1FBQ2hGLDZGQUE2RjtRQUM3RixJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNsRDtRQUVELHlGQUF5RjtRQUN6RixtR0FBbUc7UUFDbkcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFbEQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQztRQUM5QywwR0FBMEc7UUFDMUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUMsaURBQWlELEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFMUYsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQztZQUU1RCxvQ0FBb0M7WUFDcEMscUdBQXFHO1lBQ3JHLGdCQUFnQixDQUFDLG1CQUFtQixDQUFFLElBQUkseUJBQWUsQ0FBQztnQkFDeEQsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixVQUFVLEVBQUUsQ0FBQyxJQUFJLDBCQUFnQixDQUFDLDZCQUE2QixDQUFDLENBQUM7Z0JBQ2pFLFNBQVMsRUFBRSxDQUFDLEdBQUcsZ0JBQWdCLENBQUMsU0FBUyxJQUFJLENBQUM7Z0JBQzlDLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osY0FBYyxFQUFFLDJCQUEyQjtxQkFDNUM7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLElBQUkseUJBQWUsQ0FBQztnQkFDdkQsT0FBTyxFQUFFLENBQUUsaUJBQWlCLENBQUU7Z0JBQzlCLFVBQVUsRUFBRSxDQUFFLElBQUksMEJBQWdCLENBQUMsNkJBQTZCLENBQUMsQ0FBQztnQkFDbEUsU0FBUyxFQUFFLENBQUUsZ0JBQWdCLENBQUMsU0FBUyxDQUFFO2FBQzFDLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQzdCLGdCQUFnQixFQUNoQixLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzVCO1FBRUQsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLDZCQUFtQixDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQztRQUV0Rjs7O1dBR0c7UUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQStCLENBQUM7UUFDaEQsTUFBTSxXQUFXLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUEyQixDQUFDO1FBQ2xGLE1BQU0sbUJBQW1CLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxZQUE4QixDQUFDO1FBQzVFLG1CQUFtQixDQUFDLFFBQVEsR0FBRyxnREFBbUIsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3JFLG1CQUFtQixDQUFDLElBQUksR0FBRyxrQkFBa0IsQ0FBQztRQUU5QyxJQUFJLENBQUMsY0FBYyxHQUFHLGNBQWMsQ0FBQyxRQUFRLENBQUM7UUFFOUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLHFCQUFXLENBQUM7WUFDakMsV0FBVyxFQUFFLGNBQUksQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUM7WUFDekMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxjQUFjO1NBQ3JFLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxxQ0FBOEIsQ0FBQztZQUNqRCxPQUFPLEVBQUUsZ0JBQWdCLGFBQWhCLGdCQUFnQixjQUFoQixnQkFBZ0IsR0FBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxtQkFBbUI7WUFDMUUsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDN0IsUUFBUSxFQUFFLGdCQUFnQjtTQUMzQixDQUFDLENBQUM7UUFFSCxJQUFLLGdCQUFnQixLQUFLLGdEQUFtQixDQUFDLElBQUksRUFBRztZQUNuRCxJQUFJLENBQUMsWUFBWSxHQUFHLHFDQUFxQixDQUFDLE9BQU8sQ0FBQztnQkFDaEQsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2FBQ3hCLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsWUFBWSxHQUFHLHFDQUFxQixDQUFDLFFBQVEsQ0FBQztnQkFDakQsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUN2QixNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVU7YUFDeEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFBLEtBQUssQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsV0FBVywwQ0FBRSxTQUFTLENBQUMsSUFBSSxFQUFFO1FBRXhFLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLDhDQUFvQixDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRTtZQUNqRixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxHQUFHLGNBQWMsQ0FBQztRQUV4Qyw2Q0FBNkM7UUFDN0MsMkJBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNyQixDQUFDOzs7Ozs7Ozs7SUFFUyxVQUFVO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO1FBRTVCLGlIQUFpSDtRQUNqSCxpREFBaUQ7UUFDakQsSUFBSSxJQUFJLENBQUMsT0FBTyxZQUFZLGVBQVksRUFBRTtZQUN4QyxNQUFNLFlBQVksR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUM1QyxNQUFNLFNBQVMsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pDLElBQUksWUFBWSxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLDJEQUEyRCxDQUFDLENBQUM7YUFDcEY7U0FDRjtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQzs7Ozs7OztJQUdNLGtCQUFrQixDQUFDLEtBQXdCO1FBQ2hELEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFFLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFFLENBQUM7UUFDN0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUM7Ozs7Ozs7SUFHTSx1QkFBdUIsQ0FBQyxLQUE2QjtRQUMxRCxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxZQUFZLENBQUMsdUJBQXVCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDbkQsQ0FBQzs7Ozs7Ozs7Ozs7Ozs7SUFHTSwwQ0FBMEMsQ0FBQyxLQUE4QztRQUM5RixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxPQUFPLEVBQUU7WUFDdEQsbURBQW1EO1lBQ25ELE9BQU87U0FDUjtRQUVELElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxvQ0FBb0MsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNoRixDQUFDOzs7Ozs7Ozs7SUFHTSxjQUFjLENBQUMseUJBQWtDLElBQUk7UUFDMUQsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtZQUM5QixNQUFNLFNBQVMsR0FBRyx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLCtDQUErQyxDQUFDLENBQUM7WUFDMUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDekQsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQztTQUNsQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLEVBQUU7WUFDMUMsSUFBSSxzQkFBc0IsRUFBRTtnQkFDMUIsTUFBTSxRQUFRLEdBQUcsdUJBQWEsQ0FBQyx3QkFBd0IsQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO2dCQUN6RyxJQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDeEQsSUFBSSxDQUFDLGdDQUFnQyxHQUFHLElBQUksQ0FBQzthQUM5QztTQUNGO0lBQ0gsQ0FBQzs7Ozs7Ozs7Ozs7SUFHTSxrQkFBa0IsQ0FBQyxLQUFpQjtRQUN6Qyx1RUFBdUU7UUFDdkUseUVBQXlFO1FBQ3pFLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN4QyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDOUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMvQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN0RCxDQUFDOzs7Ozs7O0lBR00seUJBQXlCLENBQUMsR0FBRyxjQUFnQztRQUNsRSxjQUFjLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO0lBQzdGLENBQUM7Ozs7Ozs7SUFHTSx3QkFBd0IsQ0FBQyxHQUFHLGNBQWdDO1FBQ2pFLGNBQWMsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7SUFDcEYsQ0FBQztJQUVPLGlCQUFpQixDQUFDLEdBQXFCO1FBQzdDLE1BQU0sTUFBTSxHQUFHLGtCQUFXLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ3ZFLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTTtZQUNsQixRQUFRLEVBQUUsbUJBQW1CO1lBQzdCLE9BQU8sRUFBRSxXQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxTQUFTLENBQUM7U0FDMUMsQ0FBQyxDQUFDO1FBQ0gscURBQXFEO1FBQ3JELEdBQUcsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0QsTUFBTSxDQUFDLFNBQVMsQ0FBQztZQUNmLElBQUksRUFBRSxHQUFHO1NBQ1YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLG9CQUFvQixDQUFDLEtBTzVCO1FBQ0MsTUFBTSxFQUFFLEtBQUssRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUUxRCxNQUFNLGNBQWMsR0FBRyxJQUFJLDJCQUFpQixDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQztRQUU5RCxtRUFBbUU7UUFDbkUsTUFBTSxVQUFVLEdBQUcsVUFBVSxDQUFDLGtCQUFrQixDQUFDO1lBQy9DLGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO2FBQ2xCO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLGNBQWM7YUFDZjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFHLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQztRQUVwRCxJQUFJLFFBQVEsS0FBSyxnREFBbUIsQ0FBQyxLQUFLLEVBQUU7WUFDMUMscUdBQXFHO1lBQ3JHLG9GQUFvRjtZQUNwRixNQUFNLFVBQVUsR0FBRyxJQUFJLHlCQUFrQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7Z0JBQzlELE9BQU8sRUFBRTtvQkFDUCxFQUFFLEVBQUUsa0JBQWtCO2lCQUN2QjthQUNGLENBQUMsQ0FBQztZQUNILE1BQU0sV0FBVyxHQUFHLElBQUksNEJBQXFCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO2dCQUN0RSxpQkFBaUIsRUFBRSxVQUFVO2FBQzlCLENBQUMsQ0FBQztZQUNILENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzNFLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzVDLENBQUMsQ0FBQyxDQUFDO1lBQ0gsV0FBVyxDQUFDLG1CQUFtQixHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQzVELFdBQVcsQ0FBQyxnQkFBZ0IsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUMxRCxXQUFXLENBQUMsMkJBQTJCLEdBQUcsV0FBVyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUM7WUFDM0UsV0FBVyxDQUFDLDJCQUEyQixHQUFHLElBQUksQ0FBQztTQUNoRDtRQUVELElBQUksS0FBSyxDQUFDLHdCQUF3QixLQUFLLFNBQVMsRUFBRTtZQUNoRCxXQUFXLENBQUMsc0JBQXNCLEdBQUcsS0FBSyxDQUFDLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUM7U0FDM0Y7UUFFRCxpRkFBaUY7UUFDakYsMEJBQTBCO1FBQzFCLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQzNGLE1BQU0sbUJBQW1CLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxxQkFBcUIsRUFBRTtZQUM3RSxLQUFLO1lBQ0wsb0JBQW9CLEVBQUUsSUFBSTtZQUMxQixXQUFXO1lBQ1gsT0FBTyxFQUFFLG1CQUFTLENBQUMsT0FBTyxDQUFDO2dCQUN6QixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLFlBQVksRUFBRSxLQUFLO2FBQ3BCLENBQUM7WUFDRixJQUFJO1NBQ0wsQ0FBQyxDQUFDO1FBRUgsbUJBQW1CLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBRW5FLElBQUksS0FBSyxDQUFDLHdCQUF3QixLQUFLLFNBQVMsRUFBRTtZQUNoRCxvRkFBb0Y7WUFDcEYseUZBQXlGO1lBQ3pGLE1BQU0sVUFBVSxHQUFHLHdCQUF3QixDQUFDO1lBQzVDLGNBQWMsQ0FBQyxTQUFTLENBQUM7Z0JBQ3ZCLElBQUksRUFBRSxVQUFVO2dCQUNoQix5QkFBeUIsRUFBRTtvQkFDekIsS0FBSyxFQUFFLGVBQUssQ0FBQyxNQUFNO29CQUNuQixhQUFhLEVBQUUsSUFBSTtvQkFDbkIsTUFBTSxFQUFFLE9BQU87aUJBQ2hCO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZ0ZBQWdGO1lBQ2hGLG1CQUFtQixDQUFDLGNBQWMsQ0FBQztnQkFDakMsUUFBUSxFQUFFLEtBQUs7Z0JBQ2YsWUFBWSxFQUFFLFVBQVU7Z0JBQ3hCLGFBQWEsRUFBRSxTQUFTLEtBQUssQ0FBQyx3QkFBd0IsQ0FBQyxhQUFhLHlCQUF5QjthQUM5RixDQUFDLENBQUM7U0FDSjtRQUVELG1CQUFtQjtRQUNuQixtQkFBbUIsQ0FBQyxVQUFVLENBQzVCO1lBQ0UsSUFBSSxFQUFFLG9CQUFVLENBQUMsTUFBTTtZQUN2QixTQUFTLEVBQUUsTUFBTTtZQUNqQixTQUFTLEVBQUUsTUFBTTtTQUNsQixFQUFFO1lBQ0QsSUFBSSxFQUFFLG9CQUFVLENBQUMsS0FBSztZQUN0QixTQUFTLEVBQUUsS0FBSztZQUNoQixTQUFTLEVBQUUsS0FBSztTQUNqQixDQUNGLENBQUM7UUFFRixtQkFBbUIsQ0FBQyxlQUFlLENBQUM7WUFDbEMsYUFBYSxFQUFFLFVBQVU7WUFDekIsUUFBUSxFQUFFLFVBQVU7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxrQkFBa0IsQ0FBQyxLQUF1Qjs7UUFDaEQsSUFBSyxDQUFDLGFBQUEsS0FBSyxDQUFDLGlCQUFpQiwwQ0FBRSxXQUFXLDBDQUFFLGNBQWMsTUFBSyxTQUFTLENBQUU7WUFDdEUsQ0FBQyxhQUFBLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsV0FBVywwQ0FBRSxlQUFlLE1BQUssU0FBUyxDQUFDLEVBQUc7WUFDMUUsSUFBSSxLQUFLLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtnQkFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1RkFBdUYsQ0FBQyxDQUFDO2FBQzFHO1lBQ0QsT0FBTyxJQUFJLENBQUMsdUJBQXVCLENBQ2pDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLEVBQ25DLEtBQUssQ0FBQyxRQUFRLENBQ2YsQ0FBQztTQUNIO1FBRUQsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDOUQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssb0JBQW9CLENBQUMsR0FBUyxFQUFFLFFBQW1DOztRQUN6RSxNQUFNLFVBQVUsU0FBRyxRQUFRLGFBQVIsUUFBUSx1QkFBUixRQUFRLENBQUUsSUFBSSxtQ0FBSSxJQUFJLCtCQUFpQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDMUUsR0FBRyxFQUFFLEdBQUc7WUFDUixRQUFRLEVBQUUsV0FBVyxDQUFDLG1CQUFtQjtTQUMxQyxDQUFDLENBQUM7UUFFSCxNQUFNLHdCQUF3QixHQUFHLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxVQUFVLEVBQUUsUUFBUSxhQUFSLFFBQVEsdUJBQVIsUUFBUSxDQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXZHLE1BQU0sTUFBTSxHQUFHLElBQUkseUJBQWtCLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUNwRCxPQUFPLEVBQUU7Z0JBQ1AsRUFBRSxFQUFFLG1CQUFtQjthQUN4QjtTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLElBQUkseUJBQWtCLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO1lBQ2xFLE9BQU8sRUFBRTtnQkFDUCxFQUFFLEVBQUUsd0JBQXdCO2FBQzdCO1lBQ0Qsa0JBQWtCLEVBQUUsTUFBTTtTQUMzQixDQUFDLENBQUM7UUFDSCxNQUFNLFVBQVUsR0FBRyxJQUFJLDZCQUFzQixDQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxDQUFFLENBQUM7UUFDM0UsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLFNBQVUsQ0FBQztRQUV0QyxPQUFPO1lBQ0wsVUFBVTtZQUNWLHdCQUF3QjtZQUN4QixVQUFVO1lBQ1YsU0FBUztTQUNWLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSyx1QkFBdUIsQ0FBQyxXQUF3QyxFQUFFLFFBQWtDO1FBQzFHLElBQUksVUFBd0IsQ0FBQztRQUM3QixJQUFJLFNBQWtCLENBQUM7UUFFdkIsSUFBSyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFFO1lBQ2hELENBQUMsV0FBVyxDQUFDLGVBQWUsS0FBSyxTQUFTLENBQUMsRUFBRztZQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLG9IQUFvSCxDQUFDLENBQUM7U0FDdkk7UUFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLDREQUE0RDtrQkFDeEUsOEVBQThFLENBQUMsQ0FBQztTQUNyRjtRQUVELE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXpHLElBQUssV0FBVyxDQUFDLGNBQWMsRUFBRztZQUNoQyxJQUFLLFdBQVcsQ0FBQyxtQkFBbUIsS0FBSyxTQUFTLEVBQUc7Z0JBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMseUZBQXlGLENBQUMsQ0FBQzthQUM1RztZQUNELFVBQVUsR0FBRyxXQUFXLENBQUMsY0FBYyxDQUFDO1lBQ3hDLFNBQVMsR0FBRyxXQUFXLENBQUMsbUJBQW1CLENBQUM7U0FFN0M7YUFBTSxFQUFFLG9DQUFvQztZQUMzQyxJQUFLLFdBQVcsQ0FBQyxlQUFnQixDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUc7Z0JBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQzthQUNuRjtZQUNELFVBQVUsR0FBRyxJQUFJLDZCQUFzQixDQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsV0FBVyxDQUFDLGVBQWdCLENBQUUsQ0FBQztZQUN6RixTQUFTLEdBQUcsV0FBVyxDQUFDLGVBQWdCLENBQUMsU0FBUyxDQUFDO1NBQ3BEO1FBRUQsT0FBTztZQUNMLFVBQVUsRUFBRSxRQUFRLENBQUMsSUFBSTtZQUN6Qix3QkFBd0I7WUFDeEIsVUFBVTtZQUNWLFNBQVM7U0FDVixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssZ0NBQWdDLENBQUMsSUFBd0IsRUFBRSxXQUFtQixXQUFXLENBQUMsZ0JBQWdCO1FBQ2hILElBQUksQ0FBQyxXQUFXLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDOUQ7UUFDRCxPQUFPLEdBQUcsUUFBUSxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFZLGtCQUFrQjs7UUFDNUIsTUFBTSxpQ0FBaUMsR0FBRyxxQkFBcUIsQ0FBQztRQUNoRSxNQUFNLHNCQUFzQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDekYsSUFBSSxzQkFBc0IsS0FBSyxTQUFTLEVBQUU7WUFDeEMsT0FBTyxJQUFJLHdDQUFrQixDQUFDLElBQUksRUFBRSxpQ0FBaUMsRUFBRTtnQkFDckUsYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO2dCQUN4RCxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHO2dCQUNuQixVQUFVLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLG1DQUFJLFdBQVcsQ0FBQyx5QkFBeUI7YUFDM0UsQ0FBQyxDQUFDO1NBQ0o7YUFBTSxJQUFJLHNCQUFzQixZQUFZLHdDQUFrQixFQUFFO1lBQy9ELE9BQU8sc0JBQXNCLENBQUM7U0FDL0I7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLHNCQUFzQixDQUFDLElBQUksQ0FBQyxJQUFJLGNBQWMsd0NBQWtCLENBQUMsSUFBSSxlQUFlLE9BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMvSjtJQUNILENBQUM7SUFFRDs7T0FFRztJQUNILElBQVksNEJBQTRCOztRQUN0QyxNQUFNLGtDQUFrQyxHQUFHLHVDQUF1QyxDQUFDO1FBQ25GLE1BQU0scUNBQXFDLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUN6RyxJQUFJLENBQUMscUNBQXFDLEVBQUU7WUFDMUMsT0FBTyxJQUFJLDBEQUFxQyxDQUM5QyxJQUFJLEVBQUUsa0NBQWtDLEVBQUU7Z0JBQ3hDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0I7Z0JBQzNDLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDM0Isa0JBQWtCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxPQUM5QyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUksV0FBVyxDQUFDLHVCQUF1QixDQUNoRTtnQkFDRCxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO2FBQzVCLENBQ0YsQ0FBQztTQUNIO2FBQU0sSUFBSSxxQ0FBcUMsWUFBWSwwREFBcUMsRUFBRTtZQUNqRyxPQUFPLHFDQUFxQyxDQUFDO1NBQzlDO2FBQU07WUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixxQ0FBcUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxjQUFjLDBEQUFxQyxDQUFDLElBQUksZUFBZSxPQUFNLENBQUMscUNBQXFDLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDaE47SUFDSCxDQUFDOztBQTFzQkgsa0NBMnNCQzs7O0FBMXNCQzs7R0FFRztBQUNxQiwyQkFBZSxHQUFHO0lBQ3hDLENBQUMsZ0RBQW1CLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSTtJQUNoQyxDQUFDLGdEQUFtQixDQUFDLEtBQUssQ0FBQyxFQUFFLElBQUk7Q0FDbEMsQ0FBQztBQUVzQiw0QkFBZ0IsR0FBRyxhQUFhLENBQUM7QUFFakMsK0JBQW1CLEdBQUcsY0FBYyxDQUFDO0FBRXJDLG1DQUF1QixHQUFvQixFQUFFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLGdCQUFnQixFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQztBQUV2RyxxQ0FBeUIsR0FBb0IsRUFBRSxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO0FBRWpIOztFQUVFO0FBQ3NCLDBDQUE4QixHQUFHLElBQUksaUJBQU8sQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFckY7O0dBRUc7QUFDcUIsNkJBQWlCLEdBQUcsc0NBQXNDLENBQUM7QUFFbkY7O0dBRUc7QUFFcUIsb0JBQVEsR0FBRyxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQge1xuICBqb2luLFxufSBmcm9tICdwYXRoJztcbmltcG9ydCB7XG4gIEF1dG9TY2FsaW5nR3JvdXAsXG4gIEJsb2NrRGV2aWNlVm9sdW1lLFxuICBVcGRhdGVQb2xpY3ksXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1hdXRvc2NhbGluZyc7XG5pbXBvcnQge1xuICBJQ2VydGlmaWNhdGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHtcbiAgQ29ubmVjdGlvbnMsXG4gIElDb25uZWN0YWJsZSxcbiAgSW5zdGFuY2VUeXBlLFxuICBJU2VjdXJpdHlHcm91cCxcbiAgSVZwYyxcbiAgUG9ydCxcbiAgU3VibmV0U2VsZWN0aW9uLFxuICBTdWJuZXRUeXBlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIENsdXN0ZXIsXG4gIENvbnRhaW5lckltYWdlLFxuICBFYzJUYXNrRGVmaW5pdGlvbixcbiAgTG9nRHJpdmVyLFxuICBQbGFjZW1lbnRDb25zdHJhaW50LFxuICBTY29wZSxcbiAgVWxpbWl0TmFtZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQge1xuICBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZEVjMlNlcnZpY2UsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MtcGF0dGVybnMnO1xuaW1wb3J0IHtcbiAgQXBwbGljYXRpb25MaXN0ZW5lcixcbiAgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIsXG4gIEFwcGxpY2F0aW9uUHJvdG9jb2wsXG4gIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAsXG4gIENmblRhcmdldEdyb3VwLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQge1xuICBJR3JhbnRhYmxlLFxuICBJUHJpbmNpcGFsLFxuICBNYW5hZ2VkUG9saWN5LFxuICBQb2xpY3lTdGF0ZW1lbnQsXG4gIFNlcnZpY2VQcmluY2lwYWwsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHtcbiAgSUxvZ0dyb3VwLFxufSBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgeyBJSG9zdGVkWm9uZSwgSVByaXZhdGVIb3N0ZWRab25lLCBQcml2YXRlSG9zdGVkWm9uZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1yb3V0ZTUzJztcbmltcG9ydCB7XG4gIElTZWNyZXQsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQge1xuICBDb25zdHJ1Y3QsXG4gIElDb25zdHJ1Y3QsXG4gIFN0YWNrLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgRUNTQ29ubmVjdE9wdGlvbnMsXG4gIEluc3RhbmNlQ29ubmVjdE9wdGlvbnMsXG4gIElSZXBvc2l0b3J5LFxuICBJVmVyc2lvbixcbiAgUmVuZGVyUXVldWVFeHRlcm5hbFRMU1Byb3BzLFxuICBSZW5kZXJRdWV1ZUhvc3ROYW1lUHJvcHMsXG4gIFJlbmRlclF1ZXVlUHJvcHMsXG4gIFJlbmRlclF1ZXVlU2l6ZUNvbnN0cmFpbnRzLFxuICBTdWJuZXRJZGVudGl0eVJlZ2lzdHJhdGlvblNldHRpbmdzUHJvcHMsXG4gIFZlcnNpb25RdWVyeSxcbn0gZnJvbSAnLic7XG5pbXBvcnQge1xuICBDb25uZWN0YWJsZUFwcGxpY2F0aW9uRW5kcG9pbnQsXG4gIEltcG9ydGVkQWNtQ2VydGlmaWNhdGUsXG4gIExvZ0dyb3VwRmFjdG9yeSxcbiAgU2NyaXB0QXNzZXQsXG4gIFg1MDlDZXJ0aWZpY2F0ZVBlbSxcbiAgWDUwOUNlcnRpZmljYXRlUGtjczEyLFxufSBmcm9tICcuLi8uLi9jb3JlJztcblxuaW1wb3J0IHsgRGVwbG95bWVudEluc3RhbmNlIH0gZnJvbSAnLi4vLi4vY29yZS9saWIvZGVwbG95bWVudC1pbnN0YW5jZSc7XG5pbXBvcnQge1xuICB0YWdDb25zdHJ1Y3QsXG59IGZyb20gJy4uLy4uL2NvcmUvbGliL3J1bnRpbWUtaW5mbyc7XG5pbXBvcnQge1xuICBSZW5kZXJRdWV1ZUNvbm5lY3Rpb24sXG59IGZyb20gJy4vcnEtY29ubmVjdGlvbic7XG5pbXBvcnQge1xuICBTZWNyZXRzTWFuYWdlbWVudElkZW50aXR5UmVnaXN0cmF0aW9uLFxufSBmcm9tICcuL3NlY3JldHMtbWFuYWdlbWVudCc7XG5pbXBvcnQgeyBWZXJzaW9uIH0gZnJvbSAnLi92ZXJzaW9uJztcbmltcG9ydCB7XG4gIFdhaXRGb3JTdGFibGVTZXJ2aWNlLFxufSBmcm9tICcuL3dhaXQtZm9yLXN0YWJsZS1zZXJ2aWNlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlbmRlclF1ZXVlIGV4dGVuZHMgSUNvbnN0cnVjdCwgSUNvbm5lY3RhYmxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXBvc2l0b3J5OiBJUmVwb3NpdG9yeTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5kcG9pbnQ6IENvbm5lY3RhYmxlQXBwbGljYXRpb25FbmRwb2ludDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYmFja2VuZENvbm5lY3Rpb25zOiBDb25uZWN0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGNvbmZpZ3VyZUNsaWVudEVDUyhwYXJhbXM6IEVDU0Nvbm5lY3RPcHRpb25zKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25maWd1cmVDbGllbnRJbnN0YW5jZShwYXJhbXM6IEluc3RhbmNlQ29ubmVjdE9wdGlvbnMpOiB2b2lkO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uZmlndXJlU2VjcmV0c01hbmFnZW1lbnRBdXRvUmVnaXN0cmF0aW9uKHByb3BzOiBTdWJuZXRJZGVudGl0eVJlZ2lzdHJhdGlvblNldHRpbmdzUHJvcHMpOiB2b2lkO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHJlbmRlciBxdWV1ZSdzIFRMUyBjb25maWd1cmF0aW9uXG4gKi9cbmludGVyZmFjZSBUbHNJbmZvIHtcbiAgLyoqXG4gICAqIFRoZSBjZXJ0aWZpY2F0ZSB0aGUgUmVuZGVyIFF1ZXVlJ3Mgc2VydmVyIHdpbGwgdXNlIGZvciB0aGUgVExTIGNvbm5lY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBzZXJ2ZXJDZXJ0OiBJQ2VydGlmaWNhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBjZXJ0aWZpY2F0ZSBjaGFpbiBjbGllbnRzIGNhbiB1c2UgdG8gdmVyaWZ5IHRoZSBjZXJ0aWZpY2F0ZS5cbiAgICovXG4gIHJlYWRvbmx5IGNlcnRDaGFpbjogSVNlY3JldDtcblxuICAvKipcbiAgICogVGhlIHByaXZhdGUgaG9zdGVkIHpvbmUgdGhhdCB0aGUgcmVuZGVyIHF1ZXVlJ3MgbG9hZCBiYWxhbmNlciB3aWxsIGJlIHBsYWNlZCBpbi5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpblpvbmU6IElQcml2YXRlSG9zdGVkWm9uZTtcblxuICAvKipcbiAgICogVGhlIGZ1bGx5IHF1YWxpZmllZCBkb21haW4gbmFtZSB0aGF0IHdpbGwgYmUgZ2l2ZW4gdG8gdGhlIGxvYWQgYmFsYW5jZXIgaW4gdGhlIHByaXZhdGVcbiAgICogaG9zdGVkIHpvbmUuXG4gICAqL1xuICByZWFkb25seSBmdWxseVF1YWxpZmllZERvbWFpbk5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBSZW5kZXIgUXVldWUgcHJvdmlkZXJzXG4gKi9cbmFic3RyYWN0IGNsYXNzIFJlbmRlclF1ZXVlQmFzZSBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElSZW5kZXJRdWV1ZSB7XG4gIC8qKlxuICAgKiBUaGUgZW5kcG9pbnQgdGhhdCBEZWFkbGluZSBjbGllbnRzIGNhbiB1c2UgdG8gY29ubmVjdCB0byB0aGUgUmVuZGVyIFF1ZXVlXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZW5kcG9pbnQ6IENvbm5lY3RhYmxlQXBwbGljYXRpb25FbmRwb2ludDtcblxuICAvKipcbiAgICogQWxsb3dzIHNwZWNpZnlpbmcgc2VjdXJpdHkgZ3JvdXAgY29ubmVjdGlvbnMgZm9yIHRoZSBSZW5kZXIgUXVldWUuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY29ubmVjdGlvbnM6IENvbm5lY3Rpb25zO1xuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlcG9zaXRvcnk6IElSZXBvc2l0b3J5O1xuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGJhY2tlbmRDb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgYW4gRUNTIGNsdXN0ZXIgdG8gYmUgYWJsZSB0byBjb25uZWN0IHRvIGEgUmVuZGVyUXVldWVcbiAgICogQHJldHVybnMgQW4gZW52aXJvbm1lbnQgbWFwcGluZyB0aGF0IGlzIHVzZWQgdG8gY29uZmlndXJlIHRoZSBEb2NrZXIgSW1hZ2VzXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgY29uZmlndXJlQ2xpZW50RUNTKHBhcmFtczogRUNTQ29ubmVjdE9wdGlvbnMpOiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogQ29uZmlndXJlIGFuIEluc3RhbmNlL0F1dG9zY2FsaW5nIGdyb3VwIHRvIGNvbm5lY3QgdG8gYSBSZW5kZXJRdWV1ZVxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IGNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHBhcmFtczogSW5zdGFuY2VDb25uZWN0T3B0aW9ucyk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgY29uZmlndXJlU2VjcmV0c01hbmFnZW1lbnRBdXRvUmVnaXN0cmF0aW9uKHByb3BzOiBTdWJuZXRJZGVudGl0eVJlZ2lzdHJhdGlvblNldHRpbmdzUHJvcHMpOiB2b2lkO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgUmVuZGVyUXVldWUgZXh0ZW5kcyBSZW5kZXJRdWV1ZUJhc2UgaW1wbGVtZW50cyBJR3JhbnRhYmxlIHtcbiAgLyoqXG4gICAqIENvbnRhaW5lciBsaXN0ZW5pbmcgcG9ydHMgZm9yIGVhY2ggcHJvdG9jb2wuXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBSQ1NfUFJPVE9fUE9SVFMgPSB7XG4gICAgW0FwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUF06IDgwODAsXG4gICAgW0FwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFNdOiA0NDMzLFxuICB9O1xuXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfSE9TVE5BTUUgPSAncmVuZGVycXVldWUnO1xuXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfRE9NQUlOX05BTUUgPSAnYXdzLXJmZGsuY29tJztcblxuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBERUZBVUxUX1ZQQ19TVUJORVRTX0FMQjogU3VibmV0U2VsZWN0aW9uID0geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9OQVQsIG9uZVBlckF6OiB0cnVlIH07XG5cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgREVGQVVMVF9WUENfU1VCTkVUU19PVEhFUjogU3VibmV0U2VsZWN0aW9uID0geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9OQVQgfTtcblxuICAvKipcbiAgKiBUaGUgbWluaW11bSBEZWFkbGluZSB2ZXJzaW9uIHJlcXVpcmVkIGZvciB0aGUgUmVtb3RlIENvbm5lY3Rpb24gU2VydmVyIHRvIHN1cHBvcnQgbG9hZC1iYWxhbmNpbmdcbiAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgTUlOSU1VTV9MT0FEX0JBTEFOQ0lOR19WRVJTSU9OID0gbmV3IFZlcnNpb24oWzEwLCAxLCAxMCwgMF0pO1xuXG4gIC8qKlxuICAgKiBSZWd1bGFyIGV4cHJlc3Npb24gdGhhdCB2YWxpZGF0ZXMgYSBob3N0bmFtZSAocG9ydGlvbiBpbiBmcm9udCBvZiB0aGUgc3ViZG9tYWluKS5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFJFX1ZBTElEX0hPU1ROQU1FID0gL15bYS16XSg/OlthLXowLTktXXswLDYxfVthLXowLTldKT8kL2k7XG5cbiAgLyoqXG4gICAqIEluZm9ybWF0aW9uIGZvciB0aGUgUkNTIHVzZXIuXG4gICAqL1xuXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFJDU19VU0VSID0geyB1aWQ6IDEwMDAsIGdpZDogMTAwMCwgdXNlcm5hbWU6ICdlYzItdXNlcicgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IElQcmluY2lwYWw7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVyOiBDbHVzdGVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGVuZHBvaW50OiBDb25uZWN0YWJsZUFwcGxpY2F0aW9uRW5kcG9pbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBsb2FkQmFsYW5jZXI6IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFzZzogQXV0b1NjYWxpbmdHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2ZXJzaW9uOiBJVmVyc2lvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY2VydENoYWluPzogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcmVwb3NpdG9yeTogSVJlcG9zaXRvcnk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJhY2tlbmRDb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgU0VQIHBvbGljaWVzIGhhdmUgYmVlbiBhZGRlZFxuICAgKi9cbiAgcHJpdmF0ZSBoYXZlQWRkZWRTRVBQb2xpY2llczogYm9vbGVhbiA9IGZhbHNlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIFJlc291cmNlIFRyYWNrZXIgcG9saWNpZXMgaGF2ZSBiZWVuIGFkZGVkXG4gICAqL1xuICBwcml2YXRlIGhhdmVBZGRlZFJlc291cmNlVHJhY2tlclBvbGljaWVzOiBib29sZWFuID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIFRoZSBsb2cgZ3JvdXAgd2hlcmUgdGhlIFJDUyBjb250YWluZXIgd2lsbCBsb2cgdG9cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgbG9nR3JvdXA6IElMb2dHcm91cDtcblxuICAvKipcbiAgICogSW5zdGFuY2Ugb2YgdGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZWQgRUMyIHNlcnZpY2UgcGF0dGVybi5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcGF0dGVybjogQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRFYzJTZXJ2aWNlO1xuXG4gIC8qKlxuICAgKiBUaGUgY2VydGlmaWNhdGUgdXNlZCBieSB0aGUgQUxCIGZvciBleHRlcm5hbCBUcmFmZmljXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGNsaWVudENlcnQ/OiBJQ2VydGlmaWNhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25uZWN0aW9uIG9iamVjdCB0aGF0IGNvbnRhaW5zIHRoZSBsb2dpYyBmb3IgaG93IGNsaWVudHMgY2FuIGNvbm5lY3QgdG8gdGhlIFJlbmRlciBRdWV1ZS5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcnFDb25uZWN0aW9uOiBSZW5kZXJRdWV1ZUNvbm5lY3Rpb247XG5cbiAgLyoqXG4gICAqIENvbnN0cmFpbnRzIG9uIHRoZSBudW1iZXIgb2YgRGVhZGxpbmUgUkNTIHByb2Nlc3NlcyB0aGF0IGNhbiBiZSBydW4gYXMgcGFydCBvZiB0aGlzXG4gICAqIFJlbmRlclF1ZXVlLlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSByZW5kZXJRdWV1ZVNpemU6IFJlbmRlclF1ZXVlU2l6ZUNvbnN0cmFpbnRzO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdGVuZXIgb24gdGhlIEFMQiB0aGF0IGlzIHJlZGlyZWN0aW5nIHRyYWZmaWMgdG8gdGhlIFJDUy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgbGlzdGVuZXI6IEFwcGxpY2F0aW9uTGlzdGVuZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBFQ1MgdGFzayBmb3IgdGhlIFJDUy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgdGFza0RlZmluaXRpb246IEVjMlRhc2tEZWZpbml0aW9uO1xuXG4gIC8qKlxuICAgKiBEZXBlbmQgb24gdGhpcyB0byBlbnN1cmUgdGhhdCBFQ1MgU2VydmljZSBpcyBzdGFibGUuXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGVjc1NlcnZpY2VTdGFiaWxpemVkOiBXYWl0Rm9yU3RhYmxlU2VydmljZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMucmVwb3NpdG9yeSA9IHByb3BzLnJlcG9zaXRvcnk7XG4gICAgdGhpcy5yZW5kZXJRdWV1ZVNpemUgPSBwcm9wcz8ucmVuZGVyUXVldWVTaXplID8/IHttaW46IDEsIG1heDogMX07XG5cbiAgICBpZiAocHJvcHMudmVyc2lvbi5pc0xlc3NUaGFuKFJlbmRlclF1ZXVlLk1JTklNVU1fTE9BRF9CQUxBTkNJTkdfVkVSU0lPTikpIHtcbiAgICAgIC8vIERlYWRsaW5lIHZlcnNpb25zIGVhcmxpZXIgdGhhbiAxMC4xLjEwIGRvIG5vdCBzdXBwb3J0IGhvcml6b250YWwgc2NhbGluZyBiZWhpbmQgYSBsb2FkLWJhbGFuY2VyLCBzbyB3ZSBsaW1pdCB0byBhdCBtb3N0IG9uZSBpbnN0YW5jZVxuICAgICAgaWYgKCh0aGlzLnJlbmRlclF1ZXVlU2l6ZS5taW4gPz8gMCkgPiAxKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgcmVuZGVyUXVldWVTaXplLm1pbiBmb3IgRGVhZGxpbmUgdmVyc2lvbiBsZXNzIHRoYW4gJHtSZW5kZXJRdWV1ZS5NSU5JTVVNX0xPQURfQkFMQU5DSU5HX1ZFUlNJT04udG9TdHJpbmcoKX0gY2Fubm90IGJlIGdyZWF0ZXIgdGhhbiAxIC0gZ290ICR7dGhpcy5yZW5kZXJRdWV1ZVNpemUubWlufWApO1xuICAgICAgfVxuICAgICAgaWYgKCh0aGlzLnJlbmRlclF1ZXVlU2l6ZS5kZXNpcmVkID8/IDApID4gMSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHJlbmRlclF1ZXVlU2l6ZS5kZXNpcmVkIGZvciBEZWFkbGluZSB2ZXJzaW9uIGxlc3MgdGhhbiAke1JlbmRlclF1ZXVlLk1JTklNVU1fTE9BRF9CQUxBTkNJTkdfVkVSU0lPTi50b1N0cmluZygpfSBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIDEgLSBnb3QgJHt0aGlzLnJlbmRlclF1ZXVlU2l6ZS5kZXNpcmVkfWApO1xuICAgICAgfVxuICAgICAgaWYgKCh0aGlzLnJlbmRlclF1ZXVlU2l6ZS5tYXggPz8gMCkgPiAxKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgcmVuZGVyUXVldWVTaXplLm1heCBmb3IgRGVhZGxpbmUgdmVyc2lvbiBsZXNzIHRoYW4gJHtSZW5kZXJRdWV1ZS5NSU5JTVVNX0xPQURfQkFMQU5DSU5HX1ZFUlNJT04udG9TdHJpbmcoKX0gY2Fubm90IGJlIGdyZWF0ZXIgdGhhbiAxIC0gZ290ICR7dGhpcy5yZW5kZXJRdWV1ZVNpemUubWF4fWApO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMudmVyc2lvbiA9IHByb3BzPy52ZXJzaW9uO1xuXG4gICAgY29uc3QgZXh0ZXJuYWxQcm90b2NvbCA9IHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uPy5leHRlcm5hbFRMUz8uZW5hYmxlZCA9PT0gZmFsc2UgPyBBcHBsaWNhdGlvblByb3RvY29sLkhUVFAgOiBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTO1xuICAgIGxldCBsb2FkQmFsYW5jZXJGUUROOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gICAgbGV0IGRvbWFpblpvbmU6IElIb3N0ZWRab25lIHwgdW5kZWZpbmVkO1xuXG4gICAgaWYgKCBleHRlcm5hbFByb3RvY29sID09PSAgQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUyApIHtcbiAgICAgIGNvbnN0IHRsc0luZm8gPSB0aGlzLmdldE9yQ3JlYXRlVGxzSW5mbyhwcm9wcyk7XG5cbiAgICAgIHRoaXMuY2VydENoYWluID0gdGxzSW5mby5jZXJ0Q2hhaW47XG4gICAgICB0aGlzLmNsaWVudENlcnQgPSB0bHNJbmZvLnNlcnZlckNlcnQ7XG4gICAgICBsb2FkQmFsYW5jZXJGUUROID0gdGxzSW5mby5mdWxseVF1YWxpZmllZERvbWFpbk5hbWU7XG4gICAgICBkb21haW5ab25lID0gdGxzSW5mby5kb21haW5ab25lO1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAocHJvcHMuaG9zdG5hbWUpIHtcbiAgICAgICAgbG9hZEJhbGFuY2VyRlFETiA9IHRoaXMuZ2VuZXJhdGVGdWxseVF1YWxpZmllZERvbWFpbk5hbWUocHJvcHMuaG9zdG5hbWUuem9uZSwgcHJvcHMuaG9zdG5hbWUuaG9zdG5hbWUpO1xuICAgICAgICBkb21haW5ab25lID0gcHJvcHMuaG9zdG5hbWUuem9uZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLnZlcnNpb24gPSBwcm9wcy52ZXJzaW9uO1xuXG4gICAgY29uc3QgaW50ZXJuYWxQcm90b2NvbCA9IHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uPy5pbnRlcm5hbFByb3RvY29sID8/IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFM7XG5cbiAgICB0aGlzLmNsdXN0ZXIgPSBuZXcgQ2x1c3Rlcih0aGlzLCAnQ2x1c3RlcicsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWluQ2FwYWNpdHkgPSBwcm9wcy5yZW5kZXJRdWV1ZVNpemU/Lm1pbiA/PyAxO1xuICAgIGlmIChtaW5DYXBhY2l0eSA8IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgcmVuZGVyUXVldWVTaXplLm1pbiBjYXBhY2l0eSBtdXN0IGJlIGF0IGxlYXN0IDE6IGdvdCAke21pbkNhcGFjaXR5fWApO1xuICAgIH1cbiAgICBjb25zdCBtYXhDYXBhY2l0eSA9IHRoaXMucmVuZGVyUXVldWVTaXplLm1heCA/PyB0aGlzLnJlbmRlclF1ZXVlU2l6ZT8uZGVzaXJlZDtcbiAgICBpZiAodGhpcy5yZW5kZXJRdWV1ZVNpemU/LmRlc2lyZWQgJiYgbWF4Q2FwYWNpdHkgJiYgdGhpcy5yZW5kZXJRdWV1ZVNpemU/LmRlc2lyZWQgPiBtYXhDYXBhY2l0eSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGByZW5kZXJRdWV1ZVNpemUuZGVzaXJlZCBjYXBhY2l0eSBjYW5ub3QgYmUgbW9yZSB0aGFuICR7bWF4Q2FwYWNpdHl9OiBnb3QgJHt0aGlzLnJlbmRlclF1ZXVlU2l6ZS5kZXNpcmVkfWApO1xuICAgIH1cbiAgICB0aGlzLmFzZyA9IHRoaXMuY2x1c3Rlci5hZGRDYXBhY2l0eSgnUkNTIENhcGFjaXR5Jywge1xuICAgICAgdnBjU3VibmV0czogcHJvcHMudnBjU3VibmV0cyA/PyBSZW5kZXJRdWV1ZS5ERUZBVUxUX1ZQQ19TVUJORVRTX09USEVSLFxuICAgICAgaW5zdGFuY2VUeXBlOiBwcm9wcy5pbnN0YW5jZVR5cGUgPz8gbmV3IEluc3RhbmNlVHlwZSgnYzUubGFyZ2UnKSxcbiAgICAgIG1pbkNhcGFjaXR5LFxuICAgICAgZGVzaXJlZENhcGFjaXR5OiB0aGlzLnJlbmRlclF1ZXVlU2l6ZT8uZGVzaXJlZCxcbiAgICAgIG1heENhcGFjaXR5LFxuICAgICAgYmxvY2tEZXZpY2VzOiBbe1xuICAgICAgICBkZXZpY2VOYW1lOiAnL2Rldi94dmRhJyxcbiAgICAgICAgLy8gU2VlOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9lY3MtYW1pLXN0b3JhZ2UtY29uZmlnLmh0bWxcbiAgICAgICAgLy8gV2Ugd2FudCB0aGUgdm9sdW1lIHRvIGJlIGVuY3J5cHRlZC4gVGhlIGRlZmF1bHQgQU1JIHNpemUgaXMgMzAtR2lCLlxuICAgICAgICB2b2x1bWU6IEJsb2NrRGV2aWNlVm9sdW1lLmVicygzMCwgeyBlbmNyeXB0ZWQ6IHRydWUgfSksXG4gICAgICB9XSxcbiAgICAgIHVwZGF0ZVR5cGU6IHVuZGVmaW5lZCwgLy8gV29ya2Fyb3VuZCAtLSBTZWU6IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvMTE1ODFcbiAgICAgIHVwZGF0ZVBvbGljeTogVXBkYXRlUG9saWN5LnJvbGxpbmdVcGRhdGUoKSxcbiAgICAgIC8vIGFkZENhcGFjaXR5IGRvZXNuJ3Qgc3BlY2lmaWNhbGx5IHRha2UgYSBzZWN1cml0eUdyb3VwLCBidXQgaXQgcGFzc2VzIG9uIGl0cyBwcm9wZXJ0aWVzIHRvIHRoZSBBU0cgaXQgY3JlYXRlcyxcbiAgICAgIC8vIHNvIHRoaXMgc2VjdXJpdHkgZ3JvdXAgd2lsbCBnZXQgYXBwbGllZCB0aGVyZVxuICAgICAgLy8gQHRzLWlnbm9yZVxuICAgICAgc2VjdXJpdHlHcm91cDogcHJvcHMuc2VjdXJpdHlHcm91cHM/LmJhY2tlbmQsXG4gICAgfSk7XG5cbiAgICB0aGlzLmJhY2tlbmRDb25uZWN0aW9ucyA9IHRoaXMuYXNnLmNvbm5lY3Rpb25zO1xuXG4gICAgLyoqXG4gICAgICogVGhlIEVDUy1vcHRpbWl6ZWQgQU1JIHRoYXQgaXMgZGVmYXVsdGVkIHRvIHdoZW4gYWRkaW5nIGNhcGFjaXR5IHRvIGEgY2x1c3RlciBkb2VzIG5vdCBpbmNsdWRlIHRoZSBhd3NjbGkgb3IgdW56aXBcbiAgICAgKiBwYWNrYWdlcyBhcyBpcyB0aGUgY2FzZSB3aXRoIHRoZSBzdGFuZGFyZCBBbWF6b24gTGludXggQU1JLiBUaGVzZSBhcmUgcmVxdWlyZWQgYnkgUkZESyBzY3JpcHRzIHRvIGNvbmZpZ3VyZSB0aGVcbiAgICAgKiBkaXJlY3QgY29ubmVjdGlvbiBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2VzLlxuICAgICAqL1xuICAgIHRoaXMuYXNnLnVzZXJEYXRhLmFkZENvbW1hbmRzKFxuICAgICAgJ3l1bSBpbnN0YWxsIC15cSBhd3NjbGkgdW56aXAnLFxuICAgICk7XG4gICAgaWYgKHByb3BzLmVuYWJsZUxvY2FsRmlsZUNhY2hpbmcgPz8gZmFsc2UpIHtcbiAgICAgIC8vIEhhcyB0byBiZSBkb25lIGJlZm9yZSBhbnkgZmlsZXN5c3RlbXMgbW91bnQuXG4gICAgICB0aGlzLmVuYWJsZUZpbGVjYWNoaW5nKHRoaXMuYXNnKTtcbiAgICB9XG5cbiAgICBjb25zdCBleHRlcm5hbFBvcnROdW1iZXIgPSBSZW5kZXJRdWV1ZS5SQ1NfUFJPVE9fUE9SVFNbZXh0ZXJuYWxQcm90b2NvbF07XG4gICAgY29uc3QgaW50ZXJuYWxQb3J0TnVtYmVyID0gUmVuZGVyUXVldWUuUkNTX1BST1RPX1BPUlRTW2ludGVybmFsUHJvdG9jb2xdO1xuXG4gICAgdGhpcy5sb2dHcm91cCA9IExvZ0dyb3VwRmFjdG9yeS5jcmVhdGVPckZldGNoKHRoaXMsICdMb2dHcm91cFdyYXBwZXInLCBpZCwge1xuICAgICAgbG9nR3JvdXBQcmVmaXg6ICcvcmVuZGVyZmFybS8nLFxuICAgICAgLi4ucHJvcHMubG9nR3JvdXBQcm9wcyxcbiAgICB9KTtcbiAgICB0aGlzLmxvZ0dyb3VwLmdyYW50V3JpdGUodGhpcy5hc2cpO1xuXG4gICAgaWYgKHByb3BzLnJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5lbmFibGVkKSB7XG4gICAgICBjb25zdCBlcnJvcnMgPSBbXTtcbiAgICAgIGlmIChwcm9wcy52ZXJzaW9uLmlzTGVzc1RoYW4oVmVyc2lvbi5NSU5JTVVNX1NFQ1JFVFNfTUFOQUdFTUVOVF9WRVJTSU9OKSkge1xuICAgICAgICBlcnJvcnMucHVzaChgVGhlIHN1cHBsaWVkIERlYWRsaW5lIHZlcnNpb24gKCR7cHJvcHMudmVyc2lvbi52ZXJzaW9uU3RyaW5nfSkgZG9lcyBub3Qgc3VwcG9ydCBEZWFkbGluZSBTZWNyZXRzIE1hbmFnZW1lbnQgaW4gUkZESy4gRWl0aGVyIHVwZ3JhZGUgRGVhZGxpbmUgdG8gdGhlIG1pbmltdW0gcmVxdWlyZWQgdmVyc2lvbiAoJHtWZXJzaW9uLk1JTklNVU1fU0VDUkVUU19NQU5BR0VNRU5UX1ZFUlNJT04udmVyc2lvblN0cmluZ30pIG9yIGRpc2FibGUgdGhlIGZlYXR1cmUgaW4gdGhlIFJlcG9zaXRvcnkncyBjb25zdHJ1Y3QgcHJvcGVydGllcy5gKTtcbiAgICAgIH1cbiAgICAgIGlmIChwcm9wcy5yZXBvc2l0b3J5LnNlY3JldHNNYW5hZ2VtZW50U2V0dGluZ3MuY3JlZGVudGlhbHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICBlcnJvcnMucHVzaCgnVGhlIFJlcG9zaXRvcnkgZG9lcyBub3QgaGF2ZSBTZWNyZXRzIE1hbmFnZW1lbnQgY3JlZGVudGlhbHMnKTtcbiAgICAgIH1cbiAgICAgIGlmIChpbnRlcm5hbFByb3RvY29sICE9PSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTKSB7XG4gICAgICAgIGVycm9ycy5wdXNoKCdUaGUgaW50ZXJuYWwgcHJvdG9jb2wgb24gdGhlIFJlbmRlciBRdWV1ZSBpcyBub3QgSFRUUFMuJyk7XG4gICAgICB9XG4gICAgICBpZiAoZXh0ZXJuYWxQcm90b2NvbCAhPT0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUykge1xuICAgICAgICBlcnJvcnMucHVzaCgnRXh0ZXJuYWwgVExTIG9uIHRoZSBSZW5kZXIgUXVldWUgaXMgbm90IGVuYWJsZWQuJyk7XG4gICAgICB9XG4gICAgICBpZiAoZXJyb3JzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBEZWFkbGluZSBTZWNyZXRzIE1hbmFnZW1lbnQgaXMgZW5hYmxlZCBvbiB0aGUgc3VwcGxpZWQgUmVwb3NpdG9yeSBidXQgY2Fubm90IGJlIGVuYWJsZWQgb24gdGhlIFJlbmRlciBRdWV1ZSBmb3IgdGhlIGZvbGxvd2luZyByZWFzb25zOlxcbiR7ZXJyb3JzLmpvaW4oJ1xcbicpfWApO1xuICAgICAgfVxuICAgIH1cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IHRoaXMuY3JlYXRlVGFza0RlZmluaXRpb24oe1xuICAgICAgaW1hZ2U6IHByb3BzLmltYWdlcy5yZW1vdGVDb25uZWN0aW9uU2VydmVyLFxuICAgICAgcG9ydE51bWJlcjogaW50ZXJuYWxQb3J0TnVtYmVyLFxuICAgICAgcHJvdG9jb2w6IGludGVybmFsUHJvdG9jb2wsXG4gICAgICByZXBvc2l0b3J5OiBwcm9wcy5yZXBvc2l0b3J5LFxuICAgICAgcnVuQXNVc2VyOiBSZW5kZXJRdWV1ZS5SQ1NfVVNFUixcbiAgICAgIHNlY3JldHNNYW5hZ2VtZW50T3B0aW9uczogcHJvcHMucmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmVuYWJsZWQgPyB7XG4gICAgICAgIGNyZWRlbnRpYWxzOiBwcm9wcy5yZXBvc2l0b3J5LnNlY3JldHNNYW5hZ2VtZW50U2V0dGluZ3MuY3JlZGVudGlhbHMhLFxuICAgICAgICBwb3NpeFVzZXJuYW1lOiBSZW5kZXJRdWV1ZS5SQ1NfVVNFUi51c2VybmFtZSxcbiAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gICAgdGhpcy50YXNrRGVmaW5pdGlvbiA9IHRhc2tEZWZpbml0aW9uO1xuXG4gICAgLy8gVGhlIGZ1bGx5LXF1YWxpZmllZCBkb21haW4gbmFtZSB0byB1c2UgZm9yIHRoZSBBTEJcblxuICAgIGNvbnN0IGxvYWRCYWxhbmNlciA9IG5ldyBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcih0aGlzLCAnTEInLCB7XG4gICAgICB2cGM6IHRoaXMuY2x1c3Rlci52cGMsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy52cGNTdWJuZXRzQWxiID8/IFJlbmRlclF1ZXVlLkRFRkFVTFRfVlBDX1NVQk5FVFNfQUxCLFxuICAgICAgaW50ZXJuZXRGYWNpbmc6IGZhbHNlLFxuICAgICAgZGVsZXRpb25Qcm90ZWN0aW9uOiBwcm9wcy5kZWxldGlvblByb3RlY3Rpb24gPz8gdHJ1ZSxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IHByb3BzLnNlY3VyaXR5R3JvdXBzPy5mcm9udGVuZCxcbiAgICB9KTtcblxuICAgIHRoaXMucGF0dGVybiA9IG5ldyBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZEVjMlNlcnZpY2UodGhpcywgJ0FsYkVjMlNlcnZpY2VQYXR0ZXJuJywge1xuICAgICAgY2VydGlmaWNhdGU6IHRoaXMuY2xpZW50Q2VydCxcbiAgICAgIGNsdXN0ZXI6IHRoaXMuY2x1c3RlcixcbiAgICAgIGRlc2lyZWRDb3VudDogdGhpcy5yZW5kZXJRdWV1ZVNpemU/LmRlc2lyZWQsXG4gICAgICBkb21haW5ab25lLFxuICAgICAgZG9tYWluTmFtZTogbG9hZEJhbGFuY2VyRlFETixcbiAgICAgIGxpc3RlbmVyUG9ydDogZXh0ZXJuYWxQb3J0TnVtYmVyLFxuICAgICAgbG9hZEJhbGFuY2VyLFxuICAgICAgcHJvdG9jb2w6IGV4dGVybmFsUHJvdG9jb2wsXG4gICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICAgIC8vIFRoaXMgaXMgcmVxdWlyZWQgdG8gcmlnaHQtc2l6ZSBvdXIgaG9zdCBjYXBhY2l0eSBhbmQgbm90IGhhdmUgdGhlIEVDUyBzZXJ2aWNlIGJsb2NrIG9uIHVwZGF0ZXMuIFdlIHNldCBhIG1lbW9yeVxuICAgICAgLy8gcmVzZXJ2YXRpb24sIGJ1dCBubyBtZW1vcnkgbGltaXQgb24gdGhlIGNvbnRhaW5lci4gVGhpcyBhbGxvd3MgdGhlIGNvbnRhaW5lcidzIG1lbW9yeSB1c2FnZSB0byBncm93IHVuYm91bmRlZC5cbiAgICAgIC8vIFdlIHdhbnQgMToxIGNvbnRhaW5lciB0byBjb250YWluZXIgaW5zdGFuY2VzIHRvIG5vdCBvdmVyLXNwZW5kLCBidXQgdGhpcyBjb21lcyBhdCB0aGUgcHJpY2Ugb2YgZG93bi10aW1lIGR1cmluZ1xuICAgICAgLy8gY2xvdWRmb3JtYXRpb24gdXBkYXRlcy5cbiAgICAgIG1pbkhlYWx0aHlQZXJjZW50OiAwLFxuICAgICAgbWF4SGVhbHRoeVBlcmNlbnQ6IDEwMCxcbiAgICAgIC8vIFRoaXMgaXMgcmVxdWlyZWQgdG8gZW5zdXJlIHRoYXQgdGhlIEFMQiBsaXN0ZW5lcidzIHNlY3VyaXR5IGdyb3VwIGRvZXMgbm90IGFsbG93IGFueSBpbmdyZXNzIGJ5IGRlZmF1bHQuXG4gICAgICBvcGVuTGlzdGVuZXI6IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgLy8gQW4gZXhwbGljaXQgZGVwZW5kZW5jeSBpcyByZXF1aXJlZCBmcm9tIHRoZSBTZXJ2aWNlIHRvIHRoZSBDbGllbnQgY2VydGlmaWNhdGVcbiAgICAvLyBPdGhlcndpc2UgY2xvdWQgZm9ybWF0aW9uIHdpbGwgdHJ5IHRvIHJlbW92ZSB0aGUgY2VydCBiZWZvcmUgdGhlIEFMQiB1c2luZyBpdCBpcyBkaXNwb3NlZC5cbiAgICBpZiAodGhpcy5jbGllbnRDZXJ0KSB7XG4gICAgICB0aGlzLnBhdHRlcm4ubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMuY2xpZW50Q2VydCk7XG4gICAgfVxuXG4gICAgLy8gQW4gZXhwbGljaXQgZGVwZW5kZW5jeSBpcyByZXF1aXJlZCBmcm9tIHRoZSBzZXJ2aWNlIHRvIHRoZSBBU0cgcHJvdmlkaW5nIGl0cyBjYXBhY2l0eS5cbiAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1hdHRyaWJ1dGUtZGVwZW5kc29uLmh0bWxcbiAgICB0aGlzLnBhdHRlcm4uc2VydmljZS5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5hc2cpO1xuXG4gICAgdGhpcy5sb2FkQmFsYW5jZXIgPSB0aGlzLnBhdHRlcm4ubG9hZEJhbGFuY2VyO1xuICAgIC8vIEVuYWJsaW5nIGRyb3BwaW5nIG9mIGludmFsaWQgSFRUUCBoZWFkZXIgZmllbGRzIG9uIHRoZSBsb2FkIGJhbGFuY2VyIHRvIHByZXZlbnQgaHR0cCBzbXVnZ2xpbmcgYXR0YWNrcy5cbiAgICB0aGlzLmxvYWRCYWxhbmNlci5zZXRBdHRyaWJ1dGUoJ3JvdXRpbmcuaHR0cC5kcm9wX2ludmFsaWRfaGVhZGVyX2ZpZWxkcy5lbmFibGVkJywgJ3RydWUnKTtcblxuICAgIGlmIChwcm9wcy5hY2Nlc3NMb2dzKSB7XG4gICAgICBjb25zdCBhY2Nlc3NMb2dzQnVja2V0ID0gcHJvcHMuYWNjZXNzTG9ncy5kZXN0aW5hdGlvbkJ1Y2tldDtcblxuICAgICAgLy8gUG9saWNpZXMgYXJlIGFwcGxpZWQgYWNjb3JkaW5nIHRvXG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItYWNjZXNzLWxvZ3MuaHRtbFxuICAgICAgYWNjZXNzTG9nc0J1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KCBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QnXSxcbiAgICAgICAgcHJpbmNpcGFsczogW25ldyBTZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgIHJlc291cmNlczogW2Ake2FjY2Vzc0xvZ3NCdWNrZXQuYnVja2V0QXJufS8qYF0sXG4gICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICdzMzp4LWFtei1hY2wnOiAnYnVja2V0LW93bmVyLWZ1bGwtY29udHJvbCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcbiAgICAgIGFjY2Vzc0xvZ3NCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWyAnczM6R2V0QnVja2V0QWNsJyBdLFxuICAgICAgICBwcmluY2lwYWxzOiBbIG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgIHJlc291cmNlczogWyBhY2Nlc3NMb2dzQnVja2V0LmJ1Y2tldEFybiBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0aGlzLmxvYWRCYWxhbmNlci5sb2dBY2Nlc3NMb2dzKFxuICAgICAgICBhY2Nlc3NMb2dzQnVja2V0LFxuICAgICAgICBwcm9wcy5hY2Nlc3NMb2dzLnByZWZpeCk7XG4gICAgfVxuXG4gICAgLy8gRW5zdXJlIHRhc2tzIGFyZSBydW4gb24gc2VwYXJhdGUgY29udGFpbmVyIGluc3RhbmNlc1xuICAgIHRoaXMucGF0dGVybi5zZXJ2aWNlLmFkZFBsYWNlbWVudENvbnN0cmFpbnRzKFBsYWNlbWVudENvbnN0cmFpbnQuZGlzdGluY3RJbnN0YW5jZXMoKSk7XG5cbiAgICAvKipcbiAgICAgKiBVc2VzIGFuIGVzY2FwZS1oYXRjaCB0byBzZXQgdGhlIHRhcmdldCBncm91cCBwcm90b2NvbCB0byBIVFRQUy4gV2UgY2Fubm90IGNvbmZpZ3VyZSBzZXJ2ZXIgY2VydGlmaWNhdGVcbiAgICAgKiB2YWxpZGF0aW9uLCBidXQgYXQgbGVhc3QgdHJhZmZpYyBpcyBlbmNyeXB0ZWQgYW5kIHRlcm1pbmF0ZWQgYXQgdGhlIGFwcGxpY2F0aW9uIGxheWVyLlxuICAgICAqL1xuICAgIGNvbnN0IGxpc3RlbmVyID0gdGhpcy5sb2FkQmFsYW5jZXIubm9kZS5maW5kQ2hpbGQoJ1B1YmxpY0xpc3RlbmVyJyk7XG4gICAgdGhpcy5saXN0ZW5lciA9IGxpc3RlbmVyIGFzIEFwcGxpY2F0aW9uTGlzdGVuZXI7XG4gICAgY29uc3QgdGFyZ2V0R3JvdXAgPSBsaXN0ZW5lci5ub2RlLmZpbmRDaGlsZCgnRUNTR3JvdXAnKSBhcyBBcHBsaWNhdGlvblRhcmdldEdyb3VwO1xuICAgIGNvbnN0IHRhcmdldEdyb3VwUmVzb3VyY2UgPSB0YXJnZXRHcm91cC5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5UYXJnZXRHcm91cDtcbiAgICB0YXJnZXRHcm91cFJlc291cmNlLnByb3RvY29sID0gQXBwbGljYXRpb25Qcm90b2NvbFtpbnRlcm5hbFByb3RvY29sXTtcbiAgICB0YXJnZXRHcm91cFJlc291cmNlLnBvcnQgPSBpbnRlcm5hbFBvcnROdW1iZXI7XG5cbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGFza0RlZmluaXRpb24udGFza1JvbGU7XG5cbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gbmV3IENvbm5lY3Rpb25zKHtcbiAgICAgIGRlZmF1bHRQb3J0OiBQb3J0LnRjcChleHRlcm5hbFBvcnROdW1iZXIpLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHRoaXMucGF0dGVybi5sb2FkQmFsYW5jZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMsXG4gICAgfSk7XG5cbiAgICB0aGlzLmVuZHBvaW50ID0gbmV3IENvbm5lY3RhYmxlQXBwbGljYXRpb25FbmRwb2ludCh7XG4gICAgICBhZGRyZXNzOiBsb2FkQmFsYW5jZXJGUUROID8/IHRoaXMucGF0dGVybi5sb2FkQmFsYW5jZXIubG9hZEJhbGFuY2VyRG5zTmFtZSxcbiAgICAgIHBvcnQ6IGV4dGVybmFsUG9ydE51bWJlcixcbiAgICAgIGNvbm5lY3Rpb25zOiB0aGlzLmNvbm5lY3Rpb25zLFxuICAgICAgcHJvdG9jb2w6IGV4dGVybmFsUHJvdG9jb2wsXG4gICAgfSk7XG5cbiAgICBpZiAoIGV4dGVybmFsUHJvdG9jb2wgPT09IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUCApIHtcbiAgICAgIHRoaXMucnFDb25uZWN0aW9uID0gUmVuZGVyUXVldWVDb25uZWN0aW9uLmZvckh0dHAoe1xuICAgICAgICBlbmRwb2ludDogdGhpcy5lbmRwb2ludCxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJxQ29ubmVjdGlvbiA9IFJlbmRlclF1ZXVlQ29ubmVjdGlvbi5mb3JIdHRwcyh7XG4gICAgICAgIGVuZHBvaW50OiB0aGlzLmVuZHBvaW50LFxuICAgICAgICBjYUNlcnQ6IHRoaXMuY2VydENoYWluISxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHByb3BzLnJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5jcmVkZW50aWFscz8uZ3JhbnRSZWFkKHRoaXMpO1xuXG4gICAgdGhpcy5lY3NTZXJ2aWNlU3RhYmlsaXplZCA9IG5ldyBXYWl0Rm9yU3RhYmxlU2VydmljZSh0aGlzLCAnV2FpdEZvclN0YWJsZVNlcnZpY2UnLCB7XG4gICAgICBzZXJ2aWNlOiB0aGlzLnBhdHRlcm4uc2VydmljZSxcbiAgICB9KTtcblxuICAgIHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQgPSB0YXNrRGVmaW5pdGlvbjtcblxuICAgIC8vIFRhZyBkZXBsb3llZCByZXNvdXJjZXMgd2l0aCBSRkRLIG1ldGEtZGF0YVxuICAgIHRhZ0NvbnN0cnVjdCh0aGlzKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBvblZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCB2YWxpZGF0aW9uRXJyb3JzID0gW107XG5cbiAgICAvLyBVc2luZyB0aGUgb3V0cHV0IG9mIFZlcnNpb25RdWVyeSBhY3Jvc3Mgc3RhY2tzIGNhbiBjYXVzZSBpc3N1ZXMuIENsb3VkRm9ybWF0aW9uIHN0YWNrIG91dHB1dHMgY2Fubm90IGNoYW5nZSBpZlxuICAgIC8vIGEgcmVzb3VyY2UgaW4gYW5vdGhlciBzdGFjayBpcyByZWZlcmVuY2luZyBpdC5cbiAgICBpZiAodGhpcy52ZXJzaW9uIGluc3RhbmNlb2YgVmVyc2lvblF1ZXJ5KSB7XG4gICAgICBjb25zdCB2ZXJzaW9uU3RhY2sgPSBTdGFjay5vZih0aGlzLnZlcnNpb24pO1xuICAgICAgY29uc3QgdGhpc1N0YWNrID0gU3RhY2sub2YodGhpcyk7XG4gICAgICBpZiAodmVyc2lvblN0YWNrICE9IHRoaXNTdGFjaykge1xuICAgICAgICB2YWxpZGF0aW9uRXJyb3JzLnB1c2goJ0EgVmVyc2lvblF1ZXJ5IGNhbiBub3QgYmUgc3VwcGxpZWQgZnJvbSBhIGRpZmZlcmVudCBzdGFjaycpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB2YWxpZGF0aW9uRXJyb3JzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGNvbmZpZ3VyZUNsaWVudEVDUyhwYXJhbTogRUNTQ29ubmVjdE9wdGlvbnMpOiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSB7XG4gICAgcGFyYW0uaG9zdHMuZm9yRWFjaCggaG9zdCA9PiB0aGlzLmFkZENoaWxkRGVwZW5kZW5jeShob3N0KSApO1xuICAgIHJldHVybiB0aGlzLnJxQ29ubmVjdGlvbi5jb25maWd1cmVDbGllbnRFQ1MocGFyYW0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHBhcmFtOiBJbnN0YW5jZUNvbm5lY3RPcHRpb25zKTogdm9pZCB7XG4gICAgdGhpcy5hZGRDaGlsZERlcGVuZGVuY3kocGFyYW0uaG9zdCk7XG4gICAgdGhpcy5ycUNvbm5lY3Rpb24uY29uZmlndXJlQ2xpZW50SW5zdGFuY2UocGFyYW0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGNvbmZpZ3VyZVNlY3JldHNNYW5hZ2VtZW50QXV0b1JlZ2lzdHJhdGlvbihwcm9wczogU3VibmV0SWRlbnRpdHlSZWdpc3RyYXRpb25TZXR0aW5nc1Byb3BzKSB7XG4gICAgaWYgKCF0aGlzLnJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5lbmFibGVkKSB7XG4gICAgICAvLyBTZWNyZXRzIG1hbmFnZW1lbnQgaXMgbm90IGVuYWJsZWQsIHNvIGRvIG5vdGhpbmdcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLmlkZW50aXR5UmVnaXN0cmF0aW9uU2V0dGluZ3MuYWRkU3VibmV0SWRlbnRpdHlSZWdpc3RyYXRpb25TZXR0aW5nKHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFNFUFBvbGljaWVzKGluY2x1ZGVSZXNvdXJjZVRyYWNrZXI6IGJvb2xlYW4gPSB0cnVlKTogdm9pZCB7XG4gICAgaWYgKCF0aGlzLmhhdmVBZGRlZFNFUFBvbGljaWVzKSB7XG4gICAgICBjb25zdCBzZXBQb2xpY3kgPSBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQVdTVGhpbmtib3hEZWFkbGluZVNwb3RFdmVudFBsdWdpbkFkbWluUG9saWN5Jyk7XG4gICAgICB0aGlzLnRhc2tEZWZpbml0aW9uLnRhc2tSb2xlLmFkZE1hbmFnZWRQb2xpY3koc2VwUG9saWN5KTtcbiAgICAgIHRoaXMuaGF2ZUFkZGVkU0VQUG9saWNpZXMgPSB0cnVlO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5oYXZlQWRkZWRSZXNvdXJjZVRyYWNrZXJQb2xpY2llcykge1xuICAgICAgaWYgKGluY2x1ZGVSZXNvdXJjZVRyYWNrZXIpIHtcbiAgICAgICAgY29uc3QgcnRQb2xpY3kgPSBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQVdTVGhpbmtib3hEZWFkbGluZVJlc291cmNlVHJhY2tlckFkbWluUG9saWN5Jyk7XG4gICAgICAgIHRoaXMudGFza0RlZmluaXRpb24udGFza1JvbGUuYWRkTWFuYWdlZFBvbGljeShydFBvbGljeSk7XG4gICAgICAgIHRoaXMuaGF2ZUFkZGVkUmVzb3VyY2VUcmFja2VyUG9saWNpZXMgPSB0cnVlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZENoaWxkRGVwZW5kZW5jeShjaGlsZDogSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIC8vIE5hcnJvd2x5IGRlZmluZSB0aGUgZGVwZW5kZW5jaWVzIHRvIHJlZHVjZSB0aGUgcHJvYmFiaWxpdHkgb2YgY3ljbGVzXG4gICAgLy8gZXg6IGN5Y2xlcyB0aGF0IGludm9sdmUgdGhlIHNlY3VyaXR5IGdyb3VwIG9mIHRoZSBSZW5kZXJRdWV1ZSAmIGNoaWxkLlxuICAgIGNoaWxkLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmxpc3RlbmVyKTtcbiAgICBjaGlsZC5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy50YXNrRGVmaW5pdGlvbik7XG4gICAgY2hpbGQubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMucGF0dGVybi5zZXJ2aWNlKTtcbiAgICBjaGlsZC5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5lY3NTZXJ2aWNlU3RhYmlsaXplZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkRnJvbnRlbmRTZWN1cml0eUdyb3VwcyguLi5zZWN1cml0eUdyb3VwczogSVNlY3VyaXR5R3JvdXBbXSk6IHZvaWQge1xuICAgIHNlY3VyaXR5R3JvdXBzLmZvckVhY2goc2VjdXJpdHlHcm91cCA9PiB0aGlzLmxvYWRCYWxhbmNlci5hZGRTZWN1cml0eUdyb3VwKHNlY3VyaXR5R3JvdXApKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkQmFja2VuZFNlY3VyaXR5R3JvdXBzKC4uLnNlY3VyaXR5R3JvdXBzOiBJU2VjdXJpdHlHcm91cFtdKTogdm9pZCB7XG4gICAgc2VjdXJpdHlHcm91cHMuZm9yRWFjaChzZWN1cml0eUdyb3VwID0+IHRoaXMuYXNnLmFkZFNlY3VyaXR5R3JvdXAoc2VjdXJpdHlHcm91cCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbmFibGVGaWxlY2FjaGluZyhhc2c6IEF1dG9TY2FsaW5nR3JvdXApOiB2b2lkIHtcbiAgICBjb25zdCBzY3JpcHQgPSBTY3JpcHRBc3NldC5mcm9tUGF0aENvbnZlbnRpb24odGhpcywgJ0ZpbGVjYWNoaW5nU2NyaXB0Jywge1xuICAgICAgb3NUeXBlOiBhc2cub3NUeXBlLFxuICAgICAgYmFzZU5hbWU6ICdlbmFibGVDYWNoZUZpbGVzZCcsXG4gICAgICByb290RGlyOiBqb2luKF9fZGlybmFtZSwgJy4uJywgJ3NjcmlwdHMnKSxcbiAgICB9KTtcbiAgICAvLyBBIGNvbW1lbnQgaW4gdXNlckRhdGEgdG8gbWFrZSB0aGlzIGVhc2llciB0byB0ZXN0LlxuICAgIGFzZy51c2VyRGF0YS5hZGRDb21tYW5kcygnIyBSZW5kZXJRdWV1ZSBmaWxlIGNhY2hpbmcgZW5hYmxlZCcpO1xuICAgIHNjcmlwdC5leGVjdXRlT24oe1xuICAgICAgaG9zdDogYXNnLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVUYXNrRGVmaW5pdGlvbihwcm9wczoge1xuICAgIGltYWdlOiBDb250YWluZXJJbWFnZSxcbiAgICBwb3J0TnVtYmVyOiBudW1iZXIsXG4gICAgcHJvdG9jb2w6IEFwcGxpY2F0aW9uUHJvdG9jb2wsXG4gICAgcmVwb3NpdG9yeTogSVJlcG9zaXRvcnksXG4gICAgcnVuQXNVc2VyPzogeyB1aWQ6IG51bWJlciwgZ2lkPzogbnVtYmVyIH0sXG4gICAgc2VjcmV0c01hbmFnZW1lbnRPcHRpb25zPzogeyBjcmVkZW50aWFsczogSVNlY3JldCwgcG9zaXhVc2VybmFtZTogc3RyaW5nIH0sXG4gIH0pIHtcbiAgICBjb25zdCB7IGltYWdlLCBwb3J0TnVtYmVyLCBwcm90b2NvbCwgcmVwb3NpdG9yeSB9ID0gcHJvcHM7XG5cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBFYzJUYXNrRGVmaW5pdGlvbih0aGlzLCAnUkNTVGFzaycpO1xuXG4gICAgLy8gTW91bnQgdGhlIHJlcG8gZmlsZXN5c3RlbSB0byBSZW5kZXJRdWV1ZS5IT1NUX1JFUE9fRlNfTU9VTlRfUEFUSFxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSByZXBvc2l0b3J5LmNvbmZpZ3VyZUNsaWVudEVDUyh7XG4gICAgICBjb250YWluZXJJbnN0YW5jZXM6IHtcbiAgICAgICAgaG9zdHM6IFt0aGlzLmFzZ10sXG4gICAgICB9LFxuICAgICAgY29udGFpbmVyczoge1xuICAgICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBlbnZpcm9ubWVudCA9IGNvbm5lY3Rpb24uY29udGFpbmVyRW52aXJvbm1lbnQ7XG5cbiAgICBpZiAocHJvdG9jb2wgPT09IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFMpIHtcbiAgICAgIC8vIEdlbmVyYXRlIGEgc2VsZi1zaWduZWQgWDUwOSBjZXJ0aWZpY2F0ZSwgcHJpdmF0ZSBrZXkgYW5kIHBhc3NwaHJhc2UgZm9yIHVzZSBieSB0aGUgUkNTIGNvbnRhaW5lcnMuXG4gICAgICAvLyBOb3RlOiB0aGUgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlciBkb2VzIG5vdCB2YWxpZGF0ZSB0aGUgY2VydGlmaWNhdGUgaW4gYW55IHdheS5cbiAgICAgIGNvbnN0IHJjc0NlcnRQZW0gPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKHRoaXMsICdUbHNDYUNlcnRQZW0nLCB7XG4gICAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgICBjbjogJ3JlbmRlcmZhcm0ubG9jYWwnLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICBjb25zdCByY3NDZXJ0UGtjcyA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQa2NzMTIodGhpcywgJ1Rsc1Jjc0NlcnRCdW5kbGUnLCB7XG4gICAgICAgIHNvdXJjZUNlcnRpZmljYXRlOiByY3NDZXJ0UGVtLFxuICAgICAgfSk7XG4gICAgICBbcmNzQ2VydFBlbS5jZXJ0LCByY3NDZXJ0UGtjcy5jZXJ0LCByY3NDZXJ0UGtjcy5wYXNzcGhyYXNlXS5mb3JFYWNoKHNlY3JldCA9PiB7XG4gICAgICAgIHNlY3JldC5ncmFudFJlYWQodGFza0RlZmluaXRpb24udGFza1JvbGUpO1xuICAgICAgfSk7XG4gICAgICBlbnZpcm9ubWVudC5SQ1NfVExTX0NBX0NFUlRfVVJJID0gcmNzQ2VydFBlbS5jZXJ0LnNlY3JldEFybjtcbiAgICAgIGVudmlyb25tZW50LlJDU19UTFNfQ0VSVF9VUkkgPSByY3NDZXJ0UGtjcy5jZXJ0LnNlY3JldEFybjtcbiAgICAgIGVudmlyb25tZW50LlJDU19UTFNfQ0VSVF9QQVNTUEhSQVNFX1VSSSA9IHJjc0NlcnRQa2NzLnBhc3NwaHJhc2Uuc2VjcmV0QXJuO1xuICAgICAgZW52aXJvbm1lbnQuUkNTX1RMU19SRVFVSVJFX0NMSUVOVF9DRVJUID0gJ25vJztcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuc2VjcmV0c01hbmFnZW1lbnRPcHRpb25zICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGVudmlyb25tZW50LlJDU19TTV9DUkVERU5USUFMU19VUkkgPSBwcm9wcy5zZWNyZXRzTWFuYWdlbWVudE9wdGlvbnMuY3JlZGVudGlhbHMuc2VjcmV0QXJuO1xuICAgIH1cblxuICAgIC8vIFdlIGNhbiBpZ25vcmUgdGhpcyBpbiB0ZXN0IGNvdmVyYWdlIGJlY2F1c2Ugd2UgYWx3YXlzIHVzZSBSZW5kZXJRdWV1ZS5SQ1NfVVNFUlxuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgY29uc3QgdXNlciA9IHByb3BzLnJ1bkFzVXNlciA/IGAke3Byb3BzLnJ1bkFzVXNlci51aWR9OiR7cHJvcHMucnVuQXNVc2VyLmdpZH1gIDogdW5kZWZpbmVkO1xuICAgIGNvbnN0IGNvbnRhaW5lckRlZmluaXRpb24gPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lckRlZmluaXRpb24nLCB7XG4gICAgICBpbWFnZSxcbiAgICAgIG1lbW9yeVJlc2VydmF0aW9uTWlCOiAyMDQ4LFxuICAgICAgZW52aXJvbm1lbnQsXG4gICAgICBsb2dnaW5nOiBMb2dEcml2ZXIuYXdzTG9ncyh7XG4gICAgICAgIGxvZ0dyb3VwOiB0aGlzLmxvZ0dyb3VwLFxuICAgICAgICBzdHJlYW1QcmVmaXg6ICdSQ1MnLFxuICAgICAgfSksXG4gICAgICB1c2VyLFxuICAgIH0pO1xuXG4gICAgY29udGFpbmVyRGVmaW5pdGlvbi5hZGRNb3VudFBvaW50cyhjb25uZWN0aW9uLnJlYWRXcml0ZU1vdW50UG9pbnQpO1xuXG4gICAgaWYgKHByb3BzLnNlY3JldHNNYW5hZ2VtZW50T3B0aW9ucyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAvLyBDcmVhdGUgdm9sdW1lIHRvIHBlcnNpc3QgdGhlIFJTQSBrZXlwYWlycyBnZW5lcmF0ZWQgYnkgRGVhZGxpbmUgYmV0d2VlbiBFQ1MgdGFza3NcbiAgICAgIC8vIFRoaXMgbWFrZXMgaXQgc28gc3Vic2VxdWVudCBFQ1MgdGFza3MgdXNlIHRoZSBzYW1lIGluaXRpYWwgU2VjcmV0cyBNYW5hZ2VtZW50IGlkZW50aXR5XG4gICAgICBjb25zdCB2b2x1bWVOYW1lID0gJ2RlYWRsaW5lLXVzZXIta2V5cGFpcnMnO1xuICAgICAgdGFza0RlZmluaXRpb24uYWRkVm9sdW1lKHtcbiAgICAgICAgbmFtZTogdm9sdW1lTmFtZSxcbiAgICAgICAgZG9ja2VyVm9sdW1lQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIHNjb3BlOiBTY29wZS5TSEFSRUQsXG4gICAgICAgICAgYXV0b3Byb3Zpc2lvbjogdHJ1ZSxcbiAgICAgICAgICBkcml2ZXI6ICdsb2NhbCcsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gTW91bnQgdGhlIHZvbHVtZSBpbnRvIHRoZSBjb250YWluZXIgYXQgdGhlIGxvY2F0aW9uIHdoZXJlIERlYWRsaW5lIGV4cGVjdHMgaXRcbiAgICAgIGNvbnRhaW5lckRlZmluaXRpb24uYWRkTW91bnRQb2ludHMoe1xuICAgICAgICByZWFkT25seTogZmFsc2UsXG4gICAgICAgIHNvdXJjZVZvbHVtZTogdm9sdW1lTmFtZSxcbiAgICAgICAgY29udGFpbmVyUGF0aDogYC9ob21lLyR7cHJvcHMuc2VjcmV0c01hbmFnZW1lbnRPcHRpb25zLnBvc2l4VXNlcm5hbWV9Ly5jb25maWcvLm1vbm8va2V5cGFpcnNgLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gSW5jcmVhc2UgdWxpbWl0c1xuICAgIGNvbnRhaW5lckRlZmluaXRpb24uYWRkVWxpbWl0cyhcbiAgICAgIHtcbiAgICAgICAgbmFtZTogVWxpbWl0TmFtZS5OT0ZJTEUsXG4gICAgICAgIHNvZnRMaW1pdDogMjAwMDAwLFxuICAgICAgICBoYXJkTGltaXQ6IDIwMDAwMCxcbiAgICAgIH0sIHtcbiAgICAgICAgbmFtZTogVWxpbWl0TmFtZS5OUFJPQyxcbiAgICAgICAgc29mdExpbWl0OiA2NDAwMCxcbiAgICAgICAgaGFyZExpbWl0OiA2NDAwMCxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGNvbnRhaW5lckRlZmluaXRpb24uYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgIGNvbnRhaW5lclBvcnQ6IHBvcnROdW1iZXIsXG4gICAgICBob3N0UG9ydDogcG9ydE51bWJlcixcbiAgICB9KTtcblxuICAgIHJldHVybiB0YXNrRGVmaW5pdGlvbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVja3MgaWYgdGhlIHVzZXIgc3VwcGxpZWQgYW55IGNlcnRpZmljYXRlIHRvIHVzZSBmb3IgVExTIGFuZCB1c2VzIHRoZW0sIG9yIGNyZWF0ZXMgZGVmYXVsdHMgdG8gdXNlLlxuICAgKiBAcGFyYW0gcHJvcHNcbiAgICogQHJldHVybnMgVGxzSW5mbyBlaXRoZXIgYmFzZWQgb24gaW5wdXQgdG8gdGhlIHJlbmRlciBxdWV1ZSwgb3IgdGhlIGNyZWF0ZWQgZGVmYXVsdHNcbiAgICovXG4gIHByaXZhdGUgZ2V0T3JDcmVhdGVUbHNJbmZvKHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzKTogVGxzSW5mbyB7XG4gICAgaWYgKCAocHJvcHMudHJhZmZpY0VuY3J5cHRpb24/LmV4dGVybmFsVExTPy5hY21DZXJ0aWZpY2F0ZSAhPT0gdW5kZWZpbmVkICkgfHxcbiAgICAgICAgKHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uPy5leHRlcm5hbFRMUz8ucmZka0NlcnRpZmljYXRlICE9PSB1bmRlZmluZWQpICkge1xuICAgICAgaWYgKHByb3BzLmhvc3RuYW1lID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgaG9zdG5hbWUgZm9yIHRoZSByZW5kZXIgcXVldWUgbXVzdCBiZSBkZWZpbmVkIGlmIHN1cHBseWluZyB5b3VyIG93biBjZXJ0aWZpY2F0ZXMuJyk7XG4gICAgICB9XG4gICAgICByZXR1cm4gdGhpcy5nZXRUbHNJbmZvRnJvbVVzZXJQcm9wcyhcbiAgICAgICAgcHJvcHMudHJhZmZpY0VuY3J5cHRpb24uZXh0ZXJuYWxUTFMsXG4gICAgICAgIHByb3BzLmhvc3RuYW1lLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5jcmVhdGVEZWZhdWx0VGxzSW5mbyhwcm9wcy52cGMsIHByb3BzLmhvc3RuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgZGVmYXVsdCBjZXJ0aWZpY2F0ZSB0byB1c2UgZm9yIFRMUyBhbmQgYSBQcml2YXRlSG9zdGVkWm9uZSB0byBwdXQgdGhlIGxvYWQgYmFsYW5jZXIgaW4uXG4gICAqIEBwYXJhbSB2cGNcbiAgICogQHBhcmFtIGhvc3RuYW1lXG4gICAqIEByZXR1cm5zIGRlZmF1bHQgVGxzSW5mb1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVEZWZhdWx0VGxzSW5mbyh2cGM6IElWcGMsIGhvc3RuYW1lPzogUmVuZGVyUXVldWVIb3N0TmFtZVByb3BzKSB7XG4gICAgY29uc3QgZG9tYWluWm9uZSA9IGhvc3RuYW1lPy56b25lID8/IG5ldyBQcml2YXRlSG9zdGVkWm9uZSh0aGlzLCAnRG5zWm9uZScsIHtcbiAgICAgIHZwYzogdnBjLFxuICAgICAgem9uZU5hbWU6IFJlbmRlclF1ZXVlLkRFRkFVTFRfRE9NQUlOX05BTUUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBmdWxseVF1YWxpZmllZERvbWFpbk5hbWUgPSB0aGlzLmdlbmVyYXRlRnVsbHlRdWFsaWZpZWREb21haW5OYW1lKGRvbWFpblpvbmUsIGhvc3RuYW1lPy5ob3N0bmFtZSk7XG5cbiAgICBjb25zdCByb290Q2EgPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKHRoaXMsICdSb290Q0EnLCB7XG4gICAgICBzdWJqZWN0OiB7XG4gICAgICAgIGNuOiAnUmVuZGVyUXVldWVSb290Q0EnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCByZmRrQ2VydCA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQZW0odGhpcywgJ1JlbmRlclF1ZXVlUGVtQ2VydCcsIHtcbiAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgY246IGZ1bGx5UXVhbGlmaWVkRG9tYWluTmFtZSxcbiAgICAgIH0sXG4gICAgICBzaWduaW5nQ2VydGlmaWNhdGU6IHJvb3RDYSxcbiAgICB9KTtcbiAgICBjb25zdCBzZXJ2ZXJDZXJ0ID0gbmV3IEltcG9ydGVkQWNtQ2VydGlmaWNhdGUoIHRoaXMsICdBY21DZXJ0JywgcmZka0NlcnQgKTtcbiAgICBjb25zdCBjZXJ0Q2hhaW4gPSByZmRrQ2VydC5jZXJ0Q2hhaW4hO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGRvbWFpblpvbmUsXG4gICAgICBmdWxseVF1YWxpZmllZERvbWFpbk5hbWUsXG4gICAgICBzZXJ2ZXJDZXJ0LFxuICAgICAgY2VydENoYWluLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogR2V0cyB0aGUgY2VydGlmaWNhdGUgYW5kIFByaXZhdGVIb3N0ZWRab25lIHByb3ZpZGVkIGluIHRoZSBSZW5kZXIgUXVldWUncyBjb25zdHJ1Y3QgcHJvcHMuXG4gICAqIEBwYXJhbSBleHRlcm5hbFRMU1xuICAgKiBAcGFyYW0gaG9zdG5hbWVcbiAgICogQHJldHVybnMgVGhlIHByb3ZpZGVkIGNlcnRpZmljYXRlIGFuZCBkb21haW4gaW5mb1xuICAgKi9cbiAgcHJpdmF0ZSBnZXRUbHNJbmZvRnJvbVVzZXJQcm9wcyhleHRlcm5hbFRMUzogUmVuZGVyUXVldWVFeHRlcm5hbFRMU1Byb3BzLCBob3N0bmFtZTogUmVuZGVyUXVldWVIb3N0TmFtZVByb3BzKTogVGxzSW5mbyB7XG4gICAgbGV0IHNlcnZlckNlcnQ6IElDZXJ0aWZpY2F0ZTtcbiAgICBsZXQgY2VydENoYWluOiBJU2VjcmV0O1xuXG4gICAgaWYgKCAoZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGUgIT09IHVuZGVmaW5lZCApICYmXG4gICAgKGV4dGVybmFsVExTLnJmZGtDZXJ0aWZpY2F0ZSAhPT0gdW5kZWZpbmVkKSApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRXhhY3RseSBvbmUgb2YgZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGUgYW5kIGV4dGVybmFsVExTLnJmZGtDZXJ0aWZpY2F0ZSBtdXN0IGJlIHByb3ZpZGVkIHdoZW4gdXNpbmcgZXh0ZXJuYWxUTFMuJyk7XG4gICAgfVxuXG4gICAgaWYgKCFob3N0bmFtZS5ob3N0bmFtZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIGhvc3RuYW1lIG11c3QgYmUgc3VwcGxpZWQgaWYgYSBjZXJ0aWZpY2F0ZSBpcyBzdXBwbGllZCwgJ1xuICAgICAgICArICd3aXRoIHRoZSBjb21tb24gbmFtZSBvZiB0aGUgY2VydGlmaWNhdGUgbWF0Y2hpbmcgdGhlIGhvc3RuYW1lICsgZG9tYWluIG5hbWUuJyk7XG4gICAgfVxuXG4gICAgY29uc3QgZnVsbHlRdWFsaWZpZWREb21haW5OYW1lID0gdGhpcy5nZW5lcmF0ZUZ1bGx5UXVhbGlmaWVkRG9tYWluTmFtZShob3N0bmFtZS56b25lLCBob3N0bmFtZS5ob3N0bmFtZSk7XG5cbiAgICBpZiAoIGV4dGVybmFsVExTLmFjbUNlcnRpZmljYXRlICkge1xuICAgICAgaWYgKCBleHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZUNoYWluID09PSB1bmRlZmluZWQgKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGVDaGFpbiBtdXN0IGJlIHByb3ZpZGVkIHdoZW4gdXNpbmcgZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGUuJyk7XG4gICAgICB9XG4gICAgICBzZXJ2ZXJDZXJ0ID0gZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGU7XG4gICAgICBjZXJ0Q2hhaW4gPSBleHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZUNoYWluO1xuXG4gICAgfSBlbHNlIHsgLy8gVXNpbmcgZXh0ZXJuYWxUTFMucmZka0NlcnRpZmljYXRlXG4gICAgICBpZiAoIGV4dGVybmFsVExTLnJmZGtDZXJ0aWZpY2F0ZSEuY2VydENoYWluID09PSB1bmRlZmluZWQgKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignUHJvdmlkZWQgcmZka0NlcnRpZmljYXRlIGRvZXMgbm90IGNvbnRhaW4gYSBjZXJ0aWZpY2F0ZSBjaGFpbi4nKTtcbiAgICAgIH1cbiAgICAgIHNlcnZlckNlcnQgPSBuZXcgSW1wb3J0ZWRBY21DZXJ0aWZpY2F0ZSggdGhpcywgJ0FjbUNlcnQnLCBleHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUhICk7XG4gICAgICBjZXJ0Q2hhaW4gPSBleHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUhLmNlcnRDaGFpbjtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgZG9tYWluWm9uZTogaG9zdG5hbWUuem9uZSxcbiAgICAgIGZ1bGx5UXVhbGlmaWVkRG9tYWluTmFtZSxcbiAgICAgIHNlcnZlckNlcnQsXG4gICAgICBjZXJ0Q2hhaW4sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBIZWxwZXIgbWV0aG9kIHRvIGNyZWF0ZSB0aGUgZnVsbHkgcXVhbGlmaWVkIGRvbWFpbiBuYW1lIGZvciB0aGUgZ2l2ZW4gaG9zdG5hbWUgYW5kIFByaXZhdGVIb3N0ZWRab25lLlxuICAgKiBAcGFyYW0gaG9zdG5hbWVcbiAgICogQHBhcmFtIHpvbmVcbiAgICogQHJldHVybnMgVGhlIGZ1bGx5IHF1YWxpZmllZCBkb21haW4gbmFtZVxuICAgKi9cbiAgcHJpdmF0ZSBnZW5lcmF0ZUZ1bGx5UXVhbGlmaWVkRG9tYWluTmFtZSh6b25lOiBJUHJpdmF0ZUhvc3RlZFpvbmUsIGhvc3RuYW1lOiBzdHJpbmcgPSBSZW5kZXJRdWV1ZS5ERUZBVUxUX0hPU1ROQU1FKTogc3RyaW5nIHtcbiAgICBpZiAoIVJlbmRlclF1ZXVlLlJFX1ZBTElEX0hPU1ROQU1FLnRlc3QoaG9zdG5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgUmVuZGVyUXVldWUgaG9zdG5hbWU6ICR7aG9zdG5hbWV9YCk7XG4gICAgfVxuICAgIHJldHVybiBgJHtob3N0bmFtZX0uJHt6b25lLnpvbmVOYW1lfWA7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIHRoYXQgcnVucyBjb21tYW5kcyBkdXJpbmcgdGhlIGRlcGxveW1lbnQuXG4gICAqL1xuICBwcml2YXRlIGdldCBkZXBsb3ltZW50SW5zdGFuY2UoKTogRGVwbG95bWVudEluc3RhbmNlIHtcbiAgICBjb25zdCBDT05GSUdVUkVfUkVQT1NJVE9SWV9DT05TVFJVQ1RfSUQgPSAnQ29uZmlndXJlUmVwb3NpdG9yeSc7XG4gICAgY29uc3QgZGVwbG95bWVudEluc3RhbmNlTm9kZSA9IHRoaXMubm9kZS50cnlGaW5kQ2hpbGQoQ09ORklHVVJFX1JFUE9TSVRPUllfQ09OU1RSVUNUX0lEKTtcbiAgICBpZiAoZGVwbG95bWVudEluc3RhbmNlTm9kZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gbmV3IERlcGxveW1lbnRJbnN0YW5jZSh0aGlzLCBDT05GSUdVUkVfUkVQT1NJVE9SWV9DT05TVFJVQ1RfSUQsIHtcbiAgICAgICAgc2VjdXJpdHlHcm91cDogdGhpcy5iYWNrZW5kQ29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF0sXG4gICAgICAgIHZwYzogdGhpcy5wcm9wcy52cGMsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHRoaXMucHJvcHMudnBjU3VibmV0cyA/PyBSZW5kZXJRdWV1ZS5ERUZBVUxUX1ZQQ19TVUJORVRTX09USEVSLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIGlmIChkZXBsb3ltZW50SW5zdGFuY2VOb2RlIGluc3RhbmNlb2YgRGVwbG95bWVudEluc3RhbmNlKSB7XG4gICAgICByZXR1cm4gZGVwbG95bWVudEluc3RhbmNlTm9kZTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmV4cGVjdGVkIHR5cGUgZm9yICR7ZGVwbG95bWVudEluc3RhbmNlTm9kZS5ub2RlLnBhdGh9LiBFeHBlY3RlZCAke0RlcGxveW1lbnRJbnN0YW5jZS5uYW1lfSwgYnV0IGZvdW5kICR7dHlwZW9mKGRlcGxveW1lbnRJbnN0YW5jZU5vZGUpfS5gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVGhlIGNvbnN0cnVjdCB0aGF0IG1hbmFnZXMgRGVhZGxpbmUgU2VjcmV0cyBNYW5hZ2VtZW50IGlkZW50aXR5IHJlZ2lzdHJhdGlvbiBzZXR0aW5nc1xuICAgKi9cbiAgcHJpdmF0ZSBnZXQgaWRlbnRpdHlSZWdpc3RyYXRpb25TZXR0aW5ncygpOiBTZWNyZXRzTWFuYWdlbWVudElkZW50aXR5UmVnaXN0cmF0aW9uIHtcbiAgICBjb25zdCBJREVOVElUWV9SRUdJU1RSQVRJT05fQ09OU1RSVUNUX0lEID0gJ1NlY3JldHNNYW5hZ2VtZW50SWRlbnRpdHlSZWdpc3RyYXRpb24nO1xuICAgIGNvbnN0IHNlY3JldHNNYW5hZ2VtZW50SWRlbnRpdHlSZWdpc3RyYXRpb24gPSB0aGlzLm5vZGUudHJ5RmluZENoaWxkKElERU5USVRZX1JFR0lTVFJBVElPTl9DT05TVFJVQ1RfSUQpO1xuICAgIGlmICghc2VjcmV0c01hbmFnZW1lbnRJZGVudGl0eVJlZ2lzdHJhdGlvbikge1xuICAgICAgcmV0dXJuIG5ldyBTZWNyZXRzTWFuYWdlbWVudElkZW50aXR5UmVnaXN0cmF0aW9uKFxuICAgICAgICB0aGlzLCBJREVOVElUWV9SRUdJU1RSQVRJT05fQ09OU1RSVUNUX0lELCB7XG4gICAgICAgICAgZGVwbG95bWVudEluc3RhbmNlOiB0aGlzLmRlcGxveW1lbnRJbnN0YW5jZSxcbiAgICAgICAgICByZXBvc2l0b3J5OiB0aGlzLnJlcG9zaXRvcnksXG4gICAgICAgICAgcmVuZGVyUXVldWVTdWJuZXRzOiB0aGlzLnByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKFxuICAgICAgICAgICAgdGhpcy5wcm9wcy52cGNTdWJuZXRzQWxiID8/IFJlbmRlclF1ZXVlLkRFRkFVTFRfVlBDX1NVQk5FVFNfQUxCLFxuICAgICAgICAgICksXG4gICAgICAgICAgdmVyc2lvbjogdGhpcy5wcm9wcy52ZXJzaW9uLFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKHNlY3JldHNNYW5hZ2VtZW50SWRlbnRpdHlSZWdpc3RyYXRpb24gaW5zdGFuY2VvZiBTZWNyZXRzTWFuYWdlbWVudElkZW50aXR5UmVnaXN0cmF0aW9uKSB7XG4gICAgICByZXR1cm4gc2VjcmV0c01hbmFnZW1lbnRJZGVudGl0eVJlZ2lzdHJhdGlvbjtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmV4cGVjdGVkIHR5cGUgZm9yICR7c2VjcmV0c01hbmFnZW1lbnRJZGVudGl0eVJlZ2lzdHJhdGlvbi5ub2RlLnBhdGh9LiBFeHBlY3RlZCAke1NlY3JldHNNYW5hZ2VtZW50SWRlbnRpdHlSZWdpc3RyYXRpb24ubmFtZX0sIGJ1dCBmb3VuZCAke3R5cGVvZihzZWNyZXRzTWFuYWdlbWVudElkZW50aXR5UmVnaXN0cmF0aW9uKX0uYCk7XG4gICAgfVxuICB9XG59XG4iXX0=