"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const secrets_management_ref_1 = require("./secrets-management-ref");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 *
 * @stability stable
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 *
 * @stability stable
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 *
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 *
 * @stability stable
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 *
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 *
 * @stability stable
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    vpc,
 *    images: images.forRenderQueue(),
 *    repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *    vpc,
 *    renderQueue,
 *    deadlineGroups: ['group_name'],
 *    instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *    workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *    naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *    vpc,
 *    renderQueue: renderQueue,
 *    spotFleets: [fleet],
 *    configuration: {
 *      enableResourceTracker: true,
 *    },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *   - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *   - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *   - A policy to pass a fleet and instance role
 *   - A policy to create tags for spot fleet requests
 *
 * The Spot Fleet Requests that this construct configures Deadline to create will always use the latest version of the
 * corresponding EC2 Launch Template that was created for them.
 *
 * ![architecture diagram](/diagrams/deadline/ConfigureSpotEventPlugin.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 * - EC2 Launch Templates for each Spot Event Plugin fleet.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *    and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *    execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4;
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        (_c = props.renderQueue.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: (_e = (_d = props.configuration) === null || _d === void 0 ? void 0 : _d.awsInstanceStatus) !== null && _e !== void 0 ? _e : SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: (_g = (_f = props.configuration) === null || _f === void 0 ? void 0 : _f.deleteEC2SpotInterruptedWorkers) !== null && _g !== void 0 ? _g : false,
            DeleteTerminatedSlaves: (_j = (_h = props.configuration) === null || _h === void 0 ? void 0 : _h.deleteSEPTerminatedWorkers) !== null && _j !== void 0 ? _j : false,
            IdleShutdown: (_m = (_l = (_k = props.configuration) === null || _k === void 0 ? void 0 : _k.idleShutdown) === null || _l === void 0 ? void 0 : _l.toMinutes({ integral: true })) !== null && _m !== void 0 ? _m : 10,
            Logging: (_p = (_o = props.configuration) === null || _o === void 0 ? void 0 : _o.loggingLevel) !== null && _p !== void 0 ? _p : SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: (_r = (_q = props.configuration) === null || _q === void 0 ? void 0 : _q.preJobTaskMode) !== null && _r !== void 0 ? _r : SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: (_t = (_s = props.configuration) === null || _s === void 0 ? void 0 : _s.region) !== null && _t !== void 0 ? _t : region,
            ResourceTracker: (_v = (_u = props.configuration) === null || _u === void 0 ? void 0 : _u.enableResourceTracker) !== null && _v !== void 0 ? _v : true,
            StaggerInstances: (_x = (_w = props.configuration) === null || _w === void 0 ? void 0 : _w.maximumInstancesStartedPerCycle) !== null && _x !== void 0 ? _x : 50,
            State: (_z = (_y = props.configuration) === null || _y === void 0 ? void 0 : _y.state) !== null && _z !== void 0 ? _z : SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: (_1 = (_0 = props.configuration) === null || _0 === void 0 ? void 0 : _0.strictHardCap) !== null && _1 !== void 0 ? _1 : false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const deadlineGroups = Array.from(new Set((_2 = props.spotFleets) === null || _2 === void 0 ? void 0 : _2.map(fleet => fleet.deadlineGroups).reduce((p, c) => p.concat(c), [])));
        const deadlinePools = Array.from(new Set((_3 = props.spotFleets) === null || _3 === void 0 ? void 0 : _3.map(fleet => fleet.deadlinePools).reduce((p, c) => p === null || p === void 0 ? void 0 : p.concat(c !== null && c !== void 0 ? c : []), [])));
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: (_4 = props.renderQueue.certChain) === null || _4 === void 0 ? void 0 : _4.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
            deadlineGroups,
            deadlinePools,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        if (props.spotFleets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.spotFleets.forEach(spotFleet => {
                if (spotFleet.defaultSubnets) {
                    core_1.Annotations.of(spotFleet).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
                }
                props.renderQueue.configureSecretsManagementAutoRegistration({
                    dependent: resource,
                    role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                    registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                    vpc: props.vpc,
                    vpcSubnets: spotFleet.subnets,
                });
            });
        }
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each configuration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        var _b;
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchTemplateConfigs: fleet._launchTemplateConfigs,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: (_b = fleet.validUntil) === null || _b === void 0 ? void 0 : _b.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group.toLowerCase()]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.41.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,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