"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const tag_helpers_1 = require("./tag-helpers");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstance', () => {
    let app;
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                Ebs: assert_1.objectLike({
                    Encrypted: true,
                }),
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            Description: `Admin credentials for the MongoDB database ${core_1.Names.uniqueId(instance)}`,
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assert_1.arrayWith(assert_1.objectLike({
                Key: 'VolumeGrantAttach-6238D22B12',
                Value: '6238d22b121db8094cb816e2a49d2b61',
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: assert_1.objectLike({
                Statement: assert_1.arrayWith({
                    Action: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                }, {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Cert',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Key',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ServerCertPassphraseE4C3CB38',
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                }),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        }));
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('\",\"log_stream_name\":\"' + cloudInitLogPrefix + '-{instance_id}\",\"file_path\":\"' + cloudInitLogPath + '\",' +
                    '\"timezone\":\"Local\"},{\"log_group_name\":\"', '\",\"log_stream_name\":\"' + mongoLogPrefix + '-{instance_id}\",\"file_path\":\"' + mongoLogPath + '\"' +
                    ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}')),
            }),
        }));
        const userData = instance.userData.render();
        const token = '${Token[TOKEN.\\d+]}';
        // Make sure we add signal on exit
        const exitTrap = '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ' + token +
            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            'test \"${MONGO_SETUP_DIR} != \"\" && sudo umount \"${MONGO_SETUP_DIR}\n' +
            '}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(exitTrap)));
        const callExitTrap = 'trap exitTrap EXIT';
        expect(userData).toMatch(new RegExp(callExitTrap));
        const settings = 'set -xefuo pipefail';
        expect(userData).toMatch(new RegExp(settings));
        const createTempDir = 'mkdir -p $\\(dirname \'/tmp/' + token + token + '\'\\)\n';
        const s3Copy = 'aws s3 cp \'s3://' + token + '/' + token + token + '\' \'/tmp/' + token + token + '\'\n';
        // CloudWatch Agent
        const setE = 'set -e\n';
        const setChmod = 'chmod \\+x \'/tmp/' + token + token + '\'\n';
        const execute = '\'/tmp/' + token + token + '\' -i ${Token[AWS.Region.\\d+]} ' + token + '\n';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + setE + setChmod + execute)));
        // Make sure we mount EBS volume
        const mount = 'TMPDIR=$\\(mktemp -d\\)\n' +
            'pushd \"$TMPDIR\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'bash ./mountEbsBlockVolume.sh ' + token + ' xfs /var/lib/mongo rw \"\"\n' +
            'popd\n' +
            'rm -f /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + mount)));
        // install mongodb
        const bashCmd = 'bash /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + bashCmd)));
        // configureMongoDb
        const monogdbConfig = 'which mongod && test -f /etc/mongod.conf\n' +
            'sudo service mongod stop\n' +
            'MONGO_SETUP_DIR=$\\(mktemp -d\\)\n' +
            'mkdir -p \"${MONGO_SETUP_DIR}\"\n' +
            'sudo mount -t tmpfs -o size=50M tmpfs \"${MONGO_SETUP_DIR}\"\n' +
            'pushd \"${MONGO_SETUP_DIR}\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'cp /etc/mongod.conf .';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + monogdbConfig)));
        // Getting the server certificate
        const serverCertCmd = 'bash serverCertFromSecrets.sh \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(serverCertCmd)));
        // set mongodb certificates and credentials
        const monogdbCredentials = 'sudo mkdir -p /etc/mongod_certs\n' +
            'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
            'sudo chown root.mongod -R /etc/mongod_certs/\n' +
            'sudo chmod 640 -R /etc/mongod_certs/\n' +
            'sudo chmod 750 /etc/mongod_certs/\n' +
            'sudo chown mongod.mongod -R /var/lib/mongo\n' +
            'bash ./setMongoLimits.sh\n' +
            'bash ./setStoragePath.sh \"/var/lib/mongo\"\n' +
            'bash ./setMongoNoAuth.sh\n' +
            'sudo service mongod start\n' +
            'bash ./setAdminCredentials.sh \"' + token + '\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(monogdbCredentials)));
        // Setup for live deployment, and start mongod
        const startMongo = 'sudo service mongod stop\n' +
            'bash ./setLiveConfiguration.sh\n' +
            'sudo systemctl enable mongod\n' +
            'sudo service mongod start\n' +
            'popd';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(startMongo)));
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        }));
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        }));
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: core_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Volume', 1));
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assert_1.objectLike({
                'Fn::GetAtt': assert_1.arrayWith('Key961B73FD'),
            }),
        }));
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: core_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Size: volumeSize,
        }));
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: assert_1.arrayWith(assert_1.objectLike({
                Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
            })),
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, id, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: retention,
        }));
    });
    test('adds security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // WHEN
        instance.addSecurityGroup(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
        }));
    });
    tag_helpers_1.testConstructTags({
        constructName: 'MongoDbInstance',
        createConstruct: () => {
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.MongoDbInstance(isolatedStack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::EC2::NetworkInterface': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::EC2::Volume': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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