"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstaller', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
    });
    test('license rejection throws', () => {
        // GIVEN
        const errorString = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
        // Must throw when providing no SSPL option.
        expect(() => {
            new lib_1.MongoDbInstaller(stack, {
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            });
        }).toThrowError(errorString);
        // Must throw when explicitly rejecting the SSPL.
        expect(() => {
            new lib_1.MongoDbInstaller(stack, {
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_REJECTS_SSPL,
            });
        }).toThrowError(errorString);
    });
    test('linux installation', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        const installer = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // WHEN
        installer.installOnLinuxInstance(instance);
        const userData = instance.userData.render();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            Ref: asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            Ref: asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
            },
        }));
        // Make sure we download the mountEFS script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const installer = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // THEN
        expect(() => {
            installer.installOnLinuxInstance(windowsInstance);
        }).toThrowError('Target instance must be Linux.');
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        const installer1 = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        const installer2 = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // WHEN
        installer1.installOnLinuxInstance(instance);
        installer2.installOnLinuxInstance(instance);
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from installer1 & installer2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
});
//# sourceMappingURL=data:application/json;base64,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