"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
let stack;
let secretCert;
let certPassphraseID;
let importedAcmCertificate;
describe('ImportedAcmCertificate', () => {
    beforeEach(() => {
        // GIVEN
        stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
        secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
            subject: { cn: 'Server' },
        });
        certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
        // WHEN
        importedAcmCertificate = new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
            cert: secretCert.cert,
            certChain: secretCert.certChain,
            key: secretCert.key,
            passphrase: secretCert.passphrase,
        });
    });
    test('creates Custom::RFDK_AcmImportedCertificate', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
            X509CertificatePem: {
                Cert: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Cert',
                    ],
                },
                Key: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Key',
                    ],
                },
                Passphrase: {
                    Ref: certPassphraseID,
                },
                CertChain: '',
            },
            Tags: [
                {
                    Key: 'AcmCertImport-F4E2ABF9',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
                {
                    Key: 'Name',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
            ],
        }));
    });
    describe('creates AWS::DynamoDB::Table for database', () => {
        test('with PhysicalID partition key', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DynamoDB::Table', {
                AttributeDefinitions: assert_1.arrayWith({
                    AttributeName: 'PhysicalId',
                    AttributeType: 'S',
                }),
                KeySchema: assert_1.arrayWith({
                    AttributeName: 'PhysicalId',
                    KeyType: 'HASH',
                }),
            }));
        });
        test('with CustomResource sort key', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DynamoDB::Table', {
                AttributeDefinitions: assert_1.arrayWith({
                    AttributeName: 'CustomResource',
                    AttributeType: 'S',
                }),
                KeySchema: assert_1.arrayWith({
                    AttributeName: 'CustomResource',
                    KeyType: 'RANGE',
                }),
            }));
        });
    });
    test('creates AWS::IAM::Policy', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:ConditionCheckItem',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                    },
                    {
                        Action: 'dynamodb:DescribeTable',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Key',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: certPassphraseID,
                        },
                    },
                    {
                        Action: [
                            'acm:AddTagsToCertificate',
                            'acm:ImportCertificate',
                        ],
                        Condition: {
                            StringEquals: {
                                'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'acm:DeleteCertificate',
                            'acm:DescribeCertificate',
                            'acm:GetCertificate',
                        ],
                        Resource: '*',
                    },
                ],
            },
        }));
    });
    describe('custom resource lambda function', () => {
        test('uses correct handler', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Handler: 'x509-certificate.importCert',
            }));
        });
        test('uses RFDK lambda layer', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Layers: assert_1.arrayWith(assert_1.stringLike('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:*')),
            }));
        });
        test('sets DATABASE environment variable', () => {
            // GIVEN
            const table = importedAcmCertificate.node.findChild('Table');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Environment: {
                    Variables: assert_1.objectLike({
                        DATABASE: stack.resolve(table.tableName),
                    }),
                },
            }));
        });
    });
    describe('applyRemovalPolicy', () => {
        test('default RemovalPolicy is Delete', () => {
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Delete',
                UpdateReplacePolicy: 'Delete',
            }, assert_1.ResourcePart.CompleteDefinition));
        });
        test('Different policy can be applied', () => {
            importedAcmCertificate.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            }, assert_1.ResourcePart.CompleteDefinition));
        });
    });
    describe('metricDaysToExpiry', () => {
        let metricExpiry;
        beforeEach(() => {
            // GIVEN
            metricExpiry = importedAcmCertificate.metricDaysToExpiry();
        });
        test('uses certificate ARN', () => {
            var _a;
            // THEN
            expect((_a = metricExpiry.dimensions) === null || _a === void 0 ? void 0 : _a.CertificateArn).toEqual(importedAcmCertificate.certificateArn);
        });
        test('uses correct metric', () => {
            // THEN
            expect(metricExpiry.metricName).toEqual('DaysToExpiry');
            expect(metricExpiry.namespace).toEqual('AWS/CertificateManager');
            expect(metricExpiry.statistic).toEqual(aws_cloudwatch_1.Statistic.MINIMUM);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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