"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const tag_helpers_1 = require("./tag-helpers");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensionsMap: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
describe('HealthMonitor', () => {
    beforeEach(() => {
        app = new core_1.App();
        infraStack = new core_1.Stack(app, 'infraStack');
        hmStack = new core_1.Stack(app, 'hmStack');
        wfStack = new core_1.Stack(app, 'wfStack');
        vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
    });
    test('validating default health monitor properties', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: '*',
                    },
                ],
            },
            Description: `This key is used to encrypt SNS messages for ${core_1.Names.uniqueId(healthMonitor)}.`,
            EnableKeyRotation: true,
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: {
                            Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                        },
                        Sid: '0',
                    },
                ],
            },
            Topics: [
                {
                    Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                },
            ],
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: {
                'Fn::GetAtt': [
                    `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
                    'Arn',
                ],
            },
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating health monitor properties while passing a key', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
            encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::KMS::Key'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: 'arn:aws:kms:us-west-2:123456789012:key/testarn',
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating the target with default health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(hmStack).notTo((assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: assert_1.arrayWith(assert_1.deepObjectLike({
                CidrIp: '0.0.0.0/0',
                FromPort: 8081,
                IpProtocol: 'tcp',
                ToPort: 8081,
            })),
        })));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '8081',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('validating the target with custom health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {
            port: 7171,
        });
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('2 ASG gets registered to same LB', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, { port: 7171 });
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, { port: 7171 });
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 8,
            ActionsEnabled: true,
            DatapointsToAlarm: 8,
            Threshold: 0,
            TreatMissingData: 'notBreaching',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            ActionsEnabled: true,
            DatapointsToAlarm: 1,
            Threshold: 35,
            TreatMissingData: 'notBreaching',
        }));
    });
    test('validating LB target limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 50,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating LB listener limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'listeners-per-application-load-balancer',
                    max: 1,
                }, {
                    name: 'target-groups-per-action-on-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target group limit per lb', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'target-groups-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target limit exhaustion', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 2,
        });
        expect(() => {
            healthMonitor.registerFleet(fleet, {});
        }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
    });
    test('validating deletion protection', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            deletionProtection: false,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    test('specifying a subnet', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the public subnets
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
            ],
        }));
        // Make sure the private subnets aren't present
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
            ],
        }));
    });
    test('specifying a security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(infraStack, 'LBSecurityGroup', { vpc });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            securityGroup,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the security group
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            SecurityGroups: assert_1.arrayWith(hmStack.resolve(securityGroup.securityGroupId)),
        }));
        // HealthMonitor should not create its own security group
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'HealthMonitor',
            createConstruct: () => {
                // GIVEN
                const fleetStack = new core_1.Stack(app, 'FleetStack');
                const fleet = new TestMonitorableFleet(fleetStack, 'workerFleet', {
                    vpc,
                });
                // WHEN
                healthMonitor = new lib_1.HealthMonitor(hmStack, 'HealthMonitor', {
                    vpc,
                });
                healthMonitor.registerFleet(fleet, {});
                return hmStack;
            },
            resourceTypeCounts: {
                'AWS::KMS::Key': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::EC2::SecurityGroup': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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