"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const deployment_instance_1 = require("../lib/deployment-instance");
const tag_helpers_1 = require("./tag-helpers");
const DEFAULT_CONSTRUCT_ID = 'DeploymentInstance';
/**
 * Machine image that spies on the following user data methods:
 *
 * * `.addOnExitCommands`
 * * `.addExecuteFileCommand`
 */
class AmazonLinuxWithUserDataSpy extends aws_ec2_1.AmazonLinuxImage {
    getImage(scope) {
        const result = super.getImage(scope);
        jest.spyOn(result.userData, 'addOnExitCommands');
        jest.spyOn(result.userData, 'addExecuteFileCommand');
        return result;
    }
}
describe('DeploymentInstance', () => {
    let app;
    let depStack;
    let stack;
    let vpc;
    let target;
    beforeAll(() => {
        // GIVEN
        app = new cdk.App();
        depStack = new cdk.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
    });
    describe('defaults', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'DefaultsStack');
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
            });
        });
        describe('Auto-Scaling Group', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Auto-Scaling Group', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::AutoScaling::AutoScalingGroup', 1));
            });
            test('MaxSize is 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: '1',
                }));
            });
            test('MinSize is 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            });
            test('uses private subnets', () => {
                // GIVEN
                const privateSubnetIDs = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT }).subnetIds;
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    VPCZoneIdentifier: assert_1.arrayWith(...stack.resolve(privateSubnetIDs)),
                }));
            });
            test('waits 15 minutes for one signal', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    CreationPolicy: {
                        ResourceSignal: {
                            Count: 1,
                            Timeout: 'PT15M',
                        },
                    },
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('sets replacing update policy', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    UpdatePolicy: {
                        AutoScalingReplacingUpdate: {
                            WillReplace: true,
                        },
                        AutoScalingScheduledAction: {
                            IgnoreUnmodifiedGroupSizeProperties: true,
                        },
                    },
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('uses Launch Configuration', () => {
                // GIVEN
                const launchConfig = target.node.findChild('ASG').node.findChild('LaunchConfig');
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    LaunchConfigurationName: stack.resolve(launchConfig.ref),
                }));
            });
        });
        describe('Launch Configuration', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Launch Configuration', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::AutoScaling::LaunchConfiguration', 1));
            });
            test('uses latest Amazon Linux machine image', () => {
                // GIVEN
                const amazonLinux = aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 });
                const imageId = stack.resolve(amazonLinux.getImage(stack)).imageId;
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    ImageId: imageId,
                }));
            });
            test('uses t3.small', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    InstanceType: 't3.small',
                }));
            });
            test('Uses created Security Group', () => {
                // GIVEN
                const securityGroup = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceSecurityGroup'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: [
                        stack.resolve(securityGroup.securityGroupId),
                    ],
                }));
            });
            test('depends on policy', () => {
                // GIVEN
                const policy = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.findChild('DefaultPolicy')
                    .node.defaultChild);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    DependsOn: assert_1.arrayWith(stack.resolve(policy.logicalId)),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('Security Group', () => {
            test('creates Security Group in the desired VPC', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                    VpcId: stack.resolve(vpc.vpcId),
                }));
            });
        });
        describe('ASG IAM role', () => {
            let instanceRole;
            beforeAll(() => {
                // GIVEN
                instanceRole = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.defaultChild);
            });
            test('creates an instance profile', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                    Roles: [
                        { Ref: stack.getLogicalId(instanceRole) },
                    ],
                }));
            });
            test('creates a role that can be assumed by EC2', () => {
                // GIVEN
                const servicePrincipal = new iam.ServicePrincipal('ec2.amazonaws.com');
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                    AssumeRolePolicyDocument: {
                        Statement: [
                            {
                                Action: 'sts:AssumeRole',
                                Effect: 'Allow',
                                Principal: {
                                    Service: stack.resolve(servicePrincipal.policyFragment.principalJson).Service[0],
                                },
                            },
                        ],
                    },
                }));
            });
            test('can signal to CloudFormation', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: 'cloudformation:SignalResource',
                            Effect: 'Allow',
                            Resource: { Ref: 'AWS::StackId' },
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can write to the log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(logGroup.logGroupArn),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent install script', () => {
                // GIVEN
                const cloudWatchAgentScriptAsset = (target
                    .node.findChild('CloudWatchConfigurationScriptAsset'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                cloudWatchAgentScriptAsset.bucket.bucketArn,
                                cloudWatchAgentScriptAsset.bucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent configuration file SSM parameter', () => {
                // GIVEN
                const cloudWatchConfigSsmParam = (target
                    .node.findChild('StringParameter'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'ssm:DescribeParameters',
                                'ssm:GetParameters',
                                'ssm:GetParameter',
                                'ssm:GetParameterHistory',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(cloudWatchConfigSsmParam.parameterArn),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent installer from S3', () => {
                // GIVEN
                const cloudWatchAgentInstallerBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'CloudWatchAgentInstallerBucket', `arn:aws:s3:::amazoncloudwatch-agent-${stack.region}`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                cloudWatchAgentInstallerBucket.bucketArn,
                                cloudWatchAgentInstallerBucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch GPG installer from RFDK dependencies S3 bucket', () => {
                // GIVEN
                const rfdkExternalDepsBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'RfdkExternalDependenciesBucket', `arn:aws:s3:::rfdk-external-dependencies-${stack.region}`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                rfdkExternalDepsBucket.bucketArn,
                                rfdkExternalDepsBucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can scale the Auto-Scaling Group', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: 'autoscaling:UpdateAutoScalingGroup',
                            Condition: {
                                // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                                StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                            },
                            Effect: 'Allow',
                            Resource: '*',
                        }, 
                        // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                        // Auto-Scaling service
                        {
                            Action: 'ec2:DescribeTags',
                            Effect: 'Allow',
                            Resource: '*',
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
        });
        describe('CloudWatch Agent config SSM parameter', () => {
            test('configures log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assert_1.arrayWith('{"logs":{"logs_collected":{"files":{"collect_list":[{"log_group_name":"', stack.resolve(logGroup.logGroupName)),
                        ],
                    },
                }));
            });
            test('configures cloud-init log', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assert_1.arrayWith(assert_1.stringLike('*"log_stream_name":"cloud-init-output-{instance_id}","file_path":"/var/log/cloud-init-output.log",*')),
                        ],
                    },
                }));
            });
        });
        describe('Tags resources with RFDK meta-data', () => {
            tag_helpers_1.testConstructTags({
                constructName: 'DeploymentInstance',
                createConstruct: () => {
                    return stack;
                },
                resourceTypeCounts: {
                    'AWS::EC2::SecurityGroup': 1,
                    'AWS::IAM::Role': 1,
                    'AWS::AutoScaling::AutoScalingGroup': 1,
                    'AWS::SSM::Parameter': 1,
                },
            });
        });
        // RFDK adds the resourceLogicalId tag to the Auto-Scaling Group in order to scope down the permissions of the
        // IAM policy given to the EC2 instance profile so that only that ASG can be scaled by the instance.
        test('Tagging for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', matcher));
        });
    });
    describe('User Data', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'UserDataStack');
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, 'DeploymentInstanceNew', {
                vpc,
                // a hack to be able to spy on the user data's "addOnExitCommand" and "addExecuteFileCommand" methods.
                machineImage: new AmazonLinuxWithUserDataSpy(),
            });
        });
        test('configures self-termination', () => {
            // THEN
            expect(target.userData.addOnExitCommands).toHaveBeenCalledWith('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', `ASG="$(aws --region ${stack.region} ec2 describe-tags --filters "Name=resource-id,Values=\${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"`, `aws --region ${stack.region} autoscaling update-auto-scaling-group --auto-scaling-group-name \${ASG} --min-size 0 --max-size 0 --desired-capacity 0`);
        });
        test('configures CloudWatch Agent', () => {
            // GIVEN
            const spy = target.userData.addExecuteFileCommand;
            const cloudWatchConfigSsmParam = (target
                .node.findChild('StringParameter'));
            // THEN
            // Should have been called
            expect(spy.mock.calls.length).toBeGreaterThanOrEqual(1);
            // The first call...
            const executeFileOptions = spy.mock.calls[0][0];
            // Should have been called with arguments
            const args = executeFileOptions.arguments;
            expect(args).not.toBeUndefined();
            const splitArgs = args.split(' ');
            // Should have three arguments
            expect(splitArgs).toHaveLength(3);
            // Specify the flag to install the CloudWatch Agent
            expect(splitArgs[0]).toEqual('-i');
            // Should pass the region
            expect(stack.resolve(splitArgs[1])).toEqual(stack.resolve(stack.region));
            // Should pass the SSM parameter containing the CloudWatch Agent configuration
            expect(stack.resolve(splitArgs[2])).toEqual(stack.resolve(cloudWatchConfigSsmParam.parameterName));
        });
    });
    describe('Custom::LogRetention.LogGroupName', () => {
        beforeEach(() => {
            // We need a clean construct tree, because the tests use the same construct ID
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'Stack');
        });
        // GIVEN
        test.each([
            [
                {},
                // defaults expected final log group name
                `/renderfarm/${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                { logGroupName: 'foo' },
                // expected final log group name
                '/renderfarm/foo',
            ],
            [
                {
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                `logGroupPrefix${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                {
                    logGroupName: 'logGroupName',
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                'logGroupPrefixlogGroupName',
            ],
        ])('%s => %s', ({ logGroupName, logGroupPrefix }, expectedLogGroupName) => {
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
                logGroupName,
                logGroupProps: logGroupPrefix ? { logGroupPrefix } : undefined,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
                LogGroupName: expectedLogGroupName,
            }));
        });
    });
    // GIVEN
    test('uses specified instance type', () => {
        // GIVEN
        const instanceType = new aws_ec2_1.InstanceType('c5.large');
        // We want an isolated stack to ensure expectCDK is only searching resources
        // synthesized by the specific DeploymentInstance stack
        stack = new cdk.Stack(app, 'InstanceTypeStack');
        // WHEN
        new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
            vpc,
            instanceType,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: instanceType.toString(),
        }));
    });
    test('uses specified security group', () => {
        // GIVEN
        const securityGroupId = 'securityGroupId';
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(depStack, 'SecurityGroup', securityGroupId);
        stack = new cdk.Stack(app, 'SecurityGroupStack');
        // WHEN
        new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
            vpc,
            securityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assert_1.arrayWith(securityGroupId),
        }));
    });
    describe('.selfTermination = false', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'SelfTerminationDisabledStack');
            // Spy on user data method calls
            const machineImage = new AmazonLinuxWithUserDataSpy();
            const deploymentInstanceProps = {
                vpc,
                selfTerminate: false,
                machineImage,
            };
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        test('does not add on-exit commands', () => {
            // THEN
            expect(target.userData.addOnExitCommands).not.toHaveBeenCalledWith(expect.arrayContaining([
                expect.stringMatching(/\baws\s+.*\bautoscaling\s+update-auto-scaling-group/),
            ]));
        });
        test('is not granted IAM permissions to scale the Auto-Scaling Group', () => {
            // GIVEN
            const instanceRole = (target
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 'autoscaling:UpdateAutoScalingGroup',
                        Condition: {
                            // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                            StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                        },
                        Effect: 'Allow',
                        Resource: '*',
                    }, 
                    // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                    // Auto-Scaling service
                    {
                        Action: 'ec2:DescribeTags',
                        Effect: 'Allow',
                        Resource: '*',
                    }),
                },
                Roles: [
                    stack.resolve(instanceRole.ref),
                ],
            }));
        });
        test('does not tag for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', matcher));
        });
    });
    // GIVEN
    describe('.executionTimeout is specified', () => {
        const executionTimeout = cdk.Duration.minutes(30);
        beforeAll(() => {
            // GIVEN
            // Use a clean stack to not pollute other stacks with resources
            stack = new cdk.Stack(app, 'ExecutionTimeout');
            const deploymentInstanceProps = {
                vpc,
                executionTimeout,
            };
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        // THEN
        test('AWS::AutoScaling::AutoScalingGroup creation policy signal timeout is set accordingly', () => {
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                CreationPolicy: {
                    ResourceSignal: {
                        Count: 1,
                        Timeout: executionTimeout.toIsoString(),
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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